/*!The Treasure Box Library
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * Copyright (C) 2009-present, TBOOX Open Source Group.
 *
 * @author      ruki
 * @file        platform.c
 * @ingroup     platform
 *
 */

/* //////////////////////////////////////////////////////////////////////////////////////
 * includes
 */
#include "platform.h"
#include "impl.h"
#include "../cpu.h"
#include "../exception.h"
#include "../cache_time.h"
#include "../../network/network.h"
#if defined(TB_CONFIG_OS_ANDROID)
#   include "../android/android.h"
#elif defined(TB_CONFIG_OS_WINDOWS)
#   include "../windows/windows.h"
#endif

/* //////////////////////////////////////////////////////////////////////////////////////
 * declaration
 */
__tb_extern_c_enter__
tb_bool_t tb_process_group_init();
tb_void_t tb_process_group_exit();
__tb_extern_c_leave__

/* //////////////////////////////////////////////////////////////////////////////////////
 * implementation
 */

tb_bool_t tb_platform_init_env(tb_handle_t priv)
{
    // init the current platform environment
#if defined(TB_CONFIG_OS_ANDROID)
    if (!tb_android_init_env(priv)) return tb_false;
#elif defined(TB_CONFIG_OS_WINDOWS)
    if (!tb_windows_init_env()) return tb_false;
#endif

    // init socket environment
    if (!tb_socket_init_env()) return tb_false;

    // init dns environment
#ifndef TB_CONFIG_MICRO_ENABLE
    if (!tb_dns_init_env()) return tb_false;
#endif

    // init thread local environment
#ifndef TB_CONFIG_MICRO_ENABLE
    if (!tb_thread_local_init_env()) return tb_false;
#endif

    // init exception environment
#ifdef TB_CONFIG_EXCEPTION_ENABLE
    if (!tb_exception_init_env()) return tb_false;
#endif

    // init cpu count/cache
#ifndef TB_CONFIG_MICRO_ENABLE
    (tb_void_t)tb_cpu_count();
#endif

    // init the global process group
#ifndef TB_CONFIG_MICRO_ENABLE
    if (!tb_process_group_init()) return tb_false;
#endif

    // ok
    return tb_true;
}
tb_void_t tb_platform_exit_env()
{
    // exit all process in the process group
#ifndef TB_CONFIG_MICRO_ENABLE
    tb_process_group_exit();
#endif

    // exit exception environment
#ifdef TB_CONFIG_EXCEPTION_ENABLE
    tb_exception_exit_env();
#endif

    // exit thread local environment
#ifndef TB_CONFIG_MICRO_ENABLE
    tb_thread_local_exit_env();
#endif

    // exit dns environment
#ifndef TB_CONFIG_MICRO_ENABLE
    tb_dns_exit_env();
#endif

    // exit socket environment
    tb_socket_exit_env();

    // exit the current platform environment
#if defined(TB_CONFIG_OS_ANDROID)
    tb_android_exit_env();
#elif defined(TB_CONFIG_OS_WINDOWS)
    tb_windows_exit_env();
#endif
}

