// Copyright (C) 2006-2009 Kent-Andre Mardal and Simula Research Laboratory
//
// This file is part of SyFi.
//
// SyFi is free software: you can redistribute it and/or modify
// it under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 2 of the License, or
// (at your option) any later version.
//
// SyFi is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the
// GNU General Public License for more details.
//
// You should have received a copy of the GNU General Public License
// along with SyFi. If not, see <http://www.gnu.org/licenses/>.

#ifndef POLYGON_IS_INCLUDED
#define POLYGON_IS_INCLUDED

#include <string>
#include <ginac/ginac.h>

namespace SyFi
{

	enum  Repr_format
	{
		SUBS_PERFORMED = 1,
		SUBS_NOT_PERFORMED = 2
	};

	// TODO: Can be const too:
	// 	integrate

	class Line;
	class Triangle;
	class Rectangle;

	class Polygon
	{
		protected:
			Polygon(const std::string & subscript_="", unsigned int no_vert=0);
			Polygon(const Polygon& polygon);

			std::string subscript;
			GiNaC::exvector p;

		public:
			virtual ~Polygon() {}

			/* Return the topological dimension of the object. */
			virtual unsigned int no_space_dim() const = 0;

			virtual unsigned int no_vertices() const;
			virtual GiNaC::ex vertex(unsigned int i) const;

			virtual GiNaC::ex repr(Repr_format format = SUBS_PERFORMED) const = 0;
			virtual const std::string str() const = 0;

			virtual GiNaC::ex integrate(GiNaC::ex f, Repr_format format = SUBS_PERFORMED) = 0;
			virtual Polygon* copy() const = 0;

			// This design isn't very good, but doing it differently will require solving some memory issues:
			virtual Line line(unsigned int i) const;
			virtual Triangle triangle(unsigned int i) const;
			virtual Rectangle rectangle(unsigned int i) const;
	};

	class Line : public Polygon
	{
		protected:
			GiNaC::ex a_;
			GiNaC::ex b_;
		public:
			/* Constructing an empty object. */
			Line() {}
			/* x0_ and x1_ are points, of the type GiNaC::lst if dim>1 */
			Line(GiNaC::ex x0, GiNaC::ex x1, const std::string & subscript = "");
			Line(const Line& line);
			virtual ~Line(){}

			virtual unsigned int no_space_dim() const;

			GiNaC::ex a() const;
			GiNaC::ex b() const;

			virtual GiNaC::ex repr(Repr_format format = SUBS_PERFORMED) const;
			virtual GiNaC::ex repr(GiNaC::ex t, Repr_format format = SUBS_PERFORMED) const;
			virtual const std::string str() const;

			virtual GiNaC::ex integrate(GiNaC::ex f, Repr_format format = SUBS_PERFORMED);
			virtual Line* copy() const;
	};

	class ReferenceLine : public Line
	{
		public:
			ReferenceLine(const std::string & subscript = "");
			ReferenceLine(const ReferenceLine& line);
			virtual ~ReferenceLine(){}

			virtual GiNaC::ex repr(GiNaC::ex t, Repr_format format = SUBS_PERFORMED) const;
			virtual const std::string str() const;

			virtual GiNaC::ex integrate(GiNaC::ex f, Repr_format format = SUBS_PERFORMED);
			virtual ReferenceLine* copy() const;
	};

	class Triangle : public Polygon
	{
		protected:
			Triangle(const std::string & subscript = "");
		public:
			Triangle(GiNaC::ex x0, GiNaC::ex x1, GiNaC::ex x2, const std::string & subscript = "");
			Triangle(const Triangle& triangle);
			virtual ~Triangle(){}

			virtual unsigned int no_space_dim() const;

			virtual Line line(unsigned int i) const;

			virtual GiNaC::ex repr(Repr_format = SUBS_PERFORMED) const;
			virtual const std::string str() const;

			virtual GiNaC::ex integrate(GiNaC::ex f, Repr_format format = SUBS_PERFORMED);
			virtual Triangle* copy() const;

	};

	class ReferenceTriangle : public Triangle
	{
		public:
			ReferenceTriangle(const std::string & subscript = "");
			ReferenceTriangle(const ReferenceTriangle& triangle);
			virtual ~ReferenceTriangle(){}

			virtual const std::string str() const;

			virtual GiNaC::ex integrate(GiNaC::ex f, Repr_format format = SUBS_PERFORMED);
			virtual ReferenceTriangle* copy() const;
	};

	class Rectangle : public Polygon
	{
		protected:
			Rectangle(const std::string & subscript = "");
		public:
			Rectangle(GiNaC::ex p0, GiNaC::ex p1, const std::string & subscript = "");
			Rectangle(GiNaC::ex p0, GiNaC::ex p1, GiNaC::ex p2, GiNaC::ex p3, const std::string & subscript = "");
			Rectangle(const Rectangle& rectangle);
			virtual ~Rectangle(){}

			virtual unsigned int no_space_dim() const;

			virtual Line line(unsigned int i) const;

			virtual GiNaC::ex repr(Repr_format format = SUBS_PERFORMED) const;
			virtual const std::string str() const;

			virtual GiNaC::ex integrate(GiNaC::ex f, Repr_format format = SUBS_PERFORMED);
			virtual Rectangle* copy() const;
	};

	class ReferenceRectangle : public Rectangle
	{
		public:
			ReferenceRectangle(const std::string & subscript = "");
			ReferenceRectangle(const ReferenceRectangle& rectangle);
			virtual ~ReferenceRectangle(){}

			virtual const std::string str() const;
			virtual ReferenceRectangle* copy() const;
	};

	class Tetrahedron : public Polygon
	{
		public:
			Tetrahedron(GiNaC::ex x0, GiNaC::ex x1, GiNaC::ex x2, GiNaC::ex x3, const std::string & subscript = "");
			Tetrahedron(const Tetrahedron& tetrahedron);
			virtual ~Tetrahedron(){}

			virtual unsigned int no_space_dim() const;

			virtual Line line(unsigned int i) const;
			virtual Triangle triangle(unsigned int i) const;

			virtual GiNaC::ex repr(Repr_format format = SUBS_PERFORMED) const;
			virtual const std::string str() const;

			virtual GiNaC::ex integrate(GiNaC::ex f, Repr_format format = SUBS_PERFORMED);
			virtual Tetrahedron* copy() const;

	};

	class ReferenceTetrahedron : public Tetrahedron
	{
		public:
			ReferenceTetrahedron(const std::string & subscript = "");
			ReferenceTetrahedron(const ReferenceTetrahedron& tetrahedron);
			virtual ~ReferenceTetrahedron(){}

			virtual const std::string str() const;

			virtual GiNaC::ex integrate(GiNaC::ex f, Repr_format format = SUBS_PERFORMED);
			virtual ReferenceTetrahedron* copy() const;
	};

	class Box: public Polygon
	{
		public:
			Box(GiNaC::ex p0, GiNaC::ex p1, const std::string & subscript="");
			Box(GiNaC::ex p0, GiNaC::ex p1, GiNaC::ex p2, GiNaC::ex p3, GiNaC::ex p4, GiNaC::ex p5, GiNaC::ex p6, GiNaC::ex p7, const std::string & subscript="");
			Box(const Box& box);
			Box(){}
			virtual ~Box(){}

			virtual unsigned int no_space_dim() const;

			virtual Line line(unsigned int i) const;
			virtual Rectangle rectangle(unsigned int i) const;

			virtual GiNaC::ex repr(Repr_format format = SUBS_PERFORMED) const;
			virtual const std::string str() const;

			virtual GiNaC::ex integrate(GiNaC::ex f, Repr_format format = SUBS_PERFORMED);
			virtual Box* copy() const;
	};

	class ReferenceBox: public Box
	{
		public:
			ReferenceBox(const std::string & subscript = "");
			ReferenceBox(const ReferenceBox& box);
			virtual ~ReferenceBox(){}

			virtual const std::string str() const;
			virtual ReferenceBox* copy() const;
	};

	class Simplex : public Polygon
	{
		public:
			Simplex(GiNaC::lst vertices, const std::string & subscript = "");
			Simplex(const Simplex& simplex);
			virtual ~Simplex(){}

			virtual unsigned int no_space_dim() const;

			virtual GiNaC::ex repr(Repr_format format = SUBS_PERFORMED) const;
			virtual const std::string str() const;

			virtual GiNaC::ex integrate(GiNaC::ex f, Repr_format format = SUBS_PERFORMED);
			Simplex sub_simplex(unsigned int i);
			virtual Simplex* copy() const;
	};

	/*
	class ReferenceSimplex : public Simplex {
	  public:
		ReferenceSimplex(const std::string & subscript = "");
		virtual ~ReferenceSimplex(){}

	virtual std::string str();

	virtual GiNaC::ex integrate(GiNaC::ex f, Repr_format format = SUBS_PERFORMED);
	};
	*/

	// ---- Some tools for Polygons

	// FIXME: change to barycenter(Triangle&)
	// FIXME:       and barycenter(Tetrahedron&)
	GiNaC::ex barycenter_line(GiNaC::ex p0, GiNaC::ex p1);
	GiNaC::ex barycenter_triangle(GiNaC::ex p0, GiNaC::ex p1, GiNaC::ex p2);
	GiNaC::ex barycenter_tetrahedron(GiNaC::ex p0, GiNaC::ex p1, GiNaC::ex p2, GiNaC::ex p3);
	GiNaC::ex barycenter(Simplex& simplex);
	GiNaC::lst bezier_ordinates(Line& line, unsigned int d);
	GiNaC::lst bezier_ordinates(Triangle& triangle, unsigned int d);
	GiNaC::lst bezier_ordinates(Tetrahedron& tetrahedra, unsigned int d);
	GiNaC::lst interior_coordinates(Line& line, unsigned int d);
	GiNaC::lst interior_coordinates(Triangle& triangle, unsigned int d);
	GiNaC::lst interior_coordinates(Tetrahedron& tetrahedra, unsigned int d);

	// polynom of arbitrary order on a line, a triangle,
	// or a tetrahedron using the Bernstein basis
	GiNaC::ex bernstein(unsigned int order, Polygon& p, const std::string & a);

	// vector polynom of arbitrary order on a line, a triangle,
	// or a tetrahedron using the Bernstein basis
	GiNaC::lst bernsteinv(unsigned int no_fields, unsigned int order, Polygon& p, const std::string & a);

	GiNaC::lst normal(Triangle&, unsigned int i);
	GiNaC::lst normal(Tetrahedron&, unsigned int i);

	GiNaC::lst tangent(Triangle&, unsigned int i);

}
#endif
