/* ====================================================================
 * Copyright (c) 2003-2006, Martin Hauner
 *                          http://subcommander.tigris.org
 *
 * Subcommander is licensed as described in the file doc/COPYING, which
 * you should have received as part of this distribution.
 * ====================================================================
 */

#ifndef _SPLIT_LAYOUT_H
#define _SPLIT_LAYOUT_H

// qt
#include <QtGui/QLayout>

// sys
#include <vector>


class SplitLayoutHandle
{
public:
  virtual ~SplitLayoutHandle(){}

  virtual void recalculate() = 0;
  virtual int  calcMovePos( int curX, int diffX ) = 0;
  virtual int  calcJumpPos( int curX ) = 0;
};

///////////////////////////////////////////////////////////////////////////////

typedef std::vector<QLayoutItem*> TLayoutItems;

///////////////////////////////////////////////////////////////////////////////

class SplitLayout : public QLayout, public SplitLayoutHandle
{
public:
  enum Pos
  {
    pOne    = 0,
    pHandle = 1,
    pTwo    = 2
  };

  SplitLayout( QWidget* parent, Pos );
  virtual ~SplitLayout();

  // SplitLayoutHandle
  void recalculate();
  int calcMovePos( int curPos, int diffPos );
  int calcJumpPos( int curPos );

  void addWidgetOne( QWidget* w, bool hide = false, int stretch = 1 );
  void addWidgetTwo( QWidget* w, bool hide = false, int stretch = 1 );
  void enableHandle( bool enable = true );

  void addItem(QLayoutItem *item);
  QLayoutItem* itemAt( int index ) const;
  QLayoutItem* takeAt( int index );
  int count() const;

  QSize sizeHint() const;
  QSize minimumSize() const;

  void jumpPos( bool visible );
  void setGeometry( const QRect &newRect );
  
  void setHandlePos( int pos );
  int getHandlePos() const;

protected:
  virtual int getPos( const QRect& r ) const = 0;
  virtual int getSize( const QRect& r ) const = 0;
  virtual int getSize( const QSize& s ) const = 0;
  virtual void adjustSize( QSize& out, const QSize& item ) const = 0;
  virtual QRect calcRect( int pos, int size, const QRect& r ) const = 0;

  TLayoutItems _items;

private:
  int   _stretchOne;
  int   _stretchTwo;
  Pos   _hide;

  QRect _lastRect;

  bool  _initialized;  // initial layout done?
  int   _handlePos;    // -1 = unset
};


class HSplitLayout : public SplitLayout
{
public:
  HSplitLayout( QWidget* parent, Pos pos = pOne );

  int getPos( const QRect& r ) const;
  int getSize( const QRect& r ) const;
  int getSize( const QSize& s ) const;
  void adjustSize( QSize& out, const QSize& item ) const;
  QRect calcRect( int pos, int size, const QRect& r ) const;
};

class VSplitLayout : public SplitLayout
{
public:
  VSplitLayout( QWidget* parent, Pos pos = pOne );

  int getPos( const QRect& r ) const;
  int getSize( const QRect& r ) const;
  int getSize( const QSize& s ) const;
  void adjustSize( QSize& out, const QSize& item ) const;
  QRect calcRect( int pos, int size, const QRect& r ) const;
};


#endif //  _SPLIT_LAYOUT_H
