/* ScummVM - Graphic Adventure Engine
 *
 * ScummVM is the legal property of its developers, whose names
 * are too numerous to list here. Please refer to the COPYRIGHT
 * file distributed with this source distribution.
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301, USA.
 *
 */

#include "common/endian.h"
#include "common/system.h"
#include "common/textconsole.h"

#include "agos/agos.h"
#include "agos/intern.h"

#include "graphics/surface.h"

namespace AGOS {

static const byte polish4CD_feebleFontSize[98] = {
	8, 2, 5, 8, 8, 8, 8, 2, 4, 4, 8, 8, 3, 8, 2, 9,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 2, 3, 8, 8, 8, 8,
	7, 8, 8, 8, 8, 8, 8, 8, 8, 4, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 3, 9,
	4, 8, 8, 8, 8, 8, 7, 8, 8, 4, 5, 7, 3, 8, 8, 8,
	8, 8, 8, 7, 7, 8, 8, 8, 8, 8, 8, 5, 2, 5, 8, 8,
	8, 8,
};

static const byte polish2CD_feebleFontSize[208] = {
	4, 2, 8, 8, 8, 8, 8, 2, 4, 4, 8, 8, 3, 8, 2, 9,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 2, 8, 8, 8, 8, 8,
	7, 8, 8, 8, 8, 8, 8, 8, 8, 4, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 8,
	8, 8, 8, 8, 8, 8, 7, 8, 8, 4, 5, 7, 3, 8, 8, 8,
	8, 8, 8, 7, 7, 8, 8, 8, 8, 8, 8, 5, 2, 5, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 4, 4, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 4, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 2, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

static const byte feebleFontSize[208] = {
	8, 2, 5, 7, 8, 8, 8, 2, 4, 4, 8, 8, 3, 8, 2, 9,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 2, 3, 5, 8, 5, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 9, 4, 4, 9,
	4, 8, 8, 8, 8, 8, 7, 8, 8, 4, 5, 7, 3, 8, 8, 8,
	8, 8, 8, 7, 7, 8, 8, 8, 8, 8, 8, 5, 2, 5, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 4, 4, 4, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 4, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 2, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
	8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8, 8,
};

uint AGOSEngine::getFeebleFontSize(byte chr) {
	if (getGameType() == GType_FF && (getFeatures() & GF_DEMO) && chr == 32) {
		return 4;
	} else if (_language == Common::PL_POL) {
		if (!strcmp(getExtra(), "4CD"))
			return polish4CD_feebleFontSize[chr - 32];
		else
			return polish2CD_feebleFontSize[chr - 32];
	} else {
		return feebleFontSize[chr - 32];
	}
}

#ifdef ENABLE_AGOS2
static const byte polish4CD_feeble_windowFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
	0x90, 0x90, 0x90, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x02, 0x04, 0x04,
	0x08, 0x10, 0x10, 0x7c, 0x82, 0x80, 0x80, 0x80, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x3e, 0x62, 0x92, 0x64, 0x08, 0x10, 0x20, 0x4c, 0x92, 0x8c, 0x00, 0x00, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x50, 0x20, 0x52, 0x8a, 0x84, 0x7a, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x40, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x40, 0x20, 0x00, 0x00, 0x00,
	0x80, 0x40, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x40, 0x80, 0x00, 0x00, 0x00,
	0x10, 0x92, 0x54, 0x38, 0x38, 0x54, 0x92, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x10, 0x10, 0xfe, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x80, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x80, 0x00, 0x00,
	0x78, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x78, 0x00, 0x00, 0x00,
	0x10, 0x30, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00, 0x00, 0x00,
	0x78, 0x84, 0x84, 0x04, 0x04, 0x08, 0x10, 0x20, 0x40, 0xfc, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x02, 0x02, 0x1c, 0x02, 0x02, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x04, 0x0c, 0x14, 0x24, 0x44, 0x84, 0xfe, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00,
	0xfe, 0x80, 0x80, 0xfc, 0x02, 0x02, 0x02, 0x02, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x3e, 0x40, 0x80, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfe, 0x02, 0x02, 0x02, 0x02, 0x04, 0x08, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x78, 0x84, 0x84, 0x84, 0x78, 0x84, 0x84, 0x84, 0x84, 0x78, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x40, 0x40, 0x80, 0x00,
	0x04, 0x08, 0x08, 0x9c, 0xa2, 0xc2, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x08, 0x10, 0x10, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x3c, 0x42, 0x82, 0x02, 0x04, 0x08, 0x10, 0x10, 0x00, 0x10, 0x10, 0x00, 0x00,
	0x08, 0x10, 0x10, 0x7c, 0x80, 0x80, 0x78, 0x04, 0x04, 0xf8, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0xfe, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0xfc, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x80, 0x80, 0x80, 0x80, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x00, 0x00, 0x00,
	0xfe, 0x80, 0x80, 0x80, 0x80, 0xfc, 0x80, 0x80, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0xfe, 0x80, 0x80, 0x80, 0x80, 0xfc, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x80, 0x80, 0x86, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0xfe, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0xe0, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x1e, 0x02, 0x02, 0x02, 0x02, 0x02, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x84, 0x88, 0x90, 0xe0, 0x90, 0x88, 0x84, 0x82, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0x82, 0xc6, 0xaa, 0x92, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x82, 0x82, 0xc2, 0xa2, 0x92, 0x8a, 0x86, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x80, 0x80, 0x7c, 0x02, 0x02, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfe, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x44, 0x28, 0x10, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x92, 0xaa, 0xc6, 0x00, 0x00, 0x00,
	0x82, 0x44, 0x28, 0x10, 0x10, 0x10, 0x10, 0x28, 0x44, 0x82, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x44, 0x28, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0xfe, 0x02, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x00, 0xfe, 0x04, 0x08, 0x10, 0x20, 0x40, 0xfe, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x04, 0x04, 0x08,
	0x08, 0x10, 0x10, 0xfe, 0x04, 0x08, 0x10, 0x20, 0x40, 0xfe, 0x00, 0x00, 0x00,
	0xc0, 0x40, 0x40, 0x60, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
	0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x80, 0x80, 0x80, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x02, 0x02, 0x02, 0x7e, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x1c, 0x20, 0x20, 0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x7c,
	0x80, 0x80, 0x80, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x40, 0x00, 0x00, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x00, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0xe0,
	0x80, 0x80, 0x80, 0x84, 0x88, 0x90, 0xe0, 0x90, 0x88, 0x84, 0x00, 0x00, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xfc, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x9c, 0xa2, 0xc2, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x80, 0x80, 0x80,
	0x00, 0x00, 0x00, 0x7e, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x02,
	0x00, 0x00, 0x00, 0x9c, 0xa2, 0xc2, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x80, 0x80, 0x78, 0x04, 0x04, 0xf8, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x82, 0x82, 0x82, 0x44, 0x28, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x44, 0x28, 0x10, 0x28, 0x44, 0x82, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x7c,
	0x00, 0x00, 0x00, 0xfe, 0x04, 0x08, 0x10, 0x20, 0x40, 0xfe, 0x00, 0x00, 0x00,
	0x30, 0x40, 0x40, 0x40, 0x40, 0x80, 0x40, 0x40, 0x40, 0x40, 0x30, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00,
	0xc0, 0x20, 0x20, 0x20, 0x20, 0x10, 0x20, 0x20, 0x20, 0x20, 0xc0, 0x00, 0x00,
	0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff, 0xff,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0xc8,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x88, 0x00, 0x11, 0x00, 0x00, 0x00,
	0x03, 0x00, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0xc8, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x14, 0x00, 0x86, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x88, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x88, 0x00, 0x11, 0x00, 0x00, 0x00, 0x03,
	0x00, 0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0xc8, 0x00, 0x03, 0x00,
	0x02, 0x00, 0x0e, 0x00, 0x81, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x03, 0x00, 0x00, 0x00, 0x0e, 0x00, 0x88, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x03, 0x00, 0x88, 0x00, 0x11, 0x00, 0x00, 0x00, 0x03, 0x00,
	0x88, 0x00, 0x00, 0x00, 0x00, 0x00, 0x14, 0x00, 0xc8, 0x00, 0x03, 0x00, 0x03,
	0x00, 0x0e, 0x00, 0x7f, 0x00, 0x45, 0x46, 0x46, 0x45, 0x43, 0x54, 0x53, 0x00,
	0x53, 0x49, 0x4d, 0x4f, 0x4e, 0x00, 0x00, 0x00, 0xa8, 0x4d, 0xee, 0x00, 0xb0,
	0x4d, 0xee, 0x00, 0x00, 0x53, 0x49, 0x4d, 0x4f, 0x4e, 0x32, 0x00, 0xc0, 0x4d,
	0xee, 0x00, 0xc1, 0x4d, 0xee, 0x00, 0x4d, 0x55, 0x53, 0x49, 0x43, 0x00, 0x00,
	0x00, 0xc0, 0x4d, 0xee, 0x00, 0xd0, 0x4d, 0xee, 0x00, 0x44, 0x72, 0x81, 0x63,
	0x6b, 0x65, 0x6e, 0x20, 0x53, 0x69, 0x65, 0x20, 0x64, 0x69, 0x65, 0x20, 0x3c,
	0x4c, 0x65, 0x65, 0x72, 0x74, 0x61, 0x73, 0x74, 0x65, 0x3e, 0x2c, 0x20, 0x75,
	0x6d, 0x20, 0x66, 0x6f, 0x72, 0x74, 0x7a, 0x75, 0x66, 0x61, 0x68, 0x72, 0x65,
	0x6e, 0x2e, 0x2e, 0x2e, 0x00, 0x50, 0x72, 0x65, 0x73, 0x73, 0x20, 0x3c, 0x53,
	0x50, 0x41, 0x43, 0x45, 0x3e, 0x20, 0x74, 0x6f, 0x20, 0x63, 0x6f, 0x6e, 0x74,
	0x69, 0x6e, 0x75, 0x65, 0x2e, 0x2e, 0x2e, 0x00, 0x65, 0x6e, 0x67, 0x69, 0x6e,
	0x65, 0x73, 0x2f, 0x61, 0x67, 0x6f, 0x73, 0x2f, 0x66, 0x65, 0x65, 0x62, 0x6c,
	0x65, 0x2e, 0x63, 0x70, 0x70, 0x00, 0x5f, 0x6d, 0x6f, 0x76, 0x69, 0x65, 0x50,
	0x6c, 0x61, 0x79, 0x65, 0x72, 0x00, 0x77, 0x6f, 0x62, 0x62, 0x6c, 0x65, 0x25,
	0x64, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x63, 0x6c, 0x69, 0x70, 0x73, 0x69,
	0x6e, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x63, 0x6c, 0x69, 0x70, 0x69, 0x6e,
	0x32, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x67, 0x6f, 0x31, 0x2e, 0x73, 0x6d,
	0x6b, 0x00, 0x6d, 0x61, 0x7a, 0x65, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x67,
	0x6f, 0x32, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x72, 0x61, 0x64, 0x69, 0x6f, 0x69,
	0x6e, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x67, 0x6f, 0x33, 0x2e, 0x73, 0x6d,
	0x6b, 0x00, 0x70, 0x61, 0x64, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x67, 0x6f,
	0x34, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x62, 0x72, 0x69, 0x64, 0x67, 0x65, 0x2e,
	0x73, 0x6d, 0x6b, 0x00, 0x66, 0x67, 0x6f, 0x35, 0x2e, 0x73, 0x6d, 0x6b, 0x00,
	0x70, 0x69, 0x6c, 0x6c, 0x64, 0x69, 0x65, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66,
	0x67, 0x6f, 0x36, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x62, 0x69, 0x6b, 0x65, 0x62,
	0x75, 0x73, 0x74, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x67, 0x6f, 0x37, 0x2e,
	0x73, 0x6d, 0x6b, 0x00, 0x73, 0x74, 0x61, 0x74, 0x75, 0x65, 0x2e, 0x73, 0x6d,
	0x6b, 0x00, 0x66, 0x67, 0x6f, 0x38, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x6a, 0x75,
	0x6e, 0x6b, 0x6f, 0x75, 0x74, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x67, 0x6f,
	0x39, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x68, 0x79, 0x70, 0x6e, 0x6f, 0x2e,
	0x73, 0x6d, 0x6b, 0x00, 0x66, 0x62, 0x79, 0x65, 0x31, 0x2e, 0x73, 0x6d, 0x6b,
	0x00, 0x66, 0x62, 0x79, 0x65, 0x32, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x6d, 0x6d,
	0x66, 0x61, 0x64, 0x65, 0x69, 0x6e, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x6d, 0x61,
	0x69, 0x6e, 0x6d, 0x65, 0x6e, 0x75, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x66,
	0x61, 0x64, 0x65, 0x35, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x74, 0x65, 0x78,
	0x74, 0x30, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x74, 0x65, 0x78, 0x74, 0x31,
	0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x74, 0x65, 0x78, 0x74, 0x32, 0x2e, 0x73,
	0x6d, 0x6b, 0x00, 0x66, 0x74, 0x65, 0x78, 0x74, 0x33, 0x2e, 0x73, 0x6d, 0x6b,
	0x00, 0x66, 0x74, 0x65, 0x78, 0x74, 0x34, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66,
	0x74, 0x65, 0x78, 0x74, 0x35, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x66, 0x61,
	0x64, 0x65, 0x31, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x6d, 0x75, 0x73, 0x6f, 0x73,
	0x70, 0x31, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x6e, 0x65, 0x77, 0x63, 0x72, 0x65,
	0x64, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x61, 0x73, 0x61, 0x6c, 0x6c, 0x2e,
	0x73, 0x6d, 0x6b, 0x00, 0x6d, 0x75, 0x73, 0x35, 0x70, 0x32, 0x2e, 0x73, 0x6d,
	0x6b, 0x00, 0x63, 0x6f, 0x61, 0x63, 0x68, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x6f,
	0x75, 0x74, 0x6d, 0x69, 0x6e, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x66, 0x61,
	0x64, 0x65, 0x33, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x69, 0x64, 0x66, 0x78, 0x34,
	0x61, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x69, 0x64, 0x66, 0x78, 0x34, 0x62, 0x2e,
	0x73, 0x6d, 0x6b, 0x00, 0x69, 0x64, 0x66, 0x78, 0x34, 0x63, 0x2e, 0x73, 0x6d,
	0x6b, 0x00, 0x69, 0x64, 0x66, 0x78, 0x34, 0x64, 0x2e, 0x73, 0x6d, 0x6b, 0x00,
	0x69, 0x64, 0x66, 0x78, 0x34, 0x65, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x69, 0x64,
	0x66, 0x78, 0x34, 0x66, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x69, 0x64, 0x66, 0x78,
	0x34, 0x67, 0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x66, 0x61, 0x64, 0x65, 0x32,
	0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x73, 0x63, 0x65, 0x6e, 0x65, 0x33, 0x62,
	0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x73, 0x63, 0x65, 0x6e, 0x65, 0x33, 0x61,
	0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x73, 0x63, 0x65, 0x6e, 0x65, 0x33, 0x63,
	0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x73, 0x63, 0x65, 0x6e, 0x65, 0x33, 0x67,
	0x2e, 0x73, 0x6d, 0x6b, 0x00, 0x66, 0x66, 0x61, 0x64, 0x65, 0x34, 0x2e, 0x73,
	0x6d, 0x6b, 0x00, 0x66, 0x66, 0x61, 0x64, 0x65, 0x36, 0x2e, 0x73, 0x6d, 0x6b,
	0x00, 0x77, 0x69, 0x6e, 0x61, 0x73, 0x6f, 0x66, 0x74, 0x2e, 0x73, 0x6d, 0x6b,
	0x00, 0x66, 0x62, 0x69, 0x67, 0x74, 0x61, 0x6c, 0x6b, 0x2e, 0x73, 0x6d, 0x6b,
	0x00, 0x65, 0x70, 0x69, 0x63, 0x2e, 0x64, 0x78, 0x61, 0x00, 0x00, 0x56, 0x4f,
	0x49, 0x43, 0x45, 0x53, 0x00, 0x00, 0x00, 0xc6, 0x50, 0xee, 0x00, 0xc7, 0x50,
	0xee, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x4f, 0x75, 0x74,
	0x20, 0x6f, 0x66, 0x20, 0x69, 0x63, 0x6f, 0x6e, 0x20, 0x6d, 0x65, 0x6d, 0x6f,
	0x72, 0x79, 0x00, 0x43, 0x61, 0x6e, 0x27, 0x74, 0x20, 0x6f, 0x70, 0x65, 0x6e,
	0x20, 0x69, 0x63, 0x6f, 0x6e, 0x73, 0x20, 0x66, 0x69, 0x6c, 0x65, 0x20, 0x27,
	0x25, 0x73, 0x27, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x3c, 0x00, 0x80, 0x00,
	0x88, 0x00, 0x00, 0x03, 0x80, 0x00, 0x00, 0x00, 0x20, 0x00, 0x04, 0x00, 0xd8,
	0x00, 0x00, 0x04, 0x48, 0x00, 0x00, 0x00, 0x20, 0x89, 0x8e, 0x00, 0xa8, 0x86,
	0x10, 0x04, 0x08, 0x21, 0x88, 0x00, 0x38, 0x50, 0x84, 0x00, 0x89, 0x49, 0x28,
	0x04, 0x08, 0x52, 0x14, 0x00, 0x20, 0x20, 0x84, 0x00, 0x89, 0x48, 0x38, 0x04,
	0x08, 0x53, 0x9c, 0x00, 0x20, 0x50, 0x84, 0x00, 0x89, 0x48, 0x20, 0x04, 0x48,
	0x50, 0x90, 0x00, 0x3c, 0x89, 0xc3, 0x00, 0x88, 0x88, 0x18, 0x03, 0x86, 0x23,
	0x0c, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x02, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x0b,
	0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x0a, 0x0b, 0x0a, 0x0b, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x0a, 0x0b, 0x0a, 0x0d, 0x0a, 0x0b, 0x0a, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x0a, 0x0b, 0x0a, 0x0d, 0x03, 0x0d, 0x0a, 0x0b, 0x0a,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x0b, 0x0a, 0x0d, 0x03, 0x04, 0x03,
	0x0d, 0x0a, 0x0b, 0x0a, 0x00, 0x00, 0x00, 0x00, 0x0a, 0x0b, 0x0a, 0x0d, 0x03,
};

static const byte polish2CD_feeble_windowFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0xfe, 0x82, 0x82, 0x82, 0x82, 0x0c, 0x08, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x04, 0x06, 0x00,
	0x04, 0x08, 0x10, 0x7c, 0x82, 0x80, 0x80, 0x80, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x3e, 0x62, 0x92, 0x64, 0x08, 0x10, 0x20, 0x4c, 0x92, 0x8c, 0x00, 0x00, 0x00,
	0x7c, 0x8a, 0x92, 0x80, 0x80, 0x80, 0x80, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x40, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x40, 0x20, 0x00, 0x00, 0x00,
	0x80, 0x40, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x40, 0x80, 0x00, 0x00, 0x00,
	0xfe, 0x80, 0x80, 0x80, 0x80, 0xfc, 0x80, 0x80, 0x80, 0xfe, 0x18, 0x0c, 0x00,
	0x80, 0x80, 0x90, 0xb0, 0xe0, 0xc0, 0x80, 0x80, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x80, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x80, 0x00, 0x00,
	0x78, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x78, 0x00, 0x00, 0x00,
	0x10, 0x30, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00, 0x00, 0x00,
	0x78, 0x84, 0x84, 0x04, 0x04, 0x08, 0x10, 0x20, 0x40, 0xfc, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x02, 0x02, 0x1c, 0x02, 0x02, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x04, 0x0c, 0x14, 0x24, 0x44, 0x84, 0xfe, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00,
	0xfe, 0x80, 0x80, 0xfc, 0x02, 0x02, 0x02, 0x02, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x3e, 0x40, 0x80, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfe, 0x02, 0x02, 0x02, 0x02, 0x04, 0x08, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x78, 0x84, 0x84, 0x84, 0x78, 0x84, 0x84, 0x84, 0x84, 0x78, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00,
	0x8a, 0x92, 0xc2, 0xa2, 0x92, 0x8a, 0x86, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x04, 0x08, 0x10, 0x9c, 0xa2, 0xc2, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x7c, 0x92, 0xa2, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x04, 0x08, 0x10, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x3c, 0x42, 0x82, 0x02, 0x04, 0x08, 0x10, 0x10, 0x00, 0x10, 0x10, 0x00, 0x00,
	0x04, 0x08, 0x10, 0x7c, 0x80, 0x80, 0x78, 0x04, 0x04, 0xf8, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0xfe, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0xfc, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x80, 0x80, 0x80, 0x80, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x00, 0x00, 0x00,
	0xfe, 0x80, 0x80, 0x80, 0x80, 0xfc, 0x80, 0x80, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0xfe, 0x80, 0x80, 0x80, 0x80, 0xfc, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x80, 0x80, 0x86, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0xfe, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0xe0, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x1e, 0x02, 0x02, 0x02, 0x02, 0x02, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x84, 0x88, 0x90, 0xe0, 0x90, 0x88, 0x84, 0x82, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0x82, 0xc6, 0xaa, 0x92, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x82, 0x82, 0xc2, 0xa2, 0x92, 0x8a, 0x86, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x80, 0x80, 0x7c, 0x02, 0x02, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfe, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x44, 0x28, 0x10, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x92, 0xaa, 0xc6, 0x00, 0x00, 0x00,
	0x82, 0x44, 0x28, 0x10, 0x10, 0x10, 0x10, 0x28, 0x44, 0x82, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x44, 0x28, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0xfe, 0x02, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xfe, 0x04, 0x08, 0x7c, 0x20, 0x40, 0xfe, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x18, 0x0c, 0x00,
	0x04, 0x08, 0x10, 0xfe, 0x04, 0x08, 0x10, 0x20, 0x40, 0xfe, 0x00, 0x00, 0x00,
	0xc0, 0x40, 0x40, 0x60, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x7c, 0x8a, 0x90, 0x80, 0x7c, 0x02, 0x02, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x18, 0xfe, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x80, 0x80, 0x80, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x02, 0x02, 0x02, 0x7e, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x1c, 0x20, 0x20, 0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x7c,
	0x80, 0x80, 0x80, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x40, 0x00, 0x00, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x00, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0xe0,
	0x80, 0x80, 0x80, 0x84, 0x88, 0x90, 0xe0, 0x90, 0x88, 0x84, 0x00, 0x00, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xfc, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x9c, 0xa2, 0xc2, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x80, 0x80, 0x80,
	0x00, 0x00, 0x00, 0x7e, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x02,
	0x00, 0x00, 0x00, 0x9c, 0xa2, 0xc2, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x80, 0x80, 0x78, 0x04, 0x04, 0xf8, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x82, 0x82, 0x82, 0x44, 0x28, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x44, 0x28, 0x10, 0x28, 0x44, 0x82, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x7c,
	0x00, 0x00, 0x00, 0xfe, 0x04, 0x08, 0x10, 0x20, 0x40, 0xfe, 0x00, 0x00, 0x00,
	0x30, 0x40, 0x40, 0x40, 0x40, 0x80, 0x40, 0x40, 0x40, 0x40, 0x30, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00,
	0xc0, 0x20, 0x20, 0x20, 0x20, 0x10, 0x20, 0x20, 0x20, 0x20, 0xc0, 0x00, 0x00,
	0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x80, 0x80, 0x80, 0x80, 0x82, 0x82, 0x7c, 0x10, 0x10, 0x30,
	0x00, 0x28, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x10, 0x20, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x10, 0x28, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x00, 0x28, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x10, 0x08, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x08, 0x14, 0x08, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x80, 0x80, 0x80, 0x82, 0x7c, 0x10, 0x10, 0x30,
	0x10, 0x28, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0x28, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x10, 0x08, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0xa0, 0x00, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x40, 0xa0, 0x00, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x80, 0x40, 0x00, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x28, 0x00, 0x7c, 0x82, 0x82, 0x82, 0xfe, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x10, 0x28, 0x7c, 0x82, 0x82, 0x82, 0xfe, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x10, 0x20, 0xfe, 0x80, 0x80, 0xfc, 0x80, 0x80, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x28, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x28, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x20, 0x10, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x10, 0x28, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x10, 0x08, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0x28, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x7c,
	0x28, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x28, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x08, 0x10, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x40, 0x80, 0x00, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x10, 0x20, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x08, 0x10, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x14, 0x28, 0x00, 0x9c, 0xa2, 0xc2, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x14, 0x28, 0x82, 0xc2, 0xa2, 0x92, 0x8a, 0x86, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x10, 0x00, 0x10, 0x10, 0x20, 0x40, 0x80, 0x82, 0x84, 0x78, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x80, 0x00, 0x00, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x3c, 0x42, 0x82, 0x82, 0x82, 0x84, 0x84, 0x82, 0x82, 0x82, 0x9c, 0x80, 0x80,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

static const byte feeble_windowFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
	0x90, 0x90, 0x90, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x48, 0xfc, 0x48, 0x48, 0xfc, 0x48, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x7c, 0x92, 0x90, 0x90, 0x7c, 0x12, 0x12, 0x12, 0x92, 0x7c, 0x10, 0x10, 0x00,
	0x3e, 0x62, 0x92, 0x64, 0x08, 0x10, 0x20, 0x4c, 0x92, 0x8c, 0x00, 0x00, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x50, 0x20, 0x52, 0x8a, 0x84, 0x7a, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x40, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x40, 0x20, 0x00, 0x00, 0x00,
	0x80, 0x40, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x40, 0x80, 0x00, 0x00, 0x00,
	0x10, 0x92, 0x54, 0x38, 0x38, 0x54, 0x92, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x10, 0x10, 0xfe, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x40, 0x80, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0x80, 0x00, 0x00,
	0x78, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x84, 0x78, 0x00, 0x00, 0x00,
	0x10, 0x30, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00, 0x00, 0x00,
	0x78, 0x84, 0x84, 0x04, 0x04, 0x08, 0x10, 0x20, 0x40, 0xfc, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x02, 0x02, 0x1c, 0x02, 0x02, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x04, 0x0c, 0x14, 0x24, 0x44, 0x84, 0xfe, 0x04, 0x04, 0x04, 0x00, 0x00, 0x00,
	0xfe, 0x80, 0x80, 0xfc, 0x02, 0x02, 0x02, 0x02, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x3e, 0x40, 0x80, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfe, 0x02, 0x02, 0x02, 0x02, 0x04, 0x08, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x78, 0x84, 0x84, 0x84, 0x78, 0x84, 0x84, 0x84, 0x84, 0x78, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x80, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x40, 0x00, 0x00, 0x00, 0x40, 0x40, 0x80, 0x00,
	0x00, 0x00, 0x10, 0x20, 0x40, 0x80, 0x40, 0x20, 0x10, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xfe, 0x00, 0x00, 0xfe, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x80, 0x40, 0x20, 0x10, 0x20, 0x40, 0x80, 0x00, 0x00, 0x00, 0x00,
	0x3c, 0x42, 0x82, 0x02, 0x04, 0x08, 0x10, 0x10, 0x00, 0x10, 0x10, 0x00, 0x00,
	0x3c, 0x42, 0x82, 0x9a, 0xa6, 0xa6, 0x9c, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0xfe, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0xfc, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x80, 0x80, 0x80, 0x80, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x00, 0x00, 0x00,
	0xfe, 0x80, 0x80, 0x80, 0x80, 0xfc, 0x80, 0x80, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0xfe, 0x80, 0x80, 0x80, 0x80, 0xfc, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x80, 0x80, 0x86, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0xfe, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0xe0, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x1e, 0x02, 0x02, 0x02, 0x02, 0x02, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x84, 0x88, 0x90, 0xe0, 0x90, 0x88, 0x84, 0x82, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0x82, 0xc6, 0xaa, 0x92, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x82, 0x82, 0xc2, 0xa2, 0x92, 0x8a, 0x86, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x00,
	0xfc, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x7c, 0x82, 0x80, 0x80, 0x7c, 0x02, 0x02, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0xfe, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x44, 0x28, 0x10, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x92, 0xaa, 0xc6, 0x00, 0x00, 0x00,
	0x82, 0x44, 0x28, 0x10, 0x10, 0x10, 0x10, 0x28, 0x44, 0x82, 0x00, 0x00, 0x00,
	0x82, 0x82, 0x82, 0x82, 0x44, 0x28, 0x10, 0x10, 0x10, 0x10, 0x00, 0x00, 0x00,
	0xfe, 0x02, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0xe0, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0xe0, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x02, 0x01, 0x00, 0x00, 0x00, 0x00,
	0xe0, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0xe0, 0x00, 0x00, 0x00,
	0xa0, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xff, 0x00, 0x00, 0x00,
	0x20, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x80, 0x80, 0x80, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x80, 0x80, 0x80, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x02, 0x02, 0x02, 0x7e, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x1c, 0x20, 0x20, 0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x7c,
	0x80, 0x80, 0x80, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x40, 0x00, 0x00, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x10, 0x00, 0x00, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0x10, 0xe0,
	0x80, 0x80, 0x80, 0x84, 0x88, 0x90, 0xe0, 0x90, 0x88, 0x84, 0x00, 0x00, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x40, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xfc, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x9c, 0xa2, 0xc2, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0xfc, 0x82, 0x82, 0x82, 0x82, 0x82, 0xfc, 0x80, 0x80, 0x80,
	0x00, 0x00, 0x00, 0x7e, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x02,
	0x00, 0x00, 0x00, 0x9c, 0xa2, 0xc2, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x80, 0x80, 0x78, 0x04, 0x04, 0xf8, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0x1c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x82, 0x82, 0x82, 0x44, 0x28, 0x10, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x92, 0x92, 0x92, 0x92, 0x92, 0x92, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x44, 0x28, 0x10, 0x28, 0x44, 0x82, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x7c,
	0x00, 0x00, 0x00, 0xfe, 0x04, 0x08, 0x10, 0x20, 0x40, 0xfe, 0x00, 0x00, 0x00,
	0x30, 0x40, 0x40, 0x40, 0x40, 0x80, 0x40, 0x40, 0x40, 0x40, 0x30, 0x00, 0x00,
	0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00,
	0xc0, 0x20, 0x20, 0x20, 0x20, 0x10, 0x20, 0x20, 0x20, 0x20, 0xc0, 0x00, 0x00,
	0x98, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xfe, 0x00, 0x00,
	0x7c, 0x82, 0x82, 0x80, 0x80, 0x80, 0x80, 0x82, 0x82, 0x7c, 0x10, 0x10, 0x30,
	0x00, 0x28, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x10, 0x20, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x10, 0x28, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x00, 0x28, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x10, 0x08, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x08, 0x14, 0x08, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x7c, 0x82, 0x80, 0x80, 0x80, 0x82, 0x7c, 0x10, 0x10, 0x30,
	0x10, 0x28, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0x28, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x10, 0x08, 0x00, 0x7c, 0x82, 0x82, 0xfe, 0x80, 0x80, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0xa0, 0x00, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x40, 0xa0, 0x00, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x80, 0x40, 0x00, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x28, 0x00, 0x7c, 0x82, 0x82, 0x82, 0xfe, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x10, 0x28, 0x7c, 0x82, 0x82, 0x82, 0xfe, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x28, 0x00, 0xfe, 0x80, 0x80, 0xfc, 0x80, 0x80, 0x80, 0xfe, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x28, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x28, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x20, 0x10, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x10, 0x28, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x10, 0x08, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x00, 0x28, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x02, 0x02, 0x7c,
	0x28, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x28, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x08, 0x10, 0x00, 0x3e, 0x42, 0x82, 0x82, 0x82, 0x86, 0x7a, 0x00, 0x00, 0x00,
	0x40, 0x80, 0x00, 0x40, 0xc0, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00, 0x00, 0x00,
	0x10, 0x20, 0x00, 0x7c, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7c, 0x00, 0x00, 0x00,
	0x08, 0x10, 0x00, 0x82, 0x82, 0x82, 0x82, 0x82, 0x82, 0x7e, 0x00, 0x00, 0x00,
	0x14, 0x28, 0x00, 0x9c, 0xa2, 0xc2, 0x82, 0x82, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x14, 0x28, 0x82, 0xc2, 0xa2, 0x92, 0x8a, 0x86, 0x82, 0x82, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x10, 0x10, 0x00, 0x10, 0x10, 0x20, 0x40, 0x80, 0x82, 0x84, 0x78, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x80, 0x00, 0x00, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x80, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x3c, 0x42, 0x82, 0x82, 0x82, 0x84, 0x84, 0x82, 0x82, 0x82, 0x9c, 0x80, 0x80,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

void AGOSEngine_Feeble::windowDrawChar(WindowBlock *window, uint x, uint y, byte chr) {
	const byte *src;
	byte color, *dst;
	uint dstPitch, h, w, i;

	if (_noOracleScroll)
		return;

	_videoLockOut |= 0x8000;

	dst = getBackGround();
	dstPitch = _backGroundBuf->pitch;
	h = 13;
	w = getFeebleFontSize(chr);

	if (_language == Common::PL_POL) {
		if (!strcmp(getExtra(), "4CD"))
			src = polish4CD_feeble_windowFont + (chr - 32) * 13;
		else
			src = polish2CD_feeble_windowFont + (chr - 32) * 13;
	} else {
		src = feeble_windowFont + (chr - 32) * 13;
	}
	dst += y * dstPitch + x + window->textColumnOffset;

	color = window->textColor;

	do {
		int8 b = *src++;
		i = 0;
		do {
			if (b < 0) {
				if (dst[i] == 0)
					dst[i] = color;
			}

			b <<= 1;
		} while (++i != w);
		dst += dstPitch;
	} while (--h);

	_videoLockOut &= ~0x8000;
}
#endif

static const byte english_simon1AGAFontData[] = {
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x20,0x10,0x40,0x88,0x30,0x40,0x00,0x88,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x10,0x00,0x00,0x10,0x28,0x08,0x10,0x20,0x44,0x00,0x40,0x00,0xB8,0x30,0x00,0x00,0x48,0x08,0x00,0x00,0x74,0x30,0x00,0x40,0x88,0x20,0x00,0x00,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xB4,0x00,0x48,0x00,0xB4,0x10,0x00,0x00,0x68,0x00,0x20,0x00,0x58,0x00,0x08,0x40,0xB4,0x08,0x40,0x00,0xB4,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x10,0x00,0x40,0xA8,0x18,0x20,0x00,0x44,0x10,0x00,0x00,0x28,0x18,0x20,0x00,0x44,0x10,0x00,0x40,0xAC,0x0C,0x20,0x00,0x52,0x00,0x00,0x00,0x2C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1C,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x18,0x20,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xBA,0x0C,0x00,0x40,0xB2,0x04,0x50,0x00,0xAA,0x44,0x20,0x00,0x9A,0x44,0x00,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xBA,0x04,0x00,0x00,0x5A,0x08,0x10,0x00,0x24,0x00,0x20,0x00,0x5C,0x00,0x04,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x00,0x00,0x5A,0x00,0x08,0x10,0x24,0x04,0x00,0x00,0x5A,0x04,0x00,0x40,0xBA,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x08,0x00,0x08,0x00,0x14,0x08,0x00,0x10,0x24,0x08,0x00,0x20,0x54,0x08,0x40,0x00,0xB4,0x3C,0x40,0x00,0x82,0x08,0x00,0x00,0x74,0x04,0x08,0x10,0x22,0x00,0x00,0x00,0x1C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x18,0x60,0x82,0x00,0x40,0x00,0xBC,0x78,0x00,0x00,0x84,0x04,0x00,0x00,0x7A,0x04,0x00,0x00,0x4A,0x04,0x00,0x40,0xBA,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x08,0x10,0x00,0x24,0x00,0x00,0x20,0x58,0x00,0x40,0x00,0xB8,0x38,0x40,0x00,0x84,0x44,0x00,0x00,0xBA,0x44,0x00,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x00,0x04,0x00,0x4A,0x00,0x00,0x08,0x14,0x00,0x10,0x00,0x28,0x00,0x10,0x00,0x28,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x40,0x00,0xBA,0x18,0x20,0x00,0x44,0x04,0x00,0x40,0xBA,0x04,0x40,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x40,0x00,0xBA,0x1C,0x20,0x00,0x42,0x04,0x00,0x00,0x3A,0x00,0x08,0x00,0x34,0x00,0x10,0x20,0x48,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x05,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x40,0x00,0xBA,0x04,0x00,0x00,0x4A,0x08,0x00,0x00,0x14,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x40,0x00,0xBA,0x04,0x00,0x00,0x4A,0x08,0x00,0x00,0x14,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x40,0x00,0xBA,0x04,0x00,0x00,0x4A,0x08,0x00,0x00,0x14,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x40,0x00,0xBA,0x04,0x00,0x00,0x4A,0x08,0x00,0x00,0x14,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xBA,0x04,0x40,0x10,0xAA,0x0C,0x50,0x00,0xA2,0x1C,0x40,0x00,0xA2,0x40,0x00,0x00,0xBC,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x00,0x40,0xBA,0x0C,0x70,0x00,0x82,0x04,0x40,0x00,0xBA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x78,0x08,0x10,0x60,0x84,0x04,0x00,0x20,0x5A,0x04,0x20,0x00,0x5A,0x28,0x10,0x00,0x44,0x24,0x00,0x00,0x5A,0x24,0x00,0x00,0x5A,0x18,0x20,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x24,0x04,0x00,0x20,0x5A,0x00,0x00,0x40,0xA4,0x00,0x00,0x40,0xA0,0x00,0x40,0x00,0xA4,0x24,0x00,0x00,0x5A,0x18,0x00,0x00,0x24,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x70,0x10,0x20,0x40,0x88,0x08,0x20,0x00,0x54,0x04,0x20,0x00,0x5A,0x04,0x20,0x00,0x5A,0x24,0x00,0x00,0x5A,0x28,0x00,0x00,0x54,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x18,0x60,0x82,0x04,0x20,0x00,0x5A,0x00,0x20,0x00,0x5C,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x5C,0x20,0x04,0x00,0x5A,0x0C,0x30,0x40,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x18,0x60,0x82,0x04,0x20,0x00,0x5A,0x00,0x20,0x00,0x5C,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x24,0x04,0x00,0x20,0x5A,0x00,0x40,0x00,0xAC,0x00,0x44,0x08,0xB2,0x44,0x00,0x00,0xAA,0x24,0x00,0x00,0x5A,0x1C,0x00,0x00,0x22,0x00,0x00,0x00,0x1C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x44,0x00,0xAA,0x04,0x40,0x00,0xBA,0x4C,0x30,0x00,0x82,0x44,0x00,0x00,0xBA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x00,0x10,0x00,0x6C,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x6C,0x1C,0x20,0x40,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x0C,0x00,0x04,0x08,0x12,0x00,0x04,0x00,0x0A,0x04,0x00,0x00,0x0A,0x04,0x00,0x00,0x4A,0x04,0x00,0x40,0xAA,0x04,0x40,0x00,0xBA,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x64,0x00,0x24,0x40,0x9A,0x00,0x24,0x00,0x5A,0x20,0x08,0x00,0x54,0x20,0x10,0x00,0x48,0x28,0x00,0x00,0x54,0x24,0x00,0x00,0x5A,0x04,0x20,0x40,0x9A,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x70,0x00,0x30,0x40,0x88,0x00,0x20,0x00,0x50,0x00,0x20,0x00,0x50,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x54,0x24,0x00,0x00,0x5A,0x1C,0x20,0x40,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x04,0x40,0xAA,0x24,0x40,0x08,0x92,0x04,0x50,0x00,0xAA,0x14,0x40,0x00,0xAA,0x44,0x00,0x00,0xBA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x20,0x44,0x00,0x9A,0x14,0x40,0x00,0xAA,0x0C,0x40,0x00,0xB2,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x00,0x40,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x78,0x08,0x30,0x40,0x84,0x04,0x20,0x00,0x5A,0x04,0x20,0x00,0x5A,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x00,0x40,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xBA,0x44,0x00,0x10,0xAA,0x30,0x08,0x00,0x44,0x04,0x00,0x00,0x3A,0x00,0x00,0x00,0x04,0x07,
	0x00,0x00,0x00,0x78,0x00,0x18,0x60,0x84,0x04,0x20,0x00,0x5A,0x24,0x00,0x00,0x5A,0x28,0x10,0x00,0x44,0x24,0x00,0x00,0x5A,0x24,0x00,0x00,0x5A,0x04,0x20,0x40,0x9A,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xBA,0x00,0x40,0x00,0xBC,0x08,0x30,0x00,0x44,0x04,0x00,0x00,0x7A,0x04,0x00,0x40,0xBA,0x08,0x30,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x18,0x60,0x82,0x14,0x40,0x00,0xAA,0x10,0x00,0x00,0x6C,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x08,0x10,0x20,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x44,0x00,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xAA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xBA,0x3C,0x00,0x00,0x42,0x00,0x00,0x00,0x3C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x04,0x40,0xAA,0x00,0x44,0x00,0xAA,0x04,0x40,0x00,0xAA,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x04,0x40,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xBA,0x54,0x00,0x00,0xAA,0x6C,0x00,0x00,0x92,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x04,0x40,0x00,0xAA,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x08,0x20,0x00,0x54,0x04,0x00,0x40,0xAA,0x04,0x40,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x04,0x00,0x40,0xAA,0x04,0x40,0x00,0xAA,0x04,0x40,0x00,0xBA,0x38,0x00,0x00,0x44,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x08,0x10,0x20,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xBA,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x28,0x10,0x44,0x04,0x00,0x00,0x3A,0x04,0x08,0x30,0x42,0x04,0x40,0x00,0xBA,0x3C,0x00,0x00,0x42,0x00,0x00,0x00,0x3C,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x58,0x00,0x28,0x00,0x54,0x04,0x20,0x10,0x4A,0x04,0x20,0x00,0x5A,0x04,0x20,0x00,0x5A,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xBA,0x00,0x40,0x00,0xA4,0x04,0x40,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x0C,0x00,0x04,0x08,0x12,0x08,0x00,0x00,0x34,0x08,0x20,0x00,0x54,0x18,0x40,0x00,0xA4,0x08,0x40,0x00,0xB4,0x08,0x40,0x00,0xB4,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xBA,0x3C,0x40,0x00,0x82,0x00,0x40,0x00,0xBC,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x08,0x10,0x24,0x04,0x00,0x20,0x5A,0x00,0x20,0x00,0x54,0x10,0x20,0x40,0x88,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x30,0x40,0x00,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x04,0x10,0x20,0x4A,0x08,0x00,0x40,0xB4,0x08,0x40,0x00,0xB4,0x38,0x00,0x00,0x44,0x04,0x00,0x40,0xBA,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x58,0x20,0x08,0x00,0x54,0x24,0x00,0x10,0x4A,0x24,0x00,0x00,0x5A,0x24,0x00,0x00,0x5A,0x04,0x20,0x40,0x9A,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x60,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x04,0x00,0x00,0x04,0x0A,0x00,0x00,0x00,0x04,0x00,0x00,0x04,0x0A,0x00,0x04,0x00,0x0A,0x04,0x00,0x00,0x0A,0x04,0x00,0x00,0x4A,0x04,0x00,0x40,0xBA,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x07,
	0x00,0x00,0x00,0x60,0x00,0x00,0x60,0x90,0x00,0x20,0x00,0x54,0x20,0x04,0x00,0x5A,0x20,0x00,0x08,0x54,0x20,0x10,0x00,0x48,0x28,0x00,0x00,0x54,0x04,0x20,0x40,0x9A,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xB4,0x24,0x48,0x00,0x92,0x44,0x10,0x00,0xAA,0x44,0x00,0x00,0xBA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x08,0x10,0x60,0x84,0x04,0x00,0x40,0xBA,0x04,0x40,0x00,0xAA,0x44,0x00,0x00,0xAA,0x44,0x00,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xBA,0x04,0x40,0x00,0xAA,0x44,0x00,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x00,0x08,0x50,0xA4,0x04,0x20,0x00,0x5A,0x24,0x00,0x00,0x5A,0x38,0x00,0x00,0x44,0x20,0x00,0x00,0x58,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x00,0x14,0x20,0x4A,0x08,0x40,0x00,0xB4,0x48,0x00,0x00,0xB4,0x38,0x00,0x00,0x44,0x08,0x00,0x00,0x34,0x04,0x08,0x10,0x22,0x00,0x00,0x00,0x1C,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x00,0x08,0x50,0xA4,0x04,0x20,0x00,0x5A,0x24,0x00,0x00,0x5A,0x20,0x00,0x00,0x54,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3C,0x04,0x18,0x20,0x42,0x00,0x40,0x00,0xBC,0x38,0x00,0x00,0x44,0x04,0x00,0x00,0x7A,0x38,0x00,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x20,0x00,0x58,0x08,0x30,0x40,0x84,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x54,0x04,0x20,0x00,0x5A,0x08,0x10,0x00,0x24,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xB4,0x00,0x48,0x00,0xB4,0x08,0x40,0x00,0xB4,0x48,0x00,0x00,0xB4,0x34,0x00,0x00,0x4A,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x44,0x00,0xAA,0x04,0x40,0x00,0xAA,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x44,0x00,0xBA,0x14,0x40,0x00,0xAA,0x2C,0x40,0x00,0x92,0x48,0x00,0x00,0xB4,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x28,0x00,0x54,0x10,0x00,0x00,0x28,0x08,0x20,0x00,0x54,0x04,0x40,0x00,0xAA,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xAA,0x00,0x44,0x00,0xAA,0x04,0x40,0x00,0xBA,0x38,0x00,0x00,0x44,0x00,0x10,0x00,0x68,0x00,0x20,0x40,0x90,0x00,0x00,0x00,0x60,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7C,0x04,0x38,0x40,0x82,0x08,0x40,0x00,0xB4,0x00,0x10,0x00,0x6C,0x04,0x20,0x00,0x5A,0x3C,0x00,0x40,0x82,0x00,0x00,0x00,0x7C,0x00,0x00,0x00,0x00,0x07
};

static const byte german_simon1AGAFontData[] = {
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x20,0x10,0x40,0x88,0x30,0x40,0x00,0x88,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x00,0x00,0x38,0x00,0x28,0x10,0x44,0x04,0x00,0x00,0x3a,0x04,0x08,0x30,0x42,0x04,0x40,0x00,0xba,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x50,0x00,0x00,0x50,0xa8,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x34,0x00,0x00,0x4a,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x10,0x00,0x40,0xa8,0x18,0x20,0x00,0x44,0x10,0x00,0x00,0x28,0x18,0x20,0x00,0x44,0x10,0x00,0x40,0xac,0x0c,0x20,0x00,0x52,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x30,0x10,0x20,0x00,0x48,0x08,0x00,0x40,0xb4,0x08,0x40,0x00,0xb4,0x50,0x00,0x00,0xa8,0x48,0x00,0x00,0xb4,0x48,0x00,0x00,0xb4,0x50,0x00,0x00,0xa8,0x40,0x00,0x00,0xa0,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x18,0x20,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x0c,0x70,0x00,0x82,0x04,0x40,0x00,0xba,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x0c,0x00,0x40,0xb2,0x04,0x50,0x00,0xaa,0x44,0x20,0x00,0x9a,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x00,0x5a,0x08,0x10,0x00,0x24,0x00,0x20,0x00,0x5c,0x00,0x04,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x00,0x5a,0x00,0x08,0x10,0x24,0x04,0x00,0x00,0x5a,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x08,0x00,0x08,0x00,0x14,0x08,0x00,0x10,0x24,0x08,0x00,0x20,0x54,0x08,0x40,0x00,0xb4,0x3c,0x40,0x00,0x82,0x08,0x00,0x00,0x74,0x04,0x08,0x10,0x22,0x00,0x00,0x00,0x1c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x00,0x40,0x00,0xbc,0x78,0x00,0x00,0x84,0x04,0x00,0x00,0x7a,0x04,0x00,0x00,0x4a,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x08,0x10,0x00,0x24,0x00,0x00,0x20,0x58,0x00,0x40,0x00,0xb8,0x38,0x40,0x00,0x84,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x00,0x04,0x00,0x4a,0x00,0x00,0x08,0x14,0x00,0x10,0x00,0x28,0x00,0x10,0x00,0x28,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xba,0x18,0x20,0x00,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xba,0x1c,0x20,0x00,0x42,0x04,0x00,0x00,0x3a,0x00,0x08,0x00,0x34,0x00,0x10,0x20,0x48,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x00,0x00,0x6c,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xaa,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x50,0x00,0x00,0x50,0xa8,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x34,0x00,0x00,0x4a,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x10,0x28,0x00,0x08,0x20,0x54,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x40,0x00,0xba,0x04,0x00,0x00,0x4a,0x08,0x00,0x00,0x14,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x20,0x50,0x00,0x10,0x40,0xa8,0x00,0x00,0x00,0x60,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x40,0xba,0x0c,0x70,0x00,0x82,0x04,0x40,0x00,0xba,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x78,0x08,0x10,0x60,0x84,0x04,0x00,0x20,0x5a,0x04,0x20,0x00,0x5a,0x28,0x10,0x00,0x44,0x24,0x00,0x00,0x5a,0x24,0x00,0x00,0x5a,0x18,0x20,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x24,0x04,0x00,0x20,0x5a,0x00,0x00,0x40,0xa4,0x00,0x00,0x40,0xa0,0x00,0x40,0x00,0xa4,0x24,0x00,0x00,0x5a,0x18,0x00,0x00,0x24,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x70,0x10,0x20,0x40,0x88,0x08,0x20,0x00,0x54,0x04,0x20,0x00,0x5a,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x28,0x00,0x00,0x54,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x04,0x20,0x00,0x5a,0x00,0x20,0x00,0x5c,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x5c,0x20,0x04,0x00,0x5a,0x0c,0x30,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x04,0x20,0x00,0x5a,0x00,0x20,0x00,0x5c,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x24,0x04,0x00,0x20,0x5a,0x00,0x40,0x00,0xac,0x00,0x44,0x08,0xb2,0x44,0x00,0x00,0xaa,0x24,0x00,0x00,0x5a,0x1c,0x00,0x00,0x22,0x00,0x00,0x00,0x1c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xba,0x4c,0x30,0x00,0x82,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x00,0x10,0x00,0x6c,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x6c,0x1c,0x20,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x0c,0x00,0x04,0x08,0x12,0x00,0x04,0x00,0x0a,0x04,0x00,0x00,0x0a,0x04,0x00,0x00,0x4a,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x64,0x00,0x24,0x40,0x9a,0x00,0x24,0x00,0x5a,0x20,0x08,0x00,0x54,0x20,0x10,0x00,0x48,0x28,0x00,0x00,0x54,0x24,0x00,0x00,0x5a,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x70,0x00,0x30,0x40,0x88,0x00,0x20,0x00,0x50,0x00,0x20,0x00,0x50,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x54,0x24,0x00,0x00,0x5a,0x1c,0x20,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x04,0x40,0xaa,0x24,0x40,0x08,0x92,0x04,0x50,0x00,0xaa,0x14,0x40,0x00,0xaa,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x20,0x44,0x00,0x9a,0x14,0x40,0x00,0xaa,0x0c,0x40,0x00,0xb2,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x78,0x08,0x30,0x40,0x84,0x04,0x20,0x00,0x5a,0x04,0x20,0x00,0x5a,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x44,0x00,0x10,0xaa,0x30,0x08,0x00,0x44,0x04,0x00,0x00,0x3a,0x00,0x00,0x00,0x04,0x07,
	0x00,0x00,0x00,0x78,0x00,0x18,0x60,0x84,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x28,0x10,0x00,0x44,0x24,0x00,0x00,0x5a,0x24,0x00,0x00,0x5a,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x00,0x40,0x00,0xbc,0x08,0x30,0x00,0x44,0x04,0x00,0x00,0x7a,0x04,0x00,0x40,0xba,0x08,0x30,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x14,0x40,0x00,0xaa,0x10,0x00,0x00,0x6c,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x08,0x10,0x20,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xba,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x04,0x40,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xaa,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x04,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x54,0x00,0x00,0xaa,0x6c,0x00,0x00,0x92,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x04,0x40,0x00,0xaa,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x08,0x20,0x00,0x54,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x08,0x10,0x20,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x10,0x00,0x00,0x68,0x00,0x20,0x00,0x58,0x00,0x08,0x40,0xb4,0x08,0x40,0x00,0xb4,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xa0,0x00,0x40,0x00,0xb8,0x38,0x00,0x00,0x44,0x10,0x00,0x00,0x28,0x00,0x10,0x20,0x48,0x00,0x00,0x00,0x30,0x06,
	0x00,0x00,0x00,0x50,0x00,0x00,0x50,0xa8,0x00,0x00,0x00,0x50,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x20,0x50,0x00,0x10,0x00,0x28,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x34,0x00,0x00,0x4a,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x28,0x10,0x44,0x04,0x00,0x00,0x3a,0x04,0x08,0x30,0x42,0x04,0x40,0x00,0xba,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x58,0x00,0x28,0x00,0x54,0x04,0x20,0x10,0x4a,0x04,0x20,0x00,0x5a,0x04,0x20,0x00,0x5a,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x00,0x40,0x00,0xa4,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x0c,0x00,0x04,0x08,0x12,0x08,0x00,0x00,0x34,0x08,0x20,0x00,0x54,0x18,0x40,0x00,0xa4,0x08,0x40,0x00,0xb4,0x08,0x40,0x00,0xb4,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x08,0x10,0x24,0x04,0x00,0x20,0x5a,0x00,0x20,0x00,0x54,0x10,0x20,0x40,0x88,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x30,0x40,0x00,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x04,0x10,0x20,0x4a,0x08,0x00,0x40,0xb4,0x08,0x40,0x00,0xb4,0x38,0x00,0x00,0x44,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x58,0x20,0x08,0x00,0x54,0x24,0x00,0x10,0x4a,0x24,0x00,0x00,0x5a,0x24,0x00,0x00,0x5a,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x60,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x04,0x00,0x00,0x04,0x0a,0x00,0x00,0x00,0x04,0x00,0x00,0x04,0x0a,0x00,0x04,0x00,0x0a,0x04,0x00,0x00,0x0a,0x04,0x00,0x00,0x4a,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x07,
	0x00,0x00,0x00,0x60,0x00,0x00,0x60,0x90,0x00,0x20,0x00,0x54,0x20,0x04,0x00,0x5a,0x20,0x00,0x08,0x54,0x20,0x10,0x00,0x48,0x28,0x00,0x00,0x54,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x24,0x48,0x00,0x92,0x44,0x10,0x00,0xaa,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x08,0x10,0x60,0x84,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x00,0x08,0x50,0xa4,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x38,0x00,0x00,0x44,0x20,0x00,0x00,0x58,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x00,0x14,0x20,0x4a,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x38,0x00,0x00,0x44,0x08,0x00,0x00,0x34,0x04,0x08,0x10,0x22,0x00,0x00,0x00,0x1c,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x00,0x08,0x50,0xa4,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x20,0x00,0x00,0x54,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3c,0x04,0x18,0x20,0x42,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x04,0x00,0x00,0x7a,0x38,0x00,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x20,0x00,0x58,0x08,0x30,0x40,0x84,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x54,0x04,0x20,0x00,0x5a,0x08,0x10,0x00,0x24,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x34,0x00,0x00,0x4a,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xaa,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xba,0x14,0x40,0x00,0xaa,0x2c,0x40,0x00,0x92,0x48,0x00,0x00,0xb4,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x28,0x00,0x54,0x10,0x00,0x00,0x28,0x08,0x20,0x00,0x54,0x04,0x40,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x10,0x00,0x68,0x00,0x20,0x40,0x90,0x00,0x00,0x00,0x60,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x08,0x40,0x00,0xb4,0x00,0x10,0x00,0x6c,0x04,0x20,0x00,0x5a,0x3c,0x00,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07
};

static const byte french_simon1AGAFontData[] = {
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x20,0x10,0x40,0x88,0x30,0x40,0x00,0x88,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x10,0x28,0x00,0x08,0x20,0x54,0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x10,0x28,0x00,0x08,0x20,0x54,0x00,0x00,0x00,0x38,0x00,0x28,0x10,0x44,0x04,0x00,0x00,0x3a,0x04,0x08,0x30,0x42,0x04,0x40,0x00,0xba,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x00,0x40,0x00,0xa4,0x00,0x40,0x00,0xa0,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x10,0x00,0x00,0x28,0x00,0x10,0x20,0x48,0x00,0x00,0x00,0x30,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x10,0x00,0x40,0xa8,0x18,0x20,0x00,0x44,0x10,0x00,0x00,0x28,0x18,0x20,0x00,0x44,0x10,0x00,0x40,0xac,0x0c,0x20,0x00,0x52,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x00,0x40,0x00,0xa4,0x00,0x40,0x00,0xa0,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x10,0x00,0x00,0x28,0x00,0x10,0x20,0x48,0x00,0x00,0x00,0x30,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x18,0x20,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x10,0x28,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x0c,0x00,0x40,0xb2,0x04,0x50,0x00,0xaa,0x44,0x20,0x00,0x9a,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x00,0x5a,0x08,0x10,0x00,0x24,0x00,0x20,0x00,0x5c,0x00,0x04,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x00,0x5a,0x00,0x08,0x10,0x24,0x04,0x00,0x00,0x5a,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x08,0x00,0x08,0x00,0x14,0x08,0x00,0x10,0x24,0x08,0x00,0x20,0x54,0x08,0x40,0x00,0xb4,0x3c,0x40,0x00,0x82,0x08,0x00,0x00,0x74,0x04,0x08,0x10,0x22,0x00,0x00,0x00,0x1c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x00,0x40,0x00,0xbc,0x78,0x00,0x00,0x84,0x04,0x00,0x00,0x7a,0x04,0x00,0x00,0x4a,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x08,0x10,0x00,0x24,0x00,0x00,0x20,0x58,0x00,0x40,0x00,0xb8,0x38,0x40,0x00,0x84,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x00,0x04,0x00,0x4a,0x00,0x00,0x08,0x14,0x00,0x10,0x00,0x28,0x00,0x10,0x00,0x28,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xba,0x18,0x20,0x00,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xba,0x1c,0x20,0x00,0x42,0x04,0x00,0x00,0x3a,0x00,0x08,0x00,0x34,0x00,0x10,0x20,0x48,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x10,0x28,0x00,0x08,0x00,0x14,0x00,0x00,0x00,0x38,0x00,0x28,0x10,0x44,0x04,0x00,0x00,0x3a,0x04,0x08,0x30,0x42,0x04,0x40,0x00,0xba,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x20,0x50,0x00,0x10,0x40,0xa8,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x34,0x00,0x00,0x4a,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x10,0x28,0x00,0x08,0x00,0x14,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x10,0x28,0x00,0x08,0x20,0x54,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x40,0x00,0xba,0x04,0x00,0x00,0x4a,0x08,0x00,0x00,0x14,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x20,0x50,0x00,0x10,0x40,0xa8,0x00,0x00,0x00,0x60,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x40,0xba,0x0c,0x70,0x00,0x82,0x04,0x40,0x00,0xba,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x78,0x08,0x10,0x60,0x84,0x04,0x00,0x20,0x5a,0x04,0x20,0x00,0x5a,0x28,0x10,0x00,0x44,0x24,0x00,0x00,0x5a,0x24,0x00,0x00,0x5a,0x18,0x20,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x24,0x04,0x00,0x20,0x5a,0x00,0x00,0x40,0xa4,0x00,0x00,0x40,0xa0,0x00,0x40,0x00,0xa4,0x24,0x00,0x00,0x5a,0x18,0x00,0x00,0x24,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x70,0x10,0x20,0x40,0x88,0x08,0x20,0x00,0x54,0x04,0x20,0x00,0x5a,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x28,0x00,0x00,0x54,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x04,0x20,0x00,0x5a,0x00,0x20,0x00,0x5c,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x5c,0x20,0x04,0x00,0x5a,0x0c,0x30,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x04,0x20,0x00,0x5a,0x00,0x20,0x00,0x5c,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x24,0x04,0x00,0x20,0x5a,0x00,0x40,0x00,0xac,0x00,0x44,0x08,0xb2,0x44,0x00,0x00,0xaa,0x24,0x00,0x00,0x5a,0x1c,0x00,0x00,0x22,0x00,0x00,0x00,0x1c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xba,0x4c,0x30,0x00,0x82,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x00,0x10,0x00,0x6c,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x6c,0x1c,0x20,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x0c,0x00,0x04,0x08,0x12,0x00,0x04,0x00,0x0a,0x04,0x00,0x00,0x0a,0x04,0x00,0x00,0x4a,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x64,0x00,0x24,0x40,0x9a,0x00,0x24,0x00,0x5a,0x20,0x08,0x00,0x54,0x20,0x10,0x00,0x48,0x28,0x00,0x00,0x54,0x24,0x00,0x00,0x5a,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x70,0x00,0x30,0x40,0x88,0x00,0x20,0x00,0x50,0x00,0x20,0x00,0x50,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x54,0x24,0x00,0x00,0x5a,0x1c,0x20,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x04,0x40,0xaa,0x24,0x40,0x08,0x92,0x04,0x50,0x00,0xaa,0x14,0x40,0x00,0xaa,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x20,0x44,0x00,0x9a,0x14,0x40,0x00,0xaa,0x0c,0x40,0x00,0xb2,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x78,0x08,0x30,0x40,0x84,0x04,0x20,0x00,0x5a,0x04,0x20,0x00,0x5a,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x44,0x00,0x10,0xaa,0x30,0x08,0x00,0x44,0x04,0x00,0x00,0x3a,0x00,0x00,0x00,0x04,0x07,
	0x00,0x00,0x00,0x78,0x00,0x18,0x60,0x84,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x28,0x10,0x00,0x44,0x24,0x00,0x00,0x5a,0x24,0x00,0x00,0x5a,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x00,0x40,0x00,0xbc,0x08,0x30,0x00,0x44,0x04,0x00,0x00,0x7a,0x04,0x00,0x40,0xba,0x08,0x30,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x14,0x40,0x00,0xaa,0x10,0x00,0x00,0x6c,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x08,0x10,0x20,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xba,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x04,0x40,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xaa,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x04,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x54,0x00,0x00,0xaa,0x6c,0x00,0x00,0x92,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x04,0x40,0x00,0xaa,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x08,0x20,0x00,0x54,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x08,0x10,0x20,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x10,0x00,0x00,0x68,0x00,0x20,0x00,0x58,0x00,0x08,0x40,0xb4,0x08,0x40,0x00,0xb4,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xa0,0x00,0x40,0x00,0xb8,0x38,0x00,0x00,0x44,0x10,0x00,0x00,0x28,0x00,0x10,0x20,0x48,0x00,0x00,0x00,0x30,0x06,
	0x00,0x00,0x00,0x50,0x00,0x00,0x50,0xa8,0x00,0x00,0x00,0x50,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x20,0x50,0x00,0x10,0x00,0x28,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x34,0x00,0x00,0x4a,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x28,0x10,0x44,0x04,0x00,0x00,0x3a,0x04,0x08,0x30,0x42,0x04,0x40,0x00,0xba,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x58,0x00,0x28,0x00,0x54,0x04,0x20,0x10,0x4a,0x04,0x20,0x00,0x5a,0x04,0x20,0x00,0x5a,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x00,0x40,0x00,0xa4,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x0c,0x00,0x04,0x08,0x12,0x08,0x00,0x00,0x34,0x08,0x20,0x00,0x54,0x18,0x40,0x00,0xa4,0x08,0x40,0x00,0xb4,0x08,0x40,0x00,0xb4,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x08,0x10,0x24,0x04,0x00,0x20,0x5a,0x00,0x20,0x00,0x54,0x10,0x20,0x40,0x88,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x30,0x40,0x00,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x04,0x10,0x20,0x4a,0x08,0x00,0x40,0xb4,0x08,0x40,0x00,0xb4,0x38,0x00,0x00,0x44,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x58,0x20,0x08,0x00,0x54,0x24,0x00,0x10,0x4a,0x24,0x00,0x00,0x5a,0x24,0x00,0x00,0x5a,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x60,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x04,0x00,0x00,0x04,0x0a,0x00,0x00,0x00,0x04,0x00,0x00,0x04,0x0a,0x00,0x04,0x00,0x0a,0x04,0x00,0x00,0x0a,0x04,0x00,0x00,0x4a,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x07,
	0x00,0x00,0x00,0x60,0x00,0x00,0x60,0x90,0x00,0x20,0x00,0x54,0x20,0x04,0x00,0x5a,0x20,0x00,0x08,0x54,0x20,0x10,0x00,0x48,0x28,0x00,0x00,0x54,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x24,0x48,0x00,0x92,0x44,0x10,0x00,0xaa,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x08,0x10,0x60,0x84,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x00,0x08,0x50,0xa4,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x38,0x00,0x00,0x44,0x20,0x00,0x00,0x58,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x00,0x14,0x20,0x4a,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x38,0x00,0x00,0x44,0x08,0x00,0x00,0x34,0x04,0x08,0x10,0x22,0x00,0x00,0x00,0x1c,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x00,0x08,0x50,0xa4,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x20,0x00,0x00,0x54,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3c,0x04,0x18,0x20,0x42,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x04,0x00,0x00,0x7a,0x38,0x00,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x20,0x00,0x58,0x08,0x30,0x40,0x84,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x54,0x04,0x20,0x00,0x5a,0x08,0x10,0x00,0x24,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x34,0x00,0x00,0x4a,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xaa,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xba,0x14,0x40,0x00,0xaa,0x2c,0x40,0x00,0x92,0x48,0x00,0x00,0xb4,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x28,0x00,0x54,0x10,0x00,0x00,0x28,0x08,0x20,0x00,0x54,0x04,0x40,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x10,0x00,0x68,0x00,0x20,0x40,0x90,0x00,0x00,0x00,0x60,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x08,0x40,0x00,0xb4,0x00,0x10,0x00,0x6c,0x04,0x20,0x00,0x5a,0x3c,0x00,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07
};

static const byte italian_simon1AGAFontData[] = {
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x20,0x10,0x40,0x88,0x30,0x40,0x00,0x88,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x20,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x00,0x00,0x38,0x00,0x28,0x10,0x44,0x04,0x00,0x00,0x3a,0x04,0x08,0x30,0x42,0x04,0x40,0x00,0xba,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x10,0x28,0x00,0x08,0x00,0x14,0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x10,0x00,0x40,0xa8,0x18,0x20,0x00,0x44,0x10,0x00,0x00,0x28,0x18,0x20,0x00,0x44,0x10,0x00,0x40,0xac,0x0c,0x20,0x00,0x52,0x00,0x00,0x00,0x2c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x28,0x00,0x00,0x28,0x54,0x00,0x28,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x1c,0x20,0x40,0x82,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x00,0x00,0x00,0x28,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x20,0x50,0x00,0x10,0x00,0x28,0x00,0x00,0x00,0x60,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x30,0x00,0x10,0x20,0x48,0x10,0x20,0x00,0x48,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x18,0x20,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x10,0x28,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x0c,0x00,0x40,0xb2,0x04,0x50,0x00,0xaa,0x44,0x20,0x00,0x9a,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x00,0x5a,0x08,0x10,0x00,0x24,0x00,0x20,0x00,0x5c,0x00,0x04,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x00,0x5a,0x00,0x08,0x10,0x24,0x04,0x00,0x00,0x5a,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x08,0x00,0x08,0x00,0x14,0x08,0x00,0x10,0x24,0x08,0x00,0x20,0x54,0x08,0x40,0x00,0xb4,0x3c,0x40,0x00,0x82,0x08,0x00,0x00,0x74,0x04,0x08,0x10,0x22,0x00,0x00,0x00,0x1c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x00,0x40,0x00,0xbc,0x78,0x00,0x00,0x84,0x04,0x00,0x00,0x7a,0x04,0x00,0x00,0x4a,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x08,0x10,0x00,0x24,0x00,0x00,0x20,0x58,0x00,0x40,0x00,0xb8,0x38,0x40,0x00,0x84,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x00,0x04,0x00,0x4a,0x00,0x00,0x08,0x14,0x00,0x10,0x00,0x28,0x00,0x10,0x00,0x28,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xba,0x18,0x20,0x00,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xba,0x1c,0x20,0x00,0x42,0x04,0x00,0x00,0x3a,0x00,0x08,0x00,0x34,0x00,0x10,0x20,0x48,0x00,0x00,0x00,0x30,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x20,0x40,0x00,0x90,0x00,0x00,0x00,0x60,0x00,0x00,0x00,0x00,0x04,
	0x00,0x00,0x10,0x28,0x00,0x08,0x00,0x14,0x00,0x00,0x00,0x38,0x00,0x28,0x10,0x44,0x04,0x00,0x00,0x3a,0x04,0x08,0x30,0x42,0x04,0x40,0x00,0xba,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x10,0x28,0x00,0x08,0x00,0x14,0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x10,0x28,0x00,0x08,0x00,0x14,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x10,0x28,0x00,0x08,0x20,0x54,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x40,0x00,0xba,0x04,0x00,0x00,0x4a,0x08,0x00,0x00,0x14,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x20,0x50,0x00,0x10,0x40,0xa8,0x00,0x00,0x00,0x60,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x40,0xba,0x0c,0x70,0x00,0x82,0x04,0x40,0x00,0xba,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x78,0x08,0x10,0x60,0x84,0x04,0x00,0x20,0x5a,0x04,0x20,0x00,0x5a,0x28,0x10,0x00,0x44,0x24,0x00,0x00,0x5a,0x24,0x00,0x00,0x5a,0x18,0x20,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x24,0x04,0x00,0x20,0x5a,0x00,0x00,0x40,0xa4,0x00,0x00,0x40,0xa0,0x00,0x40,0x00,0xa4,0x24,0x00,0x00,0x5a,0x18,0x00,0x00,0x24,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x70,0x10,0x20,0x40,0x88,0x08,0x20,0x00,0x54,0x04,0x20,0x00,0x5a,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x28,0x00,0x00,0x54,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x04,0x20,0x00,0x5a,0x00,0x20,0x00,0x5c,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x5c,0x20,0x04,0x00,0x5a,0x0c,0x30,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x04,0x20,0x00,0x5a,0x00,0x20,0x00,0x5c,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x18,0x00,0x24,0x04,0x00,0x20,0x5a,0x00,0x40,0x00,0xac,0x00,0x44,0x08,0xb2,0x44,0x00,0x00,0xaa,0x24,0x00,0x00,0x5a,0x1c,0x00,0x00,0x22,0x00,0x00,0x00,0x1c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xba,0x4c,0x30,0x00,0x82,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x00,0x10,0x00,0x6c,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x6c,0x1c,0x20,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x0c,0x00,0x04,0x08,0x12,0x00,0x04,0x00,0x0a,0x04,0x00,0x00,0x0a,0x04,0x00,0x00,0x4a,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x64,0x00,0x24,0x40,0x9a,0x00,0x24,0x00,0x5a,0x20,0x08,0x00,0x54,0x20,0x10,0x00,0x48,0x28,0x00,0x00,0x54,0x24,0x00,0x00,0x5a,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x70,0x00,0x30,0x40,0x88,0x00,0x20,0x00,0x50,0x00,0x20,0x00,0x50,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x54,0x24,0x00,0x00,0x5a,0x1c,0x20,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x04,0x40,0xaa,0x24,0x40,0x08,0x92,0x04,0x50,0x00,0xaa,0x14,0x40,0x00,0xaa,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x20,0x44,0x00,0x9a,0x14,0x40,0x00,0xaa,0x0c,0x40,0x00,0xb2,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x78,0x08,0x30,0x40,0x84,0x04,0x20,0x00,0x5a,0x04,0x20,0x00,0x5a,0x28,0x10,0x00,0x44,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x44,0x00,0x10,0xaa,0x30,0x08,0x00,0x44,0x04,0x00,0x00,0x3a,0x00,0x00,0x00,0x04,0x07,
	0x00,0x00,0x00,0x78,0x00,0x18,0x60,0x84,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x28,0x10,0x00,0x44,0x24,0x00,0x00,0x5a,0x24,0x00,0x00,0x5a,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x38,0x00,0x18,0x20,0x44,0x04,0x00,0x40,0xba,0x00,0x40,0x00,0xbc,0x08,0x30,0x00,0x44,0x04,0x00,0x00,0x7a,0x04,0x00,0x40,0xba,0x08,0x30,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x18,0x60,0x82,0x14,0x40,0x00,0xaa,0x10,0x00,0x00,0x6c,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x08,0x10,0x20,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xba,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x04,0x40,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xaa,0x28,0x00,0x00,0x54,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x04,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x54,0x00,0x00,0xaa,0x6c,0x00,0x00,0x92,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x04,0x40,0x00,0xaa,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x08,0x20,0x00,0x54,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x44,0x04,0x00,0x40,0xaa,0x04,0x40,0x00,0xaa,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x10,0x00,0x00,0x28,0x10,0x00,0x00,0x28,0x08,0x10,0x20,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x04,0x40,0x00,0xba,0x08,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x20,0x00,0x54,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x10,0x00,0x00,0x68,0x00,0x20,0x00,0x58,0x00,0x08,0x40,0xb4,0x08,0x40,0x00,0xb4,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x00,0x06,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xa0,0x00,0x40,0x00,0xb8,0x38,0x00,0x00,0x44,0x10,0x00,0x00,0x28,0x00,0x10,0x20,0x48,0x00,0x00,0x00,0x30,0x06,
	0x00,0x00,0x00,0x50,0x00,0x00,0x50,0xa8,0x00,0x00,0x00,0x50,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x20,0x50,0x00,0x10,0x00,0x28,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x34,0x00,0x00,0x4a,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x28,0x10,0x44,0x04,0x00,0x00,0x3a,0x04,0x08,0x30,0x42,0x04,0x40,0x00,0xba,0x3c,0x00,0x00,0x42,0x00,0x00,0x00,0x3c,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x58,0x00,0x28,0x00,0x54,0x04,0x20,0x10,0x4a,0x04,0x20,0x00,0x5a,0x04,0x20,0x00,0x5a,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x00,0x40,0x00,0xa4,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x0c,0x00,0x04,0x08,0x12,0x08,0x00,0x00,0x34,0x08,0x20,0x00,0x54,0x18,0x40,0x00,0xa4,0x08,0x40,0x00,0xb4,0x08,0x40,0x00,0xb4,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x00,0x08,0x30,0x44,0x04,0x00,0x40,0xba,0x3c,0x40,0x00,0x82,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x18,0x00,0x08,0x10,0x24,0x04,0x00,0x20,0x5a,0x00,0x20,0x00,0x54,0x10,0x20,0x40,0x88,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x30,0x40,0x00,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x04,0x10,0x20,0x4a,0x08,0x00,0x40,0xb4,0x08,0x40,0x00,0xb4,0x38,0x00,0x00,0x44,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x58,0x20,0x08,0x00,0x54,0x24,0x00,0x10,0x4a,0x24,0x00,0x00,0x5a,0x24,0x00,0x00,0x5a,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x00,0x00,0x60,0x00,0x40,0x20,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x04,0x00,0x00,0x04,0x0a,0x00,0x00,0x00,0x04,0x00,0x00,0x04,0x0a,0x00,0x04,0x00,0x0a,0x04,0x00,0x00,0x0a,0x04,0x00,0x00,0x4a,0x04,0x00,0x40,0xba,0x18,0x20,0x00,0x44,0x00,0x00,0x00,0x38,0x07,
	0x00,0x00,0x00,0x60,0x00,0x00,0x60,0x90,0x00,0x20,0x00,0x54,0x20,0x04,0x00,0x5a,0x20,0x00,0x08,0x54,0x20,0x10,0x00,0x48,0x28,0x00,0x00,0x54,0x04,0x20,0x40,0x9a,0x00,0x00,0x00,0x64,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x60,0x00,0x20,0x40,0x90,0x00,0x20,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x20,0x00,0x00,0x50,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x05,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x24,0x48,0x00,0x92,0x44,0x10,0x00,0xaa,0x44,0x00,0x00,0xba,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x78,0x08,0x10,0x60,0x84,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xaa,0x44,0x00,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x38,0x08,0x10,0x20,0x44,0x04,0x00,0x40,0xba,0x04,0x40,0x00,0xaa,0x44,0x00,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x00,0x00,0x38,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x00,0x08,0x50,0xa4,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x38,0x00,0x00,0x44,0x20,0x00,0x00,0x58,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x34,0x00,0x14,0x20,0x4a,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x38,0x00,0x00,0x44,0x08,0x00,0x00,0x34,0x04,0x08,0x10,0x22,0x00,0x00,0x00,0x1c,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x58,0x00,0x08,0x50,0xa4,0x04,0x20,0x00,0x5a,0x24,0x00,0x00,0x5a,0x20,0x00,0x00,0x54,0x10,0x20,0x40,0x88,0x00,0x00,0x00,0x70,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x3c,0x04,0x18,0x20,0x42,0x00,0x40,0x00,0xbc,0x38,0x00,0x00,0x44,0x04,0x00,0x00,0x7a,0x38,0x00,0x40,0x84,0x00,0x00,0x00,0x78,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x20,0x00,0x00,0x20,0x50,0x00,0x20,0x00,0x58,0x08,0x30,0x40,0x84,0x20,0x00,0x00,0x58,0x20,0x00,0x00,0x54,0x04,0x20,0x00,0x5a,0x08,0x10,0x00,0x24,0x00,0x00,0x00,0x18,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x48,0x00,0x00,0x48,0xb4,0x00,0x48,0x00,0xb4,0x08,0x40,0x00,0xb4,0x48,0x00,0x00,0xb4,0x34,0x00,0x00,0x4a,0x00,0x00,0x00,0x34,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xaa,0x28,0x00,0x00,0x54,0x10,0x00,0x00,0x28,0x00,0x00,0x00,0x10,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xba,0x14,0x40,0x00,0xaa,0x2c,0x40,0x00,0x92,0x48,0x00,0x00,0xb4,0x00,0x00,0x00,0x48,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x28,0x00,0x54,0x10,0x00,0x00,0x28,0x08,0x20,0x00,0x54,0x04,0x40,0x00,0xaa,0x00,0x00,0x00,0x44,0x00,0x00,0x00,0x00,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x44,0x00,0x00,0x44,0xaa,0x00,0x44,0x00,0xaa,0x04,0x40,0x00,0xba,0x38,0x00,0x00,0x44,0x00,0x10,0x00,0x68,0x00,0x20,0x40,0x90,0x00,0x00,0x00,0x60,0x07,
	0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x00,0x7c,0x04,0x38,0x40,0x82,0x08,0x40,0x00,0xb4,0x00,0x10,0x00,0x6c,0x04,0x20,0x00,0x5a,0x3c,0x00,0x40,0x82,0x00,0x00,0x00,0x7c,0x00,0x00,0x00,0x00,0x07
};

void AGOSEngine::renderStringAmiga(uint vgaSpriteId, uint color, uint width, uint height, const char *txt) {
	VgaPointersEntry *vpe = &_vgaBufferPointers[2];
	byte *p, *dst, *dst_org, chr;
	uint count;

	if (vgaSpriteId >= 100) {
		vgaSpriteId -= 100;
		vpe++;
	}

	dst = vpe->vgaFile2;

	count = 2000;
	if (vgaSpriteId == 1)
		count *= 2;

	p = dst + vgaSpriteId * 8;
	WRITE_BE_UINT16(p + 4, height);
	WRITE_BE_UINT16(p + 6, width);
	dst += READ_BE_UINT32(p);

	width /= 8;	// convert width from pixels to bytes

	const byte *imgSrc = NULL;
	switch (_language) {
	case Common::IT_ITA:
		imgSrc = italian_simon1AGAFontData;
		break;
	case Common::FR_FRA:
		imgSrc = french_simon1AGAFontData;
		break;
	case Common::DE_DEU:
		imgSrc = german_simon1AGAFontData;
		break;
	case Common::EN_ANY:
		imgSrc = english_simon1AGAFontData;
		break;
	default:
		error("renderStringAmiga: Unknown language %d", _language);
	}


	uint charsize = width * height;
	memset(dst, 0, count);
	dst_org = dst;
	int delta = 0;
	while ((chr = *txt++) != 0) {
		int img_width = 1;
		if (chr == 10) {
			dst += width * 10;
			dst_org = dst;
			delta = 0;
		} else if ((signed char)(chr -= '!') < 0) {
			img_width = 7;
		} else {
			const byte *img = imgSrc + chr * 41;
			img_width = img[40];
			byte *cur_dst = dst_org;
			for (int row = 0; row < 10; row++) {
				int col = color;
				for (int plane = 0; plane < 3; plane++) {
					chr = img[plane] >> delta;
					if (chr) {
						if (col & 1) cur_dst[charsize * 0] |= chr;
						if (col & 2) cur_dst[charsize * 1] |= chr;
						if (col & 4) cur_dst[charsize * 2] |= chr;
						if (col & 8) cur_dst[charsize * 3] |= chr;
					}
					chr = img[plane] << (8 - delta);
					if (((8 - delta) < img_width) && (chr)) {
						if (col & 1) cur_dst[charsize * 0 + 1] |= chr;
						if (col & 2) cur_dst[charsize * 1 + 1] |= chr;
						if (col & 4) cur_dst[charsize * 2 + 1] |= chr;
						if (col & 8) cur_dst[charsize * 3 + 1] |= chr;
					}
					col++;
				}
				chr = img[3] >> delta;
				if (chr) {
					cur_dst[charsize * 0] |= chr;
					cur_dst[charsize * 1] |= chr;
					cur_dst[charsize * 2] |= chr;
					cur_dst[charsize * 3] |= chr;
				}
				chr = img[3] << (8 - delta);
				if (((8 - delta) < img_width) && (chr)) {
					cur_dst[charsize * 0 + 1] |= chr;
					cur_dst[charsize * 1 + 1] |= chr;
					cur_dst[charsize * 2 + 1] |= chr;
					cur_dst[charsize * 3 + 1] |= chr;
				}
				cur_dst += width;
				img += 4;
			}
		}
		delta += img_width - 1;
		if (delta >= 8) {
			delta -= 8;
			dst_org++;
		}
	}
}

void AGOSEngine::renderString(uint vgaSpriteId, uint color, uint width, uint height, const char *txt) {
	VgaPointersEntry *vpe = &_vgaBufferPointers[2];
	byte *src, *dst, *p, *dst_org, chr;
	const int textHeight = (getGameType() == GType_FF || getGameType() == GType_PP) ? 15: 10;
	uint count = 0;

	if (vgaSpriteId >= 100) {
		vgaSpriteId -= 100;
		vpe++;
	}

	src = dst = vpe->vgaFile2;

	if (getGameType() == GType_FF || getGameType() == GType_PP) {
		if (vgaSpriteId == 1)
			count = 45000;
	} else {
		count = 4000;
		if (vgaSpriteId == 1)
			count *= 2;
	}

	p = dst + vgaSpriteId * 8;

	if (getGameType() == GType_FF || getGameType() == GType_PP) {
		if (vgaSpriteId != 1)
			WRITE_LE_UINT32(p, READ_LE_UINT32(p - 8) + READ_LE_UINT16(p - 4) * READ_LE_UINT16(p - 2));

		WRITE_LE_UINT16(p + 4, height);
		WRITE_LE_UINT16(p + 6, width);
	} else {
		WRITE_BE_UINT16(p + 4, height);
		WRITE_BE_UINT16(p + 6, width);
	}
	dst += readUint32Wrapper(p);

	if (count != 0)
		memset(dst, 0, count);

	if (_language == Common::HE_ISR)
		dst += width - 1; // For Hebrew, start at the right edge, not the left.

	dst_org = dst;
	while ((chr = *txt++) != 0) {
		if (chr == 10) {
			dst_org += width * textHeight;
			dst = dst_org;
		} else if ((chr -= ' ') == 0) {
			dst += (_language == Common::HE_ISR ? -6 : 6); // Hebrew moves to the left, all others to the right
		} else {
			byte *img_hdr, *img;
			uint i, img_width, img_height;

			if (getGameType() == GType_FF || getGameType() == GType_PP) {
				img_hdr = src + 96 + chr * 8;
				img_height = READ_LE_UINT16(img_hdr + 4);
				img_width = READ_LE_UINT16(img_hdr + 6);
				img = src + READ_LE_UINT32(img_hdr);
			} else {
				img_hdr = src + 48 + chr * 4;
				img_height = img_hdr[2];
				img_width = img_hdr[3];
				img = src + READ_LE_UINT16(img_hdr);
			}

			if (_language == Common::HE_ISR)
				dst -= img_width - 1; // For Hebrew, move from right edge to left edge of image.
			byte *cur_dst = dst;

			// Occurs in Amiga and Macintosh ports of The Feeble Files, when
			// special characters are used by French/German/Spanish versions.
			// Due to the English image data, been used by all languages.
			if (img_width == 0 || img_height == 0)
				continue;

			assert(img_width < 50 && img_height < 50);

			do {
				for (i = 0; i != img_width; i++) {
					chr = *img++;
					if (chr) {
						if (chr == 0xF)
							chr = 207;
						else
							chr += color;
						cur_dst[i] = chr;
					}
				}
				cur_dst += width;
			} while (--img_height);

			if (_language != Common::HE_ISR) // Hebrew character movement is done higher up
				dst += img_width - 1;
		}
	}
}

static const byte czech_simonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x70, 0x70, 0x20, 0x20, 0x00, 0x20, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x50, 0x20, 0x78, 0x80, 0x70, 0x08, 0xf0, 0x00,
	0x10, 0x20, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x50, 0x20, 0xb0, 0x48, 0x48, 0x40, 0xe0, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2a, 0x44, 0x3a, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3e, 0x08, 0x14, 0x00,
	0x00, 0x70, 0x88, 0xf0, 0x88, 0x88, 0xf0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
	0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
	0x50, 0x20, 0xf8, 0x90, 0x20, 0x48, 0xf8, 0x00,
	0x70, 0x88, 0x98, 0xa8, 0xc8, 0x88, 0x70, 0x00,
	0x20, 0x60, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x40, 0x88, 0xf8, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x10, 0x30, 0x50, 0x90, 0xf8, 0x10, 0x38, 0x00,
	0xf8, 0x80, 0xf0, 0x08, 0x08, 0x88, 0x70, 0x00,
	0x30, 0x40, 0x80, 0xf0, 0x88, 0x88, 0x70, 0x00,
	0xf8, 0x88, 0x08, 0x10, 0x20, 0x20, 0x20, 0x00,
	0x70, 0x88, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x70, 0x88, 0x88, 0x78, 0x08, 0x10, 0x60, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x00,
	0x08, 0x10, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0x20, 0x50, 0x20, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x10, 0x20, 0x70, 0x88, 0xf8, 0x80, 0x70, 0x00,
	0x50, 0x20, 0x70, 0x88, 0xf8, 0x80, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x10, 0x20, 0x00, 0x20, 0x00,
	0x20, 0x40, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x70, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xf0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xe0, 0x50, 0x48, 0x48, 0x48, 0x50, 0xe0, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x48, 0xf8, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x30, 0x48, 0x80, 0x98, 0x88, 0x48, 0x38, 0x00,
	0x88, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0xf8, 0x00,
	0x18, 0x08, 0x08, 0x08, 0x88, 0x88, 0x70, 0x00,
	0xc8, 0x48, 0x50, 0x60, 0x50, 0x48, 0xc8, 0x00,
	0xe0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xf8, 0x00,
	0x88, 0xd8, 0xa8, 0xa8, 0x88, 0x88, 0x88, 0x00,
	0x88, 0xc8, 0xa8, 0x98, 0x88, 0x88, 0x88, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0xa8, 0x70, 0x08,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xc8, 0x00,
	0x70, 0x88, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0xf8, 0xa8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x78, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x20, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xa8, 0xd8, 0x88, 0x00,
	0x88, 0x88, 0x50, 0x20, 0x50, 0x88, 0x88, 0x00,
	0x88, 0x88, 0x88, 0x70, 0x20, 0x20, 0x70, 0x00,
	0xf8, 0x88, 0x10, 0x20, 0x40, 0x88, 0xf8, 0x00,
	0x00, 0x0e, 0x08, 0x08, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x50, 0x20, 0x70, 0x88, 0x80, 0x88, 0x70, 0x00,
	0x10, 0x20, 0x88, 0x88, 0x88, 0x70, 0x20, 0xc0,
	0x10, 0x20, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x80, 0x88, 0x70, 0x00,
	0x18, 0x10, 0x50, 0xb0, 0x90, 0x90, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0xf8, 0x80, 0x70, 0x00,
	0x30, 0x48, 0x40, 0xe0, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x88, 0x70,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0xc8, 0x00,
	0x40, 0x00, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x08, 0x00, 0x08, 0x08, 0x08, 0x08, 0x88, 0x70,
	0xc0, 0x40, 0x48, 0x50, 0x60, 0x50, 0xc8, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x90, 0xd8, 0xa8, 0x88, 0x88, 0x00,
	0x00, 0x00, 0xf0, 0x88, 0x88, 0x88, 0x88, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x70, 0x40, 0xe0,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x10, 0x38,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x78, 0x80, 0x70, 0x08, 0xf0, 0x00,
	0x40, 0x40, 0xf0, 0x40, 0x40, 0x48, 0x30, 0x00,
	0x00, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x50, 0x20, 0x00,
	0x00, 0x00, 0x88, 0x88, 0xa8, 0xd8, 0x90, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x70, 0x20, 0xc0,
	0x00, 0x00, 0xf8, 0x90, 0x20, 0x48, 0xf8, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0e, 0x08, 0x30, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0c, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00,
	0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
};

static const byte english_simonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x70, 0x70, 0x20, 0x20, 0x00, 0x20, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x90, 0x00, 0x90, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2a, 0x44, 0x3a, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3e, 0x08, 0x14, 0x00,
	0x00, 0x70, 0x88, 0xf0, 0x88, 0x88, 0xf0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
	0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x70, 0x88, 0x98, 0xa8, 0xc8, 0x88, 0x70, 0x00,
	0x20, 0x60, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x40, 0x88, 0xf8, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x10, 0x30, 0x50, 0x90, 0xf8, 0x10, 0x38, 0x00,
	0xf8, 0x80, 0xf0, 0x08, 0x08, 0x88, 0x70, 0x00,
	0x30, 0x40, 0x80, 0xf0, 0x88, 0x88, 0x70, 0x00,
	0xf8, 0x88, 0x08, 0x10, 0x20, 0x20, 0x20, 0x00,
	0x70, 0x88, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x70, 0x88, 0x88, 0x78, 0x08, 0x10, 0x60, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x70, 0x08, 0xf8, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x50, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x20, 0x50, 0x00, 0x70, 0xf8, 0x80, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x10, 0x20, 0x00, 0x20, 0x00,
	0x20, 0x50, 0x00, 0xc0, 0x40, 0x40, 0xe0, 0x00,
	0x70, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xf0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xe0, 0x50, 0x48, 0x48, 0x48, 0x50, 0xe0, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x48, 0xf8, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x30, 0x48, 0x80, 0x98, 0x88, 0x48, 0x38, 0x00,
	0x88, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0xf8, 0x00,
	0x18, 0x08, 0x08, 0x08, 0x88, 0x88, 0x70, 0x00,
	0xc8, 0x48, 0x50, 0x60, 0x50, 0x48, 0xc8, 0x00,
	0xe0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xf8, 0x00,
	0x88, 0xd8, 0xa8, 0xa8, 0x88, 0x88, 0x88, 0x00,
	0x88, 0xc8, 0xa8, 0x98, 0x88, 0x88, 0x88, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0xa8, 0x70, 0x08,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xc8, 0x00,
	0x70, 0x88, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0xf8, 0xa8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x78, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x20, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xa8, 0xd8, 0x88, 0x00,
	0x88, 0x88, 0x50, 0x20, 0x50, 0x88, 0x88, 0x00,
	0x88, 0x88, 0x88, 0x70, 0x20, 0x20, 0x70, 0x00,
	0xf8, 0x88, 0x10, 0x20, 0x40, 0x88, 0xf8, 0x00,
	0x00, 0x0e, 0x08, 0x08, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x98, 0x90, 0x90, 0xe8, 0x00,
	0x00, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x80, 0x88, 0x70, 0x00,
	0x18, 0x10, 0x50, 0xb0, 0x90, 0x90, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0xf8, 0x80, 0x70, 0x00,
	0x30, 0x48, 0x40, 0xe0, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x88, 0x70,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0xc8, 0x00,
	0x40, 0x00, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x08, 0x00, 0x08, 0x08, 0x08, 0x08, 0x88, 0x70,
	0xc0, 0x40, 0x48, 0x50, 0x60, 0x50, 0xc8, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x90, 0xd8, 0xa8, 0x88, 0x88, 0x00,
	0x00, 0x00, 0xf0, 0x88, 0x88, 0x88, 0x88, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x70, 0x40, 0xe0,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x10, 0x38,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x78, 0x80, 0x70, 0x08, 0xf0, 0x00,
	0x40, 0x40, 0xf0, 0x40, 0x40, 0x48, 0x30, 0x00,
	0x00, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x50, 0x20, 0x00,
	0x00, 0x00, 0x88, 0x88, 0xa8, 0xd8, 0x90, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x70, 0x20, 0xc0,
	0x00, 0x00, 0xf8, 0x90, 0x20, 0x48, 0xf8, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0e, 0x08, 0x30, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0c, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00,
	0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
};

static const byte french_simonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x70, 0x70, 0x20, 0x20, 0x00, 0x20, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x50, 0x00, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x20, 0x50, 0x00, 0x70, 0x08, 0xf8, 0x78, 0x00,
	0x70, 0x88, 0x80, 0x80, 0x88, 0x70, 0x20, 0x60,
	0x00, 0x10, 0x28, 0x10, 0x2a, 0x44, 0x3a, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3e, 0x08, 0x14, 0x00,
	0x70, 0x88, 0x80, 0x80, 0x88, 0x70, 0x20, 0x60,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
	0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
	0x20, 0x40, 0x00, 0x70, 0xf8, 0x80, 0x70, 0x00,
	0x70, 0x88, 0x98, 0xa8, 0xc8, 0x88, 0x70, 0x00,
	0x20, 0x60, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x40, 0x88, 0xf8, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x10, 0x30, 0x50, 0x90, 0xf8, 0x10, 0x38, 0x00,
	0xf8, 0x80, 0xf0, 0x08, 0x08, 0x88, 0x70, 0x00,
	0x30, 0x40, 0x80, 0xf0, 0x88, 0x88, 0x70, 0x00,
	0xf8, 0x88, 0x08, 0x10, 0x20, 0x20, 0x20, 0x00,
	0x70, 0x88, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x70, 0x88, 0x88, 0x78, 0x08, 0x10, 0x60, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x70, 0x08, 0xf8, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x20, 0x10, 0x00, 0x70, 0xf8, 0x80, 0x70, 0x00,
	0x20, 0x50, 0x00, 0x70, 0xf8, 0x80, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x10, 0x20, 0x00, 0x20, 0x00,
	0x20, 0x50, 0x00, 0xc0, 0x40, 0x40, 0xe0, 0x00,
	0x70, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xf0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xe0, 0x50, 0x48, 0x48, 0x48, 0x50, 0xe0, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x48, 0xf8, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x30, 0x48, 0x80, 0x98, 0x88, 0x48, 0x38, 0x00,
	0x88, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0xf8, 0x00,
	0x18, 0x08, 0x08, 0x08, 0x88, 0x88, 0x70, 0x00,
	0xc8, 0x48, 0x50, 0x60, 0x50, 0x48, 0xc8, 0x00,
	0xe0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xf8, 0x00,
	0x88, 0xd8, 0xa8, 0xa8, 0x88, 0x88, 0x88, 0x00,
	0x88, 0xc8, 0xa8, 0x98, 0x88, 0x88, 0x88, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0xa8, 0x70, 0x08,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xc8, 0x00,
	0x70, 0x88, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0xf8, 0xa8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x78, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x20, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xa8, 0xd8, 0x88, 0x00,
	0x88, 0x88, 0x50, 0x20, 0x50, 0x88, 0x88, 0x00,
	0x88, 0x88, 0x88, 0x70, 0x20, 0x20, 0x70, 0x00,
	0xf8, 0x88, 0x10, 0x20, 0x40, 0x88, 0xf8, 0x00,
	0x00, 0x0e, 0x08, 0x08, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x80, 0x70, 0x20, 0x60,
	0xa0, 0x00, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x20, 0x10, 0x00, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x80, 0x88, 0x70, 0x00,
	0x18, 0x10, 0x50, 0xb0, 0x90, 0x90, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0xf8, 0x80, 0x70, 0x00,
	0x30, 0x48, 0x40, 0xe0, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x88, 0x70,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0xc8, 0x00,
	0x40, 0x00, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x08, 0x00, 0x08, 0x08, 0x08, 0x08, 0x88, 0x70,
	0xc0, 0x40, 0x48, 0x50, 0x60, 0x50, 0xc8, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x90, 0xd8, 0xa8, 0x88, 0x88, 0x00,
	0x00, 0x00, 0xf0, 0x88, 0x88, 0x88, 0x88, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x70, 0x40, 0xe0,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x10, 0x38,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x78, 0x80, 0x70, 0x08, 0xf0, 0x00,
	0x40, 0x40, 0xf0, 0x40, 0x40, 0x48, 0x30, 0x00,
	0x00, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x50, 0x20, 0x00,
	0x00, 0x00, 0x88, 0x88, 0xa8, 0xd8, 0x90, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x70, 0x20, 0xc0,
	0x00, 0x00, 0xf8, 0x90, 0x20, 0x48, 0xf8, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0e, 0x08, 0x30, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0c, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00,
	0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
};

static const byte german_simonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x70, 0x70, 0x20, 0x20, 0x00, 0x20, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x50, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0x50, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x50, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2a, 0x44, 0x3a, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3e, 0x08, 0x14, 0x00,
	0x60, 0x90, 0x90, 0xa0, 0x90, 0x90, 0xa0, 0x80,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
	0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
	0x50, 0x00, 0x70, 0x88, 0xf8, 0x88, 0x88, 0x00,
	0x70, 0x88, 0x98, 0xa8, 0xc8, 0x88, 0x70, 0x00,
	0x20, 0x60, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x40, 0x88, 0xf8, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x10, 0x30, 0x50, 0x90, 0xf8, 0x10, 0x38, 0x00,
	0xf8, 0x80, 0xf0, 0x08, 0x08, 0x88, 0x70, 0x00,
	0x30, 0x40, 0x80, 0xf0, 0x88, 0x88, 0x70, 0x00,
	0xf8, 0x88, 0x08, 0x10, 0x20, 0x20, 0x20, 0x00,
	0x70, 0x88, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x70, 0x88, 0x88, 0x78, 0x08, 0x10, 0x60, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x00,
	0x88, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x50, 0x00, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x50, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x20, 0x40, 0x00, 0x70, 0xf8, 0x80, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x10, 0x20, 0x00, 0x20, 0x00,
	0x20, 0x50, 0x00, 0xc0, 0x40, 0x40, 0xe0, 0x00,
	0x70, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xf0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xe0, 0x50, 0x48, 0x48, 0x48, 0x50, 0xe0, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x48, 0xf8, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x30, 0x48, 0x80, 0x98, 0x88, 0x48, 0x38, 0x00,
	0x88, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0xf8, 0x00,
	0x18, 0x08, 0x08, 0x08, 0x88, 0x88, 0x70, 0x00,
	0xc8, 0x48, 0x50, 0x60, 0x50, 0x48, 0xc8, 0x00,
	0xe0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xf8, 0x00,
	0x88, 0xd8, 0xa8, 0xa8, 0x88, 0x88, 0x88, 0x00,
	0x88, 0xc8, 0xa8, 0x98, 0x88, 0x88, 0x88, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0xa8, 0x70, 0x08,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xc8, 0x00,
	0x70, 0x88, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0xf8, 0xa8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x78, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x20, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xa8, 0xd8, 0x88, 0x00,
	0x88, 0x88, 0x50, 0x20, 0x50, 0x88, 0x88, 0x00,
	0x88, 0x88, 0x88, 0x70, 0x20, 0x20, 0x70, 0x00,
	0xf8, 0x88, 0x10, 0x20, 0x40, 0x88, 0xf8, 0x00,
	0x00, 0x0e, 0x08, 0x08, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x98, 0x90, 0x90, 0xe8, 0x00,
	0x00, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x80, 0x88, 0x70, 0x00,
	0x18, 0x10, 0x50, 0xb0, 0x90, 0x90, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0xf8, 0x80, 0x70, 0x00,
	0x30, 0x48, 0x40, 0xe0, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x88, 0x70,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0xc8, 0x00,
	0x40, 0x00, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x08, 0x00, 0x08, 0x08, 0x08, 0x08, 0x88, 0x70,
	0xc0, 0x40, 0x48, 0x50, 0x60, 0x50, 0xc8, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x90, 0xd8, 0xa8, 0x88, 0x88, 0x00,
	0x00, 0x00, 0xf0, 0x88, 0x88, 0x88, 0x88, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x70, 0x40, 0xe0,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x10, 0x38,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x78, 0x80, 0x70, 0x08, 0xf0, 0x00,
	0x40, 0x40, 0xf0, 0x40, 0x40, 0x48, 0x30, 0x00,
	0x00, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x50, 0x20, 0x00,
	0x00, 0x00, 0x88, 0x88, 0xa8, 0xd8, 0x90, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x70, 0x20, 0xc0,
	0x00, 0x00, 0xf8, 0x90, 0x20, 0x48, 0xf8, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0e, 0x08, 0x30, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0c, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00,
	0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
};

static const byte hebrew_simonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x70, 0x70, 0x20, 0x20, 0x00, 0x20, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x90, 0x00, 0x90, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2a, 0x44, 0x3a, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3e, 0x08, 0x14, 0x00,
	0x00, 0x70, 0x88, 0xf0, 0x88, 0x88, 0xf0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
	0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x70, 0x88, 0x98, 0xa8, 0xc8, 0x88, 0x70, 0x00,
	0x20, 0x60, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x40, 0x88, 0xf8, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x10, 0x30, 0x50, 0x90, 0xf8, 0x10, 0x38, 0x00,
	0xf8, 0x80, 0xf0, 0x08, 0x08, 0x88, 0x70, 0x00,
	0x30, 0x40, 0x80, 0xf0, 0x88, 0x88, 0x70, 0x00,
	0xf8, 0x88, 0x08, 0x10, 0x20, 0x20, 0x20, 0x00,
	0x70, 0x88, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x70, 0x88, 0x88, 0x78, 0x08, 0x10, 0x60, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x00,
	0x05, 0x05, 0x04, 0x06, 0x05, 0x03, 0x04, 0x05,
	0x06, 0x03, 0x05, 0x05, 0x04, 0x06, 0x05, 0x03,
	0x04, 0x06, 0x05, 0x06, 0x06, 0x06, 0x05, 0x05,
	0x05, 0x06, 0x05, 0x06, 0x06, 0x06, 0x06, 0x06,
	0x70, 0x88, 0x08, 0x10, 0x20, 0x00, 0x20, 0x00,
	0x00, 0x00, 0x90, 0x50, 0xa0, 0x90, 0x90, 0x00,
	0x00, 0x00, 0xe0, 0x20, 0x20, 0x20, 0xf0, 0x00,
	0x00, 0x00, 0xe0, 0x20, 0x60, 0xa0, 0xa0, 0x00,
	0x00, 0x00, 0xf8, 0x10, 0x10, 0x10, 0x10, 0x00,
	0x00, 0x00, 0xf0, 0x10, 0x10, 0x90, 0x90, 0x00,
	0x00, 0x00, 0xc0, 0x40, 0x40, 0x40, 0x40, 0x00,
	0x00, 0x00, 0xe0, 0x40, 0x20, 0x40, 0x40, 0x00,
	0x00, 0x00, 0xf0, 0x90, 0x90, 0x90, 0x90, 0x00,
	0x00, 0x00, 0xb8, 0xa8, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x00, 0xc0, 0x40, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0xf0, 0x10, 0x10, 0x10, 0x10, 0x10,
	0x00, 0x00, 0xe0, 0x10, 0x10, 0x10, 0xe0, 0x00,
	0x80, 0x80, 0xe0, 0x20, 0x20, 0x20, 0xc0, 0x00,
	0x00, 0x00, 0xf8, 0x48, 0x48, 0x48, 0x78, 0x00,
	0x00, 0x00, 0xb0, 0xd0, 0x90, 0x90, 0xb0, 0x00,
	0x00, 0x00, 0xc0, 0x40, 0x40, 0x40, 0x40, 0x40,
	0x00, 0x00, 0x60, 0x20, 0x20, 0x20, 0xe0, 0x00,
	0x00, 0x00, 0xf8, 0x48, 0x48, 0x48, 0x30, 0x00,
	0x00, 0x00, 0x50, 0x50, 0x50, 0x50, 0xe0, 0x00,
	0x00, 0x00, 0xf8, 0x48, 0x68, 0x08, 0x08, 0x08,
	0x00, 0x00, 0xf8, 0x48, 0x68, 0x08, 0xf8, 0x00,
	0x00, 0x00, 0xd8, 0x48, 0x30, 0x10, 0x10, 0x10,
	0x00, 0x00, 0x90, 0x50, 0x20, 0x10, 0xf0, 0x00,
	0x00, 0x00, 0xf0, 0x10, 0x90, 0xa0, 0x80, 0x80,
	0x00, 0x00, 0xf0, 0x10, 0x10, 0x10, 0x10, 0x00,
	0x00, 0x00, 0xa8, 0xa8, 0xc8, 0x88, 0x70, 0x00,
	0x00, 0x00, 0xf0, 0x50, 0x50, 0x50, 0xd0, 0x00,
	0x00, 0x0e, 0x08, 0x08, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x98, 0x90, 0x90, 0xe8, 0x00,
	0x00, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x80, 0x88, 0x70, 0x00,
	0x18, 0x10, 0x50, 0xb0, 0x90, 0x90, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0xf8, 0x80, 0x70, 0x00,
	0x30, 0x48, 0x40, 0xe0, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x88, 0x70,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0xc8, 0x00,
	0x40, 0x00, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x08, 0x00, 0x08, 0x08, 0x08, 0x08, 0x88, 0x70,
	0xc0, 0x40, 0x48, 0x50, 0x60, 0x50, 0xc8, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x90, 0xd8, 0xa8, 0x88, 0x88, 0x00,
	0x00, 0x00, 0xf0, 0x88, 0x88, 0x88, 0x88, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x70, 0x40, 0xe0,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x10, 0x38,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x78, 0x80, 0x70, 0x08, 0xf0, 0x00,
	0x40, 0x40, 0xf0, 0x40, 0x40, 0x48, 0x30, 0x00,
	0x00, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x50, 0x20, 0x00,
	0x00, 0x00, 0x88, 0x88, 0xa8, 0xd8, 0x90, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x70, 0x20, 0xc0,
	0x00, 0x00, 0xf8, 0x90, 0x20, 0x48, 0xf8, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0e, 0x08, 0x30, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0c, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00,
	0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
};

static const byte italian_simonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x70, 0x70, 0x20, 0x20, 0x00, 0x20, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x50, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0x50, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x20, 0x10, 0x00, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2a, 0x44, 0x3a, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3e, 0x08, 0x14, 0x00,
	0x20, 0x10, 0x00, 0xc0, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
	0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
	0x20, 0x40, 0x00, 0x70, 0xf8, 0x80, 0x70, 0x00,
	0x70, 0x88, 0x98, 0xa8, 0xc8, 0x88, 0x70, 0x00,
	0x20, 0x60, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x40, 0x88, 0xf8, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x10, 0x30, 0x50, 0x90, 0xf8, 0x10, 0x38, 0x00,
	0xf8, 0x80, 0xf0, 0x08, 0x08, 0x88, 0x70, 0x00,
	0x30, 0x40, 0x80, 0xf0, 0x88, 0x88, 0x70, 0x00,
	0xf8, 0x88, 0x08, 0x10, 0x20, 0x20, 0x20, 0x00,
	0x70, 0x88, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x70, 0x88, 0x88, 0x78, 0x08, 0x10, 0x60, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x70, 0x08, 0xf8, 0x78, 0x00,
	0x20, 0x10, 0x00, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x20, 0x10, 0x00, 0x70, 0xf8, 0x80, 0x70, 0x00,
	0x20, 0x50, 0x00, 0x70, 0xf8, 0x80, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x10, 0x20, 0x00, 0x20, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x70, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xf0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xe0, 0x50, 0x48, 0x48, 0x48, 0x50, 0xe0, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x48, 0xf8, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x30, 0x48, 0x80, 0x98, 0x88, 0x48, 0x38, 0x00,
	0x88, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0xf8, 0x00,
	0x18, 0x08, 0x08, 0x08, 0x88, 0x88, 0x70, 0x00,
	0xc8, 0x48, 0x50, 0x60, 0x50, 0x48, 0xc8, 0x00,
	0xe0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xf8, 0x00,
	0x88, 0xd8, 0xa8, 0xa8, 0x88, 0x88, 0x88, 0x00,
	0x88, 0xc8, 0xa8, 0x98, 0x88, 0x88, 0x88, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0xa8, 0x70, 0x08,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xc8, 0x00,
	0x70, 0x88, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0xf8, 0xa8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x78, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x20, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xa8, 0xd8, 0x88, 0x00,
	0x88, 0x88, 0x50, 0x20, 0x50, 0x88, 0x88, 0x00,
	0x88, 0x88, 0x88, 0x70, 0x20, 0x20, 0x70, 0x00,
	0xf8, 0x88, 0x10, 0x20, 0x40, 0x88, 0xf8, 0x00,
	0x00, 0x0e, 0x08, 0x08, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x80, 0x70, 0x20, 0x60,
	0xa0, 0x00, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x20, 0x10, 0x00, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x80, 0x88, 0x70, 0x00,
	0x18, 0x10, 0x50, 0xb0, 0x90, 0x90, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0xf8, 0x80, 0x70, 0x00,
	0x30, 0x48, 0x40, 0xe0, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x88, 0x70,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0xc8, 0x00,
	0x40, 0x00, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x08, 0x00, 0x08, 0x08, 0x08, 0x08, 0x88, 0x70,
	0xc0, 0x40, 0x48, 0x50, 0x60, 0x50, 0xc8, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x90, 0xd8, 0xa8, 0x88, 0x88, 0x00,
	0x00, 0x00, 0xf0, 0x88, 0x88, 0x88, 0x88, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x70, 0x40, 0xe0,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x10, 0x38,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x78, 0x80, 0x70, 0x08, 0xf0, 0x00,
	0x40, 0x40, 0xf0, 0x40, 0x40, 0x48, 0x30, 0x00,
	0x00, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x50, 0x20, 0x00,
	0x00, 0x00, 0x88, 0x88, 0xa8, 0xd8, 0x90, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x70, 0x20, 0xc0,
	0x00, 0x00, 0xf8, 0x90, 0x20, 0x48, 0xf8, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0e, 0x08, 0x30, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0c, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00,
	0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
};

static const byte polish_simonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x70, 0x70, 0x20, 0x20, 0x00, 0x20, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x70, 0x88, 0xf8, 0x80, 0x70, 0x08,
	0x00, 0x10, 0x78, 0x80, 0x70, 0x08, 0xf0, 0x00,
	0xc0, 0x40, 0x40, 0x60, 0xc0, 0x40, 0xe0, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2a, 0x44, 0x3a, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3e, 0x08, 0x14, 0x00,
	0x00, 0x20, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
	0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
	0x00, 0x20, 0x70, 0x88, 0x80, 0x88, 0x70, 0x00,
	0x70, 0x88, 0x98, 0xa8, 0xc8, 0x88, 0x70, 0x00,
	0x20, 0x60, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x40, 0x88, 0xf8, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x10, 0x30, 0x50, 0x90, 0xf8, 0x10, 0x38, 0x00,
	0xf8, 0x80, 0xf0, 0x08, 0x08, 0x88, 0x70, 0x00,
	0x30, 0x40, 0x80, 0xf0, 0x88, 0x88, 0x70, 0x00,
	0xf8, 0x88, 0x08, 0x10, 0x20, 0x20, 0x20, 0x00,
	0x70, 0x88, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x70, 0x88, 0x88, 0x78, 0x08, 0x10, 0x60, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x00,
	0x00, 0x20, 0xf0, 0x88, 0x88, 0x88, 0x88, 0x00,
	0x50, 0x00, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x20, 0xf8, 0x90, 0x20, 0x48, 0xf8, 0x00,
	0x08, 0x20, 0xf8, 0x90, 0x20, 0x48, 0xf8, 0x00,
	0x70, 0x88, 0x08, 0x10, 0x20, 0x00, 0x20, 0x00,
	0x00, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x04,
	0x70, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xf0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xe0, 0x50, 0x48, 0x48, 0x48, 0x50, 0xe0, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x48, 0xf8, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x30, 0x48, 0x80, 0x98, 0x88, 0x48, 0x38, 0x00,
	0x88, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0xf8, 0x00,
	0x18, 0x08, 0x08, 0x08, 0x88, 0x88, 0x70, 0x00,
	0xc8, 0x48, 0x50, 0x60, 0x50, 0x48, 0xc8, 0x00,
	0xe0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xf8, 0x00,
	0x88, 0xd8, 0xa8, 0xa8, 0x88, 0x88, 0x88, 0x00,
	0x88, 0xc8, 0xa8, 0x98, 0x88, 0x88, 0x88, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0xa8, 0x70, 0x08,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xc8, 0x00,
	0x70, 0x88, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0xf8, 0xa8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x78, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x20, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xa8, 0xd8, 0x88, 0x00,
	0x88, 0x88, 0x50, 0x20, 0x50, 0x88, 0x88, 0x00,
	0x88, 0x88, 0x88, 0x70, 0x20, 0x20, 0x70, 0x00,
	0xf8, 0x88, 0x10, 0x20, 0x40, 0x88, 0xf8, 0x00,
	0x00, 0x0e, 0x08, 0x08, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0xe0, 0x40, 0x40, 0x60, 0xc0, 0x48, 0xf8, 0x00,
	0x10, 0x78, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x48, 0xf8, 0x10,
	0x20, 0xf8, 0x10, 0x20, 0x40, 0x88, 0xf8, 0x00,
	0x00, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x80, 0x88, 0x70, 0x00,
	0x18, 0x10, 0x50, 0xb0, 0x90, 0x90, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0xf8, 0x80, 0x70, 0x00,
	0x30, 0x48, 0x40, 0xe0, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x88, 0x70,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0xc8, 0x00,
	0x40, 0x00, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x08, 0x00, 0x08, 0x08, 0x08, 0x08, 0x88, 0x70,
	0xc0, 0x40, 0x48, 0x50, 0x60, 0x50, 0xc8, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x90, 0xd8, 0xa8, 0x88, 0x88, 0x00,
	0x00, 0x00, 0xf0, 0x88, 0x88, 0x88, 0x88, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x70, 0x40, 0xe0,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x10, 0x38,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x78, 0x80, 0x70, 0x08, 0xf0, 0x00,
	0x40, 0x40, 0xf0, 0x40, 0x40, 0x48, 0x30, 0x00,
	0x00, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x50, 0x20, 0x00,
	0x00, 0x00, 0x88, 0x88, 0xa8, 0xd8, 0x90, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x70, 0x20, 0xc0,
	0x00, 0x00, 0xf8, 0x90, 0x20, 0x48, 0xf8, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0e, 0x08, 0x30, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0c, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00,
	0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
};

static const byte russian_simonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x70, 0x70, 0x20, 0x20, 0x00, 0x20, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x64, 0x28, 0x30, 0x28, 0x64, 0x00,
	0x00, 0x00, 0x60, 0x30, 0x28, 0x28, 0x70, 0x00,
	0x3c, 0x44, 0x44, 0x3c, 0x24, 0x44, 0x44, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2a, 0x44, 0x3a, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x48, 0x54, 0x54, 0x74, 0x54, 0x54, 0x48, 0x00,
	0x00, 0x00, 0x3c, 0x44, 0x3c, 0x24, 0x64, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
	0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x70, 0x88, 0x98, 0xa8, 0xc8, 0x88, 0x70, 0x00,
	0x20, 0x60, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x40, 0x88, 0xf8, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x10, 0x30, 0x50, 0x90, 0xf8, 0x10, 0x38, 0x00,
	0xf8, 0x80, 0xf0, 0x08, 0x08, 0x88, 0x70, 0x00,
	0x30, 0x40, 0x80, 0xf0, 0x88, 0x88, 0x70, 0x00,
	0xf8, 0x88, 0x08, 0x10, 0x20, 0x20, 0x20, 0x00,
	0x70, 0x88, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x70, 0x88, 0x88, 0x78, 0x08, 0x10, 0x60, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x70, 0x08, 0xf8, 0x78, 0x00,
	0x00, 0x00, 0x60, 0x30, 0x28, 0x28, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x58, 0x14, 0x14, 0x38, 0x00,
	0x00, 0x00, 0x78, 0x04, 0x1c, 0x04, 0x78, 0x00,
	0x70, 0x88, 0x08, 0x10, 0x20, 0x00, 0x20, 0x00,
	0x00, 0x00, 0x54, 0x54, 0x54, 0x54, 0x38, 0x04,
	0x38, 0x44, 0x44, 0x7c, 0x44, 0x44, 0x44, 0x00,
	0x7c, 0x24, 0x20, 0x38, 0x24, 0x24, 0x78, 0x00,
	0x78, 0x24, 0x24, 0x38, 0x24, 0x24, 0x78, 0x00,
	0x7c, 0x24, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x38, 0x28, 0x28, 0x28, 0x28, 0x28, 0x7c, 0x44,
	0x7c, 0x24, 0x20, 0x38, 0x20, 0x24, 0x7c, 0x00,
	0x54, 0x54, 0x54, 0x38, 0x54, 0x54, 0x54, 0x00,
	0x38, 0x44, 0x04, 0x18, 0x04, 0x44, 0x38, 0x00,
	0x44, 0x44, 0x4c, 0x54, 0x64, 0x44, 0x44, 0x00,
	0x64, 0x28, 0x28, 0x30, 0x28, 0x24, 0x64, 0x00,
	0x1c, 0x24, 0x24, 0x24, 0x24, 0x24, 0x64, 0x00,
	0x44, 0x6c, 0x54, 0x44, 0x44, 0x44, 0x44, 0x00,
	0x44, 0x44, 0x44, 0x7c, 0x44, 0x44, 0x44, 0x00,
	0x38, 0x44, 0x44, 0x44, 0x44, 0x44, 0x38, 0x00,
	0x7c, 0x44, 0x44, 0x44, 0x44, 0x44, 0x44, 0x00,
	0x78, 0x24, 0x24, 0x38, 0x20, 0x20, 0x70, 0x00,
	0x38, 0x44, 0x40, 0x40, 0x40, 0x44, 0x38, 0x00,
	0x7c, 0x54, 0x10, 0x10, 0x10, 0x10, 0x38, 0x00,
	0x64, 0x24, 0x24, 0x1c, 0x04, 0x04, 0x38, 0x00,
	0x38, 0x54, 0x54, 0x54, 0x38, 0x10, 0x38, 0x00,
	0x6c, 0x28, 0x10, 0x10, 0x28, 0x28, 0x6c, 0x00,
	0x48, 0x48, 0x48, 0x48, 0x48, 0x48, 0x3c, 0x04,
	0x4c, 0x48, 0x48, 0x38, 0x08, 0x08, 0x1c, 0x00,
	0x54, 0x54, 0x54, 0x54, 0x54, 0x54, 0x3c, 0x00,
	0x54, 0x54, 0x54, 0x54, 0x54, 0x54, 0x38, 0x04,
	0x38, 0x44, 0x04, 0x1c, 0x04, 0x44, 0x38, 0x00,
	0x00, 0x00, 0x44, 0x64, 0x54, 0x54, 0x64, 0x00,
	0x00, 0x00, 0x48, 0x54, 0x74, 0x54, 0x48, 0x00,
	0x00, 0x00, 0x3c, 0x44, 0x3c, 0x24, 0x64, 0x00,
	0x00, 0x00, 0x78, 0x04, 0x18, 0x04, 0x78, 0x00,
	0x00, 0x00, 0x64, 0x28, 0x30, 0x28, 0x64, 0x00,
	0x3c, 0x44, 0x44, 0x3c, 0x24, 0x44, 0x44, 0x00,
	0x00, 0x00, 0x38, 0x04, 0x3c, 0x44, 0x3c, 0x00,
	0x3c, 0x40, 0x20, 0x38, 0x44, 0x44, 0x38, 0x00,
	0x30, 0x48, 0x50, 0x78, 0x44, 0x44, 0x38, 0x00,
	0x00, 0x00, 0x78, 0x04, 0x38, 0x40, 0x3c, 0x00,
	0x38, 0x04, 0x04, 0x3c, 0x44, 0x44, 0x38, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x78, 0x40, 0x38, 0x00,
	0x28, 0x00, 0x38, 0x44, 0x78, 0x40, 0x38, 0x00,
	0x00, 0x00, 0x54, 0x54, 0x38, 0x54, 0x54, 0x00,
	0x40, 0x00, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x3c, 0x00,
	0x38, 0x00, 0x44, 0x44, 0x44, 0x44, 0x3c, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x1c, 0x24, 0x24, 0x24, 0x64, 0x00,
	0x00, 0x00, 0x44, 0x6c, 0x54, 0x44, 0x44, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x44, 0x44, 0x38, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x7c, 0x44, 0x44, 0x00,
	0x00, 0x00, 0x7c, 0x44, 0x44, 0x44, 0x44, 0x00,
	0x00, 0x00, 0x78, 0x24, 0x24, 0x38, 0x20, 0x70,
	0x00, 0x00, 0x3c, 0x40, 0x40, 0x40, 0x3c, 0x00,
	0x00, 0x00, 0x7c, 0x54, 0x10, 0x10, 0x38, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x3c, 0x04, 0x38, 0x00,
	0x30, 0x10, 0x38, 0x54, 0x54, 0x38, 0x10, 0x38,
	0x00, 0x00, 0x44, 0x28, 0x10, 0x28, 0x44, 0x00,
	0x00, 0x00, 0x48, 0x48, 0x48, 0x48, 0x3c, 0x04,
	0x00, 0x00, 0x4c, 0x48, 0x48, 0x38, 0x08, 0x1c,
	0x00, 0x00, 0x54, 0x54, 0x54, 0x54, 0x3c, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0e, 0x08, 0x30, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0c, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00,
	0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
};

static const byte spanish_simonFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x70, 0x70, 0x20, 0x20, 0x00, 0x20, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x50, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0x50, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x50, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2a, 0x44, 0x3a, 0x00,
	0x30, 0x30, 0x60, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3e, 0x08, 0x14, 0x00,
	0x60, 0x90, 0x90, 0xa0, 0x90, 0x90, 0xa0, 0x80,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x60,
	0x00, 0x00, 0x00, 0xf0, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x30, 0x30, 0x00,
	0x20, 0x40, 0x00, 0x70, 0xf8, 0x80, 0x70, 0x00,
	0x70, 0x88, 0x98, 0xa8, 0xc8, 0x88, 0x70, 0x00,
	0x20, 0x60, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x40, 0x88, 0xf8, 0x00,
	0x70, 0x88, 0x08, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x10, 0x30, 0x50, 0x90, 0xf8, 0x10, 0x38, 0x00,
	0xf8, 0x80, 0xf0, 0x08, 0x08, 0x88, 0x70, 0x00,
	0x30, 0x40, 0x80, 0xf0, 0x88, 0x88, 0x70, 0x00,
	0xf8, 0x88, 0x08, 0x10, 0x20, 0x20, 0x20, 0x00,
	0x70, 0x88, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x70, 0x88, 0x88, 0x78, 0x08, 0x10, 0x60, 0x00,
	0x00, 0x00, 0x30, 0x30, 0x00, 0x30, 0x30, 0x00,
	0x20, 0x40, 0x00, 0x70, 0x08, 0xf8, 0x78, 0x00,
	0x20, 0x40, 0x00, 0xc0, 0x40, 0x40, 0xe0, 0x00,
	0x20, 0x40, 0x00, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x20, 0x40, 0x00, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x70, 0x88, 0x08, 0x10, 0x20, 0x00, 0x20, 0x00,
	0x50, 0xa0, 0x00, 0xf0, 0x88, 0x88, 0x88, 0x00,
	0x70, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xf0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xe0, 0x50, 0x48, 0x48, 0x48, 0x50, 0xe0, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x48, 0xf8, 0x00,
	0xf8, 0x48, 0x40, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x30, 0x48, 0x80, 0x98, 0x88, 0x48, 0x38, 0x00,
	0x88, 0x88, 0x88, 0xf8, 0x88, 0x88, 0x88, 0x00,
	0xf8, 0x20, 0x20, 0x20, 0x20, 0x20, 0xf8, 0x00,
	0x18, 0x08, 0x08, 0x08, 0x88, 0x88, 0x70, 0x00,
	0xc8, 0x48, 0x50, 0x60, 0x50, 0x48, 0xc8, 0x00,
	0xe0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xf8, 0x00,
	0x88, 0xd8, 0xa8, 0xa8, 0x88, 0x88, 0x88, 0x00,
	0x88, 0xc8, 0xa8, 0x98, 0x88, 0x88, 0x88, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xf0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xe0, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0xa8, 0x70, 0x08,
	0xf0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xc8, 0x00,
	0x70, 0x88, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0xf8, 0xa8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x88, 0x88, 0x88, 0x88, 0x88, 0x78, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x20, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xa8, 0xd8, 0x88, 0x00,
	0x88, 0x88, 0x50, 0x20, 0x50, 0x88, 0x88, 0x00,
	0x88, 0x88, 0x88, 0x70, 0x20, 0x20, 0x70, 0x00,
	0xf8, 0x88, 0x10, 0x20, 0x40, 0x88, 0xf8, 0x00,
	0x00, 0x0e, 0x08, 0x08, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x20, 0x00, 0x20, 0x40, 0x80, 0x88, 0x70, 0x00,
	0x20, 0x00, 0x20, 0x20, 0x70, 0x70, 0x20, 0x00,
	0x50, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x08, 0x78, 0x88, 0x78, 0x00,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x80, 0x88, 0x70, 0x00,
	0x18, 0x10, 0x50, 0xb0, 0x90, 0x90, 0x70, 0x00,
	0x00, 0x00, 0x70, 0x88, 0xf8, 0x80, 0x70, 0x00,
	0x30, 0x48, 0x40, 0xe0, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x88, 0x70,
	0xc0, 0x40, 0x50, 0x68, 0x48, 0x48, 0xc8, 0x00,
	0x40, 0x00, 0xc0, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x08, 0x00, 0x08, 0x08, 0x08, 0x08, 0x88, 0x70,
	0xc0, 0x40, 0x48, 0x50, 0x60, 0x50, 0xc8, 0x00,
	0xc0, 0x40, 0x40, 0x40, 0x40, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x90, 0xd8, 0xa8, 0x88, 0x88, 0x00,
	0x00, 0x00, 0xf0, 0x88, 0x88, 0x88, 0x88, 0x00,
	0x00, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x70, 0x40, 0xe0,
	0x00, 0x00, 0x68, 0x90, 0x90, 0x70, 0x10, 0x38,
	0x00, 0x00, 0xb0, 0x48, 0x48, 0x40, 0xe0, 0x00,
	0x00, 0x00, 0x78, 0x80, 0x70, 0x08, 0xf0, 0x00,
	0x40, 0x40, 0xf0, 0x40, 0x40, 0x48, 0x30, 0x00,
	0x00, 0x00, 0x90, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x50, 0x20, 0x00,
	0x00, 0x00, 0x88, 0x88, 0xa8, 0xd8, 0x90, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x88, 0x70, 0x20, 0xc0,
	0x00, 0x00, 0xf8, 0x90, 0x20, 0x48, 0xf8, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0e, 0x08, 0x30, 0x08, 0x08, 0x0e, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0c, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xf8, 0x00,
	0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc, 0xfc,
	0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0, 0xf0,
};

static const byte english_waxworksFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x20,
	0x00, 0x50, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x90, 0x00, 0x90, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2A, 0x44, 0x3A, 0x00,
	0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14, 0x00,
	0x00, 0x70, 0x88, 0xF0, 0x88, 0x88, 0xF0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x10,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x20, 0x60, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x10, 0x20, 0x48, 0xF8, 0x00,
	0x00, 0x70, 0x88, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x10, 0x20, 0x40, 0x90, 0xF8, 0x10, 0x00,
	0x00, 0xF8, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x80, 0xF0, 0x88, 0x70, 0x00,
	0x00, 0xF8, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00,
	0x00, 0x70, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x78, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x20, 0x50, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x20, 0x10, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x20, 0x50, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x3C, 0x42, 0x04, 0x08, 0x00, 0x08, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x50, 0x50, 0x70, 0x50, 0x50, 0x88, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xF0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xF0, 0x48, 0x48, 0x48, 0x48, 0x48, 0xF0, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xF8, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xE0, 0x00,
	0x38, 0x40, 0x80, 0x98, 0x88, 0x48, 0x30, 0x00,
	0xC8, 0x48, 0x48, 0x78, 0x48, 0x48, 0xC8, 0x00,
	0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x78, 0x10, 0x10, 0x10, 0x90, 0x90, 0x60, 0x00,
	0xC8, 0x50, 0x60, 0x40, 0x60, 0x50, 0xC8, 0x00,
	0xE0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xF8, 0x00,
	0x88, 0xD8, 0xA8, 0x88, 0x88, 0x88, 0x88, 0x00,
	0xC8, 0x48, 0x68, 0x58, 0x58, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xC0, 0x00,
	0x60, 0x90, 0x90, 0x90, 0xB0, 0x90, 0x68, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x50, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x80, 0x60, 0x10, 0x88, 0x70, 0x00,
	0xF8, 0xA8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x50, 0x50, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xA8, 0xD8, 0x88, 0x00,
	0x88, 0x50, 0x70, 0x20, 0x70, 0x50, 0x88, 0x00,
	0x88, 0x48, 0x30, 0x20, 0x40, 0x80, 0xC0, 0x00,
	0xF8, 0x88, 0x10, 0x20, 0x40, 0x80, 0xF8, 0x00,
	0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x60, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x40, 0x50, 0x68, 0x48, 0x48, 0xF0, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x00,
	0x10, 0x10, 0x70, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x48, 0x70, 0x40, 0x38, 0x00,
	0x10, 0x38, 0x20, 0x70, 0x20, 0x20, 0x20, 0x40,
	0x00, 0x00, 0x78, 0x90, 0x90, 0x90, 0x70, 0x10,
	0xC0, 0x40, 0x58, 0x68, 0x48, 0x48, 0xC8, 0x00,
	0x20, 0x00, 0x60, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x10, 0x18, 0x10, 0x10, 0x10, 0x10, 0x10, 0x60,
	0x40, 0x40, 0x70, 0x48, 0x70, 0x50, 0xC8, 0x00,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x50, 0x60, 0x00,
	0x00, 0x00, 0xA8, 0xF8, 0xA8, 0xA8, 0xA8, 0x00,
	0x00, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x48, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x48, 0x48, 0x30, 0x00,
	0x00, 0x00, 0xF0, 0x48, 0x48, 0x70, 0x40, 0xC0,
	0x00, 0x00, 0x70, 0x90, 0x90, 0x70, 0x18, 0x10,
	0x00, 0x00, 0xD8, 0x68, 0x40, 0x40, 0xE0, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x30, 0x08, 0x70, 0x00,
	0x20, 0x20, 0x70, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x50, 0x50, 0x20, 0x00,
	0x00, 0x00, 0xA8, 0xA8, 0xA8, 0xF8, 0xA8, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x70, 0x10, 0x60,
	0x00, 0x00, 0x78, 0x10, 0x20, 0x40, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0E, 0x08, 0x30, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0C, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x14, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0x00, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0x00,
};

static const byte french_waxworksFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x20,
	0x00, 0x50, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x10, 0x28, 0x10, 0x2A, 0x44, 0x3A, 0x00,
	0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14, 0x00,
	0x00, 0x70, 0x88, 0xF0, 0x88, 0x88, 0xF0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x10,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x20, 0x60, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x10, 0x20, 0x48, 0xF8, 0x00,
	0x00, 0x70, 0x88, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x10, 0x20, 0x40, 0x90, 0xF8, 0x10, 0x00,
	0x00, 0xF8, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x80, 0xF0, 0x88, 0x70, 0x00,
	0x00, 0xF8, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00,
	0x00, 0x70, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x78, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x20, 0x50, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x20, 0x10, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x20, 0x50, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x3C, 0x42, 0x04, 0x08, 0x00, 0x08, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x50, 0x50, 0x70, 0x50, 0x50, 0x88, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xF0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xF0, 0x48, 0x48, 0x48, 0x48, 0x48, 0xF0, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xF8, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xE0, 0x00,
	0x38, 0x40, 0x80, 0x98, 0x88, 0x48, 0x30, 0x00,
	0xC8, 0x48, 0x48, 0x78, 0x48, 0x48, 0xC8, 0x00,
	0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x78, 0x10, 0x10, 0x10, 0x90, 0x90, 0x60, 0x00,
	0xC8, 0x50, 0x60, 0x40, 0x60, 0x50, 0xC8, 0x00,
	0xE0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xF8, 0x00,
	0x88, 0xD8, 0xA8, 0x88, 0x88, 0x88, 0x88, 0x00,
	0xC8, 0x48, 0x68, 0x58, 0x58, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xC0, 0x00,
	0x60, 0x90, 0x90, 0x90, 0xB0, 0x90, 0x68, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x50, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x80, 0x60, 0x10, 0x88, 0x70, 0x00,
	0xF8, 0xA8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x50, 0x50, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xA8, 0xD8, 0x88, 0x00,
	0x88, 0x50, 0x70, 0x20, 0x70, 0x50, 0x88, 0x00,
	0x88, 0x48, 0x30, 0x20, 0x40, 0x80, 0xC0, 0x00,
	0xF8, 0x88, 0x10, 0x20, 0x40, 0x80, 0xF8, 0x00,
	0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x60, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x40, 0x50, 0x68, 0x48, 0x48, 0xF0, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x00,
	0x10, 0x10, 0x70, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x48, 0x70, 0x40, 0x38, 0x00,
	0x10, 0x38, 0x20, 0x70, 0x20, 0x20, 0x20, 0x40,
	0x00, 0x00, 0x78, 0x90, 0x90, 0x90, 0x70, 0x10,
	0xC0, 0x40, 0x58, 0x68, 0x48, 0x48, 0xC8, 0x00,
	0x20, 0x00, 0x60, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x10, 0x18, 0x10, 0x10, 0x10, 0x10, 0x10, 0x60,
	0x40, 0x40, 0x70, 0x48, 0x70, 0x50, 0xC8, 0x00,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x50, 0x60, 0x00,
	0x00, 0x00, 0xA8, 0xF8, 0xA8, 0xA8, 0xA8, 0x00,
	0x00, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x48, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x48, 0x48, 0x30, 0x00,
	0x00, 0x00, 0xF0, 0x48, 0x48, 0x70, 0x40, 0xC0,
	0x00, 0x00, 0x70, 0x90, 0x90, 0x70, 0x18, 0x10,
	0x00, 0x00, 0xD8, 0x68, 0x40, 0x40, 0xE0, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x30, 0x08, 0x70, 0x00,
	0x20, 0x20, 0x70, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x50, 0x50, 0x20, 0x00,
	0x00, 0x00, 0xA8, 0xA8, 0xA8, 0xF8, 0xA8, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x70, 0x10, 0x60,
	0x00, 0x00, 0x78, 0x10, 0x20, 0x40, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0E, 0x08, 0x30, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0C, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x14, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0x00, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0x00,
};

static const byte german_waxworksFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x20,
	0x00, 0x50, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x90, 0x00, 0x90, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2A, 0x44, 0x3A, 0x00,
	0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14, 0x00,
	0x00, 0x70, 0x88, 0xF0, 0x88, 0x88, 0xF0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x10,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x50, 0x00, 0x20, 0x50, 0x50, 0x70, 0x50, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x20, 0x60, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x10, 0x20, 0x48, 0xF8, 0x00,
	0x00, 0x70, 0x88, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x10, 0x20, 0x40, 0x90, 0xF8, 0x10, 0x00,
	0x00, 0xF8, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x80, 0xF0, 0x88, 0x70, 0x00,
	0x00, 0xF8, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00,
	0x00, 0x70, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x78, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00,
	0x50, 0x00, 0x70, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x50, 0x00, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x90, 0x00, 0x90, 0x90, 0x90, 0x60, 0x00,
	0x20, 0x50, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x3C, 0x42, 0x04, 0x08, 0x00, 0x08, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x50, 0x50, 0x70, 0x50, 0x50, 0x88, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xF0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xF0, 0x48, 0x48, 0x48, 0x48, 0x48, 0xF0, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xF8, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xE0, 0x00,
	0x38, 0x40, 0x80, 0x98, 0x88, 0x48, 0x30, 0x00,
	0xC8, 0x48, 0x48, 0x78, 0x48, 0x48, 0xC8, 0x00,
	0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x78, 0x10, 0x10, 0x10, 0x90, 0x90, 0x60, 0x00,
	0xC8, 0x50, 0x60, 0x40, 0x60, 0x50, 0xC8, 0x00,
	0xE0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xF8, 0x00,
	0x88, 0xD8, 0xA8, 0x88, 0x88, 0x88, 0x88, 0x00,
	0xC8, 0x48, 0x68, 0x58, 0x58, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xC0, 0x00,
	0x60, 0x90, 0x90, 0x90, 0xB0, 0x90, 0x68, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x50, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x80, 0x60, 0x10, 0x88, 0x70, 0x00,
	0xF8, 0xA8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x50, 0x50, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xA8, 0xD8, 0x88, 0x00,
	0x88, 0x50, 0x70, 0x20, 0x70, 0x50, 0x88, 0x00,
	0x88, 0x48, 0x30, 0x20, 0x40, 0x80, 0xC0, 0x00,
	0xF8, 0x88, 0x10, 0x20, 0x40, 0x80, 0xF8, 0x00,
	0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x60, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x40, 0x50, 0x68, 0x48, 0x48, 0xF0, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x00,
	0x10, 0x10, 0x70, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x48, 0x70, 0x40, 0x38, 0x00,
	0x10, 0x38, 0x20, 0x70, 0x20, 0x20, 0x20, 0x40,
	0x00, 0x00, 0x78, 0x90, 0x90, 0x90, 0x70, 0x10,
	0xC0, 0x40, 0x58, 0x68, 0x48, 0x48, 0xC8, 0x00,
	0x20, 0x00, 0x60, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x10, 0x18, 0x10, 0x10, 0x10, 0x10, 0x10, 0x60,
	0x40, 0x40, 0x70, 0x48, 0x70, 0x50, 0xC8, 0x00,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x50, 0x60, 0x00,
	0x00, 0x00, 0xA8, 0xF8, 0xA8, 0xA8, 0xA8, 0x00,
	0x00, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x48, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x48, 0x48, 0x30, 0x00,
	0x00, 0x00, 0xF0, 0x48, 0x48, 0x70, 0x40, 0xC0,
	0x00, 0x00, 0x70, 0x90, 0x90, 0x70, 0x18, 0x10,
	0x00, 0x00, 0xD8, 0x68, 0x40, 0x40, 0xE0, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x30, 0x08, 0x70, 0x00,
	0x20, 0x20, 0x70, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x50, 0x50, 0x20, 0x00,
	0x00, 0x00, 0xA8, 0xA8, 0xA8, 0xF8, 0xA8, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x70, 0x10, 0x60,
	0x00, 0x00, 0x78, 0x10, 0x20, 0x40, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0E, 0x08, 0x30, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0C, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x14, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0x00, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0x00,
};

static const byte italian_waxworksFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x20,
	0x00, 0x50, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2A, 0x44, 0x3A, 0x00,
	0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x10,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x20, 0x60, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x10, 0x20, 0x48, 0xF8, 0x00,
	0x00, 0x70, 0x88, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x10, 0x20, 0x40, 0x90, 0xF8, 0x10, 0x00,
	0x00, 0xF8, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x80, 0xF0, 0x88, 0x70, 0x00,
	0x00, 0xF8, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00,
	0x00, 0x70, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x78, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x20, 0x10, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x20, 0x50, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x3C, 0x42, 0x04, 0x08, 0x00, 0x08, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x50, 0x50, 0x70, 0x50, 0x50, 0x88, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xF0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xF0, 0x48, 0x48, 0x48, 0x48, 0x48, 0xF0, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xF8, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xE0, 0x00,
	0x38, 0x40, 0x80, 0x98, 0x88, 0x48, 0x30, 0x00,
	0xC8, 0x48, 0x48, 0x78, 0x48, 0x48, 0xC8, 0x00,
	0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x78, 0x10, 0x10, 0x10, 0x90, 0x90, 0x60, 0x00,
	0xC8, 0x50, 0x60, 0x40, 0x60, 0x50, 0xC8, 0x00,
	0xE0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xF8, 0x00,
	0x88, 0xD8, 0xA8, 0x88, 0x88, 0x88, 0x88, 0x00,
	0xC8, 0x48, 0x68, 0x58, 0x58, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xC0, 0x00,
	0x60, 0x90, 0x90, 0x90, 0xB0, 0x90, 0x68, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x50, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x80, 0x60, 0x10, 0x88, 0x70, 0x00,
	0xF8, 0xA8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x50, 0x50, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xA8, 0xD8, 0x88, 0x00,
	0x88, 0x50, 0x70, 0x20, 0x70, 0x50, 0x88, 0x00,
	0x88, 0x48, 0x30, 0x20, 0x40, 0x80, 0xC0, 0x00,
	0xF8, 0x88, 0x10, 0x20, 0x40, 0x80, 0xF8, 0x00,
	0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x60, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x40, 0x50, 0x68, 0x48, 0x48, 0xF0, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x00,
	0x10, 0x10, 0x70, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x48, 0x70, 0x40, 0x38, 0x00,
	0x10, 0x38, 0x20, 0x70, 0x20, 0x20, 0x20, 0x40,
	0x00, 0x00, 0x78, 0x90, 0x90, 0x90, 0x70, 0x10,
	0xC0, 0x40, 0x58, 0x68, 0x48, 0x48, 0xC8, 0x00,
	0x20, 0x00, 0x60, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x10, 0x18, 0x10, 0x10, 0x10, 0x10, 0x10, 0x60,
	0x40, 0x40, 0x70, 0x48, 0x70, 0x50, 0xC8, 0x00,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x50, 0x60, 0x00,
	0x00, 0x00, 0xA8, 0xF8, 0xA8, 0xA8, 0xA8, 0x00,
	0x00, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x48, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x48, 0x48, 0x30, 0x00,
	0x00, 0x00, 0xF0, 0x48, 0x48, 0x70, 0x40, 0xC0,
	0x00, 0x00, 0x70, 0x90, 0x90, 0x70, 0x18, 0x10,
	0x00, 0x00, 0xD8, 0x68, 0x40, 0x40, 0xE0, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x30, 0x08, 0x70, 0x00,
	0x20, 0x20, 0x70, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x50, 0x50, 0x20, 0x00,
	0x00, 0x00, 0xA8, 0xA8, 0xA8, 0xF8, 0xA8, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x70, 0x10, 0x60,
	0x00, 0x00, 0x78, 0x10, 0x20, 0x40, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0E, 0x08, 0x30, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0C, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x14, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0x00, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0x00,
};

static const byte spanish_waxworksFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x20,
	0x00, 0x50, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x90, 0x00, 0x90, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2A, 0x44, 0x3A, 0x00,
	0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14, 0x00,
	0x00, 0x70, 0x88, 0xF0, 0x88, 0x88, 0xF0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x10,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x20, 0x60, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x10, 0x20, 0x48, 0xF8, 0x00,
	0x00, 0x70, 0x88, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x10, 0x20, 0x40, 0x90, 0xF8, 0x10, 0x00,
	0x00, 0xF8, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x80, 0xF0, 0x88, 0x70, 0x00,
	0x00, 0xF8, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00,
	0x00, 0x70, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x78, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00,
	0x20, 0x40, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x20, 0x40, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x40, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x20, 0x40, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x3C, 0x42, 0x04, 0x08, 0x00, 0x08, 0x00,
	0x28, 0x50, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x00,
	0x20, 0x50, 0x50, 0x70, 0x50, 0x50, 0x88, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xF0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xF0, 0x48, 0x48, 0x48, 0x48, 0x48, 0xF0, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xF8, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xE0, 0x00,
	0x38, 0x40, 0x80, 0x98, 0x88, 0x48, 0x30, 0x00,
	0xC8, 0x48, 0x48, 0x78, 0x48, 0x48, 0xC8, 0x00,
	0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x78, 0x10, 0x10, 0x10, 0x90, 0x90, 0x60, 0x00,
	0xC8, 0x50, 0x60, 0x40, 0x60, 0x50, 0xC8, 0x00,
	0xE0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xF8, 0x00,
	0x88, 0xD8, 0xA8, 0x88, 0x88, 0x88, 0x88, 0x00,
	0xC8, 0x48, 0x68, 0x58, 0x58, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xC0, 0x00,
	0x60, 0x90, 0x90, 0x90, 0xB0, 0x90, 0x68, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x50, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x80, 0x60, 0x10, 0x88, 0x70, 0x00,
	0xF8, 0xA8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x50, 0x50, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xA8, 0xD8, 0x88, 0x00,
	0x88, 0x50, 0x70, 0x20, 0x70, 0x50, 0x88, 0x00,
	0x88, 0x48, 0x30, 0x20, 0x40, 0x80, 0xC0, 0x00,
	0xF8, 0x88, 0x10, 0x20, 0x40, 0x80, 0xF8, 0x00,
	0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x80, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x20, 0x00, 0x20, 0x40, 0x88, 0x70, 0x00,
	0x20, 0x00, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00,
	0x00, 0x90, 0x00, 0x90, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x00, 0x60, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x40, 0x50, 0x68, 0x48, 0x48, 0xF0, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x00,
	0x10, 0x10, 0x70, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x48, 0x70, 0x40, 0x38, 0x00,
	0x10, 0x38, 0x20, 0x70, 0x20, 0x20, 0x20, 0x40,
	0x00, 0x00, 0x78, 0x90, 0x90, 0x90, 0x70, 0x10,
	0xC0, 0x40, 0x58, 0x68, 0x48, 0x48, 0xC8, 0x00,
	0x20, 0x00, 0x60, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x10, 0x18, 0x10, 0x10, 0x10, 0x10, 0x10, 0x60,
	0x40, 0x40, 0x70, 0x48, 0x70, 0x50, 0xC8, 0x00,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x50, 0x60, 0x00,
	0x00, 0x00, 0xA8, 0xF8, 0xA8, 0xA8, 0xA8, 0x00,
	0x00, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x48, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x48, 0x48, 0x30, 0x00,
	0x00, 0x00, 0xF0, 0x48, 0x48, 0x70, 0x40, 0xC0,
	0x00, 0x00, 0x70, 0x90, 0x90, 0x70, 0x18, 0x10,
	0x00, 0x00, 0xD8, 0x68, 0x40, 0x40, 0xE0, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x30, 0x08, 0x70, 0x00,
	0x20, 0x20, 0x70, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x50, 0x50, 0x20, 0x00,
	0x00, 0x00, 0xA8, 0xA8, 0xA8, 0xF8, 0xA8, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x70, 0x10, 0x60,
	0x00, 0x00, 0x78, 0x10, 0x20, 0x40, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0E, 0x08, 0x30, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0C, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x14, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0x00, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0x00,
};

static const byte english_elvira1Font[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x20, 0x20, 0x20, 0x20, 0x20, 0x20, 0x00, 0x20,
	0x00, 0x50, 0x28, 0x28, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x90, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x90, 0x00, 0x90, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2A, 0x44, 0x3A, 0x00,
	0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14, 0x00,
	0x00, 0x70, 0x88, 0xF0, 0x88, 0x88, 0xF0, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x10,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x10, 0x20, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x70, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x20, 0x60, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x10, 0x20, 0x48, 0xF8, 0x00,
	0x00, 0x70, 0x88, 0x30, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x10, 0x20, 0x40, 0x90, 0xF8, 0x10, 0x00,
	0x00, 0xF8, 0x80, 0x70, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x80, 0xF0, 0x88, 0x70, 0x00,
	0x00, 0xF8, 0x08, 0x10, 0x20, 0x40, 0x40, 0x00,
	0x00, 0x70, 0x88, 0x70, 0x88, 0x88, 0x70, 0x00,
	0x00, 0x70, 0x88, 0x78, 0x08, 0x88, 0x70, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00,
	0x40, 0x20, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x20, 0x50, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x20, 0x10, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x20, 0x50, 0x00, 0x78, 0x70, 0x40, 0x38, 0x00,
	0x00, 0x3C, 0x42, 0x04, 0x08, 0x00, 0x08, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x50, 0x50, 0x70, 0x50, 0x50, 0x88, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x48, 0x48, 0xF0, 0x00,
	0x30, 0x48, 0x80, 0x80, 0x80, 0x48, 0x30, 0x00,
	0xF0, 0x48, 0x48, 0x48, 0x48, 0x48, 0xF0, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xF8, 0x00,
	0xF8, 0x40, 0x40, 0x70, 0x40, 0x40, 0xE0, 0x00,
	0x38, 0x40, 0x80, 0x98, 0x88, 0x48, 0x30, 0x00,
	0xC8, 0x48, 0x48, 0x78, 0x48, 0x48, 0xC8, 0x00,
	0x70, 0x20, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x78, 0x10, 0x10, 0x10, 0x90, 0x90, 0x60, 0x00,
	0xC8, 0x50, 0x60, 0x40, 0x60, 0x50, 0xC8, 0x00,
	0xE0, 0x40, 0x40, 0x40, 0x40, 0x48, 0xF8, 0x00,
	0x88, 0xD8, 0xA8, 0x88, 0x88, 0x88, 0x88, 0x00,
	0xC8, 0x48, 0x68, 0x58, 0x58, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x88, 0x88, 0x88, 0x88, 0x70, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x40, 0x40, 0xC0, 0x00,
	0x60, 0x90, 0x90, 0x90, 0xB0, 0x90, 0x68, 0x00,
	0xF0, 0x48, 0x48, 0x70, 0x50, 0x48, 0xC8, 0x00,
	0x70, 0x88, 0x80, 0x60, 0x10, 0x88, 0x70, 0x00,
	0xF8, 0xA8, 0x20, 0x20, 0x20, 0x20, 0x70, 0x00,
	0x88, 0x50, 0x50, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x50, 0x50, 0x50, 0x20, 0x00,
	0x88, 0x88, 0x88, 0x88, 0xA8, 0xD8, 0x88, 0x00,
	0x88, 0x50, 0x70, 0x20, 0x70, 0x50, 0x88, 0x00,
	0x88, 0x48, 0x30, 0x20, 0x40, 0x80, 0xC0, 0x00,
	0xF8, 0x88, 0x10, 0x20, 0x40, 0x80, 0xF8, 0x00,
	0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x0E, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x10, 0x30,
	0x00, 0x50, 0x00, 0x60, 0x20, 0x28, 0x30, 0x00,
	0x20, 0x10, 0x00, 0x98, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x60, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x40, 0x40, 0x50, 0x68, 0x48, 0x48, 0xF0, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x40, 0x48, 0x30, 0x00,
	0x10, 0x10, 0x70, 0x90, 0x90, 0x90, 0x68, 0x00,
	0x00, 0x00, 0x70, 0x48, 0x70, 0x40, 0x38, 0x00,
	0x10, 0x38, 0x20, 0x70, 0x20, 0x20, 0x20, 0x40,
	0x00, 0x00, 0x78, 0x90, 0x90, 0x90, 0x70, 0x10,
	0xC0, 0x40, 0x58, 0x68, 0x48, 0x48, 0xC8, 0x00,
	0x20, 0x00, 0x60, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x10, 0x18, 0x10, 0x10, 0x10, 0x10, 0x10, 0x60,
	0x40, 0x40, 0x70, 0x48, 0x70, 0x50, 0xC8, 0x00,
	0x40, 0x40, 0x40, 0x40, 0x40, 0x50, 0x60, 0x00,
	0x00, 0x00, 0xA8, 0xF8, 0xA8, 0xA8, 0xA8, 0x00,
	0x00, 0x00, 0xD8, 0x68, 0x48, 0x48, 0x48, 0x00,
	0x00, 0x00, 0x30, 0x48, 0x48, 0x48, 0x30, 0x00,
	0x00, 0x00, 0xF0, 0x48, 0x48, 0x70, 0x40, 0xC0,
	0x00, 0x00, 0x70, 0x90, 0x90, 0x70, 0x18, 0x10,
	0x00, 0x00, 0xD8, 0x68, 0x40, 0x40, 0xE0, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x30, 0x08, 0x70, 0x00,
	0x20, 0x20, 0x70, 0x20, 0x20, 0x28, 0x30, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x90, 0xE8, 0x00,
	0x00, 0x00, 0x88, 0x88, 0x50, 0x50, 0x20, 0x00,
	0x00, 0x00, 0xA8, 0xA8, 0xA8, 0xF8, 0xA8, 0x00,
	0x00, 0x00, 0x88, 0x50, 0x20, 0x50, 0x88, 0x00,
	0x00, 0x00, 0x98, 0x90, 0x90, 0x70, 0x10, 0x60,
	0x00, 0x00, 0x78, 0x10, 0x20, 0x40, 0x78, 0x00,
	0x20, 0x50, 0x00, 0x60, 0x90, 0x90, 0x60, 0x00,
	0x00, 0x0E, 0x08, 0x30, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0C, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x14, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
	0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC,
	0x00, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0xFC, 0x00,
};

static const byte english_pnFont[] = {
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x10, 0x10, 0x10, 0x10, 0x00, 0x10, 0x00,
	0x00, 0x24, 0x24, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x24, 0x7E, 0x24, 0x24, 0x7E, 0x24, 0x00,
	0x00, 0x08, 0x3E, 0x28, 0x3E, 0x0A, 0x3E, 0x08,
	0x00, 0x62, 0x64, 0x08, 0x10, 0x26, 0x46, 0x00,
	0x00, 0x10, 0x28, 0x10, 0x2A, 0x44, 0x3A, 0x00,
	0x00, 0x08, 0x10, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x00, 0x04, 0x08, 0x08, 0x08, 0x08, 0x04, 0x00,
	0x00, 0x20, 0x10, 0x10, 0x10, 0x10, 0x20, 0x00,
	0x00, 0x00, 0x14, 0x08, 0x3E, 0x08, 0x14, 0x00,
	0x00, 0x00, 0x08, 0x08, 0x3E, 0x08, 0x08, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x08, 0x08, 0x10,
	0x00, 0x00, 0x00, 0x00, 0x3E, 0x00, 0x00, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x18, 0x18, 0x00,
	0x00, 0x00, 0x02, 0x04, 0x08, 0x10, 0x20, 0x00,
	0x00, 0x3C, 0x46, 0x4A, 0x52, 0x62, 0x3C, 0x00,
	0x00, 0x18, 0x28, 0x08, 0x08, 0x08, 0x3E, 0x00,
	0x00, 0x3C, 0x42, 0x02, 0x3C, 0x40, 0x7E, 0x00,
	0x00, 0x3C, 0x42, 0x0C, 0x02, 0x42, 0x3C, 0x00,
	0x00, 0x08, 0x18, 0x28, 0x48, 0x7E, 0x08, 0x00,
	0x00, 0x7E, 0x40, 0x7C, 0x02, 0x42, 0x3C, 0x00,
	0x00, 0x3C, 0x40, 0x7C, 0x42, 0x42, 0x3C, 0x00,
	0x00, 0x7E, 0x02, 0x04, 0x08, 0x10, 0x10, 0x00,
	0x00, 0x3C, 0x42, 0x3C, 0x42, 0x42, 0x3C, 0x00,
	0x00, 0x3C, 0x42, 0x42, 0x3E, 0x02, 0x3C, 0x00,
	0x00, 0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x00,
	0x00, 0x00, 0x10, 0x00, 0x00, 0x10, 0x10, 0x20,
	0x00, 0x00, 0x04, 0x08, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x00, 0x00, 0x3E, 0x00, 0x3E, 0x00, 0x00,
	0x00, 0x00, 0x10, 0x08, 0x04, 0x08, 0x10, 0x00,
	0x00, 0x3C, 0x42, 0x04, 0x08, 0x00, 0x08, 0x00,
	0x00, 0x3C, 0x4A, 0x56, 0x5E, 0x40, 0x3C, 0x00,
	0x00, 0x3C, 0x42, 0x42, 0x7E, 0x42, 0x42, 0x00,
	0x00, 0x7C, 0x42, 0x7C, 0x42, 0x42, 0x7C, 0x00,
	0x00, 0x3C, 0x42, 0x40, 0x40, 0x42, 0x3C, 0x00,
	0x00, 0x78, 0x44, 0x42, 0x42, 0x44, 0x78, 0x00,
	0x00, 0x7E, 0x40, 0x7C, 0x40, 0x40, 0x7E, 0x00,
	0x00, 0x7E, 0x40, 0x7C, 0x40, 0x40, 0x40, 0x00,
	0x00, 0x3C, 0x42, 0x40, 0x4E, 0x42, 0x3C, 0x00,
	0x00, 0x42, 0x42, 0x7E, 0x42, 0x42, 0x42, 0x00,
	0x00, 0x3E, 0x08, 0x08, 0x08, 0x08, 0x3E, 0x00,
	0x00, 0x02, 0x02, 0x02, 0x42, 0x42, 0x3C, 0x00,
	0x00, 0x44, 0x48, 0x70, 0x48, 0x44, 0x42, 0x00,
	0x00, 0x40, 0x40, 0x40, 0x40, 0x40, 0x7E, 0x00,
	0x00, 0x42, 0x66, 0x5A, 0x42, 0x42, 0x42, 0x00,
	0x00, 0x42, 0x62, 0x52, 0x4A, 0x46, 0x42, 0x00,
	0x00, 0x3C, 0x42, 0x42, 0x42, 0x42, 0x3C, 0x00,
	0x00, 0x7C, 0x42, 0x42, 0x7C, 0x40, 0x40, 0x00,
	0x00, 0x3C, 0x42, 0x42, 0x52, 0x4A, 0x3C, 0x00,
	0x00, 0x7C, 0x42, 0x42, 0x7C, 0x44, 0x42, 0x00,
	0x00, 0x3C, 0x40, 0x3C, 0x02, 0x42, 0x3C, 0x00,
	0x00, 0xFE, 0x10, 0x10, 0x10, 0x10, 0x10, 0x00,
	0x00, 0x42, 0x42, 0x42, 0x42, 0x42, 0x3C, 0x00,
	0x00, 0x42, 0x42, 0x42, 0x42, 0x24, 0x18, 0x00,
	0x00, 0x42, 0x42, 0x42, 0x42, 0x5A, 0x24, 0x00,
	0x00, 0x42, 0x24, 0x18, 0x18, 0x24, 0x42, 0x00,
	0x00, 0x82, 0x44, 0x28, 0x10, 0x10, 0x10, 0x00,
	0x00, 0x7E, 0x04, 0x08, 0x10, 0x20, 0x7E, 0x00,
	0x00, 0x0E, 0x08, 0x08, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x00, 0x40, 0x20, 0x10, 0x08, 0x04, 0x00,
	0x00, 0x70, 0x10, 0x10, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x10, 0x38, 0x54, 0x10, 0x10, 0x10, 0x00,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0xFF,
	0x00, 0x1C, 0x22, 0x78, 0x20, 0x20, 0x7E, 0x00,
	0x00, 0x00, 0x38, 0x04, 0x3C, 0x44, 0x3C, 0x00,
	0x00, 0x20, 0x20, 0x3C, 0x22, 0x22, 0x3C, 0x00,
	0x00, 0x00, 0x1C, 0x20, 0x20, 0x20, 0x1C, 0x00,
	0x00, 0x04, 0x04, 0x3C, 0x44, 0x44, 0x3C, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x78, 0x40, 0x3C, 0x00,
	0x00, 0x0C, 0x10, 0x18, 0x10, 0x10, 0x10, 0x00,
	0x00, 0x00, 0x3C, 0x44, 0x44, 0x3C, 0x04, 0x38,
	0x00, 0x40, 0x40, 0x78, 0x44, 0x44, 0x44, 0x00,
	0x00, 0x10, 0x00, 0x30, 0x10, 0x10, 0x38, 0x00,
	0x00, 0x04, 0x00, 0x04, 0x04, 0x04, 0x24, 0x18,
	0x00, 0x20, 0x28, 0x30, 0x30, 0x28, 0x24, 0x00,
	0x00, 0x10, 0x10, 0x10, 0x10, 0x10, 0x0C, 0x00,
	0x00, 0x00, 0x68, 0x54, 0x54, 0x54, 0x54, 0x00,
	0x00, 0x00, 0x78, 0x44, 0x44, 0x44, 0x44, 0x00,
	0x00, 0x00, 0x38, 0x44, 0x44, 0x44, 0x38, 0x00,
	0x00, 0x00, 0x78, 0x44, 0x44, 0x78, 0x40, 0x40,
	0x00, 0x00, 0x3C, 0x44, 0x44, 0x3C, 0x04, 0x06,
	0x00, 0x00, 0x1C, 0x20, 0x20, 0x20, 0x20, 0x00,
	0x00, 0x00, 0x38, 0x40, 0x38, 0x04, 0x78, 0x00,
	0x00, 0x10, 0x38, 0x10, 0x10, 0x10, 0x0C, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x44, 0x44, 0x38, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x28, 0x28, 0x10, 0x00,
	0x00, 0x00, 0x44, 0x54, 0x54, 0x54, 0x28, 0x00,
	0x00, 0x00, 0x44, 0x28, 0x10, 0x28, 0x44, 0x00,
	0x00, 0x00, 0x44, 0x44, 0x44, 0x3C, 0x04, 0x38,
	0x00, 0x00, 0x7C, 0x08, 0x10, 0x20, 0x7C, 0x00,
	0x00, 0x0E, 0x08, 0x30, 0x08, 0x08, 0x0E, 0x00,
	0x00, 0x08, 0x08, 0x08, 0x08, 0x08, 0x08, 0x00,
	0x00, 0x70, 0x10, 0x0C, 0x10, 0x10, 0x70, 0x00,
	0x00, 0x14, 0x28, 0x00, 0x00, 0x00, 0x00, 0x00,
	0x3C, 0x42, 0x99, 0xA1, 0xA1, 0x99, 0x42, 0x3C,
	0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF, 0xFF,
	0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00, 0x00,
};

void AGOSEngine::windowDrawChar(WindowBlock *window, uint x, uint y, byte chr) {
	const byte *src;
	byte color, *dst;
	uint dstPitch, h, w, i;

	_videoLockOut |= 0x8000;

	Graphics::Surface *screen = _system->lockScreen();

	if (getGameType() == GType_SIMON1 || getGameType() == GType_SIMON2) {
		dst = (byte *)screen->getPixels();
		dstPitch = screen->pitch;
		h = 8;
		w = 6;

		switch (_language) {
		case Common::CZ_CZE:
			src = czech_simonFont + (chr - 32) * 8;
			break;
		case Common::RU_RUS:
			src = russian_simonFont + (chr - 32) * 8;
			break;
		case Common::PL_POL:
			src = polish_simonFont + (chr - 32) * 8;
			break;
		case Common::HE_ISR:
			src = hebrew_simonFont + (chr - 32) * 8;
			break;
		case Common::ES_ESP:
			src = spanish_simonFont + (chr - 32) * 8;
			break;
		case Common::IT_ITA:
			src = italian_simonFont + (chr - 32) * 8;
			break;
		case Common::FR_FRA:
			src = french_simonFont + (chr - 32) * 8;
			break;
		case Common::DE_DEU:
			src = german_simonFont + (chr - 32) * 8;
			break;
		case Common::EN_ANY:
			src = english_simonFont + (chr - 32) * 8;
			break;
		default:
			error("windowDrawChar: Unknown language %d", _language);
		}
	} else if (getGameType() == GType_ELVIRA2 || getGameType() == GType_WW) {
		dst = (byte *)screen->getPixels();
		dstPitch = screen->pitch;
		h = 8;
		w = 6;

		switch (_language) {
		case Common::ES_ESP:
			src = spanish_waxworksFont + (chr - 32) * 8;
			break;
		case Common::IT_ITA:
			src = italian_waxworksFont + (chr - 32) * 8;
			break;
		case Common::FR_FRA:
			src = french_waxworksFont + (chr - 32) * 8;
			break;
		case Common::DE_DEU:
			src = german_waxworksFont + (chr - 32) * 8;
			break;
		case Common::EN_ANY:
			src = english_waxworksFont + (chr - 32) * 8;
			break;
		default:
			error("windowDrawChar: Unknown language %d", _language);
		}
	} else if (getGameType() == GType_ELVIRA1) {
		dst = (byte *)screen->getPixels();
		dstPitch = screen->pitch;
		h = 8;
		w = 6;

		src = english_elvira1Font + (chr - 32) * 8;
	} else {
		dst = (byte *)screen->getPixels();
		dstPitch = screen->pitch;
		h = 8;
		w = 8;

		src = english_pnFont + (chr - 32) * 8;
	}
	dst += y * dstPitch + x + window->textColumnOffset;

	color = window->textColor;
	if (getGameType() == GType_ELVIRA2 || getGameType() == GType_WW)
		color += dst[0] & 0xF0;

	do {
		int8 b = *src++;
		i = 0;
		do {
			if (b < 0) {
				dst[i] = color;
			}

			b <<= 1;
		} while (++i != w);
		dst += dstPitch;
	} while (--h);

	_system->unlockScreen();

	_videoLockOut &= ~0x8000;
}

} // End of namespace AGOS
