//! Contains basic data about various HIR declarations.

pub mod adt;

use base_db::CrateId;
use hir_expand::{
    name::Name, AstId, ExpandResult, HirFileId, InFile, MacroCallId, MacroCallKind, MacroDefKind,
};
use intern::Interned;
use smallvec::SmallVec;
use syntax::{ast, Parse};
use triomphe::Arc;

use crate::{
    attr::Attrs,
    db::DefDatabase,
    expander::{Expander, Mark},
    item_tree::{self, AssocItem, FnFlags, ItemTree, ItemTreeId, MacroCall, ModItem, TreeId},
    macro_call_as_call_id, macro_id_to_def_id,
    nameres::{
        attr_resolution::ResolvedAttr,
        diagnostics::DefDiagnostic,
        proc_macro::{parse_macro_name_and_helper_attrs, ProcMacroKind},
        DefMap, MacroSubNs,
    },
    path::ImportAlias,
    type_ref::{TraitRef, TypeBound, TypeRef},
    visibility::RawVisibility,
    AssocItemId, AstIdWithPath, ConstId, ConstLoc, ExternCrateId, FunctionId, FunctionLoc,
    HasModule, ImplId, Intern, ItemContainerId, ItemLoc, Lookup, Macro2Id, MacroRulesId, ModuleId,
    ProcMacroId, StaticId, TraitAliasId, TraitId, TypeAliasId, TypeAliasLoc,
};

#[derive(Debug, Clone, PartialEq, Eq)]
pub struct FunctionData {
    pub name: Name,
    pub params: Box<[Interned<TypeRef>]>,
    pub ret_type: Interned<TypeRef>,
    pub attrs: Attrs,
    pub visibility: RawVisibility,
    pub abi: Option<Interned<str>>,
    pub legacy_const_generics_indices: Box<[u32]>,
    pub rustc_allow_incoherent_impl: bool,
    flags: FnFlags,
}

impl FunctionData {
    pub(crate) fn fn_data_query(db: &dyn DefDatabase, func: FunctionId) -> Arc<FunctionData> {
        let loc = func.lookup(db);
        let krate = loc.container.module(db).krate;
        let item_tree = loc.id.item_tree(db);
        let func = &item_tree[loc.id.value];
        let visibility = if let ItemContainerId::TraitId(trait_id) = loc.container {
            trait_vis(db, trait_id)
        } else {
            item_tree[func.visibility].clone()
        };

        let crate_graph = db.crate_graph();
        let cfg_options = &crate_graph[krate].cfg_options;
        let enabled_params = func
            .params
            .clone()
            .filter(|&param| item_tree.attrs(db, krate, param.into()).is_cfg_enabled(cfg_options));

        // If last cfg-enabled param is a `...` param, it's a varargs function.
        let is_varargs = enabled_params
            .clone()
            .next_back()
            .map_or(false, |param| item_tree[param].type_ref.is_none());

        let mut flags = func.flags;
        if is_varargs {
            flags |= FnFlags::IS_VARARGS;
        }
        if flags.contains(FnFlags::HAS_SELF_PARAM) {
            // If there's a self param in the syntax, but it is cfg'd out, remove the flag.
            let is_cfgd_out = match func.params.clone().next() {
                Some(param) => {
                    !item_tree.attrs(db, krate, param.into()).is_cfg_enabled(cfg_options)
                }
                None => {
                    stdx::never!("fn HAS_SELF_PARAM but no parameters allocated");
                    true
                }
            };
            if is_cfgd_out {
                cov_mark::hit!(cfgd_out_self_param);
                flags.remove(FnFlags::HAS_SELF_PARAM);
            }
        }

        let attrs = item_tree.attrs(db, krate, ModItem::from(loc.id.value).into());
        let legacy_const_generics_indices = attrs
            .by_key("rustc_legacy_const_generics")
            .tt_values()
            .next()
            .map(parse_rustc_legacy_const_generics)
            .unwrap_or_default();
        let rustc_allow_incoherent_impl = attrs.by_key("rustc_allow_incoherent_impl").exists();

        Arc::new(FunctionData {
            name: func.name.clone(),
            params: enabled_params
                .clone()
                .filter_map(|id| item_tree[id].type_ref.clone())
                .collect(),
            ret_type: func.ret_type.clone(),
            attrs: item_tree.attrs(db, krate, ModItem::from(loc.id.value).into()),
            visibility,
            abi: func.abi.clone(),
    