% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/ports.R
\name{freePort}
\alias{freePort}
\title{Find a TCP port that can be opened}
\usage{
freePort(ports = 1024:65535, default = "random", randomize = TRUE)
}
\arguments{
\item{ports}{(integer vector, or character string)
Zero or more TCP ports in [0, 65535] to scan.
If \code{"random"}, then a random set of ports is considered.
If \code{"auto"}, then the port given by environment variable
\env{R_PARALLEL_PORT} is used, which may also specify
\code{"random"}. If \env{R_PARALLEL_PORT} is not set, the default
is \code{"random"}. The \code{"auto"} alternative is used for full
backward compatibility with the \strong{parallelly} package.}

\item{default}{(integer) \code{NA_integer_} or a port to returned if
an available port could not be found.
If \code{"first"}, then \code{ports[1]}.  If \code{"random"}, then a random port
among \code{ports} is used. If \code{length(ports) == 0}, then \code{NA_integer_}.}

\item{randomize}{(logical) If TRUE, \code{ports} is randomly shuffled
before searched.  This shuffle does \emph{not} forward the RNG seed.}
}
\value{
Returns an integer representing the first port among \code{ports} \emph{that
can be opened}. If none can be opened, then \code{default} is returned.
}
\description{
Find a TCP port that can be opened
}
\section{Validation of TCP port returned}{

This function attempts to validate that the TCP port returned is
available and can be be opened. Thus, it is likely that this port
is available momentarily after \code{freePort()} returns. However, due
to race conditions, there is always a risk that the port is taken
by another process on the same system after \code{freePort()} returns
but before, say, \code{makeClusterPSOCK()}, opens it.
}

\section{Controlling the random set of TCP ports}{

For backward compatibility with the \strong{parallel} package, the default
set of of ports \code{freePort("random")}, or \code{freePort("auto")} with
environment variable \code{R_PARALLEL_PORT=random} samples from is
\code{11000:11999}.
The set can be overriden by setting environment variable
\code{R_PARALLELLY_RANDOM_PORTS},  e.g.
\code{R_PARALLELLY_RANDOM_PORTS="10001:10010,11000:54000"}.
}

\examples{
# Make sure to undo any changes at the end
oenv <- Sys.getenv("R_PARALLELLY_RANDOM_PORTS")

# Get a random, free TCP port in 1024:65535
port <- freePort()
message("A free TCP port: ", port)

# Get a random, free TCP port in 11000:11999, which is what
# parallelly::makeClusterPSOCK() and parallel::makePSOCKcluster()
# default to (but the latter does not make sure it is available)
Sys.unsetenv("R_PARALLELLY_RANDOM_PORTS")
port <- freePort("random")
message("A free TCP port: ", port)

# Customize the range of ports to sample from to 30000:50000
Sys.setenv(R_PARALLELLY_RANDOM_PORTS = "30000:50000")
port <- freePort("random")
message("A free TCP port: ", port)

# Test if a specific port is free
isPortFree <- function(port) !is.na(freePort(port, default = NA_integer_))
message("TCP port 1024 is free: ", isPortFree(1024))

# Undo changes
Sys.setenv(R_PARALLELLY_RANDOM_PORTS = oenv)
}
