% Generated by roxygen2: do not edit by hand
% Please edit documentation in R/tags.R
\name{tagAddRenderHook}
\alias{tagAddRenderHook}
\title{Modify a tag prior to rendering}
\usage{
tagAddRenderHook(tag, func, replace = FALSE)
}
\arguments{
\item{tag}{A \code{\link[=tag]{tag()}} object.}

\item{func}{A function (\emph{hook}) to call when the \code{tag} is rendered. This function
should have at least one argument (the \code{tag}) and return anything that can
be converted into tags via \code{\link[=as.tags]{as.tags()}}.}

\item{replace}{If \code{TRUE}, the previous hooks will be removed. If \code{FALSE},
\code{func} is appended to the previous hooks.}
}
\value{
A \code{\link[=tag]{tag()}} object with a \code{.renderHooks} field containing a list of functions
(e.g. \code{func}). When the return value is \emph{rendered} (such as with \code{\link[=as.tags]{as.tags()}}),
these functions will be called just prior to writing the HTML.
}
\description{
Adds a hook to call on a \code{\link[=tag]{tag()}} object when it is is rendered as HTML (with,
for example, \code{\link[=print]{print()}}, \code{\link[=renderTags]{renderTags()}}, \code{\link[=as.tags]{as.tags()}}, etc).
}
\details{
The primary motivation for \code{\link[=tagAddRenderHook]{tagAddRenderHook()}} is to create tags that can
change their attributes (e.g., change CSS classes) depending upon the context
in which they're rendered (e.g., use one set of CSS classes in one a page
layout, but a different set in another page layout). In this situation,
\code{\link[=tagAddRenderHook]{tagAddRenderHook()}} is preferable to \code{\link[=tagFunction]{tagFunction()}} since the latter is more a
"black box" in the sense that you don't know anything about the tag structure
until it's rendered.
}
\examples{
# Have a place holder div and return a span instead
obj <- div("example", .renderHook = function(x) {
  x$name <- "span"
  x
})
obj$name # "div"
print(obj) # Prints as a `span`

# Add a class to the tag
# Should print a `span` with class `"extra"`
spanExtra <- tagAddRenderHook(obj, function(x) {
  tagAppendAttributes(x, class = "extra")
})
spanExtra

# Replace the previous render method
# Should print a `div` with class `"extra"`
divExtra <- tagAddRenderHook(obj, replace = TRUE, function(x) {
  tagAppendAttributes(x, class = "extra")
})
divExtra

# Add more child tags
spanExtended <- tagAddRenderHook(obj, function(x) {
  tagAppendChildren(x, " ", tags$strong("bold text"))
})
spanExtended

# Add a new html dependency
newDep <- tagAddRenderHook(obj, function(x) {
  fa <- htmlDependency(
    "font-awesome", "4.5.0", c(href="shared/font-awesome"),
    stylesheet = "css/font-awesome.min.css")
  attachDependencies(x, fa, append = TRUE)
})
# Also add a jqueryui html dependency
htmlDependencies(newDep) <- htmlDependency(
  "jqueryui", "1.11.4", c(href="shared/jqueryui"),
  script = "jquery-ui.min.js")
# At render time, both dependencies will be found
renderTags(newDep)$dependencies

# Ignore the original tag and return something completely new.
newObj <- tagAddRenderHook(obj, function(x) {
  tags$p("Something else")
})
newObj
}
\seealso{
\code{\link[=tagFunction]{tagFunction()}}
}
