\name{fracdiff.sim}
\alias{fracdiff.sim}
\title{Simulate fractional ARIMA Time Series}
\description{
  Generates simulated long-memory time series data from the
  fractional ARIMA(p,d,q) model.  This is a test problem generator for
  \code{\link{fracdiff}}.

  Note that the MA coefficients have \emph{inverted} signs
  compared to other parametrizations, see the details in
  \code{\link{fracdiff}}.
}
\usage{
fracdiff.sim(n, ar = NULL, ma = NULL, d,
             rand.gen = rnorm, innov = rand.gen(n+q, ...),
             n.start = NA, backComp = TRUE, allow.0.nstart = FALSE,
             start.innov = rand.gen(n.start, ...),
             ..., mu = 0)
}
\arguments{
  \item{n}{length of the time series.}
  \item{ar}{vector of autoregressive parameters; empty by default.}
  \item{ma}{vector of moving average parameters; empty by default.}
  \item{d}{fractional differencing parameter.}
  \item{rand.gen}{a function to generate the innovations; the default,
    \code{\link{rnorm}} generates white N(0,1) noise.}
  \item{innov}{an optional times series of innovations.  If not
    provided, \code{rand.gen()} is used.}
  \item{n.start}{length of \dQuote{burn-in} period.  If \code{NA}, the
    default, the same value as in \code{\link{arima.sim}} is computed.}
  \item{backComp}{logical indicating if back compatibility with older
    versions of \code{fracdiff.sim} is desired.  Otherwise, for
    \code{d = 0}, compatibility with \R's \code{\link{arima.sim}} is
    achieved.}
  \item{allow.0.nstart}{logical indicating if \code{n.start = 0} should
    be allowed even when \eqn{p + q > 0}.  This not recommended unless
    for producing the same series as with older versions of
    \code{fracdiff.sim}.}
  \item{start.innov}{an optional vector of innovations to be used for
    the burn-in period.  If supplied there must be at least
    \code{n.start} values.}
  \item{\dots}{additional arguments for \code{rand.gen()}.  Most usefully,
    the standard deviation of the innovations generated by \code{rnorm}
    can be specified by \code{sd}.}
  \item{mu}{time series mean (added at the end).}
}
\value{
  a list containing the following elements :
  \item{series}{time series}
  \item{ar, ma, d, mu, n.start}{same as input}
}
\seealso{
  \code{\link{fracdiff}}, also for references;
  \code{\link[stats]{arima.sim}}
}
\examples{
## Pretty (too) short to "see" the long memory
fracdiff.sim(100, ar = .2, ma = .4, d = .3)

## longer with "extreme" ar:
r <- fracdiff.sim(n=1500, ar=-0.9, d= 0.3)
plot(as.ts(r$series))

## Show that  MA  coefficients meaning is inverted
## compared to   stats :: arima :

AR <- 0.7
MA <- -0.5
n.st <- 2

AR <- c(0.7, -0.1)
MA <- c(-0.5, 0.4)
n <- 512 ; sd <- 0.1
n.st <- 10

set.seed(101)
Y1 <- arima.sim(list(ar = AR, ma = MA), n = n, n.start = n.st, sd = sd)
plot(Y1)

# For our fracdiff, reverse the MA sign:
set.seed(101)
Y2 <- fracdiff.sim(n = n, ar = AR, ma = - MA, d = 0,
                   n.start = n.st, sd = sd)$series
lines(Y2, col=adjustcolor("red", 0.5))
## .. no, you don't need glasses ;-)  Y2 is Y1 shifted slightly

##' rotate left by k (k < 0: rotate right)
rot <- function(x, k) {
  stopifnot(k == round(k))
  n <- length(x)
  if(k <- k \%\% n) x[c((k+1):n, 1:k)] else x
}
k <- n.st - 2
Y2.s <- rot(Y2, k)
head.matrix(cbind(Y1, Y2.s))
plot(Y1, Y2.s); i <- (n-k+1):n
text(Y1[i], Y2.s[i], i, adj = c(0,0)-.1, col=2)

## With  backComp = FALSE,  get *the same* as arima.sim():
set.seed(101)
Y2. <- fracdiff.sim(n = n, ar = AR, ma = - MA, d = 0,
                    n.start = n.st, sd = sd, backComp = FALSE)$series
stopifnot( all.equal( c(Y1), Y2., tolerance= 1e-15))
}
\keyword{ts}
