
##############################################################################
#
# Copyright (c) 2003-2018 by The University of Queensland
# http://www.uq.edu.au
#
# Primary Business: Queensland, Australia
# Licensed under the Apache License, version 2.0
# http://www.apache.org/licenses/LICENSE-2.0
#
# Development until 2012 by Earth Systems Science Computational Center (ESSCC)
# Development 2012-2013 by School of Earth Sciences
# Development from 2014 by Centre for Geoscience Computing (GeoComp)
#
##############################################################################

from __future__ import print_function, division

__copyright__="""Copyright (c) 2003-2018 by The University of Queensland
http://www.uq.edu.au
Primary Business: Queensland, Australia"""
__license__="""Licensed under the Apache License, version 2.0
http://www.apache.org/licenses/LICENSE-2.0"""
__url__="https://launchpad.net/escript-finley"

"""
test for unary operations without tagged data

:remark: use see `test_util`
:var __author__: name of author
:var __copyright__: copyrights
:var __license__: licence agreement
:var __url__: url entry point on documentation
:var __version__: version
:var __date__: date of the version
"""

__author__="Lutz Gross, l.gross@uq.edu.au"

import esys.escriptcore.utestselect as unittest
import numpy
from esys.escript import *
from test_util_base import Test_util_base

class Test_util_binary_no_tagged_data(Test_util_base):
   """
   test for the binary operations inner, ... without tagged data
   """
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_float_rank0_float_rank0(self):
      arg0=-0.63638833874
      arg1=0.675249449251
      res=inner(arg0,arg1)
      ref=-0.42972087524346775
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      res=inner(arg0,arg1*1j)
      self.assertTrue(isinstance(res,complex),"wrong type of result.")
      self.assertTrue(Lsup(res-ref*1j)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_float_rank0_array_rank0(self):
      arg0=-0.0120734848986
      arg1=numpy.array(0.186721047866)
      res=inner(arg0,arg1)
      ref=numpy.array(-0.0022543737516678149)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_float_rank0_constData_rank0(self):
      arg0=0.416970345798
      arg1=Data(-0.0686904741385,self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(-0.028641890754520932,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg1=arg1*1j
      res=inner(arg0,arg1)
      ref=Data(-0.028641890754520932j,self.functionspace) 
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_float_rank0_expandedData_rank0(self):
      arg0=-0.702135729267
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.388305302896)+(1.-msk_arg1)*(0.618444759567)
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(-0.27264302702700538)+(1.-msk_ref)*numpy.array(-0.43423216227036321)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg1.promote()
        res=inner(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank0_float_rank0(self):
      arg0=numpy.array(-0.172668075783)
      arg1=-0.72035494477
      res=inner(arg0,arg1)
      ref=numpy.array(0.12438230219442528)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array(-0.172668075783j)
      res=inner(arg0,arg1)
      ref=ref*1j
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result."+str(type(res)))
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank0_array_rank0(self):
      arg0=numpy.array(0.367217155332)
      arg1=numpy.array(0.353427284375)
      res=inner(arg0,arg1)
      ref=numpy.array(0.12978456198487195)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array(0.367217155332j)
      arg1=numpy.array(0.353427284375j)
      res=inner(arg0,arg1)
      ref=numpy.array(-0.12978456198487195)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank1_array_rank1(self):
      arg0=numpy.array([0.17057698496056495, 0.65197508237616231])
      arg1=numpy.array([0.32087931119793067, 0.47355654012965243])
      res=inner(arg0,arg1)
      ref=numpy.array(0.363481689701)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array([0.17057698496056495j, 0.65197508237616231j])
      arg1=numpy.array([0.32087931119793067j, 0.47355654012965243j])
      res=inner(arg0,arg1)
      ref=numpy.array(-0.363481689701)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank2_array_rank2(self):
      arg0=numpy.array([[-0.035529960222147716, 0.99948208956536089, 0.29174060409603397, 0.80304088757032743, 
-0.22356450624851254], [0.49066384343149982, 0.079758333795518332, -0.99232738609739557, 0.25810714747140073, 
-0.29886716922556156], [-0.69989595642288926, -0.64245939648716655, 0.16655367423389711, -0.95321218749485381, 
0.21638802862009299], [-0.47773968220001661, -0.23536079683934341, 0.30508717152899334, -0.14545129929727274, 
0.84846996077931758]])
      arg1=numpy.array([[0.71176134831396065, -0.11756983608048777, -0.38521912408149217, 0.27444535320199082, 
0.57303910984221029], [-0.2450861922400549, -0.53694004187819711, 0.45031452258385873, 0.45834369692056742, 
-0.3311126283689303], [0.63516776015807475, -0.60784580998742488, -0.25877736622125513, 0.55939114578330851, 
-0.89510238302072809], [-0.73226548502549438, -0.45276749570816377, 0.0051676384025589606, -0.63694640946938041, 
0.50513135858382707]])
      res=inner(arg0,arg1)
      ref=numpy.array(-0.400418822582)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=inner(arg0,arg1)
      ref=numpy.array(0.400418822582)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank3_array_rank3(self):
      arg0=numpy.array([[[0.89352907932481784, -0.2813007089590529], [-0.69957301452894849, 0.17341921348802947]], 
[[0.054935566453697016, 0.26169631254615577], [-0.58304701755746646, -0.57114551586649376]], [[0.47352481574743033, 
-0.048416145171658709], [0.93223229353677217, -0.454772692829994]], [[-0.32462211909078809, 0.089935113935614819], 
[-0.286419458027239, 0.86148561956606584]], [[-0.46565312018771388, 0.68711061108884097], [-0.80180026607064825, 
-0.2937284863457541]], [[0.25592134854363269, -0.76990158619153037], [-0.5453961772646585, -0.3664968003943212]]])
      arg1=numpy.array([[[0.32367504239132705, 0.14792860294053156], [0.6280980890935608, 0.76580645778924739]], 
[[0.011893082020882684, -0.82206139884528118], [0.52472067780122855, -0.98039256498560623]], [[0.73565201035010919, 
0.59105957526157393], [0.26717809906364232, -0.018751742395435533]], [[-0.54650947408049366, 0.40974338641807573], 
[-0.5411625415474377, -0.55103964175712683]], [[0.17755009800646215, -0.9728086543767589], [0.63120223112293616, 
-0.39282275192220806]], [[-0.51566477827268198, 0.54812090272268477], [-0.7413890093151918, -0.73686610466046076]]])
      res=inner(arg0,arg1)
      ref=numpy.array(-0.568960512795)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank4_array_rank4(self):
      arg0=numpy.array([[[[0.21900292911625985, -0.60296685402518957, -0.036182884002972315, 0.43073749211345658], 
[0.17662974671291853, 0.47378352574048255, 0.13148163346860686, -0.9075698563504826], [-0.26566039804635011, 
-0.16850033546278254, 0.07451540452840133, -0.35668991292063268]], [[0.90587343082002314, -0.55800425115866381, 
0.82036937796381881, -0.52043951259439081], [0.84970150242946874, -0.92369260371025752, -0.1994015294909357, 
-0.3565446913147281], [0.52982924316795788, 0.35685085558316776, 0.8621190418377902, -0.86631488594347084]]], 
[[[-0.055899301832907478, -0.62636120778693516, -0.98921303748822575, -0.70197234469007541], [-0.40991408807660368, 
-0.053106017924667848, -0.066485118669267873, -0.024378451385486777], [-0.9243594466057472, -0.96050256112690779, 
-0.64964483254915906, -0.99257146022549003]], [[-0.53042944297798789, -0.3730158086125106, 0.21336738115690945, 
0.50992862268150629], [0.3421173202861485, -0.22668604447876262, 0.27710925912205209, -0.22342248055928193], 
[0.94642089014990094, -0.86328145577679227, 0.026956203656329336, -0.19213054671644891]]], [[[0.5092821302248649, 
0.66099009757380012, 0.27398216648635487, 0.79901877176611036], [0.94090012810807888, -0.72354491228080819, 
-0.96828772385126149, -0.22141913324735873], [-0.48542816727608407, -0.76982959274647911, -0.3068381860033953, 
0.2572561109009035]], [[0.52546472411875222, 0.47516779288045119, -0.63172905926969736, -0.65471479410742206], 
[0.17532321209404156, -0.87251359508673287, -0.041241206248513462, 0.14482583937594073], [-0.70481434929401998, 
0.5951965426459036, 0.19848055508804241, -0.90212192321127316]]]])
      arg1=numpy.array([[[[0.40293190800871881, -0.52178068293538615, -0.0010302315027994613, -0.69940520456709776], 
[0.51617551422053642, -0.61106171553887001, -0.50747804751707726, -0.13235539501135385], [0.77055869833341051, 
0.31268222395947043, 0.98450151992025936, -0.24145694940836915]], [[0.10755497690663995, 0.93179498417987583, 
0.76190767263619685, 0.29396668842920293], [0.49604049008825579, -0.92333650195578332, -0.027861799684368416, 
0.60296582674521226], [0.60520530056161781, 0.49193959262973541, -0.98823171213192729, 0.26461406021984279]]], 
[[[0.88914934413791347, -0.1774138189223089, 0.58911406224725549, -0.79660932378069171], [0.12292073845561458, 
-0.11657999170114852, -0.80012642473751705, 0.88884685738952696], [-0.66836072518944745, -0.14373852043843982, 
0.99356869621344757, 0.82527812224016284]], [[0.94254195204021229, 0.61233093725048682, 0.8263349635204229, 
0.8169097790734059], [0.67347372348904333, 0.13773189624189031, -0.26566944449497609, 0.73618620492089559], 
[-0.1682516688141209, -0.65535861723124555, 0.12215518357441701, -0.56159828576022308]]], [[[0.23636310358669732, 
-0.36815150321797252, 0.21433798217009592, 0.44508327099916811], [-0.075644838498539313, -0.22190339086905841, 
-0.062431398652471382, 0.87125454618723563], [0.6058326030522907, 0.99421458438737109, -0.95649568451053169, 
-0.70291984701882626]], [[0.36370399365467687, -0.061169008055809604, -0.033210650166190181, 0.67962607885974613], 
[0.94337960804017129, -0.31509433991294244, -0.044306808278911758, 0.95119341821522396], [0.38655537876329316, 
-0.29896841578308009, 0.68221891453643635, 0.23502327204928264]]]])
      res=inner(arg0,arg1)
      ref=numpy.array(-0.859513882924)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=inner(arg0,arg1)
      ref=-ref
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank0_constData_rank0(self):
      arg0=numpy.array(-0.810401259044)
      arg1=Data(-0.325113918466,self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(0.26347272885752965,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array(-0.810401259044j)
      arg1=Data(-0.325113918466j,self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(-0.26347272885752965,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank1_constData_rank1(self):
      arg0=numpy.array([-0.070866698348063917, 0.050603972040196954])
      arg1=Data(numpy.array([0.82981473915326354, -0.90417609429403045]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(-0.104561132599,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank2_constData_rank2(self):
      arg0=numpy.array([[-0.012332909244574752, -0.43045347177805371, -0.56902413378307704, -0.61290486950010425, 
-0.43154713220921792], [-0.19526684671168137, -0.47024244784236813, 0.77901986141687618, -0.71109341477698207, 
-0.32125703210104373], [-0.52994874156774552, 0.91638894385906422, 0.4049789497535381, -0.92738222662475445, 
0.93783732431033617], [-0.6882772029887636, -0.4796143837430713, 0.34926097584118532, -0.35815861799291304, 
-0.83700114852955054]])
      arg1=Data(numpy.array([[0.82605893992853674, -0.43129145518369238, 0.42275995124864907, 0.79933162121258117, 
0.4565204718350151], [-0.43642891929309124, 0.32815803774379337, 0.88547286455610918, 0.47079208955831264, 0.729559199963703], 
[-0.35223043377128738, 0.4159016971778966, -0.38353234059288832, -0.31502231691509142, 0.31931352402769808], 
[0.98254332028266433, 0.34217713422490603, 0.7072477440265148, -0.83027415700191431, 0.28019515667625261]]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(-0.226903257331,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=inner(arg0,arg1)
      ref=Data(0.226903257331,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank3_constData_rank3(self):
      arg0=numpy.array([[[0.10241974677245125, -0.5612793405960681], [0.48956883482708147, 0.21824369883334827]], 
[[-0.94208184708149623, -0.051386571005039627], [-0.91342350452431909, -0.25978647540698407]], [[-0.5951270487657907, 
-0.27571085909871629], [-0.73723013197562359, 0.317886957344657]], [[-0.45037371104994639, -0.029486570998041861], 
[-0.82789139856903593, -0.94832948730039912]], [[0.27098807734732322, -0.89513169154431971], [-0.24930925111402669, 
0.97881744824711525]], [[0.08702563256538931, 0.23178250733209138], [-0.67548328336784969, -0.36820646116643152]]])
      arg1=Data(numpy.array([[[-0.31373964435579982, 0.017883438358496484], [0.6888236499822904, 0.61142611013031667]], 
[[-0.03634678500533739, -0.63105984256155212], [-0.25953474206005089, -0.016239653369635842]], [[0.24128973568149914, 
-0.21787692878990894], [-0.57616711332523507, -0.97817129188864271]], [[0.009337648257052189, 0.49174072868617746], 
[-0.15474415928388274, -0.92936833360630899]], [[-0.70978657661432498, 0.87557995989779003], [-0.89720836523803316, 
0.67329579802870576]], [[0.26372536245876654, -0.76542154521883288], [0.58655422134985846, 
0.6979828223763247]]]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(0.856438930143,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=inner(arg0,arg1)
      ref=-ref
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank4_constData_rank4(self):
      arg0=numpy.array([[[[0.016639529837428668, 0.84838979094454325, -0.50467114288267378, -0.54181350086674529], 
[-0.54620830647663876, -0.31499053555167711, -0.48094938871622994, 0.27646333695070058], [0.82724160718728901, 
0.1737915495866984, -0.34238117387678146, -0.60400537575240221]], [[-0.36746819329091229, -0.64176561322584069, 
0.52245880667087774, 0.73688903719870424], [0.3179495016926901, 0.85279408078258245, -0.60047774438704749, 
0.15873638993823813], [0.5076892764693155, 0.84293811414455178, 0.48925585544262318, -0.82237799915319121]]], 
[[[0.25242600532246828, 0.98620415816453733, -0.67524390609349005, 0.51688560582684695], [-0.69177865939521888, 
0.63232405584020346, 0.0023024450747841652, -0.79237123558851374], [-0.75432092404493756, 0.79156821576387371, 
0.053607107332242165, -0.95012768912242951]], [[0.88260562556496636, 0.99069034151790114, -0.53215163846733593, 
0.3971989742729396], [-0.85212374540238023, 0.50628972942019668, -0.29218975010767423, -0.90647005705651607], 
[0.13419889681450714, -0.68500671786361833, 0.18920196138340883, -0.16974207058474455]]], [[[0.61164050051299412, 
0.36642578193074971, 0.37341536941942066, -0.82463585738792067], [-0.08223679232670289, 0.0040037760019404622, 
-0.016299163145779616, 0.84470568309483696], [0.75150985970177553, -0.1990527406518785, 0.35874622129606437, 
0.22351265216298954]], [[-0.10960282976903146, -0.035365424124620626, 0.91643455408768437, -0.32048481948855656], 
[0.59775606736847942, -0.60614780477219266, 0.71467667645915967, -0.93549262644917519], [0.41148771993312305, 
0.17822720261692204, -0.92517551904060591, 0.23574681120190943]]]])
      arg1=Data(numpy.array([[[[-0.52344546615220522, -0.45296337582365509, 0.63184857984224196, 0.42541233474254758], 
[-0.23897433488742448, 0.71055269293239243, -0.46634951228173249, 0.076642826379174744], [-0.97484132074029883, 
0.16864792330825473, 0.95635345134512728, 0.21669053973675578]], [[-0.9069669442268995, -0.21691444531397885, 
0.15933955144867706, -0.81348324723411847], [-0.31275687451151102, -0.34212629791189086, 0.30384255829756257, 
0.55895935486113646], [0.85216976110340892, 0.46078915037373602, 0.91937027831781348, -0.3482307708329897]]], 
[[[-0.45307960753759979, -0.63289481149081328, -0.63520551311755002, -0.92739929533503007], [-0.82080462816910016, 
-0.063643912885194887, 0.7430471470832356, -0.6508836953472783], [0.10170220900992355, 0.77593531995384457, 
0.91230159203931382, 0.78425954312077284]], [[0.6357266342050405, -0.12466997751467157, 0.32490859933816507, 
0.1101144072176754], [-0.23463819036749256, 0.64140028824878947, -0.81732691350469833, -0.25015974677132657], 
[-0.23974953965755108, -0.82125675953865751, 0.97770789947702896, -0.03498443568673304]]], [[[-0.077955029958820754, 
-0.43754142290157572, -0.71454906310952526, 0.33099228123870361], [-0.57320856346297955, 0.57803655157871647, 
-0.46331816432125361, -0.27964923157478383], [0.25214848873585183, -0.14481715104089021, 0.11289539377069313, 
0.089390090920334897]], [[-0.91410123483641836, -0.86128585400035518, -0.92255914908006731, 0.29388782909507194], 
[-0.27855479268337269, 0.030076820585910991, 0.98119952695768076, 0.75737484900128438], [-0.31289264565854547, 
0.04614406591056075, 0.68094161393432828, 0.28036598839232596]]]]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(-1.21336440422,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank0_expandedData_rank0(self):
      arg0=numpy.array(0.685017565314)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.074549723403)+(1.-msk_arg1)*(-0.879253747732)
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(0.051067870020334076)+(1.-msk_ref)*numpy.array(-0.60230426156491856)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array(0.685017565314j)
      if not arg1.isLazy():
        arg1=arg1*1j
        ref=-ref
        res=inner(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank1_expandedData_rank1(self):
      arg0=numpy.array([-0.33806363855530708, -0.38292988645561055])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.11856789813520696, -0.13776690018680293])+(1.-msk_arg1)*numpy.array([-0.34564715796422996, 
-0.069488375244919798])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.0126715683864)+(1.-msk_ref)*(0.14345991152)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank2_expandedData_rank2(self):
      arg0=numpy.array([[0.8235093234178823, -0.039529146873249488, 0.063372460107542938, -0.02619523847089944, 
-0.95775695528662586], [-0.34622913082399309, 0.65703970311335413, 0.13991122080937823, -0.72993544844032421, 
-0.61110494663001846], [0.94702026611810308, 0.28367545790239079, 0.67686520536412309, 0.42751460713834732, 
0.19113835646707988], [-0.75397238968468683, -0.20449122930723362, -0.21008054171449841, -0.13985634479509779, 
0.64669654227550311]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.52739632868989683, -0.36965406751101404, -0.39081726084476509, 0.97579405276136999, 
0.69376649986660421], [0.778547720225927, 0.25692204796805784, -0.57060516104094372, -0.41495750688346966, 
-0.25648389392393289], [-0.33884072596680825, 0.84308494012737856, -0.52829054733196146, 0.65321147965010384, 
0.48804191576280909], [-0.24641018217509747, 0.087863580087031101, -0.56288330676632148, -0.6022514263107599, 
-0.63532666639301216]])+(1.-msk_arg1)*numpy.array([[-0.14157133127819188, -0.54776353090035768, 0.81467492509210548, 
-0.62049750034243933, -0.37900634079761653], [0.82176313795168232, 0.42864041831927735, 0.24199983631564392, 
0.41189660175178577, 0.40908911745866505], [0.030519565802787252, -0.68547715815876176, 0.14694228902994233, 
-0.16803721604883659, -0.53884356711170156], [-0.36084080845702471, -0.2746282620691598, 0.63989515977938782, 
0.26879023564652971, 0.44513874904187345]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.0941431468163)+(1.-msk_ref)*(0.0194138324902)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        res=inner(arg0,arg1)
        ref=-ref
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank3_expandedData_rank3(self):
      arg0=numpy.array([[[0.6807678093224121, -0.57723267525409039], [0.70068737825030447, -0.098432327342198533]], 
[[0.098140634624181322, -0.54424338697813068], [0.71804851090573774, 0.29950212330472437]], [[0.42465927843511464, 
0.58260173822051264], [-0.40034392645695749, 0.27714450388518275]], [[0.18874402904939913, 0.062894210730298328], 
[-0.83705822051765666, 0.096190962372082689]], [[-0.75297233787755746, 0.45002414314126327], [-0.72288520216827656, 
0.70912440219389161]], [[0.14137295031492436, -0.34614151067367471], [0.93935035354863272, -0.83373700012705565]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-0.10440843510343933, -0.89208525680149098], [-0.62476879751924952, -0.77336672258946537]], 
[[-0.16279839471814728, -0.0072511576320637428], [0.39123164448829528, 0.29096390323811661]], [[0.43544714395872375, 
0.19856771196494227], [-0.61262553243481022, 0.92741953379021758]], [[-0.044525429250246829, -0.89645569569903594], 
[0.60174376712715039, 0.51259492479278346]], [[-0.77538544226011674, -0.68714118832869286], [-0.40232551304269637, 
0.50588918195801913]], [[-0.95544279028286105, -0.73188009487106576], [0.94088511797831398, 
0.42873552842727891]]])+(1.-msk_arg1)*numpy.array([[[-0.22185702833951537, 0.60513168028384201], [-0.6423119626413778, 
-0.58036119662568053]], [[0.30081297970455267, -0.6070556981649895], [-0.90233647395001904, 0.62817784677024058]], 
[[-0.52044268719947984, 0.11509410080733606], [-0.75548193826004817, -0.60670270362377821]], [[0.22643631677427467, 
-0.45635249547105694], [-0.32459092263105682, 0.75218467015134483]], [[-0.045797508987332813, 0.52811472165708517], 
[0.87346978951632415, -0.44939888990809318]], [[-0.17980683491748972, -0.72256211726739084], [-0.71437002134646899, 
0.82199165311469602]]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(2.29079134711)+(1.-msk_ref)*(-2.46432742627)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_array_rank4_expandedData_rank4(self):
      arg0=numpy.array([[[[0.15001435572408761, 0.88232864246715459, -0.24610469990288886, 0.67560609162045515], 
[0.86401150897253132, 0.43421666202257314, -0.86140940673783661, 0.32139301112348884], [-0.45383172852577358, 
0.52020632092451913, 0.81512898306376735, -0.9742734405822282]], [[0.98066673304755825, -0.94023788286751175, 
-0.40227847101427572, 0.35551124176897919], [0.023933521307637706, -0.53811922182304706, 0.6913422958258757, 
-0.84406449580488818], [-0.50650935800253394, 0.9426192897711656, 0.48842047540240663, 0.94440029359694688]]], 
[[[0.38346320727782701, -0.87406936246287126, 0.68998833790835068, 0.1571712550522113], [-0.33552688217408133, 
0.84039147835532146, -0.87599849362515503, 0.47001164445580623], [-0.91299234258139927, -0.17750683498323405, 
0.36632349854464397, 0.70016097477658779]], [[-0.066200692553295193, -0.058261135948909137, -0.22814373144827504, 
-0.64926256398498539], [0.26196442127305009, 0.64998457480300531, 0.50294142170873934, -0.22074066985171492], 
[-0.28710043984106881, -0.35115422025792342, -0.023340121762502664, -0.73626684924280084]]], [[[-0.63319886833240191, 
0.4626606248706171, -0.45536290831045778, -0.6161109835018006], [-0.18669620353738425, -0.34302502854744232, 
0.09082321884037392, -0.024033185577168048], [-0.64389952822168084, -0.62679870762489798, -0.23800700726467983, 
-0.93865957831833335]], [[0.32502225087125924, 0.28665512174603269, 0.29368673585583949, -0.94801857817809299], 
[-0.94387554673357887, -0.86999806926650147, 0.23024162709965412, -0.84247401709034264], [-0.26926348184769733, 
0.09437686600404227, -0.99267356825664055, -0.69396226836589103]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-0.69982398293267623, 0.41309680911744429, 0.22295573078252606, 0.065367899473801172], 
[-0.42418776970603456, 0.72498482032837264, 0.58244790217178544, -0.46788783378843202], [-0.16876136986509649, 
0.25776299059074637, 0.031694131873162279, -0.37723534572713935]], [[-0.33211559209104569, -0.23554524976887059, 
-0.60874446346290112, 0.18836992215618942], [-0.97274913731614254, 0.17533771183253855, 0.76831525607694795, 
0.36098529286068604], [-0.78559144936195024, 0.85956422754002038, 0.58705783447497617, -0.75059133979061365]]], 
[[[0.023467803523911757, -0.22375429863866336, -0.83149004680439464, 0.79185129225030026], [-0.63331631885344386, 
0.56090673973895022, 0.76276329432083534, -0.20867007167164942], [0.14833356982262336, 0.52730370024687101, 
-0.09927133106555841, -0.10049391484766845]], [[-0.1045637502068355, -0.59290766899230785, 0.81880317570587868, 
-0.7594690941697726], [0.50563811016995319, 0.7841853659235094, 0.52355731289255059, 0.012124560297168907], 
[-0.4414905530370985, 0.37628519305253927, -0.59792951426456353, 0.62279388927523338]]], [[[0.52498756740874608, 
0.85236651782771622, 0.28393668963979235, -0.97371138479150177], [-0.94199870660473217, -0.7761222781471373, 
0.28133263444008993, -0.062866327413718448], [-0.6517057507208186, 0.65491103442817566, 0.59299834678865948, 
0.82740874618675231]], [[0.44012931494554919, 0.99719641511647206, 0.9149812057157205, -0.99082386522433707], 
[-0.56711688982673825, 0.35833134825795643, -0.20106138526283757, -0.80539330297231193], [0.97328132737404571, 
-0.58869057173184602, -0.60198505227481314, 0.15998823045583488]]]])+(1.-msk_arg1)*numpy.array([[[[-0.3906025329404188, 
-0.66797630599954649, 0.55626332612710394, -0.457068692078344], [-0.78410812521096473, -0.49243638570973625, 
0.29841020790238093, -0.13552326940959891], [0.5302429043181327, -0.16736605349594891, -0.32299170682193212, 
-0.87075419197777704]], [[0.54629425419275934, 0.93422482754651637, -0.026459574386814344, -0.34680953003122483], 
[0.29185098278590171, -0.53003159945694334, -0.12574506159971754, 0.65975689963329986], [-0.71584634254643165, 
-0.19571211281268908, 0.46323808195948635, -0.49723992236554437]]], [[[0.027689417062008559, 0.022300669629664371, 
-0.84390270292480429, -0.505113378158903], [-0.54750825000226544, 0.94307048396077531, 0.3108339975708474, 
-0.61172090959047609], [0.43765690868955454, -0.7275628428159473, -0.3370827804832035, -0.059131771918396803]], 
[[-0.23280105822411001, -0.79969955562783657, -0.81404675045904007, 0.25135673576902229], [0.55826423982562923, 
0.64622779798664243, -0.16536107142796919, -0.53329031234256563], [0.41394958803272464, 0.10216346250295305, 
0.18870302903326386, -0.58477587277594778]]], [[[-0.63316090072805942, 0.13981387221419306, -0.71451383537974622, 
-0.0020177533913716683], [-0.88954622708813003, -0.95431668093249011, -0.31610740921578206, -0.78120894488186576], 
[-0.50298104954776512, 0.33848149773071357, -0.12181804287981279, 0.76299698123850668]], [[0.74324353622072326, 
-0.56716250169963001, 0.39532337596723721, -0.60851190458351456], [0.2290837754652173, 0.48925543744198086, 
0.22910725686311295, 0.59838985543456413], [0.36495477921428865, -0.98765458011552831, -0.57313136561592271, 
-0.10846619791113143]]]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(4.14398942825)+(1.-msk_ref)*(-1.80134112143)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank0_float_rank0(self):
      arg0=Data(0.240153729026,self.functionspace)
      arg1=0.494786577508
      res=inner(arg0,arg1)
      ref=Data(0.11882484166036042,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      res=res*1j
      ref=ref*1j
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank0_array_rank0(self):
      arg0=Data(-0.68099713877,self.functionspace)
      arg1=numpy.array(0.599013007948)
      res=inner(arg0,arg1)
      ref=Data(-0.40792614449895914,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      res=res*1j
      ref=ref*1j
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")       
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank1_array_rank1(self):
      arg0=Data(numpy.array([0.033542800183077981, 0.33729160037543515]),self.functionspace)
      arg1=numpy.array([-0.79953029369586592, -0.40929278080095122])
      res=inner(arg0,arg1)
      ref=Data(-0.16486950194,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank2_array_rank2(self):
      arg0=Data(numpy.array([[0.11805480437705862, 0.31753327694424094, 0.73062236450582518, -0.90810070531761888, 
-0.74339066588611802], [-0.87396679950262013, 0.033682508122346633, -0.66916008311259567, -0.61758963219973451, 
-0.014600527977821409], [-0.62476994832958632, 0.94174223303834825, 0.19392565663742545, -0.2183301186062907, 
-0.82572189921234873], [-0.58500710929329358, -0.49464134340105126, 0.43417012225850549, -0.38613577382401987, 
-0.52016166685121545]]),self.functionspace)
      arg1=numpy.array([[-0.18028690878941611, 0.062613895915825779, 0.38680633581348545, 0.90119541158920313, 
0.56476663391541426], [-0.64735554161587672, -0.85217678865091129, 0.14569414336970454, -0.91439705751775535, 
-0.58339013293759967], [-0.54453380416009844, 0.6384627434546204, 0.7632588493700363, -0.33123026273510781, 
-0.13210066089445971], [0.26324992126960245, 0.069723410191333324, -0.40664489631445289, 0.76527538981583887, 
0.90980377502679532]])
      res=inner(arg0,arg1)
      ref=Data(0.192897864145,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank3_array_rank3(self):
      arg0=Data(numpy.array([[[0.031191586406912997, 0.25749528378875186], [-0.43560426520028694, 0.82186118773863481]], 
[[-0.27636706429065239, -0.96873716307434288], [0.57212427498318874, -0.89488694319467221]], [[-0.53664653426082554, 
0.17125910714144532], [0.55223023402118465, 0.7026196657323478]], [[0.47151493133476885, -0.9878569008612994], 
[0.923273401741177, -0.61291345033616373]], [[-0.21627815010371965, -0.68308350321451483], [0.42003476980191601, 
0.35437944514165998]], [[0.57930415742194397, 0.15851258736510698], [-0.034124513547037383, 
0.13910191031173214]]]),self.functionspace)
      arg1=numpy.array([[[0.4376882511962199, 0.87791312611900185], [0.3872095550010406, -0.80749936076640538]], 
[[0.99370860605609601, 0.1415181468078317], [0.88396994180208766, -0.080771360869665632]], [[0.10037555131668507, 
-0.30418314544151359], [0.18496158262509454, -0.64787225673616877]], [[-0.63902641553526074, 0.82753535127444788], 
[0.92366200098645801, 0.71478320656855199]], [[0.87210131386272871, 0.66318134839062681], [0.40332439860072489, 
-0.61714848609477224]], [[-0.026058266846441747, -0.6502882417343101], [-0.039896755710729703, 0.14651276739152008]]])
      res=inner(arg0,arg1)
      ref=Data(-2.37679561615,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=inner(arg0,arg1)
      ref=-ref
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank4_array_rank4(self):
      arg0=Data(numpy.array([[[[0.84900275444177553, -0.246217726373545, -0.73088633570089478, -0.047481367331499991], 
[0.032816845637419156, -0.76812807628518143, -0.81281363329523049, -0.58084890710073589], [-0.73993039657350201, 
0.83123891495121938, 0.46715743977232393, -0.76686166407968437]], [[0.065941685788379445, 0.6522770195720593, 
0.55982464930930576, -0.80087140104814658], [0.99239590675011136, 0.31694749481754458, 0.51372543692541339, 
-0.52022278210177308], [-0.32907326950409344, -0.55598341424051401, -0.067704116072573095, 0.31263074534856905]]], 
[[[-0.23768937860779404, -0.80388720531878, -0.95492614309454349, -0.76905056297799268], [-0.28127606377571679, 
-0.15077250683701848, 0.077788021738735891, -0.20476199518683469], [0.53944050035171465, -0.76150320944042771, 
-0.41896215933942926, 0.029055212182700796]], [[-0.58861237768203778, -0.10490589957463636, -0.20817583378648052, 
0.45198229030492376], [-0.50700996207133375, 0.90434965349211405, -0.92835653637992666, -0.83045937562172556], 
[0.13469897080944304, 0.84359291895905919, 0.79200244141954079, -0.19219096641228584]]], [[[0.93312085161368596, 
-0.049735922253603126, -0.2192679149037029, -0.25989030793661816], [0.72644194255242556, 0.74091598098057099, 
-0.30432653743395388, 0.21345622350494531], [-0.97093017386630387, -0.70687128737696248, 0.59112558415095928, 
0.49788697633828582]], [[0.80030801684525965, -0.029383632137730986, -0.021414167385272354, 0.39283038151162608], 
[-0.33822921166771125, -0.26639948236330291, -0.53349415516836562, 0.020369480646021065], [0.51135584516748556, 
-0.3710959845124091, 0.26001287550270291, -0.85402920503239654]]]]),self.functionspace)
      arg1=numpy.array([[[[-0.60896275596671168, -0.96198549401243993, 0.87828678808797989, 0.20466313727661678], 
[-0.60814598635215744, -0.42907263934755946, 0.34070813799556454, -0.44693550838732787], [-0.45080256697753263, 
0.38604990495638059, 0.80729145776923339, 0.73718833304242581]], [[-0.035964177002679509, 0.94827643636590153, 
-0.26496089896423247, 0.17351685300419173], [-0.86296994839532148, -0.34374708342173599, 0.93025018200816256, 
-0.51672268415125755], [0.44209583101516747, -0.96811144437281405, 0.4995052724717195, 0.9123136046672522]]], 
[[[0.088471951920575664, -0.6656163426311541, -0.09894089828246333, 0.069975899888207627], [0.19541385901389585, 
0.69922988340466619, -0.50164250295332447, -0.50835814069727259], [0.21174890510515954, 0.1461844479384613, 
0.71251500577579563, 0.47855838895259528]], [[-0.71000611047816142, 0.18410545018704094, -0.16709456416117296, 
0.7936780300267583], [-0.37262949205049356, -0.55397580023642123, 0.24386454933063217, 0.47508493827539522], 
[0.19879432084889914, -0.090159004691243494, -0.76971400164621384, -0.79794747780441577]]], [[[-0.074209069031669017, 
-0.087535757179307172, 0.14630099406390684, 0.37408318110918937], [0.90736322792861746, 0.56357314444323792, 
-0.031967808854433954, -0.48045831936901751], [-0.87344873813351165, 0.96387391665293354, 0.27247943246116457, 
-0.097155528382333989]], [[-0.21814477511318953, 0.32096345155185513, 0.91631079955393258, -0.53655097537736762], 
[-0.44317638848986629, 0.77227497773824183, 0.56920867302973188, 0.021731072527379158], [0.89544436460468901, 
-0.81538646463898146, 0.025217872921130091, -0.2345693190627598]]]])
      res=inner(arg0,arg1)
      ref=Data(1.37543330539,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=inner(arg0,arg1)
      ref=-ref
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank0_constData_rank0(self):
      arg0=Data(-0.109880112782,self.functionspace)
      arg1=Data(0.679037989257,self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(-0.074612770843086082,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=inner(arg0,arg1)
      ref=-ref
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank1_constData_rank1(self):
      arg0=Data(numpy.array([-0.38855187589347273, 0.14448607308006922]),self.functionspace)
      arg1=Data(numpy.array([-0.60202560601945909, -0.80720775185258908]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(0.11728790033,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank2_constData_rank2(self):
      arg0=Data(numpy.array([[0.86706829877617553, 0.49321667183194462, -0.46368761238578982, 0.29128551483742715, 
-0.92830470873637738], [0.29788055279544867, 0.43176703576888875, -0.85641119676627464, 0.46156140731641848, 
0.72244217301749325], [0.44947710909496763, -0.075355916107305854, -0.41900674017982698, 0.75202312792256043, 
0.22573770677908533], [0.0031182461358558733, 0.71442090495797195, 0.54936572245556348, -0.74234307901918961, 
-0.5861969161328362]]),self.functionspace)
      arg1=Data(numpy.array([[-0.76298566426626668, 0.42924119580309217, 0.088601689342662659, -0.92253185082545741, 
-0.10621318192598928], [0.6270532350670448, 0.85936605057441562, 0.46565958269064978, -0.87889242410910473, 
-0.08233008340904413], [0.8819854104670195, -0.80865701059438289, -0.050226129858086166, 0.033418509554747455, 
0.90566607852023706], [-0.74726523489863084, 0.99737958054864184, 0.56971288996407088, -0.9525569131192575, 
-0.41645026842223332]]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(1.71527222467,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank3_constData_rank3(self):
      arg0=Data(numpy.array([[[0.15940026922242234, 0.15456201881086518], [0.59670596059649061, -0.78199474933041069]], 
[[-0.40931306095332465, -0.30330934407899113], [-0.41127966248227921, 0.87766929968889151]], [[-0.50526082253226456, 
-0.55791104709617079], [-0.46341249236547877, 0.94360358235444308]], [[-0.085176359264879542, 0.99973232174249427], 
[0.14305711539257082, -0.16880638300304596]], [[0.34290503849199472, 0.17643564090355612], [-0.063107186944666438, 
0.90573192192593233]], [[0.046518471261303596, 0.099777293593690031], [0.27897146863434386, 
-0.11021578865267534]]]),self.functionspace)
      arg1=Data(numpy.array([[[-0.020655884066142161, -0.63581685932798959], [-0.028189504194750636, -0.33312032709984463]], 
[[0.18371093040455544, 0.53810369487480392], [-0.6278500905042923, -0.23388196616213763]], [[-0.6265531075236217, 
-0.81242119009526403], [0.85751366598472889, 0.47472755352559814]], [[-0.25421798028957743, 0.76915921038357227], 
[0.23926613772390404, 0.90786986627874433]], [[-0.32449040895249714, 0.72903050951989146], [0.85899390598941605, 
0.14162684860538022]], [[-0.47547023426125623, 0.52010210858651917], [-0.026594493728175239, 
-0.85487359950804409]]]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(1.65664288426,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank4_constData_rank4(self):
      arg0=Data(numpy.array([[[[-0.77842547938200712, -0.27838514037726103, -0.89144217523800173, 0.0054425153781361146], 
[0.99706601886824098, 0.78178928147355919, -0.48010670862462801, 0.69424527709964767], [0.66013789019771019, 
0.45824582488789067, -0.96353240437680876, 0.834555866938125]], [[0.3621919162667917, 0.25713435067190016, 
-0.72525415889975187, -0.37623226602641213], [-0.53778025315253108, -0.27139019205119497, -0.067900376115989092, 
0.4568352783260623], [-0.41907694255774697, -0.2194008083746839, 0.66797370775622289, 0.26820893540122404]]], 
[[[-0.40605657137194817, -0.12790178125270946, -0.17410735990789283, -0.15888167658286534], [-0.16153499568694052, 
-0.85631531260351879, -0.049876455481712823, -0.46438180769276838], [-0.44000243900407199, -0.40478747773712298, 
0.92325256219560048, -0.35254132751685119]], [[-0.98549823069498244, 0.51509155773569848, -0.35801892388132384, 
0.41842044524973021], [-0.81537594633586319, 0.0061482420516145719, 0.73869381108433863, 0.10350459228602715], 
[0.6485592561496587, 0.29062672359343233, 0.67430545525394114, -0.87352827882424622]]], [[[0.90625102942488267, 
0.57609384066425284, 0.94149558862395133, 0.13561300628620221], [0.051759965405846264, -0.73180643620986086, 
0.15917838120678218, -0.19984359746002767], [-0.75895521708451175, 0.02467549025441973, -0.71190411029359635, 
0.064476244560973006]], [[0.010518277032398826, -0.47261451925786702, -0.72252534248328493, -0.88093533929134749], 
[0.7554786003180507, -0.53706340160103583, -0.89826064239038872, -0.70850211109993722], [-0.065050104790129737, 
0.59873154953592889, -0.27115597493835475, 0.60224768895299086]]]]),self.functionspace)
      arg1=Data(numpy.array([[[[-0.20249524630900306, -0.96774449919067917, -0.52843683450160439, -0.46516378685072324], 
[0.97628567894067331, -0.3008376220458906, 0.2085767180427871, -0.64493892545106712], [0.63542883246455828, 
-0.75580660556926271, -0.11642715040710971, -0.57722651873841335]], [[0.9635443621483939, -0.45469939476511745, 
-0.41046595433690669, -0.00087468895810327574], [-0.0064587635049175862, -0.85926878616987445, -0.89729357458235381, 
0.73746038100106448], [0.011783382508033791, 0.032903862162879216, -0.37599153545615205, 0.59273806574501897]]], 
[[[0.45474791914542756, -0.53886378385785316, 0.72930018614416792, 0.3818111441831058], [0.819037183960172, 
0.064783887997734713, -0.67285340441766239, -0.82697093043866676], [-0.41303793669871669, 0.65375901690488325, 
0.48148908822783976, 0.61591587360434841]], [[0.54012711113369427, -0.63211866188134502, -0.073210952913265626, 
-0.81920966825262687], [-0.51175703045000764, 0.19787169073795052, -0.21697146593273309, 0.063276039413608931], 
[0.32218457663689404, -0.064958869530121088, -0.35773287769104711, -0.74493151067112429]]], [[[0.17263744713312823, 
-0.8114330555959357, 0.062894929516047027, -0.73765216130276956], [-0.10051063265176641, -0.094306413867799588, 
0.90282523931118108, 0.6151602807402341], [0.88428575660803332, -0.031152847200344835, -0.52110285906370746, 
-0.52480281782828131]], [[-0.5571306490178789, 0.67789963231677008, 0.79860247005714258, 0.44194411259296795], 
[-0.27253617209725989, -0.24175511809765515, 0.32343391507911723, 0.48083522723160876], [0.9522975767175601, 
0.15975334942746611, -0.76757362234650128, -0.9820616906911519]]]]),self.functionspace)
      res=inner(arg0,arg1)
      ref=Data(-1.34087782434,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=inner(arg0,arg1)
      ref=-ref
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")       
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank0_expandedData_rank0(self):
      arg0=Data(0.504999017422,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.949315936357)+(1.-msk_arg1)*(0.226218043381)
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(0.47940361508382373)+(1.-msk_ref)*numpy.array(0.11423988963078481)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank1_expandedData_rank1(self):
      arg0=Data(numpy.array([-0.081888632374237824, -0.70704843929112626]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-0.63647001611409015, 0.28001115497616635])+(1.-msk_arg1)*numpy.array([-0.95518664215107396, 
-0.099716134713024651])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.145861790943)+(1.-msk_ref)*(0.148723065209)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank2_expandedData_rank2(self):
      arg0=Data(numpy.array([[-0.57859909800275866, -0.87214342898618158, 0.016011568175356761, -0.23815542580177862, 
0.60391292640371552], [0.35788399426640716, 0.017142972407319101, -0.2967562302731197, 0.3534441869214251, 
-0.90980604318790004], [-0.07925842687212481, 0.47853292187338226, 0.64048368330147909, 0.14555091638444151, 
0.28561330493337755], [-0.3552511008600614, -0.38420485695947626, 0.26739266799421046, -0.60565377378093421, 
0.11194898284027333]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-0.93077040816398893, 0.90091471866981743, 0.065383612598433727, 0.6037065816632452, 
-0.74867109215247041], [-0.63866698099654573, 0.53705324255622111, 0.53082384464137666, -0.05830484773571265, 
-0.25706797972194462], [0.09395735902475999, -0.73147043052977256, 0.4501866674408872, 0.7474043741109182, 
-0.95363680556286257], [-0.16128566467235839, 0.82346724291291906, -0.76766336820905723, 0.84787123161224653, 
-0.58085210405634502]])+(1.-msk_arg1)*numpy.array([[-0.45783242037211935, 0.64404206556021504, -0.43367995912720581, 
0.68924549824928949, 0.67655360249298435], [-0.58959025803189502, 0.4825575464599865, -0.36957742684587358, 
-0.68246492861111885, 0.61875212284828685], [0.12481128840438638, 0.31263132178344266, -0.49457193786999665, 
0.55938196857186795, -0.72677278202984485], [0.41178913492830782, -0.43314660108339909, 0.93644188167580755, 
0.17440462131539314, 0.40728030234091483]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-2.28127930167)+(1.-msk_ref)*(-1.04924200455)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank3_expandedData_rank3(self):
      arg0=Data(numpy.array([[[-0.78262529392534508, 0.19262299116314141], [0.40314342927705571, 0.56765433666196907]], 
[[-0.077381984119281988, -0.10070169049292366], [0.73590227931222296, 0.67072024003607411]], [[0.52384371186421408, 
-0.12399733883675723], [0.6575827271672896, -0.072890130875433945]], [[0.69043721918464129, 0.25684356518514662], 
[0.64215553103016898, 0.79678359995253389]], [[-0.34984047601112822, -0.52668353895434161], [0.64670167849951898, 
0.073109332448158648]], [[0.19460045426646722, -0.27300730637033577], [-0.3615187862955016, 
0.66093465992733225]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-0.56847726275050459, 0.22553822811057311], [-0.50483648978341789, 0.18596122471399257]], 
[[0.82907064642940376, 0.39924212625792066], [-0.61560545397453392, -0.77936842489418856]], [[0.22312691317620548, 
0.33763279650280675], [0.25967515105910666, 0.83249352707317881]], [[-0.5125813274545159, -0.24474831588504764], 
[0.61761543752610981, -0.90901051596869187]], [[0.96045364618153561, -0.63252896492430533], [0.30994914138101914, 
0.19728471412527138]], [[0.5793481344933038, 0.98228090549135483], [0.56156762048276998, 
-0.43886014095897963]]])+(1.-msk_arg1)*numpy.array([[[-0.072477101298686453, -0.88184779719335049], [0.71644251282215632, 
-0.4218857555860287]], [[0.49993394558426107, 0.51515328476076028], [-0.40601362900867066, 0.96411337080457771]], 
[[0.36949361406832071, -0.56377042221000795], [-0.26007700280810964, -0.32112177435121891]], [[0.033011707625483888, 
-0.19851785034552694], [-0.3443811451793517, 0.74419473885301524]], [[-0.81079826467032756, 0.029779666629704549], 
[-0.57319734205670558, 0.15998162467162769]], [[-0.61086054296645664, -0.92839666154121758], [-0.40381664895113789, 
0.41316312614238138]]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-1.68558433172)+(1.-msk_ref)*(1.11559168604)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_constData_rank4_expandedData_rank4(self):
      arg0=Data(numpy.array([[[[0.20663983007903708, 0.30744479922588486, -0.61313072201040253, 0.25088112795266682], 
[0.33976801897833919, -0.064270205661395075, -0.85843808732189264, 0.50003701608777096], [-0.22943788417321742, 
0.74280772825811958, -0.0089561666890796143, 0.20113324496421714]], [[0.88307214121599276, -0.39885344149136404, 
-0.3459359788500429, 0.25014778756241052], [-0.88472844025773534, 0.74531568368805612, -0.72531133224660183, 
0.10223747230426095], [0.53780460336790248, -0.85214383675588068, 0.44977767427703097, 0.41749848532253431]]], 
[[[0.810262475188956, -0.83945779116489216, -0.40109970785754068, 0.81761928138724738], [-0.64032533906144784, 
-0.20290569312629381, -0.7513065784748898, -0.11353011082332332], [-0.04329051434534259, -0.53099678691971941, 
0.72579753561683891, 0.10912003488729338]], [[0.25838798321207412, -0.024144789607555728, 0.037189255130095189, 
0.83617814730817952], [-0.057771627027428885, 0.57135552620608943, -0.29473564102969707, 0.22430933502957195], 
[0.0066994836377327616, -0.10368706774933312, -0.043730730137522489, -0.48065952960266367]]], [[[-0.43491153992500364, 
0.88082656940550486, 0.68378859836467565, -0.012356816132639503], [0.62178344914568107, 0.25078866625698182, 
-0.17819520948670231, 0.77144460986479291], [-0.13233066669247773, -0.52312935440794983, -0.69513281910345093, 
0.35433089338923129]], [[0.98606210920219017, -0.48210615134767609, 0.39736669460124818, 0.4619219074410863], 
[0.29477660552266371, -0.048973046813332921, -0.90547262460988409, -0.3696640293795237], [0.24863189129449159, 
0.042312225378325463, -0.5264617262481659, 0.63154246835173189]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[0.089737235920831493, -0.56346650919450236, 0.28624099902385591, -0.075846454418770026], 
[-0.95811348785909334, 0.2752992871874782, 0.13840105973254935, 0.27184883084441624], [0.0074529560285576668, 
-0.90632485195229795, 0.24414236087203656, 0.69344832107210985]], [[-0.99378322357862214, 0.7208847594355694, 
0.89881914525516038, -0.97829419124154593], [-0.45862576369309904, -0.92319150295659647, -0.013676613305693364, 
0.49602557059441765], [0.8901810990644663, -0.49566489033100725, -0.37871731857378066, -0.36883668970072847]]], 
[[[0.18437957632400348, -0.12147921286972108, -0.393537207341218, -0.10680783054154741], [0.59524170515902464, 
-0.66587261729761993, 0.92947386439191648, -0.59547765733667379], [-0.41650352833637716, 0.47904744601647264, 
0.25048486764121192, -0.41733337222442124]], [[-0.79428459272932228, 0.5722780947544559, 0.12114888559828185, 
0.56254489926953277], [-0.34164943537144987, 0.81303102768839741, -0.038515748676847927, 0.41404092010551152], 
[0.20093724259501333, -0.88224230847965268, 0.51276578069438661, 0.72798848537859406]]], [[[-0.45399090015655497, 
0.79734969926852561, 0.59881323493903804, 0.8863764997702912], [0.73518245422321593, -0.84664030571345616, 0.38783733192343139, 
0.10097672989077644], [-0.20113660943856826, 0.47646146092328245, -0.45757028576837544, -0.88890427591403598]], 
[[0.83506841533390697, 0.6716787741303738, -0.10053183201377869, 0.11446135026109738], [-0.41932879449869054, 
-0.53813698492040141, 0.79149344313352699, -0.6411930529087273], [0.97161445556347514, 0.30396839332542291, 
0.43157396752624089, 0.34620878385763731]]]])+(1.-msk_arg1)*numpy.array([[[[-0.2365951445271044, 0.75967395787052205, 
-0.030266120399357277, -0.062905760835206603], [-0.0034883077622471781, -0.11075118427509789, 0.13174467113806743, 
0.14259302169144994], [0.85476603592412981, 0.14614550828496253, -0.52196009130408605, 0.81755710115293234]], 
[[-0.94646680517447002, -0.10727315772865564, 0.61196965517547364, 0.45137437283758808], [0.10990216419068144, 
-0.65862678223718785, -0.84446480541514646, -0.49352662713553697], [0.34069886888804035, 0.59232100391463494, 
-0.45038974037479473, 0.93140701293250738]]], [[[-0.24206999953430408, -0.67008991280004926, 0.29703689217222728, 
0.34177624779386373], [0.42746834677094858, 0.89522807405287197, 0.81107598861107144, -0.38725554644468985], 
[0.010653314070785402, -0.53865696273892572, -0.18015869599227829, -0.98478472344271717]], [[-0.29749384360660525, 
0.91543915007368271, 0.65063936887232066, -0.64142279626575327], [-0.6680021459857739, -0.59184516083841476, 
-0.99887826884567055, -0.77809132563010275], [0.97229142472558627, -0.84596239894234082, 0.41786543173372448, 
0.48987897019575266]]], [[[-0.94580762978353428, 0.48552438448269752, 0.10135988934582585, 0.1698050051381188], 
[0.57086484517167979, -0.72706535861218913, 0.59049934683374494, 0.01124997633120417], [0.85945579027607155, 
-0.17921010647927127, -0.60581393629582103, -0.70955535208358023]], [[-0.45839119112204441, -0.71839988876461813, 
0.56529846008009965, 0.27304931740440574], [-0.802780743166704, -0.2200444749049042, -0.60485461325660217, 
-0.54399770036370776], [-0.72600872797946381, -0.98033359034866319, 0.53661806935321499, 0.44150780904733433]]]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-1.15490107051)+(1.-msk_ref)*(-0.542684806247)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank0_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.796068672144)+(1.-msk_arg0)*(0.141569698859)
      arg1=-0.603765866012
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(-0.48063909124250953)+(1.-msk_ref)*numpy.array(-0.08547495183288302)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank0_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.62865123941)+(1.-msk_arg0)*(0.56747818964)
      arg1=numpy.array(0.933535236912)
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(0.58686808371773458)+(1.-msk_ref)*numpy.array(0.52976088620826978)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg0.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        res=inner(arg0,arg1)
        ref=-ref
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")       
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank1_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.39152578782990766, -0.52109249508064992])+(1.-msk_arg0)*numpy.array([-0.11199372735313928, 
0.6249548954816011])
      arg1=numpy.array([0.21561276669150931, 0.99522225948885312])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.603020808702)+(1.-msk_ref)*(0.597821745753)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank2_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.85357846934618076, 0.2964009731213777, -0.023080151467793186, -0.27661013218309982, 
0.91448024743450684], [0.038642001670084358, 0.45125127830899081, 0.078079050913618886, 0.15283070827828116, 
0.11079659530969765], [0.75962692381603292, 0.35160201211707931, 0.18704639739340623, -0.54900920331954639, 
-0.22652306109506082], [-0.45955692269483905, 0.044781495048763054, -0.2231804184681454, 0.2212845809191446, 
-0.53383944584935916]])+(1.-msk_arg0)*numpy.array([[-0.83645714719341768, 0.96709801321284594, -0.058716806047270031, 
0.79069168020660108, -0.006189196069628089], [0.76088034415206862, 0.35707511806818282, 0.49950296883713152, 
-0.14173588322467801, -0.93983638713043782], [-0.41603986165671203, -0.43918722106447605, 0.59689715924249764, 
0.70425582964428801, 0.21973881370350368], [-0.62574249966976314, 0.2359074345188692, 0.95673105277827286, 0.73034029598894068, 
-0.46841895831725866]])
      arg1=numpy.array([[-0.55385492909427314, -0.47641952194387671, -0.89304150409657335, -0.87332537255000009, 
-0.44614032725195774], [0.68166147218263906, -0.81572051421033853, 0.6817858041358531, 0.80364966651683156, 
0.69869146527717207], [-0.28246872658949052, 0.7565359744867568, -0.58420487370735263, -0.053675921069090915, 
0.012529594167807634], [0.84491978275008273, 0.20578097220399294, -0.83873899314782618, -0.62655075305381036, 
0.037134445285017748]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.284110545531)+(1.-msk_ref)*(-3.19151566867)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank3_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.50531540411783182, -0.84947843142415125], [-0.95083916187690498, -0.21527209334022501]], 
[[0.2583894311796151, 0.54220175557198469], [0.65696482234713316, -0.55740417358634153]], [[0.29656481425005188, 
-0.31044204944929144], [0.50754888593660263, 0.38016247619963761]], [[0.69329849552127065, 0.58139227981628316], 
[-0.051956674604789699, -0.43018500683995664]], [[0.3632473698109191, -0.52776919844933068], [0.2062693834122924, 
0.37147496020488502]], [[-0.3209875527716135, -0.33881385226990934], [-0.41746150569840856, 
0.49408272714889878]]])+(1.-msk_arg0)*numpy.array([[[-0.41041698765894008, 0.89834680164904657], [0.55229868254055225, 
-0.045937486672473327]], [[-0.57946737794748171, 0.82419491486371754], [0.52384160056564988, 0.66720084983331551]], 
[[0.92072943418247588, 0.044221354735691332], [0.47640305718056308, 0.041941185364050781]], [[-0.59582446422571267, 
0.54567956683968699], [-0.40584249131142558, -0.44707794880717588]], [[-0.47053963882295413, -0.63790818284295314], 
[0.83414330884494636, -0.48717768922188109]], [[0.082591022081887511, 0.19857141903282627], [-0.10712395103176831, 
0.79089222821183669]]])
      arg1=numpy.array([[[0.49639001737089017, 0.99796022318233724], [-0.26032037322246437, 0.87839121298696399]], 
[[-0.70668657882973518, -0.052141122088173564], [-0.92298831309774809, 0.10209313818414234]], [[-0.015592425837284374, 
0.24294783103686157], [-0.066724432625728358, 0.18394037896882964]], [[0.86297004238084862, 0.02978315080350713], 
[-0.42676542085722868, -0.21841822625410234]], [[0.71093267788754444, 0.70330080989264721], [0.22278712618694252, 
0.71358709166414203]], [[0.70882998330176572, -0.18543709228630068], [0.80986692291703566, -0.66366121056967531]]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-1.35746173436)+(1.-msk_ref)*(-1.32984776059)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank4_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[0.019006423843182851, 0.97637325818737142, -0.3778371247022172, -0.24522935505580201], 
[0.25482670730200452, 0.49070982279658582, -0.13970200142272571, -0.69060997302262028], [-0.0035452601352568003, 
0.42044690617808511, 0.23511837122113199, -0.52221931154567769]], [[0.19381933190495082, 0.50012173987864328, 
-0.2386525690285064, -0.37667383268862786], [-0.90971892925469122, -0.35065845094282455, 0.8087148795715664, 
-0.44030179135579273], [-0.13870392554075761, -0.04716338746550508, -0.29904290935096611, -0.79131372436671921]]], 
[[[0.44896770989300028, -0.53803971979467313, -0.25044275134156724, -0.38914200762310536], [0.68803157043978946, 
0.58126023831649132, -0.57827031016518915, 0.86926678146573488], [0.21238462391570478, -0.033692118582254516, 
0.89777850769742762, -0.95875294470846728]], [[0.11907635360994329, 0.98460261094680579, 0.86174977469907055, 
-0.26058948337859], [0.086267828609031483, -0.91293579034927186, -0.38540499021322527, 0.12009708559161325], 
[0.73392875969958893, 0.18353973800235357, 0.52212765409565609, 0.37309668927918338]]], [[[0.51977741225129326, 
0.59772951631814597, -0.87702243675428693, -0.46372790641150741], [0.36789568031225017, 0.89534194799267564, 
-0.23506346221068353, -0.55222847869683522], [0.16542389254588907, 0.14294883580010453, 0.089959855117651921, 
-0.68981061483411676]], [[-0.41115958809013176, -0.986489932594244, -0.96413374324085255, 0.32322608077112491], 
[0.18297301423492152, -0.22781262416705839, -0.024144967009124052, -0.24163855793719113], [-0.058032940521662413, 
-0.24375443215880965, -0.078100049154545825, -0.57894164618957267]]]])+(1.-msk_arg0)*numpy.array([[[[-0.40001705419647693, 
-0.67263334791472262, -0.78918590831661017, 0.78865053701091914], [-0.090034058366400105, -0.37881095982421065, 
0.91867249206684143, 0.83980019508043813], [0.67448518939744662, 0.96143694144932024, 0.78309732697025192, 
-0.56731276091215888]], [[-0.59467823931614316, -0.021013646322696644, 0.72513102655611639, -0.62960315578871229], 
[0.86606083067435691, -0.72841084190946459, 0.15543842306958466, 0.31989412769972758], [0.47194268630299918, 
0.19444174622614785, -0.66695252858363485, 0.8443318710904939]]], [[[0.18223491454007679, 0.82014337464069054, 
-0.96721600534140117, -0.97710964280180557], [-0.33988905556481974, -0.32222643536859397, 0.95343454998321575, 
0.39976170447642656], [-0.32911640832985767, -0.0051494009105315097, 0.68395988105583672, -0.66658071774810845]], 
[[-0.51922775999439819, -0.4962702374477268, 0.66698724811873289, -0.13569420026129531], [-0.46821413546121105, 
0.052814763063328218, 0.98772928696039708, -0.18783043849008596], [0.94396487905072735, -0.62988912286541021, 
0.77657229350292489, 0.45494324806001996]]], [[[0.49245323985257272, -0.11950235129083819, 0.027258906056445653, 
0.60373375693159259], [0.43120373969315473, -0.1588341022553017, 0.32757005714599741, -0.80325727791557533], 
[0.20663774771076104, 0.75387367922315862, 0.6484883492094573, -0.0062877488438248808]], [[-0.84891489459846925, 
0.79397666107969944, -0.97762738566042362, 0.49200737923277527], [-0.43113202215326463, -0.43968935940027642, 
0.22521681397809279, -0.24856511685500804], [0.37263947745544268, -0.1323935505978322, -0.24082796491432612, 
0.56535324006815268]]]])
      arg1=numpy.array([[[[-0.81163163217881018, 0.14583559156390025, 0.90627553136559591, -0.030656591956136126], 
[-0.30432921218385545, 0.059587722748475569, -0.40268326742636673, -0.61586544340169969], [-0.8514338629390128, 
-0.8041774874930705, 0.85523791052074705, 0.38333292243623918]], [[0.16821600296096539, 0.83859885983438409, 
0.43414495762074878, 0.024013057438022178], [0.98564703788576091, -0.79662656719344027, -0.4795278224208761, 
0.95284714676502769], [0.23808354955330957, -0.46334260291805918, 0.095014259985530369, -0.10374068985654117]]], 
[[[-0.55773665932704719, 0.86766281454843952, -0.060782779594130254, -0.81799578780935178], [-0.39579207222264756, 
0.14496469860413042, -0.39443931203825033, 0.46619641301760439], [0.59557714470818546, -0.1844163855468639, 
0.31701568421598036, 0.99386589088114263]], [[0.28596875555199719, -0.5113185780054228, 0.11275973679314433, 
0.95984183359351416], [0.56397724998403853, 0.21456711389176819, -0.8869207954469096, 0.082002513156396395], 
[0.80154311652059129, -0.14918064076265192, 0.87456914801889662, 0.32879945324200155]]], [[[-0.79827538301490764, 
-0.63675913901470316, 0.73918094313334581, -0.69235823201746327], [-0.13014921576975857, 0.21884429385731696, 
-0.61439214698250044, -0.98441367018944836], [0.83244430972608208, -0.9972055581758188, 0.64180363710022958, 
0.40388529426265696]], [[0.75678188832694704, -0.95635607269517875, 0.8829297874583959, 0.64433269944660188], 
[-0.88188838313425677, 0.92295774091578653, -0.069016329600187332, -0.16861637091927673], [-0.87830132902116187, 
0.45182237326637598, -0.94926829916048394, -0.98497920687933416]]]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-1.17042066276)+(1.-msk_ref)*(-2.30685891421)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank0_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.291531225166)+(1.-msk_arg0)*(0.0245199868034)
      arg1=Data(-0.700555766847,self.functionspace)
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(-0.20423388100607154)+(1.-msk_ref)*numpy.array(-0.017177618158112615)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy() and not arg0.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        res=inner(arg0,arg1)
        ref=-ref
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")       
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank1_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.13795126128122193, -0.049083343324659578])+(1.-msk_arg0)*numpy.array([0.41150496249996782, 
-0.895867771687894])
      arg1=Data(numpy.array([-0.48311607076511565, -0.98030703413828491]),self.functionspace)
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.114763218027)+(1.-msk_ref)*(0.67942081766)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg0.isLazy() and not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        res=inner(arg0,arg1)
        msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
        ref=msk_ref*(0.114763218027)+(1.-msk_ref)*(0.67942081766)
        ref=-ref
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank2_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.75796400257798324, -0.70348397505435734, 0.89005890441617064, 0.28168276991104246, 
-0.22815083366244493], [0.87001455744515122, -0.61199454394107877, -0.88968426082566543, -0.48988343936641043, 
0.0091061395097682407], [0.00038701284385123103, -0.58154294262613138, 0.12466518547258865, 0.26742919042369873, 
0.2060419497399868], [0.05209424073622726, 0.36005249160908215, 0.84151943818677521, 0.90383961792029832, 
-0.3365232541585006]])+(1.-msk_arg0)*numpy.array([[-0.75334843408531493, -0.089920064534031008, -0.41287254028214426, 
0.93961825826695544, -0.099839208277126223], [-0.79338565350602552, 0.50638366051092198, -0.35393904632388651, 
0.29272444192355396, -0.6584314991602549], [0.85093602922976408, 0.36590867303067798, -0.5094538711100558, 0.89305104272018543, 
-0.10276418819907129], [0.023578509730667108, -0.16387847175810877, -0.18004804284015163, -0.81549358690093499, 
-0.47628686906637374]])
      arg1=Data(numpy.array([[-0.72674958907045362, -0.93280779919345203, 0.43718221749563302, -0.0088212548455395279, 
0.921486643264948], [0.49429499981575908, -0.81673962137327916, -0.54992531147908963, -0.8645863398574325, 
-0.086203025330993999], [0.040650358243210727, 0.49655770251078479, 0.6536740821874063, -0.066082623968604537, 
0.37594522027749044], [0.12475948518736812, 0.71132898153986934, 0.33196050203008243, 0.095617436608621098, 
-0.64995797842956304]]),self.functionspace)
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(3.92500564327)+(1.-msk_ref)*(-0.613040413195)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank3_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.93288754023006093, 0.60106570738315845], [0.30801084889880515, -0.10038465186627255]], 
[[-0.49486005038268965, 0.60921104657314995], [0.87331640600526983, 0.96637335220871035]], [[0.43202597147178401, 
0.91322807330597522], [-0.62307024365185915, -0.17477126639044194]], [[-0.31822550805442673, -0.90911614050867762], 
[0.55699981820870859, -0.68063487208669682]], [[0.0034300727389189323, -0.86962091301762912], [-0.57714211960863127, 
-0.85825991865608864]], [[-0.17754376238064573, -0.24779798715479107], [-0.011140993027736412, 
0.098200156176100561]]])+(1.-msk_arg0)*numpy.array([[[0.24010403647853407, -0.88838665532645722], [0.90501777439104525, 
0.53632909560538189]], [[-0.54131241789007545, 0.15141641667204264], [-0.99744268542316106, 0.29568079322935215]], 
[[-0.07533138726578037, 0.51085817705272096], [-0.68401123133882247, -0.026930646609903741]], [[0.31995995928285237, 
0.31660009563403202], [0.35070586390968561, 0.66058148339954159]], [[0.55381138010714537, -0.22547846639192781], 
[0.20473202038390381, -0.42712246976137891]], [[-0.83406606159475327, -0.36707521115120145], [0.99394483242876586, 
0.64057999530308241]]])
      arg1=Data(numpy.array([[[0.56165585181397337, 0.76350367189248303], [-0.11994286131079535, -0.16253635268259248]], 
[[-0.95622051234148109, 0.83826617585901686], [0.49593782958986288, 0.49840249855170282]], [[0.69853814174312601, 
0.087064491712804282], [-0.01647409369557451, 0.43352480342262467]], [[0.98625851181227997, -0.85890738533846744], 
[-0.75112351254266674, -0.69214254512586582]], [[0.21952135478796331, 0.74856199577848415], [-0.32886277865741609, 
-0.99776445778129808]], [[0.097666284406329407, 0.62425865896691324], [-0.0010713686346326412, 
-0.078146997352248615]]]),self.functionspace)
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(3.91262521622)+(1.-msk_ref)*(-1.17757862828)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg0.isLazy() and not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        res=inner(arg0,arg1)
        msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
        ref=-(msk_ref*(3.91262521622)+(1.-msk_ref)*(-1.17757862828))
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank4_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-0.44681514011031398, 0.33985438742685536, -0.4431136159656559, -0.13354837017362287], 
[0.73305422778738927, 0.61064286420775193, 0.99914094831721179, 0.20213178104609941], [0.77095705236519541, 
-0.9529046335984086, 0.9520486082598747, -0.042873934491836962]], [[-0.25981270183444449, 0.85055011439346706, 
0.1307660257197174, -0.75354074853278541], [-0.89931998085420384, -0.31754146844196685, -0.96168179930810993, 
-0.32969850226885389], [-0.81759489072088565, -0.79455455035559308, 0.30554468587756012, -0.97694761502203065]]], 
[[[0.9524044737827746, -0.5491619770191285, 0.51931562514984453, 0.75704791120318649], [-0.63299678476462362, 
-0.60042020485151748, 0.63938597028752708, 0.97527046159828057], [0.7349335457464401, -0.47635296268781113, 
0.47610605929806882, -0.91077052361951272]], [[0.92333547515269676, -0.13572130553211714, 0.79962527069075584, 
0.62582507802038934], [-0.34955705672146808, -0.22067602283071563, 0.98794413498735012, -0.39365343011975495], 
[-0.9395621189187584, -0.10369246837169155, -0.48126680305363423, -0.51669696645207086]]], [[[0.65028702636009528, 
-0.53411372748691854, 0.32701223381980959, 0.92786702524886722], [0.40479639977686066, -0.081428651882120562, 
-0.83257147695365452, -0.83547803542192933], [-0.21493301104465234, -0.858545831833744, -0.046828862716373321, 
-0.61361609559691499]], [[0.97566639940685507, -0.0056602780003758291, 0.72342396028118405, -0.18050425435644346], 
[-0.7670123805919975, 0.64963751109942836, 0.74915475749297489, -0.95417005675424726], [0.86875515153318328, 
-0.50354690200796526, -0.20723844381933199, 0.588505435821433]]]])+(1.-msk_arg0)*numpy.array([[[[-0.45187484182260351, 
0.26218751303890997, 0.85384327705187646, 0.75303646136734614], [-0.80410026271033752, 0.53147720985892377, 
0.50441076578741106, 0.58596387599943167], [0.98671580330013953, 0.4923260155932212, -0.53541284591070015, 
0.24215788450392051]], [[-0.085406833040039487, -0.13311435037468855, -0.73318155510231131, -0.38549084622077223], 
[-0.68928525519152739, -0.68740139381758092, 0.90616221111805806, 0.19007205825157003], [0.17561487416286736, 
-0.90180859526252921, -0.73315538314293272, -0.54323538049256781]]], [[[0.89111566972666645, 0.20443274795271837, 
-0.728969135510944, -0.67352997540305459], [0.60201785288645926, -0.035196174636839217, 0.34578764531428541, 
0.34953478993313891], [-0.35854486353024217, -0.55920923257522803, -0.42049272504701074, 0.42935874870770507]], 
[[-0.081340864936127133, -0.75079067727008786, -0.31332393287911153, 0.89072681020132172], [0.69085093202892045, 
-0.97027053638239891, -0.12717831450461681, -0.83994461823002342], [-0.72897706097294757, -0.42745265769195995, 
-0.33355490737943816, -0.97263634639973628]]], [[[-0.76033244671178002, -0.69089710234761292, 0.87282786896240006, 
0.92436829090199435], [0.58083081564855843, -0.81615646614553428, 0.90398073899985509, 0.42059572648475063], 
[0.019855973515656711, -0.45396080542761208, 0.53736366245784306, 0.59033308886792213]], [[-0.94138407203855867, 
-0.67937937246377555, -0.054229586011050657, -0.5411240345139039], [0.80177386348099788, 0.23753684966099531, 
-0.34285918400305304, -0.52316423985274541], [-0.15503198735542156, 0.62711500005635878, -0.59892199914183242, 
-0.11115015793655636]]]])
      arg1=Data(numpy.array([[[[-0.56067461849346856, 0.82436543003426865, -0.85784285960585782, -0.84913542810300635], 
[0.31644257130645426, -0.15652880988651585, -0.92651670576264755, 0.66860658189646704], [-0.80377814282619164, 
-0.38511769253035455, 0.4108650741538038, -0.50997661822690676]], [[-0.019249788314307237, -0.5845033284576342, 
0.14061092565362476, 0.83198289487612698], [0.84011230181354235, -0.35733816393052509, 0.96442765761882199, 
-0.76747774313560524], [0.59919634836449132, -0.55144540917616425, 0.97222443025153171, -0.21855343547764483]]], 
[[[-0.080736106169028155, -0.59334688941447178, -0.34294452941442688, 0.7014605270562615], [-0.44551858269727185, 
-0.1899099392185104, 0.73618040445628052, 0.99607451421460791], [-0.38697159025089833, 0.53225190075118189, 
0.80372671392552042, -0.26158470174624071]], [[0.78921994505062454, -0.78939118206556924, -0.7493958134408456, 
-0.43440374824874217], [-0.040305328457116429, 0.31872351340112015, -0.72449815921690686, 0.22002072913721205], 
[-0.061081502789878339, -0.5769041864612805, -0.53698267097296615, -0.060752999644593242]]], [[[-0.61527880545476776, 
-0.016981853524558055, -0.67572274024216328, 0.39259015365922023], [-0.95150026727961223, -0.26067466866724942, 
-0.535056872864345, 0.50070596980230753], [-0.39057009491319405, -0.92296854823798524, 0.16321388919501878, 
0.56519023268513768]], [[-0.39144376514710566, 0.085591582644400699, 0.86408260436397399, -0.6950100726632451], 
[0.32560339228050483, -0.4547551945702446, -0.43659989630539719, 0.036767150522588832], [-0.17649832158303713, 
0.45518147529791864, -0.64714458530939223, -0.87261235042399221]]]]),self.functionspace)
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.755579480427)+(1.-msk_ref)*(-0.59812450412)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg0.isLazy() and not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j      
        res=inner(arg0,arg1)
        msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
        ref=-(msk_ref*(-0.755579480427)+(1.-msk_ref)*(-0.59812450412))
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank0_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.403235728374)+(1.-msk_arg0)*(-0.295926359992)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.541361045232)+(1.-msk_arg1)*(-0.810551377619)
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(0.21829611538783306)+(1.-msk_ref)*numpy.array(0.23986351876555856)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg0.isLazy() and not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j  
        res=inner(arg0,arg1)
        msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
        ref=-(msk_ref*numpy.array(0.21829611538783306)+(1.-msk_ref)*numpy.array(0.23986351876555856))
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank1_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.90850727965316946, 0.50515315395193428])+(1.-msk_arg0)*numpy.array([0.74762075729523225, 
0.81043766812190188])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-0.61574683887358339, 0.93798419219177864])+(1.-msk_arg1)*numpy.array([-0.007714614761548555, 
-0.86522376090217512])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.0855848124973)+(1.-msk_ref)*(-0.706977533319)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg0.isLazy() and not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j 
        res=inner(arg0,arg1)
        msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
        ref=-(msk_ref*(-0.0855848124973)+(1.-msk_ref)*(-0.706977533319))
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank2_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[0.97936996315172942, 0.70300297214989427, -0.25061232251134102, 0.44815350126104292, 
-0.30428700430274702], [-0.12023987608121867, 0.12967971629631569, 0.25525677189486284, -0.64554781120983828, 
-0.31958253742431997], [0.076147714924661436, 0.29338897471920244, -0.28506107676730741, -0.85770078975051978, 
0.91865830392579051], [0.61918931910660246, 0.84653161661238596, 0.85127368640798529, -0.28022811001072645, 
0.8813404201048407]])+(1.-msk_arg0)*numpy.array([[-0.49148868927913258, -0.46133680864660875, 0.042472319927527868, 
-0.16381315384583361, -0.081512938414658453], [0.13382615090707151, 0.23827946149774837, -0.2677152798942255, 
0.90613469033837268, 0.70139580236050714], [0.72992965875251414, -0.43511959013079471, -0.15960126320034851, 
0.12536531981177279, 0.17187813577533584], [0.76790552207840812, 0.038936831804500249, 0.97968906398820965, 
0.58515284859248307, -0.33790017411269457]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.77792892176813555, -0.48474074670279599, 0.93420142438233378, -0.44466398223378789, 
-0.51263103365865992], [0.63495992637336518, -0.66572912657297234, 0.82268523377741287, -0.66447875079745655, 
-0.43956314173978117], [-0.70214786437828991, -0.98050935373364578, 0.15802372275413235, 0.3060961579403354, 
-0.40748742528675685], [0.72454726587244744, 0.88591188793736531, 0.46813774668437813, -0.83119094161369311, 
0.92495780926047999]])+(1.-msk_arg1)*numpy.array([[-0.46866746866959152, 0.77485146518085779, 0.078830102748751463, 
0.86057905914440269, 0.63438949285317259], [-0.46737572651749382, 0.13973168612107201, 0.54911400282390876, 
-0.44226114996595678, -0.28915038149170735], [-0.79873536394315914, 0.17697126209294067, 0.31687102417355439, 
-0.0020160127671704497, -0.045717467295027348], [-0.49355269181235517, -0.94737060160283404, -0.076589075013161478, 
-0.54223770216012301, 0.1330028382259909]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(2.38259814517)+(1.-msk_ref)*(-2.66813933913)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank3_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.82515479906730982, 0.86187150324023643], [-0.64054582590394871, -0.4585412457900413]], 
[[0.18034602851741566, -0.48817089152477067], [0.84083037658278736, 0.27098186545704439]], [[0.16449913181804132, 
0.4832607127000943], [-0.58311581170613991, 0.27572947234154332]], [[-0.91649111471063804, -0.31690517728986922], 
[-0.33677447363475643, 0.75657122713227598]], [[0.15263583122141222, 0.7147460693527834], [-0.77672397702369111, 
-0.88628760970188258]], [[0.26526071178948207, -0.10765846826010272], [0.26436608712533771, 
-0.10491818539219988]]])+(1.-msk_arg0)*numpy.array([[[0.74852211953177816, 0.3186875609801596], [0.18785460974395507, 
-0.0051836560942222221]], [[0.27784421485642063, 0.10151035726096147], [-0.46057699701818189, 0.1048205876836934]], 
[[0.8619190179244669, 0.7182321218553569], [-0.52469116547623029, -0.64351673986699343]], [[-0.95304560847909769, 
-0.50575897884598531], [-0.49654624538600833, 0.27045902883502704]], [[0.72623722391565204, 0.28200506732420449], 
[0.49675058588390431, -0.80454908927028956]], [[-0.7252012828553136, -0.15094050576566165], [-0.7527450566268663, 
0.39277655861690786]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.77848925392078994, -0.52609071667174501], [0.72333872332332971, -0.52349654559260528]], 
[[-0.781853676722845, 0.26887365320155632], [-0.25467080531596542, -0.12846944357675794]], [[0.27015633772991388, 
0.46747008919478983], [-0.45708381185132296, 0.033007137251819252]], [[0.61986039250907288, -0.25968752477805968], 
[0.40222628722983833, -0.75395899203083494]], [[-0.60867569571751678, -0.32633068042911528], [-0.64743474613387764, 
0.066047300331868453]], [[0.78079227379994265, -0.46586476049812209], [-0.70304649915648398, 
-0.13207766237217911]]])+(1.-msk_arg1)*numpy.array([[[0.65793158785296946, 0.69958336185059955], [-0.83288605047716091, 
-0.74606078974779488]], [[0.91843015701277997, 0.75420640449017595], [-0.61909674393108149, -0.88241525774172858]], 
[[-0.2168876481400932, -0.88985122212027634], [-0.20849132046611341, -0.096336219615894558]], [[0.99177271903176045, 
-0.36477583556657356], [-0.87653688087840109, 0.86997207008395683]], [[0.88000911935290405, -0.11788561531713126], 
[-0.26746816228835169, -0.37355939266452176]], [[0.26514166517441073, 0.67136051090322457], [0.57373631491666877, 
-0.47031764453494174]]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-2.28253572206)+(1.-msk_ref)*(0.205671973309)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg0.isLazy() and not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j       
        res=inner(arg0,arg1)
        msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
        ref=-(msk_ref*(-2.28253572206)+(1.-msk_ref)*(0.205671973309))
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")       
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_inner_expandedData_rank4_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-0.85848399921195329, -0.57896707886291332, -0.43034381488523143, -0.44028867476623801], 
[0.1542009868569123, 0.89134745980511632, -0.31625638349919094, -0.46821880735750909], [-0.00014140379166804884, 
-0.45054390630032426, 0.063612421180540002, 0.2139474282001117]], [[0.14851515293243445, 0.030852297972091103, 
-0.32904205566508327, -0.60470261985184459], [0.2050582049845926, 0.66891227664110531, 0.12664539701770172, 
-0.1512184013561424], [0.43610165485599883, -0.031232254854981267, -0.94367086360711649, 0.99864791296239752]]], 
[[[-0.0739525337370186, 0.13690549306105271, 0.44836619794469867, 0.86486218754491251], [0.18292690193610994, 
0.82337273414879975, -0.99426240224917328, -0.2611768138402506], [0.63533397617306964, -0.46590412506632273, 
0.19710132665811919, 0.92162975601912445]], [[-0.50199990627637336, -0.41579217916435351, -0.033835742233214949, 
-0.99227315756588963], [0.41300955053414445, 0.2404521065620393, -0.70048488468070169, -0.02128933259204846], 
[-0.2371289244231205, 0.72390162758131571, 0.57245237356825229, 0.3193794886016077]]], [[[0.036010838035816839, 
-0.16697536583993622, -0.40253894630502418, 0.19689771264364264], [0.692038111953033, -0.29737990759908306, 0.3210505962031367, 
-0.39445483259245995], [0.24932266060009267, -0.87499616800979152, -0.024935835903332482, 0.61934652294663839]], 
[[0.1921206747357862, 0.91683813432396177, -0.35927087384140122, 0.41710362048575012], [0.32246985496822211, 
0.83179661383912151, -0.56990023935119183, 0.67618438981012918], [0.50718837602815503, -0.84834818782953691, 
0.37937837632206328, -0.25426188524343107]]]])+(1.-msk_arg0)*numpy.array([[[[-0.52660798398775155, -0.91198538123189898, 
0.82532828731900576, 0.75474274557208942], [-0.14733998379424951, 0.45177887906173697, -0.36321745594766464, 
0.34951620533919336], [-0.69560845662216808, -0.11401790466463391, -0.73834803054149334, 0.10722259179197136]], 
[[0.0177056943396805, -0.47868540405574977, 0.40133180486325815, 0.32115356551548468], [0.91979852980436405, 
0.34980505610258494, 0.4856493865847229, -0.44919356645331976], [-0.85304974198046613, 0.71519478383129353, 
0.67204216667633654, 0.7475580019235577]]], [[[0.24217862286310465, 0.95354035165723561, -0.31263155472319304, 
-0.44578837088375711], [-0.11528633107193453, -0.23498522545753464, 0.9180284456824146, 0.20783304264420299], 
[0.15122575609614941, -0.091557510721294477, 0.40550183981931331, 0.96902348433008778]], [[0.85700567482802037, 
5.7867086591834038e-05, 0.72809005526213588, -0.2944154096127316], [0.048631557146810112, 0.90940535486443252, 
0.87370296139337378, 0.29272107817043769], [-0.56122498183076353, 0.73490381573551034, 0.98989427623480863, 
-0.86335822872138035]]], [[[0.44987693855421917, 0.61924124200328845, 0.92027614760278231, -0.92836895964474153], 
[-0.26038483014295299, 0.7091524621105445, -0.61699126503434165, 0.7867106439773095], [-0.010275133721955321, 
-0.64211611345792963, -0.7266804879141564, -0.37259764086320635]], [[0.23436695670222973, -0.62268616476340166, 
0.56504976209184354, 0.75852938727342356], [0.018278357410181023, -0.50994535892748472, -0.47740324751282759, 
-0.4048677745526883], [-0.75546010712197464, -0.54279502794620971, 0.65458780889854662, 0.63133081314691397]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-0.85032148694968157, -0.89307913993242494, 0.82344755515608292, 0.52020307096338647], 
[-0.71582198836216415, -0.61956713081684667, -0.025632675537781591, -0.96725308876812122], [-0.74840121493833189, 
-0.95704548609145279, 0.02925160124606685, 0.28204436866124238]], [[0.77582565278870219, 0.20399239563725824, 
0.42072111286039204, -0.98584512646851774], [0.0051541472756353457, -0.31450623803480027, -0.18137342071886264, 
0.84383008132128734], [0.60539191928307545, 0.93603849652654358, 0.52593897438047366, -0.91238023684979264]]], 
[[[0.11132156128689674, -0.23856418739373941, -0.45006040991635454, 0.48701817686331794], [-0.64125441277940864, 
0.88477096108526365, -0.46055858667528038, -0.3010739289286819], [0.302230111668218, -0.12141629391555764, 
-0.69959171269062126, -0.088172608374930794]], [[-0.59067769345245158, -0.43595737005003898, 0.45225872072232964, 
-0.40597848146882853], [-0.37249427288094927, -0.16711051368782659, -0.56238851046142724, 0.40758376843097333], 
[-0.78419565596843221, 0.94913458725501987, -0.75346190090301124, -0.80854672867882327]]], [[[0.047814610347620023, 
0.89226292666026841, 0.65891391291871959, -0.043438552217093696], [-0.85376353357903323, 0.58078752052539806, 
-0.24143839852724791, -0.58458556312805987], [-0.63293655693372997, 0.96460464839254167, 0.65159578698950993, 
0.061326179501126621]], [[-0.17234535569232556, -0.049385685202222529, 0.83562180983328016, -0.9741260950898365], 
[-0.46358357818427764, -0.93135550143789358, -0.22307605504315453, 0.40839527256023089], [-0.82005299493558326, 
-0.98533424617530185, 0.94391858495261372, 0.67582394987210059]]]])+(1.-msk_arg1)*numpy.array([[[[0.41324938725852634, 
-0.96435227534002066, -0.21728111111808257, 0.70938235599301858], [-0.3679456961352745, -0.24296128506749803, 
-0.51762107537142632, -0.83438579427741177], [-0.96896461424687486, -0.16496310799157143, -0.86728612607361533, 
-0.39351263591764996]], [[0.076791725845727354, 0.85612310092343114, -0.31486157389407832, -0.88870188389020055], 
[-0.66810623930448432, -0.36426484354507327, -0.87988747852457183, 0.32315595765127236], [-0.75543936721157112, 
0.49986454498032074, -0.30458644036080273, 0.059409472050175882]]], [[[0.0079330968533004587, 0.21116245050181348, 
-0.44143395119942364, -0.28063483186401439], [0.88751279455182308, -0.073463439487072346, 0.60096508702459972, 
0.79177470299857999], [0.95243642490697522, -0.21215275520223864, 0.18355745101229259, -0.14013995159823978]], 
[[0.1627217916908017, 0.41334334239730808, 0.33573546545915911, 0.077565211932971945], [0.91790152497156097, 
0.63377408643159061, 0.98034579788904486, -0.49206922858999569], [0.49897607647039011, -0.010403252244100214, 
0.046849195786473175, -0.32202425604129448]]], [[[0.44597817005287776, -0.15805646513978289, 0.66962570699549295, 
-0.62314847800938744], [-0.61692082273961146, 0.22088164537570232, 0.68823694440407768, -0.54817719482377592], 
[0.13092507486506988, -0.38768295304299594, -0.26250766251874946, -0.020094679080584044]], [[-0.81049573082826165, 
0.16068165668178569, 0.25337709546983245, 0.53604096791339551], [-0.43325951698250842, 0.57308382811271863, 0.8779157187028388, 
-0.4315436666110668], [-0.47430817479538279, -0.29432135991384145, -0.84855309577270099, 0.77697355338037299]]]])
      res=inner(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.113297048661)+(1.-msk_ref)*(5.16080189005)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_float_rank0(self):
      arg0=-0.081861603174
      arg1=0.765137400833
      res=outer(arg0,arg1)
      ref=-0.062635374280529704
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result") 
      arg0=arg0*1j
      arg1=arg1*1j
      res=outer(arg0,arg1)
      ref=-ref
      self.assertTrue(isinstance(res,complex),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result") 
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_array_rank0(self):
      arg0=0.470792845884
      arg1=numpy.array(0.986628916768)
      res=outer(arg0,arg1)
      ref=numpy.array(0.46449783555674595)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=numpy.array(arg1*1j)    
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_constData_rank0(self):
      arg0=0.806587192294
      arg1=Data(0.459316802107,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(0.37047904978481383,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j    
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_expandedData_rank0(self):
      arg0=0.62510111232
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.932536673263)+(1.-msk_arg1)*(-0.39927472313)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(0.58292971173616104)+(1.-msk_ref)*numpy.array(-0.24958707354962301)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j    
        ref=-ref
        res=outer(arg0,arg1)        
        msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
        ref=-(msk_ref*numpy.array(0.58292971173616104)+(1.-msk_ref)*numpy.array(-0.24958707354962301))
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_array_rank1(self):
      arg0=-0.147579432862
      arg1=numpy.array([0.4366735032511968, -0.4840678581242126])
      res=outer(arg0,arg1)
      ref=numpy.array([-0.064444027955627856, 0.071438459968642856])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j    
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_constData_rank1(self):
      arg0=0.307392721359
      arg1=Data(numpy.array([-0.55600277881165461, -0.97997933943118731]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([-0.1709112072620588, -0.30123851602330931]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j    
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_expandedData_rank1(self):
      arg0=-0.533630322003
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.37338763703449107, -0.059381069842770806])+(1.-msk_arg1)*numpy.array([0.66805487237042271, 
-0.44911334201438247])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.19925096498253622, 0.031687539421061568])+(1.-msk_ref)*numpy.array([-0.35649433665848967, 
0.23966049731483582])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_array_rank2(self):
      arg0=0.507057355338
      arg1=numpy.array([[0.82727299573137292, -0.94610810308954418, -0.57533662315989043, -0.28952788602325241, 
0.018136635315888716], [-0.92320697591850909, -0.40737177126670399, -0.067864270738446253, -0.44712910432163211, 
-0.26252030844042507], [-0.6037930078105489, 0.95214729595957825, -0.34240635693853072, -0.64429323264604199, 
0.76924375662754518], [-0.56483829149320042, 0.816975372319108, 0.76638795986714259, -0.46644827039746395, 
-0.35838294865216191]])
      res=outer(arg0,arg1)
      ref=numpy.array([[0.41947485735778095, -0.47973107261607756, -0.29172866656833152, -0.14680724418344251, 
0.009196314337997431], [-0.46811888763848197, -0.20656085297769719, -0.034411077642552856, -0.22672010113180602, 
-0.13311285332021847], [-0.30615768571176444, 0.48279328978113084, -0.17361966180004085, -0.32669362260742862, 
0.39005070484553961], [-0.28640541027796246, 0.41425337166409515, 0.38860265209282813, -0.23651602638954558, 
-0.18172071014166363]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j    
      ref=-ref
      res=outer(arg0,arg1)      
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_constData_rank2(self):
      arg0=-0.461803900652
      arg1=Data(numpy.array([[-0.52799308286320312, -0.69180625252696748, -0.70146211143834436, -0.79159352615299805, 
0.23646350770741109], [0.25829241615476461, 0.3018732714936927, 0.29415929156311038, -0.56921835144236077, 
0.60564013367395853], [-0.5177543192396723, 0.47622606084243757, -0.6682456782013706, -0.51288857119335751, 
-0.48979664027252068], [-0.38711325402410046, 0.62799192622923905, -0.10406963163713212, 0.86589686023678203, 
0.48939305358190266]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.24382926518345957, 0.31947882591234072, 0.32393793922175912, 0.36556097810826205, 
-0.10919977022111776], [-0.11928044528907927, -0.1394062542783433, -0.13584390825684978, 0.26286725501873759, 
-0.27968697612198423], [0.23910096420426005, -0.21992305248913621, 0.30859846078718056, 0.2368539427768824, 
0.22618999900405529], [0.17877041070238711, -0.29000912111057531, 0.048059761829436061, -0.39987454761959623, 
-0.22600362109607669]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_expandedData_rank2(self):
      arg0=0.542639921047
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.74293955171720949, -0.6474141214322553, 0.89013588888960116, -0.14757583754509063, 
0.081887518972254991], [0.21487307077342988, -0.050194366663281365, -0.93093206605908807, -0.4171126339311515, 
-0.67949074748359117], [-0.39271355958770982, 0.91153629318962737, -0.43013281164693429, 0.75192193609147018, 
-0.50864788267834871], [-0.17454337482838866, -0.48266661220717322, -0.6330500907575769, -0.85640884246823479, 
-0.72710868516954452]])+(1.-msk_arg1)*numpy.array([[0.13502712344398327, -0.30609572077649028, 0.076158340221367027, 
-0.76620762911605222, -0.75414774859793399], [-0.045647655526027675, 0.51826952770744628, 0.9282009241583189, 
0.62578373001108756, 0.27753347187078115], [0.7635669473363762, -0.13422090527602437, -0.85991252821868414, 
-0.50396182803855938, 0.35889330854675716], [0.57190812057601392, 0.4595840391218784, 0.51334244288848341, 
-0.85329159837300739, 0.86017378063831629]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.40314865968646924, -0.35131274773866755, 0.48302326846809335, -0.080080540833902758, 
0.044435436829833551], [0.11659870615960571, -0.02723746716316373, -0.50516090282636039, -0.22634196674407769, 
-0.36871880556661635], [-0.21310205496873427, 0.494635982167832, -0.23340723495178708, 0.40802286003413124, 
-0.27601264689726801], [-0.094714203136141789, -0.26191417234009035, -0.34351825126744434, -0.46472162666085692, 
-0.3945581995129398]])+(1.-msk_ref)*numpy.array([[0.073271107604837349, -0.16609975775495828, 0.041326555724787953, 
-0.41577484736909115, -0.40923067475690406], [-0.024770240190621184, 0.28123373559619913, 0.50367887620095897, 
0.33957523384567084, 0.15060074126384146], [0.41434190801665766, -0.072833621441829532, -0.46662286641985401, 
-0.27346980657751113, 0.19474983661408432], [0.31034017739546715, 0.24938864670352595, 0.27856010267904557, 
-0.46303008557113867, 0.4667646324122165]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j    
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_array_rank3(self):
      arg0=0.329599447984
      arg1=numpy.array([[[0.32822728567176518, 0.76182410862170369], [-0.91107903577061466, -0.82291242821270738]], 
[[-0.44260136214831292, -0.61872702571531413], [-0.82643984233761358, 0.59956489999270834]], [[0.1872384874131392, 
-0.95740409540478066], [0.34436766116193884, -0.24392537634176192]], [[0.55128906940654687, -0.055866350184200497], 
[-0.14960910116281423, 0.39785898000498165]], [[-0.69334745701646527, -0.36482981315924334], [0.62969646397122481, 
0.85157868691814409]], [[-0.4054544548193193, 0.40561559734997887], [0.023758608459736985, -0.58272572766033925]]])
      res=outer(arg0,arg1)
      ref=numpy.array([[[0.10818353217063062, 0.25109680566245424], [-0.30029114725959566, -0.27123148207790621]], 
[[-0.14588116464095621, -0.20393208612841801], [-0.27239411582628553, 0.19761626006805122]], [[0.061713702092689962, 
-0.31555986134283281], [0.1135033910224429, -0.080397669391482268]], [[0.18170457295589357, -0.018413518181581431], 
[-0.04931107715661414, 0.13113410018503457]], [[-0.22852693909358954, -0.12024770502531482], [0.2075476069222584, 
0.2806798651229786]], [[-0.13363756449101502, 0.13369067698016712], [0.0078308242331922451, -0.1920660781627985]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_constData_rank3(self):
      arg0=0.676272532347
      arg1=Data(numpy.array([[[-0.78502641500144188, -0.2959496388941778], [0.9709955494182978, -0.72028076814454289]], 
[[-0.42229143063252117, -0.57707492065624755], [-0.37028192685951566, -0.77738667668782058]], [[-0.52303963001352782, 
-0.74377516733057392], [0.32553195575683924, -0.14808641834981451]], [[-0.51098172394733266, 0.48839251208215906], 
[-0.05041292302866851, 0.43615558206766214]], [[-0.18673861291251259, -0.77362964466616568], [-0.10240459910802069, 
0.31449844481000633]], [[0.71163512256337746, 0.6064924623513579], [-0.012160009387012227, 
-0.56867060961205484]]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.53089180163193517, -0.20014261174200376], [0.65665761910231268, -0.48710609907360664]], 
[[-0.28558409518208988, -0.39025991794586762], [-0.25041149635943355, -0.52572525645611778]], [[-0.35371733510683534, 
-0.50299471590710387], [0.22014832007939297, -0.10014677714355522]], [[-0.34556290443665305, 0.33028644092488008], 
[-0.034092875119587848, 0.29496003998196829]], [[-0.12628619464122146, -0.52318447889672626], [-0.069253417562711334, 
0.21268665969070522]], [[0.48125928644266136, 0.41015419336342923], [-0.0082234803415122126, 
-0.38457631323338359]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j    
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_expandedData_rank3(self):
      arg0=0.305104655943
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.30293977445085996, 0.80288896011525268], [0.43540848013586975, 0.40694710798426192]], 
[[-0.59373430985932441, -0.10176913923597497], [-0.30052400560538772, 0.48490916950152085]], [[0.76780025804496965, 
-0.13092427930556072], [0.71353871888096498, -0.21625028363463095]], [[0.048762689975171636, 0.027421493530511487], 
[-0.80924187302600847, -0.71846315628216839]], [[-0.71726816616769762, 0.39535977450661641], [0.98564221665367024, 
-0.34445301274369311]], [[0.055019366867498087, -0.40840665672813459], [0.45115914313174343, 
-0.18779261237584977]]])+(1.-msk_arg1)*numpy.array([[[-0.61380336879006725, 0.58597512263121998], [-0.6415509561610433, 
-0.94320197837698028]], [[-0.67863445716181414, -0.49823099756251277], [-0.84557582756626659, -0.97939605205818969]], 
[[-0.44458557380422148, -0.40082933239532137], [0.63008051889144734, 0.76000112210178639]], [[0.15432738338689655, 
-0.14349590068753426], [0.79389836449546203, 0.6986714070934783]], [[-0.27877479322191001, -0.2681783738236474], 
[-0.4386996383193229, -0.32550724130405539]], [[-0.74293716545466415, 0.67250397682981777], [-0.47603169981880589, 
-0.37618199584110656]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.09242833565526147, 0.24496515993634904], [0.13284515452649295, 0.12416145736851268]], 
[[-0.18115110233114812, -0.031050238212201299], [-0.091691273332825993, 0.14794804532433831]], [[0.23425943356371101, 
-0.039945607192100482], [0.21770398532614302, -0.065978968385907263]], [[0.014877723747726992, 0.0083664253490682624], 
[-0.24690346324422066, -0.21920645410514972]], [[-0.2188418570574189, 0.12062610797451954], [0.30072402939495479, 
-0.10509421794167362]], [[0.016786664998306396, -0.12460677248584391], [0.1376507551407222, 
-0.057296400387559537]]])+(1.-msk_ref)*numpy.array([[[-0.187274265651311, 0.17878373818152046], [-0.19574018374937929, 
-0.28777531509740889]], [[-0.20705453256337916, -0.15201259709141823], [-0.25798912194327256, -0.2988182954950877]], 
[[-0.13564512853273156, -0.12229489555231286], [0.19224049993272416, 0.23187988087511388]], [[0.047086003210833269, 
-0.043781267408492426], [0.24222208735305073, 0.21316789927842547]], [[-0.085055487371535093, -0.081822470476801093], 
[-0.13384930221170924, -0.099313774865009377]], [[-0.22667358825326839, 0.20518409447092045], [-0.14523948799114966, 
-0.1147748784130293]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_array_rank4(self):
      arg0=-0.311141703693
      arg1=numpy.array([[[[-0.96242947025367509, 0.31613060708727825, -0.16474461524119355, 0.097607895254317345], 
[-0.4608076213705925, -0.045189493507184153, -0.23256772880073062, -0.025792894346069906], [0.47221318146065183, 
-0.53932971340006897, -0.9206012533737864, -0.43277979347555129]], [[0.52164453678492451, -0.4177205567007356, 
0.36074511135280884, 0.44545791479432495], [-0.60244720292977383, 0.73113422773618209, -0.35506077006676295, 
0.12888290818921266], [-0.018600374399256658, 0.34889421111020491, 0.0095060116754939195, -0.12269843664561075]]], 
[[[0.11970738714252849, 0.0046044121318216114, 0.7711293226528344, 0.083756858882147567], [-0.96727123111763413, 
-0.62793542034570082, 0.66371387282856142, -0.27404421582471739], [-0.025508922135425216, -0.5860353376499825, 
-0.93418277964334107, 0.098139531888658205]], [[0.57463111535317091, 0.58390453459973113, -0.19312942733947369, 
0.68367140971995921], [0.020894590195250196, -0.71205314610397163, 0.15500610556452066, 0.99622189868944999], 
[0.089625133479853458, 0.87254489846494532, -0.27731691759675536, -0.94015242319703418]]], [[[0.13787664510865594, 
-0.92567081934687612, -0.23120204812863565, -0.27798371946865919], [-0.17735552235919139, 0.21911400717697926, 
-0.11484383008164167, -0.93977080140677649], [-0.041189774819279545, -0.96264056886361749, -0.67189498094862232, 
0.15088102270862103]], [[0.095815805990953073, 0.7429753189232895, -0.37866380561605983, 0.91725934981023949], 
[-0.18496465764990355, -0.47182302550197419, 0.13864024675702846, 0.87186119166113274], [0.28701359062133958, 
-0.17167237546109382, 0.63761401351486469, -0.43423124188146778]]]])
      res=outer(arg0,arg1)
      ref=numpy.array([[[[0.29945194505864181, -0.098361415678494227, 0.051258920260317739, -0.030369886823271755], 
[0.14337646838775525, 0.014060335998828468, 0.072361519362965046, 0.0080252450899979972], [-0.14692521378574402, 
0.16780796587930935, 0.28643744239621199, 0.13465584226569072]], [[-0.16230536989716976, 0.12997028567926514, 
-0.11224284854506973, -0.1386005345324347], [0.1874464491043781, -0.22748634924576874, 0.11047421291295953, 
-0.040100847630841464], [0.0057873521799039112, -0.10855553925329554, -0.0029577166680344004, 0.038176600618327103]]], 
[[[-0.037245960380109341, -0.0014326246351975983, -0.23993049121748097, -0.026060251768527446], [0.30095841878272594, 
0.19537689649525561, -0.20650906515625561, 0.085266584198790005], [0.0079368894925766695, 0.18234003338045129, 
0.29066322161846631, -0.030535301151425942]], [[-0.17879170422573282, -0.18167705168916282, 0.06009061905556938, 
-0.21271868718615194], [-0.0065011783913076883, 0.22154942899842625, -0.048228863768091407, -0.30996617881405725], 
[-0.027886116724593339, -0.27148510625662159, 0.086284858203819523, 0.29252062668419959]]], [[[-0.042899174258519579, 
0.2880147957900609, 0.071936599151949429, 0.086492328074269056], [0.055182699386120027, -0.06817550549594581, 
0.035732704950179113, 0.29240188823021263], [0.012815856711982919, 0.29951762663978643, 0.20905454907482426, 
-0.046945378460433888]], [[-0.02981229311669948, -0.23117060653130411, 0.11781810160608347, -0.28539763682787384], 
[0.057550218704099279, 0.14680381999605527, -0.043136762576336647, -0.27127237655685715], [-0.08930189756883819, 
0.053414435377910949, -0.19838831046325628, 0.13510744839552941]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_constData_rank4(self):
      arg0=-0.672892796531
      arg1=Data(numpy.array([[[[-0.15035387426968971, 0.7448082478008573, 0.19832246991022884, 0.66992547222564558], 
[-0.87242392963594284, -0.043981690167481657, 0.070163273395474457, 0.41278977409290363], [0.92244264078304616, 
-0.52871594235556008, -0.80680164879181415, 0.3638857320985347]], [[0.10604864161257521, -0.88216765288310506, 
-0.91687431340483028, 0.64065686176858061], [-0.57691631049459136, 0.20733489672099492, -0.23698932317186117, 
0.12997631174408308], [0.74193649559644603, -0.46758879464846692, -0.11845259801302221, 0.92706200519521409]]], 
[[[-0.91657636595994929, 0.62692336839030194, 0.77538458911871211, -0.78481494192413681], [0.95702122108174481, 
0.41867966052078098, 0.0038112330327841892, -0.4421255675128164], [0.79198271993779445, 0.41795257294183141, 
0.56507785727994664, 0.49783786847345235]], [[0.91348936829492056, -0.053708934298375421, -0.11362449808283137, 
-0.12015710605876295], [0.72087868244402276, 0.25135052985676531, 0.19612782696734055, -0.8311278206221322], 
[0.9492318741345187, 0.96648002899763163, 0.84111363429424535, 0.73138033303662486]]], [[[0.82118391220944731, 
-0.64895386300413671, 0.52527934759865125, 0.83792978829018172], [-0.97163515948825285, 0.94417103202880415, 
-0.56367072296837883, -0.22800894175864372], [-0.74379926261052098, -0.041615774090976387, 0.87421839143037627, 
-0.73199242523809449]], [[0.8704284040109711, -0.91798362066932904, 0.27031454140249722, 0.23116809961765994], 
[0.34664387098973815, 0.69188536416441937, 0.98111032452439617, -0.47437466946671369], [0.55282380606543646, 
0.12145535131127505, -0.43470110890054059, 0.1573076701434708]]]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.10117203892652984, -0.50117610474171603, -0.13344976139273398, -0.45078802447294447], 
[0.58704777777287598, 0.029594962492935646, -0.047212361248816302, -0.27776326546857588], [-0.62070500819550267, 
0.35576914902190249, 0.54289101770095904, -0.24485608788933896]], [[-0.071359367022948705, 0.59360425895727842, 
0.6169581208139775, -0.43109338733192759], [0.38820282953277591, -0.13951415847295701, 0.15946840841698903, 
-0.087460123892198857], [-0.4992437233699471, 0.31463713165734236, 0.079705899933288138, -0.62381334523299992]]], 
[[[0.61675763412457241, -0.42185221856648425, -0.52175070455875916, 0.52809632103027082], [-0.64397268579274924, 
-0.28172652761827743, -0.0025645512536596514, 0.29750310954134268], [-0.53291946722279082, -0.28123727562395545, 
-0.38023681964257788, -0.33499151553589501]], [[-0.61468041563286802, 0.036140354998707852, 0.076457106269333219, 
0.080852851118895397], [-0.48507407258899582, -0.169131960944747, -0.1319730019655079, 0.55925992349274367], 
[-0.63873129034228371, -0.65033744950311534, -0.56597930558020459, -0.49214055762443826]]], [[[-0.55256873915248883, 
0.4366763796961381, -0.35345668916538403, -0.56383691853880769], [0.65380629967542914, -0.6353258861449701, 
0.37928996910057294, 0.15342557445393845], [0.50049716587533277, 0.028002954607859499, -0.58825525818799951, 
0.49255243005761967]], [[-0.58570500295454042, 0.61770456568139775, -0.18189270770719163, -0.15555134902037324], 
[-0.23325416375045002, -0.46556467757113401, -0.66018206997418782, 0.31920329794069852], [-0.37199115685201789, 
-0.081726430997440744, 0.29250724482300328, -0.10585119807854079]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j    
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_float_rank0_expandedData_rank4(self):
      arg0=0.184226502646
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[0.65730545940807961, -0.4190764869923187, -0.91233525335447063, 0.94305454000779299], 
[0.9595976861826756, -0.25922356850125716, -0.47777030754140215, 0.79357928766377217], [0.82177864020114644, 
0.57356814342818963, -0.44599790358764069, -0.88625831217822171]], [[0.49549282141744944, -0.71265510458079873, 
-0.44671696505642289, -0.67951574493726463], [0.74664559903254291, 0.24573973378307845, 0.52339278211342077, 
0.065647498013209127], [-0.29054338210583186, -0.24512034012153916, 0.52316907153411552, 0.4542799080891522]]], 
[[[0.68013291879022675, -0.59070959949748247, -0.4160977604455971, -0.43475815273698326], [-0.88030221968057565, 
0.25403036852465499, -0.12823056233409758, -0.65367638618314627], [-0.52340105656477909, -0.48022748873278265, 
-0.28561993280419773, -0.24563030027621902]], [[-0.77189443540773661, 0.76873304337010162, 0.57399920199108667, 
0.69732176176072125], [0.14482102250488738, -0.79852504962461524, -0.79507037231493638, -0.089040013221562742], 
[0.5770105324062218, -0.96386024653463465, -0.46049683884974102, 0.11283893421851321]]], [[[-0.55499542827162029, 
-0.68767713721104951, 0.14947438804422863, -0.25844280664845853], [0.91170284074015062, 0.9253661175288237, 
0.83613688420269416, -0.93229398965659227], [-0.17563135055601586, 0.49891895617816684, -0.46929202386808755, 
-0.057064086850217244]], [[0.11910718268125309, 0.044264612600426467, -0.44246145558194883, -0.34283821975409845], 
[0.74769433079688574, 0.99312538575709097, 0.9078007214123327, 0.58344305988196599], [-0.87486562805907941, 
-0.74611549307091285, -0.00091221631963400718, -0.95022821552442194]]]])+(1.-msk_arg1)*numpy.array([[[[0.29776538578889933, 
0.83374470873797013, 0.61415531489559494, -0.71595656682753428], [0.21585625924589302, -0.14697143751100739, 
-0.084575556424320242, 0.47798305756602555], [-0.1338160730259188, -0.49693412037852136, 0.45330650019972452, 
0.39188345056255458]], [[-0.37922219968267679, -0.20117902721507952, -0.83581818808135022, 0.60961342345292491], 
[0.99750808819140802, -0.67257491810321568, -0.63236999997628573, -0.88694252684793451], [-0.69149659826752496, 
-0.53943479886510315, -0.92239128362138367, 0.14263274986737273]]], [[[0.10260430817581145, -0.55598580788903007, 
-0.33118324349604222, -0.13224437802571609], [-0.87170370823813759, -0.7395010730547209, 0.53349827266138816, 
0.18763994168424558], [-0.079441770664999467, 0.15424036810085751, 0.54771635762931381, 0.70513379076545357]], 
[[-0.2022059207172906, 0.17229783384265307, 0.02013953811621394, 0.016855182454158291], [0.2794366381731519, 
0.14338154424900629, 0.18037912383416299, 0.80754634988290008], [-0.019647626451636979, -0.56963585459384491, 
-0.85708942953768896, -0.94946928910942985]]], [[[0.15117646226304982, -0.46746072048210907, -0.95653799430305986, 
0.94491270358315815], [0.29755076195009256, 0.5322007738011707, 0.96013909022432498, 0.37016824938675996], 
[0.0015505646618996227, -0.083969383227496142, -0.31112268986924185, 0.63143348905277041]], [[0.66576238925902853, 
-0.67268178702623427, 0.026781312366727317, 0.54021600142687909], [0.059925616543585969, -0.4159475667789827, 
-0.45691210248484726, 0.70846379196415565], [-0.69182695381065407, 0.80175306625454734, 0.31836086192875279, 
0.57079550481579777]]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.12109308595670129, -0.077204995539657414, -0.16807633296590838, 0.17373563970982189], 
[0.17678332567237773, -0.047755851428334765, -0.08801795282633168, 0.14619833673839361], [0.15139340483322833, 
0.105666453092785, -0.082164633965282549, -0.16327226929330937]], [[0.09128290957580644, -0.13128995750955394, 
-0.082297104144863603, -0.12518480918250632], [0.13755190742559817, 0.045271771715951509, 0.096422821758778823, 
0.012094008966416618], [-0.053525791152222002, -0.045157662987925187, 0.096381608341148561, 0.083690398689492282]]], 
[[[0.12529850896296191, -0.1088243635946864, -0.076656235165616884, -0.080093973975456464], [-0.16217499920303352, 
0.046799126359105415, -0.02362346803110723, -0.12042451448862652], [-0.09642434613201388, -0.088470630723586588, 
-0.052618761306428334, -0.045251611163710535]], [[-0.14220341224687463, 0.14162100004828904, 0.10574586550426303, 
0.12846514938794293], [0.026679870485655267, -0.14710947716735806, -0.14647303404882633, -0.016403530231338868], 
[0.1063006323749541, -0.17756860225833554, -0.084835722100706279, 0.020787922213349297]]], [[[-0.10224486673485472, 
-0.12668835393782565, 0.027537143744500309, -0.047612014402794466], [0.16795982580174315, 0.17047696349920111, 
0.15403857390976763, -0.17175326115207679], [-0.032355949467882575, 0.091914094400366408, -0.086456028276758454, 
-0.010512717147088228]], [[0.021942699705354392, 0.0081547147703450798, -0.081513126517405443, -0.06315988619858888], 
[0.13774511161075656, 0.18296001650672933, 0.16724095200507291, 0.10748567441498308], [-0.16117343494229214, 
-0.13745424785825541, -0.00016805442222254076, -0.1750572208613658]]]])+(1.-msk_ref)*numpy.array([[[[0.054856275632848166, 
0.15359787179018658, 0.113143685744508, -0.13189817435288698], [0.039766443715062841, -0.027076033921469671, 
-0.015581058969369883, 0.088057147019305837], [-0.024652467131351836, -0.091548435042671669, 0.08351107115837525, 
0.072195317541883797]], [[-0.069862779573163628, -0.037062508589506049, -0.15397986163792568, 0.11230694896862831], 
[0.18376742644834052, -0.12390612492939977, -0.1164993134737172, -0.16339831976896946], [-0.12739199989025174, 
-0.099378186400325627, -0.16992892025248146, 0.026276732670810578]]], [[[0.018902432851615351, -0.10242732090806175, 
-0.061012730684148059, -0.024362919258238698], [-0.16059092551203377, -0.1362356963916424, 0.098284520939950423, 
0.034568250213138987], [-0.014635279573597717, 0.028415163582012386, 0.10090386900791133, 0.12990433217005182]], 
[[-0.037251689588008036, 0.03174182734226863, 0.0037102366720506495, 0.0031051713149854066], [0.051479634561722588, 
0.026414680440939704, 0.033230615134270516, 0.14877143976325899], [-0.003619613506474992, -0.10494202127344078, 
-0.15789858805836002, -0.17491740650216633]]], [[[0.027850710925077213, -0.086118653658676339, -0.17621964933822257, 
0.1740779626866551], [0.05481673623364039, 0.098045487262745729, 0.17688306664548908, 0.068194801975018524], 
[0.0002856551047878403, -0.015469385801321393, -0.057317045048345332, 0.11632678334158844]], [[0.12265107656626197, 
-0.12392581301732901, 0.0049338275135852533, 0.099522104616139501], [0.011039886754714482, -0.076628565511696972, 
-0.084175318657294898, 0.13051780664469484], [-0.12745286013659204, 0.14770416338157275, 0.058650508172417133, 
0.10515565957812352]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_float_rank0(self):
      arg0=numpy.array(-0.447383148159)
      arg1=-0.833832446992
      res=outer(arg0,arg1)
      ref=numpy.array(0.37304258517261873)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array(arg0*1j)
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_float_rank0(self):
      arg0=numpy.array([-0.7077557598550499, 0.34384980137545629])
      arg1=-0.117774156535
      res=outer(arg0,arg1)
      ref=numpy.array([0.083355337649769559, -0.040496620331747418])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_float_rank0(self):
      arg0=numpy.array([[-0.74034215690736938, -0.026692072802847244, 0.20164819911474718, 0.83226009454281358, 
0.91414431655496875], [0.69310910714139085, 0.40494661103705387, -0.070740970341285792, 0.8162325138343971, 
-0.24023752347865712], [0.55012364496204058, -0.40587261759026627, 0.18284956859225754, -0.82971461005420166, 
0.78296352261416358], [-0.32346840994150194, -0.29416615770821286, -0.93637076734800884, 0.61578785330116848, 
-0.12974392129838419]])
      arg1=0.964178232458
      res=outer(arg0,arg1)
      ref=numpy.array([[-0.71382179226108633, -0.025735915575689355, 0.19442480420079461, 0.80244706690161316, 
0.88139805134749083], [0.66828071382412468, 0.39044070766956146, -0.068206903746024325, 0.78699362246359428, 
-0.23163179075773749], [0.53041724363284937, -0.39133354303128232, 0.17629957385098963, -0.79999276616663406, 
0.75491638531320893], [-0.31188119975339523, -0.28362860598806422, -0.90282831138693886, 0.59372924396502325, 
-0.12509626470964516]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank3_float_rank0(self):
      arg0=numpy.array([[[-0.15791645491015305, 0.41702383789148212], [-0.90077914152118699, 0.0088427934648849238]], 
[[0.015777707296524834, 0.02048903309684591], [-0.15738286559336179, -0.84730278576106199]], [[-0.82290207937519555, 
-0.14283446280549805], [0.65481568668046997, -0.045837257239726314]], [[0.5564462185637844, 0.086215516976181572], 
[-0.58292048601627267, 0.26964649866065393]], [[0.38964230741750172, 0.75334221328525208], [-0.20459531306113932, 
0.51971180708870413]], [[-0.96004303118597267, 0.076424350124239648], [-0.36806507639747976, 0.95496844011225734]]])
      arg1=-0.639352122152
      res=outer(arg0,arg1)
      ref=numpy.array([[[0.10096422056954935, -0.26662507574394978], [0.57591505572195523, -0.0056536587675272655]], 
[[-0.010087510642728482, -0.013099706791313899], [0.10062306910750114, 0.54172483418175643]], [[0.52612419077194139, 
0.091321516911156322], [-0.41865779889767035, 0.029306147689852638]], [[-0.35576507070229002, -0.055122073741165685], 
[0.37269144978046176, -0.17239906114958373]], [[-0.24911863612763688, -0.48165094277071724], [0.13080844758802121, 
-0.33227884676968744]], [[0.6138055493461263, -0.048862070436030913], [0.23532318768481883, -0.61056109877409215]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array(arg0*1j)
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank4_float_rank0(self):
      arg0=numpy.array([[[[-0.87020281833627244, 0.38653786763242937, -0.39861176181522451, -0.65499761474764417], 
[0.05055951134247616, -0.025737969733530264, -0.12216547307173986, -0.87787279426303932], [-0.80700029009802332, 
0.94697847812541758, 0.29159653307353262, 0.63930031280993882]], [[0.23785001278341023, 0.93992568249991604, 
0.22199263237551592, -0.052778005340911482], [-0.16862506415237455, 0.25374353614534706, -0.11888781793537628, 
-0.65094094688477777], [-0.44952265798288749, -0.15066801692167542, 0.94480116118414803, 0.32746587459986021]]], 
[[[0.78917025548681541, -0.084561499269018459, -0.84479426524784618, 0.74962830861495], [0.97673242354717993, 
-0.71494837460876259, -0.80013195830983519, 0.12796340276698337], [0.019165508803590026, -0.21892235269927185, 
-0.39596112938741901, 0.64963194873588415]], [[0.95797829118030275, 0.53000085273002395, -0.93284698498085139, 
-0.86034783100269485], [-0.41924351261022297, 0.064151217188172405, -0.89760216010433047, -0.81302206425904222], 
[0.68233887569792473, 0.18494775692731746, 0.73925938459522, -0.36536865017631448]]], [[[-0.94205154403397806, 
0.21091249594021089, -0.076130048892607549, 0.36845534393432722], [0.5111382941147895, -0.54496836419397821, 
-0.11635448135542692, -0.16379745454182637], [-0.8580412291711903, 0.74391783752126805, -0.6049896435642117, 
-0.90606576695648267]], [[0.91914643333426516, -0.92427239857047083, 0.1023644003817421, -0.1239878505251808], 
[0.084055849795886317, -0.41367241918571263, -0.71001744057354998, -0.19230922867658373], [-0.50023319705649949, 
0.94827281200621139, -0.2159049077858417, -0.39167467063244432]]]])
      arg1=-0.568664125355
      res=outer(arg0,arg1)
      ref=numpy.array([[[[0.49485312457061714, -0.21981021841376658, 0.22667620888885412, 0.37247364570005387], 
[-0.028751380295943364, 0.014636260046930409, 0.069471121892915821, 0.49921476472250564], [0.4589121141297911, 
-0.53851268799316132, -0.16582048743679897, -0.3635471532232159]], [[-0.13525676948514392, -0.53450201613747816, 
-0.12623924612506784, 0.030012958245168815], [0.095891024619134027, -0.14429484604656836, 0.067607237001580431, 
0.37016676421796135], [0.25562740912907533, 0.085679496061730795, -0.53727452595913372, -0.18621809516292637]]], 
[[[-0.4487728130925599, 0.048087091020520438, 0.4804041919520522, -0.42628672645983823], [-0.55543268934228696, 
0.40656549212084198, 0.45500634024081343, -0.072768196511931019], [-0.0108987373007763, 0.1244932882183814, 0.2251688893176586, 
-0.36942238393052951]], [[-0.54476788706308565, -0.30139247135510183, 0.53047661480414698, 0.48924894681818404], 
[0.2384087454092334, -0.036480495812768111, 0.51043414729245151, 0.46233648106615199], [-0.38802163994444683, 
-0.10517315442943466, -0.42039029135128647, 0.20777204388463608]]], [[[0.53571091732737131, -0.11993837003027148, 
0.04329242766674498, -0.20952733589077499], [-0.29066601095821287, 0.30990395817049171, 0.066166619371113802, 
0.093145736222396441], [0.48793726510512936, -0.42303938640998484, 0.34403590650625121, 0.51524709688037884]], 
[[-0.52268560258516061, 0.52560055512280734, -0.058210962210568262, 0.070507442573543411], [-0.047799546305145553, 
0.23524066443971348, 0.40376144683052478, 0.1093593593230564], [0.28446467347764942, -0.53924872923740019, 0.12277737554587888, 
0.22273133399889083]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_array_rank0(self):
      arg0=numpy.array(-0.207683000649)
      arg1=numpy.array(0.928165743638)
      res=outer(arg0,arg1)
      ref=numpy.array(-0.19276424673857764)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_array_rank0(self):
      arg0=numpy.array([0.93891187273728738, -0.83684388514200214])
      arg1=numpy.array(0.325006863482)
      res=outer(arg0,arg1)
      ref=numpy.array([0.30515280284446328, -0.27198000633418834])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_array_rank0(self):
      arg0=numpy.array([[0.074246212290295599, -0.1783493867998498, 0.06354609168547376, 0.77790826354226272, 
-0.50614134071462624], [-0.41467785638114374, 0.87706967672955005, -0.25212998844060541, -0.93711898146715567, 
-0.34260493725944197], [-0.43608695675866183, 0.86498660822160822, 0.45142390637338137, -0.29020859837544655, 
0.83090104827681044], [0.31739964223258488, 0.33895596714672016, -0.83141613296839489, 0.49329189927812389, 
0.018359301817349793]])
      arg1=numpy.array(0.771453015092)
      res=outer(arg0,arg1)
      ref=numpy.array([[0.057277464330503514, -0.13758817218653971, 0.049022824028066493, 0.60011967537460065, 
-0.3904642633569666], [-0.31990448259708865, 0.67661804655870939, -0.19450643977759668, -0.7229432637527089, 
-0.26430361183417556], [-0.33642059763373061, 0.66729652692669539, 0.34825233365631891, -0.22388229822233913, 
0.64100111893618472], [0.24485891098942519, 0.26148860283873598, -0.64139848257453524, 0.38055152301852974, 
0.014163338741977116]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=outer(arg0,arg1)
      ref=-ref
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank3_array_rank0(self):
      arg0=numpy.array([[[0.22890218267521778, 0.14190434160962129], [-0.84482969323319335, 0.91841360418106577]], 
[[-0.45435795577157667, -0.55641507914234856], [-0.98245728870817817, -0.32379807029523966]], [[0.5404127425337717, 
-0.65902737365430708], [0.2880201704204961, 0.69027355927695666]], [[0.31775056241027944, 0.46458796273435676], 
[-0.98236559656202838, 0.53823105267672466]], [[-0.99671747177264791, -0.97760339687001663], [-0.020380770488373612, 
-0.12334078575248042]], [[-0.95617586546311739, 0.58649753976234509], [-0.44759240577104609, -0.45666066208325384]]])
      arg1=numpy.array(0.628984462801)
      res=outer(arg0,arg1)
      ref=numpy.array([[[0.14397591640387697, 0.089255626076413061], [-0.53138475075635072, 0.5776678874546719]], 
[[-0.28578409473020422, -0.34997643964855274], [-0.61795036996273456, -0.20366395530055093]], [[0.33991121855325124, 
-0.41451797858890305], [0.18116021216769854, 0.43417134386733591]], [[0.1998601668022463, 0.29221861016413575], 
[-0.61789269702744543, 0.33853896953051882]], [[-0.62691980354697951, -0.61489734741241586], [-0.012819187977093804, 
-0.077579437867938736]], [[-0.60141976308130241, 0.36889783998134418], [-0.28152866889756928, -0.28723246122264223]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank4_array_rank0(self):
      arg0=numpy.array([[[[0.72014732698955375, -0.13486060745334227, 0.23474332003161913, 0.67651069382593376], 
[-0.78542889973998498, -0.13478196489366945, 0.96798211398773448, -0.68984751443494652], [0.22216169271582276, 
0.24976428201837564, 0.02856353065993078, 0.22585321890236276]], [[0.55690843170751414, -0.37328922333704595, 
0.015239158523985052, 0.84768206207577035], [-0.85714719370844827, -0.045370799462534528, -0.5453445151133498, 
-0.014684318738259261], [-0.34003935275514685, 0.52980156993147798, 0.79481595972951635, 0.51231503305548731]]], 
[[[0.89326872458915108, -0.50630791010133658, -0.43917997165505662, 0.80947422995533214], [0.48814414134967454, 
-0.11445907719210546, -0.12746204293519114, -0.28181681024183125], [-0.47017192506529093, 0.82642588873841349, 
-0.81703786624207453, -0.29160845380722367]], [[0.060809028860670633, -0.96985541156429012, 0.79244483057402282, 
-0.59923632202828148], [-0.39639332317995435, 0.10597687153371615, 0.39091924958652591, -0.90537659080526978], 
[-0.72195636813718789, -0.5289409025268379, -0.92922629277285518, 0.47252160039886726]]], [[[0.30055684213557465, 
-0.19201302940683496, -0.2904517457543192, 0.39597497120161962], [-0.052595824562017501, 0.018124718453008004, 
-0.72467427579590482, 0.43256723364373473], [0.21958817986407686, 0.15716805388254484, -0.17828449289135739, 
0.062420281231962527]], [[-0.23906809121972139, 0.43514000120426499, 0.61688858358287857, 0.72030306154158108], 
[-0.39263270024044195, 0.61790128793808297, -0.23700797179115174, -0.87988848288027777], [0.57534289451001275, 
0.30915542819777286, 0.18564180823835774, -0.69127657529842268]]]])
      arg1=numpy.array(0.114228590767)
      res=outer(arg0,arg1)
      ref=numpy.array([[[[0.08226141430682879, -0.015404937139412466, 0.026814398639240705, 0.077276863194720422], 
[-0.089718436365181162, -0.015395953910646709, 0.11057123276873601, -0.078800309418203748], [0.025377217081398371, 
0.028530221958956544, 0.0032627718546214312, 0.025798894915467288]], [[0.063614865340356416, -0.042640301930397237, 
0.0017407476026737514, 0.096829527369603638], [-0.097910716017431093, -0.0051826424845894574, -0.06229393544405485, 
-0.0016773690358486836], [-0.038842216070632991, 0.060518486719556792, 0.090790706999233098, 0.058521024254812586]]], 
[[[0.10203682758628999, -0.057834839065194266, -0.050166909255364066, 0.092465100550213786], [0.055760017357669445, 
-0.013074499088175694, -0.014559809540803381, -0.032191537088449837], [-0.053707076418539827, 0.094401464644172678, 
-0.093329084064324511, -0.033310022734219957]], [[0.0069461296726802913, -0.11078521691099369, 0.090519856257273906, 
-0.068449920601848985], [-0.045279450696398826, 0.012105588689219772, 0.044654154984065199, -0.1034198920813558], 
[-0.082468058527763025, -0.060420173894805439, -0.10614420992733231, 0.053975476520654854]]], [[[0.034332184522605724, 
-0.021933377758095979, -0.033177893603407532, 0.045231662939468953], [-0.006007946919961509, 0.0020703610469405293, 
-0.082778521289453816, 0.049411545511217646], [0.025083248335021995, 0.017953085308636513, -0.02036518637863605, 
0.0071301807604233769]], [[-0.027308411157448493, 0.049705429124028748, 0.070466313563085761, 0.082279203645240634], 
[-0.044849880037611277, 0.070581993354444636, -0.027073086618310722, -0.10050842143176002], [0.065720608047837387, 
0.035314388891081662, 0.021205602142554264, -0.078963549026759283]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=outer(arg0,arg1)
      ref=-ref
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_constData_rank0(self):
      arg0=numpy.array(0.887539290361)
      arg1=Data(-0.160935502236,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(-0.14283658144885403,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_constData_rank0(self):
      arg0=numpy.array([-0.80443673361283907, 0.88962940157758075])
      arg1=Data(-0.952119275543,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.76591972002781683, -0.84703330133200139]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_constData_rank0(self):
      arg0=numpy.array([[-0.95383898291833091, -0.46383909201486806, -0.71519060195390449, 0.0048290787630054233, 
-0.044981813281804284], [-0.52949066543669643, 0.99055946009096418, 0.68357813283801949, -0.84851215078442399, 
-0.15535224483408228], [0.27022658310544156, -0.8170007898109839, 0.73088394412821422, -0.029016832652413393, 
-0.16030994451808289], [0.64656138098619609, 0.077804961279747342, 0.25404457715323114, 0.57257780429924776, 
-0.24032882851783999]])
      arg1=Data(0.148359356872,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.14151093806492476, -0.068814869383281421, -0.10610521774656882, 0.00071643901956229322, 
-0.0066734728894116005], [-0.078554894593759389, 0.14695876444227876, 0.10141521215940949, -0.1258847169882043, 
-0.023047959132160099], [0.040090642079161652, -0.12120971174003259, 0.10843347189871702, -0.0043049186307659408, 
-0.023783480268841486], [0.095923430661193479, 0.011543094016891225, 0.037689890083197612, 0.084947274804849357, 
-0.035655030436637015]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank3_constData_rank0(self):
      arg0=numpy.array([[[0.81268236173682173, -0.61774115797698559], [-0.72454518228764386, 0.24997841496722639]], 
[[-0.75470763598792434, 0.67603998942603849], [-0.35958699159539176, -0.52667774173871607]], [[-0.084304086883719265, 
0.4423511992191278], [-0.55266737759737361, -0.80092220260002445]], [[0.51243243708803732, -0.60985056807272109], 
[0.59800145126702575, -0.47236649314802048]], [[-0.50307511490707713, -0.26267426698901741], [-0.21398082083038661, 
0.15170633594316407]], [[-0.71895932140375174, -0.32112868090794633], [0.32074928270899106, -0.55325289435955582]]])
      arg1=Data(-0.635300701687,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.51629767466022392, 0.39245139112385014], [0.46030406271142948, -0.15881146243533611]], 
[[0.47946629071182573, -0.42948867965097798], [0.22844586807814618, 0.33459873888964281]], [[0.053558445552326576, 
-0.28102602725609183], [0.3511099727872431, 0.50882643730866328]], [[-0.32554868684931848, 0.38743849382094486], 
[-0.37991074159991289, 0.3000947645504653]], [[0.31960397350184117, 0.16687714613329696], [0.13594216562115033, 
-0.096379141675088126]], [[0.45675536137236566, 0.20401327631270727], [-0.20377224437069213, 
0.35148195199710708]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank4_constData_rank0(self):
      arg0=numpy.array([[[[-0.18026562838092008, 0.27831868086485678, 0.98579853033629061, 0.88127902674675784], 
[-0.5364354806419025, -0.13897951092749539, 0.42877943629615789, -0.79530200142915541], [-0.68639139086178824, 
-0.68881420582826625, 0.78988873526541759, -0.55015997231733227]], [[0.67967080280078607, -0.84205267815405049, 
-0.76960487747871231, -0.66764363542422145], [0.13744122174468965, -0.027930655143734606, -0.72106750744828885, 
0.030892117028429311], [-0.63383485403404816, 0.83772516805124941, -0.88145536554766912, -0.55175259156627643]]], 
[[[0.95237671798280288, -0.31426345768375819, -0.42409201981500755, -0.80473471628180415], [0.072934421982475017, 
-0.84154816359610995, 0.14914477114807712, -0.034838092348106864], [0.99012346249899785, 0.017641123115833768, 
-0.44204693967023934, -0.17622878218081217]], [[-0.91456939505022539, 0.7675074592143587, 0.70302295778984458, 
0.7321972361192377], [0.73700357930770388, -0.54386864178108807, -0.8730357957009629, -0.58210683464430324], 
[-0.55715123434724068, -0.44056110733015719, 0.55416164304167537, 0.86092420650071788]]], [[[0.21182039332567482, 
-0.90059380223329022, 0.56588168450013643, 0.60138762580131289], [-0.98715507058350171, -0.90566299557741159, 
-0.090968976101634125, -0.91128877303181222], [-0.11950637352599403, 0.90900379164478329, 0.39364466601609749, 
-0.74427105029357277]], [[0.88229174877567651, 0.15812352807834085, 0.14579456797030232, 0.14467696011463982], 
[-0.30198142121146998, 0.8616472362664942, 0.31192655241301304, -0.40610229411455534], [-0.85333199406519222, 
0.051210495043784832, -0.31613404527326283, 0.46299292449748264]]]])
      arg1=Data(0.338135219123,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.060954157752957767, 0.094109348140309079, 0.33333320206655742, 0.29799147681766613], 
[-0.18138772879229101, -0.046993867381099209, 0.1449854286475093, -0.26891961652234059], [-0.23209310335330599, 
-0.23291234242289091, 0.26708920058189306, -0.18602846279231652]], [[0.22982063583666248, -0.28472766684086931, 
-0.26023051388452234, -0.22575382696035681], [0.046473717631196391, -0.0094443381972803067, -0.24381831963362285, 
0.010445712760586458], [-0.21432188725670331, 0.28326438326400094, -0.29805110317675221, -0.18656698345103803]]], 
[[[0.3220321102229175, -0.10626354312630162, -0.14340044804853397, -0.27210914962596733], [0.024661696758665703, 
-0.28455707270026925, 0.050431099873229621, -0.011779985989960235], [0.33479561395108726, 0.0059650850303512139, 
-0.14947163880812167, -0.059589157878517755]], [[-0.30924812279865005, 0.25952130290011216, 0.23771682188088594, 
0.24758167287655555], [0.24920686678376869, -0.18390114236286731, -0.29520415008171336, -0.1968308220855445], 
[-0.18839245471074709, -0.1489692265642277, 0.18738156859955105, 0.29110879521355876]]], [[[0.071623935111932435, 
-0.30452248265911952, 0.19134452738624036, 0.20335033662828797], [-0.33379189610029752, -0.30623655546131173, 
-0.030759814667536179, -0.30813882895359362], [-0.040409313798827007, 0.30736619627159828, 0.13310512540001893, 
-0.25166425467804671]], [[0.29833391380282542, 0.053467133815298003, 0.049298278187605613, 0.048920375610437397], 
[-0.10211055403246573, 0.29135327704184211, 0.10547335315050814, -0.13731748820684589], [-0.28854160079804264, 
0.017316071963036032, -0.10689605467076786, 0.15655421397743211]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=outer(arg0,arg1)
      ref=-ref
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_expandedData_rank0(self):
      arg0=numpy.array(-0.219684990046)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.984785150581)+(1.-msk_arg1)*(0.959198032324)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(-0.2163425160025613)+(1.-msk_ref)*numpy.array(-0.21072141018301316)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_expandedData_rank0(self):
      arg0=numpy.array([0.3549348644413739, -0.33299431526460643])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.0403903428868)+(1.-msk_arg1)*(0.867826851273)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.01433594087726137, -0.013449754572887379])+(1.-msk_ref)*numpy.array([0.30802200581530792, 
-0.2889814081080247])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_expandedData_rank0(self):
      arg0=numpy.array([[-0.49543918245473906, -0.18156754427494026, 0.73371800638179052, -0.62302790811506226, 
-0.01418456574302196], [0.3452768699724631, -0.18619769930544483, 0.90490194291210213, -0.40087053280296869, 
-0.84660055652106725], [0.40670965695263761, -0.87788723802393731, -0.97163148178830383, -0.42472364241307847, 
-0.3871763715678489], [0.47028774699673925, -0.39379776090155527, 0.18107041954039649, 0.99960890315982254, 
0.62771463148164619]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.779222341332)+(1.-msk_arg1)*(0.817056551009)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.38605727973990189, 0.14148148695978688, -0.57172946281013015, 0.4854772652764815, 
0.011052930529052624], [-0.26904745102766325, 0.14508940720338592, -0.70511981063167628, 0.31236727514166129, 
0.65969006782516337], [-0.31691725113289343, 0.6840693490383325, 0.7571169581507825, 0.33095415106009546, 0.30169647876145583], 
[-0.36645871931446328, 0.30685601326093603, -0.14109411626020171, -0.7789175899363241, 
-0.48912926483136554]])+(1.-msk_ref)*numpy.array([[-0.40480182965139777, -0.14835095150053357, 0.59948910370781616, 
-0.50904903378710897, -0.011589592363559949], [0.28211072852302999, -0.15213405000039656, 0.7393560604774887, 
-0.3275338949333037, -0.69172053079376206], [0.33230478957194803, -0.71728351887502739, -0.7938778673621274, 
-0.34702323440218974, -0.31634499078556977], [0.38425168454314829, -0.3217550403174585, 0.14794477247950574, 
0.81673700277407779, 0.5128783518165454]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        res=outer(arg0,arg1)
        ref=-ref      
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank3_expandedData_rank0(self):
      arg0=numpy.array([[[0.54236255244194465, 0.75659586809114421], [0.3661458269609339, -0.51975633455419379]], 
[[-0.80861697268642407, -0.75737083498588476], [-0.27761315519041752, -0.65538128106110061]], [[0.45251432733535735, 
0.95093655581500647], [-0.36525727052008605, -0.4908409940140579]], [[-0.27352299292749049, -0.23117084777931862], 
[-0.14429773179196048, -0.48471103153587536]], [[0.68495464141255469, -0.1919787494952232], [-0.77482644344771479, 
-0.9802309778068532]], [[-0.072370570945621493, -0.22381199096031423], [-0.58589816807084705, -0.20352533557704966]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.325076605672)+(1.-msk_arg1)*(0.972088798165)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.17630937759136275, -0.24595161666443635], [-0.11902544260938283, 0.16896062501333384]], 
[[0.26286246076957165, 0.24620354027208669], [0.090245542179160948, 0.213049122268229]], [[-0.14710182154807064, 
-0.30912722777364771], [0.11873659369764412, 0.15956092425870005]], [[0.088915926114081431, 0.075148234526391064], 
[0.046907816857081411, 0.1575682168633965]], [[-0.22266272986959096, 0.062407800247038729], [0.25187795022079568, 
0.31865015903987687]], [[0.023525979553538398, 0.072756042330043777], [0.19046178774584188, 
0.066161325257617018]]])+(1.-msk_ref)*numpy.array([[[0.52722456177304045, 0.73547836810939327], [0.35592625688361718, 
-0.50524931059547862]], [[-0.78604750115463928, -0.73623170474671928], [-0.26986463838387154, -0.63708880184658223]], 
[[0.43988410861191157, 0.92439477367345957], [-0.35506250112093157, -0.4771410319612836]], [[-0.26588863746540259, 
-0.22471859158860277], [-0.14027020867559534, -0.47118216410307007]], [[0.66583673416833034, -0.18662039187004836], 
[-0.75320010619762001, -0.9528715531404548]], [[-0.070350621333050561, -0.21756512930754782], [-0.56954504604711753, 
-0.19784469885724082]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        res=outer(arg0,arg1)
        ref=-ref    
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank4_expandedData_rank0(self):
      arg0=numpy.array([[[[-0.822250636341469, 0.63638809184653833, 0.52959023053761745, -0.0036888432821744477], 
[-0.0095192693710410836, -0.61885383255801396, 0.73352016357184868, 0.6177409427823235], [-0.68277734563867454, 
0.41285719119533026, -0.26534099849346537, 0.55786359386333229]], [[0.062308714666052589, 0.1809406597141674, 
-0.3972832420118888, 0.53326051657459539], [-0.8537018610811522, -0.36944105535194449, -0.74127572741761094, 
-0.071601359497779793], [0.62774540796583267, -0.31605339710016711, 0.16176277052083066, -0.62123826544507965]]], 
[[[0.92456868073754506, 0.61315530468207013, 0.33222920883688434, -0.54113394998908704], [-0.16602855481677925, 
-0.21715771822207275, 0.63670831412113249, -0.94084038285711369], [0.55803542236941817, 0.037332690808041802, 
-0.28711960314612517, 0.046507744312877186]], [[-0.049053830386043717, -0.079295986576197075, -0.61783181583030999, 
-0.88326371464697995], [0.45009667779065543, 0.95843215991445696, 0.72270648953330463, -0.50524740319664985], 
[0.031675253637922784, 0.056197570308277767, -0.32562149073832658, -0.02611660637834845]]], [[[-0.49213582326329286, 
-0.6446132567206968, -0.17581570927366008, 0.54391357448983135], [-0.96359375506215361, 0.52153541000680015, 
0.67931642406343462, 0.74271670340750484], [-0.94227463976154291, 0.03116700024629071, 0.28594734622339146, 
-0.089203926776785236]], [[-0.83316454913290783, -0.6887417787132184, -0.79411226326946527, 0.39282591376873621], 
[-0.44894050979038158, 0.18717976723150787, -0.54009302765234679, 0.5825689649809529], [-0.25538648970381628, 
-0.4225818231655476, -0.49715341260075241, -0.71712430014819817]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.0433066760346)+(1.-msk_arg1)*(-0.474220233298)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.03560894192731321, -0.027559852925898149, -0.022934792545000734, 0.00015975154096367178], 
[0.00041224791473810842, 0.026800502439382718, -0.031766320088679183, -0.026752306882404632], [0.029568817311362693, 
-0.017879472627665899, 0.011491036660463316, -0.024159217930957023]], [[-0.0026983833201773056, -0.0078359385317347419, 
0.017205016655798717, -0.023093740433358587], [0.036970989928007195, 0.015999264098020646, 0.032102187779613538, 
0.0031008168794098494], [-0.027185567015006656, 0.013687222077863069, -0.0070054078974107628, 0.026903764301949202]]], 
[[[-0.040039996328471571, -0.026553718138784876, -0.014387742716342342, 0.023434712663520253], [0.007190144835949046, 
0.0094043789514640531, -0.02757372068820307, 0.040744669660695822], [-0.024166659252403602, -0.0016167547463250957, 
0.012434195636642468, -0.0020140958160594523]], [[0.0021243583407863733, 0.0034340456015022032, 0.026756242292054064, 
0.038251215543365844], [-0.019492191009345811, -0.041506511050591728, -0.031298015810347801, 0.021880585607578374], 
[-0.001371749947612444, -0.0024337299712742584, 0.014101584409319901, 0.0011310234115512418]]], [[[0.021312766663102251, 
0.027916057476434795, 0.0076139939633141267, -0.023555088961271925], [0.041730042579474977, -0.022585965041755498, 
-0.029418936301922497, -0.032164591659981562], [0.040806782559806382, -0.001349739182637529, -0.01238342908586028, 
0.0038631255579396201]], [[0.036081587212842318, 0.029827117082252233, 0.034390362520542221, -0.017011984585592475], 
[0.01944212121631634, -0.008106133539733466, 0.023389633777105828, -0.025229125434263269], [0.011059939973226046, 
0.018300614113956164, 0.021530061779014453, 0.031056269743083013]]]])+(1.-msk_ref)*numpy.array([[[[0.38992788859509775, 
-0.30178810938339307, -0.25114240267777288, 0.0017493241218717072], [0.0045142301419594938, 0.29347300885288535, 
-0.34785010309766623, -0.29294525400382271], [0.32378683213920972, -0.19578523352731481, 0.12583007020903647, 
-0.2645502036302062]], [[-0.02954805320542013, -0.085805721862706297, 0.18839975171217543, -0.25288292657849815], 
[0.40484269572865111, 0.17519642345877637, 0.35152794839395946, 0.033954813405475195], [-0.29768957381716588, 
0.14987891570739648, -0.076711178775283217, 0.29460375517287252]]], [[[-0.43844917547917728, -0.29077065163410137, 
-0.15754981292296344, 0.25661666800917293], [0.078734099999305959, 0.10298058379768445, -0.30193996526515826, 
0.4461655458544711], [-0.26463168818444949, -0.017703917344623388, 0.13615792518832087, -0.022054913358206019]], 
[[0.023262318889819274, 0.037603761253741638, 0.2929883478418393, 0.4188615248233527], [-0.21344495154843937, 
-0.45450792247472693, -0.34272204007230178, 0.23959854141701173], [-0.015021046169942029, -0.026650024902359754, 
0.15441629930469944, 0.012385023169686635]]], [[[0.23338076492211271, 0.30568864898892922, 0.083375366669169365, 
-0.25793482218839608], [0.45695565532985627, -0.24732264380647698, -0.32214559310237417, -0.3522112883640634], 
[0.44684569949829844, -0.014780022127987881, -0.13560201723693716, 0.042302306967164985]], [[0.39510348686524543, 
0.32661528698330894, 0.37658410275227222, -0.18628599647282285], [0.21289667328961806, -0.088764432885149416, 
0.25612304157579907, -0.27626599048531247], [0.12110944072844429, 0.20039685076896629, 0.23576020730831521, 
0.34007485291978418]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_array_rank1(self):
      arg0=numpy.array(-0.0705991943231)
      arg1=numpy.array([0.31103878225265236, 0.89166364596776915])
      res=outer(arg0,arg1)
      ref=numpy.array([-0.021959087430276469, -0.062950735012525461])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_array_rank1(self):
      arg0=numpy.array([-0.55998676190775054, 0.049723485599342698])
      arg1=numpy.array([0.79135932201024461, -0.21409044108347763])
      res=outer(arg0,arg1)
      ref=numpy.array([[-0.44315074423802975, 0.11988781285773868], [0.039349143851881999, -0.010645322964171226]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_array_rank1(self):
      arg0=numpy.array([[0.39370173512983175, -0.95148127262943816, 0.19145825873523825, 0.22194728033627098, 
-0.55485368848311123], [0.21660110788510956, 0.89997912244714051, 0.90465802099620984, 0.82682412007740247, 
-0.37263197268419423], [-0.41711154749590573, -0.12043205783375033, -0.21821535243959223, 0.082355203560406265, 
-0.75205178168271147], [-0.45919525734405653, -0.76847783785798152, -0.84715014107714515, -0.47796513160077603, 
0.35432747942008547]])
      arg1=numpy.array([-0.75239679614805688, 0.18991554154474533])
      res=outer(arg0,arg1)
      ref=numpy.array([[[-0.29621992414961629, 0.074770078234287884], [0.71589146112126512, -0.18070108116110323], 
[-0.14405258046847899, 0.036360898890916739], [-0.16699242263878492, 0.042151237939446307], [0.4174701375456249, 
-0.10537533872636949]], [[-0.16296997961487605, 0.041135916703192388], [-0.67714140832936831, 0.17092002241851337], 
[-0.68066179660718984, 0.17180861797029279], [-0.62209981892417388, 0.15702675052675744], [0.28036710238991797, 
-0.070768602889205501]], [[0.31383339197227755, -0.079215965427251692], [0.090612694467631247, -0.022871919482844782], 
[0.16418453204586825, -0.04144248683194262], [-0.06196379130497072, 0.015640533083202294], [0.56584135107551004, 
-0.14282632138796272]], [[0.3454970404320506, -0.087208315973275191], [0.57820026311513117, -0.14594588474193354], 
[0.63739305200281837, -0.16088697781237343], [0.35961943368690824, -0.090773006807466847], [-0.26659486030288887, 
0.067292295138250136]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=outer(arg0,arg1)
      ref=-ref    
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank3_array_rank1(self):
      arg0=numpy.array([[[-0.11650349333382004, -0.93704280994921563], [0.16842803438649012, -0.29159530904824105]], 
[[0.62977412573709302, -0.38013561791327355], [0.37907571072272361, -0.8115279466802332]], [[-0.88418803236921328, 
-0.51276945192377066], [-0.16107750386484887, 0.67441583586588005]], [[0.53090414717970158, -0.48588821848507924], 
[-0.017722197481937263, -0.62956203569481728]], [[-0.26077893723743117, -0.052624881493034126], [0.40460561788751304, 
-0.045268579762606898]], [[-0.42518607569172384, 0.93698214557284532], [0.36176722067350453, -0.50067180425631896]]])
      arg1=numpy.array([0.81982036515978107, 0.2126887982468002])
      res=outer(arg0,arg1)
      ref=numpy.array([[[[-0.095511936447322462, -0.024778987988724282], [-0.76820677862291331, -0.19929850915390346]], 
[[0.13808073265387649, 0.035822756224733326], [-0.23905577274280818, -0.062019055855874694]]], [[[0.5163016537299655, 
0.13394590196995157], [-0.31164292108789893, -0.080850587744778962]], [[0.31077398758790681, 0.080625157358167762], 
[-0.66530713758475613, -0.17260290372311216]]], [[[-0.7248753555668368, -0.18805689002881085], [-0.42037883931892644, 
-0.10906031850733718]], [[-0.13205461803750645, -0.034259380721609022], [0.55289983682910471, 0.14344069364892528]]], 
[[[0.43524603180570509, 0.11291736504789307], [-0.39834105670527314, -0.10334298127187019]], [[-0.01452901841107556, 
-0.0037693128847257049], [-0.51612777799406018, -0.13390079279373981]]], [[[-0.21379188355197046, -0.055464758769106969], 
[-0.043142949562109442, -0.011192722802633705]], [[0.33170392540223981, 0.086055082632399194], [-0.03711210359124506, 
-0.0096281198280482819]]], [[[-0.34857620383444338, -0.090432315470145769], [0.76815704473172519, 0.19928560652059688]], 
[[0.2965841349553916, 0.076943835410132652], [-0.41046094139062184, -0.10648728436333367]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_constData_rank1(self):
      arg0=numpy.array(0.45620442526)
      arg1=Data(numpy.array([0.31772359770243996, -0.5126401738892179]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.14494691128130036, -0.23386871589418687]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_constData_rank1(self):
      arg0=numpy.array([0.8539936228705296, 0.89072822837147902])
      arg1=Data(numpy.array([0.90381714995212459, 0.33301162500766246]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.77185408230013164, 0.2843898040982959], [0.80505544874861534, 
0.29662285477018252]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_constData_rank1(self):
      arg0=numpy.array([[0.22004350247352922, -0.35970181275916224, 0.15646348648285158, 0.4212211618405588, 
0.37626278163592164], [0.33038291515486939, -0.90764100376178236, -0.65433267556937502, 0.43104735731894261, 
-0.033957692220172087], [-0.21077756673600101, 0.57526900030575967, -0.70337369772725622, 0.37091995859980731, 
0.46236022366388907], [0.44196297866078726, -0.73366403186605034, 0.48888423737293074, 0.51899052032344106, 
0.62487345661284288]])
      arg1=Data(numpy.array([-0.28684372213436249, -0.13123413337250533]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.063118097280988925, -0.02887721835136434], [0.10317820683031562, 0.047205155669967837], 
[-0.044880568840860662, -0.020533350053017728], [-0.12082464590410658, -0.055278594132305549], [-0.10792861678507662, 
-0.049378520068318395]], [[-0.094768265112624012, -0.043357515551431247], [0.26035112388079856, 0.11911348054202836], 
[0.18769122017445578, 0.085870781615659628], [-0.12364322838954604, -0.056568126380260077], [0.0097405508315272452, 
0.0044564083098445504]], [[0.060460221784978521, 0.0276612113049645], [-0.16501230127621783, -0.07549492871119387], 
[0.20175832950749614, 0.092306637658250992], [-0.10639606153869237, -0.048677359317411267], [-0.13262512752262628, 
-0.060677443258448219]], [[-0.12677430584465005, -0.058000628487279486], [0.2104469216965614, 0.096281763408519253], 
[-0.14023337434087066, -0.064158299211114753], [-0.14886917260202534, -0.068109271163192403], [-0.17924102815779291, 
-0.082004726546068246]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=arg0*1j
      arg1=arg1*1j
      res=outer(arg0,arg1)
      ref=-ref    
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank3_constData_rank1(self):
      arg0=numpy.array([[[-0.66447070728135871, 0.10916086666511138], [0.26746102869120603, -0.70631701651855239]], 
[[-0.87747366027630269, 0.72562057325760065], [0.3023068504577977, -0.0090824437971253058]], [[-0.37222489903654399, 
0.080330054398139561], [0.27150597071903082, -0.27524188286388296]], [[-0.54612661303996024, 0.17675616726836862], 
[-0.94091695872832548, -0.31531671754763124]], [[-0.43884398137503489, 0.69552266955434705], [-0.44929811099008043, 
0.35789835388832625]], [[-0.24615638375578608, 0.35713986793045427], [-0.45039744128586578, -0.65163958989322479]]])
      arg1=Data(numpy.array([-0.95289915314530238, 0.048853322366110286]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.63317357425826681, -0.032461601665653525], [-0.1040192974017919, 0.005332871008954668]], 
[[-0.25486338773922163, 0.01306635985502296], [0.67304888689264508, -0.034505932900650085]]], [[[0.83614390778459757, 
-0.042867503593248958], [-0.69144322976197647, 0.035448975780835307]], [[-0.288067941791259, 0.014768694018898286], 
[0.0086546530027705078, -0.00044370755469304134]]], [[[0.35469279107151847, -0.018184422985325136], [-0.076546440808103258, 
0.0039243900431994872]], [[-0.25871780957205776, 0.013263968711860512], [0.26227775709111256, -0.013446480432204441]]], 
[[[0.52040358707589041, -0.026680099479553146], [-0.16843080210323788, 0.0086351260197597233]], [[0.89659897315227477, 
-0.045966919504494969], [0.30046503312369433, -0.015404269249778172]]], [[[0.41817405821518361, -0.021438986490541878], 
[-0.66276296281169733, 0.033978593188676118]], [[0.42813578947223174, -0.021949705454682798], [-0.34104103833228383, 
0.017484523656806621]]], [[[0.23456220962219862, -0.01202555716809737], [-0.340318277705355, 0.017447469097796535]], 
[[0.42918334038011258, -0.02200341139200963], [0.62094681336520607, -0.031834758951573609]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_expandedData_rank1(self):
      arg0=numpy.array(0.0952653164761)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.03600206368683323, -0.21567891629700786])+(1.-msk_arg1)*numpy.array([0.98400030929841975, 
-0.057944651857548823])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.0034297479909205435, -0.020546720218266729])+(1.-msk_ref)*numpy.array([0.09374110087793984, 
-0.0055201155973095091])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
          arg0=numpy.array(arg0*1j)
          arg1=arg1*1j
          ref=-ref
          res=outer(arg0,arg1)
          self.assertTrue(isinstance(res,Data),"wrong type of result.")
          self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
          self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")          
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_expandedData_rank1(self):
      arg0=numpy.array([-0.6816375542209947, -0.73859164214492656])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.65424365167993637, -0.84029348680087512])+(1.-msk_arg1)*numpy.array([-0.4070349497467638, 
0.0094425568140983351])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.44595704259572422, 0.57277559717078019], [-0.48321889305717752, 
0.6206337462999445]])+(1.-msk_ref)*numpy.array([[0.27745030762784956, -0.006436401332354777], [0.30063261194383994, 
-0.006974193543371655]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
          arg0=numpy.array(arg0*1j)
          arg1=arg1*1j
          ref=-ref
          res=outer(arg0,arg1)
          self.assertTrue(isinstance(res,Data),"wrong type of result.")
          self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
          self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")          
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_expandedData_rank1(self):
      arg0=numpy.array([[0.11413458360117934, -0.39371474720074584, -0.52902000724565745, 0.4389579328234896, 
0.59175704124133555], [-0.35473395573288258, -0.37884322851013308, 0.91011404456304401, 0.40604179011912978, 
-0.77022685281477199], [-0.55045292580057259, -0.69989776505223489, 0.78949225848162996, 0.0304730903118684, 
-0.47456543934448558], [-0.16824189615650686, 0.97460850007093991, -0.71636079682124976, -0.032193665122135284, 
0.7901741074913109]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-0.57288820519795935, 0.13608562597774565])+(1.-msk_arg1)*numpy.array([-0.44236731972391885, 
0.68478136995014638])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.065386356750296082, 0.015532076255075834], [0.22555453488380359, -0.053578917829483381], 
[0.30306932246477614, -0.071992018840776828], [-0.25147382229265536, 0.059735865066181809], [-0.33901062927000353, 
0.080529627384065799]], [[0.20322289922258346, -0.04827419242147124], [0.21703481723257054, -0.051555117899231596], 
[-0.52139360151517788, 0.12385343946549975], [-0.23261655237671477, 0.055256451181486192], [0.44125387930432752, 
-0.10481680341016721]], [[0.31534798870785552, -0.074908730978852506], [0.40096317444283791, -0.095246025477558532], 
[-0.45229080297922442, 0.10743854820005679], [-0.01745767401560161, 0.0041469495705669874], [0.27187294279504337, 
-0.064581534880598201]], [[0.096383797928202675, -0.022895303754141116], [-0.55834171437631597, 0.13263020781538562], 
[0.41039465116510582, -0.097486407461336438], [0.018443371030564225, -0.0043810950706636978], [-0.45268142623459651, 
0.10753133804936152]]])+(1.-msk_ref)*numpy.array([[[-0.050489409835459249, 0.0781572365171051], [0.17416653745497421, 
-0.2696085239577023], [0.23402116268558962, -0.36226304529271769], [-0.19418064421467912, 0.30059021458935353], 
[-0.26177397626168614, 0.40522419737888704]], [[0.15692270921261856, -0.24291520417459792], [0.1675878635915837, 
-0.25942478501550525], [-0.40260471053644903, 0.62322914224674986], [-0.17961961839090143, 0.27804985329478749], 
[0.34072318845906002, -0.52743699944288935]], [[0.24350238542058847, -0.37693990862278237], [0.30961189840691822, 
-0.47927695037751511], [-0.34924557432730197, 0.54062959032808566], [-0.013480299284966142, 0.020867404530375777], 
[0.20993224143642408, -0.32497357168530994]], [[0.074424716668023833, -0.11520891613306304], [-0.43113494995653046, 
0.66739374384363559], [0.31689460564510707, -0.49055052782583386], [0.014241425352168394, -0.022045622106052046], 
[-0.34954720204617096, 0.54109650782703411]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank3_expandedData_rank1(self):
      arg0=numpy.array([[[0.87420183956214492, -0.94796363776977888], [-0.89619505429056767, 0.98269351205225663]], 
[[-0.78543686020072778, 0.49801945489413346], [-0.32439744666765336, -0.90769684324544953]], [[0.047647877322015475, 
-0.020522477822865426], [-0.13937854856216125, -0.72983275447960549]], [[-0.29260969381059909, 0.4335589962680908], 
[0.1678945182964815, -0.50664460296582137]], [[0.14014475688327788, 0.85900221376298114], [0.33192132101855942, 
-0.56956032203941187]], [[0.60043591489976333, -0.696389117775569], [-0.20269845166148004, -0.77081614130858345]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.72311213346341008, -0.73911599083212276])+(1.-msk_arg1)*numpy.array([-0.85972430731293925, 
-0.31600218264079616])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.63214595728342038, -0.64613655883523913], [-0.68548400855344005, 0.70065508340303362]], 
[[-0.64804951770740904, 0.66239209553082101], [0.71059760204075861, -0.72632448884480227]]], [[[-0.56795892368055045, 
0.58052894316333237], [0.36012391053478138, -0.36809414285775111]], [[-0.2345757297499296, 0.2397673402171733], 
[-0.65636660085721954, 0.67089325167055047]]], [[[0.034454758225325448, -0.035217308057908903], [-0.014840052722447738, 
0.015168491530377444]], [[-0.10078631960981793, 0.10301691402126495], [-0.52775092016322467, 0.539431059468931]]], 
[[[-0.21158961996345749, 0.21627250376790499], [0.31351177077367381, -0.32045038711087054]], [[0.12140656332218028, 
-0.12409352324598588], [-0.36636085975833743, 0.37446912772083052]]], [[[0.10134037414357799, -0.10358323084371088], 
[0.62115492344394152, -0.6349022723524127]], [[0.24001633458372393, -0.24532835606293965], [-0.41185597960602605, 
0.42097114176282285]]], [[[0.43418249543122239, -0.44379178617233073], [-0.50356742067539362, 0.51471233278939754]], 
[[-0.14657370983066273, 0.14981766693991197], [-0.55738650444968318, 
0.56972253603268719]]]])+(1.-msk_ref)*numpy.array([[[[-0.75157257096926233, -0.27624968937023692], [0.81498738183947717, 
0.29955857859935919]], [[0.77048067226724026, 0.28319959322770621], [-0.8448454989500459, -0.31053329467546265]]], 
[[[0.67525916057412061, 0.24819976214996387], [-0.42815943088722647, -0.15737523474412571]], [[0.27889237013043444, 
0.10251030119007973], [0.7803690398093357, 0.28683418364172264]]], [[[-0.040964038325601658, -0.015056833231957783], 
[0.017643673030608136, 0.0064851477852228088]], [[0.11982712611688695, 0.044043925558949155], [0.62745495929927331, 
0.23062874337829964]]], [[[0.25156366632436855, 0.092465301906004382], [-0.3727412077458776, -0.13700558912426949]], 
[[-0.14434299844408216, -0.053055034235113242], [0.43557468033862989, 0.16010080036037913]]], [[[-0.12048585403501635, 
-0.044286049060779549], [-0.73850508320766028, -0.2714465744423778]], [[-0.28536082779507677, -0.10488786190688115], 
[0.48966485333826798, 0.17998230491004891]]], [[[-0.51620935102301002, -0.18973905964424856], [0.59870265189986993, 
0.22006048118437826]], [[0.17426478594807124, 0.064053153142937608], [0.66268937315215459, 0.24357958306826874]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_array_rank2(self):
      arg0=numpy.array(0.218264810985)
      arg1=numpy.array([[0.62659455796785046, -0.37652765022503543, 0.94590766670555371, 0.094449960134427524, 
-0.22520111876829429], [0.12465652166384422, 0.040617779034448631, 0.23922827763552656, -0.30156812276149814, 
-0.20731745558322934], [0.12441485550270071, 0.0098506262715623016, 0.7491573134680285, 0.4197162216745971, 
0.15950621707742796], [-0.13087565182032113, -0.03397043006609568, 0.67705732058749746, -0.2032399434958112, 
-0.89027035091146112]])
      res=outer(arg0,arg1)
      ref=numpy.array([[0.1367635427591869, -0.08218273640705627, 0.20645835808290763, 0.020615102696297342, 
-0.049153479621609805], [0.027208132139027282, 0.0088654318635911934, 0.052215114800425164, -0.0658217093136899, 
-0.04525010525679922], [0.027155384920043782, 0.0021500450812480623, 0.16351467942225445, 0.09160928179121422, 
0.034814594321363766], [-0.028565549407122866, -0.0074145494974611941, 0.14777778810415348, -0.04436012785174917, 
-0.19431468986738801]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_array_rank2(self):
      arg0=numpy.array([0.16636827760669792, 0.40745449772685505])
      arg1=numpy.array([[0.7102545042272097, -0.16509784417725548, 0.20930882056968447, -0.72285238509450433, 
-0.53798767304345074], [-0.81804276796566344, 0.55893736573743813, -0.90691450200816082, 0.37895267866036297, 
-0.17452628062456754], [-0.82868114455863062, 0.10780418724989116, 0.61897790911348216, -0.076685948682186478, 
-0.70998524155215792], [0.73039426394358054, -0.6291725941235613, 0.85291511223351368, -0.003280370193414317, 
0.98782742462344664]])
      res=outer(arg0,arg1)
      ref=numpy.array([[[0.11816381853068003, -0.027467043972348996, 0.034822347966067792, -0.12025970627206621, 
-0.089504082537874244], [-0.13609636631506308, 0.092989446827762556, -0.15088180363563389, 0.063045704443169065, 
-0.029035636704612518], [-0.13786625470536643, 0.017935196951554336, 0.10297828861580524, -0.012758109198890991, 
-0.11811902176320789], [0.1215144356661054, -0.10467436080167493, 0.14189801816701311, -0.0005457495389906904, 
0.16434314720726304]], [[0.28939639227813418, -0.067269859175030214, 0.085283820355021217, -0.29452945549934045, 
-0.21920549710315873], [-0.33331520514053564, 0.22774154361731933, -0.36952639289693606, 0.1544059733458045, 
-0.071111518012019329], [-0.33764985953185223, 0.043925300968756235, 0.25220533306185283, -0.031246034703007672, 
-0.28928667999011437], [0.29760242795770758, -0.25635920332211809, 0.3475240986587505, -0.0013366015895157769, 
0.40249472714075923]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_array_rank2(self):
      arg0=numpy.array([[-0.81956487123327637, 0.26071542752343602, -0.47037712698936929, 0.094964802681870131, 
-0.46512807889738683], [0.076723197191971071, 0.081140132594234249, 0.093588154099443432, -0.74062097527274462, 
0.030840140350320233], [-0.8940842367217956, -0.96021909382187043, 0.28300310355728842, -0.76699825705987146, 
-0.71884907337947257], [0.607311761899342, -0.50060925046611637, -0.49716368786464571, 0.7905725723857413, 
0.65601088235599914]])
      arg1=numpy.array([[0.90239786983064629, -0.98569087469079975, -0.39644746029235201, 0.35389309790413259, 
-0.32994298913565023], [0.49986542917608778, 0.17646704298247662, -0.044357612957905967, 0.21383297018993774, 
-0.48466538221114286], [-0.90245331337920187, -0.60180396016839999, -0.82878400962931353, 0.56657680356113072, 
0.41317536821994039], [-0.8755948255170094, -0.31647123327854643, 0.24493355541838535, 0.22973481239643379, 
0.69502961990367562]])
      res=outer(arg0,arg1)
      ref=numpy.array([[[[-0.73957359398893652, 0.80783761479178084, 0.32491441174526092, -0.29003835121414567, 
0.27040968340528149], [-0.4096721460966668, -0.14462618935885049, 0.036353941352061714, -0.17524999067914535, 
0.39721472156310195], [0.7396190335736692, 0.49321738512309049, 0.6792422601320468, -0.46434644505433942, 
-0.33862401745193699], [0.71760676042737093, 0.25936870555096808, -0.20073893780717755, -0.18828258193948416, 
-0.56962186093966893]], [[0.2352690464291349, -0.25698481780096144, -0.10335996910070097, 0.09226539031766913, 
-0.086021227470861453], [0.13032262907182957, 0.046007680554972952, -0.011564714026239559, 0.055749554241675769, 
-0.12635974232898764], [-0.23528350141759999, -0.15689957676060126, -0.21607677739509398, 0.14771531356530201, 
0.10772119276761485], [-0.22828107927197547, -0.082508932883085279, 0.06385795661573955, 0.059895409830952602, 
0.18120494449463803]], [[-0.42446731741226623, 0.46364644173669678, 0.18647981737454861, -0.16646321865351346, 
0.15519763529991185], [-0.23512526445715623, -0.083006060686406896, 0.020864806543246229, -0.10058213817354636, 
0.22797551003568195], [0.42449339678934589, 0.28307481779483684, 0.38984104134416625, -0.26650476907790493, 
-0.19434824264607031], [0.41185977843344895, 0.14886082948434515, -0.11521114210099156, -0.10806200102447626, 
-0.32692603578280427]], [[0.085696035649007254, -0.093605939420331769, -0.03764855484039175, 0.033607388212941701, 
-0.031332970859533446], [0.047469621849195502, 0.016758157916683988, -0.0042124119619863058, 0.020306605820965655, 
-0.04602615238841435], [-0.085701300834655811, -0.057150194330560136, -0.078705309940336921, 0.053804854354307471, 
0.039237117316015674], [-0.083150689834489297, -0.030053628222785254, 0.023260066760475869, 0.021816721128383787, 
0.066003350712207753]], [[-0.41973058759542264, 0.45847250293161657, 0.18439884558952974, -0.16460561676319402, 
0.15346574868232638], [-0.23250144677989149, -0.082079776691141942, 0.020631971299584634, -0.099459718629367935, 
0.22543147813593661], [0.41975637594664955, 0.27991591986596737, 0.38549071421975595, -0.26353078018821086, 
-0.19217946526786128], [0.40726373908521918, 0.14719965676113705, -0.11392547408926021, -0.10685611194580481, 
-0.3232777918825776]]], [[[0.069234849712631316, -0.075625355349228679, -0.030416716672266243, 0.027151809935376289, 
-0.025314281017562862], [0.038351273892126232, 0.013539115736628589, -0.0034032578859345508, 0.016405949138027464, 
-0.037185077691507543], [-0.069239103518940165, -0.046172323906909257, -0.063586959000342266, 0.043469583824017291, 
0.031700135250803743], [-0.067178434458411013, -0.024280684836416194, 0.018792085471295353, 0.017625989313352068, 
0.053324894582130404]], [[0.073220682810813176, -0.079979088269338222, -0.032167799494768862, 0.028714932888125641, 
-0.026771617887004651], [0.040559147202621569, 0.014318559266110586, -0.0035991825969682134, 0.017350435554230489, 
-0.039325813376447354], [-0.07322518150769447, -0.048830453123799245, -0.067247644433303611, 0.045972116965767557, 
0.033525104162037522], [-0.071045880241275547, -0.025678517830482094, 0.019873941163425009, 0.018640713139358167, 
0.056394795515904469]], [[0.08445375090072002, -0.092248989474977755, -0.037102786006173623, 0.033120201781381385, 
-0.030878755311258225], [0.046781482814716134, 0.016515224812117129, -0.0041513471169879728, 0.020012232965677588, 
-0.045358938477042088], [-0.08445893976008606, -0.056321721761895538, -0.077564365608342803, 0.053024877200849189, 
0.038668320031062062], [-0.081945303459161165, -0.029617958548113513, 0.022922879328620416, 0.021500457024564171, 
0.065046539171222786]], [[-0.66833479043802047, 0.73002333693094479, 0.29361730468612446, -0.26210065131205157, 
0.24436269839804986], [-0.37021082166152319, -0.13069519347717917, 0.032852178569655249, -0.15836918292753943, 
0.35895334805415413], [0.66837585309302427, 0.44570863590292037, 0.61381482150211797, -0.41961866482035887, 
-0.30600634416972761], [0.64848389361817615, 0.23438523343652531, -0.1814029286909854, -0.17014642081114781, 
-0.51475351493650523]], [[0.027830076957407138, -0.030398844917494179, -0.012226495316944143, 0.010914112808373068, 
-0.010175488092547638], [0.015415919992063605, 0.0054422683727855725, -0.0013679950092270034, 0.0065946388121835226, 
-0.014947148410333245], [-0.027831786844226113, -0.018559718594971984, -0.025559815177069185, 0.017473308141061086, 
0.012742386345198203], [-0.027003467308958726, -0.0097600172511493073, 0.0075537852256059431, 0.0070850538576605069, 
0.021434811025459081]]], [[[-0.80681971066690761, 0.88129067334156275, 0.35445742493578192, -0.31641024032072806, 
0.29499682560305551], [-0.44692180070851523, -0.1577764014315399, 0.039659442524270186, -0.19118468793822496, 
0.43333167831972702], [0.80686928186969908, 0.53806343438331783, 0.74100271865665412, -0.50656738895622833, 
-0.36941358372717226], [0.78285553124992913, 0.28295194105025451, -0.21899123094380268, -0.20540227438989039, 
-0.62141502721061759]], [[-0.86649966483556939, 0.94647919848408657, 0.38067642106990418, -0.33981490977932066, 
0.31681755804071332], [-0.47998032943634339, -0.16944702410205875, 0.042593026918541728, -0.20532650086502105, 
0.46538495411361408], [0.86655290278952168, 0.57786365329131406, 0.79581423070031576, -0.54403786489596084, 
-0.39673887766166882], [0.84076286991306148, 0.30388172083941561, -0.23518987663041085, -0.22059575337864104, 
-0.66738071180326641]], [[0.25538139780555885, -0.27895357668559462, -0.11219586166014048, 0.10015284503437284, 
-0.093374889922357707], [0.14146346781782879, 0.049940720839618299, -0.012553342133480382, 0.060515394206625515, 
-0.13716180735253283], [-0.25539708850187232, -0.17031238846072397, -0.23454844690374935, 0.16034299381136813, 
0.11692991151966856], [-0.24779605308001609, -0.089562341204431253, 0.069316956348724146, 0.06501566490334218, 
0.19669553949698271]], [[-0.69213759333464642, 0.75602318288766357, 0.30407451106004657, -0.27143538927798816, 
0.25306569759616782], [-0.38339591294254394, -0.135349914396069, 0.034022211826050246, -0.16400951543761769, 
0.371737503413203], [0.6921801184397538, 0.46158258854089107, 0.63567589086477516, -0.43456342082194044, -0.3169047872847649], 
[0.67157970506218845, 0.2427328843342331, -0.187863610101379, -0.17620620069404128, -0.53308650707110417]], 
[[-0.64868787254737004, 0.70856297191008322, 0.28498588947480247, -0.2543957255037767, 0.23717921200821557], 
[-0.35932780057766306, -0.12685317032996887, 0.031886428972115993, -0.15371363247901712, 0.34840126090158796], 
[0.64872772809087409, 0.43260621912315134, 0.59577061735375592, -0.40728321023822228, -0.29701073058812655], 
[0.62942052897876322, 0.22749505279354201, -0.17607025935204601, -0.1651446570141834, -0.49962139823904422]]], 
[[[0.54803684026106292, -0.59862166179657317, -0.24076720561066772, 0.2149234408121751, -0.2003782580583072], 
[0.30357415450550063, 0.10717051079085479, -0.026938900079114956, 0.12986327787822058, -0.29434298720226715], 
[-0.54807051178022215, -0.36548262336787241, -0.50333027712197964, 0.34408875682200768, 0.25092626084706138], 
[-0.5317590361946819, -0.19219670227285171, 0.14875102908940974, 0.13952065368609301, 0.42209966303593122]], 
[[-0.45174872123813992, 0.49344596997025192, 0.19846526594614977, -0.17716215848691977, 0.16517251248774784], 
[-0.25023725783376488, -0.088341034119429562, 0.022205831375323396, -0.10704676293172814, 0.24262797371559403], 
[0.45177647679142546, 0.30126862942744326, 0.41489694185883325, -0.28363358896222568, -0.20683941139564599], [0.43833086931408, 
0.15842842688566058, -0.12261600359199887, -0.11500737223975258, -0.34793825707172882]], [[-0.4486394528862046, 
0.49004971035580636, 0.19709928140351843, -0.17594279766386267, 0.16403567326376461], [-0.24851494020522769, 
-0.087733005875737027, 0.022052994443025126, -0.10630998804668029, 0.24095802880041983], [0.44866701740527282, 
0.29919507620887009, 0.41204131467055755, -0.28168141311701467, -0.2054157897990585], [0.43531395252923738, 
0.15733800543983473, -0.12177206969360403, -0.11421580656190354, -0.34554348900647436]], [[0.7134110052674274, 
-0.7792601703814569, -0.31342048849911874, 0.27977817675962907, -0.26084387766161171], [0.3951798981904423, 
0.13951000411196171, -0.035067912181022814, 0.16905048130394262, -0.38316315796098172], [-0.71345483741623117, 
-0.47576970486225817, -0.65521390644481536, 0.44792008104541392, 0.32664511370006416], [-0.69222125357662645, 
-0.25019347697910849, 0.19363775097069844, 0.18162204160280435, 0.54947135449153284]], [[0.59198282282377634, 
-0.64662394043616811, -0.26007384823418078, 0.232157723415788, -0.21644619143005375], [0.32791716125306553, 
0.11576430057368851, -0.029099076815721794, 0.14027675545110513, -0.31794576503173938], [-0.59201919439498518, 
-0.39478994691540664, -0.54369132943946885, 0.37168054882657897, 0.27104753787372793], [-0.57439973407376044, 
-0.20760857298335048, 0.16067907780860699, 0.15070853698807446, 0.45594699421656493]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg0=numpy.array(arg0*1j)
      arg1=arg1*1j
      ref=-ref
      res=outer(arg0,arg1)
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")          
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_constData_rank2(self):
      arg0=numpy.array(0.372750342838)
      arg1=Data(numpy.array([[0.76871462138898106, -0.69295903418971228, 0.054927886711077623, 0.35251075234789675, 
-0.21985084950520184], [0.084613974030040451, -0.55016187666396577, 0.54086086224118057, -0.068569600221233706, 
-0.0033024490285564756], [0.86799036985074474, 0.44034217230811157, -0.096308260724639938, 0.40820711124699538, 
-0.98327313126459037], [0.011621419115650511, -0.81993724395384415, 0.87297909269920226, -0.80204874976487361, 
-0.89363699263737328]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.28653863866720669, -0.25830071756679701, 0.020474388602912477, 0.13139850379170509, 
-0.081949479526255378], [0.031539887828570062, -0.20507302814280526, 0.20160607182797236, -0.025559341990718815, 
-0.0012309890075989335], [0.3235437079418127, 0.16413769569380987, -0.035898937203226067, 0.15215934066616374, 
-0.36651539678211714], [0.0043318879596210098, -0.30563188878931291, 0.32540325609389825, -0.29896394644752133, 
-0.33310349537816136]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=numpy.array(arg0*1j)
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_constData_rank2(self):
      arg0=numpy.array([-0.7248354981517402, -0.71175522139340708])
      arg1=Data(numpy.array([[-0.52973733182056404, 0.93852863726967595, -0.5026798935486676, 0.88752851790782139, 
0.68830786230505958], [-0.84886692931572005, 0.14548667127138337, 0.38695425878446543, 0.84252686359018125, 
-0.74463622630528481], [-0.55814036535876954, -0.4005814259625835, 0.11271709200032509, -0.57891927518335184, 
0.71421601651008437], [0.12439171343058342, 0.44403183920210854, -0.76908159470076143, -0.53483151543574237, 
-0.37397019025429445]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.3839724227997322, -0.68027887232503947, 0.36436023105121224, -0.64331217540159136, 
-0.49890997225564726], [0.61528888357509803, -0.10545390384543163, -0.28047818292797538, -0.61069337887661224, 
0.53973877003582305], [0.40455994976341803, 0.29035563743792364, -0.081701349530271167, 0.41962124121716921, 
-0.51768912211503848], [-0.090163529570405437, -0.32185003936329376, 0.55745764081426119, 0.38766486791811644, 
0.27106686914687256]], [[0.37704331189029833, -0.66800265800393088, 0.35778503892274621, -0.63170305675644389, 
-0.48990671492176041], [0.60418546920865202, -0.1035508979205533, -0.27541671413025892, -0.59967289432452231, 
0.52999872211146914], [0.39725931931452813, 0.28511592152208531, -0.080226978771512414, 0.41204881687703737, 
-0.50834697895385239], [-0.088536451532290145, -0.3160419800170185, 0.54739784070583497, 0.38066912367713823, 
0.26617523555897993]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_constData_rank2(self):
      arg0=numpy.array([[0.11795870957734778, -0.82634566459596437, -0.81327808756997855, 0.76035332192267013, 
0.34229080331266304], [-0.18303348598622482, -0.68197677284763358, 0.049397167409949372, 0.40385722090682474, 
0.27340856521325163], [-0.51879714918404063, -0.77485833329140297, 0.14980458961576404, 0.33593403206236161, 
-0.68622055619837052], [-0.42127435895109344, 0.88103478757968023, 0.16906790486401468, -0.86313143450198027, 
0.30859617370497516]])
      arg1=Data(numpy.array([[0.71881854358742059, 0.10699636106889332, 0.86427296110378049, -0.48139841430720809, 
-0.57445230950683612], [0.54036999379716311, 0.018193274282380045, -0.74821022888831834, 0.67727540924340257, 
0.79758060153710808], [0.92275615326190508, -0.32084725677028247, 0.45755998301402356, 0.78740423840981788, 
-0.17005371414933568], [0.88434579513768496, 0.46409423804178584, 0.72546943412878195, -0.23777811168984098, 
0.73729617122869739]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.08479090782184065, 0.012621152681158628, 0.10194852321439524, -0.056785135744259696, 
-0.067761653143153577], [0.063741347162632786, 0.0021460551573362981, -0.088257913092238044, 0.079890533302821878, 
0.094081578541242064], [0.10884712509333168, -0.037846728380054477, 0.053973185150567385, 0.092881187878556412, 
-0.020059316679890804], [0.10431628881459487, 0.05474395744169152, 0.085575438287639818, -0.028047999220672113, 
0.086970504934456391]], [[-0.5939925871246502, -0.088415979096824415, -0.71418821443562552, 0.39780149260613329, 
0.4746961754781131], [-0.44653240165203389, -0.015033933328050005, 0.61828027884821601, -0.55966359816574329, 
-0.65907727224603063], [-0.76251554672722455, 0.26513073962963107, -0.37810270825624148, -0.6506680786944401, 
0.14052314943574493], [-0.73077531381569683, -0.38350226156979722, -0.59948852168920652, 0.19648691173071509, 
-0.60926149461803791]], [[-0.58459937043861465, -0.087017795907056467, -0.70289426094492502, 0.39151078174698639, 
0.4671894756758771], [-0.439471075135558, -0.014796191315010117, 0.60850298405058745, -0.55081324958764899, 
-0.64865482630101234], [-0.7504573596182722, 0.2609380433882092, -0.37212350793419696, -0.64037861315843214, 
0.13830095942754353], [-0.71921905702012845, -0.37743767436686998, -0.59000839397873028, 0.19337972794111463, 
-0.59962682010954249]], [[0.54655606747631091, 0.081355038572370489, 0.65715281702320216, -0.36603288348679153, 
-0.43678672181967271], [0.41087211985100563, 0.01383331653725795, -0.56890413303175424, 0.51496860727475702, 
0.60644305987982161], [0.70162070645727403, -0.24395727751506019, 0.34790725306359332, 0.59870542837089513, 
-0.12930090645873554], [0.67241526306128385, 0.35287559558024229, 0.55161309419317905, -0.18079537710387028, 
0.56060559303460589]], [[0.24604497672057668, 0.036623870381803238, 0.29583268613762698, -0.16477824994665644, 
-0.19662974248590948], [0.1849636792628897, 0.0062273904690034786, -0.25610548029293395, 0.23182514389383688, 
0.27300450480673372], [0.31585094496172028, -0.10982306526056425, 0.15661857414959859, 0.26952122929709221, 
-0.05820782242247808], [0.30270343262385391, 0.15885518955210115, 0.24832151538672387, -0.081389260860483786, 
0.2523696987292216]]], [[[-0.13156786382434668, -0.019583916954280335, -0.15819089291446184, 0.088112029918889223, 
0.10514400874187398], [-0.098905803687049446, -0.0033299784134075525, 0.13694752644398009, -0.123964079126567, 
-0.14598395785432702], [-0.16889527544676564, 0.058725791875782175, -0.083748798738854546, -0.14412134263647744, 
0.031125524105657915], [-0.1618648937013103, -0.084944786214908891, -0.13278519950504486, 0.043521356673813509, 
-0.13494988842428499]], [[-0.49021755061878514, -0.072969033028204039, -0.5894140848730246, 0.32830253704319784, 
0.3917631321923421], [-0.36851978451348505, -0.01240739048262939, 0.51026199730884458, -0.461886097924876, 
-0.54393144472215127], [-0.62929826352685037, 0.2188103767492133, -0.31204528060012182, -0.53699140143727631, 
0.11597268318631791], [-0.60310329144937291, -0.31650149075691858, -0.49475330348674562, 0.16215914926404193, 
-0.5028188634874633]], [[0.035507599934963802, 0.0052853171599755132, 0.042692636147536107, -0.023779718062417327, 
-0.028376316901741236], [0.026692747046911769, 0.00089869621546185359, -0.036959465934232799, 0.033455486773038327, 
0.039398222497056649], [0.045581540181239222, -0.015848945655704657, 0.022602167081037312, 0.038895538984033459, 
-0.0084001717865184108], [0.043684177290701016, 0.022924940770542988, 0.035836135088460681, -0.011745565189564714, 
0.036420342400898663]], [[0.29030005934950692, 0.043211253028426433, 0.34904287617628504, -0.19441622575106127, 
-0.23199671326093799], [0.21823232395636041, 0.0073474851908776114, -0.30217010369289549, 0.27352256456557295, 
0.32210868518597002], [0.372661735631025, -0.1295764814548247, 0.1847889031382175, 0.31799888745444393, -0.068677420401234285], 
[0.35714943514494163, 0.18742780921442601, 0.29298606952009665, -0.096028407379531749, 0.29776238269766414]], 
[[0.19653114665091584, 0.029253721562885138, 0.23629963024799305, -0.13161844975166823, -0.15706018172570282], 
[0.14774178468837604, 0.0049741970180766787, -0.20456708515823371, 0.18517289789545652, 0.21806536790818287], 
[0.25228943590503677, -0.087722388126170669, 0.12510081845486398, 0.21528306306646142, -0.046494141994754296], 
[0.2417877150009666, 0.12688733974674193, 0.19834955709121985, -0.06501057235623571, 0.20158308831286206]]], 
[[[-0.37292101119377785, -0.055509407095608122, -0.44838234833749058, 0.24974812496429724, 0.29802422051433475], 
[-0.28034241228656592, -0.0094386188320220894, 0.3881693337375981, -0.35136855152793167, -0.41378254232194395], 
[-0.47872326170430801, 0.16645464213594244, -0.23738081476837347, -0.40850307414244419, 0.088223382108833098], 
[-0.45879607741032458, -0.24077076764881802, -0.37637147424617123, 0.1233586064830539, -0.38250715173775651]], 
[[-0.55698253862310243, -0.082907022006087835, -0.6696891061497009, 0.37301557295920751, 0.44511915909986421], 
[-0.41871019275435556, -0.014097210187558348, 0.57975693090798153, -0.52479249478559575, -0.61801197557259813], 
[-0.71500529495090615, 0.24861117062213989, -0.35454416581908893, -0.610126735800818, 0.13176753751576692], 
[-0.68524270887364702, -0.35960728777920181, -0.56213603658288525, 0.18424435131716724, -0.57130008238040131]], 
[[0.10768231693031474, 0.016028545960305677, 0.12947205625415303, -0.072115691896970854, -0.086055592479499454], 
[0.080949905161456975, 0.0027254359876389767, -0.11208532628493141, 0.10145896473855656, 0.11948123469876069], 
[0.13823310685482076, -0.048064391629815836, 0.068544585480011758, 0.11795676879669599, -0.025474826860777677], 
[0.13247905891902742, 0.069523446872890429, 0.10867865085844275, -0.035620252441307933, 0.11045035035618911]], 
[[0.24147561166851664, 0.035943718989873627, 0.29033870062606959, -0.16171811034664768, -0.19297808056016716], 
[0.18152867082179433, 0.006111739986096397, -0.25134927902075527, 0.22751985904382227, 0.26793446736908455], 
[0.30998519517562628, -0.10778351264298884, 0.15370997000428663, 0.26451588067200321, -0.057126829861366607], 
[0.29708184869799775, 0.15590504864228658, 0.24370987214488157, -0.079877759796142839, 0.24768287562499769]], 
[[-0.4932680607862624, -0.073423102403897655, -0.59308187207584895, 0.33034548761890592, 0.3942009833392196], 
[-0.3708129976963993, -0.012484598797124345, 0.51343723942105191, -0.46476030803048674, -0.54731620399982528], 
[-0.63321424072685328, 0.22017198299562465, -0.3139870660380002, -0.54033297443453954, 0.11669435430715584], 
[-0.60685626341107246, -0.31847100615749324, -0.49783203859276987, 0.16316822805560094, -0.50594778870348578]]], 
[[[-0.30281982115194922, -0.045074823419397766, -0.36409603764775844, 0.20280080838734199, 0.2420020284354675], 
[-0.22764402273330622, -0.0076643599605310678, 0.31520178453557718, -0.28531876386235389, -0.33600025662437272], 
[-0.38873350693358599, 0.13516472241711763, -0.19275828852590599, -0.33171321577146995, 0.071639269415513879], 
[-0.37255220793772326, -0.19551100262394949, -0.30562167080121511, 0.10016982157473925, -0.3106039718914651]], 
[[0.63330414285787817, 0.094267516246131194, 0.76145454469693052, -0.42412874969034597, -0.5061124684810121], 
[0.47608476269951672, 0.016028907542755561, -0.65919924007356345, 0.59670319631570223, 0.70269625585291962], 
[0.8129802714769454, -0.28267759471412895, 0.40312626243972233, 0.69373052592673379, -0.14982323792269561], 
[0.77913940976611373, 0.40888316843009836, 0.63916380879320223, -0.20949078812375652, 0.64958357560178692]], 
[[0.12152914514172762, 0.018089650593991421, 0.14612081876443422, -0.081389021311778581, -0.097121448412615288], 
[0.091359222702666984, 0.0030758987655383546, -0.12649833579597286, 0.1145055344567002, 0.13484528126205939], 
[0.15600844953236792, -0.054244973483518207, 0.07735870767779511, 0.13312478486899304, -0.028750625165572231], 
[0.14951449075922954, 0.07846344048518604, 0.12265359727103547, -0.040200647165923094, 0.12465311893389569]], 
[[-0.6204348806732346, -0.09235192261588572, -0.74598116071878029, 0.41551010390795912, 0.49582784595761104], 
[-0.46641032790797154, -0.015703186929638673, 0.64580376816942919, -0.58457769553317385, -0.68841688873567641], 
[-0.79645984226047728, 0.2769333529921591, -0.39493440450959588, -0.67963334983160539, 0.14677870623610581], 
[-0.76330665475298443, -0.40057432542511012, -0.62617547336691548, 0.20523376263602452, -0.63638350192544324]], 
[[0.22182465213926089, 0.033018667626216448, 0.26671132883329551, -0.14855770868284679, -0.17727378468979577], 
[0.1667561124707857, 0.0056143748307076096, -0.23089481376185872, 0.20900459983698519, 0.246130321855664], 
[0.28475901815934557, -0.099012235783046854, 0.14120125999864111, 0.24298993513234984, -0.052477925510804588], 
[0.27290572861157342, 0.14321770609822104, 0.22387709151205562, -0.073377415458279152, 
0.22752677732850421]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_expandedData_rank2(self):
      arg0=numpy.array(-0.921419557017)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.27411787614041705, -0.32088094729570171, 0.75322329571656921, -0.7137139228727496, 
0.22883188436579371], [-0.54384414564382388, 0.65122352185543342, -0.89272014052793924, -0.92983942126507046, 
0.029342402377037446], [-0.30097855865431367, 0.84358871207542063, -0.40336790135540901, -0.57151413265873607, 
0.85041879066142778], [-0.87329972577643389, 0.2208570947409334, 0.67495300452707463, -0.081365038924114508, 
0.61526159570110073]])+(1.-msk_arg1)*numpy.array([[0.073197150752966511, -0.26693292886439668, 0.57752280048264093, 
-0.51305472225006565, -0.8102674893157904], [0.71818111351797986, -0.13429691550499201, 0.12139067548095128, 
-0.17097142145570587, 0.55948298386490847], [-0.54227231534025067, 0.80440840691869786, -0.8401661844295587, 
-0.60447001903774922, 0.7220179464068277], [-0.13044206585017437, 0.15860167865578378, -0.8719997620994473, 
0.94997502334238826, 0.31743055979649304]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.25257757200387571, 0.29566598031255498, -0.694034675474408, 0.65762996665061724, 
-0.21085017352380497], [0.50110863176568243, -0.60005008902739709, 0.82256979642583683, 0.85677222763945171, 
-0.027036663400078514], [0.2773275301870175, -0.77729913738548106, 0.37167107298197177, 0.52660429894364225, 
-0.78359250537059433], [0.80467544646840894, -0.2035020464003586, -0.62191489843895464, 0.074971338122167649, 
-0.56691406696077651]])+(1.-msk_ref)*numpy.array([[-0.067445286221740147, 0.24595722106761103, -0.53214080298820987, 
0.472738654901382, 0.74659631107102176], [-0.66174612347605788, 0.12374380439342375, -0.11185174242771087, 0.15753641142036548, 
-0.51551856315162214], [0.49966031658365728, -0.74119763796416382, 0.77414555347815084, 0.55697049717211089, 
-0.66528145633685021], [0.12019187053211269, -0.14613868848924111, 0.80347763451302112, -0.87532556518581417, 
-0.29248672579149548]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=numpy.array(arg0*1j)
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_expandedData_rank2(self):
      arg0=numpy.array([0.45174073833470918, -0.32999422808025969])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-0.097138691944465005, -0.21553892514741024, -0.51682088548807403, 0.4592939442072681, 
0.13793149312241337], [-0.86233231279442091, 0.11269037329575693, -0.42541177929681973, 0.41252622674889983, 
0.34353528222730056], [-0.93182078713230698, 0.64040311928947813, -0.020493489734802806, -0.30033328959216421, 
-0.62520987906799363], [-0.22284178783794584, -0.19029020302641642, -0.12015018256454324, -0.67086908239846776, 
0.19266225130706749]])+(1.-msk_arg1)*numpy.array([[0.44810843186804195, 0.067119478562676527, 0.97389928428461525, 
-0.89095081193287706, 0.40325892981648548], [-0.8596003320912915, 0.36057172930421189, -0.25785774808618123, 
-0.49573945621014581, 0.12835464234318317], [-0.75805229637792237, 0.54720140452346189, -0.018817847608054938, 
-0.49377906267057603, -0.47063696225714668], [-0.96006565494914842, 0.54119283890237635, 0.53704331265262861, 
0.059162490255933786, -0.963426135668078]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.043881504419860488, -0.097367713185960714, -0.23346904839718075, 0.20748178546885201, 
0.062309274542727881], [-0.38955063567162906, 0.050906832435839232, -0.19217583127582769, 0.18635490225397966, 
0.15518888203738346], [-0.42094141037477822, 0.28929617793967971, -0.0092577441838546035, -0.13567278198685628, 
-0.28243277238432968], [-0.10066671376974026, -0.085961836813015058, -0.054276732182756866, -0.30305889460861268, 
0.087033387654681951]], [[0.032055207664939869, 0.07112660122526851, 0.17054790916239329, -0.15156435058061529, 
-0.04551659660088845], [0.28456468590925998, -0.037187172747809617, 0.14038343172530382, -0.13613127375886538, 
-0.11336466027693221], [0.30749548135886562, -0.2113293330101218, 0.0067627333257069778, 0.099108252065771313, 
0.20631565143119507], [0.073536503761607933, 0.06279466865893818, 0.039648866749088719, 0.22138292498899451, 
-0.063577430900280743]]])+(1.-msk_ref)*numpy.array([[[0.20242883386607799, 0.030320602802544179, 0.43994998174637695, 
-0.40247877760246653, 0.18216848669536381], [-0.38831648869168123, 0.16288493921850758, -0.11648484950577695, 
-0.2239457079700185, 0.057983020900797093], [-0.34244310406208445, 0.24719316649721854, -0.0085007883723327796, 
-0.22306011834502665, -0.21260588881764811], [-0.43370076781652445, 0.2444788526272168, 0.24260434257541652, 
0.026726107029935567, -0.43521883385765325]], [[-0.14787319607055013, -0.022149040517439978, -0.321381142545419, 
0.29400862544127038, -0.13307311926126275], [0.28366314804600062, -0.11898658947930775, 0.08509156853421343, 
0.16359115918099476, -0.042356291120556541], [0.25015288238770073, -0.18057330509015374, 0.0062097810955520508, 
0.16294424062817092, 0.15530748106608544], [0.31681612471131321, -0.17859051311615401, -0.17722119340446973, 
-0.019523280303312754, 0.31792506395213493]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank2_expandedData_rank2(self):
      arg0=numpy.array([[0.61580845642221882, 0.17822708313385949, 0.58972637660522143, -0.74843177869717703, 
0.9976037022648363], [0.55187701135213363, 0.22487493386286173, 0.29391218079007464, 0.80458982942274737, 
-0.61481939252808426], [-0.22926550831669279, -0.77143931293288204, 0.84539028644557157, 0.11279761558864387, 
-0.99157575517310503], [-0.70875966152114378, 0.13774326141678772, 0.99325430842393869, 0.28694347396680775, 
0.3742883562671242]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.25127802526427567, -0.23594707373025448, -0.18977891359341159, -0.037150166425794717, 
-0.93771552478023867], [-0.21845508307326078, 0.41466029902966173, -0.001028329177914955, 0.50999614420739126, 
-0.5183973685029688], [0.039164165095863623, -0.6280324353802238, 0.29056793167506711, -0.015424500823010723, 
-0.39665605959047312], [0.48049748658112046, 0.26031355791525357, 0.63368537749662179, 0.039647163322244383, 
-0.96028049287581796]])+(1.-msk_arg1)*numpy.array([[-0.34344273002027159, 0.47629490451433143, 0.37993140152802929, 
-0.72076603796295546, -0.30431457202317413], [-0.96759705784412997, 0.22558806065373616, 0.42407326960757774, 
-0.61572292288676489, -0.30944640576414106], [-0.064828948526520991, -0.23692625772967313, 0.4289947836296979, 
-0.29011540864147944, -0.49760550944029203], [0.010984444410195859, 0.62280811555934634, -0.29684745695028547, 
-0.10474252571079257, -0.35566145942346195]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.15473913287081689, -0.14529820327116746, -0.11686745984144443, -0.022877386642497183, 
-0.57745314987806962], [-0.13452648750493229, 0.25535131868503169, -0.00063325380374573768, 0.31405993834563695, 
-0.31923348331115337], [0.024117624054748717, -0.38674768461458248, 0.17893418949061982, -0.0094985380429014766, 
-0.24426415578692889], [0.2958944155262756, 0.16030329028556817, 0.39022881417352573, 0.024415058446990923, 
-0.59134884805022492]], [[0.044784549498488105, -0.042052158724912941, -0.033823742210066503, -0.0066211658000068308, 
-0.16712630279091828], [-0.038934612251912228, 0.07390369558747055, -0.00018327610988122208, 0.090895125191598508, 
-0.092392450892552616], [0.0069801149084086841, -0.11193238907127138, 0.051787074914685788, -0.0027490637904810161, 
-0.070694852508180375], [0.085637665486503894, 0.046394926127432644, 0.11293989645580155, 0.0070661982734553539, 
-0.17114799123560198]], [[0.14818527935961659, -0.13914421286154799, -0.11191763106951802, -0.021908433036564867, 
-0.55299557871511384], [-0.12882872459178671, 0.24453611566880001, -0.00060643284004921255, 0.30075817820605882, 
-0.30571260176893755], [0.023096141174752338, -0.37036729250733225, 0.17135557350441089, -0.0090962349812983689, 
-0.23391854078079449], [0.28336204172940016, 0.15351377129057595, 0.37370098157879467, 0.023380977968702612, 
-0.56630273558833222]], [[-0.18806445939605604, 0.17659028807032834, 0.1420365698599349, 0.027804365136953689, 
0.70181609812323087], [0.16349872638996013, -0.31034494515787303, 0.00076963423571309558, -0.38169732133783985, 
0.38798506458061288], [-0.029311705743887109, 0.47003943269114079, -0.2174702739359303, 0.011544186586481986, 
0.29687000021031124], [-0.35961958854143095, -0.19482693916950383, -0.47427027421418871, -0.029673196965564842, 
0.71870443733125022]], [[0.2506758883014385, -0.23538167429185616, -0.18932414681258586, -0.037061143566127629, 
-0.93546847919197995], [-0.21793159965245734, 0.41366664949423465, -0.0010258649950349147, 0.50877404160208484, 
-0.51715513406291036], [0.039070316095744829, -0.62652748267771285, 0.28987164439848295, -0.015387539126622512, 
-0.3957055535732375], [0.47934607154227427, 0.25968976912598885, 0.63216687866172028, 0.039552156914569626, 
-0.9579793749056178]]], [[[0.13867456560131439, -0.13021376588753436, -0.1047346196515868, -0.020502322818301964, 
-0.51750364131421567], [-0.12056033836115324, 0.22884148655487174, -0.0005675112333939019, 0.28145514786628684, 
-0.28609159042222909], [0.021613802385206764, -0.34659666346983992, 0.16035776172760699, -0.0085124274158016825, 
-0.21890536070150413], [0.26517551685660068, 0.14366106835671069, 0.3497163922703842, 0.021880358002870158, 
-0.5299567284680603]], [[0.056506129312494491, -0.053058582600226736, -0.042676520642884179, -0.0083541412179948937, 
-0.21086871661713485], [-0.049125072358105482, 0.093246707319849645, -0.00023124545587287649, 0.1146853491989516, 
-0.11657457395678666], [0.0088070390357265303, -0.14122875236985979, 0.065341444418099248, -0.0034685836024421924, 
-0.089198005166710989], [0.10805184051620076, 0.058537994119798868, 0.1424999573544154, 0.0089156532299397833, 
-0.21594301232524582]], [[0.07385367239004674, -0.069347718991095625, -0.055778334362210739, -0.010918886430919539, 
-0.27560601484886921], [-0.064206609870738993, 0.12187371277487236, -0.00030223847125104912, 0.14989407893852374, 
-0.15236330109254351], [0.0115108251721478, -0.1845863826895032, 0.085401454466280377, -0.0045334486744893826, 
-0.11658204749783375], [0.14122406414520675, 0.07650932549609557, 0.1862478512348138, 0.011652784234181108, 
-0.28223813383129936]], [[0.20217574348506837, -0.18984061580542186, -0.15269418371615734, -0.029890646067556849, 
-0.75447637412999424], [-0.17576673802644699, 0.33363145926466092, -0.00082738319784902775, 0.41033771067408381, 
-0.41709725029700478], [0.031511088913965229, -0.50530851005452693, 0.23378800258216276, -0.012410396486117223, 
-0.31914543132539791], [0.38660339076636258, 0.20944564115946235, 0.50985680978769621, 0.031899704374540414, 
-0.7726319179609461]], [[-0.15449060284863858, 0.14506483653961416, 0.11667975637014111, 0.022840642754224341, 
0.57652568930954007], [0.13431042146977437, -0.25494119315493041, 0.00063223672048457694, -0.31355551957325356, 
0.31872075519115273], [-0.024078888193108473, 0.38612652040840251, -0.17864679924060667, 0.0094832822260523877, 
0.24387183759999828], [-0.29541917281107583, -0.16004582354428049, -0.38960205884640275, -0.024375844869244035, 
0.59039906928647978]]], [[[-0.057609384191028935, 0.054094525794602985, 0.043509759092783228, 0.00851725178965956, 
0.21498582644519576], [0.050084215665156487, -0.095067304235787228, 0.000235760411691559, -0.11692452524126092, 
0.11885063619986905], [-0.0089789922185020514, 0.14398617553681753, -0.066617204556014317, 0.0035363060217187998, 
0.090939553128906211], [-0.11016150050591385, -0.05968092017716746, -0.14528220018461835, -0.0090897270523892966, 
0.22015919532577868]], [[-0.19384574716500419, 0.18201884844699157, 0.14640291471165023, 0.0286590988628573, 
0.72339062016296429], [0.16852483919273198, -0.31988525618398567, 0.00079329355447954836, -0.39343107508576891, 
0.39991210978414432], [-0.030212776613142995, 0.48448891044928449, -0.2241555255717424, 0.011899066317236065, 
0.30599607808113888], [-0.37067465091411628, -0.2008161122652572, -0.48884981223160789, -0.030585380433049966, 
0.74079812364697029]], [[0.21242800175564358, -0.19946736424681422, -0.1604372501240636, -0.031406389836203252, 
-0.79273559609842548], [-0.18467980525479508, 0.35054978897429206, -0.00086933949827786299, 0.43114578643762352, 
-0.43824809985135532], [0.033109004748793802, -0.53093252044319728, 0.24564330699068226, -0.013039723169044989, 
-0.33532917983756177], [0.40620790781719063, 0.2200665532916421, 0.53571146279823922, 0.033517326757746539, 
-0.81181180094038241]], [[0.028343562099633311, -0.026614267321890657, -0.021406608942340102, -0.0041904501915509366, 
-0.10577207529566482], [-0.024641212483882933, 0.046772693009819899, -0.00011599307930903727, 0.057526349025995904, 
-0.058473987094562439], [0.0044176244393334093, -0.070840561223218301, 0.032775369859471555, -0.0017398469144806845, 
-0.044741857730592402], [0.054198970782686796, 0.029362748638236957, 0.07147819961500862, 0.0044721054876027027, 
-0.10831734989267999]], [[-0.24916119765983072, 0.23395939781496136, 0.18818016956231853, 0.036837204328463928, 
0.92981597962150964], [0.21661476396977195, -0.41116709915064237, 0.0010196662811575599, -0.50569981182781576, 
0.51403026215308167], [-0.038834236580655136, 0.62274173638534969, -0.28812011627979184, 0.015294561051745037, 
0.39331453183241155], [-0.47644965811545342, -0.25812061277161535, -0.62834705673336688, -0.039313165911725907, 
0.9521908549013407]]], [[[-0.17809572813400945, 0.16722976811395951, 0.13450763856231679, 0.026330539381400425, 
0.66461493794636373], [0.15483215073657766, -0.29389449318651933, 0.00072883824007131958, -0.36146469454551905, 
0.36741914343361581], [-0.027757980397102495, 0.44512405632438701, -0.2059428289029194, 0.010932263982449683, 
0.28113381453565439], [-0.34055723595099524, -0.18449974919737977, -0.4491306336654039, -0.028100310056547437, 
0.68060807709602189]], [[0.034611854722271317, -0.032500119457352532, -0.026140766506491261, -0.0051171850856654121, 
-0.12916399466438469], [-0.030090715615586239, 0.057116661968406063, -0.00014164541477605007, 0.070248532213112461, 
-0.071405744247479266], [0.0053945998309697764, -0.086507235924800002, 0.040023774572054083, -0.0021246210490874235, 
-0.054636699308723467], [0.066185290904252722, 0.035856438458254883, 0.087285890608512981, 0.0054611295819299863, 
-0.13227216696363556]], [[0.24958298120600111, -0.23435544754259599, -0.18849872357467046, -0.036899562861086956, 
-0.93138998506398674], [-0.21698145245962572, 0.41186312854357027, -0.0010213923864420761, 0.50655586751358772, 
-0.51490041974120593], [0.038899975717292985, -0.62379592227138614, 0.28860785002609307, -0.015320451897743989, 
-0.39398034015069999], [0.47725619873357156, 0.25855756294049009, 0.62941073138376957, 0.039379715786606789, 
-0.95380273684436956]], [[0.07210258950085055, -0.067703473008461745, -0.054455820752140158, -0.010659997812662601, 
-0.2690713502730499], [-0.062684260442749024, 0.11898406671968646, -0.00029507234669234872, 0.1463400653285459, 
-0.14875074181349326], [0.011237901587616705, -0.18020980877183612, 0.083376571738193794, -0.0044259598503585823, 
-0.11381786770887546], [0.13787561803190629, 0.074695276628862661, 0.18183188362084865, 0.011376494776614206, 
-0.2755462206083456]], [[0.094050439042214648, -0.088312242392534906, -0.071032037623038613, -0.013904874726560809, 
-0.35097600241615928], [-0.08176519396168884, 0.15520252173304627, -0.00038489163770331165, 0.19088561851795571, 
-0.19403009895017886], [0.014658690978305074, -0.23506522792090287, 0.10875619353059893, -0.005773211059285588, 
-0.14846374454751265], [0.17984461444293215, 0.097432333706147095, 0.23718105833372266, 0.014839471590537065, 
-0.35942180723387379]]]])+(1.-msk_ref)*numpy.array([[[[-0.21149493744321629, 0.29330642995073852, 0.23396496992130594, 
-0.443853821279526, -0.18739948686437899], [-0.59585445062967402, 0.13891903541845912, 0.26114790556696588, 
-0.37916738272667555, -0.19055971347901932], [-0.039922214723592371, -0.14590119305840279, 0.26417861552018801, 
-0.17865552197981069, -0.30642968067561804], [0.0067643137568983817, 0.38353050428983193, -0.18280117425741635, 
-0.064501333079727732, -0.21901933433643572]], [[-0.061210795995042602, 0.084888651543109414, 0.067714065485299832, 
-0.12846002856808619, -0.054237098526819127], [-0.17245200126846361, 0.040205902040139567, 0.075581341877197367, 
-0.10973850056476241, -0.055151730285599582], [-0.011554274398516954, -0.042226675832980669, 0.076458488965962232, 
-0.051706423054358575, -0.08868677849888143], [0.0019577254870752356, 0.11100127378823799, -0.052906256387953307, 
-0.018667954837507841, -0.063388504496175141]], [[-0.20253723674626006, 0.28088366823476657, 0.22405556878166821, 
-0.42505474394799519, -0.17946232990739516], [-0.57061750693629165, 0.13303522961472675, 0.25008719270080598, 
-0.363108048306788, -0.18248870762479602], [-0.038231340913671634, -0.13972166349355497, 0.25298953933248269, 
-0.17108870873548282, -0.29345109406101871], [0.006477816601046283, 0.36728637330913932, -0.1750587751917663, 
-0.061769430163904945, -0.20974294376392319]], [[0.2570434533096862, -0.35647424257006316, -0.28435273462853433, 
0.53944420781713176, 0.22775869642277441], [0.72418038706443744, -0.16883727348792241, -0.31738991147032691, 
0.46082660236076622, 0.23159952387750446], [0.048520045256771838, 0.17732314049268505, -0.32107332896378538, 
0.21713159131700083, 0.37242377651991265], [-0.0082211072679231507, -0.46612938571511858, 0.22217007020703583, 
0.078392634822963284, 0.26618833869033548]], [[-0.34261973898416559, 0.47515356011337373, 0.37902097277103008, 
-0.71903886793860194, -0.30358534370345769], [-0.965278407205867, 0.22504748449491163, 0.42305706379207364, 
-0.61424746744116299, -0.3087048800428539], [-0.06467359906399385, -0.23635851187487469, 0.42796678440128899, 
-0.28942020574481575, -0.4964130984850153], [0.010958122410933675, 0.6213156818825899, -0.29613612206150641, 
-0.10449153143365647, -0.35480918867376049]]], [[[-0.18953814741420519, 0.26285620842561908, 0.20967540639411625, 
-0.39777420691511434, -0.16794421651905295], [-0.53399457247613602, 0.12449686471030776, 0.23403628862535761, 
-0.33980332650374806, -0.17077635758677381], [-0.035777606361917715, -0.13075415502669735, 0.23675235907521289, 
-0.16010802466826263, -0.27461704138226428], [0.0060620623524625409, 0.34371348146074632, -0.16382328736920471, 
-0.057804992050746219, -0.19628138327975833]], [[-0.077231661198989249, 0.10710678515187833, 0.085437048791039943, 
-0.1620822151175165, -0.068432719257216348], [-0.21758832438859832, 0.050729100219760161, 0.095363448456011579, 
-0.13846065156200915, -0.069586740030311497], [-0.014578405512300274, -0.053278776537335579, 0.096470173596240993, 
-0.065239683330849799, -0.11189900602518128], [0.0024701262102630752, 0.14005393379566156, -0.066753552249054132, 
-0.023553968541843572, -0.079979347165419878]], [[-0.10094200175675486, 0.13998887408500751, 0.11166646677373258, 
-0.21184191805711397, -0.089441759509529345], [-0.28438856139702823, 0.066303078866943227, 0.12464029948514045, 
-0.18096846702808803, -0.090950067955789027], [-0.019054017639757281, -0.069635513095759508, 0.12608679240417073, 
-0.085268452434620889, -0.1462523204527523], [0.0032284620113680104, 0.18305089145780429, -0.087247083434246198, 
-0.030785104153119507, -0.10453323516213034]], [[-0.27633052756349302, 0.38322203597810967, 0.30568894154778242, 
-0.57992102353832375, -0.24484840959498205], [-0.77851875172076079, 0.18150585924119797, 0.34120503965630772, 
-0.4954044014971376, -0.24897743082925253], [-0.052160712636609596, -0.19062845729248759, 0.34516483978386703, 
-0.23342390715175859, -0.40036833196038385], [0.0088379722543031371, 0.50110507546099725, -0.23884044475220653, 
-0.084274770894954318, -0.28616159296976862]], [[0.21115525063925022, -0.29283534385772314, -0.23358919348980664, 
0.44314093761525841, 0.18709850030873187], [0.59489743531568962, -0.1386959144127187, -0.26072847000752947, 
0.37855839341485725, 0.19025365121190829], [0.03985809475131008, 0.14566685785130995, -0.26375431226892782, 
0.17836857930399133, 0.30593751703270822], [-0.0067534494395351287, -0.38291450726975823, 0.18250757315568117, 
0.064397736029366728, 0.21866756242838475]]], [[[0.078739572075770259, -0.10919799339212885, -0.087105165896797149, 
0.16524679207098567, 0.069768835043069841], [0.22183663131237086, -0.051719561395955745, -0.097225373720103217, 
0.14116402889787399, 0.070945387514289374], [0.014863041837569547, 0.054319018911965271, -0.098353707134072321, 
0.066513456632693826, 0.11408378006301542], [-0.0025183542312800086, -0.14278841919747509, 0.068056883110224772, 
0.024013848399459123, 0.081540905283376805]], [[0.2649452236786316, -0.36743261389196852, -0.29309401935640983, 
0.55602725711149792, 0.23476022435702151], [0.74644240949915375, -0.17402749851657953, -0.32714679173927064, 
0.47499286858879181, 0.2387191226522388], [0.050011599509460526, 0.18277422947873798, -0.33094344113508456, 0.2238064315136252, 
0.38387245231423561], [-0.0084738322487509302, -0.48045866475612514, 0.22899979823560151, 0.080802502069188556, 
0.27437123189434159]], [[-0.29034314790948651, 0.4026550857599368, 0.32119031636744816, -0.60932860729374261, 
-0.25726458321223272], [-0.81799715389474137, 0.19070995521476297, 0.35850742286746023, -0.52052617815034674, 
-0.26160298560849976], [-0.054805763364800794, -0.2002951568885657, 0.3626680230163663, -0.24526074841369436, 
-0.42067086416262306], [0.0092861426063809357, 0.52651593121334239, -0.25095195666184134, -0.088548313813679572, 
-0.30067274305965053]], [[-0.038739521037540997, 0.053724929546237393, 0.042855356179613349, -0.081300690479495341, 
-0.034325958113092678], [-0.10914264097540498, 0.025445795347007808, 0.047834453646614886, -0.069452077564897519, 
-0.034904816722671095], [-0.0073125508149104951, -0.026724716942247633, 0.048389588693396116, -0.032724326340283931, 
-0.056128714968637361], [0.0012390191380361006, 0.070251270404350835, -0.033483685337544811, -0.011814707150909627, 
-0.040117764579743719]], [[0.34054948437856364, -0.47228247962890013, -0.37673076638413183, 0.71469412839624447, 
0.30175095156405918], [0.9594457833350678, -0.22368765160076465, -0.42050077255986168, 0.61053592223883546, 
0.30683955348118125], [0.064282813592263413, 0.23493033292863835, -0.42538082654294046, 0.28767140541102892, 
0.49341355880155519], [-0.010891908761196951, -0.61756142751369736, 0.29434674129669508, 0.10386014903041751, 
0.35266528021378796]]], [[[0.24341835308106527, -0.33757861530782302, -0.26928005154825979, 0.51084989310256024, 
0.21568589306309663], [0.68579376320646013, -0.15988771751215328, -0.30056602702723156, 0.43639957041603278, 
0.21932312980832713], [0.045948143614428671, 0.16792377423395441, -0.30405419763972097, 0.20562209883080329, 
0.3526827124419577], [-0.0077853311021682369, -0.44142126917646374, 0.21039350311149663, 0.074237277069651039, 
0.25207849559708889]], [[-0.047306921742877518, 0.065606413544001499, 0.052332990361121878, -0.099280664787473716, 
-0.041917281647125951], [-0.13327997448473866, 0.031073235211133744, 0.058413235235428483, -0.084811683527500278, 
-0.042624157163655446], [-0.0089297508042640566, -0.032634995454959591, 0.05909114062794376, -0.039961442573541499, 
-0.06854180576926798], [0.0015130331979117809, 0.085787621073987977, -0.040888736863611809, -0.014427577100436309, 
-0.048989969381242157]], [[-0.34112597128951433, 0.47308196598922819, 0.37736850147326051, -0.71590397257235772, 
-0.30226175977820469], [-0.96106994652200917, 0.22406631317332423, 0.4212126021251531, -0.61156944595265983, 
-0.30735897575153542], [-0.064391632434560725, -0.23532802626875834, 0.42610091713159282, -0.28815837957332102, 
-0.49424881614705896], [0.010910346736070288, 0.61860684410071509, -0.2948450155605607, -0.10403596493744989, 
-0.3532622769126994]], [[-0.098548650060661186, 0.13666971453403123, 0.10901883622353085, -0.20681911085038249, 
-0.08732108047505191], [-0.27764566117785688, 0.064731021809417991, 0.12168505719766103, -0.17667767449412519, 
-0.088793626676505033], [-0.018602243703815295, -0.067984443466907643, 0.12309725352834454, -0.083246723206886156, 
-0.14278465354382056], [0.0031519146386568826, 0.17871072429331988, -0.085178440535527325, -0.030055184199512499, 
-0.102054734723073]], [[-0.12854661489118116, 0.17827163690907599, 0.14220389977219083, -0.26977433560232222, 
-0.11390140095068722], [-0.36216031230938489, 0.084434984415575218, 0.15872568701824527, -0.2304579207232765, 
-0.1158221865662299], [-0.024264720582517545, -0.08867873956216038, 0.16056775241193022, -0.10858681942818438, 
-0.18624794819787185], [0.0041113496427998084, 0.23310982584253287, -0.11110654672399826, -0.039203907779559545, 
-0.13311994303517405]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=numpy.array(arg0*1j)
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)      
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")      
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_array_rank3(self):
      arg0=numpy.array(-0.409786441515)
      arg1=numpy.array([[[-0.24581316871434389, -0.33314609913795179], [-0.79698015548755907, -0.41895026934279977]], 
[[-0.95131583957856281, -0.020459344268369817], [-0.84875716041095783, 0.40273550306252726]], [[0.54396865218080359, 
-0.63003080302737158], [-0.61695250839228399, -0.46130757001642531]], [[0.21174308619500604, -0.7300390108976802], 
[0.83728873908140167, -0.61171328391097091]], [[0.53377663455002256, 0.27485489141149699], [0.14771677004966133, 
-0.8418463957002067]], [[0.76892157851423759, 0.10470902366721901], [0.11160616603087448, -0.73241482746681186]]])
      res=outer(arg0,arg1)
      ref=numpy.array([[[0.1007309036848804, 0.13651875447021333], [0.32659166187500405, 0.17168014004557156]], 
[[0.38983633265737877, 0.0083839618834575125], [0.34780917647484783, -0.16503554867158765]], [[-0.22291097827266779, 
0.25817808081717608], [0.25281877299758398, 0.18903758756078076]], [[-0.086769445807171969, 0.29916008844260283], 
[-0.34310957290841881, 0.2506718098410905]], [[-0.2187344276358959, -0.11263180788440072], [-0.06053232955068201, 
0.34497723879588438]], [[-0.31509363746314295, -0.04290833820305831], [-0.045734693628880323, 0.30013366586015877]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_array_rank3(self):
      arg0=numpy.array([-0.49129863317869038, 0.54128675640520529])
      arg1=numpy.array([[[0.093841980947059422, 0.89307566094746704], [-0.78966131499119219, 0.20641795950763542]], 
[[-0.54236894611313802, -0.93859447771012605], [0.54563092327710838, -0.33381561458664288]], [[-0.87336998291594892, 
0.440422743370255], [0.04740961922232878, 0.55570236388504091]], [[0.7408413781394021, 0.0022393702192720077], 
[-0.44183603347710343, 0.95376042666640082]], [[-0.29549285253968072, 0.33527228096563499], [-0.20900772183724192, 
-0.87406076379355291]], [[0.048859208273817734, -0.82158991022488981], [-0.26546800212877186, 0.55957256702419222]]])
      res=outer(arg0,arg1)
      ref=numpy.array([[[[-0.046104436974070996, -0.43876685154864609], [0.38795952472925999, -0.10141286136963554]], 
[[0.2664651219039515, 0.46113018400805172], [-0.26806772682607022, 0.16400315518012215]], [[0.42908547886590187, 
-0.21637909183861539], [-0.023292281123452295, -0.27301581183088786]], [[-0.36397435648210558, -0.0011001995279094016], 
[0.217073439336395, -0.4685811940011273]], [[0.14517523456681744, -0.16471881338111832], [0.10268520806242887, 
0.42942485856689466]], [[-0.024004462243119612, 0.40364599992689132], [0.13042406659854328, -0.27491723734327678]]], 
[[[0.050795421481472869, 0.4834100277386893], [-0.42743321185025152, 0.11173130776566899]], [[-0.29357712761649007, 
-0.50804876041955194], [0.29534279265504343, -0.18068997125701405]], [[-0.47274360519424358, 0.23839499820596746], 
[0.025662199011260219, 0.30079433007403888]], [[0.40100762658383915, 0.0012121414423801585], [-0.23915999342376301, 
0.51625788773790071]], [[-0.15994636769212542, 0.18147844547646322], [-0.11313311181692208, -0.47311751573486854]], 
[[0.026446842367061171, -0.44471573760087441], [-0.14369431380165307, 0.30288921977785938]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_constData_rank3(self):
      arg0=numpy.array(0.387355709924)
      arg1=Data(numpy.array([[[-0.10395844223183359, -0.034376748558501147], [0.12862393234867664, -0.91804162386511479]], 
[[0.80081261014870431, -0.74967915907766769], [0.3734576493144337, -0.18944617576320444]], [[-0.62224883551171062, 
-0.5896500922539134], [-0.36425230899400796, -0.41356142099185367]], [[0.55848582677774394, -0.40903018734005703], 
[0.43711358327212779, 0.4802731028544085]], [[0.21468754528887946, 0.58178173257469279], [0.45898096180989745, 
-0.24878056263588544]], [[0.42123126772180419, 0.61407672776484268], [0.55341950102533555, 
0.22644467592489304]]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.04026889619326831, -0.013316029842744909], [0.049823214628092742, -0.35560866495172894]], 
[[0.31019933711995984, -0.29039250287949242], [0.14466095287660874, -0.073383057905076002]], [[-0.2410316394288011, 
-0.22840433009155836], [-0.14109521174170145, -0.16019537782533158]], [[0.21633267391378774, -0.15844017859730997], 
[0.16931844236564411, 0.18603652871340198]], [[0.083160446517138942, 0.22535647604207926], [0.17778889630331099, 
-0.096366571455027653]], [[0.16316633675041714, 0.23786612683094052], [0.21437020370525914, 
0.087714638201317044]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_constData_rank3(self):
      arg0=numpy.array([0.56605108848823593, -0.18851042340241464])
      arg1=Data(numpy.array([[[0.44970194057812041, 0.057775049092888997], [0.6868268383241618, -0.59156660892409763]], 
[[0.56945239876636711, -0.2197598623523993], [0.3022233880249896, -0.24972154753827924]], [[0.18469333360262574, 
0.36337543839803699], [0.24891933100108354, 0.5292816133280307]], [[0.71451631627973322, -0.95542670983685118], 
[-0.2721393858590675, 0.048385636119469666]], [[-0.59563969298710884, -0.43356870481545284], [0.46921748373561711, 
-0.36221706760565597]], [[-0.027495688126619067, 0.58580762011459719], [0.53855251006612104, 
-0.7732493783904304]]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.25455427295951705, 0.032703629426491083], [0.38877907943632545, -0.33485692289478003]], 
[[0.3223391501639391, -0.12439530929060053], [0.17107387775814786, -0.14135515380300973]], [[0.10454586252228719, 
0.20568906243509877], [0.14090105825892682, 0.29960043334114139]], [[0.40445273857274766, -0.54082032907388355], 
[-0.1540447955860452, 0.027388741992621507]], [[-0.33716249656215164, -0.24542203729522175], [0.26560106740625722, 
-0.20503336538719852]], [[-0.015563964192805787, 0.33159704101057075], [0.30484823453099946, -0.43769865231075494]]], 
[[[-0.084773503223268987, -0.010891198966595796], [-0.12947401809662953, 0.11151647191901229]], [[-0.10734771279896853, 
0.041427024698907154], [-0.056972258838703042, 0.047075114659147238]], [[-0.034816618517034395, -0.068500057746451987], 
[-0.046923888480060058, -0.099775101027580179]], [[-0.13469377330982613, 0.18010789360132076], [0.051301110852765903, 
-0.0091211967514763931]], [[0.11228429072028415, 0.081732220118797549], [-0.088452386526816792, 0.068281692777923259]], 
[[0.0051832238104897056, -0.11043084250016359], [-0.10152276169699764, 0.14576556771603397]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_expandedData_rank3(self):
      arg0=numpy.array(-0.0528042862132)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.84057186546492013, 0.95475104878375916], [0.82290236437194797, -0.94329886724569456]], 
[[-0.54658057476605704, -0.1365510319173604], [-0.40460022411662555, -0.1163723229781406]], [[0.37751551926701143, 
0.54195098813444509], [0.39841038225251446, -0.14876508383575748]], [[-0.16099764993361343, -0.94944586040930856], 
[0.80917016966538169, 0.028354118907784942]], [[-0.72728317469000103, -0.48372986468381352], [-0.37012289584931302, 
0.11281353766308144]], [[-0.5644983995881232, 0.50017748063404488], [0.66734901930012702, 
-0.71438652176093376]]])+(1.-msk_arg1)*numpy.array([[[0.74423031805071438, 0.52445338133253472], [-0.60489290108393923, 
0.60530980668442291]], [[-0.49209968010950811, -0.15600012165422306], [0.52935145057545641, 0.2918272989589823]], 
[[-0.56506933354856748, -0.53581802436899206], [0.80213522312628971, -0.32830786440359261]], [[0.47913514471837093, 
-0.69421694646042065], [-0.56527548892904078, 0.33046868249119821]], [[0.99709158711162793, 0.84770941063786132], 
[-0.202469867157685, 0.79247373048264391]], [[0.92657022957706991, -0.88882680191552499], [-0.027318369583105628, 
-0.53152810510410364]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.044385797366810691, -0.050414947642373205], [-0.043452771973852146, 0.049810223370671203]], 
[[0.028861797108546687, 0.0072104797720782153], [0.021364626036197258, 0.006144957449837893]], [[-0.019934437529316978, 
-0.02861733509100204], [-0.021037775854790015, 0.0078554340654006863]], [[0.0085013659867543053, 0.050134810957033538], 
[-0.042727653234230618, -0.0014972190101310502]], [[0.038403668914408086, 0.025543010224658234], [0.019544075326502097, 
-0.0059570383314900173]], [[0.029807935058769851, -0.026411514844819781], [-0.03523888861925209, 
0.037722670361948735]]])+(1.-msk_ref)*numpy.array([[[-0.039298550722924083, -0.027693386453387144], [0.031940937877196263, 
-0.031962952279848106]], [[0.025984972353948643, 0.0082374750731305912], [-0.027952025503582674, -0.015409732219068238]], 
[[0.029838082819026012, 0.028293488316995598], [-0.042356177903685531, 0.017336062438026448]], [[-0.025300389316533296, 
0.036657630334980845], [0.029848968706740923, -0.017450162894779127]], [[-0.052650709546660845, -0.04476269034498264], 
[0.010691276814952039, -0.041846009680883298]], [[-0.048926879599259482, 0.046933864842350362], [0.0014425270063455088, 
0.028066962192300718]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank1_expandedData_rank3(self):
      arg0=numpy.array([0.20678794651781152, 0.2502637168038282])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.053314958879399743, 0.79556982667577225], [0.15870834897630259, 0.47094597084800416]], 
[[0.069161910646639901, -0.61982663962747275], [-0.62942126336988258, -0.62334021680981344]], [[0.45765903223210813, 
0.41444013187142548], [0.25471122760688547, -0.7489874822821323]], [[-0.013047736550351852, -0.082011819940301489], 
[0.73058388881809155, 0.21765601080646824]], [[-0.39232638166275224, -0.47018056584844681], [-0.10867343405645791, 
0.22042469754020755]], [[-0.53349238995623782, 0.96587951492430824], [-0.72626294178900785, 
0.47479682180950289]]])+(1.-msk_arg1)*numpy.array([[[-0.68299337038412622, 0.84196499441255401], [-0.62088074728991893, 
-0.44309842738306671]], [[-0.26453631540983413, 0.09336790644058901], [-0.10107396285692016, -0.10155806106462872]], 
[[-0.031104927846619335, 0.60884094737193872], [0.090208804348483884, 0.8858826510976896]], [[-0.11796959941075391, 
0.36030799184669116], [0.9595854250855973, 0.0077306962795962164]], [[-0.88641372669713081, -0.50131085906678696], 
[0.76027428371304806, 0.15483479187808658]], [[-0.59528068006432378, 0.92672986021064641], [0.69245958914554029, 
-0.69230260953398992]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.011024890865352634, 0.16451425076981416], [0.032818973580041828, 0.09738595023249591]], 
[[0.014301849479867032, -0.12817267800560067], [-0.13015673054690463, -0.12889924341606873]], [[0.094638371480606553, 
0.085701223824263081], [0.052671211711858748, -0.15488158342866787]], [[-0.0026981146479526531, -0.016959055835643451], 
[0.15107594212769027, 0.045008639521928157]], [[-0.081128366828803714, -0.097227673704382969], [-0.022472356269573734, 
0.045581170566149215]], [[-0.11031979580192995, 0.19973224147481758], [-0.1501824223645338, 0.098182259795170368]]], 
[[[0.013342799770401842, 0.19910226180085616], [0.039718941302608529, 0.11786068907820885]], [[0.017308716819682358, 
-0.15512011860719832], [-0.15752130480630805, -0.15599943949212802]], [[0.11453545043525039, 0.10371932779481163], 
[0.063744978532565014, -0.18744439115546785]], [[-0.0032653750449682141, -0.020524582880106161], [0.18283863945261036, 
0.054471402249120934]], [[-0.098185058475117645, -0.11766913597815938], [-0.027197017524804881, 0.055164304081771989]], 
[[-0.13351378839700537, 0.24172459738963603], [-0.18175726318899943, 
0.11882441735269111]]]])+(1.-msk_ref)*numpy.array([[[[-0.14123479654701251, 0.17410821223445269], [-0.1283906547645266, 
-0.091627413903815985]], [[-0.054702921442987697, 0.01930735764351655], [-0.020900877225600075, -0.021000982899885081]], 
[[-0.0064321241559871052, 0.12590096926300215], [0.018654093409050009, 0.18318985427624612]], [[-0.02439469121367863, 
0.074507349747933632], [0.19843069956187193, 0.0015986148088105868]], [[-0.18329967430890029, -0.10366504311350089], 
[0.15721555791932124, 0.032017968661982241]], [[-0.12309686943222785, 0.19163656476969809], [0.14319229648597373, 
-0.14315983499445606]]], [[[-0.17092845942470511, 0.2107132889204002], [-0.1553839235087135, -0.11089145934681743]], 
[[-0.066203841524054896, 0.023366599296013896], [-0.025295145616664919, -0.025416297833424133]], [[-0.0077844348538098507, 
0.15237079843166534], [0.022575990664680915, 0.22170428491573674]], [[-0.029523510418393975, 0.090172017233676349], 
[0.24014941507270302, 0.0019347127844132758]], [[-0.22183719386915671, -0.12545991886417421], [0.1902690680323956, 
0.038749530505957135]], [[-0.14897715553440818, 0.23192685928940851], [0.17329751051601472, -0.1732582242149657]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_array_rank4(self):
      arg0=numpy.array(-0.730935877851)
      arg1=numpy.array([[[[-0.78191809977016868, -0.68664195999519873, -0.8091776474747967, 0.2261814966467186], 
[-0.66902295768628917, 0.73975108805058309, 0.86251664890996871, -0.8523231851989983], [0.9122388979308611, 
-0.30021501495916425, 0.46570071050475015, -0.12990023109379245]], [[0.40617071029300522, 0.036831899135205903, 
0.8675573142267925, -0.97899632074763021], [-0.76466833566117609, 0.76346806464430306, 0.047664613252238253, 
-0.42508616770889129], [0.023624890751065886, -0.64092450309316806, 0.7730603866640835, -0.66041329891979572]]], 
[[[0.10268058043631068, -0.25518645921054062, 0.75164024165418319, 0.22733024998903351], [0.15083427638758629, 
0.71619440821272562, -0.23384587894353004, 0.88967050676253456], [0.042075387601280712, 0.58956579004336107, 
-0.98905184748058761, 0.15987692907760476]], [[0.015268771894321409, -0.038912224301465237, 0.64817891334974553, 
0.38179161125282923], [-0.20446335810878913, 0.23547727215007752, -0.5759362456810484, 0.81575606232508835], 
[0.64623346560637107, -0.31553003581827754, 0.53002340359386113, 0.49644058502816635]]], [[[0.99568301932228476, 
0.22822554384281291, 0.41332850760217266, -0.48455525766166874], [0.41057334644739329, 0.3253517871792273, 
-0.97380219806987234, -0.72358766087548076], [0.25192451036393582, 0.41091345762120102, 0.72457229680370649, 
0.84860644997230583]], [[0.53535627062493352, -0.38101327657706063, -0.80377189834711249, -0.22316933585178012], 
[0.68592233680235259, 0.23682060765030966, 0.43467238902828464, 0.20557286974186306], [-0.49884304184952444, 
-0.51082452482911989, -0.6428107713982214, 0.52471220548819542]]]])
      res=outer(arg0,arg1)
      ref=numpy.array([[[[0.57153199266316856, 0.50189124379848726, 0.59145697409447462, -0.16532417080514383], 
[0.48901288287896394, -0.54071061093555584, -0.63044436393219294, 0.62299359558627143], [-0.66678813966900963, 
0.21943792550325641, -0.34039735764866835, 0.094948739447601321]], [[-0.29688474468542064, -0.026921756527314725, 
-0.63412876706049903, 0.71558353511866146], [0.55892352119143773, -0.5580462000420604, -0.034839775929957718, 
0.31071073115665637], [-0.017268280260266566, 0.46847471430468179, -0.56505757235821896, 0.4827197743904787]]], 
[[[-0.075052920199474737, 0.18652493857876923, -0.54940081986170974, -0.16616383583784314], [-0.11025018422139511, 
-0.52349218847901424, 0.17092634280745009, -0.65029209285870193], [-0.030754410372267206, -0.43093478829631865, 
0.72293348037847094, -0.11685978350347632]], [[-0.01116049318828395, 0.028442340828930216, -0.47377722303386527, 
-0.27906518652727086], [0.14944960414763064, -0.17211878663299818, 0.4209724653231412, -0.59626537352794118], 
[-0.4723552254797484, 0.23063222371922024, -0.38741312178750426, -0.36286623481847408]]], [[[-0.7277804417897632, 
-0.16681823823679007, -0.30211663554507717, 0.35417882262629546], [-0.30010278940778728, -0.23781129417227123, 
0.71178696449952827, 0.52889618210424016], [-0.18414066313507077, -0.30035138886718138, -0.5296158878308016, 
-0.6202769004606089]], [[-0.3913111056323243, 0.27849627378777597, 0.58750571811038799, 0.16312247441026681], 
[-0.50136524538830218, -0.1731006787461089, -0.31771764425202204, -0.15026058600713754], [0.36462227670419278, 
0.37337997248384136, 0.46985345548409868, -0.38353097653769841]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_constData_rank4(self):
      arg0=numpy.array(0.749217389397)
      arg1=Data(numpy.array([[[[-0.23764815269520856, -0.18455544660624201, -0.86854924762358765, -0.52190811115135682], 
[0.60231705734854768, -0.38772913109778928, 0.31513849762036106, -0.83855541759102703], [0.60463621224720465, 
0.83322375980887253, 0.82300214150592166, 0.63478805277209704]], [[-0.92311897579065683, 0.50028758892228264, 
-0.40790848163492055, 0.61140861252818102], [0.2482285795009469, -0.68985582771863574, -0.57852022126144687, 
0.088837349136333676], [-0.29717142897784332, 0.14598829284286263, -0.69389346371608074, 0.41237743422733653]]], 
[[[0.68613166005853499, -0.089223037792774296, -0.85328947134450228, -0.2366765680382874], [-0.082832060713332822, 
-0.13298655372469881, -0.096193670872633641, 0.9301320089812195], [0.8616956824865214, -0.45302600597414466, 
0.5867348229104401, 0.49530471719458924]], [[-0.53236094581109183, 0.15096918538813942, 0.79579488271314314, 
0.15431250414984166], [0.17451306913175046, 0.11950361241978413, -0.82875828286528663, 0.9324872081668687], 
[-0.433803042765466, 0.30160290882483398, 0.24618890795144455, 0.034820689896103874]]], [[[0.31785046073691126, 
0.37491673301292194, 0.24333760131866189, -0.70222242846076033], [-0.53548006611759313, 0.4194970567389269, 
0.60139635888872478, 0.18758102830458601], [0.082890811553601784, 0.57208279180229082, -0.69611056660004111, 
0.64094977905094641]], [[-0.04569723508861423, -0.70575800156534818, 0.40407628063560219, -0.74110273778927382], 
[-0.75353144830274577, -0.86137928176577705, 0.06012369990444677, -0.083786106712061326], [-0.59306811754723965, 
-0.48642200425558246, -0.86997408610813132, -0.27730301946656444]]]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.17805012855724847, -0.13827214990526759, -0.6507321998669976, -0.3910226325417735], 
[0.45126641329577116, -0.29049340739412999, 0.23610724248551976, -0.62826030083199469], [0.45300396447454949, 
0.62426573010729247, 0.61660751592694618, 0.47559424771811448]], [[-0.69161678914441582, 0.37482416131991358, 
-0.30561212772328006, 0.45807796453301197], [0.18597716830734645, -0.51685198230344431, -0.43343740988669271, 
0.066558486800845598], [-0.2226460022220616, 0.10937696764620802, -0.51987704940478408, 0.30896034471790745]]], 
[[[0.514061771131468, -0.06684745144914396, -0.63929931012040386, -0.17732220043701213], [-0.062059220285990778, 
-0.099635838606480592, -0.072069970967678337, 0.69687107556320149], [0.64559738968694469, -0.33941496152475487, 
0.43959193228908511, 0.37109090717239257]], [[-0.3988540780373353, 0.11310873895584569, 0.59622336452158076, 
0.11561361151040921], [0.13074822607049297, 0.089534184520623714, -0.62092011712920792, 0.69863563174858279], 
[-0.32501278321308014, 0.22596614398418796, 0.18444901091380161, 0.026088266380950405]]], [[[0.23813909241184159, 
0.28089413594907464, 0.18231276240201874, -0.52611725462716996], [-0.40119097721058639, 0.31429448970953106, 
0.45057660999928112, 0.14053896832670726], [0.062103237437161951, 0.42861437579287853, -0.5215381414395287, 
0.48021072019493094]], [[-0.034237163175738057, -0.52876616747861038, 0.30274097609492739, -0.55524705848121436], 
[-0.56455886452568493, -0.64536033676494542, 0.045045721483279219, -0.062774008138522491], [-0.44433694676314811, 
-0.36443582417346981, -0.65179971363669942, -0.20776024431655701]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_array_rank0_expandedData_rank4(self):
      arg0=numpy.array(-0.305801562699)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-0.076413284641204005, 0.38935210549404298, -0.87416483595564998, -0.17808529342420831], 
[-0.45136890149782793, -0.76051790558909405, 0.090218653691126249, -0.21900622726658181], [-0.021002797765071968, 
-0.74561261640475429, 0.0089512091020149676, -0.40272388162468853]], [[0.78254729019255809, 0.091319539172886266, 
-0.92162394941705261, 0.0035206880978480015], [-0.97199275524939011, -0.91028332618997254, 0.57471596823472937, 
-0.34679234179543195], [0.32401911995733079, -0.45404235298596274, 0.89010307011958845, -0.62562287427214014]]], 
[[[-0.89077603808072126, 0.35306075395242598, 0.41112703301475717, -0.78019271431447534], [-0.50972742129771409, 
0.25265771402863657, -0.20898468571319029, -0.32322942473076854], [-0.67890323338268299, 0.27809696504865955, 
0.50181490703872633, 0.2474557882082471]], [[-0.50803662282881779, 0.28745039939039341, 0.75142850079254853, 
0.38461906474766083], [-0.79004617450693693, -0.80488457439045957, 0.28994104666897669, 0.84024531664112345], 
[0.09295599333181026, -0.77393733706441026, 0.082149420520204242, 0.65119514574683679]]], [[[-0.56328794490739398, 
-0.77708657567269745, -0.35569077221534151, 0.9655445481998306], [0.61573578226884251, 0.96366054275303337, 
0.79633760080273963, 0.78496512183264811], [0.87001146332336354, -0.23162665449416164, 0.092499885172479646, 
-0.82048676789744923]], [[-0.27665579014160069, 0.78610284117250617, -0.46563337780422298, 0.59181244373631237], 
[0.97715866220645253, 0.36730209557650206, -0.57698674789558768, -0.73888924091888231], [0.12486787492493812, 
-0.48572943068852403, 0.37108312575493785, 0.17693016092234015]]]])+(1.-msk_arg1)*numpy.array([[[[0.24779411139647256, 
0.030923197183163165, 0.26385000008947146, 0.26084330968092395], [-0.40869869895284539, 0.27562112292321661, 
-0.034057342292344694, -0.21741963202156644], [0.0085135581535666915, -0.79692161059190791, -0.53555664354946408, 
0.36274753547938765]], [[-0.18566438430195764, -0.79711944059062279, -0.10551670269570645, 0.29702369829533692], 
[0.582560976676052, 0.93278036958236088, -0.79230495881995266, -0.23574276521562432], [-0.54366001095056271, 
-0.12021671908123022, -0.74952574526028304, 0.22443224791911232]]], [[[0.79516700008567032, 0.43491471551636662, 
0.87961514755304537, -0.94994363819548955], [0.70962114713121704, 0.049025057757413171, -0.60055359926960605, 
0.15148560426616231], [0.9200852427312125, 0.78218919602399173, -0.35575228880385246, 0.57639041676455971]], 
[[-0.91251891001791252, -0.60692661160038575, -0.25804212419225925, 0.75833778551217645], [-0.8441072639167031, 
-0.77616174870793708, 0.82337316848673225, 0.74305007885292218], [0.49959972075119308, -0.29782295278750781, 
-0.59591336226686775, -0.13699931234570717]]], [[[-0.1854359391396716, 0.98823418676405672, 0.7025739902289343, 
0.90216845426126446], [-0.21513498062117953, 0.45844653154002812, 0.79638962266025293, -0.053347997254850288], 
[-0.45013562465321955, -0.33375799278898621, -0.07278996036103158, -0.59715058872691285]], [[0.32569967219899443, 
0.81866841960423642, -0.13546178282279953, 0.70390122030123692], [0.01050667752783685, 0.4836265262041195, 0.37130052883088704, 
-0.30679879418041689], [0.85442902550949595, 0.17981560260205076, -0.95381997614648251, 0.84655270932562532]]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.023367301854272833, -0.1190644823003728, 0.26732097289208628, 0.054458761022900794], 
[0.138029315431939, 0.23256756399000567, -0.027589005283380732, 0.066972446539016614], [0.0064226883776180854, 
0.22800950326494837, -0.0027372937314451046, 0.12315359233719049]], [[-0.23930418422705066, -0.027925657784056007, 
0.28183404395291045, -0.0010766319220990319], [0.29723690348774096, 0.27836606364808447, -0.17574904119446835, 
0.10604964005322114], [-0.099085553227430109, 0.13884686107481159, -0.27219490980608729, 0.19131645261289915]]], 
[[[0.27240070446024833, -0.10796653028647371, -0.12572328916387296, 0.2385841512440387], [0.15587544198356698, 
-0.077263123778010506, 0.063907843471332701, 0.098844063193091136], [0.20760966969008737, -0.085042486493835376, 
-0.15345578275828739, -0.07567236673308915]], [[0.15535839316956876, -0.087902781332143645, -0.22978800979921479, 
-0.11761711104380929], [0.24159735476888958, 0.24613496064122914, -0.088664425162067378, -0.25694833087969221], 
[-0.028426088023140866, 0.23667124710569462, -0.025121421169927098, -0.19913649319163421]]], [[[0.17225433380240421, 
0.23763428919342208, 0.10877079398120119, -0.29526503169537649], [-0.18829296442773819, -0.29468889988561175, 
-0.24352128276175403, -0.24004356092093421], [-0.26605086505066022, 0.07083179290714435, -0.028286609435257626, 
0.25090613579720472]], [[0.084601772955133644, -0.24039147727297611, 0.14239141457752291, -0.18097717011950412], 
[-0.29881664590797025, -0.11232155481005192, 0.17644344916330479, 0.225953484534754], [-0.038184791282996997, 
0.14853681895363158, -0.1134777997472311, -0.054105519698704661]]]])+(1.-msk_ref)*numpy.array([[[[-0.075775826492745932, 
-0.009456362022272409, -0.080685742345592323, -0.079766291720105256], [0.12498070081298421, -0.084285370102877968, 
0.010414788494386745, 0.066487263233719507], [-0.0026034593874927556, 0.24369987386791345, 0.16377405851146176, 
-0.11092876321494609]], [[0.056776458857157167, 0.24376037059066927, 0.03226717257523308, -0.090830311097463659], 
[-0.17814805703521466, -0.28524569467359262, 0.24228809454161065, 0.072090505998011312], [0.16625208092584495, 
0.036762460557632756, 0.22920614418401311, -0.068631732133799569]]], [[[-0.24316331123317722, -0.13299759964586186, 
-0.26898768669576834, 0.29049424903651661], [-0.2170032557172539, -0.014991939273644384, 0.18365022914138368, 
-0.046324534511052534], [-0.28136350504384455, -0.23919467847070952, 0.10878960585009977, -0.17626109017155014]], 
[[0.27904970867621398, 0.18559910627123855, 0.078909684820260756, -0.23190087986361196], [0.25812932039162706, 
0.23735147566237183, -0.25178880160798389, -0.22722587527712226], [-0.15277837532988947, 0.091074724370164006, 
0.1822312374146508, 0.041894603804057927]]], [[[0.05670659996953898, -0.30220355862540099, -0.21484822412394811, 
-0.27588452313118017], [0.065788613265257867, -0.1401936657590519, -0.24353719112707614, 0.016313900927415532], 
[0.13765217744561678, 0.1020637157582809, 0.022259283627229495, 0.18260958319954562]], [[-0.099599468729128809, 
-0.2503500820476085, 0.04142442487325633, -0.21525409315411986], [-0.0032129584067909834, -0.14789374747609313, 
-0.11354428194759202, 0.093819550694657403], [-0.26128573121651361, -0.054987892273438095, 0.29167963923948115, 
-0.25887714141917151]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_float_rank0(self):
      arg0=Data(0.855868237407,self.functionspace)
      arg1=-0.331240406713
      res=outer(arg0,arg1)
      ref=Data(-0.28349814305116411,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_float_rank0(self):
      arg0=Data(numpy.array([0.76708082216518658, 0.24900288446126861]),self.functionspace)
      arg1=0.0458016595743
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.035133574682782366, 0.011404745347113238]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_float_rank0(self):
      arg0=Data(numpy.array([[0.27130062800907284, 0.44990188647787566, 0.088311580580226368, -0.99526858922829242, 
0.85698394752482043], [-0.62451037819636279, -0.939987131220962, 0.64573112346806516, -0.63920340736457271, 
-0.56916567178378163], [0.74097375520088504, -0.5438764905274347, 0.63856368133120744, -0.16653574269539528, 
0.63460309111640489], [0.34795479077961011, -0.21677917281420012, -0.66518992587141823, -0.1291185912908297, 
-0.89585459262077771]]),self.functionspace)
      arg1=0.961957356457
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.26097963492462783, 0.4327864293811024, 0.084951974599460092, -0.95740594105835508, 
0.82438201268673272], [-0.60075235248949543, -0.904227535852556, 0.62116580451310155, -0.61488641998648663, 
-0.54751310501498129], [0.71278515475677606, -0.52318599106667352, 0.61427103082257406, -0.16020028279880186, 
0.61046111192953467], [0.3347176707047691, -0.20853232001520014, -0.63988434263284266, -0.12420657874752894, 
-0.86177391568700312]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank3_float_rank0(self):
      arg0=Data(numpy.array([[[0.57447118168186195, -0.9350773708453195], [0.77222865652792017, -0.78557350460581077]], 
[[-0.23304322070648698, -0.23819108901843822], [0.30653294545689524, 0.79724898528053201]], [[-0.88280206793438487, 
0.40332879737058502], [0.77270626667504283, 0.18441274328652613]], [[-0.76004623321329823, 0.1704480317704693], 
[-0.51375334514695004, -0.037200815454657032]], [[0.43916408941534923, 0.85295616833129495], [-0.30158632262856933, 
0.53164939486293217]], [[0.072519001758186263, 0.1379034515343005], [-0.92697335821079752, 
0.1044833237344418]]]),self.functionspace)
      arg1=0.835880092036
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.4801890242164556, -0.78161255880327773], [0.64549056049165143, -0.6566452533312096]], 
[[-0.19479618877258173, -0.19909918941096647], [0.25622478666067777, 0.66640455519216402]], [[-0.73791667379485848, 
0.33713451226702801], [0.64588978530538588, 0.15414694083101407]], [[-0.63530751537019969, 0.14247411648371164], 
[-0.4294361934254064, -0.031095421046065351]], [[0.3670885194795544, 0.71296908048772056], [-0.25209000311566765, 
0.44439514510908834]], [[0.060617189864015708, 0.11527074976061906], [-0.77483857597646988, 
0.087335530259407157]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank4_float_rank0(self):
      arg0=Data(numpy.array([[[[-0.99003147993777652, 0.99248118182244327, -0.28393801264996088, -0.25782938592976867], 
[-0.25796348811729675, -0.9366938080443088, -0.41605694953379224, 0.5317307438090042], [0.25368074706940624, 
0.054844991120822861, -0.26663260028699876, -0.42921257694890991]], [[0.17425190117495082, -0.020499236232457418, 
0.1047698276591571, -0.64415325405356194], [0.78001969337194033, -0.6623207724498501, -0.83880153563372839, 
0.2428790987632552], [0.43639348261735389, 0.74734546870049678, 0.53186214770861451, 0.032918407378838221]]], 
[[[0.87184649475290432, -0.070427396650536878, 0.08083953652836029, -0.93929653796714008], [-0.42554636590081141, 
0.58060523059381164, -0.23351855337671146, -0.18988344428505166], [-0.84737202935955835, -0.27240603315182432, 
-0.29324807086273341, -0.1206112357169844]], [[-0.72707031300431013, -0.12170258962411951, 0.91097572803827753, 
-0.0017605112124874367], [0.75740702212793365, -0.20946431282305711, 0.65275147924456922, 0.046018209195791693], 
[-0.16409908608468871, 0.4434084280382713, -0.11952060793439201, -0.65385351680468551]]], [[[0.68236855318799572, 
0.26826896833718883, -0.9998479914754328, 0.87396403269938672], [0.6901199070263444, 0.99922382186703151, -0.11227870700938802, 
0.77890290565720632], [-0.54285727890099156, -0.72884280010118552, -0.92003470771768225, -0.69492614208004144]], 
[[-0.88632998203428381, 0.013555649733227204, -0.30928316046881354, 0.23624644768272107], [-0.94817677989959215, 
-0.62591771484007364, 0.72923780693280738, -0.51992672063312817], [-0.9931941841403773, 0.37974651536759785, 
-0.51815249801128216, -0.47193207425752437]]]]),self.functionspace)
      arg1=-0.874147612629
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.86543365461470378, -0.86757505566883686, 0.2482037358924544, 0.22538094217598878], 
[0.22549816728306471, 0.81880865606586484, 0.36369518915247584, -0.46481116026183678], [-0.22175441942054497, 
-0.047942618052900565, 0.23307625098981871, 0.37519514945003146]], [[-0.15232188340806688, 0.017919358413311041, 
-0.091584294723755344, 0.56308502919782411], [-0.68185235276432443, 0.57896612203132436, 0.73323635984337365, 
-0.21231218434126997], [-0.38147232099661249, -0.6532902572732936, -0.46492602666697119, -0.02877554722174502]]], 
[[[-0.7621225319668089, 0.061563940645709775, -0.070665687862263488, 0.8210838262142246], [0.37199033981494378, 
-0.50753467620322112, 0.20412968593872172, 0.16598615949946155], [0.74072823647285579, 0.23812308354527709, 
0.25634210105258221, 0.10543202375817955]], [[0.63556677832580022, 0.10638602817063451, -0.79632725782720293, 
0.0015389466734016624], [-0.66208554018122201, 0.18310272898515223, -0.57060114722138566, -0.040226707705941854], 
[0.14344682433545528, -0.38760441878902691, 0.10447865408576004, 0.57156449072358828]]], [[[-0.59649084170207478, 
-0.23450667821427396, 0.87401473473968649, -0.76397357270737676], [-0.60326666915450633, -0.87346911836662722, 
0.098148163681275422, -0.68087611544967686], [0.47453739434932518, 0.63711619368994876, 0.80424614328680499, 
0.60746802805242717]], [[0.77478323779636249, -0.011849638851929214, 0.27035913635002179, -0.20651426823382299], 
[0.82884646849904142, 0.54714447612935924, -0.63746148796878288, 0.45449270158323285], [0.86819832494285754, 
-0.33195450981259139, 0.45294176911407463, 0.41253829603504794]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_array_rank0(self):
      arg0=Data(0.72749091269,self.functionspace)
      arg1=numpy.array(0.544429243769)
      res=outer(arg0,arg1)
      ref=Data(0.39606732744475071,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_array_rank0(self):
      arg0=Data(numpy.array([-0.71865788045453294, -0.23655700550904335]),self.functionspace)
      arg1=numpy.array(-0.0695063340636)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.049951274716309058, 0.016442210249995634]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_array_rank0(self):
      arg0=Data(numpy.array([[0.9481360650271351, 0.20080183704454591, -0.34235850408590363, -0.21548109592585418, 
-0.3378140780018144], [-0.44688833119774674, -0.64497216507845367, -0.6086273707942087, 0.14685104020157436, 
0.57295381791887801], [0.064227911010919403, -0.91053622229373166, -0.75875775638735554, -0.13519195091034986, 
0.1532033919276008], [-0.78483268225608693, -0.89080247571077642, 0.73468888286281442, -0.32300084310035859, 
-0.66646383399504394]]),self.functionspace)
      arg1=numpy.array(0.18612714645)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.17647386022947992, 0.037374672930944604, -0.063722211428272668, -0.040106881498517877, 
-0.062876370368989881], [-0.083178049867473028, -0.12004682862549122, -0.11328207577706595, 0.027332965065878692, 
0.1066422591766607], [0.011954557798883093, -0.16947550879455675, -0.14122541604290076, -0.025162892045901657, 
0.028515310165888364], [-0.14607866758873334, -0.16580252285431119, 0.13674554529552091, -0.060119225227093877, 
-0.12404701163337635]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank3_array_rank0(self):
      arg0=Data(numpy.array([[[-0.26866259702851858, 0.40120836097358192], [-0.25200668304318952, -0.11455848141269542]], 
[[0.94398866644768331, 0.82628729899896092], [-0.24330240460710928, -0.094657592656990586]], [[-0.44847718163489403, 
-0.23055820282066608], [-0.26922667564184399, 0.77233417483910416]], [[-0.55642929541911657, 0.0075577282469232454], 
[0.084036871591784301, -0.80233885576629538]], [[-0.16845827560251547, -0.4472604377306062], [-0.13739162918347314, 
-0.99320600213394195]], [[-0.58138064046843674, -0.11018358249045401], [0.063922835096953401, 
-0.54131725355090565]]]),self.functionspace)
      arg1=numpy.array(-0.0182349338312)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.0048990446797230677, -0.0073160079148620613], [0.0045953251903030727, 0.0020889663283588564]], 
[[-0.017213570870039975, -0.015067294222775286], [0.0044366032489731432, 0.0017260749387172708]], [[0.0081779517318981317, 
0.0042042135726663796], [0.0049093306159126299, -0.014083462573735854]], [[0.010146451383687458, -0.00013781467449654628], 
[-0.0015324067928540101, 0.014630595945068301]], [[0.0030718255089234531, 0.0081557644873139733], [0.002505327267116123, 
0.018111045729624971]], [[0.010601437509660291, 0.0020091903359937642], [-0.0011656286682931991, 
0.0098708843001668878]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank4_array_rank0(self):
      arg0=Data(numpy.array([[[[0.95462962342599722, -0.94724113177473135, 0.49513578884703269, 0.53035952876880899], 
[-0.61444141225101778, -0.25095323015371473, -0.30817817857191754, 0.40591894392835703], [-0.12576247497385351, 
-0.84457431957082418, 0.51631066561800831, -0.88990809350170674]], [[0.82039700166326313, -0.67660864343626925, 
-0.051274119820798436, 0.12256730192970489], [-0.093676451895046586, -0.014472530095783753, -0.29970090215351486, 
-0.054009917955756048], [-0.76514661163061826, -0.012158493375796775, 0.69391579183245367, 0.69139942245021779]]], 
[[[-0.30363471850512314, -0.69687459624795034, 0.24579045612403783, 0.5283226969362016], [0.072033061111246743, 
0.24904040981223075, -0.43822789190142641, -0.78423278697539733], [-0.70651970260583874, -0.11576991731667685, 
-0.196095882982803, -0.49017897450254355]], [[0.22539170133650122, 0.85382114442168566, 0.41170067506447894, 
-0.29281346939592434], [-0.61023932590586205, -0.85662278469281516, 0.47503142183387981, -0.24738801228027674], 
[0.44543053294663659, -0.54165686048766837, 0.046479281819532092, 0.72957276124480908]]], [[[-0.37555395269063441, 
-0.74769587040326813, 0.16503520916337711, -0.17115445642731308], [0.90114286973352353, 0.46163510304640254, 
-0.87719837601530304, 0.29280571120417842], [0.14051749500984201, -0.67883102537652262, 0.30867803228715318, 
-0.95551022441636113]], [[0.15049183021264345, -0.53128173055033634, -0.69946290860329308, -0.75378481192125379], 
[-0.51780039536351685, 0.38122248695605632, -0.77441203786786561, 0.55202816275395561], [-0.45762426631314512, 
0.96921999872840758, 0.52912481251169008, -0.73547276091528513]]]]),self.functionspace)
      arg1=numpy.array(0.0681689406737)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.065076090164669864, -0.064572424515626697, 0.033752882215332007, 0.036154047252364915], 
[-0.041885820179195378, -0.017107215858218408, -0.021008179971993609, 0.027671064406977381], [-0.0085730946954685194, 
-0.057573736685342103, 0.035196351133705256, -0.060664092030950717]], [[0.055925594535252726, -0.046123694473710092, 
-0.003495302432159468, 0.0083552831337797926], [-0.0063858244917548166, -0.00098657704549761542, -0.020430293018753081, 
-0.0036817988929165736], [-0.052159233974919353, -0.00082883161361609265, 0.047303504445960265, 0.047131966210829553]]], 
[[[-0.020698457112247038, -0.047505203008625298, 0.016755275021677719, 0.036015198584005634], [0.0049104174694365663, 
0.016976820921840368, -0.029873531164582749, -0.053460318329685168], [-0.048162699691727581, -0.0078919126253580597, 
-0.013367648613408738, -0.033414981432352053]], [[0.015364713516749069, 0.058204082940020441, 0.028065198893786877, 
-0.019960784023706889], [-0.041599368404426747, -0.058395067789452075, 0.032382388813130387, -0.01686417873251525], 
[0.030364527574687536, -0.036924174388078795, 0.0031684434049112082, 0.049734202278434551]]], [[[-0.025601115120736075, 
-0.050969635431480281, 0.011250275382527588, -0.011667417986230454], [0.061429954825379572, 0.031469175952461063, 
-0.059797684053640755, 0.01996025515599403], [0.0095789287809408662, -0.046275191896349495, 0.021042254470253027, 
-0.065136119801339168]], [[0.010258868645640094, -0.036216912770899022, -0.047681645520021625, -0.05138471212458539], 
[-0.035297904432346656, 0.025987533096782377, -0.052790848266402686, 0.037631175076978191], [-0.031195761461139799, 
0.066070700593066689, 0.036069877953084542, -0.050136399005945999]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_constData_rank0(self):
      arg0=Data(-0.761549634663,self.functionspace)
      arg1=Data(0.0997529748633,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(-0.075966841563735246,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_constData_rank0(self):
      arg0=Data(numpy.array([0.55868192127741945, 0.65764903734185398]),self.functionspace)
      arg1=Data(-0.995214717967,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([-0.55600847071728854, -0.65450200121934787]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_constData_rank0(self):
      arg0=Data(numpy.array([[0.28355419590890651, -0.64569032712320906, 0.50141124723693098, -0.80757327909511289, 
0.55012578249109922], [0.73848320842017556, 0.30326193928008305, 0.4209398421529722, -0.31301100573218643, 
-0.56419991074326048], [-0.64831653760568275, 0.50941318603805041, -0.53963934101870326, -0.050527384769094574, 
0.97712479575240763], [-0.30360813532673103, 0.91177458447138848, -0.7012270814638073, 0.84659160172487069, 
-0.012813769261334373]]),self.functionspace)
      arg1=Data(-0.912549180814,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.25875714919310722, 0.58922417907594948, -0.4575624229170972, 0.73695033428569789, 
-0.50201683215703385], [-0.67390224688888734, -0.27674143426217118, -0.38412830812876142, 0.28563793686674155, 
0.51486016636421783], [0.59162072530037657, -0.46486458561498423, 0.49244743858174261, 0.046108723579722101, 
-0.89167443191712059], [0.27705735518094093, -0.83203915014659047, 0.63990419875454296, -0.77255647263822824, 
0.011693194642573103]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank3_constData_rank0(self):
      arg0=Data(numpy.array([[[-0.84881460202203995, 0.54149795397130895], [0.6989075063336776, -0.76126229191446915]], 
[[-0.64412555419430562, 0.73975769511094036], [0.18952045967857267, 0.44246537421386645]], [[-0.14357708781339995, 
-0.20689271372893048], [0.22560575090442936, 0.3430078217353314]], [[0.67664786945362243, 0.10124599613681817], 
[-0.17759393001224488, -0.95180121682446628]], [[0.001928327449682854, 0.37269203891262626], [-0.43934319126823351, 
0.7467473182371025]], [[0.017458344573415996, -0.44215311866986218], [-0.65332164942710658, 
0.92129804824183625]]]),self.functionspace)
      arg1=Data(-0.901088980022,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.7648574839634189, -0.4879378390277509], [-0.62977785201160286, 0.68596506215005937]], 
[[0.58041443863474806, -0.66658750695059343], [-0.17077479770497611, -0.39870067274521642]], [[0.12937573161223787, 
0.18642874438788787], [-0.20329085596946309, -0.30908056822689572]], [[-0.60971993851970441, -0.091231651390189017], 
[0.16002793325274831, 0.85765758765160716]], [[-0.0017375946147822365, -0.33582868920592185], [0.39588730809929545, 
-0.67288577932408178]], [[-0.015731521904523791, 0.39841930271556358], [0.58870093870825346, 
-0.83017151858606031]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank4_constData_rank0(self):
      arg0=Data(numpy.array([[[[0.36506979897896308, 0.61233242903390339, -0.24865941965977933, 0.57815860382927831], 
[-0.73624130099188112, 0.13597465903021155, -0.99478808372728134, 0.91700497409411619], [-0.49724567921575669, 
-0.54691125683738284, 0.9507305292564705, -0.43122965813968106]], [[0.08995318683066178, 0.76455311872055609, 
0.071910852797488545, -0.35672973572366673], [-0.26623785199264605, 0.97237948150930698, -0.91926047313140868, 
0.32892783516690316], [-0.087694443509900077, 0.60849775878089041, 0.72436347181268745, -0.83397913799482115]]], 
[[[0.53469221329579275, -0.24350908703846508, 0.90393177678133441, -0.3445113911551021], [-0.38976994345460647, 
-0.34763449585953965, -0.93882937967617619, 0.46110064961193653], [0.77347702173187827, 0.90339541842388771, 
0.075137977932457023, -0.8468468433600711]], [[0.85846586709643158, 0.29065862494062333, 0.12332472946607553, 
0.57985673762797574], [-0.83233995096314151, -0.15708811684665114, 0.963272973347304, 0.31822269071413434], 
[-0.19626619878721407, 0.87629225061940486, -0.20561672906490625, -0.98671240041738906]]], [[[-0.36810699494738608, 
0.20825338962819551, -0.43615912691127967, -0.22491076709170787], [0.85985869915219282, 0.39753233629467633, 
0.36508767523776808, 0.035424381439172814], [-0.10016681257513804, 0.77433650969734646, -0.94187407357980368, 
0.93922055606159649]], [[0.46394552974372494, 0.28410240599782344, -0.36314219052725183, 0.52252472849970011], 
[0.80853947102134094, -0.10437974451611942, -0.84506815967908877, 0.64333966413214849], [-0.89726693506002442, 
0.6859164881500488, 0.18387818447739512, -0.092837085709315481]]]]),self.functionspace)
      arg1=Data(-0.625847677008,self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.2284780856367617, -0.38322682826753468, 0.15562292016023038, -0.36183921914874262], 
[0.46077490794311682, -0.08509942448601282, 0.62258581131595925, -0.57390543284158413], [0.31120005323944666, 
0.34228313962120194, -0.59501249319574878, 0.26988407980367352]], [[-0.056296993017436327, -0.47849379330048197, 
-0.045005240174972469, 0.22325847642233471], [0.16662434120119735, -0.6085614396728436, 0.57531703167456749, 
-0.20585872154247678], [0.054883363757180281, -0.38082690879759484, -0.45334119614342061, 0.52194390618719344]]], 
[[[-0.33463587960543817, 0.15239959645336235, -0.56572360267231248, 0.2156116538572152], [0.24393661367860511, 
0.21756624168154021, 0.58756418637719687, -0.28857877042651042], [-0.48407879726996261, -0.56538792404026061, 
-0.04702492894410662, 0.52999712969845847]], [[-0.53726886871296053, -0.18190802522142877, -0.077182495453983466, 
-0.36290199224190617], [0.52091802479123506, 0.098313233014037937, -0.60286215269399968, -0.19915893175467633], 
[0.12283274458616834, -0.54842546943026704, 0.12868475223925496, 0.61753166367621082]]], [[[0.23037890767821731, 
-0.13033490012784815, 0.27296917638326201, 0.14075988111843279], [-0.53814056941952093, -0.24879468920558637, 
-0.22848927345180842, -0.022170266833151632], [0.062689166963445897, -0.48461670581656724, 0.58946970098398255, 
-0.58780900320931251]], [[-0.2903592320483564, -0.17780483082612158, 0.22727169636507724, -0.32702088751077341], 
[-0.50602254970798366, 0.065325820632101905, 0.52888394464858368, -0.40263263432421231], [0.56155242696340457, 
-0.42927924073019363, -0.11507973460762629, 0.058101874431367724]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_expandedData_rank0(self):
      arg0=Data(-0.642877126562,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.547089422365)+(1.-msk_arg1)*(0.222624469009)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(-0.35171127582219308)+(1.-msk_ref)*numpy.array(-0.14312017893859127)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_expandedData_rank0(self):
      arg0=Data(numpy.array([-0.3829468436692347, -0.26589916801218894]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.230991317554)+(1.-msk_arg1)*(-0.606772663729)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.088457395972133118, 0.06142039915553054])+(1.-msk_ref)*numpy.array([0.23236167639977173, 
0.161340346458065])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_expandedData_rank0(self):
      arg0=Data(numpy.array([[-0.87174569179957362, -0.56872338095146957, 0.26797673469459404, -0.48582324665863297, 
-0.94500956212991372], [-0.35643086869262919, -0.44329490689890672, 0.81325997657563187, -0.71848389647741118, 
-0.73924415246545094], [0.75438515837096842, 0.27031972889573064, 0.84218967617015683, -0.039217444296754778, 
-0.94152621651398638], [-0.14387273018259394, -0.68139020051338228, 0.90325594873061155, -0.98700102624342323, 
-0.37376869960732795]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.0527910607114)+(1.-msk_arg1)*(-0.405654453504)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.046020379740734634, 0.03002351053182907, -0.01414677607051794, 0.025647124509388729, 
0.049888057167299285], [0.018816363628586782, 0.023402008343175927, -0.042932856797595058, 0.03792952699913682, 
0.039025482933386632], [-0.039824792695377205, -0.014270465219636713, -0.044460086325253584, 0.0020703304828178165, 
0.049704167657409905], [0.0075951940337911118, 0.035971311443487773, -0.04768383962741439, 0.05210483109867816, 
0.019731646113009465]])+(1.-msk_ref)*numpy.array([[0.35362752220125299, 0.23070517229470502, -0.10870595586426984, 
0.1970763636227523, 0.38334733748168076], [0.14458776925139519, 0.17982455319909638, -0.32990253135430569, 0.29145619237682907, 
0.29987768267425641], [-0.30601969915035704, -0.10965640189649449, -0.34163799283335194, 0.015908730934015996, 
0.38193430281948687], [0.058362613736320623, 0.27640896941210458, -0.36640979825637765, 0.40038136190847112, 
0.1516209375760387]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank3_expandedData_rank0(self):
      arg0=Data(numpy.array([[[-0.77328627374203962, 0.079523600460909227], [-0.66474016911635925, -0.43502454702821791]], 
[[-0.0082597579492271755, 0.35076593869127404], [-0.44497667487629888, 0.1174660507470533]], [[-0.94247002006062308, 
-0.3077056658142272], [0.84316414408989693, -0.49022494198661248]], [[-0.41017795822495784, 0.50503730298836658], 
[0.83082960996619759, 0.8780217163336772]], [[-0.82118917871286246, 0.10197202621331503], [0.93307721066846172, 
0.4777386015723819]], [[0.69942467243314588, -0.83580669402499019], [-0.72490505988022047, 
0.23436457019346668]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.871700854773)+(1.-msk_arg1)*(0.012858077492)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.6740743058054266, -0.069320790496428114], [0.57945457362090469, 0.37921126949188244]], 
[[0.0072000380645622156, -0.30576296858255642], [0.38788654784386817, -0.10239525684305484]], [[0.82155192208509165, 
0.26822729190886141], [-0.7349869051173944, 0.42732950096094091]], [[0.35755247679387892, -0.44024144870738124], 
[-0.72423488117853474, -0.76537228063762108]], [[0.71583130901461889, -0.088889102413116095], [-0.81336420210922156, 
-0.41644514734886523]], [[-0.60968908480953632, 0.72857340960686301], [0.63190036032710706, 
-0.20429579616623125]]])+(1.-msk_ref)*numpy.array([[[-0.009942974831298395, 0.0010225206171716164], [-0.008547280606563382, 
-0.0055935793366241457]], [[-0.00010620460777657516, 0.0045101756212571021], [-0.0057215445677053622, 0.0015103875831883589]], 
[[-0.012118352551854714, -0.0039565032957760695], [0.01084146990320918, -0.0063033502925898544]], [[-0.0052740999723792188, 
0.0064938087781903336], [0.010682871507618564, 0.011289671268303747]], [[-0.010558914095506591, 0.0013111642150701356], 
[0.011997579080822433, 0.006142799959951809]], [[0.0089932566379832025, -0.010746867240130867], [-0.0093208854343046401, 
0.0030134778049339369]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg1.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1) 
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank4_expandedData_rank0(self):
      arg0=Data(numpy.array([[[[-0.83504325403320423, 0.72986446282641149, 0.86742233864227591, -0.0059036297732719678], 
[0.29820188050100183, -0.17552230466048235, 0.76126162856103852, 0.46771314116943086], [-0.26052995472384133, 
-0.34668561183634683, 0.96178401739697628, -0.80594576244112615]], [[0.57742472247806353, -0.77451120347821623, 
0.29491164193254282, -0.62602578123637809], [0.14785524502051084, 0.78988617504439018, -0.12588877364162165, 
0.2155924612104001], [0.8585041497903676, 0.54583569168921997, 0.80556935761893178, -0.46444010468878583]]], 
[[[0.28107157933058624, 0.047773080942044244, -0.83081398509678506, 0.055612189618716279], [0.38529122516130831, 
0.049472585315569528, -0.54818234037853664, -0.016806731189929369], [-0.28894913844719183, 0.99248029239537039, 
0.058901579190173026, -0.54663411835204334]], [[0.45320846987436503, -0.98170747061466113, 0.24373236322199898, 
-0.46104918112498638], [0.68019717040089955, -0.47457064947511274, -0.72411938225628325, -0.72698055420509688], 
[-0.67048338311954292, -0.41460665211487036, -0.4987048196003625, -0.47176296727437927]]], [[[0.1963630166123862, 
-0.31000614317997255, 0.91699734932975696, 0.23826455365961907], [-0.4962649474436871, 0.36154518492720933, 
-0.41701367639334053, -0.6258638542675814], [0.5472178354039805, 0.61426522480721935, -0.16313628065434438, 
0.5905860497489801]], [[0.054834609730624484, 0.78207162444878109, 0.66991726836949272, 0.68954181735556452], 
[-0.0052733385953620537, 0.25102504584418339, 0.58525398946693841, -0.11076231426807115], [-0.82109331075278047, 
-0.72719539932325872, -0.75213299670452183, 0.5381412754258712]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.767088501143)+(1.-msk_arg1)*(0.319392052905)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.64055207812595427, 0.5598706368270967, 0.66538970160711142, -0.0045286065140827374], 
[0.22874723355155543, -0.13464114159918497, 0.58395504163061207, 0.35877737242457119], [-0.19984953247198076, 
-0.26593854635140807, 0.73777346032839741, -0.61823172691356432]], [[0.44293586488864523, -0.59411863819461253, 
0.22622332937967313, -0.48021717820552656], [0.11341805828892357, 0.60591260208842612, -0.096567830683489536, 
0.16537849792762915], [0.65854866148779023, 0.41870428260826925, 0.61794299110268414, -0.35626666377644667]]], 
[[[0.21560677650261212, 0.036646181054818809, -0.63730785455658556, 0.04265947117990472], [0.29555246841256128, 
0.037949851317392397, -0.42050436983406647, -0.01289225023759724], [-0.22164956151803497, 0.7613202199075908, 
0.045182724095949089, -0.41931674652032708]], [[0.3476510058612664, -0.75305651219474512, 0.18696429318401911, 
-0.35366552530240108], [0.52177142792457676, -0.36403768819235283, -0.555463651583611, -0.55765842368533891], 
[-0.51432009339849838, -0.31803999533473809, -0.38255073258006228, -0.36188394746130614]]], [[[0.15062781209312515, 
-0.23780214771706604, 0.70341812224952238, 0.18276999934227742], [-0.38067913470441744, 0.27733715400130349, 
-0.31988639598072471, -0.48009296588973766], [0.41976450915878921, 0.47119579060167477, -0.12513996500919464, 
0.4530317676979459]], [[0.042062998589029399, 0.59991815018493377, 0.51388583328340731, 0.52893959915074173], 
[-0.0040451173991361278, 0.19255842616598259, 0.44894160556819007, -0.084964497635031278], [-0.6298512370439433, 
-0.55782322890500768, -0.57695257310230985, 0.41280198436964627]]]])+(1.-msk_ref)*numpy.array([[[[-0.26670617917053724, 
0.23311290912488289, 0.27704780147502889, -0.0018855724328792466], [0.095243310793489563, -0.056060429216212543, 
0.24314091434427901, 0.14938386032897505], [-0.083211197082619412, -0.11072862927720316, 0.30718617176810004, 
-0.25741267159654352]], [[0.18442486751064571, -0.24737272327720133, 0.094192434742560657, -0.19994765944084364], 
[0.047223790239943692, 0.25228336700908505, -0.040207873851150808, 0.068858518776934732], [0.27419940282941913, 
0.17433558211770256, 0.25729245088765929, -0.14833847848818729]]], [[[0.089772028735781384, 0.015258342395699169, 
-0.26535538428264493, 0.01776209140889061], [0.12305895537073776, 0.015801150586481249, -0.17508508306003137, 
-0.0053679363773820998], [-0.092288058513918331, 0.31699031805638828, 0.018812696296924055, -0.17459059324863618]], 
[[0.14475118358732478, -0.31354956439226267, 0.077846179848978347, -0.14725544444989982], [0.21724957063484182, 
-0.15157409398454316, -0.23127797604748224, -0.23219181162992941], [-0.21414706417356219, -0.13242206976723658, 
-0.15928235612601679, -0.15067734260254473]]], [[[0.06271678699054288, -0.099013498483561571, 0.29288166591131454, 
0.07609980492795361], [-0.15850308034906932, 0.11547465883199276, -0.1331908541929305, -0.19989594125385893], 
[0.17477702783617033, 0.1961914311796239, -0.052104431581555574, 0.18862849084666464]], [[0.017513738572134958, 
0.2497874616518197, 0.2139662516213636, 0.22023417660936923], [-0.0016842624396383862, 0.080175404722865939, 
0.18692547316696756, -0.035376602938641172], [-0.26225067814828773, -0.23226043145327582, -0.24022530187540778, 
0.17187804671144227]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_array_rank1(self):
      arg0=Data(0.094765405644,self.functionspace)
      arg1=numpy.array([0.43204953548936986, -0.029784398113217314])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.040943349488930969, -0.0028225305690599833]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_array_rank1(self):
      arg0=Data(numpy.array([-0.77657105275802074, 0.22498494171069083]),self.functionspace)
      arg1=numpy.array([0.22091855299981256, -0.068451744610724008])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.17155895327684303, 0.053157643375473115], [0.049703347769472994, 
-0.015400611771238836]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_array_rank1(self):
      arg0=Data(numpy.array([[-0.19358546521433406, 0.53808130011597055, -0.33754340271808458, -0.26681469630310262, 
-0.027436007634214166], [0.48475778158319649, -0.51694798347645632, -0.36535005917852104, -0.3145900258061618, 
-0.45174258519432553], [-0.14077024477916988, 0.20624720031697841, -0.39314518635241624, -0.82591162366689486, 
-0.8368370323312504], [-0.16007342247871792, 0.84886675840320991, -0.98705708408575532, 0.56621771577631375, 
-0.89012344094023943]]),self.functionspace)
      arg1=numpy.array([0.093504543477614499, -0.20572310259722482])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.018101120548767929, 0.03982500252161994], [0.050313046321185106, -0.11069575450940593], 
[-0.031561841775035078, 0.069440476068388893], [-0.024948386370939966, 0.054889947142010558], [-0.0025653913686855419, 
0.0056442206133916839]], [[0.045327055064157948, -0.099725874835443026], [-0.048336985196639454, 0.10634814304215551], 
[-0.034161890493007052, 0.075160947708285047], [-0.029415596745616123, 0.064718436154984627], [-0.042239984197992783, 
0.092933886201467808]], [[-0.013162657473308326, 0.028959691509341617], [0.019285050309175173, -0.042429813951200129], 
[-0.03676086117030436, 0.080879047507583193], [-0.077226489323828348, 0.16990910169186516], [-0.078248064673295287, 
0.17215671065943897]], [[-0.014967592291771834, 0.032930801115678199], [0.079372898717814624, -0.1746315032303572], 
[-0.092294322033783904, 0.2030604457786914], [0.0529439290226019, -0.11648406523501688], [-0.083230585983840435, 
0.18311895596474367]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank3_array_rank1(self):
      arg0=Data(numpy.array([[[0.52793389561713155, 0.15497295848870052], [-0.93668753153367779, 0.10326774528137728]], 
[[-0.67071187831479984, -0.60665708959908526], [0.47550232291478522, 0.86657992322827249]], [[-0.48175286497054448, 
-0.85598675032131721], [0.16819608414084075, 0.72547085781731191]], [[0.078203982287778251, 0.094239263802861961], 
[-0.63867238064066001, 0.47206552186393114]], [[-0.26618052116044821, 0.49845726599722706], [-0.10756974178355838, 
0.8587717008057989]], [[0.29948403844539584, 0.87726990965275875], [-0.26732263279729618, 
0.59819481827892873]]]),self.functionspace)
      arg1=numpy.array([0.44042322360500719, -0.49121081330391614])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.23251434815804645, -0.25932683823679598], [0.068253689949198448, -0.076124392979348615]], 
[[-0.4125389421486792, 0.46011104417629545], [0.045481513271244951, -0.050726233147726979]]], [[[-0.29539708755757343, 
0.32946092723961007], [-0.26718587102406083, 0.2979965223785534]], [[0.20942226588979879, -0.23357188276687302], 
[0.38166192329957543, -0.42567342888180493]]], [[[-0.21217514977127494, 0.23664221661367285], [-0.37699644393968895, 
0.42046994780271041]], [[0.074077461575048104, -0.082619735285356294], [0.31951421383139034, -0.35635913009673148]]], 
[[[0.034442849977932184, -0.038414641743184604], [0.041505160354219132, -0.046291345417766132]], [[-0.28128614870924368, 
0.31372277952924693], [0.20790861889209256, -0.23188368892751923]]], [[[-0.11723208319034543, 0.13075075028488403], 
[0.21953215591983727, -0.24484759902774436]], [[-0.047376212438673015, 0.052839420348393965], [0.37822300080964472, 
-0.42183794559520382]]], [[[0.13189972563036714, -0.14710979809630417], [0.3863700415809414, -0.43092446580758464]], 
[[-0.11773509567916281, 0.13131176787090398], [0.26345889021021729, -0.29383976320098087]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_constData_rank1(self):
      arg0=Data(-0.919108703748,self.functionspace)
      arg1=Data(numpy.array([-0.93688283577102016, 0.53170944224710404]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([0.86109716874912567, -0.4886987762342358]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_constData_rank1(self):
      arg0=Data(numpy.array([-0.29779080932637259, -0.053805000864227903]),self.functionspace)
      arg1=Data(numpy.array([0.5272408344369186, 0.62962043017047886]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.15700747479688201, -0.18749517746888575], [-0.028368193552534646, 
-0.033876727789458161]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_constData_rank1(self):
      arg0=Data(numpy.array([[-0.61169447974141988, 0.55875640915843894, 0.63173687966141756, -0.037640820319297719, 
-0.99090884499283916], [0.41618393390700192, 0.12956118624983004, -0.71693208691742694, 0.71158379483885348, 
-0.61635924994323665], [0.4413481405572679, 0.08995759548061022, 0.33046412157090854, 0.4466056588564935, 0.23129456001205373], 
[-0.64922128479527008, -0.03976505531208363, 0.086830616590950704, -0.25200669262905184, 
0.37855163655036073]]),self.functionspace)
      arg1=Data(numpy.array([0.7021467630050735, 0.76928149433839033]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.42949929889851046, -0.47056524345402373], [0.39232900399893633, 0.42984096540855693], 
[0.44357200512519002, 0.48598349081460718], [-0.026429380144050489, -0.028956386503352194], [-0.6957634379448181, 
-0.76228783702921965]], [[0.29222220200751886, 0.32016259859560831], [0.090970967536415606, 0.09966902296652376], 
[-0.50339154412354337, -0.55152258716297897], [0.49963625815296731, 0.5474082450406158], [-0.43277465219587863, 
-0.47415376484562249]], [[0.30989116825059387, 0.33952095709136498], [0.063163434474430291, 0.069202713478412253], 
[0.23203431325032853, 0.25421993326729203], [0.31358271770583507, 0.34356546862510468], [0.16240272661314625, 
0.1779306247584132]], [[-0.45584862359299383, -0.49943392012359505], [-0.027920904868097222, -0.030590521172928441], 
[0.060967836369070667, 0.066797186485410387], [-0.17694568348510326, -0.19386408508895242], [0.26579880623410884, 
0.29121276864970469]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank3_constData_rank1(self):
      arg0=Data(numpy.array([[[0.38252460007169198, -0.051755825978095027], [0.011862285227803859, 0.6098177254103021]], 
[[-0.83769475721431519, -0.45494251353843351], [0.037872621083969049, 0.33679832205419591]], [[0.80112411616460832, 
-0.0098861829681524682], [0.75418188513559348, -0.019244760229890856]], [[-0.091164744830903643, 0.137855274995063], 
[0.9289097954794594, 0.12542265606508685]], [[-0.22835257905635764, -0.75524782434503912], [-0.40829474194126947, 
-0.792401786421711]], [[-0.84502099168268718, 0.21593138093987796], [0.75161492138409214, 
0.14572990444308442]]]),self.functionspace)
      arg1=Data(numpy.array([-0.55350465181593345, -0.6950619290155835]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.21172914557371106, -0.26587828642174482], [0.02864709043745153, 0.035973504242129577]], 
[[-0.0065658300547568656, -0.0082450228529704107], [-0.33753694777441379, -0.42386108457158]]], [[[0.46366794491994245, 
0.58224973387562284], [0.25181279755235625, 0.31621322105122179]], [[-0.020962671946439068, -0.026323817067499786], 
[-0.18641943798079832, -0.23409569141620115]]], [[[-0.44342592497903893, -0.55683087356227712], [0.0054720482615758436, 
0.0068715094044450612]], [[-0.41744318173786099, -0.52420311591095481], [0.010652064310326861, 0.013376300168830322]]], 
[[[0.050460110345517742, 0.063365143400381338], [-0.076303535987132104, -0.095817953363042219]], [[-0.51415589291526809, 
-0.64564983432742418], [-0.069422023575135466, -0.087176513266857339]]], [[[0.12639421476185966, 0.15871918409459548], 
[0.41803318404884215, 0.5249440096740855]], [[0.22599303897647877, 0.28379013094061861], [0.4385980748916728, 
0.55076831422566885]]], [[[0.46772304977848056, 0.5873419205376299], [-0.11951902382326084, -0.15008568217107038]], 
[[-0.41602235536036208, -0.52241891713412325], [-0.080662180017938692, -0.10129130849746691]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_expandedData_rank1(self):
      arg0=Data(0.94577337941,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.21258989853496657, 0.94855972467563565])+(1.-msk_arg1)*numpy.array([0.092694441816000728, 
0.70626041960124941])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.20106186676583832, 0.89712253637866823])+(1.-msk_ref)*numpy.array([0.087667935488839999, 
0.66796230378977828])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_expandedData_rank1(self):
      arg0=Data(numpy.array([0.80743269872209877, -0.67335052968308773]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-0.8696787996472739, -0.85444435201596103])+(1.-msk_arg1)*numpy.array([0.50146272797380598, 
0.012381584608076812])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.70220710022059385, -0.6899063090561024], [0.58559868039664387, 
0.57534055701467002]])+(1.-msk_ref)*numpy.array([[0.40489740375643585, 0.009997296274555461], [-0.33766019349748838, 
-0.008337146554164487]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_expandedData_rank1(self):
      arg0=Data(numpy.array([[-0.55391791827152037, -0.3590000067754664, 0.024151024496113394, -0.54258909914201925, 
-0.59450570915289913], [-0.99285035958068391, -0.46489595794231819, -0.99762597209037041, 0.57921432631499536, 
-0.24793237890852571], [0.056552525067596671, -0.013315620681917606, -0.91314148739809764, -0.24033483034116565, 
-0.71124375711070598], [-0.63975316839291207, 0.60169935268198982, 0.20660245627981144, -0.72877339290849275, 
0.80719704598441488]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.54261649421559821, 0.91431185496179834])+(1.-msk_arg1)*numpy.array([0.25664152255339001, 
-0.95718588355394463])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.30056499889569466, -0.50645371935141159], [-0.19479932509987957, -0.32823796212617484], 
[0.013104744243796084, 0.022081568006269269], [-0.29441779477604213, -0.49609564571859072], [-0.32258860369170422, 
-0.54356361772096662]], [[-0.53873698139636672, -0.90777485396770363], [-0.2522602148736629, -0.42505988567048314], 
[-0.54132830751410499, -0.91214125310001382], [0.3142912471444923, 0.52958252511351178], [-0.13453219824587753, 
-0.2266875132649456]], [[0.030686332891219043, 0.051706644097327913], [-0.0072252754127268446, -0.012174629845651772], 
[-0.49548563261477258, -0.83489608718553021], [-0.1304096430776239, -0.21974098454116026], [-0.38593259401614172, 
-0.6502985988938883]], [[-0.34714062139668322, -0.58493390611101115], [0.32649199332409612, 0.55014085127998347], 
[0.11210590052288269, 0.18889907504085823], [-0.39544446353761303, -0.66632615271696749], [0.43799843123325022, 
0.73802982843369436]]])+(1.-msk_ref)*numpy.array([[[-0.14215833791480725, 0.53020241201708695], [-0.09213430833553303, 
0.34362973868124691], [0.0061981556979067605, -0.023117019721045257], [-0.13925089252468009, 0.51935862626899254], 
[-0.15257485036368287, 0.5690524724933822]], [[-0.25480662795046749, 0.95034234867208855], [-0.11931160647523331, 
0.44499184826367533], [-0.25603224841607841, 0.95491349755168409], [0.14865044659021648, -0.55441577670092168], 
[-0.063629743213368034, 0.23731737316718857]], [[0.014513726137586764, -0.054131278674034118], [-0.0034173411655507436, 
0.012745524147490483], [-0.23435002163251498, 0.87404614142491133], [-0.061679896781367427, 0.23004510692889601], 
[-0.1825346807314851, 0.68079248407223825]], [[-0.16418722719471227, 0.61236270174460505], [0.15442103799169507, 
-0.57593812653074694], [0.053022768942921003, -0.19775695465860654], [-0.18703351315243549, 0.69757160400172169], 
[0.207160278882039, -0.77263761766272621]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank3_expandedData_rank1(self):
      arg0=Data(numpy.array([[[-0.33561040691589383, 0.39295988096864809], [-0.74025647782746185, 0.24467768682694735]], 
[[-0.36609502502031499, -0.85371902689107326], [-0.87097730147411467, -0.71741878239906942]], [[-0.3037948221187714, 
-0.87597603983988059], [-0.01638418537184827, -0.62994381943706013]], [[-0.36032580215061172, -0.83457235834116617], 
[0.33584754764220115, -0.67422681043058752]], [[0.3773462048543037, 0.10333686426053723], [-0.22292593411219541, 
-0.46206698428249293]], [[0.26067951497054764, -0.11270196798769594], [0.27159358928923694, 
-0.66048112318849106]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.23855386295122938, 0.34303848989880215])+(1.-msk_arg1)*numpy.array([-0.87981738521624453, 
-0.73300661027850178])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.080061159016420455, -0.11512728718275073], [0.093742097589926285, 0.13480036415829807]], 
[[-0.17659104236041209, -0.25393646429173861], [0.05836880737053942, 0.08393386420104805]]], [[[-0.087333382425823114, 
-0.12558468454243304], [-0.20365797173983, -0.29285848578258861]], [[-0.20777499980948755, -0.29877873823381401], 
[-0.17114302189506544, -0.2461022557392141]]], [[[-0.0724714283610145, -0.10421331701869856], [-0.20896746815652353, 
-0.3004934978942056]], [[-0.003908510711763429, -0.005620406208180875], [-0.15027553156896242, -0.21609497654077278]]], 
[[[-0.085957112024028823, -0.12360561904132039], [-0.19909045999460284, -0.28629044151663563]], [[0.080117729852744138, 
0.11520863557939669], [-0.16083941013350289, -0.23128574689939468]]], [[[0.090017394837980089, 0.12944427228226438], 
[0.024651408154617992, 0.035448521866812188]], [[-0.053179842734475458, -0.076472175797127381], [-0.11022786404281368, 
-0.15850676052035992]]], [[[0.06218610528847697, 0.089423107163048854], [-0.026885489825670659, -0.038661112907122354]], 
[[0.064789699877737111, 0.093167054735975321], [-0.15756032334298134, 
-0.22657044710524468]]]])+(1.-msk_ref)*numpy.array([[[[0.29527587066410155, 0.246004646747608], [-0.34573293496872265, 
-0.28804219032427225]], [[0.65129051871154442, 0.54261289154901071], [-0.21527168264484398, -0.1793503618318055]]], 
[[[0.32209676765404915, 0.26835007332996441], [0.7511168419486608, 0.62578169003168671]], [[0.76630097196565627, 
0.63843211938305755], [0.63119751723537121, 0.52587270983647194]]], [[[0.26728396603877158, 0.22268361278144105], 
[0.77069894888400459, 0.64209622764821672]], [[0.014415091132757789, 0.012009716181593114], [0.55423552405024834, 
0.46175298375145202]]], [[[0.31702090507409708, 0.264121194830302], [0.73427127008947946, 0.61174705541979335]], 
[[-0.29548451119784952, -0.2461784724675575], [0.59319646939572812, 0.49421270887261098]]], [[[-0.33199575127618686, 
-0.2765972625217103], [-0.090917569710151858, -0.075746604588426053]], [[0.19613411244748058, 0.163406183306749], 
[0.40653456590617848, 0.33869815387051988]]], [[[-0.22935036924082611, -0.1910798076376051], [0.099157150783659539, 
0.082611287526377225]], [[-0.23895276156995107, -0.19907989625827519], [0.58110277478838646, 0.4841370292613334]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_array_rank2(self):
      arg0=Data(-0.53334445403,self.functionspace)
      arg1=numpy.array([[-0.14622035743864048, 0.50649767048635241, -0.024778745805814406, -0.014030242656037251, 
0.47192766812552156], [0.93282861324086097, 0.32755079828488132, -0.33403983401182868, -0.2189428902004753, 
0.20544718376209503], [0.009457548649367542, 0.38281810872422306, -0.0070076677237429497, 0.017701352393783898, 
0.67621360244218298], [0.6183268723103903, 0.30250428096384541, 0.31053406436293884, 0.97314794751525824, 0.69542049607892209]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[0.077985816706134276, -0.27013772353284116, 0.013215606653341953, 0.007482952109287914, 
-0.25170000449789953], [-0.49751896743219715, -0.17469740167823117, 0.17815829289519891, 0.11677197623764954, 
-0.10957411605552697], [-0.0050441311208559341, -0.20417391519018996, 0.0037375007161409937, -0.0094409181280493883, 
-0.36065477460195949], [-0.32978120812425593, -0.16133898057229873, -0.16562162101526468, -0.51902306075761517, 
-0.37089866480225198]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_array_rank2(self):
      arg0=Data(numpy.array([0.6082991034824301, -0.89790957751207823]),self.functionspace)
      arg1=numpy.array([[0.76657659534322575, 0.68288841558950208, -0.52316353807759675, -0.14719334499260706, 
-0.41525172281721101], [-0.64498670026315219, 0.900497810651681, 0.80284887559478979, 0.30499901681381969, 
-0.83593848020492501], [0.89752625807910302, 0.44866872380829048, -0.5104140244105988, 0.36221640249806852, 
-0.96865683502897593], [-0.15265985147484296, 0.59080359638445579, 0.25838669533340086, 0.84017981147952447, 
0.59099373775915232]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.46630785569789784, 0.41540041098163127, -0.31823991118729827, -0.089537579797582909, 
-0.25259725070924405], [-0.39234483152816635, 0.54777201090730865, 0.48837225125618766, 0.18553062849086915, 
-0.50850062807512098], [0.54596441814145857, 0.27292478245318913, -0.31048439345382645, 0.22033591290620613, 
-0.58923308433025423], [-0.0928628507899079, 0.35938529801485997, 0.15717639512309556, 0.51108062608703186, 
0.35950096084262273]], [[-0.68831646685528314, -0.61317204872986231, 0.46975355144497893, 0.13216631421490138, 
0.37285849899596457], [0.5791397355341964, -0.80856560871280225, -0.72088569469136476, -0.27386153832889609, 
0.75059716758689299], [-0.80589742319780389, -0.40286394423758543, 0.45830564101476035, -0.32523757693498556, 
0.86976624949505466], [0.13707474274083284, -0.53048820762218307, -0.23200788844155604, -0.75440549955975733, 
-0.53065893738360437]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_array_rank2(self):
      arg0=Data(numpy.array([[0.16196797258458528, -0.87553179470356857, -0.60222680248837368, 0.14709037205386077, 
0.3556166421447251], [-0.6419686001157352, -0.42480335802533564, -0.19594124821840864, 0.70726884481089414, 
0.75204164044023791], [0.039321557679237396, 0.43748758848062752, 0.75704401346022876, 0.76454523005635644, 
0.30534874386382027], [0.15112591468115033, 0.089900406644022102, 0.35917646346824705, -0.3421665984079667, 
-0.20565912906789818]]),self.functionspace)
      arg1=numpy.array([[0.21860412872011858, -0.98840740295497787, -0.069274004080670748, -0.04734481193948481, 
-0.038090044095202558], [-0.86828963553517347, -0.69761872791299773, -0.64672359733995943, -0.59855991393624142, 
-0.36198596308426945], [-0.28082063984983763, -0.61440801497291919, -0.65080269815869363, -0.08863549331887266, 
-0.87236881392923471], [-0.71107952303930211, -0.83721058185920727, -0.12005709440444634, 0.53721957267291631, 
0.44292111553365765]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.03540686752741732, -0.160090343144213, -0.011220169993762529, -0.0076683432022368221, 
-0.0061693672177574122], [-0.14063511188384054, -0.11299189099710567, -0.10474850988376293, -0.096947535730656884, 
-0.058630132544837656], [-0.045483949696384196, -0.09951442052488324, -0.10540919357334143, -0.014356111151892359, 
-0.14129580813813747], [-0.11517210869308966, -0.13560130057009678, -0.019445404175084332, 0.087012365018589527, 
0.071739035097889398]], [[-0.19139486514793533, 0.86538210740746502, 0.060651593119051994, 0.041451888167280078, 
0.03334904466701076], [0.76021518292261792, 0.61078737686848739, 0.56622707185620269, 0.52405823568621102, 
0.31693021991667014], [0.2458673987975328, 0.537933752029497, 0.56979845431680587, 0.077603192539908744, 0.76378663330288632], 
[0.6225727309835577, 0.73300448328001067, 0.10511380333082067, -0.47035281661220257, -0.38779151919528992]], 
[[-0.13164926544987388, 0.5952454298374138, 0.041718661973068893, 0.028512314708729315, 0.022938845462094994], 
[0.52290729084214282, 0.42012469586705137, 0.38947428411982227, 0.3604688230675388, 0.21799764909391406], [0.1691177160095069, 
0.37001297428036994, 0.39193082796291628, 0.053378669728404292, 0.52536388140317802], [0.42823114747491675, 
0.50419065172250122, 0.072301600079234543, -0.32352802548498089, -0.26673896716241818]], [[0.032154562625952315, 
-0.14538521264143797, -0.01018953903388653, -0.0069639660029988905, -0.005602678757511307], [-0.12771704554137983, 
-0.10261299824046391, -0.095126814548745872, -0.08804240043740863, -0.053244649988340303], [-0.041306012395915859, 
-0.090373503515240747, -0.095726811005818702, -0.01303742768945047, -0.12831705340903637], [-0.10459295160373282, 
-0.12314561597310006, -0.017659242683655499, 0.079019826819075351, 0.065149431674356759]], [[0.077739266214421829, 
-0.35149412170983746, -0.024634988719088115, -0.016836603044893077, -0.013545453580280448], [-0.30877824459808556, 
-0.24808482951769487, -0.22998567408179366, -0.21285786671644183, -0.12872823269555231], [-0.099864492988332443, 
-0.21849371519147551, -0.23143627021792171, -0.031520256508898711, -0.31022886832129093], [-0.25287171228110933, 
-0.29772601588880282, -0.042694300777761471, 0.19104422052836662, 0.15751011984107519]]], [[[-0.14033698649397452, 
0.63452651681903649, 0.044471735424079929, 0.03039388264353381, 0.024452612286143812], [0.55741468181951725, 
0.44784931817282714, 0.41517624244614615, 0.38425667003504382, 0.23238362198275467], [0.1802780330480053, 0.39443065327205262, 
0.41779489708847989, 0.056901203566484281, 0.5600333862627751], [0.45649072597650547, 0.53746290523823537, 
0.077072884828785077, -0.34487809702360556, -0.28434144850084203]], [[-0.092863767958509091, 0.41987878387237565, 
0.029427829557329737, 0.020112235096971152, 0.016180778638975154], [0.36885235291393648, 0.2963507782388044, 
0.27473035586423977, 0.25427026141947123, 0.15377285267623284], [0.1192935508110344, 0.26100258795817677, 0.27646317158976197, 
0.037652655202089311, 0.37058520159371811], [0.30206896921014958, 0.35564986654813641, 0.051000656857773538, 
-0.22821267846839069, -0.18815437721902542]], [[-0.042833565847117706, 0.19366978028331397, 0.01357363482865376, 
0.0092768015480884699, 0.0074634107847082144], [0.170133755001869, 0.13669228432781116, 0.12671982891509115, 
0.11728257667017039, 0.070927981444274549], [0.055024346697669368, 0.12038787336918849, 0.12751909302112266, 
0.017367349197354328, 0.17093303430810691], [0.13932980932687153, 0.16404408643115326, 0.02352413693508254, 
-0.10526347363689131, -0.08678651623995487]], [[0.15461188959077027, -0.6990697620905032, -0.048995344841561164, 
-0.03348551044822845, -0.026939901486009932], [-0.61411420748623446, -0.49340399180947137, -0.45740745160257895, 
-0.42334277887979366, -0.2560213939483702], [-0.19861568954565081, -0.43455164699245108, -0.46029247252651229, 
-0.062689122968882793, -0.61699928327677966], [-0.50292439282868884, -0.59213296109501801, -0.084912642470785227, 
0.37995866657417571, 0.31326430572584263]], [[0.16439940756968691, -0.74332352474153685, -0.052096935668691363, 
-0.035605270037304716, -0.028645299245797128], [-0.65298996188512815, -0.52463833254152281, -0.48636307505495496, 
-0.45014197957837859, -0.27222851749423338], [-0.21118881466214914, -0.46206041147986443, -0.48943072872619697, 
-0.066657581796754739, -0.65605767389624636], [-0.53476141098993868, -0.62961721937532433, -0.090287934222408328, 
0.40401148870954356, 0.33309512231155203]]], [[[0.0085958548563875789, -0.038865718705879404, -0.0027239617471298216, 
-0.0018616717534910992, -0.0014977598658942031], [-0.034142500985980338, -0.027431455047747157, -0.025430179235327116, 
-0.02353630818032329, -0.014233851926492399], [-0.011042304987375741, -0.024159480199343396, -0.025590575833450396, 
-0.0034852856629657121, -0.034302900634486315], [-0.027960754479714533, -0.03292042418424472, -0.0047208319624260859, 
0.021124310413273343, 0.01741634819180889]], [[0.095636593105673365, -0.43241597115517316, -0.030306516989649795, 
-0.020712767602474032, -0.016663921536330933], [-0.379865938753006, -0.30519953495358043, -0.28293354701377521, 
-0.26186253330913822, -0.15836436605357448], [-0.12285554452349227, -0.26879588081367173, -0.28471810299413258, 
-0.038776928225864371, -0.38165052867160615], [-0.31108846575241911, -0.3662692385080476, -0.052523488710992272, 
0.23502689533326737, 0.19377249072196931]], [[0.16549294696525504, -0.74826790726683801, -0.052443470077691248, 
-0.035842106447187334, -0.028835839854709232], [-0.65733347053146696, -0.52812808164427516, -0.48959822772967981, 
-0.45313619954270129, -0.27403930630958157], [-0.21259358425439054, -0.46513390955723105, -0.49268628658480323, 
-0.067100969597146648, -0.66042158811452734], [-0.53831849601105852, -0.63380525900206763, -0.090888504592315636, 
0.40669886140569361, 0.33531077894988187]], [[0.16713274388359242, -0.75568216528161936, -0.052963109386781393, 
-0.036197250136248346, -0.0291215615256234], [-0.66384669915578909, -0.53336107082386552, -0.49444944151115372, 
-0.45762612710289663, -0.27675464142343453], [-0.21470008069856733, -0.46974271715593979, -0.49756809858503592, 
-0.067765843630636147, -0.6669654155395176], [-0.54365245753044744, -0.64008535691316371, -0.09178907886134513, 
0.41072866177999229, 0.3386332261724983]], [[0.066750496108133092, -0.3018089589180033, -0.021152730128449971, 
-0.014456678854190486, -0.011630747118187625], [-0.26513114952063954, -0.21301700226411008, -0.19747623807484771, 
-0.18276951784766768, -0.1105319591241169], [-0.085748229629182188, -0.18760871559184417, -0.19872178638594207, 
-0.0270647365466678, -0.26637672151926256], [-0.21712723914733534, -0.25564119952020703, -0.036659282968337772, 
0.16403932169473331, 0.13524540625896436]]], [[[0.033036748905903843, -0.14937397284919135, -0.010469097230317108, 
-0.0071550280097616888, -0.005756392754132836], [-0.13122106537841574, -0.10542826835455232, -0.097736695193885337, 
-0.090457914485085111, -0.05470545977284734], [-0.042439276058652609, -0.092852973250212315, -0.098353153036193158, 
-0.013395120001029619, -0.13183753494436584], [-0.10746254333035064, -0.12652421496421079, -0.018143738205833169, 
0.081187799304811195, 0.066936858716619474]], [[0.019652600066000812, -0.088858227455614322, -0.0062277611367119465, 
-0.0042563178458444369, -0.0034243104532474429], [-0.078059591319401836, -0.062716207321863907, -0.058140714387147165, 
-0.053810779663678976, -0.032542685280703795], [-0.025245889716534881, -0.055235530391411852, -0.058507427209493333, 
-0.0079683668924601572, -0.07842631111580145], [-0.063926338277470546, -0.07526557175582109, -0.010793181607459476, 
0.048296258040422958, 0.039818788397699714]], [[0.07851745785324965, -0.35501267545920356, -0.024881591795980235, 
-0.017005142115993393, -0.013681047331464441], [-0.31186920055765677, -0.25056822754100783, -0.23228789453402926, 
-0.2149886330614775, -0.13001683804575534], [-0.10086416429015496, -0.22068089794451889, -0.23375301154023265, 
-0.031835783028036121, -0.31333434542709176], [-0.25540302832994444, -0.30070633597038343, -0.04312168258246251, 
0.19295662621858101, 0.15908683987279001]], [[-0.074799031122100276, 0.33819999891035724, 0.023703250334382715, 
0.016199813253598407, 0.013033140821264918], [0.29709971102396349, 0.23870182711568327, 0.22128721341197746, 
0.20480720969492902, 0.12385950565997629], [0.096087443100167647, 0.21022990051787482, 0.22268294546368689, 
0.030328105247130716, 0.29849546961935869], [0.24330766159591738, 0.2864654969459195, 0.041079527607113538, 
-0.18381859377967324, -0.15155281146521365]], [[-0.044957934723226296, 0.20327500565598383, 0.014246831346276771, 
0.0097368927893578729, 0.0078335652947771964], [0.17857169022284652, 0.14347166000404216, 0.13300461177659414, 
0.1230993105950835, 0.074445717902715203], [0.05775332821580751, 0.12635861725166669, 0.13384351609835515, 
0.018228698360462859, 0.17941061049868173], [0.1462399954062793, 0.17217999911159285, 0.024690837473640866, 
-0.1104841094341404, -0.09109077086643394]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_constData_rank2(self):
      arg0=Data(0.401591456378,self.functionspace)
      arg1=Data(numpy.array([[-0.20636599075220774, -0.41037145249758167, -0.65372099878190437, -0.71522947074190424, 
0.6239040269837941], [0.27537558542853469, -0.082977678406752631, -0.81947272137821492, -0.57172533214450616, 
0.87255906268418171], [0.065289537431470146, 0.58518655847804424, 0.65745809407314848, -0.17463204844297064, 
-0.81657024710887893], [0.45031562839620509, 0.52969267736231962, -0.92934239922061845, -0.94765722904536687, 
-0.30658080708702773]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[-0.082874818773104889, -0.16480166926453246, -0.26252876796582264, -0.28723004479983538, 
0.25055452683663249], [0.11058848240323885, -0.033323126718247949, -0.32909324364046688, -0.22960000878421025, 
0.35041226475931919], [0.026219720423361716, 0.2350059222721321, 0.2640295535064574, -0.070130738664517259, 
-0.32792763477154407], [0.18084290903748679, 0.21272005373479075, -0.37321596757699904, -0.38057104675963827, 
-0.12312023281567695]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_constData_rank2(self):
      arg0=Data(numpy.array([-0.6037699296970036, -0.64977695945749225]),self.functionspace)
      arg1=Data(numpy.array([[0.11246978049741929, 0.57470907517187886, -0.10635824940352423, 0.81226307149406152, 
0.18553308629923349], [0.86526504782660396, 0.38800189087870085, -0.90036808440952543, -0.22258074352878832, 
0.061709096654924744], [0.49362869521577335, -0.50310428478554359, -0.8247507935097822, 0.54201170525013609, 
-0.7474481860057034], [-0.6772962728394869, -0.39473202374126393, -0.12896798728005354, -0.71086115611699374, 
0.27274538987649644]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.067905871463964276, -0.34699205791275528, 0.064215912765062194, -0.49042001757144171, 
-0.1120192984713563], [-0.52242101709554312, -0.23426387437813767, 0.54361517502536494, 0.13438755987228332, 
-0.037258096949009514], [-0.29803816260685112, 0.30375923865522891, 0.49795972861494914, -0.32725036917382772, 
0.4512867387168164], [0.40893112303633961, 0.23832732622341887, 0.077866992613241981, 0.42919659025308804, 
-0.16467546487091408]], [[-0.073080272002464669, -0.37343271543781081, 0.069109139910643616, -0.52778982887501491, 
-0.12055512469428045], [-0.5622292919016123, -0.25211468891891992, 0.58503843628018815, 0.14462783876392396, 
-0.040097149195305506], [-0.32074855267827435, 0.32690557245798679, 0.53590406291694026, -0.35218671782780392, 
0.48567460965480408], [0.44009151281753389, 0.25648777418710111, 0.083800426642185732, 0.46190120061813794, 
-0.17722367013999815]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_constData_rank2(self):
      arg0=Data(numpy.array([[-0.92127095180878116, 0.69094113165284088, 0.47472214054501838, -0.75831139586704532, 
-0.96280360258309083], [-0.41455067519827504, 0.38490685299598559, 0.2447115140349736, 0.00037197105984443368, 
0.57403409938038674], [0.60014207779041695, 0.059576141602880606, -0.21413099513921741, 0.54640485285359564, 
0.27570424665373161], [0.26235808321486132, -0.54732464402956138, -0.69381431133305127, -0.43435017056214087, 
0.19733412935079175]]),self.functionspace)
      arg1=Data(numpy.array([[-0.77313075143792598, 0.0084607718125822018, -0.64464610044251924, -0.57638099822004474, 
-0.93184039847213129], [0.17235025217443223, -0.13940553724244564, -0.24589043482940109, 0.11872633083126116, 
0.72206642689275036], [0.27659434094381496, 0.35400790191399789, 0.79422056366365057, 0.77235596300316445, 
0.72206224814221165], [-0.90875871953557441, -0.28709729641379167, -0.14124077404126978, -0.11279523096346988, 
-0.41136629847249173]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.71226290324985631, -0.0077946633008145116, 0.59389372653449879, 0.53100307083467602, 
0.85847749083429425], [-0.15878128086522264, 0.12843027198276238, 0.22653171493595742, -0.10937911980968021, 
-0.66521882437264979], [-0.25481833174623092, -0.32613719674413849, -0.73169233463251804, -0.71154911317111302, 
-0.66521497461116363], [0.83721301051106789, 0.26449439952886161, 0.13012102233520961, 0.10391496978920721, 
0.3789798213358076]], [[-0.5341878363141318, 0.0058458952508420045, -0.44541250615534517, -0.39824533917335175, 
-0.64384685944016862], [0.11908387827805471, -0.096321019660967649, -0.16989581530363551, 0.082032905381541163, 
0.49890539412580021], [0.19111040694049122, 0.2445986203625056, 0.54875965503971991, 0.53365250311622614, 0.49890250685517412], 
[-0.62789877807529637, -0.1983673308786163, -0.097589060251598136, -0.07793486452694344, -0.28422989579042374]], 
[[-0.36702228524379077, 0.0040165157055319775, -0.30602777669607167, -0.27362082124449405, -0.44236526860901304], 
[0.08181848063572017, -0.066178895043562072, -0.11672963356175863, 0.056362017911272308, 0.34278091979021946], 
[0.13130545759548645, 0.16805538896646399, 0.37703408604727923, 0.36665447601957124, 0.34277893604481896], 
[-0.4314078845768779, -0.13629144309824281, -0.067050122585106856, -0.053546393486248157, -0.1952846897589422]], 
[[0.58627385931063136, -0.0064158996833117606, 0.48884248426681426, 0.43707627931148307, 0.70662519329070561], 
[-0.13069516030443096, 0.10571280753791433, 0.18646151886583789, -0.09003152965882627, -0.54755120008577141], 
[-0.20974464077002977, -0.26844822624836778, -0.60226650425809436, -0.58568632841116564, -0.54754803129161733], 
[0.68912209311737016, 0.21770915159319723, 0.10710448851657722, 0.085533909039054626, 0.31194375200733482]], 
[[0.74437307275220732, -0.0081460615817876107, 0.62066758789719856, 0.55494170154669709, 0.89717929268143093], 
[-0.16593944369964753, 0.1342201534770579, 0.23674419649447009, -0.11431013904581014, -0.69520815711664008], 
[-0.26630602791480074, -0.34084008330567861, -0.76467841994093577, -0.74362710365597917, -0.69520413380056711], 
[0.87495616904764772, 0.27641831127906408, 0.13598712607855884, 0.1085996547258206, 0.39606495415058607]]], 
[[[0.32050187502514199, -0.0035074186676044851, 0.26723847620238139, 0.2389391319835753, 0.38629506636365168], 
[-0.071447913409503849, 0.05779065959023412, 0.10193404578332567, -0.049218080609913097, -0.29933312480639557], 
[-0.11466237079428039, -0.14675421476397255, -0.32924467092312093, -0.32018068595637578, -0.29933139250253826], 
[0.37672654077579221, 0.11901637807593664, 0.058551458244335386, 0.046759339155051823, 0.17053217678558658]], 
[[-0.29758332449039365, 0.003256609052298156, -0.24812870181746413, -0.2218529961515622, -0.35867175527043327], 
[0.066338793177525238, -0.05365814663020442, -0.094644913451999269, 0.045698578368020992, 0.27792831602934442], 
[0.1064630573291825, 0.13626006746142846, 0.30570093774447354, 0.29728510311223189, 0.27792670759962512], 
[-0.34978745886909945, -0.11050571686628821, -0.054364541850942243, -0.043415657383104544, -0.1583377073736541]], 
[[-0.18919399673137172, 0.0020704482801614183, -0.15775232325603056, -0.14104706673541656, -0.22803207474906834], 
[0.04217609115391481, -0.03411414008345777, -0.060172220593820748, 0.029053700173535084, 0.17669796855874853], 
[0.067685819945866643, 0.086629809657718854, 0.19435491661184207, 0.18900439708044442, 0.17669694597037741], 
[-0.22238372215003435, -0.070256014080766552, -0.034563243659110725, -0.027602291744995251, -0.1006660697221663]], 
[[-0.00028758226501068873, 3.1471622582281117e-06, -0.00023978969320618513, -0.00021439705078210269, -0.00034661766062553807], 
[6.4109305965778964e-05, -5.1854825436255177e-05, -9.1464125649100977e-05, 4.4162759110745077e-05, 0.00026858781408937966], 
[0.00010288509014784348, 0.00013168069446825413, 0.0002954270648162116, 0.0002872940661354553, 0.00026858625971511293], 
[-0.0003380319440485181, -0.00010679188562550962, -5.2537480413379293e-05, -4.1956561606879575e-05, -0.00015301635802709439]], 
[[-0.44380341460495148, 0.0048567715274985867, -0.37004884368659985, -0.33086234721321167, -0.53490816390321061], 
[0.098934921784932739, -0.080023532019606242, -0.14114949430354692, 0.068152962391460847, 0.41449075105419381], 
[0.15877458339739445, 0.20321260714874206, 0.45590968597204679, 0.44335865962359283, 0.41448835230889181], 
[-0.52165849312267687, -0.16480363798143483, -0.081077020522569002, -0.064748308820518144, -0.23613828265910014]]], 
[[[-0.4639882955716233, 0.0050776651753136745, -0.38687925015906333, -0.34591048987069228, -0.55923663290811498], 
[0.10343463844766608, -0.083663128776170687, -0.14756919646730587, 0.07125266687350551, 0.43334244573811742], 
[0.1659959024790921, 0.21245503780889283, 0.47664517930097938, 0.46352331243053752, 0.43333993789408654], 
[-0.54538434615223841, -0.17229916799778416, -0.084764531601854431, -0.067693164275266796, -0.24687822509823401]], 
[[-0.046060147125207368, 0.00050406013957605804, -0.038405527363708322, -0.034338555967167061, -0.055515455530660382], 
[0.010267963028836155, -0.0083052440269815869, -0.014649203364190285, 0.0070732566975936642, 0.043017931695248535], 
[0.016478423622624156, 0.021090424892967006, 0.047316596764745297, 0.046013988219705738, 0.043017682741414716], 
[-0.054140338157903846, -0.01710414918495224, -0.0084145803543831509, -0.0067199046520093047, -0.024507616848450015]], 
[[0.16555125717813401, -0.0018117134878740671, 0.13803871100037254, 0.12342103672819367, 0.19953591183576236], 
[-0.036905531010606243, 0.029851046417642118, 0.05265276350523454, -0.025422987370125903, -0.15461680254716362], 
[-0.059227421476175081, -0.075804064323990838, -0.17006723965732767, -0.16538535095957618, -0.15461590774715231], 
[0.19459340895559352, 0.061476429782864085, 0.030244027499690442, 0.024152955053165674, 0.08808627485865099]], 
[[-0.42244239447602977, 0.0046230067772818273, -0.35223775765493898, -0.31493737452003212, -0.50916211581020088], 
[0.094173014178650746, -0.076171862063934961, -0.13435572686106556, 0.064872643327702567, 0.39454059973685485], 
[0.15113249016354247, 0.19343163555432813, 0.43396597022193678, 0.42201904631534121, 0.3945383164472816], 
[-0.49655017442725752, -0.15687135600164295, -0.077174644356947958, -0.061631861577182097, -0.22477254178579015]], 
[[-0.21315543139002682, 0.0023326707186971034, -0.17773166748077057, -0.15891068889978324, -0.25691235506227206], 
[0.047517696436332503, -0.038434698624787204, -0.067793037093998518, 0.032733353599794568, 0.1990767802604175], 
[0.076258234398599897, 0.097601481906666898, 0.21896998218178887, 0.21294181892830485, 0.19907562816114829], 
[-0.2505486381595653, -0.079153943824087511, -0.038940681203838215, -0.031098124178917125, -0.11341543541909244]]], 
[[[-0.20283710202171965, 0.0022197518752673944, -0.16912811526403432, -0.15121821389447934, -0.24447586080532094], 
[0.045217481802082023, -0.036574169540466002, -0.064511343162710441, 0.031148812584023172, 0.18943996371338578], 
[0.072566761118097123, 0.092876834589071117, 0.20837018473262209, 0.20263383001307855, 0.18943886738440421], 
[-0.23842019576214504, -0.075322296383291257, -0.037055658749250882, -0.029592740591353535, -0.10792527356643546]], 
[[0.42315351331907014, -0.0046307889205369004, 0.35283069744974671, 0.31546752467618921, 0.51001921438612385], 
[-0.094331540419776261, 0.076300086046971319, 0.13458189471327603, -0.064981846759155956, -0.3952047500647719], 
[-0.15138689919766465, -0.19375724889873078, -0.43469648728816512, -0.42272945251481608, -0.39520246293162081], 
[0.49738604267856829, 0.15713542556152799, 0.077304556374597697, 0.061735609635313313, 0.22515091287721484]], 
[[0.53640917987930903, -0.0058702045684928122, 0.4472646902320635, 0.39990138534549696, 0.64652420433825786], 
[-0.11957907152048142, 0.096721556817881457, 0.17060230270454543, -0.082374027462791474, -0.50098002071131065], 
[-0.19190511218055217, -0.2456157486729188, -0.55104159342484349, -0.53587162057501614, -0.50097712143438333], 
[0.63050980516248001, 0.19919221299691575, 0.097994870373590698, 0.078258945492572313, 0.28541182508031826]], 
[[0.3358094737538993, -0.003674937679882433, 0.28000214367942722, 0.25035118488565344, 0.40474503601306355], 
[-0.074860361428392627, 0.060550818878563145, 0.10680255230774935, -0.051568802046775448, -0.31362967567806171], 
[-0.12013879916546898, -0.15376339257669061, -0.34496983729126629, -0.33547294426511104, -0.3136278606370525], 
[0.39471950483010948, 0.12470075966525992, 0.061347954295154328, 0.048992627807579216, 0.17867702190504334]], 
[[-0.1525650837093265, 0.001669599039271629, -0.12721067697020758, -0.11373964245809277, -0.1838839137263929], 
[0.034010586956230986, -0.027509470318417387, -0.048522574872747463, 0.023428757125600986, 0.14248834968431801], 
[0.05458150345350378, 0.069857841107499263, 0.15672682344306155, 0.1524121915081218, 0.14248752507421869], 
[-0.17932911070949292, -0.056654095026781766, -0.02787162517426588, -0.022258348697097797, 
-0.081176610353327086]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_expandedData_rank2(self):
      arg0=Data(0.0596557925971,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.31928510411621724, 0.064704454835101899, -0.23987836773877858, -0.90618834188613073, 
0.86835655239726695], [-0.51412650198807031, -0.46973335346415701, -0.8729202258712625, -0.97687278919286924, 
-0.90832968295672534], [-0.049162725267693608, -0.20127692395668162, -0.062009945166549674, -0.30360297570745054, 
-0.522325587313895], [-0.6617808520544175, -0.89148228241511585, 0.0438043718299026, -0.63153844785264757, 
-0.46117950601537605]])+(1.-msk_arg1)*numpy.array([[-0.024236153865825116, -0.12283916168511944, -0.38330902066210149, 
-0.45060899935199306, 0.17935121229739748], [0.82995869977987646, -0.049795990724521344, 0.50232359921601777, 
0.41922814719074886, -0.47855235126117446], [0.026221206389239082, 0.75310141641819639, 0.88473844001614332, 
-0.53523113043558856, -0.61177775227866893], [0.34495492330617306, -0.92331067160852265, 0.71145038846183661, 
-0.17562305144957313, -0.76964110985703527]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.019047205950485778, 0.0038599955377482727, -0.014310134154344371, -0.054059383777427077, 
0.051802498390104021], [-0.030670623971249079, -0.028022315510176303, -0.052074747948349376, -0.058276120505795247, 
-0.054187127176214024], [-0.002932841342075465, -0.012007334430132764, -0.0036992524278103649, -0.018111676150652026, 
-0.031159746904932027], [-0.039479061254859865, -0.053182082143704276, 0.0026131845207288946, -0.037674926662162808, 
-0.027512028960864989]])+(1.-msk_ref)*numpy.array([[-0.0014458269683699464, -0.0073280675522834401, -0.022866603437198128, 
-0.026881437007708439, 0.010699338722843706], [0.049511844058188738, -0.002970619294826859, 0.029966512451436323, 
0.02500938739965845, -0.028548419813669051], [0.001564246850000988, 0.044926861902391355, 0.05277977288024395, 
-0.031929637308752111, -0.036496086705428019], [0.020578559360085652, -0.055080829928124457, 0.042442136817172668, 
-0.01047693233253744, -0.045913550423797578]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_expandedData_rank2(self):
      arg0=Data(numpy.array([0.81754009937055727, 0.068794605163091882]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.073594571193924319, 0.49278314621057606, -0.79450929904989653, 0.55543766467178646, 
-0.15150111473618222], [0.03670305604595292, -0.67735370664124428, 0.65738905375957324, -0.63103985741826785, 
0.9628578377269259], [-0.86786828643525094, -0.5752751384508854, 0.39109303732782164, -0.83914853897407826, 
0.95198513521267536], [0.70762159671469527, -0.6304267832395658, -0.5483870867810221, -0.57642535770284908, 
0.82988669590837638]])+(1.-msk_arg1)*numpy.array([[-0.38948009589738009, -0.83859298264283355, -0.21024311874563439, 
0.50084730030558045, -0.46145224636212845], [0.17583332556646702, 0.38526563864204744, 0.30977882677375379, 
0.87702995861563915, 0.94144902733384161], [0.91242354539656567, -0.12921860172255517, -0.77641969028149282, 
-0.59418535977868281, 0.72270161672714472], [-0.99183426227178062, -0.96151014763149401, -0.46360827099737434, 
0.43715054312385027, -0.90366235547661167]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.060166513047014439, 0.40286998232113019, -0.64954321129608417, 0.45409256356992256, 
-0.12385823639616861], [0.030006220087011481, -0.5537638166364981, 0.53744191233571814, -0.51590038774051294, 
0.78717489233499094], [-0.70951712513283027, -0.47031049385454793, 0.31973424060012035, -0.68603757993952585, 
0.77828602204106401], [0.57850903049488434, -0.51539917501553545, -0.44832843342048723, -0.47125084421609625, 
0.67846565183923746]], [[0.0050629094674330792, 0.033900821974582759, -0.054657953526542527, 0.038211114833805376, 
-0.010422459370043937], [0.0025249722489601636, -0.046598280804141169, 0.045224820391928426, -0.043412137833263531, 
0.066239424774612268], [-0.059704656098882217, -0.039575826009871681, 0.026905091085001845, -0.05772889241190713, 
0.065491441498088637], [0.048680548350864099, -0.043369961637204038, -0.037726073111638621, -0.039654954889161508, 
0.057091727575119655]]])+(1.-msk_ref)*numpy.array([[[-0.31841559630279831, -0.68558339036127414, -0.17188218019128182, 
0.40946275166129958, -0.37725571534566138], [0.14375079445626501, 0.31497010849948065, 0.25325661282350931, 
0.71700715951758531, 0.7696723313588234], [0.74594283597154443, -0.10564138849278222, -0.63475423074598891, 
-0.48577035807799468, 0.59083755155437234], [-0.81086428133679489, -0.78607310164045086, -0.37901835194020567, 
0.35738809846536562, -0.73878021189378096]], [[-0.026794129416143425, -0.057690673133453295, -0.01446359234236296, 
0.034455592271522915, -0.031745425090104432], [0.012096384206858489, 0.026504197493286089, 0.021311112075786228, 
0.060334929719165709, 0.064766614116608529], [0.062769817547065176, -0.0088895426852300067, -0.053413486033765389, 
-0.040876747219664181, 0.049717972373472079], [-0.068232846460213673, -0.066146710966614813, -0.031893747953608068, 
0.030073599011036449, -0.062167094945763078]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank2_expandedData_rank2(self):
      arg0=Data(numpy.array([[0.096587985349336769, -0.41147338952206325, 0.24409171143097264, 0.87622408202359137, 
0.13972144257176877], [0.90926242262038559, 0.23018290769581995, -0.42857420242821154, -0.66176433450416416, 
0.13873484339387732], [0.67546872313826256, -0.1473051146243205, 0.18477243685801326, -0.36311347294725893, 
-0.078844790086926464], [-0.10370292228364386, -0.61063338424861202, 0.73079465938438748, -0.97666680454888644, 
-0.64935172531905838]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.9652330546955592, -0.20265266885029432, 0.3716803443575416, 0.92942145427133416, 
0.2223474274250592], [0.74680455546934854, -0.60335567874715323, -0.23996232680339369, 0.4017292028969488, 
-0.8708951237392375], [0.84719581080672524, -0.83137189315267035, 0.69592012701030304, 0.45226323247994227, 
-0.5411054267749229], [0.96631753795233633, 0.99859139869408753, -0.95454913298656785, 0.0076583508435648096, 
0.67698130053309269]])+(1.-msk_arg1)*numpy.array([[-0.87833112768788291, -0.6595510056659506, 0.24220570379344464, 
0.43217280424913507, -0.95579951001036889], [-0.42609761853809225, -0.82180511560918923, 0.71834570005375431, 
-0.26719977697253161, -0.5984655911967709], [0.69700867656891874, -0.5273527604562207, -0.0099409341444107735, 
0.93986037339584372, -0.08491685305213803], [0.43714405229483755, 0.21144112509039981, -0.81170014133570789, 
-0.14134978605899451, 0.78010563049507331]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.093229916145630254, -0.019573813009916223, 0.035899855655442674, 0.089770945808518895, 
0.021476090062594339], [0.072132347462491392, -0.05827690945926918, -0.023177477705678951, 0.038802214363811226, 
-0.084118005452534297], [0.081828936562219456, -0.080300536235680495, 0.06721752303197974, 0.04368319447281635, 
-0.052264283033782874], [0.09333466419854744, 0.096451931387038239, -0.092197977672128728, 0.00073970467907831874, 
0.065388259937665311]], [[-0.39716771669431683, 0.083386180547522851, -0.15293657111152531, -0.38243219608355117, 
-0.091490049614100066], [-0.30729020174949057, 0.24826480622147623, 0.098738111967393447, -0.16530087678600419, 
0.35835016848322077], [-0.34859853186153583, 0.34208741082890387, -0.28635261349755414, -0.18609428522472674, 
0.22265048404386012], [-0.3976139526958628, -0.41089378756823425, 0.3927715672153298, -0.0031512075797507644, 
-0.27855979037340622]], [[0.23560538825038466, -0.049465836765722498, 0.090724091359485587, 0.22686407341375342, 
0.05427316409245668], [0.18228880204896003, -0.14727412022698877, -0.058572815028398721, 0.098058768666916679, 
-0.21257828123039915], [0.20679347537696408, -0.20293098823524303, 0.16986833482120473, 0.11039370643333296, 
-0.13207934968607776], [0.23587010162454963, 0.24374788352748855, -0.23299753151564245, 0.0018693399641445674, 
0.16524552425388822]], [[0.84576044728944333, -0.17756914873297999, 0.32567526854089923, 0.81438146058193106, 
0.19482617048582959], [0.65436813606716615, -0.52867477574394528, -0.21026076952354866, 0.35200480203044804, 
-0.7630992803372354], [0.742333371618355, -0.7284680738979139, 0.60978197445134386, 0.39628393571275949, -0.47412960585384045], 
[0.84671069763558282, 0.87498983163738098, -0.83639893779757046, 0.0067104314377171717, 0.59318731860674612]], 
[[0.13486375482001853, -0.02831492323278207, 0.05193171388920749, 0.12986010634794204, 0.031066703311950935], 
[0.10434460980934589, -0.084301725818420936, -0.033527882463848378, 0.056130183751968474, -0.12168272301756533], 
[0.11837142082667494, -0.11616048022491351, 0.097234964060608081, 0.063190871264268769, -0.075604030812404827], 
[0.13501528038510036, 0.1395246307652982, -0.13337098186651442, 0.0010700358275835976, 0.094588803904595842]]], 
[[[0.87765014570575928, -0.18426445662930535, 0.33795497035091743, 0.84508800314611521, 0.20217216052391968], [0.67904131933, 
-0.54860864615940363, -0.21818872660687844, 0.36527726826343609, -0.79187221005941955], [0.77032331536796483, 
-0.75593522166649341, 0.63277402063567456, 0.41122596242683895, -0.49200683124240407], [0.8786362255791077, 
0.90798163438446533, -0.86793565716955534, 0.0069634506412966125, 0.61555365739141921]], [[0.22218015113394224, 
-0.046647180568278865, 0.085554462397602574, 0.21393693281905327, 0.051180577363385424], [0.1719016440584189, 
-0.13888216450880478, -0.055235226121059751, 0.092471196029143693, -0.2004651718804086], [0.19500999511920977, 
-0.1913675997424602, 0.16018891835927587, 0.10410326589614372, -0.12455322050503934], [0.22242978064333463, 0.2298586717514409, 
-0.21972089496937211, 0.0017628214653264834, 0.15582952421240501]], [[-0.41367398657349558, 0.086851705922463357, 
-0.15929260714127641, -0.39832605848400554, -0.095292371370659393], [-0.3200611667300311, 0.2585826787995934, 
0.10284166282258229, -0.172170772723681, 0.3732431830551623], [-0.36308626891701429, 0.35630454602913797, -0.29825341338718031, 
-0.19382835414769609, 0.2319038267096396], [-0.41413876812031558, -0.42797051224699079, 0.40909513334825914, 
-0.0032821716046962093, -0.2901367209347836]], [[-0.63875681008202823, 0.13410830853720779, -0.24596479573204708, 
-0.61505797015976194, -0.14714159733865723], [-0.49420861965485158, 0.39927926921541812, 0.15879850950311858, 
-0.26585005860598765, 0.57632733198421815], [-0.56064397193322835, 0.55017226759764393, -0.46053511971902661, 
-0.29929167706279108, 0.35808427264629861], [-0.63947448242273019, -0.66083217239837533, 0.63168657174238296, 
-0.0050680234493910709, -0.44800207981904566]], [[0.13391145668178223, -0.028114986276296862, 0.05156501436702593, 
0.12894313990524325, 0.030847335522847089], [0.10360781304887423, -0.083706455601792851, -0.033291135829499233, 
0.055733838050655356, -0.12082349860445452], [0.11753557813621994, -0.11534024939860703, 0.096548369835421602, 
0.062744668730913514, -0.075070176643196082], [0.13406191229657449, 0.13853942131229713, -0.13242922447665287, 
0.001062480104937332, 0.093920894710042002]]], [[[0.65198473898605414, -0.13688553946886944, 0.25105844761877838, 
0.62779512297396523, 0.15018873289588225], [0.50444311951671861, -0.40754788992155933, -0.16208704648717481, 
0.27135551172815403, -0.58826241721948191], [0.57225427257370376, -0.56156571112087428, 0.4700722795978669, 0.3054896681656098, 
-0.36549979170684183], [0.65271727350677422, 0.67451725701274701, -0.64476808403117258, 0.005172976465647558, 
0.45727969465956853]], [[-0.14218376576111236, 0.029851774613917068, -0.054750415729194583, -0.13690853385574153, 
-0.03275291328327113], [-0.11000813064537711, 0.088877377417084097, 0.035347678055292563, -0.05917676628067195, 
0.12828730602817015], [-0.1247962760201288, 0.12246533201629244, -0.10251259407862437, -0.0666206873008236, 
0.07970759691492188], [-0.14234351569156006, -0.14709762044749308, 0.14060996944913212, -0.001128114248844576, 
-0.099722808073548788]], [[0.17834846365200246, -0.037444627459248876, 0.068676282959168486, 0.17173146697383296, 
0.041083675994438434], [0.13798889757073687, -0.1114834990542321, -0.044338423877582002, 0.074228483776296469, 
-0.1609174142610599], [0.15653843445865898, -0.15361461063307871, 0.1285868577262318, 0.083565779566601106, 
-0.099981368302297757], [0.17854884626608888, 0.18451216616215843, -0.1763743694026319, 0.0014150521476790914, 
0.12508748460680658]], [[-0.35048912669399601, 0.073585914388261162, -0.13496214066590007, -0.33748545209215614, 
-0.080737346573201849], [-0.27117479574930903, 0.21908657593232947, 0.087133553862085406, -0.14587328604824512, 
0.31623375295378736], [-0.30762821312839894, 0.30188233543340359, -0.25269797421260864, -0.16422287303214539, 
0.19648267074685097], [-0.35088291717571751, -0.362601990835071, 0.34660965077754757, -0.0027808503718553881, 
-0.24582103115692333]], [[-0.076103597582434179, 0.015978107136056876, -0.029305058730306911, -0.073280039464309263, 
-0.017530936241696909], [-0.058881648411941219, 0.047571451838574336, 0.018919779285584023, -0.031674254674198217, 
0.068665543218948022], [-0.066796975865579714, 0.065549342399792945, -0.05486967633139455, -0.03565859962891587, 
0.042663343788965552], [-0.07618910343716756, -0.078733729212645626, 0.075261226017983601, -0.00060382106467290358, 
-0.05337644853330617]]], [[[-0.10009748845669773, 0.021015673968355086, -0.038544337865268126, -0.096383720841051429, 
-0.023058077986229056], [-0.077445814776909053, 0.062569747062511227, 0.024884794527494685, -0.041660492307092477, 
0.090314369334334552], [-0.087856681327118477, 0.086215694824417238, -0.072168950846973015, -0.046901018849617011, 
0.056114214020097773], [-0.10020995253959321, -0.10355684621188818, 0.098989534554025677, -0.00079419336235107981, 
-0.070204939196663457]], [[-0.58940352677737495, 0.1237464850070685, -0.22696042653373513, -0.56753576801497141, 
-0.13577276208753655], [-0.45602379307852858, 0.36842912001899253, 0.14652900770812771, -0.24530926271646117, 
0.53179763673450431], [-0.51732604507415747, 0.50766343268499059, -0.42495206232302524, -0.27616702822044392, 
0.33041703798686067], [-0.5900657484586217, -0.60977324526612564, 0.5828795675071663, -0.004676444693369192, 
-0.41338738261754909]], [[0.70538716143279301, -0.14809748810578791, 0.27162201065464148, 0.67921623509876172, 
0.16249031249009094], [0.54576078074093148, -0.44092910773766175, -0.17536318688137117, 0.29358155599583718, 
-0.63644550531254007], [0.61912617399038072, -0.6075621394782591, 0.50857471217723405, 0.33051155493226148, 
-0.39543695605102341], [0.70617969600503749, 0.7297652610728248, -0.69757940850658129, 0.0055966818961690817, 
0.49473431893268111]], [[-0.94271108317447228, 0.19792413451932062, -0.36300785423730991, -0.90773508182236295, 
-0.217159351442898], [-0.72937921881280021, 0.58927746276840665, 0.23436323893118613, -0.39235557688733425, 
0.85057435759960809], [-0.82742802536780724, 0.81197333027717677, -0.6796820866684079, -0.44171048608113539, 
0.52847970809232536], [-0.94377026197145564, -0.97529107041255747, 0.93227645149890126, -0.0074796570464987118, 
-0.66118516353100498]], [[-0.62677574940154646, 0.13159286015845043, -0.24135127287575139, -0.60352142487963911, 
-0.14438168561871631], [-0.48493882657015391, 0.3917900509755155, 0.15581995092135942, -0.26086355101218378, 
0.56551725117202867], [-0.55012806143042559, 0.53985277320045832, -0.45189693515839852, -0.29367791030922491, 
0.3513677424558016], [-0.62747996047541432, -0.64843704763077747, 0.61983812640663916, -0.0049729633333674755, 
-0.43959897550990373]]]])+(1.-msk_ref)*numpy.array([[[[-0.084836234092983673, -0.063704702872403166, 0.023394160969527032, 
0.041742700485197244, -0.092318749069784767], [-0.041155910536744542, -0.079376500466470379, 0.069383563952551089, 
-0.025808288143568937, -0.057804585754595877], [0.067322663840799338, -0.050935940700877751, -0.00096017480149906942, 
0.090779219975979944, -0.0082019477585116921], [0.042222863318603476, 0.020422672292478818, -0.078400481359387944, 
-0.013652691064998048, 0.075348831209193265]], [[0.36140988623246934, 0.27138768786405426, -0.099661201901465518, 
-0.17782760862364674, 0.39328606408749373], [0.17532783136714794, 0.33815093644628413, -0.29558014004971767, 
0.10994559791042692, 0.24625266532206078], [-0.2868005226741005, 0.21699162781873782, 0.0040904298674163128, 
-0.38672753351865979, 0.034941025352910195], [-0.17987314490716688, -0.087002396425305384, 0.33399300843094154, 
0.05816167557791295, -0.32099270796505402]], [[-0.21439334816043149, -0.16099093374902101, 0.059120404757285125, 
0.1054897994230941, -0.23330273818331601], [-0.10400689694562466, -0.20059581713177532, 0.17534223132520102, 
-0.065221250855199434, -0.14608049038776863], [0.17013404074594465, -0.12872243782760667, -0.0024264996285318172, 
0.22941212704834449, -0.020727499990828784], [0.1067032398665175, 0.051611026090206055, -0.19812927666739533, 
-0.034502311189541807, 0.1904173184444804]], [[-0.76961488607106099, -0.57791447448738409, 0.2122264704672889, 
0.37868021867875962, -0.83749454825743386], [-0.37335699465597827, -0.72008543302695316, 0.62943180160519496, 
-0.23412687929466486, -0.52438996326909648], [0.61073578778907911, -0.46207918843335888, -0.0087104858951433054, 
0.82352829290912299, -0.074406191613941852], [0.38303614593411683, 0.18526980573437093, -0.71123121122029997, 
-0.1238540865337735, 0.68354733996198058]], [[-0.12272169221623944, -0.092153417961307452, 0.033841330333130616, 
0.060383807649975793, -0.13354568634803848], [-0.059534973938537496, -0.11482379626577513, 0.10036829747673767, 
-0.037333538293457001, -0.083618475731579267], [0.097387057775248739, -0.073682488435147575, -0.0013889616591680253, 
0.13131864718690853, -0.011864705207099632], [0.061078397598303427, 0.029542859016628474, -0.1134119146831337, 
-0.019749596015373603, 0.10899748405113086]]], [[[-0.79863348902437969, -0.59970494525353391, 0.22022854500370298, 
0.39295849098221425, -0.86907257801140547], [-0.38743455290472267, -0.74723651034063743, 0.65316475150981346, 
-0.24295471653367082, -0.54416227330651723], [0.63376379784448389, -0.47950204854797113, -0.0090389178632566507, 
0.854579720038805, -0.077211703527486311], [0.39747866002369653, 0.19225546964127693, -0.73804843695221511, 
-0.12852404890887456, 0.70932073548375374]], [[-0.20217681289094538, -0.15181736825789072, 0.055751613159687574, 
0.099478792709122327, -0.22000871038842668], [-0.098080388797362389, -0.18916549107022265, 0.16535090196916249, 
-0.061504821599211917, -0.13775654993757064], [0.16043948386184903, -0.12138759178323009, -0.0022882331265731298, 
0.21633979357633437, -0.019546408147919796], [0.10062308903915929, 0.048670132979783821, -0.18683949870976127, 
-0.032536304757241431, 0.17956698233723689]], [[0.37643006251670602, 0.28266654621400961, -0.10380311632683918, 
-0.18521811489223666, 0.40963101268396923], [0.18261444702152321, 0.35220447197363247, -0.30786443546827302, 
0.11451493130499873, 0.25648691342788416], [-0.29871993764606758, 0.22600978871084049, 0.0042604279223322225, 
-0.40279990992200482, 0.036393172569533699], [-0.18734866355849639, -0.090618211546141803, 0.3478737406838176, 
0.060578871823631905, -0.33433314839918316]], [[0.5812482141886639, 0.43646733233607998, -0.16028309638398161, 
-0.28599654819472725, 0.63251402666141798], [0.28197620696566983, 0.54384131542323277, -0.47537556414000065, 
0.17682328258788846, 0.39604318368197228], [-0.46125548299325869, 0.34898324857224478, 0.0065785556684257182, 
-0.6219660745271357, 0.056194944748236025], [-0.28928634284934673, -0.13992419543226015, 0.53715420384796075, 
0.093540247103636481, -0.51624608340752354]], [[-0.1218551314477461, -0.091502705481339944, 0.033602370384887378, 
0.059957426316596549, -0.13260269533723321], [-0.059114586378386305, -0.11401300401432811, 0.099659578199622775, 
-0.037069919213163119, -0.083028030071308212], [0.096699389587962634, -0.073162202635222676, -0.0013791539417136764, 
0.13039138171518344, -0.011780926309689263], [0.060647111635689206, 0.029334251376441842, -0.11261109199099752, 
-0.019610140432652665, 0.10822783247741592]]], [[[-0.59328520531192452, -0.44550607564173661, 0.16360237747816228, 
0.29191921226124556, -0.64561267460288085], [-0.28781561432617964, -0.55510365210903134, 0.48522005278717079, 
-0.18048509217446446, -0.40424478872786829], [0.47080756077829777, -0.35621029574880142, -0.0067147900933267014, 
0.6348462863459412, -0.057358678304047152], [0.29527713483107981, 0.14282186678373002, -0.54827805803917784, 
-0.095477359505135612, 0.52693695414347641]], [[0.12938266744217228, 0.097155236490208713, -0.035678138959957584, 
-0.063661264467432865, 0.14079415637994674], [0.062766358539903672, 0.12105609675366458, -0.10581599568630604, 
0.03935989377453164, 0.088157042509952072], [-0.10267294299613051, 0.077681758826455422, 0.0014643504436152503, 
-0.13844624003393141, 0.012508686772381773], [-0.064393554730631006, -0.031146359167736632, 0.11956758236003361, 
0.020821546437543366, -0.11491354931915458]], [[-0.16229138283113692, -0.121866846549051, 0.044752938110824911, 
0.079853622184873838, -0.17660540461231089], [-0.078731095316679475, -0.15184693383349121, 0.13273048550540764, 
-0.04937115391913232, -0.11057994566109892], [0.12878799168081792, -0.097440254633296031, -0.0018368106265078076, 
0.1736602914986323, -0.015690293868757366], [0.080772171800503917, 0.039068491934953181, -0.14997981311259251, 
-0.026117544419479246, 0.14414201835323157]], [[0.31893386617242953, 0.23949185625322059, -0.087948154272072768, 
-0.15692776786425933, 0.34706367952115341], [0.15472178608192302, 0.29840850961467635, -0.26084100192324872, 
0.09702383898722898, 0.21731091925889398], [-0.25309324122331284, 0.1914888923175822, 0.0036096871215169839, 
-0.34127596426927237, 0.030834453423513885], [-0.15873289500701665, -0.076777121255450886, 0.29473925731218981, 
0.051326011716233542, -0.28326686475477719]], [[0.069251833389364539, 0.052002160593353122, -0.01909665787345043, 
-0.034074574032301415, 0.075359811731954704], [0.033595577290175149, 0.064795051832568853, -0.056637815930584499, 
0.021067310326672823, 0.047185893912157745], [-0.054955502792842816, 0.041579017699931935, 0.00078379086588402738, 
-0.074103093851415627, 0.0066952514537382034], [-0.034466531040934868, -0.016671031123496133, 0.063998327257142426, 
0.011144694210653387, -0.061507264682013472]]], [[[0.091085504673921791, 0.068397366682675223, -0.025117439277146856, 
-0.044817582732152486, 0.099119202305350165], [0.044187568220501509, 0.085223592036320711, -0.074494548305464231, 
0.027709397705589417, 0.062062630693313713], [-0.072281836617252043, 0.054688022333656515, 0.0010309039210046522, 
-0.097466267259745679, 0.0088061258126374761], [-0.045333115681888685, -0.021927062562815951, 0.084175676674559649, 
0.014658385878485595, -0.080899233572263585]], [[0.5363383089909517, 0.4027438626743749, -0.14789888859170799, 
-0.26389914203886228, 0.58364308946079668], [0.26018943082818941, 0.50182163893726106, -0.43864586588426235, 
0.16316110408321133, 0.36544306930883058], [-0.42561676702392509, 0.32201920081022967, 0.0060702662591941309, 
-0.57391012052786816, 0.051853065358969122], [-0.26693475205694889, -0.12911300978328494, 0.49565120429890003, 
0.086312898224021092, -0.47635854122060384]], [[-0.64187969728537131, -0.48199635253227857, 0.17700263480468623, 
0.31582957727644223, -0.69849317735779204], [-0.31138986400404378, -0.60057078954196463, 0.5249632011910228, 
-0.19526817000022553, -0.43735545787192026], [0.50937021838114571, -0.38538658095302031, -0.0072647815820272984, 
0.68684494144469888, -0.062056782702231299], [0.31946253879871667, 0.1545200449902904, -0.59318612830968787, 
-0.10329766875703894, 0.57009702852148991]], [[0.85783685581474456, 0.6441615731407685, -0.23655427076745766, 
-0.42208883173893413, 0.93349765323121836], [0.4161553995234889, 0.80262977622395504, -0.70158439943293305, 
0.26096515235193757, 0.58450147658661045], [-0.68074523688741417, 0.51504793542481142, 0.0097089803850525878, 
-0.9179304276066419, 0.082935471522779011], [-0.42694408468235029, -0.20650752799226216, 0.79276058329022536, 
0.13805164387390689, -0.76190327334621755]], [[0.57034583316556098, 0.42828058346510506, -0.15727669164039007, 
-0.28063215607515152, 0.6206500608843436], [0.27668722375205218, 0.53364056969685525, -0.46645901970543219, 
0.17350663618198101, 0.38861466418771345], [-0.45260378669238094, 0.34243742485401502, 0.0064551627379562729, 
-0.61029995502360557, 0.05514090503807078], [-0.28386024457061743, -0.13729965938085395, 0.52707888721806551, 
0.091785727450887866, -0.50656293709308775]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_array_rank3(self):
      arg0=Data(0.63085697321,self.functionspace)
      arg1=numpy.array([[[0.7511494295457346, -0.63812308385015082], [-0.79031758533022489, 0.72347714731302926]], 
[[-0.1549764196296024, -0.42229395140139703], [0.84788909172282212, 0.61113423428304059]], [[0.055651654082436774, 
0.84180242209460854], [0.67477676686322829, -0.31421937445641834]], [[-0.039714909406222576, 0.36656647335745873], 
[-0.54627624730718272, -0.40593534119731323]], [[-0.90428276936389351, -0.34475114430785903], [0.57493034903473594, 
0.067723576808715213]], [[-0.65528381944986314, 0.75340720688533924], [0.90833199251890773, 0.61485294800008217]]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[0.47386785555184341, -0.40256439721330978], [-0.49857735975627532, 0.45641060334069861]], 
[[-0.09776795500649571, -0.26640708398609037], [0.53489674602226489, 0.3855382932649753]], [[0.035108234048591057, 
0.53105692804367921], [0.42568762873594851, -0.19822748349360064]], [[-0.025054427539329675, 0.23125101586264965], 
[-0.34462217991287447, -0.25608714066681543]], [[-0.57047309080710695, -0.2174886634088331], [0.36269881979877788, 
0.04272389068051935]], [[-0.41339036693180603, 0.47529219013048912], [0.57302757147053218, 
0.38788426974474366]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_array_rank3(self):
      arg0=Data(numpy.array([-0.37351169248831639, 0.754681078152595]),self.functionspace)
      arg1=numpy.array([[[0.33151316835686417, -0.47169541761088807], [-0.27677524227489081, -0.10665081466068571]], 
[[-0.41169023354507717, 0.80275407203831062], [0.67347081713473345, -0.21720762663964099]], [[-0.57688106717229926, 
0.53185800711579634], [0.78853005774813067, -0.23925046278153528]], [[0.65670628554389765, 0.52680954318837214], 
[0.70580059327607159, 0.99072646018628818]], [[-0.47118853616110168, -0.077236143350451369], [-0.74397504936769643, 
0.43281688949676145]], [[0.061635190625851877, -0.97480843561904251], [0.69244102903953375, 0.35790333135525043]]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.12382404459513652, 0.17618375377082601], [0.10337878918095829, 0.03983532628917047]], 
[[0.15377111591233203, -0.29983803209891724], [-0.25154922474948371, 0.081129588247542628]], [[0.21547182376399163, 
-0.1986551844012841], [-0.29452519644741415, 0.089362845282144193]], [[-0.2452874761812168, -0.1967695240952857], 
[-0.26362477415380331, -0.37004791693713912]], [[0.17599442762262535, 0.028848602624097316], [0.27788337985840705, 
-0.16166216893346397]], [[-0.023021464367501947, 0.36410234863995655], [-0.25863482070490768, -0.13368107904170631]]], 
[[[0.25018671531734099, -0.35597960632222353], [-0.20887703824596029, -0.080487351793978881]], [[-0.31069482931669246, 
0.60582330857725819], [0.50825568237954977, -0.16392248585537059]], [[-0.43536122573941038, 0.40138317423423975], 
[0.5950887141370873, -0.18055779720047635]], [[0.49560380760385458, 0.39757319403447677], [0.53265435269432693, 
0.74768251312769196]], [[-0.35559707248320321, -0.058288655936067017], [-0.56146389237544325, 0.3266387168080685]], 
[[0.046514912113658609, -0.73566948118522346], [0.52257214235264771, 0.27010287198158583]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_constData_rank3(self):
      arg0=Data(-0.22015011451,self.functionspace)
      arg1=Data(numpy.array([[[0.21611300599883188, 0.85367468515937839], [0.20206252928537971, -0.33534565187777221]], 
[[-0.28753014455938852, 0.94369326713724155], [0.5960541281912437, -0.35610462964096889]], [[-0.43821404099900829, 
0.71729335722593879], [0.96696190055936571, -0.11902778733421071]], [[-0.52697654397718829, -0.74626763346059466], 
[-0.894797071002152, 0.51441543102202125]], [[0.13108682526670457, -0.12314148797673896], [0.97800726769710322, 
-0.31679390749747505]], [[0.60304956080461203, -0.17187084537541608], [0.20646553622807273, 
-0.79892531291287883]]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[-0.04757730301771268, -0.18793657969200497], [-0.044484088960328079, 0.073826383661274864]], 
[[0.063299794249785699, -0.20775418082244668], [-0.13122138457537647, 0.07839647499295023]], [[0.096472871305759725, 
-0.15791221473045161], [-0.21287677313481526, 0.026203981011481633]], [[0.11601394650058774, 0.1642909049613514], 
[0.19698967764421019, -0.11324861604513645]], [[-0.028858779593198887, 0.027109612678993505], [-0.21530841197499159, 
0.069742215011594805]], [[-0.13276142986625553, 0.037837386290304122], [-0.045453411442949655, 
0.17588349912259521]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_constData_rank3(self):
      arg0=Data(numpy.array([-0.6964528596908941, 0.14349836695029716]),self.functionspace)
      arg1=Data(numpy.array([[[0.18464657487749325, -0.13451960456249568], [-0.15051112471229877, -0.95360578587102784]], 
[[0.48599623445465157, 0.94785127929647262], [-0.80233042263825016, -0.29668787050614687]], [[0.97413159829837248, 
0.8251455931670626], [0.30707955779124552, -0.98124009531244316]], [[-0.19974203930633738, -0.76739425913226422], 
[-0.6100504802439064, 0.089384999831634726]], [[-0.94203847780455074, 0.14571372034187569], [-0.62694687714414532, 
-0.2985007731314655]], [[0.04558018528447616, 0.32028111128322179], [0.88398088330267788, 
-0.87923676066261702]]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[-0.12859763510555897, 0.093686563282038365], [0.10482390322117328, 0.66414147658765976]], 
[[-0.33847346728494832, -0.6601337340277007], [0.55878531726341296, 0.20662911584960766]], [[-0.67843673735016286, 
-0.57467500802253979], [-0.21386643617632811, 0.6833874704237165]], [[0.13911091447538965, 0.53445392628304045], 
[0.42487140152167191, -0.062252438746212092]], [[0.65608539180583625, -0.10148273722829852], [0.43663894546131565, 
0.20789171706735196]], [[-0.031744450386614234, -0.22306069585817731], [-0.61565101408823253, 0.61234695630883784]]], 
[[[0.026496481957886047, -0.019303343577517874], [-0.021598100604067388, -0.13684087298684725]], [[0.069739665988236249, 
0.13601511069079383], [-0.11513310540313063, -0.042574224911593309]], [[0.13978629355049932, 0.11840704511570777], 
[0.04406541506686313, -0.14080635126348953]], [[-0.028662656451781482, -0.11011982299251308], [-0.087541247672245096, 
0.01282660150569217]], [[-0.1351809831692968, 0.020909680911311457], [-0.089965853034773433, -0.042834373477766439]], 
[[0.0065406821536142948, 0.045959816434168722], [0.12684981316921548, -0.12616903931775481]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_expandedData_rank3(self):
      arg0=Data(0.11946137623,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-0.60009572891304597, -0.90370197398073504], [0.24596895409604413, -0.62521236806063096]], 
[[0.73989407342463687, 0.98218131541014264], [-0.8161448228849888, -0.76997935470965029]], [[0.93799867068875042, 
0.15725815651379604], [0.04143890929725047, -0.3138816191100573]], [[0.18267431352457963, -0.74895957924499701], 
[-0.18371856301252509, 0.46216104492375054]], [[0.19599041150492469, 0.47743431899460353], [0.6887260182789392, 
0.83062239870725207]], [[0.3689738891803549, 0.46162556151726752], [-0.81813499745096308, 
-0.45772317035683363]]])+(1.-msk_arg1)*numpy.array([[[0.67627092574222458, 0.29414155783672391], [0.18312599623788461, 
-0.83022275044559457]], [[0.74444175782501998, -0.83170336216878682], [-0.68427091786799332, 0.52224497857627394]], 
[[0.74612578255800099, 0.2171108033711453], [-0.90597048931563728, 0.99802670455100317]], [[-0.13764985206780667, 
0.34424923204205204], [-0.77946707677156035, -0.14045604097079534]], [[-0.67766328759702876, -0.0061437012722671547], 
[0.68403256260133172, -0.36106006739046248]], [[-0.73725728883864083, -0.70852286233785988], [0.94731906938117993, 
-0.29743929808334824]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.071688261645656878, -0.10795748151344513], [0.029383789766150491, -0.074688729924497987]], 
[[0.088388764275677739, 0.11733273164622091], [-0.097497783744775149, -0.091982793382250072]], [[0.11205461210232524, 
0.018786275800520176], [0.0049503491341168805, -0.037496730192166879]], [[0.02182252489550443, -0.089471742077198421], 
[-0.021947272376461796, 0.055210394466454836]], [[0.023413284286249073, 0.057034960806495873], [0.082276157988963622, 
0.099227294876975916]], [[0.044078128594395742, 0.055146424881768082], [-0.097735532737364253, 
-0.054680239863155122]]])+(1.-msk_ref)*numpy.array([[[0.080788255493456543, 0.035138555305591283], [0.021876483534055111, 
-0.099179552345630417]], [[0.088932036912806905, -0.099356428259745139], [-0.081743945562629497, 0.062388103869893227]], 
[[0.08913321282501406, 0.025936355365103254], [-0.10822848147735126, 0.11922564363988694]], [[-0.016443840765866784, 
0.041124487025840865], [-0.093116209717052936, -0.016779071954178975]], [[-0.080954588956841503, -0.00073393500913062064], 
[0.081715471314442442, -0.043132732552136767]], [[-0.088073770360212786, -0.084641116225251653], [0.11316803975713453, 
-0.03553250789390186]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank1_expandedData_rank3(self):
      arg0=Data(numpy.array([-0.28740085940932802, 0.24764677399374602]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.67242579391337953, -0.89317712926090165], [-0.99701004112330294, 0.89526514770181609]], 
[[-0.91740715313646759, -0.35134117664448827], [-0.049093269571241605, 0.15259130609547955]], [[0.88272014101965879, 
-0.4251717004763671], [0.022740311589840356, -0.60226254447714189]], [[-0.63411600184138606, -0.57221024361603012], 
[-0.72835035653650548, 0.47257931918100105]], [[0.63109086308110807, -0.78774731648430785], [0.032191936052698766, 
0.29192410722269302]], [[-0.8070392067870551, 0.25497980893334793], [-0.57544468016613104, 
-0.18247344197239346]]])+(1.-msk_arg1)*numpy.array([[[-0.72180000146745971, 0.67186892782174601], [0.15238074428723825, 
-0.38705902177094997]], [[0.029915034122554252, 0.44216489583112395], [0.42650164147091751, 0.11272393150830373]], 
[[0.36411532614042019, -0.93726403450581586], [0.037190266021924545, 0.10796202997472304]], [[-0.18544138361845919, 
-0.59619018146266511], [-0.18514933596332872, 0.1897906248393959]], [[0.72764377986819939, 0.18040101973592271], 
[-0.44782228345532848, 0.088378488666651434]], [[-0.17616843716837183, -0.57623381041958188], [0.6551520906908892, 
-0.97896984058814596]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.19325575105970497, 0.25669987455433962], [0.28654154265856674, -0.25729997284872091]], 
[[0.26366360423968577, 0.10097575611351045], [0.01410944786598865, -0.043854872510232658]], [[-0.25369452714697316, 
0.1221947121134333], [-0.0065355850941560207, 0.17309077287277921]], [[0.18224548389442138, 0.164453715778068], 
[0.20932851841968214, -0.13581970247169484]], [[-0.18137605641488502, 0.22639925575498199], [-0.0092519900875957569, 
-0.083899239298102793]], [[0.23194376160762203, -0.073281416219470444], [0.16538329562227197, 0.052443024042244026]]], 
[[[0.16652407861283194, -0.22119243466645738], [-0.24690632032355803, 0.2217095256973893]], [[-0.22719292191303272, 
-0.087008508967174381], [-0.012157789834123318, 0.037788744694037743]], [[0.21860279526282306, -0.10529240001640756], 
[0.005631564804836558, -0.14914837623702917]], [[-0.15703678219383158, -0.14170602087768536], [-0.18037361613346031, 
0.11703274385133573]], [[0.15628761633896529, -0.19508308164956931], [0.0079722291120638156, 0.072294063404704331]], 
[[-0.1998606560472859, 0.06314492711588536], [-0.14250701865500531, 
-0.045188959243998253]]]])+(1.-msk_ref)*numpy.array([[[[0.20744594074340214, -0.19309570726639358], [-0.043794356865585325, 
0.11124109549910483]], [[-0.0085976065160814647, -0.12707857106250103], [-0.1225769382982308, -0.032396954791484724]], 
[[-0.10464705765686452, 0.26937048901042554], [-0.010688514416362645, -0.031028380198311035]], [[0.053296013022000051, 
0.17134557052377317], [0.053212078274927077, -0.054545988686675741]], [[-0.20912544767797239, -0.051847408110423335], 
[0.12870450912770912, -0.025400053596093177]], [[0.050630960242988271, 0.16561009233529964], [-0.18829127390837958, 
0.28135677352084598]]], [[[-0.17875144183209754, 0.16638617252169238], [0.037736599741500496, -0.095853918086750867]], 
[[0.0074083616943633928, 0.10950071002585861], [0.10562175561331, 0.0279157179899234]], [[0.090171985880355757, 
-0.23211041452572834], [0.0092100494042988389, 0.026736448437056269]], [[-0.045923960418048117, -0.14764457512597506], 
[-0.04585163575840262, 0.04700103597573372]], [[0.18019863470097505, 0.044675730562783365], [-0.110901743820225, 
0.021886647608739072]], [[-0.043627545144267227, -0.14270244421653328], [0.16224630173485682, -0.24243872285882614]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_array_rank4(self):
      arg0=Data(-0.448440211752,self.functionspace)
      arg1=numpy.array([[[[-0.25515102245128873, 0.36730616433963759, 0.17450639786221234, -0.36562008126248435], 
[0.35434412138378724, -0.0087194181035530338, -0.82333920735532318, 0.93803882900265978], [-0.60591702633892441, 
0.63607654180945716, 0.31940940747376056, 0.15366531796524296]], [[-0.86048365345962585, -0.011426882553942974, 
-0.43114257438679471, 0.9724566228920053], [-0.25770702633276033, 0.46387524920783285, -0.13014386176362192, 
0.78555502852532433], [-0.71597477612394567, 0.93125380345187714, -0.01746761040089484, -0.30285112237192524]]], 
[[[-0.13998741907158574, 0.32762499088219132, -0.68181548853288021, -0.72915647255318494], [-0.063111239566042032, 
-0.34496963082412591, 0.52408052730259103, -0.97614011386580235], [0.21818049843376852, -0.99285280659954767, 
-0.94141588395163112, 0.23244192959824495]], [[0.38929568410850446, 0.33251835049575984, -0.28250822038803847, 
0.68001471362974253], [-0.0057615102952528474, -0.27448682595580909, -0.41348165426752881, 0.36856267355724026], 
[0.20140002479968566, 0.3740960197975931, 0.96136038238721211, 0.11315580646280576]]], [[[-0.084662037656419331, 
-0.77212322168240699, 0.34567399161637669, 0.45692298418094368], [0.64724676592952934, 0.79532036096776637, 
0.43505835701500573, -0.46844161826170061], [-0.68218528288427494, 0.34353392644124114, 0.4129843557441697, 
0.40791753970232913]], [[0.89919955957474107, 0.80499016523171996, 0.96080359935353998, -0.26014772593819702], 
[-0.15237087519497927, 0.92286473964411764, 0.62941743892544189, -0.3482537369458607], [0.32637149670430898, 
-0.58702804227088135, 0.48043802099880728, -0.424008055959604]]]])
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.1144199785366858, -0.16471485411412445, -0.078255686009334413, 0.16395874666197513], 
[-0.15890215282626996, 0.0039101377007078053, 0.36921840848979187, -0.42065433110914857], [0.27171755959530958, 
-0.28524229909924004, -0.14323602232297705, -0.06890970772720606]], [[0.38587547176620018, 0.0051242736321505207, 
0.19334166735313171, -0.43608865388890861], [0.11556619345853075, -0.20802031498107337, 0.05836174092744581, 
-0.35227446333440793], [0.32107188021380589, -0.41761165281441576, 0.0078331789069712287, 0.13581062144566713]]], 
[[[0.062775987851017831, -0.14692022028631643, 0.3057534820531857, 0.32698308295177891], [0.028301617634900023, 
0.15469825429463233, -0.23501878263844897, 0.43774047936118315], [-0.097840908917702488, 0.44523512282964284, 
0.42216873834556196, -0.10423630812898074]], [[-0.17457583901559048, -0.1491145995076017, 0.12668804617237148, 
-0.30494594217430571], [0.0025836928968120437, 0.12309093035463968, 0.18542180059512053, -0.16527832337373399], 
[-0.090315869767942716, -0.16775969833345258, -0.43111265344729277, -0.050743613811100416]]], [[[0.037965862093964173, 
0.34625110102956386, -0.15501411799745879, -0.20490263978026216], [-0.29025147676895774, -0.35665363108272113, 
-0.19509766174409979, 0.21006805848652566], [0.30591931271042971, -0.15405442671715883, -0.18519879194000163, 
-0.18292662788129241]], [[-0.40323724090261637, -0.36098996015444473, -0.43086296954577319, 0.11666070130641476], 
[0.068329227537208714, -0.41384965926406664, -0.28225608959185672, 0.15617097953927772], [-0.14635810309175751, 
0.26324697958006427, -0.21544772787021091, 0.19014226239889684]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_constData_rank4(self):
      arg0=Data(-0.417410053051,self.functionspace)
      arg1=Data(numpy.array([[[[-0.94576928300862351, -0.62048842948887883, -0.39889033328627965, -0.017179219367599252], 
[-0.096071312182654411, 0.091364138729949085, 0.10187595078927769, 0.47143692177797614], [0.46966210476308135, 
0.43389890996641811, 0.23972434500924034, 0.63268538631155091]], [[0.38944441855279255, -0.91092437832945938, 
0.90538426583677034, -0.52350775380487624], [0.17979340245138054, 0.7143507616587641, -0.40349151482927992, 
-0.82811672312040918], [0.078986528205921003, -0.9170053419240094, 0.61367718689934736, 0.76517759465324175]]], 
[[[0.043013142565961893, 0.13675292997260069, 0.17643812054635499, -0.76496364495945546], [-0.36356111513980127, 
0.69414565540107609, -0.5927406081713007, -0.063802585021933922], [-0.69540657661054039, -0.97942078046711467, 
0.68334764317920427, -0.84217667903769078]], [[0.86049775957003605, 0.81374832485360726, 0.85720321784346187, 
0.68990788631562938], [0.26635690263035317, -0.24450877678071303, -0.90021483752868448, 0.68721263455353809], 
[0.1715336960840641, 0.10045919693030037, 0.15564365826854853, -0.16002747995296263]]], [[[-0.86904748630712514, 
0.74810020405840127, 0.36169408128210412, 0.27880264117974929], [0.97015926227527594, 0.67344800575323349, 0.6377297902349004, 
-0.14473893506001301], [0.054955764189549949, -0.15602273858521687, 0.50576167221108159, 0.63985303207983679]], 
[[-0.69541655011200687, 0.87626714412326745, 0.96026810922009309, -0.048868982009782957], [-0.66540954681536535, 
0.56553627820198216, 0.87398572584185885, 0.43190709586551845], [0.15933540203546537, -0.5872633487310277, 
-0.44598472235855247, 0.1405540914022323]]]]),self.functionspace)
      res=outer(arg0,arg1)
      ref=Data(numpy.array([[[[0.39477360659474714, 0.25899810827055769, 0.16650083517860403, 0.0071707788676063939], 
[0.040101131514852277, -0.038136309994237734, -0.042524046023585463, -0.19678251052960069], [-0.19604168406525743, 
-0.18111376702790477, -0.10006335156795147, -0.26408924066497141]], [[-0.16255801540858286, 0.38022899308405611, 
-0.37791649443457365, 0.21851739928836489], [-0.075047573655471725, -0.29817718932109111, 0.16842141461056562, 
0.34566424533020779], [-0.032969770928757094, 0.38276724842065918, -0.25615502713991728, -0.31939282037773636]]], 
[[[-0.017954118120353448, -0.057082047754759051, -0.073647045257493549, 0.31930351562470272], [0.15175406435782798, 
-0.28974337484616608, 0.24741588870233444, 0.02663184039880389], [0.29026969603510189, 0.40882007993414554, 
-0.28523617599178797, 0.35153301227553668]], [[-0.35918041547249674, -0.33966673144740245, -0.3578052406356283, 
-0.2879744874273914], [-0.11118004885746713, 0.10206042148750134, 0.37575872309025155, -0.28684946224639074], 
[-0.071599889182503512, -0.041932678720149365, -0.064967227654944892, 0.066797078896802822]]], [[[0.36274915736339763, 
-0.31226454586356933, -0.15097474565623839, -0.11637502524563091], [-0.40495422913435619, -0.28110396780862668, 
-0.26619482557422797, 0.060415486561962321], [-0.022939088445824762, 0.065125459590036058, -0.21111000642884961, 
-0.26708108806536324]], [[0.29027385907487807, -0.36576271511544456, -0.40082556241285561, 0.020398404373257625], 
[0.27774863423692187, -0.23606052788662107, -0.36481042818956999, -0.18028236379838028], [-0.066508198616544778, 
0.24512962554879536, 0.18615850661967143, -0.058668690748757441]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_constData_rank0_expandedData_rank4(self):
      arg0=Data(0.0446296352099,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[0.83493992475885648, 0.26024474473146442, -0.27641842981279918, 0.39870398910379912], 
[-0.43615518465977599, 0.1307617326084376, 0.86325047901245844, -0.84740155321183575], [-0.34277425115768612, 
-0.99665619154375484, 0.62572624536880328, -0.36861771215142802]], [[0.16172507619941046, -0.055604708141027581, 
-0.90158191963756384, -0.67799237877879848], [0.44875857145027864, 0.91322040050478948, 0.35541011910030074, 
-0.42252559157082814], [0.71841452661281235, -0.42399254291293587, 0.011968355890833005, -0.53503912065223402]]], 
[[[0.21733911718324173, 0.82586341121125884, -0.20351620055918773, 0.26849828803684539], [-0.0027675040975294607, 
-0.90282176942342085, 0.45961809987326774, -0.11579856226490515], [0.46881150577395014, -0.84463688913836088, 
0.90015960010663676, -0.84682738008523284]], [[0.58676349652971305, -0.56400170636377922, -0.41853456169608139, 
-0.3251869433384682], [-0.99772483767357212, -0.50911148213419288, 0.49774853321201795, -0.66890484092032088], 
[0.81808185587295612, 0.15836639889620208, 0.88622045509309211, 0.4269146725362889]]], [[[-0.010073445030529093, 
0.68142450068881155, 0.72926994273213608, 0.78881989849172096], [0.31238844618913486, -0.15409616695845152, 
0.27529205688076686, 0.075112836227879232], [0.08126758577522053, -0.30328648172851302, 0.82803294830929541, 
-0.089323169102796474]], [[-0.082243386649075045, 0.51484339977082172, 0.62765845253957075, 0.67404265033697675], 
[0.4548897701434913, -0.8102518961352132, -0.29182113685584543, 0.5064938009939024], [0.80135108524261534, 
-0.58862292165723362, 0.5819953900635888, 0.0037514290888855939]]]])+(1.-msk_arg1)*numpy.array([[[[-0.024467085503008024, 
0.9104785955182606, 0.60676090282793549, 0.72485116145544759], [0.39311257256698462, 0.35984428315119699, 0.45151886370107963, 
0.42708476399446682], [-0.35235399139098145, 0.6862957160449068, 0.21487591789820559, -0.50613288972477966]], 
[[-0.81610963170031847, 0.1774050944192096, -0.0095511449550185201, -0.48048006558641121], [-0.47063578677355622, 
-0.16145416553119429, 0.74620701119135546, -0.42746436133938515], [-0.03424527257764054, 0.74238485619319361, 
-0.04847517230856413, 0.50122429886182962]]], [[[-0.20315484399875117, 0.96514809821932857, 0.29423326020356955, 
-0.20935538616946547], [0.88318603351275926, 0.22679185747668562, -0.055942671603035565, 0.8146375864496469], 
[0.15353310928014818, 0.25602474775840256, 0.27815841161798804, 0.42634852514419119]], [[-0.2872372834622503, 
0.10221917675293035, 0.67883549368623219, -0.77139665041407635], [0.09322742044113852, -0.16991436282799399, 
0.98682491498336566, -0.4934527011871821], [0.72697976062816294, 0.066156633283350308, 0.80418716417699621, 
0.94451153492052131]]], [[[0.9829866607602209, -0.78349886466420493, -0.81693247875887742, 0.62902855141793013], 
[0.44277257894516309, 0.91490629885114383, 0.69990829951854105, 0.75473359841841403], [0.44370819987613053, 
-0.34912347902903607, 0.82994105470259694, -0.40646982696858935]], [[0.066450599902513563, 0.6750221435230741, 
-0.89762862866476367, 0.61411954144575187], [0.32079212183711636, 0.25113487890755271, -0.51828827116657927, 
0.56840005789848536], [-0.93967160610378597, -0.87552179468722069, 0.63506723414018063, 0.18149549290406863]]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.037263064264134484, 0.011614628022648007, -0.012336453687827108, 0.017794013590417962], 
[-0.019465446786254285, 0.0058358484257236319, 0.03852655397306165, -0.037819222196111744], [-0.015297889788499957, 
-0.044480402258244657, 0.027925934072044113, -0.016451274025210866]], [[0.0072177311550662608, -0.0024816178402847114, 
-0.040237272185228455, -0.030258552539962], [0.020027931341123167, 0.040756693340729649, 0.015861823965338792, 
-0.018857163018635732], [0.032062578252193011, -0.018922632521904612, 0.00053414335746963825, -0.023878600777712686]]], 
[[[0.0096997655167207712, 0.036857982775527862, -0.0090828537902529535, 0.011982980649555929], [-0.00012351269831452854, 
-0.040292606228886273, 0.020512588133192261, -0.0051680475917088046], [0.020922886484875877, -0.037695836247034757, 
0.040173794583411318, -0.037793597058924149]], [[0.02618704080458218, -0.025171190412753206, -0.018679044811214136, 
-0.014512974656204769], [-0.04452809554518683, -0.022721459728799433, 0.022214335463494508, -0.029852979040382366], 
[0.036510694799414087, 0.0070678346122364398, 0.039551695626319508, 0.019053046101030778]]], [[[-0.00044957417701907556, 
0.030411726888801647, 0.032547051513649644, 0.035204744315963134], [0.013941782397195606, -0.0068772557185931447, 
0.012286184074760247, 0.0033522584804280956], [0.0036269427075339795, -0.013535565043624957, 0.036954808424787489, 
-0.0039864604528463124]], [[-0.0036704923445715679, 0.022977273121975131, 0.028012167773225339, 0.030082277600425492], 
[0.020301564502200407, -0.036161246552610743, -0.013023870884402581, 0.022604633574412541], [0.035764006609402162, 
-0.026270026269723468, 0.025974241952357285, 0.00016742491175261598]]]])+(1.-msk_ref)*numpy.array([[[[-0.0010919571006476659, 
0.040634327584364298, 0.027079517752815172, 0.03234984291719889], [0.017544470710073558, 0.01605971908939096, 
0.020151122177349014, 0.019060637220761574], [-0.015725430100517131, 0.030629127453172839, 0.0095898338311804256, 
-0.022588526236128465]], [[-0.036422675154037201, 0.0079175246482998327, -0.00042626411517895712, -0.021443650052730429], 
[-0.021004303480408573, -0.007205640510769315, 0.033302946700509012, -0.019077578511791909], [-0.0015283540228022688, 
0.033132365317225507, -0.0021634292568662524, 0.022369457616520563]]], [[[-0.0090667265787799795, 0.043074207547017337, 
0.013131523069492692, -0.0093434545139623043], [0.039416270498116329, 0.010121637867750704, -0.0024967010263083887, 
0.036356978311487327], [0.0068521266598083573, 0.011426291097153549, 0.012414108441064478, 0.019027779149446446]], 
[[-0.012819295179590958, 0.0045620045699353321, 0.030296180450720758, -0.034427151110086984], [0.0041607057658441197, 
-0.0075832160299289164, 0.044041635971707259, -0.022022614047303258], [0.03244484152178518, 0.0029525464101483089, 
0.035890579777669941, 0.042153205255006411]]], [[[0.043870336085885607, -0.034967268517301774, -0.036459398518094197, 
0.028073314786367955], [0.019760778679250912, 0.040831934368928353, 0.031236652087864882, 0.033683485178037674], 
[0.019802535100094696, -0.015581253512262571, 0.037039966517062137, -0.018140600101422457]], [[0.0029656660331254404, 
0.030125992024011561, -0.040060838251233959, 0.027407931109969495], [0.014316835375786994, 0.011208058034116069, 
-0.023131016475712668, 0.025367487237271863], [-0.041937200997473834, -0.039074218315171307, 0.028342818993410072, 
0.0081000776405420489]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.668020810252)+(1.-msk_arg0)*(-0.199373680103)
      arg1=-0.414108752084
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(-0.27663326409916661)+(1.-msk_ref)*numpy.array(0.082562385865791135)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.89803697641820968, -0.11561484456563709])+(1.-msk_arg0)*numpy.array([-0.32840922337831913, 
-0.52183272187384655])
      arg1=0.777737979352
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.69843746342312674, -0.089918055595606172])+(1.-msk_ref)*numpy.array([-0.25541632579090445, 
-0.40584912667006418])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[0.17536854353567821, -0.78928042811432308, -0.97636831926807344, -0.27888945307416857, 
0.010622511883872532], [0.75263624986742128, -0.5592719942188058, -0.013841027001748607, -0.052258497265722292, 
0.58265643840157866], [0.98297043285336128, 0.77043885820702096, -0.18809135815129996, 0.91236177100903726, 
0.6082829915489476], [-0.27330126762450457, 0.20962667467243068, -0.81369505041848655, 0.49080308139066298, 
-0.1249881775189563]])+(1.-msk_arg0)*numpy.array([[0.63107319493892855, 0.84913314407214524, -0.47663244422210194, 
0.54298257454386545, -0.25310528901917007], [-0.74538586542084051, -0.63889727230382065, -0.34508951051308467, 
-0.30076076768983873, -0.5692494007765323], [0.46403709770754098, 0.07981428366460297, 0.071007166588557213, 
-0.58626153560215855, -0.90404379525966672], [0.93173030530645828, 0.32579774985788545, -0.65037365393573832, 
0.99004422028261851, -0.44078045182846037]])
      arg1=-0.234522178058
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.041127812792920319, 0.18510376510025423, 0.22898002482198859, 0.06540576197247587, 
-0.0024912146234572496], [-0.17650989260462979, 0.13116168621127111, 0.0032460277990155029, 0.012255776600817243, 
-0.13664585699370058], [-0.23052836687979938, -0.18068499908755373, 0.04411159498760945, -0.21396906971427798, 
-0.14265585205395143], [0.064095208549426297, -0.049162104323322527, 0.19082953549950035, -0.11510420764552287, 
0.029312499623298231]])+(1.-msk_ref)*numpy.array([[-0.14800066019136418, -0.19914055440939471, 0.11178087895227631, 
-0.12734145602979641, 0.059358803658881971], [0.17480951665245678, 0.14983557985627627, 0.080931143630643076, 
0.070535070313143822, 0.1335016093285635], [-0.10882699085428094, -0.01871821964519536, -0.016652755366105595, 
0.1374913322412929, 0.21201831992449846], [-0.21851142056350839, -0.076406797903203996, 0.15252704587282326, 
-0.23218732691483107, 0.10337279160838554]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank3_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.25740843478106368, 0.73423343206754943], [-0.6427051620706361, -0.89580613442810764]], 
[[-0.22740639543899688, -0.84905678784073912], [0.84197910761975647, -0.25666510559167866]], [[0.28439305888670252, 
0.6223153710205096], [-0.77856279651648896, 0.7399236312518287]], [[0.048181875839945487, -0.38581915202623618], 
[0.7742861694449017, -0.83641627537952057]], [[0.48993888851279577, -0.8103853288250773], [0.11418354207494752, 
0.35973611692648522]], [[-0.1081267874550631, -0.69353637113732813], [-0.82769375941263057, 
-0.18143269032927378]]])+(1.-msk_arg0)*numpy.array([[[0.58533346116207041, 0.28874612633378605], [-0.13007576734424164, 
0.18467312201063169]], [[-0.83082036907737877, -0.060060583903163822], [0.34197864311618154, 0.6939585737259415]], 
[[-0.79098633992461753, 0.84652008035450477], [0.14677195874921889, -0.42095550171758123]], [[-0.96050106947722624, 
-0.43788444443669405], [-0.39609118372006691, -0.90094996813009498]], [[0.086117360299087897, -0.32697201394953446], 
[0.78954648940984051, -0.22063927607971023]], [[0.2610626132674041, 0.34277618940295862], [0.30158294235422911, 
-0.99775481707329794]]])
      arg1=-0.613278501288
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.15786305910143908, -0.4502895788139839], [0.39415725856478695, 0.54937864356673072]], 
[[0.13946345337815058, 0.52070827435543277], [-0.51636768523691223, 0.15740719129020936]], [[-0.17441214893076726, 
-0.38165263806798844], [0.47747582500628227, -0.45377925564174931]], [[-0.02954890860436972, 0.23661459132288482], 
[-0.47485306156525126, 0.51295611981770339]], [[-0.30046898726987104, 0.49699189992768461], [-0.070026311555487289, 
-0.22061842664786538]], [[0.066311834159534969, 0.42533094627986845], [0.50760678829806782, 
0.11126876840979985]]])+(1.-msk_ref)*numpy.array([[[-0.35897242781523425, -0.17708179161072052], [0.079772671650772412, 
-0.1132560554948694]], [[0.5095242707873775, 0.036833864882618791], [-0.20972814972282014, -0.42558987405065324]], 
[[0.48509491708830665, -0.51915256619006867], [-0.090012086892835644, 0.25816295920232651]], [[0.58905465637458332, 
0.26854511582149565], [0.24291420752526091, 0.55253324619036059]], [[-0.052813925659109515, 0.20052490667811296], 
[-0.48421188772252538, 0.13531332455944978]], [[-0.16010408820698105, -0.21021726771428267], [-0.18495433490104857, 
0.61190157886766627]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank4_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[0.51373130732142314, -0.49826403224687299, -0.14925439016692477, 0.16168858249137918], 
[0.85919417540887144, -0.064282795632149003, 0.5515090118931254, -0.44099930119256969], [-0.71487224437524155, 
0.0042457584859505459, 0.64173479370803466, 0.46078959800040109]], [[-0.40754479414590783, 0.725128391302996, 
0.72850950188907149, -0.56409961600821523], [-0.022347866841244679, -0.32715203968889051, -0.0021558078400554237, 
-0.21285177956047807], [-0.9534140635323769, -0.42255393690504173, 0.27521318354608226, 0.57194316501709985]]], 
[[[0.98916441835287316, 0.48416942773906935, 0.68388501815994296, -0.097292654680867008], [0.24471140330887642, 
0.71914389963411529, -0.44963477073130309, 0.527935384037723], [-0.68654030358372786, 0.88089482181212042, 0.22551553093490395, 
0.50153512068782025]], [[-0.050412827743439026, -0.42713936461634372, -0.7302549257426092, -0.15609649997827968], 
[-0.41466745825868179, -0.39687267255639691, -0.35974562857433545, 0.60426450876362203], [0.26980797832737724, 
0.06245476882456602, -0.8699550919775223, -0.26835386848971177]]], [[[0.44368144640849461, -0.45077538942190087, 
-0.4844355295838132, 0.50519709918605082], [-0.86012856869553977, -0.63594022089366686, 0.66789619211143192, 
0.75734366205209036], [-0.50346727384050727, -0.072080239588779182, 0.83356371123426221, 0.1323134254267162]], 
[[0.68690502648922536, -0.34609252078420827, -0.394197615091467, -0.1394901997472644], [0.7938132612482065, 
0.023217155424228419, 0.073861196778270521, -0.39694336802580144], [0.002027476811779616, 0.85655908725606755, 
0.26981208768492859, -0.47583056631722642]]]])+(1.-msk_arg0)*numpy.array([[[[0.99538415389773904, 0.23823136326685446, 
-0.95985216090673009, 0.81660962284280081], [-0.13147899307334221, 0.96465576217747939, -0.053627611872602898, 
0.63155894105576849], [-0.31614712525834432, -0.31560053874951, 0.3475495545383982, -0.49451553112039814]], 
[[0.25647863330323406, -0.84355062453784013, -0.023875899680008139, 0.98678089365882671], [-0.18117294237811077, 
0.96325207513942224, 0.44001398623201249, 0.22639152697392451], [0.041655855711577594, 0.43114749731298319, 
-0.24049119940790997, -0.3179433057487997]]], [[[-0.018558917849502032, -0.71289989545026167, -0.029025319641343916, 
0.84936380123980859], [0.5757942789674213, -0.4399905546039673, -0.2984122303891934, 0.69320778125482319], 
[-0.97557323408510177, 0.97758894999817825, -0.13294946758508241, -0.12124450243022733]], [[0.23090968905588616, 
0.43492506953951815, 0.023440743934802777, -0.74633022318926501], [-0.26926132428283522, -0.43824040687843446, 
0.91308299706660057, -0.9114403702994931], [0.65526167223687182, -0.67644618397949108, -0.32545238214563077, 
-0.95407218158865947]]], [[[0.54539062662972126, 0.73588406488339753, 0.84117839560109031, 0.37807152717999637], 
[0.68558935225302253, 0.43008087372839232, -0.27646314630559865, 0.9023041381229171], [0.62363434616615621, 
-0.42667050127974804, 0.60746106586901494, 0.8106742973298775]], [[0.42330331017042222, -0.74611903242880073, 
-0.5622261508844506, -0.27517600179999246], [0.6470216876552195, -0.66974562018521988, -0.6884238802306839, 
-0.10904474913681272], [0.30444409123221572, -0.89746722334726736, -0.94500306008683066, -0.79495475061134968]]]])
      arg1=-0.76173064645
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.39132488082748135, 0.37954298338612641, 0.11369164310733181, -0.12316314846472041], 
[-0.65447453466015737, 0.048966175472481346, -0.42010131615227608, 0.33592268278135434], [0.54454009683701776, 
-0.0032341243561731046, -0.48882905926059128, -0.35099755836221841]], [[0.31043935950203905, -0.55235251826638099, 
-0.55492801381883072, 0.42969196516405778], [0.017023055055756787, 0.2492017346796101, 0.0016421448996271027, 
0.16213572364260689], [0.72624471094890908, 0.32187228351861241, -0.20963831621408227, -0.43566663682105666]]], 
[[[-0.75347685183713764, -0.36880669118294196, -0.52093617698035022, 0.074110796744880206], [-0.18640417543612378, 
-0.54779394755877064, 0.34250058457549182, -0.40214456136681187], [0.52295878926271877, -0.67100458207328417, 
-0.17178209116352866, -0.38203467169884381]], [[0.038400995866375444, 0.32536514433339164, 0.55625755665911503, 
0.11890348783701614], [0.31586491104110764, 0.30231007742466937, 0.27402927021144119, -0.46028679488722324], 
[-0.20552100574864393, -0.04757371143061348, 0.66267145459438881, 0.2044133657219899]]], [[[-0.33796575499055326, 
0.34336942878803367, 0.36900938911316017, -0.38482411294758651], [0.65518629066244971, 0.48441515556480208, 
-0.50875699817844355, -0.57689187727964608], [0.38350645196888022, 0.05490572749822184, -0.63495102461562225, 
-0.10078719108428839]], [[-0.52323660987729848, 0.26362927958841792, 0.3002724041726178, 0.10625396002690435], 
[-0.60467188865107102, -0.017685218810024477, -0.056262337169472605, 0.30236392833027997], [-0.001544391222498996, 
-0.65246730725806967, -0.20552413597222802, 0.36245472488142572]]]])+(1.-msk_ref)*numpy.array([[[[-0.75821461501447551, 
-0.1814681303458934, 0.73114880702378227, -0.62203657590522621], [0.10015157838833413, -0.73480785732503751, 
0.040849795459280198, -0.48107780044160203], [0.24081895409630472, 0.24040260240158956, -0.26473914685189626, 
0.37668763519983867]], [[-0.19536763514665006, 0.6425583625423954, 0.018187004497824698, -0.75166124803111245], 
[0.13800498251690224, -0.73373862589012495, -0.33517213817949237, -0.1724493641926193], [-0.031730541899602255, 
-0.32841826184345962, 0.18318951679049039, 0.24218715982243991]]], [[[0.014136896490911101, 0.54303769821536818, 
0.022109475493814833, -0.64698643738951311], [-0.43860014833998712, 0.33515428959031413, 0.227309741162906, 
-0.52803761133931226], [0.743124030258829, -0.74465946284435602, 0.10127168388875005, 0.092355653214669153]], 
[[-0.17589098671607722, -0.33129575437758918, -0.017855533030723045, 0.56850260337503], [0.20510460260991051, 
0.33382114843196126, -0.69552330161792086, 0.69427206246873618], [-0.4991328971868112, 0.51526978901124132, 
0.24790705344043945, 0.72674601964136154]]], [[[-0.4154407545903539, -0.56054544445578403, -0.64075136306087788, 
-0.28798866880310592], [-0.52223442049083835, -0.32760578197085055, 0.21059045115492694, -0.6873127144267569], 
[-0.47504139365348436, 0.32500799676090997, -0.46272171039752807, -0.61751545656537676]], [[-0.32244310410048294, 
0.56834173290053736, 0.42826488936423129, 0.20960999373859718], [-0.49285624840469228, 0.51016576422065252, 
0.52439356731964248, 0.083062727251947591], [-0.23190439442215702, 0.68362828820787835, 0.71983779185704111, 
0.60554139608157376]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.771414181912)+(1.-msk_arg0)*(0.183128004107)
      arg1=numpy.array(-0.536035695845)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(0.41350553778552301)+(1.-msk_ref)*numpy.array(-0.098163147110422982)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.84211476238897176, -0.63311767547216014])+(1.-msk_arg0)*numpy.array([0.7228079105361378, 
0.53424529529876619])
      arg1=numpy.array(-0.42076964503)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.35433632964514156, 0.26639669957079831])+(1.-msk_ref)*numpy.array([-0.30413562794134791, 
-0.22479420326194333])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[0.3799622126195461, 0.37236705072111187, 0.6232161190449097, -0.67341084973043053, 
0.38354033302504886], [0.52272884055284385, 0.42129933951125409, 0.27778957298543006, -0.62481768620975697, 
-0.43452070079407457], [0.57971563148932437, 0.58259832375302456, 0.10661588551405843, -0.14869063205343003, 
0.013692674452755238], [-0.65472061118789893, -0.62775351208846275, 0.037602772251926764, -0.046802628906359445, 
-0.75513434598744889]])+(1.-msk_arg0)*numpy.array([[0.010910815632426019, -0.74920910177769695, 0.16281908062279138, 
0.83277196168888179, 0.34638945889459172], [-0.85272886856179153, -0.7706397301146295, -0.7113468219626875, 
-0.76118910476955093, 0.68080399454121632], [0.69545723388366931, 0.64141052030309353, 0.98073714527561151, 
0.85583897830349431, -0.26940020551838795], [0.63045657920698028, -0.28809893405512432, -0.95967764455003657, 
-0.22259961891726254, 0.60520236510218428]])
      arg1=numpy.array(0.885486966476)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.33645158702807043, 0.32972617015873018, 0.55184975071215336, -0.59629653051996545, 
0.3396199660116257], [0.46286957531076389, 0.37305507412225103, 0.24597904630159007, -0.55326791756256188, 
-0.38476241721726179], [0.513330635946322, 0.51588322237419026, 0.094406977042018447, -0.13166361672042198, 
0.012124684764116527], [-0.5797465678902225, -0.55586755311399938, 0.033296764732454416, -0.041443117893403977, 
-0.6686616213104225]])+(1.-msk_ref)*numpy.array([[0.00966138503613812, -0.66341489478949855, 0.14417417378512093, 
0.73740871812233011, 0.30672335117590771], [-0.75508029904947116, -0.68239143686524806, -0.62988833949223244, 
-0.67402303129713126, 0.60284306389118758], [0.61581831634558604, 0.56796065588911349, 0.86842995968063885, 
0.75783426069005966, -0.23855037075254473], [0.55826108381695705, -0.25510785116150159, -0.84978204626764597, 
-0.19710906129380706, 0.53589880637856058]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank3_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.4463432675487522, 0.76019878661702078], [-0.085592688955817842, 0.82447078133220941]], 
[[0.43940759438778776, -0.091163458108980633], [0.079709332561558099, -0.86851544097558375]], [[0.087497133213726874, 
-0.92480764823561001], [-0.28404925251082513, -0.1517033585351546]], [[-0.95839918828928683, 0.15658266453206582], 
[0.15208993384197234, -0.96379349512777934]], [[0.93942990915614644, -0.75969946811255618], [-0.8659092222707192, 
-0.60537319805077083]], [[0.327337965548961, -0.11291815862827148], [-0.84266681171659097, 
0.21912726347102573]]])+(1.-msk_arg0)*numpy.array([[[0.56292885389644121, 0.18488731830320915], [0.54430645574055037, 
0.59866881602820388]], [[0.91497718043284904, 0.0089825862396000211], [-0.60272668870203394, 0.68427682734608464]], 
[[0.45077649549700283, -0.26629280817811218], [0.95186464074863908, 0.65977924770938334]], [[0.0025375671708003367, 
0.57653705835051272], [0.51442792009275506, 0.13344738186607263]], [[0.75394596315267992, 0.82486677200307179], 
[-0.210421866790929, 0.51981409505314136]], [[-0.56875483910531988, -0.50228072629755971], [0.67568239247773842, 
-0.21188179097900317]]])
      arg1=numpy.array(-0.325283555839)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.14518812519300994, -0.24728016445519885], [0.027841894217361048, -0.26818678743701413]], 
[[-0.14293206476507503, 0.02965397381625948], [-0.025928135129168733, 0.2825137909415249]], [[-0.028461378617470667, 
0.30082472028508633], [0.092396550890101722, 0.049346607897018058]], [[0.31175149587985107, -0.050933765901719386], 
[-0.04947235448741924, 0.31350617518956198]], [[-0.30558110131172256, 0.24711774435657044], [0.28166603085391267, 
0.19691794647151911]], [[-0.10647765739483614, 0.036730420157384688], [0.27410565690259836, 
-0.071278495443101927]]])+(1.-msk_ref)*numpy.array([[[-0.1831114992797489, -0.060140804327185715], [-0.17705393938935302, 
-0.19473712124751616]], [[-0.29762703076264446, -0.0029218875926466342], [0.19605708050000109, -0.22258399957729288]], 
[[-0.14663018134386127, 0.086620671538501434], [-0.30962591502003084, -0.21461533976362016]], [[-0.00082542887249798121, 
-0.18753802441315198], [-0.16733494307057895, -0.043408238890787147]], [[-0.24524622380468517, -0.2683155966905113], 
[0.068446773056011934, -0.16908697721406382]], [[0.18500659646475776, 0.16338366067941362], [-0.21978837124289144, 
0.068921662387163921]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank4_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[0.31403475340174825, -0.44529723351568418, 0.51246391185524365, 0.015942733483527372], 
[-0.19794432119927841, 0.62979766487987732, -0.84218684969223756, 0.15189290257478816], [0.86628145613116603, 
0.38957710836696569, 0.34806921903528654, 0.36452391358710057]], [[-0.64769996681692077, -0.047944207922439253, 
0.94071577948782714, 0.52995994246574862], [0.26288341509963287, -0.15842925139810515, 0.91677042237312678, 
0.15971871842058727], [0.44159924798644834, -0.14690277912758298, 0.77053530275794735, -0.35986252789064088]]], 
[[[-0.74464208984073332, 0.72496627953920334, 0.26229282682528532, -0.068267617453663831], [-0.76150062127382534, 
-0.20925073378676129, 0.63271920674188653, -0.96019340322931912], [-0.51401548152996024, -0.73307789292371095, 
0.455130321312188, -0.42605394012282183]], [[-0.035194814480391523, 0.064715103894955606, 0.90184393500337712, 
0.24845297240730058], [0.81903131284113417, -0.23449420225909279, 0.62678067988682762, -0.089125260315905397], 
[-0.41604544571592372, -0.44637819653861577, -0.41262781940722992, 0.44373825265876921]]], [[[-0.55741425141821543, 
-0.63116860073803238, -0.66473958174575998, -0.93061641622253921], [0.52565065522145038, 0.0082169278534585111, 
-0.99664732047156646, 0.57986628098319626], [0.72377726280554611, -0.064307043837671163, -0.46706787517406378, 
-0.057762943884072993]], [[-0.51171804164063572, -0.76248011577579433, -0.42796445301156694, 0.54372186320210147], 
[-0.28113071911784271, 0.35689467924681773, -0.99216842853938347, -0.24168176746071879], [0.23856810831436648, 
-0.36199751109310641, 0.42562845583970299, 0.93737052769639173]]]])+(1.-msk_arg0)*numpy.array([[[[0.82548941551363519, 
-0.68138599365956676, -0.51539648999145315, 0.27055203069465716], [-0.013577166973204546, 0.74914204467550438, 
0.22588966745390393, 0.44467674134569868], [0.45398425526542119, 0.55326654311637169, -0.15622051962428807, 
0.22712387279217716]], [[-0.76302079685336022, -0.010006803251566554, 0.90680167826394986, 0.93372627404849129], 
[-0.86287658390765731, -0.83203302829617942, -0.82197141311309196, -0.10756454509262281], [0.89889902656412257, 
0.74650286195648552, 0.48209134818193422, 0.49202847344226019]]], [[[-0.8236922495556962, 0.4402185595395165, 
0.59617768874750054, 0.43160190247889241], [0.090939262879082605, -0.56163456612849871, 0.65668545765448738, 
0.69677153031341676], [-0.75813962516501898, 0.56072548596055438, 0.49834724021091015, -0.55130227447148217]], 
[[-0.23707881592363633, 0.33187755120540041, 0.51813023043316853, 0.63995125436627198], [0.76663867303590205, 
-0.39574894154855045, 0.93322035844601747, -0.28350509209773556], [0.14854930412492307, -0.24385741736058764, 
0.44739292318740342, 0.877983527645539]]], [[[0.046091094481068406, -0.91843340666079021, -0.82002661750842343, 
0.86971525588010068], [-0.91383310935561468, -0.27915760973248127, -0.48475397039245682, 0.12823696714625843], 
[0.90996461848980581, 0.4084380779897745, 0.89206120622010365, -0.32816874514194683]], [[0.60341747123153389, 
0.38813934596379518, 0.84260955813589744, 0.69588078179445301], [-0.39269177831721969, 0.55534794867574577, 
-0.53195052385209407, -0.38118771487043079], [0.63206575369569284, -0.7867420000743337, -0.63506205778412328, 
0.50965870349872211]]]])
      arg1=numpy.array(-0.185196674525)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.058158192015161848, 0.082467566822116423, -0.094906612289467437, -0.0029525412239815281], 
[0.036658630027138955, -0.11663643315912228, 0.15597020389136565, -0.028130060440742365], [-0.16043244487783462, 
-0.072148384940478288, -0.0644612618697155, -0.067508616581029787]], [[0.11995187994419834, 0.0088791078699525343, 
-0.17421743403397827, -0.09814681897591386], [-0.048685134264126427, 0.029340570506353563, -0.16978283352603138, 
-0.02957937551082641], [-0.081782712199661795, 0.027205906172852647, -0.1427005756745906, 0.066645343451368841]]], 
[[[0.1379052387495647, -0.13426134411314417, -0.048575759279704027, 0.012642935730137226], [0.14102738270834209, 
0.038752540039144058, -0.11717749299645085, 0.1778246251785443], [0.095193957833518195, 0.13576358793698434, 
-0.084288621982337525, 0.07890377287885679]], [[0.0065179626022793371, -0.011985022032860851, -0.16701849770282004, 
-0.046012664265588428], [-0.15168187546970907, 0.043427546453686856, -0.11607769757131881, 0.016505701826646471], 
[0.077050232997701029, 0.082667757579247422, 0.076417299970563096, -0.08217884875176823]]], [[[0.10323126569528213, 
0.11689032592103916, 0.12310755996419938, 0.17234706554243087], [-0.09734875330869848, -0.0015217477132692136, 
0.18457576942520409, -0.10738930690704496], [-0.13404114216811672, 0.011909450667245447, 0.086499417259515418, 
0.010697505118082368]], [[0.094768479606095093, 0.14120878183282187, 0.079257593512488803, -0.10069548093135777], 
[0.052064474287338569, -0.066095707752040392, 0.18374629353380867, 0.044758659626956847], [-0.044182020307449242, 
0.067040735240631366, -0.078824974604560685, -0.17359790452675691]]]])+(1.-msk_ref)*numpy.array([[[[-0.15287789460839482, 
0.12619042009340337, 0.095449716008077043, -0.050105336370532542], [0.0025144461729029381, -0.13873861542047528, 
-0.041834015221934556, -0.082352653735666539], [-0.084076374361690728, -0.10246312391088253, 0.028931520726925779, 
-0.04206258594626329]], [[0.14130891417036545, 0.0018532266848122476, -0.16793665526782495, -0.17292300087004153], 
[0.15980187386485956, 0.15408974993509877, 0.15222637226284458, 0.019920596047906927], [-0.16647311045309063, 
-0.13824984755745015, -0.089281714500383358, -0.091122037052930288]]], [[[0.15254506544941565, -0.081527013290735459, 
-0.11041012538180912, -0.079931237057588811], [-0.016841649068926021, 0.10401285394507391, -0.12161596296628709, 
-0.12903977031747296], [0.14040493740590096, -0.10384449532109434, -0.092292251645580928, 0.10209934788997602]], 
[[0.043906208309291206, -0.061462618832613371, -0.095955995646896192, -0.11851684416649069], [-0.14197893280821403, 
0.073291387921428475, -0.17282930698287327, 0.052504200267295825], [-0.027510837126881686, 0.045161582753342373, 
-0.082855681580154411, -0.16259962960734572]]], [[[-0.0085359174230937845, 0.17009081268589335, 0.15186620258422989, 
-0.16106837317232078], [0.16923885292315027, 0.051699060990696315, 0.089774823279287527, -0.023749059866609593], 
[-0.16852242127937361, -0.075641373792932301, -0.16520676886438154, 0.060775760283205027]], [[-0.11175090902213367, 
-0.071882116124654588, -0.15604848808942492, -0.12887480665392315], [0.072725211457507086, -0.10284859329881565, 
0.098515468029035644, 0.070594697163641584], [-0.11705647566533781, 0.14570200212261239, 0.11761138121837963, 
-0.094387097030490988]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg0.isLazy():
        arg0=arg0*1j
        arg1=numpy.array(arg1*1j)
        ref=-ref
        res=outer(arg0, arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.791652550411)+(1.-msk_arg0)*(0.27657970676)
      arg1=Data(0.524788097784,self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(0.41544983603581132)+(1.-msk_ref)*numpy.array(0.14514573819592874)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.2777005478551926, -0.11730389760971582])+(1.-msk_arg0)*numpy.array([0.14278735863381065, 
-0.59771418621042871])
      arg1=Data(-0.929492640069,self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.25812061537453374, 0.10903310947963972])+(1.-msk_ref)*numpy.array([-0.13271979894502137, 
0.55557093694743187])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[0.13317664602533652, -0.8278119096032821, 0.81800967280777881, -0.49731655159751642, 
-0.46639600919465596], [0.54488677268272023, 0.29640062843549808, -0.067602651839557826, 0.070422007044289581, 
0.10857751872635713], [-0.10303795811028849, 0.92312867631464468, 0.83674986777882965, 0.56457088498604824, 
0.24440215677678134], [0.88430294741407822, 0.78808073457631256, -0.48973872679301778, 0.44081702009759072, 
-0.80189819054052025]])+(1.-msk_arg0)*numpy.array([[-0.8024246446648049, 0.150768401322257, 0.36988293952586004, 
-0.03966117250769452, 0.56627044745398591], [0.26077699778899999, 0.62989660003720349, -0.99418330021016099, 
-0.90421489639872443, -0.2575086193377818], [-0.20703755181091976, 0.57518613221448489, -0.29410545024012147, 
0.089755770194008822, 0.38574047734125716], [-0.34763454259115556, 0.48021712485292833, -0.84799225914637599, 
0.96309475859407079, -0.78914836659374887]])
      arg1=Data(-0.355722959141,self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.047373990612636169, 0.29447170209642776, -0.29098482141732834, 0.17690691536417824, 
0.16590776852238098], [-0.19382873517560922, -0.10543650863839409, 0.024047815358161487, -0.025050724734458879, 
-0.038623516257551525], [0.036652967362861402, -0.32837806440676748, -0.29765113902731405, -0.20083082585221726, 
-0.086939458429134253], [-0.31456686123144295, -0.28033841094567619, 0.17421130910086821, -0.15680873482893182, 
0.28525359726906774]])+(1.-msk_ref)*numpy.array([[0.28544086908801036, -0.053631781863345038, -0.1315758537839937, 
0.014108389647447693, -0.20143539924255741], [-0.09276436532946776, -0.22406868251823067, 0.35365382547954732, 
0.32164999864653043, 0.091601728075207015], [0.073648010583535067, -0.20460691300833245, 0.10462006105897831, 
-0.031928188173412579, -0.13721674406038065], [0.12366158819023207, -0.17082425668297477, 0.30165031575240137, 
-0.34259491746048659, 0.28071819216619254]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank3_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.71614527258053018, -0.04214798739915282], [-0.85602591015480667, -0.63437029957306024]], 
[[0.48616819208344642, -0.68488217634113524], [0.16967198365605829, 0.51842248585142947]], [[0.12768328872289358, 
-0.6502475147161737], [0.25458052156920608, 0.82355211118303573]], [[0.34057178868714955, 0.67446046215066358], 
[0.051489994935964178, -0.39981816352451194]], [[0.83849369299165977, -0.50780980914803031], [-0.099483057187065294, 
0.16116058957979695]], [[-0.59619809542046154, 0.17687119993864409], [-0.59048580388867711, 
0.52200367122135582]]])+(1.-msk_arg0)*numpy.array([[[0.15979749909800622, 0.79913484946863145], [0.78599981054426094, 
-0.33299470529963315]], [[-0.63169398011950184, 0.97937647208491163], [0.41559571088800507, -0.89012278000386758]], 
[[-0.61202692897310218, 0.50430213793464462], [-0.674353480635832, -0.90845669378286154]], [[-0.88096226389709842, 
0.27048679451685098], [-0.88479897767137117, -0.97502331724884006]], [[-0.46088647641014102, -0.54914351217085633], 
[0.8665650718293656, -0.99478510823778965]], [[-0.24029797246396223, -0.071361817533693195], [-0.72546360451150349, 
-0.61207451014397418]]])
      arg1=Data(-0.631902761531,self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.45253417540070151, 0.026633429630480432], [0.54092513656854391, 0.40086034413319321]], 
[[-0.30721102314585552, 0.43277893855303179], [-0.10721619502663338, -0.32759260044906235]], [[-0.080683422745301839, 
0.41089320022754167], [-0.16087013461147476, -0.52040485332089326]], [[-0.21520825377081634, -0.42619342857619036], 
[-0.032536669991230949, 0.25264620164122104]], [[-0.52984648012739777, 0.32088642073295326], [0.062863618562010407, 
-0.10183782160536888]], [[0.37673922291545731, -0.1117653996764552], [0.37312961012185503, 
-0.32985556137387168]]])+(1.-msk_ref)*numpy.array([[[-0.10097648096570927, -0.50497551821454645], [-0.49667545084542514, 
0.2104202738538973]], [[0.3991691704797517, -0.61887069728852517], [-0.26261607739039178, 0.56247104278571403]], 
[[0.38674150654917872, -0.31866991360667352], [0.42612582666153559, 0.57405629353232368]], [[0.55668248736080128, 
-0.17092135241275061], [0.55910691738996665, 0.61611992672624138]], [[0.29123543719566308, 0.34700530181736156], 
[-0.54758486193491496, 0.62860745702494858]], [[0.15184495239017518, 0.045093729567381551], [0.45842245508074203, 
0.38677157322244959]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank4_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[0.33614499471732762, -0.38763051892321343, -0.67310246736468238, 0.54147114643914462], 
[0.79807775395112168, 0.65747863038277421, -0.82356537264935969, 0.97369681738798897], [-0.55352515849093664, 
-0.79137549467897039, 0.1987852106562471, -0.7374247699467178]], [[0.90001316105185181, 0.89817527500492433, 
-0.39221865336499651, 0.2541763072784724], [0.85319027771590972, -0.64517033625387277, -0.6040202851658163, 
-0.027827968094352507], [0.33853315146414942, 0.73257354535595054, -0.43164449645334657, 0.5741811015673548]]], 
[[[0.36107836699350915, -0.017101929430308527, 0.90242939202759342, -0.39777338170880627], [0.97430916784023713, 
0.67115836087484748, 0.58825808307274419, 0.05087528266536756], [-0.96855449930468684, -0.92599616247707472, 
0.017156628965631304, 0.98848943258679478]], [[-0.1015215647727179, 0.033919722549923037, -0.48391259268061093, 
-0.038419572012241332], [-0.31296886312169048, 0.6763235149893454, -0.43060770967905682, -0.56601062303268557], 
[-0.51970228140602592, -0.098958271363178474, -0.89913559217507522, -0.75328978108805522]]], [[[0.7928607866930224, 
0.93404095404258913, 0.058432793246443682, -0.36756309946219856], [0.57169716993400965, 0.18136666084464381, 
0.0030118825408098626, -0.61854504527165433], [-0.49261584481107556, 0.67908138854072941, -0.079794244062769026, 
0.70216268024066131]], [[-0.23110518553918236, 0.221210552858077, -0.9089179127211533, -0.65776227434369638], 
[-0.36972745210471825, 0.67252662597655788, 0.42985612896673686, -0.42719967674258852], [0.61250180881279626, 
0.40884633171139528, 0.73839275838093421, -0.58021136703013321]]]])+(1.-msk_arg0)*numpy.array([[[[-0.31009462545392563, 
0.19911016528579473, 0.15798365667222125, -0.33924318891169092], [-0.46683321503505382, 0.96265767240208344, 
0.16348606888355133, -0.04670429266016729], [-0.22730536037283411, -0.60736704221980231, 0.11646128721190485, 
0.19817799892539867]], [[0.22650372920304318, -0.30264633978965572, -0.17506598233551052, -0.13724912529266642], 
[0.058668115865703419, 0.051609198623017383, -0.87565988190737931, -0.094481322335332907], [0.50273756152670601, 
-0.019114983721407741, -0.31015293298428226, -0.43861313303089045]]], [[[0.77531930002959948, -0.39709154910793831, 
-0.33698975134292386, 0.65281235409873872], [0.34828176289241219, 0.66750619882050244, 0.64624691865888506, 
-0.7148248388267775], [0.2906366405892904, 0.92778907252451792, 0.40218686572635032, 0.92138695697192396]], 
[[-0.98774207903531885, -0.44795498518746713, 0.54602942308871483, -0.62025586524331833], [-0.20252787584458476, 
-0.31858728562069905, -0.52867934500457392, -0.53999227776338721], [-0.040113310138338187, -0.98575276362842268, 
0.24615633737850562, -0.054113937410190882]]], [[[-0.64735009236087149, 0.21069057561826776, 0.3280043377298274, 
-0.99782439129055489], [0.65027332347529598, -0.68432726910756392, 0.51695839859198411, -0.71134830648984892], 
[0.19173752494194862, 0.14071904346667186, -0.85589771591759933, -0.69208088909502607]], [[-0.73371064987758849, 
-0.32688482596206314, 0.15057874653305237, 0.16123884704326841], [0.40523773076670322, -0.80911074177084941, 
0.087570240521975373, 0.51863272802496585], [0.78101380659994502, 0.185492143914324, 0.043925946630618329, 
-0.73751576574241118]]]])
      arg1=Data(0.880552249443,self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.29599323123725768, -0.34132892539048859, -0.59270189174342325, 0.47679363600532304], 
[0.70274916147186239, 0.57894428694407751, -0.72519234144953448, 0.85739092282622142], [-0.48740782343233829, 
-0.69684747199342112, 0.17504076439930547, -0.64934103997137049]], [[0.79250861349227075, 0.79089025879942942, 
-0.34536901749394633, 0.22381551912910561], [0.75127861824541242, -0.56810619086207059, -0.53187142081179917, 
-0.02450397990290264], [0.29809612803269342, 0.64506928324541857, -0.38008553231156911, 0.50559646057263929]]], 
[[[0.31794836828124234, -0.015059142429669011, 0.79463623111313331, -0.35026024603213129], [0.85793012939439584, 
0.59099000440064298, 0.5179919783025736, 0.04479834459202417], [-0.85286284307061977, -0.81538800384452448, 
0.015107308228540955, 0.87041659341467059]], [[-0.089395042227562629, 0.029868087991808047, -0.42611032201857585, 
-0.033830440558006082], [-0.27558543642733863, 0.59553819247488249, -0.37917258738527582, -0.49840192731991273], 
[-0.45762501293259517, -0.08713792844981097, -0.79173586824378417, -0.66331101121930924]]], [[[0.69815534921747391, 
0.82247186315383614, 0.051453127534377988, -0.32365851404358109], [0.50340922898543528, 0.15970282118066795, 
0.0026521199463674116, -0.54466123099561092], [-0.43377399025952373, 0.59796664423423229, -0.070263001102053524, 
0.61829092756065118]], [[-0.20350019098440666, 0.19478744991964977, -0.80034971260540311, -0.5791940502719084], 
[-0.32556433963153913, 0.59219483331378808, 0.37851078129840471, -0.37617163631689371], [0.53933984553784908, 
0.36001055706487795, 0.65019340436455431, -0.51090642439062561]]]])+(1.-msk_ref)*numpy.array([[[[-0.27305451998355529, 
0.17532690392932046, 0.1391128642579125, -0.29872135310431469], [-0.41107103761369862, 0.84767037887695795, 
0.14395802570796035, -0.041125569960541924], [-0.2001542463866896, -0.53481841526402496, 0.10255024842743875, 
0.17450608274381887]], [[0.19944836825690682, -0.26649591528739036, -0.15415474454643513, -0.1208550260105046], 
[0.051660341396111906, 0.045444595939434639, -0.77106427876029882, -0.083195740912701099], [0.44268669068169419, 
-0.016831741913946766, -0.27310586281057025, -0.38622178092547427]]], [[[0.68270915367742724, -0.34965985680169365, 
-0.29673708358415807, 0.57483538686564883], [0.30668028975479333, 0.58777408488836003, 0.5690541779205146, 
-0.62944061978645638], [0.25592074764137812, 0.81696675481984904, 0.35414654931165929, 0.81132935756882019]], 
[[-0.86975850956378953, -0.39444776985590929, 0.48080743676268434, -0.54616769737005122], [-0.1783363766498072, 
-0.2805327509971603, -0.46552978647768706, -0.47549141486625446], [-0.035321865474907586, -0.86800681340739605, 
0.21675351659322686, -0.04765014931274672]]], [[[-0.57002558000532488, 0.18552406029704943, 0.28882495741497266, 
-0.87863651229972162], [0.57259963773877232, -0.60258591616766632, 0.45520888074848337, -0.62637935141691348], 
[0.16883490889021455, 0.1239104702640073, -0.75366265904413743, -0.60941338368895026]], [[-0.64607056318979839, 
-0.28783916880959021, 0.13259245397794606, 0.14197922946150238], [0.35683299538568819, -0.7124642837145978, 
0.077110172275866376, 0.45668321529700362], [0.68772346424741138, 0.16333552457771272, 0.038679091114492303, 
-0.6494211665239582]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.264591594097)+(1.-msk_arg0)*(-0.427834372758)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.12978686673)+(1.-msk_arg1)*(0.427514232563)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array(0.034340513960799644)+(1.-msk_ref)*numpy.array(-0.18290528353399987)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.21293714109293549, -0.97325505842674231])+(1.-msk_arg0)*numpy.array([-0.6125640006994324, 
0.78735279410138892])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.233178513628)+(1.-msk_arg1)*(0.731625996957)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.04965236605634122, 0.22694216790531338])+(1.-msk_ref)*numpy.array([-0.44816774771145035, 
0.57604777294099929])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.32941699506058231, 0.50568121904021801, -0.35213527917652732, 0.69364322007306822, 
-0.48890801916381332], [0.91487273015996218, 0.35548718050125561, 0.9280950949893787, 0.76483319225020319, 
-0.13978070240719553], [0.47902080752887843, 0.24418645006654782, 0.39670242841894154, -0.64460121466056619, 
0.092516635563920646], [-0.66523647553523979, 0.58631041062426514, -0.45387928658778942, 0.89804929988327054, 
-0.34436500698567896]])+(1.-msk_arg0)*numpy.array([[0.45830684062404137, 0.02169027792878131, -0.48337085922610523, 
-0.4796207471344418, -0.87794907194563687], [-0.53017579701142581, 0.24506177783476679, 0.88874600023393135, 
0.86671281296747926, 0.64119254334737619], [-0.14087777246442301, 0.94002508153033104, 0.89390247775514986, 
0.51517305006431546, 0.83904630357552934], [-0.00086796920917087128, -0.77123745415412803, -0.77950764862612276, 
0.43488999834439879, 0.21147604980506163]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.581754419525)+(1.-msk_arg1)*(0.905971318599)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.19163979274307527, -0.29418228404733882, 0.20485625493154649, -0.40353000885092577, 
0.28442440088966742], [-0.5322312540732842, -0.20680623834101824, -0.5399234232493666, -0.44494508979082209, 
0.081318041389667833], [-0.27867247182426697, -0.14205654651428778, -0.23078339096894263, 0.37499960545984318, 
-0.053821961618876771], [0.38700425967173191, -0.34108867259407061, 0.26404628090321297, -0.52244414915825133, 
0.20033586474360976]])+(1.-msk_ref)*numpy.array([[0.41521285272297331, 0.019650769695910581, -0.43792013470526781, 
-0.43452264070869051, -0.79539667837310557], [-0.48032406590756543, 0.2220189420031087, 0.80517838573126754, 
0.78521695001054748, 0.58090205397208527], [-0.12763122128084287, 0.85163576262989737, 0.80985000647049044, 
0.46673200747328902, 0.76015188601569894], [-0.00078635520893561705, -0.69871901329273034, -0.70621157228359122, 
0.39399786524546737, 0.19159123569393893]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank3_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.33745884555915162, -0.66375221821549824], [0.35050591871233383, 0.042176466180520178]], 
[[0.20748673291497721, -0.76371032387997184], [-0.32482792758474766, -0.70921662870826352]], [[0.50212745454878882, 
0.36618836225151474], [0.19785234293440346, -0.77302966670008377]], [[0.64935634997914415, 0.57990195989895033], 
[-0.80742363450638099, -0.21732793929102523]], [[-0.1072984723427084, 0.25601710633311225], [0.55861416589548618, 
0.51491586728123639]], [[-0.1317501155335572, 0.83648625632547402], [0.091416210119983576, 
-0.48358358241352173]]])+(1.-msk_arg0)*numpy.array([[[0.42886586178498498, -0.069787877992939507], [0.30684797181993506, 
0.46417129346574049]], [[0.77136474775699138, 0.078418535081947294], [-0.49859534298556052, -0.72222304926459113]], 
[[0.4388243207897955, -0.11542252838195743], [-0.38927617211922882, 0.048609512811642386]], [[0.96883423294606907, 
-0.87835366478764509], [-0.51994722843107555, 0.07625890302971694]], [[-0.22900078843687521, 0.19274199077969256], 
[-0.7277710383793552, -0.51046116944335274]], [[0.81208765776357139, -0.22720534187686625], [0.60017241969465607, 
0.43244947860361416]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.637368901774)+(1.-msk_arg1)*(-0.36037041184)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.21508577378804244, 0.42305502237423392], [-0.22340157247505449, -0.026881967930196909]], 
[[-0.13224559109074599, 0.48676521040503384], [0.20703521947029596, 0.45203262375982139]], [[-0.32004042425646079, 
-0.2333970742907589], [-0.12610493052956287, 0.49270506970354627]], [[-0.41387954364634211, -0.36961147531752869], 
[0.51462671519190484, 0.13851806999078145]], [[0.068388709479126703, -0.16317734189895694], [-0.35604329743234431, 
-0.32819136083517669]], [[0.083973426446253802, -0.53315032654342054], [-0.058265849448537937, 
0.30822113683896346]]])+(1.-msk_ref)*numpy.array([[[-0.15455056723538993, 0.02514948633372573], [-0.11057892997688877, 
-0.16727360019035784]], [[-0.27797703182771805, -0.028259719783337627], [0.17967900909300136, 0.26026781770351548]], 
[[-0.15813930120824105, 0.041594864088571208], [0.14028361446594037, -0.017517430151252737]], [[-0.3491391915310551, 
0.31653267192032503], [0.18737359684455307, -0.027481452291253424]], [[0.082525108440584458, -0.069458510596057663], 
[0.26226714882568447, 0.18395510186041289]], [[-0.29265236367809527, 0.081878082624318074], [-0.21628438206011838, 
-0.15584199670419457]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank4_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[0.13152428269210881, -0.3249269355703337, 0.50929908793572332, 0.54884977050839079], 
[0.93776551306946487, 0.10668163504602424, -0.84845010039488855, -0.9726067912742018], [0.1083611448838746, 
0.84834672122815569, -0.92827615365710492, -0.18820948907656354]], [[-0.83470741457517228, -0.75974999350460037, 
-0.61856244293820262, 0.53074924989214667], [-0.42304235638745968, 0.60869483277582392, 0.24882199006295203, 
-0.83711417504221308], [0.34239912938615635, 0.32163181395462992, 0.57530808877203632, -0.48131089429291762]]], 
[[[0.27536846712519769, 0.31419276389072537, 0.86070347055427066, -0.28452411904104302], [0.17730288224396284, 
0.16756821629920848, 0.66092392041239179, -0.71636704098355231], [-0.19092608481195517, 0.41713386566086319, 
-0.75212160203626044, 0.69628026918488284]], [[0.39790165641326158, 0.63795247582254455, 0.46050525883551074, 
-0.77107410798654508], [-0.89469611490295264, -0.91344706342184856, 0.083105874082978648, 0.73203700836399621], 
[0.83297065676902582, -0.3314079580353988, 0.52631137651761151, -0.065404468481846223]]], [[[0.069369776635408664, 
0.14627252260314005, 0.3527723050915994, -0.90634105218740446], [0.2810853455317881, -0.89740910295554155, 0.23938138895920114, 
0.98563119512268726], [0.26555452729196527, -0.024169136008622916, 0.34220225993250453, -0.97686411145602192]], 
[[-0.61998394665672785, 0.67363885378633603, -0.39612107107938477, -0.95171740177791797], [0.72427341015815094, 
-0.7096180514308017, -0.92874702670260212, -0.15181963621277306], [0.42525537412781644, 0.55610853161171336, 
0.82710212515693238, -0.93985339008584168]]]])+(1.-msk_arg0)*numpy.array([[[[0.24363200118490158, -0.1513324253842474, 
0.57133660366702466, 0.57053434632287159], [0.48129939414584433, -0.48919366668235287, 0.83801855845680429, 
0.71261770393977741], [0.67852259714775998, 0.44877787453470797, 0.12516534181874928, 0.0025008086222404469]], 
[[0.097901205990076567, 0.72150826959730896, 0.52873140790976669, 0.37216804509190715], [0.80008770995803724, 
0.057197703972295288, 0.62784333251223257, -0.10801727402453931], [0.41920817219938389, -0.72998152074020539, 
0.76777140255742826, 0.072679087275026255]]], [[[0.80301108161918444, 0.093661774013343857, 0.4100575037780303, 
0.017742627021203994], [-0.63220586939490575, 0.87175869618491353, 0.79014166286733545, -0.96226421942608464], 
[-0.14275240493695485, -0.23295462209719209, -0.65589256553715414, -0.0025931052289895895]], [[0.84256758603185, 
0.68983388564910175, -0.35000681991373761, -0.50778119657276477], [0.047412329402986009, 0.44722652266486396, 
0.38058248940630457, 0.049682360264559744], [0.61095978397269701, -0.6566569411821479, -0.2181239549048084, 
-0.33434575728366167]]], [[[0.71884062279606309, -0.56019619993759928, -0.11950311285462911, 0.89269367228125018], 
[-0.51800440731713415, -0.50124129173870924, -0.35877777621882045, 0.28245223081806858], [-0.85623006074111352, 
0.0017039409286681018, 0.84801473994961341, 0.69242336798734838]], [[0.63349834503210944, 0.76984527473816655, 
0.29278199906355118, 0.45303590963602947], [-0.88505218143270015, 0.012178144653693268, -0.83892758808741852, 
-0.92284598773509319], [-0.87068117618378538, -0.45190637812759382, -0.3925910200527376, 0.22267783977567768]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.0844199344821)+(1.-msk_arg1)*(0.941872663068)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.011103271327677369, 0.02743031061232783, -0.042994995635343869, -0.046333861666852098], 
[-0.07916610317292809, -0.0090060566410321927, 0.071626101886695687, 0.082107401596245758], [-0.009147840751505619, 
-0.071617374624213276, 0.078365012073059251, 0.015888632736759226]], [[0.070465945250186693, 0.064138044674459355, 
0.052219000905951218, -0.044805816902336273], [0.035713208009396515, -0.051385977902547973, -0.021005536098828367, 
0.07066912381112847], [-0.028905312069518709, -0.027152136661419458, -0.048567471161176476, 0.040632234161744961]]], 
[[[-0.023246587953154593, -0.026524132542415327, -0.072660530592736042, 0.024019507488031472], [-0.014967897702528691, 
-0.014146097841267077, -0.055795154058888623, 0.060475658664990994], [0.016117967570755429, -0.035214413609368951, 
0.063494056366498033, -0.058779934705789759]], [[-0.033590831764739752, -0.053855906211653766, -0.038875823779571529, 
0.065094025677093298], [0.075530187401526228, 0.077113141246969275, -0.0070157924451654547, -0.061798516284585228], 
[-0.070319328269980483, 0.02797743810420586, -0.044431171922818015, 0.0055214409440761929]]], [[[-0.0058561919986014023, 
-0.012348316774693406, -0.029781014882943865, 0.076513252244128194], [-0.023729206453681276, 0.075759217675176599, 
-0.020208561172177762, -0.083206920915803742], [-0.022418095795421516, 0.0020403568783377087, -0.02888869236313989, 
0.082466804287064474]], [[0.052339004156735223, -0.056868547901261682, 0.033440514867514023, 0.080343920703597707], 
[-0.061143113832702163, 0.059905909409127184, 0.078404763144709524, 0.012816603742183573], [-0.035900030822045237, 
-0.046946645803616044, -0.069823907215781231, 0.079342361613857368]]]])+(1.-msk_ref)*numpy.array([[[[0.22947032176470283, 
-0.14253587450525856, 0.53812632840430608, 0.53737070414310251], [0.45332274209734597, -0.4607581415942949, 
0.78930677135443816, 0.67119513455943514], [0.63908188552763667, 0.42269161181417431, 0.1178898138226899, 
0.0023554432768539851]], [[0.092210469603479797, 0.67956891531147867, 0.49799765921586775, 0.35053490773966883], 
[0.75358074206646097, 0.053872953761782826, 0.59134847158302517, -0.10173851754288017], [0.3948407175294632, 
-0.68754963893028576, 0.72314289555451294, 0.068454445481108447]]], [[[0.75633418591808432, 0.088217464517657301, 
0.38622195309458718, 0.016711295362290466], [-0.59545742581444261, 0.82108568472870647, 0.74421283220613821, 
-0.90633036292626579], [-0.13445458779738592, -0.21941359028877122, -0.61776727738923365, -0.0024423749276449754]], 
[[0.7935913760709179, 0.64973567895113027, -0.32966185556424821, -0.47826522787204034], [0.044656376957065855, 
0.42123043589716824, 0.35846024281431099, 0.046794456969903725], [0.57544631875804841, -0.61848722191356853, 
-0.20544499028519989, -0.31491112879837785]]], [[[0.67705633171466328, -0.52763348667601528, -0.1125567151493511, 
0.84080376641583598], [-0.4878941906009488, -0.47210547028977462, -0.33792297953697298, 0.26603403483022003], 
[-0.80645968750943642, 0.0016048953801958397, 0.79872190143758515, 0.65217464157702321]], [[0.59667477328480667, 
0.72509621906824828, 0.27576336115647193, 0.42670213867449464], [-0.83360645508049946, 0.011470261536206077, 
-0.79016296151343279, -0.86920340807002505], [-0.82007079809573435, -0.42563826382462472, -0.36977074955380512, 
0.20973416995583236]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.809916642498)+(1.-msk_arg0)*(0.858853571425)
      arg1=numpy.array([0.18792098079616504, -0.0087948976168672477])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.15220032982133061, -0.0071231339489653553])+(1.-msk_ref)*numpy.array([0.16139660550243365, 
-0.0075535292285617128])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.69225378146116934, -0.05852530104787812])+(1.-msk_arg0)*numpy.array([-0.90322445384304317, 
0.97425063291510239])
      arg1=numpy.array([-0.94088931049844371, -0.95517417934997439])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.65133418312893998, -0.6612229376090889], [0.055065830149651891, 
0.055901856399617179]])+(1.-msk_ref)*numpy.array([[0.84983423360171428, 0.86273667646835761], [-0.91666200625616312, 
-0.93057904877587605]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.68854400539697558, -0.65841481073462838, 0.093032278271191648, 0.90655175598475313, 
-0.3675552924858676], [0.10025318514000503, 0.70593657398898335, -0.31748455253330365, -0.43299186356982067, 
0.69988564454522106], [0.64562779777595503, 0.070769172172916894, -0.86753891003133443, -0.072516413535663293, 
-0.3535574126470018], [0.761728829327315, 0.31243787616296226, -0.3731441451355042, 0.59820441601917862, 
0.8188916120757328]])+(1.-msk_arg0)*numpy.array([[-0.38284500969415047, -0.21780851164808479, 0.13254498807361514, 
-0.27159469939107406, 0.80967081260907348], [0.2985837063418848, -0.7067565097923707, -0.16488817957428337, 
-0.34413067911586759, 0.93265287047179957], [0.37419307563750981, -0.23034427559306847, 0.36688817263928009, 
-0.07634503898964029, -0.28028449653149945], [0.23430328570643932, -0.54406841361731795, 0.44301529307210474, 0.14940947211625, 
-0.041130763768178147]])
      arg1=numpy.array([0.9852682146970666, -0.49287947409927302])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.67840052293784547, 0.33936920727426834], [-0.64871518510261428, 0.32451914565405604], 
[0.091661746721457701, -0.045853700388562166], [0.8931966301495885, -0.44682075273353761], [-0.3621405468300089, 
0.18116045926283889]], [[0.09877627674058724, -0.049412737168582729], [0.69553686794348935, -0.34794164733513266], 
[-0.31280743826838514, 0.15648161928725773], [-0.42661312039779303, 0.21341280200555737], [0.68957507949317576, 
-0.34495926841307928]], [[0.636116547673514, -0.31821668943168452], [0.069726615922399149, -0.034880672363028185], 
[-0.85475851306681194, 0.42759212173690064], [-0.071448117300517172, 0.035741851767023131], [-0.34834888075162557, 
0.17426119160935391]], [[0.75050720375461022, -0.37544050480510194], [0.30783510845082501, -0.15399421609189462], 
[-0.36764706570232131, 0.18391509001761011], [0.58939179699511746, -0.29484267797139552], [0.80682787666026012, 
-0.40361486710419309]]])+(1.-msk_ref)*numpy.array([[[-0.37720501920703681, 0.18869644703958396], [-0.21459980341733373, 
0.10735334467545342], [0.13059236376633479, -0.065328704016217848], [-0.26759362459023001, 0.13386345260402271], 
[0.79774291603166503, -0.39907012431229116]], [[0.29418503528510204, -0.14716578015639994], [-0.69634472462865893, 
0.34834577686270135], [-0.16245908231380349, 0.081269999233759271], [-0.33906101983497994, 0.16961494814405451], 
[0.91891322862184444, -0.45968545631531799]], [[0.36868054358537372, -0.18443208633180533], [-0.22695089317927167, 
0.11353196541608959], [0.36148325484977267, -0.18083164958369166], [-0.075220340266300825, 0.0376289026673024], 
[-0.27615540550485662, 0.13814647524862497]], [[0.23085158000564018, -0.11548328023872152], [-0.53605331455780003, 
0.26816015357772943], [0.4364888868886504, -0.21835314466731429], [0.1472084038508088, -0.073640862042107291], 
[-0.040524834186999675, 0.020272509215361077]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank3_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.11711260368278764, 0.10980142262859216], [-0.91665357370244083, -0.093333775529722329]], 
[[0.47884791251865955, 0.81001994543770106], [-0.48359066455925381, -0.38107346610484627]], [[0.62252050489004418, 
0.50277828007388337], [0.28603879958825051, 0.69684786424599521]], [[-0.34602034382597702, 0.91199419628949729], 
[0.69639120035850022, -0.6737254134675501]], [[-0.51849356148680248, -0.36035315909332022], [-0.7751326363239297, 
0.29837003569811249]], [[0.56940063451215206, 0.23880928332203633], [0.87412068108193952, 
0.84235747779910075]]])+(1.-msk_arg0)*numpy.array([[[-0.73797274562881965, 0.087647337119837188], [0.42606202197474197, 
0.99157082131401553]], [[-0.64032420926864053, 0.15915908019775871], [0.12876515050279203, -0.30175036005486278]], 
[[-0.3189537853451887, 0.56434647528486237], [0.41060862455421976, 0.42765641045758773]], [[0.79174083353568236, 
-0.054672767013720813], [0.26963343963365571, -0.98947217950649047]], [[-0.43169389761932053, -0.23379247312147111], 
[0.54814164739271654, 0.61555291583611349]], [[0.35096141936800329, -0.80263157305871813], [-0.26078342225015771, 
-0.13421208138753649]]])
      arg1=numpy.array([0.13180785873603296, 0.57438234807218902])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.015436361522429887, 0.067267412292167261], [0.014472690402844927, 0.06306799895107755]], 
[[-0.12082214475245109, -0.52650963203197132], [-0.012302125100322251, -0.053609273143204531]]], [[[0.063115918009303745, 
0.27504178836193383], [0.10676699454162163, 0.46526115824581316]], [[-0.063741050000290428, -0.27776594141533451], 
[-0.050228477588398018, -0.21888187224930933]]], [[[0.082053094768830853, 0.35756478932182822], [0.066270128515524038, 
0.28878696906853379]], [[0.037702161689152566, 0.16429563734724961], [0.091850024851042406, 0.40025711251470475]]], 
[[[-0.045608200598807935, -0.19874797756751086], [0.12020800219260797, 0.5238333678929703]], [[0.091789832961869613, 
0.39999481283872562], [-0.088802304125206247, -0.38697598494339785]]], [[[-0.068341526107995085, -0.29781354930710152], 
[-0.047497378288855564, -0.20698049365525237]], [[-0.10216857303027334, -0.44522250371912492], [0.039327515516361919, 
0.17137848169866471]]], [[[0.075051478397985275, 0.32705367344488423], [0.031476940280964233, 0.13716783689594786]], 
[[0.1152159752502932, 0.50207948929830548], [0.11102933543898488, 
0.48383526601441434]]]])+(1.-msk_ref)*numpy.array([[[[-0.09727060740688584, -0.42387851844756169], [0.011552607829680957, 
0.050343083297166816]], [[0.056158322805235361, 0.24472250460623687], [0.13069682674252994, 0.56954077662621316]]], 
[[[-0.084399762920542978, -0.36779092284718951], [0.020978417559263121, 0.091418166200998491]], [[0.016972258767596036, 
0.073960429495662491], [-0.039773068831658431, -0.17332008033994056]]], [[[-0.042040615482101612, -0.18320142415308244], 
[0.074385300492525258, 0.32415065360038281]], [[0.054121443581039388, 0.23584634591014464], [0.056368475737152653, 
0.24563829320675309]]], [[[0.10435766394222021, 0.45476195903085737], [-0.0072063003512525554, -0.03140307229294468]], 
[[0.035539806321743563, 0.15487268817555999], [-0.13042020925962614, -0.56833535381704448]]], [[[-0.056900648274614878, 
-0.24795735456302051], [-0.030815685270742646, -0.13428626967311472]], [[0.072249376826875572, 0.31484288650558639], 
[0.081134711775079638, 0.35356272916062942]]], [[[0.046259473185855396, 0.20158604413934197], [-0.10579314899880343, 
-0.46101740757034126]], [[-0.034373304480648022, -0.14978939441034672], [-0.017690207064197366, -0.077089050447028948]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.754627721144)+(1.-msk_arg0)*(0.58366899685)
      arg1=Data(numpy.array([-0.23430637466555004, -0.19529124738101356]),self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.17681408556336167, -0.14737218897049123])+(1.-msk_ref)*numpy.array([-0.13675736665662333, 
-0.11398544645247927])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.42201616119905205, 0.27767742277886431])+(1.-msk_arg0)*numpy.array([0.36965951817707077, 
0.016971039947976774])
      arg1=Data(numpy.array([-0.88941504482415179, -0.014645484245880747]),self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.37534752292937135, -0.0061806310403477865], [-0.24697047742751854, 
-0.0040667203207446247]])+(1.-msk_ref)*numpy.array([[-0.32878073692913373, -0.0054138426498021577], [-0.015094298256042233, 
-0.00024854909819430667]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.32565014483404586, 0.010119090846729684, 0.81211966143393899, -0.87604363255028828, 
-0.076988721786289904], [-0.36406220548242252, 0.40674278274405906, 0.60188613892112408, -0.30971097973586437, 
0.62504329060164121], [0.80632167789462628, 0.035718851870698343, 0.62756915889350329, 0.15400741967310783, 
-0.80078011954154671], [-0.60228312653172322, -0.95589393004111911, -0.12966642233915548, -0.65230700307710321, 
0.59350325591256503]])+(1.-msk_arg0)*numpy.array([[0.38013816651553789, 0.38544277320184328, -0.11994397447910532, 
0.90332228918106083, 0.1620135444547588], [-0.29343253074237796, 0.61947664145034387, -0.085366125919957181, 
-0.92522686233194484, -0.20398584627625538], [0.92366281232068936, 0.60773805084890609, 0.27891181403544163, 
-0.87242066265766005, 0.9594846291366943], [-0.3720465977870473, 0.24525390349168763, 0.97924605524931319, 
-0.58203005863290769, -0.62666251303222253]])
      arg1=Data(numpy.array([0.2674613723521404, -0.54031949922685807]),self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.087098834643987197, 0.17595512317988546], [0.0027064659248223037, -0.0054675420989360657], 
[0.21721063916127695, -0.43880408877827143], [-0.23430783220225432, 0.47334345684044943], [-0.020591509184598226, 
0.041598507601684057]], [[-0.097372577099875665, 0.19670990855368803], [0.10878798286705453, -0.21977105668640873], 
[0.16098129271557488, -0.32521081717344891], [-0.082835723672680231, 0.16734288147594181], [0.16717493628381266, 
-0.33772307777298632]], [[0.21565990252697725, -0.43567132521578444], [0.0095534131401797968, -0.019299592155734051], 
[0.16785050848353486, -0.3390878536635582], [0.041191035818181447, -0.083213211874994183], [-0.21417774972489312, 
0.43267711318151209]], [[-0.16108747156671252, 0.32542531732040708], [-0.2556647023518786, 0.51648812959381074], 
[-0.034680759266822764, 0.070061296384830773], [-0.17446692623791391, 0.35245419324479299], [0.15873919532183822, 
-0.32068138202418695]]])+(1.-msk_ref)*numpy.array([[[0.10167227569967223, -0.20539606376869141], [0.1030910530837798, 
-0.2082622461970314], [-0.032080380019551616, 0.064808068225829227], [0.24160381914064355, -0.48808264693076986], 
[0.043332364939504292, -0.087539077207763577]], [[-0.078481867365118044, 0.15854731806759129], [0.16568607266240379, 
-0.33471530869118571], [-0.02283214119093737, 0.046124982408008171], [-0.24746244633636685, 0.49991811492643357], 
[-0.054558334385460011, 0.11021753030935316]], [[0.24704412337392909, -0.49907302820758626], [0.16254645311066332, 
-0.32837271929578776], [0.0745981365471442, -0.15070149168808439], [-0.2333388277027815, 0.47138589556235055], 
[0.25662507565968473, -0.51842825433100626]], [[-0.099508093623068478, 0.20102403140535369], [0.065595945602606176, 
-0.13251546631806083], [0.26191049380740122, -0.52910573819218509], [-0.15567055823215423, 0.31448218981551151], 
[-0.1676080157372393, 0.33859797522581492]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank3_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.8308383256533951, -0.0020690666107840983], [0.21524076447995588, -0.10475888330800176]], 
[[-0.087242765977021852, 0.70888825028190872], [0.78270518430092428, 0.29269417305487044]], [[-0.44144040991368905, 
-0.12863921174128756], [-0.24101563144741855, -0.92331317605137442]], [[0.047840171049458258, 0.80686689373407749], 
[-0.61618586167476153, -0.10436202485840806]], [[0.13957673157835604, -0.20644022281349739], [0.52298215287361449, 
0.80141225819897266]], [[0.46802794923437463, -0.53954185089423423], [-0.059424631459731536, 
0.46892840066385233]]])+(1.-msk_arg0)*numpy.array([[[0.24025366440042739, -0.94615132172696148], [-0.60120188385701789, 
-0.61359598051779263]], [[0.80388555717190679, -0.52479497642943729], [-0.59558320626551553, -0.37112136946130847]], 
[[0.18761274066806388, 0.78545898789654789], [-0.92703360405653634, -0.18394419220048652]], [[-0.45611388588357693, 
-0.73857176458521323], [-0.76968431314751107, 0.59284454646853635]], [[-0.85058232257517341, 0.84176506815894236], 
[-0.54396196879800685, -0.31031616416812891]], [[-0.92721672793037735, -0.55179586860640684], [0.092088579256883074, 
0.22532501099469493]]])
      arg1=Data(numpy.array([0.59121762951075518, -0.29269610838823557]),self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.49120626539948509, 0.24318314461854629], [-0.0012232686569276269, 
0.00060560774497254164]], [[0.12725413454992227, -0.063000134129791857], [-0.061935298659550622, 0.030662517463349404]]], 
[[[-0.051579461292896422, 0.025535618086499859], [0.41910723091969698, -0.20748883213966021]], [[0.46274910366817118, 
-0.22909476146017724], [0.17304595516511126, -0.085670445401073339]]], [[[-0.2609873527194273, 0.12920789006704428], 
[-0.076053769827816142, 0.037652196662805087]], [[-0.14249269029938061, 0.070544337385392661], [-0.54587902724114012, 
0.27025017345381913]]], [[[0.028283952523249768, -0.014002631890803963], [0.47703393224416768, -0.23616679978326849]], 
[[-0.36429994447739461, 0.18035520375605432], [-0.061700668947730519, 0.030546358539572341]]], [[[0.082520224378614629, 
-0.040853566154534164], [-0.12205109916746805, 0.060424249832310935]], [[0.30919626869836975, -0.15307484090260826], 
[0.47380905555325792, -0.23457024918946712]]], [[[0.27670637469112702, -0.13698995935782812], [-0.31898665410753446, 
0.15792180006932802]], [[-0.035132889746172728, 0.017393358370668535], [0.27723873745075239, 
-0.13725351798702887]]]])+(1.-msk_ref)*numpy.array([[[[0.14204220194809319, -0.070321312596018273], [-0.55938134158988206, 
0.27693480981586704]], [[-0.35544115263134646, 0.17596945176062512], [-0.36276876107905687, 0.17959715562022152]]], 
[[[0.4752713135091074, -0.23529417417372556], [-0.31026804194376456, 0.1536054473025921]], [[-0.35211929138471326, 
0.17432488669530419], [-0.21941349631369997, 0.10862578058103756]]], [[[0.11091995980378878, -0.054913519077593557], 
[0.46437720090211398, -0.22990078905588179]], [[-0.54807860986711743, 0.27133912825246864], [-0.10875104927504238, 
0.053839749217700038]]], [[[-0.26966257039902747, 0.13350275937995873], [-0.43665664788164527, 0.21617708125952398]], 
[[-0.45505093509068528, 0.22528360314574855], [0.35050014743150681, -0.17352329163052907]]], [[[-0.50287926445664655, 
0.2489621356815801], [0.49766634820188915, -0.24638135962728031]], [[-0.321599905736761, 0.15921555137837942], 
[-0.18346438697835152, 0.090828333621976162]]], [[[-0.54818687592971649, 0.27139272789769486], [-0.326231445411308, 
0.16150850336580144]], [[0.054444391533267709, -0.02695396877549127], [0.13321611886976839, -0.065951753840683594]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.196322146119)+(1.-msk_arg0)*(-0.426019435555)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.4008017758092961, -0.69247468010822177])+(1.-msk_arg1)*numpy.array([0.043612026398168968, 
0.10558868904036522])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.078686264795153185, 0.13594811533185513])+(1.-msk_ref)*numpy.array([-0.018579570869536998, 
-0.044982833705918678])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.084694054093183979, -0.71334990178012059])+(1.-msk_arg0)*numpy.array([0.338433690248783, 
-0.43486916422203836])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-0.29984942829353645, -0.5647027318292841])+(1.-msk_arg1)*numpy.array([-0.89076776806819269, 
0.45228449535701198])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.025395463699703066, 0.047826963716118151], [0.21389756022201953, 
0.40283063828538557]])+(1.-msk_ref)*numpy.array([[-0.30146582290199048, 0.15306831080598213], [0.38736743481574548, 
-0.19668458048649018]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.91685841531765533, -0.22384099195951723, 0.43367807439332395, 0.49498927216479727, 
-0.4553686876237637], [-0.48723087581788382, -0.028233650781919417, 0.46252772974398493, -0.49004398489623546, 
-0.60474523114706269], [-0.22700145502566982, -0.022434491114989852, -0.43737706246764052, -0.093536595481270579, 
0.41370261256966478], [-0.66811934938107242, -0.47714142742132659, 0.98632342655140781, 0.19496584964968111, 
-0.19394544936355596]])+(1.-msk_arg0)*numpy.array([[-0.64866638345178118, -0.18375521953732088, -0.62929258930351617, 
-0.39782127311369653, -0.44126747720393511], [-0.21672598946323585, -0.90341157394747817, -0.71731330077304722, 
0.22435617216144998, 0.93087894537294358], [0.80243746270150695, -0.82807947233128676, -0.71606775170573944, 
-0.79843563358641201, 0.87736535077078859], [-0.56915521568888772, 0.59242256876910404, -0.067116567370258728, 
0.1996295935260104, -0.42950669257857754]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-0.90604925456086072, 0.69995302293583572])+(1.-msk_arg1)*numpy.array([0.76093140511844881, 
-0.11727039054567712])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.83071888373641367, -0.64175781940575283], [0.20281096390508421, -0.15667817897902017], 
[-0.39293369602346068, 0.30355427915259936], [-0.44848466106053758, 0.34646923737255897], [0.41258645997186855, 
-0.31873668945257766]], [[0.44145517183382893, -0.34103872439640254], [0.025581078244489752, -0.019762229213319219], 
[-0.41907290474826481, 0.32374768262595149], [0.4440039872172678, -0.34300776859964305], [0.54792896588003159, 
-0.42329325264741724]], [[0.20567449911023888, -0.15889035465605075], [0.020326753951188808, -0.015703089873964293], 
[0.3962851614108247, -0.30614339703702081], [0.084748762609965986, -0.065471222762241779], [-0.37483494372862536, 
0.28957239426458975]], [[0.60534903846440791, -0.46765215828120543], [0.43231363463519801, -0.33397658449147721], 
[-0.89365760538281691, 0.69038006400708962], [-0.17664866273991842, 0.13646693583154795], [0.17572412982132102, 
-0.13575270356667005]]])+(1.-msk_ref)*numpy.array([[[-0.49359062261306635, 0.076069360121242324], [-0.13982511740038261, 
0.021549046359948257], [-0.47884849420935149, 0.07379738771512373], [-0.30271470033641529, 0.046652656065421669], 
[-0.33577428146186344, 0.051747609386811146]], [[-0.16491361168794619, 0.025415541425751972], [-0.68743423836412398, 
0.10594342810030563], [-0.54582621786738728, 0.084119610925264002], [0.17071965732980873, -0.026310335930706412], 
[0.70833502389781366, -0.10916453747463313]], [[0.61059986601314054, -0.094102154639487928], [-0.63011167643078969, 
0.097109203123148233], [-0.54487844046545686, 0.083973544899697022], [-0.60755474856154745, 0.093632858576263689], 
[0.66761484916425684, -0.10288897733613538]], [[-0.43308807800463911, 0.066745054424944952], [0.45079293767735523, 
-0.069473626007626091], [-0.051071103915778009, 0.0078707860675954911], [0.15190442710497187, -0.023410640397270015], 
[-0.32682513109159467, 0.05036841758067187]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank3_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.13883456031409991, -0.3292702283004636], [0.30614442524975405, 0.14505263944761793]], 
[[-0.77383501589957482, 0.64758425412559206], [-0.27726204439273783, -0.9519754075218414]], [[0.607081071850609, 
-0.82120053072857457], [-0.29135529803867821, 0.68144743891650239]], [[0.23511839162296355, -0.47188904024163913], 
[0.6432406982842358, -0.5274484392482397]], [[0.018132367332716726, -0.49504783001903019], [0.63952767645173836, 
0.76314424887778909]], [[0.98218793657759296, 0.022328732960420039], [-0.79791121219488392, 
-0.76829567228829099]]])+(1.-msk_arg0)*numpy.array([[[0.6479716279159895, -0.01948819418153902], [-0.56839741520435338, 
-0.78827860664172444]], [[-0.68170115669068876, 0.67613005216917665], [-0.40665000876541457, -0.26812785551717999]], 
[[-0.60645082773233461, -0.91576122624446632], [0.091252226719450436, -0.62939783050003162]], [[-0.91514322861177599, 
-0.45280213878396358], [0.23315706784438728, -0.29481090681351674]], [[0.96787113640604794, 0.88227228556255488], 
[0.70113912024859593, 0.86814108302189719]], [[0.90063169303264656, -0.90300469456878085], [0.82130049361510138, 
0.3235079425689702]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.12214824086500831, -0.69411170222450136])+(1.-msk_arg1)*numpy.array([-0.65090991803994647, 
0.19335409710404261])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.016958397313634199, 0.096366692987210095], [-0.040219779156121301, 0.22855031865748496]], 
[[0.037395002994886489, -0.2124984281366484], [0.01771792474135284, -0.10068273447914293]]], [[[-0.094522585911878801, 
0.53712794012697795], [0.079101277453319574, -0.44949580896489877]], [[-0.033867070981208766, 0.19245082959568852], 
[-0.11628212137554232, 0.66077727059084868]]], [[[0.074153884988995597, -0.42138207617050105], [-0.10030820022590659, 
0.57000489825167489]], [[-0.035588537122124751, 0.20223312187375386], [0.083237605905615963, -0.47300064180286039]]], 
[[[0.028719297931755101, -0.16319842703370216], [-0.057640416148993331, 0.32754370498321034]], [[0.078570719748198972, 
-0.44648089602614782], [-0.064426899001166688, 0.36610813400225217]]], [[[0.0022148367724094909, -0.012585888354671948], 
[-0.060469221580864187, 0.34361849197705463]], [[0.078117180663066041, -0.44390364412159627], [0.093216727526670026, 
-0.52970735363140065]]], [[[0.11997252865178533, -0.68174814056224364], [0.0027274154518598368, -0.015498634843673482]], 
[[-0.097463450936071436, 0.55383950972060614], [-0.09384596483421366, 
0.53328301690374336]]]])+(1.-msk_ref)*numpy.array([[[[-0.42177115921900743, 0.1252879690647328], [0.012685058877452126, 
-0.0037681221901597339]], [[0.36997551494478309, -0.10990196901310938], [0.51309836324180802, -0.15241689825364341]]], 
[[[0.4437260440292729, -0.1318097116467096], [-0.44009975684178348, 0.13073251576208039]], [[0.26469252387643949, 
-0.07862744528218775], [0.17452708045891424, -0.05184361941196753]]], [[[0.39474485857451164, -0.11725975223418485], 
[0.59607806471894642, -0.17706618506338967]], [[-0.059396979414920094, 0.017643991906072732], [0.40968129026529571, 
-0.12169664923557687]]], [[[0.59567580393050312, -0.1769466926891084], [0.29473340304418222, -0.087551148711352667]], 
[[-0.15176424792102436, 0.045081874336477507], [0.19189534319126847, -0.057002896703351576]]], [[[-0.62999692207129054, 
0.18714184969285508], [-0.57427978108443889, 0.17059096117486786]], [[-0.45637840729561374, 0.13556812153999004], 
[-0.56508164119689341, 0.16785863526662462]]], [[[-0.58623010149605825, 0.17414082782961263], [0.58777471173145202, 
-0.17459965739905839]], [[-0.53459263698517323, 0.15880181539405244], [-0.2105745283828401, 0.06255158614140971]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.135679058676)+(1.-msk_arg0)*(-0.629820830028)
      arg1=numpy.array([[-0.15137218142962316, -0.56892857471855862, 0.57687741496105271, -0.39999495525550022, 
-0.21860517149339675], [0.39059915618285679, 0.20078616449879449, -0.67471414803258778, -0.51436995307718436, 
-0.52869142419732817], [0.33338414638611669, 0.25052717773521804, -0.59775810180367017, -0.14479228654166709, 
0.71189655719882228], [-0.94444987904345812, 0.92044245306127359, -0.54335603171718994, 0.094025208931912241, 
0.3880528931344569]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.020538035086050812, 0.077191693471492615, -0.078270184633157325, 0.054270939004074564, 
0.029660143889852865], [-0.052996125830392773, -0.02724247779429043, 0.091544580480203933, 0.069789231044550065, 
0.07173235476498166], [-0.045233247159053046, -0.03399129164778137, 0.081103256568464663, 0.019645281141468216, 
-0.096589454755171444], [0.12814207055494686, -0.12488476559645874, 0.073722034909124351, -0.012757251839663067, 
-0.052650651256845288]])+(1.-msk_ref)*numpy.array([[0.095337352951132076, 0.3583230671558062, -0.36332941231509264, 
0.25192515472597321, 0.13768209055835237], [-0.24600748475526601, -0.12645930878273987, 0.4249490247454204, 
0.32396091078846023, 0.33298087161656914], [-0.20997227979503139, -0.15778723502573058, 0.37648050383386161, 
0.091193198091303551, -0.44836728054893343], [0.59483420673885656, -0.57971382977992481, 0.34221694689676119, 
-0.059219035133039298, -0.24440379524865352]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.86660731561558046, 0.10293397253942249])+(1.-msk_arg0)*numpy.array([0.68626330708301464, 
-0.64806613946281844])
      arg1=numpy.array([[-0.61871824119745367, 0.1369785624298181, -0.69937967814104596, -0.1603390367364621, 
-0.47726573003741879], [0.12320225807433371, 0.79412062731893673, -0.20323882416546035, 0.60524988458207907, 
-0.062024186681118199], [0.29346975098615879, 0.55556808629568799, -0.79331957665352171, -0.35436718901142772, 
-0.64062927975535078], [0.45054591806262989, 0.89434005681699191, -0.53782149229694065, 0.67608265546193191, 
0.5902052140526306]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.53618575412651859, 0.11870662428418587, -0.60608754546990051, -0.13895098221457336, 
-0.41360197314303782], [0.10676797814757631, 0.68819074511582456, -0.17612825183889655, 0.52451397775431541, 
-0.053750613922963479], [0.25432303311648791, 0.48145936790639132, -0.68749654874899713, -0.3070971984114324, 
-0.55517402043352726], [0.39044638861381298, 0.77504163588565911, -0.46608003971981732, 0.58589817518411813, 
0.51147615621246922]], [[-0.063687126449058482, 0.014099747583640465, -0.071989928584400564, -0.016504334004428443, 
-0.049126857549679097], [0.012681697849418309, 0.081741990845436402, -0.020920179545592013, 0.062300774999060358, 
-0.0063843959286142352], [0.030208007289160427, 0.057186830138539854, -0.081659535518239873, -0.036476422502574643, 
-0.065942516690287292], [0.046376481157607639, 0.092057974849305793, -0.055360102719204517, 0.069591873491698333, 
0.060752167295917453]]])+(1.-msk_ref)*numpy.array([[[-0.42460362635675086, 0.094003361252564144, -0.47995861082772856, 
-0.11003479760526946, -0.32752995825286829], [0.084549189066187302, 0.5449758479267317, -0.13947534759945213, 
0.4153607874049105, -0.042564923470918448], [0.20139752184059012, 0.3812659922110605, -0.5444261162479429, 
-0.24318919905269412, -0.43964036813911678], [0.30919313172241331, 0.61375276504804011, -0.36908715592402058, 
0.46397071899877179, 0.40503618205339681]], [[0.40097034198805875, -0.088771168143058882, 0.45324428803161615, 
0.10391030054298601, 0.3092997591632537], [-0.079843211763335295, -0.51464268921437495, 0.13171220016587246, 
-0.39224195611142443, 0.040195775215753432], [-0.19018780857071457, -0.36004486489439247, 0.51412355540212529, 
0.22965337613492687, 0.4151701441578961], [-0.29198355376957985, -0.57959150798834569, 0.3485438982330103, 
-0.43814627648298499, -0.38249201456191473]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.82198701632641225, -0.0055809644583895146, -0.81997433832370969, -0.013440107755883002, 
-0.0052025136098590163], [0.012338171120628516, -0.94554806940956548, -0.25511026330949549, -0.51527037834293887, 
0.62279006579732687], [0.5635167688932019, 0.86230385418831657, 0.73570223287298075, 0.47639158369102952, 0.19187623056909731], 
[-0.27036352797170204, -0.40710182516029136, -0.80722681219478321, -0.90792685183815003, 
-0.31384743450951436]])+(1.-msk_arg0)*numpy.array([[0.89229130193725426, 0.18307842930067797, 0.29290544525661, 
0.027650258983314302, 0.14508277074613707], [-0.1224177638677284, 0.96014567322005839, -0.95987661764430232, 
-0.32853655808001747, 0.38933705747241021], [-0.66540729123951414, -0.70473240520780589, -0.36230465711735671, 
-0.96645064654281287, -0.50720451022365531], [0.30009717637530686, -0.93720424349855591, -0.90344207829791445, 
0.095855935879436416, -0.44243063558063711]])
      arg1=numpy.array([[0.87835048566625273, -0.87601670858751768, -0.81895933805461363, -0.61129077480889138, 
-0.0016464416367591994], [-0.079158624147162682, -0.29676575864334032, -0.8121801333168146, -0.54247405024734552, 
-0.51631405473326453], [-0.24550181580863573, -0.17128786684120079, 0.72400612674452747, -0.11041705408522606, 
0.67698164797895566], [-0.036509002781882849, -0.60084796882118874, 0.48115497910310912, -0.34643162579331177, 
-0.50453128354835663]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.72199269500165819, 0.72007436054393781, 0.67317394278016551, 0.50247308009302138, 
0.001353353648555269], [0.065067361279230143, 0.24393760049508348, 0.66760152450467614, 0.44590662599731978, 
0.424403449337588], [0.20179930507925692, 0.14079640259771445, -0.59512363592477635, 0.090761384839067066, 
-0.55647012492995929], [0.030009926265732567, 0.49388922915711408, -0.39550314566356187, 0.2847622984469525, 
0.41471816440724873]], [[-0.0049020428425125256, 0.0048890181155823009, 0.0045705829585490024, 0.0034115920879498112, 
9.1887322575657513e-06], [0.00044178146794032893, 0.0016562391514554832, 0.0045327484578512002, 0.0030275283940290432, 
0.0028815303888333277], [0.001370136908498085, 0.00095595149699409748, -0.0040406524610174617, 0.00061623365444971945, 
-0.0037782105163525134], [0.00020375544693693209, 0.0033533111588865856, -0.0026853088373516017, 0.0019334225908145693, 
0.0028157711616290208]], [[-0.72022485830049465, 0.71831122098456379, 0.67152564133535508, 0.50124274859730855, 
0.00135003989169023], [0.064908040457684946, 0.24334030658070671, 0.66596686741611733, 0.44481480040935001, 
0.42336427539714022], [0.20130518897495533, 0.14045165527599332, -0.59366644471965579, 0.090539150863186507, 
-0.55510757885883877], [0.029936445398932865, 0.49267991566729918, -0.39453473562123026, 0.28406504313427783, 
0.41370270539117571]], [[-0.011805125174786605, 0.011773758959370197, 0.011006901751140621, 0.0082158138836087111, 
2.2128353011816019e-05], [0.0010639004383453086, 0.0039885637744228614, 0.010915788508965411, 0.0072909096900946133, 
0.0069393165314919495], [0.0032995708587330052, 0.0023021273876210773, -0.0097307203593659346, 0.0014840171049925997, 
-0.0090987062975924175], [0.00049068493144833775, 0.0080754614458602072, -0.0064667747664254203, 0.0046560783807078473, 
0.0067809548171038738]], [[-0.0045696303559049564, 0.0045574888488904607, 0.0042606471021502581, 0.0031802485755245205, 
8.5656350230782907e-06], [0.00041182381946332843, 0.0015439278982821139, 0.0042253781972378382, 0.0028222286294071587, 
0.0026861308967113017], [0.0012772265379895288, 0.000891127458445066, -0.0037666517280097161, 0.00057444622663892773, 
-0.0035220062372353024], [0.0001899385838551262, 0.0031259197352483804, -0.0025032153272353556, 0.0018023152480752904, 
0.0026248308692599638]]], [[[0.01083723859603739, -0.010808444055082557, -0.010104460453754479, -0.007542210184053693, 
-2.0314078654462699e-05], [-0.00097667265040120971, -0.0036615467128846739, -0.01002081746563774, -0.0066931376604521808, 
-0.0063703711592845757], [-0.0030290434138719705, -0.0021133790119741663, 0.0089329114841574381, -0.0013623445079392132, 
0.0083527154182894501], [-0.00045045432376637312, -0.0074133650567978935, 0.0059365724677165983, -0.0042743326806354244, 
-0.006224993312129971]], [[-0.83052260598667949, 0.82831590757544926, 0.77436542102247563, 0.57800481196842468, 
0.0015567897110331862], [0.074848284239467086, 0.2806062901520755, 0.76795535707051754, 0.51293529091616519, 
0.48819975766206297], [0.23213376797439825, 0.16196091180498012, -0.68458259538398514, 0.10440463232017708, 
-0.64011869027220758], [0.034521017096477785, 0.56813063692753385, -0.45495516157774463, 0.32756775495128299, 
0.47705858111587868]], [[-0.22407622367634097, 0.22348085319127922, 0.20892493237088261, 0.15594655052016179, 
0.00042002415947735609], [0.020194177449400059, 0.075707990828744728, 0.20719548766519372, 0.13839069779716881, 
0.13171701445339637], [0.062630032873900315, 0.043697292811580531, -0.18470139363148436, 0.028168523741540826, 
-0.17270496647160757], [0.0093138213128532363, 0.153282483534949, -0.12274757341166896, 0.088378263274868377, 
0.128711108593899]], [[-0.45258798706695413, 0.45138546086862624, 0.42198548796688357, 0.31498002881332571, 
0.00084836260509248021], [0.040788094213415012, 0.15291460473538324, 0.41849236457677358, 0.27952080911217614, 
0.26604133832618604], [0.12649981351559422, 0.088259563952820461, -0.37305891085025844, 0.05689463723400718, 
-0.34882858988534277], [0.018812007676344184, 0.30959916022108025, -0.24792490812404788, 0.17850595489247917, 
0.25997002535981029]], [[0.54702795676119953, -0.54557450358077786, -0.51003974003236807, -0.38070582186452839, 
-0.0010253874952887204], [-0.049299204741037314, -0.18482276635187955, -0.50581771866766068, -0.33784744944688672, 
-0.32155526411941443], [-0.15289609202082347, -0.10667638186031521, 0.45090382331289203, -0.068766644378884939, 
0.42161744508839655], [-0.022737444244723608, -0.37420214603633833, 0.29965854109433676, -0.21575417502209157, 
-0.31421707127789084]]], [[[0.49496522763842138, -0.49365010511969559, -0.4614973200354513, -0.34447260227452836, 
-0.00092779747131777881], [-0.044607212109440506, -0.16723248142883493, -0.45767712448594133, -0.30569322400379262, 
-0.290951627857437], [-0.13834439000189641, -0.096523585272962481, 0.40798959320195816, -0.062221861548812513, 
0.38149051086909613], [-0.020573435283159542, -0.33858790598615962, 0.27113889916106015, -0.19522003040946587, 
-0.28431183871070981]], [[0.75740500911818942, -0.75539258414837984, -0.70619179362800577, -0.52711839114746928, 
-0.001419732969073578], [-0.068258786694342724, -0.25590225746927209, -0.70034605925426996, -0.46777746432543255, 
-0.44521959936809141], [-0.21169716198201677, -0.14770218775286259, 0.62431327354776089, -0.095213051305810245, 
0.58376388426701165], [-0.03148185381138955, -0.51811351929573246, 0.41490179294250989, -0.29872932613429737, 
-0.43505927036232633]], [[0.64620441354972924, -0.64448744854187601, -0.60251021363895751, -0.44972798796155583, 
-0.0012112907884587881], [-0.058237176536220639, -0.21833123127414955, -0.59752273757625574, -0.39909937004262164, 
-0.37985340293096509], [-0.18061623406478455, -0.12601686609912122, 0.53265292405966713, -0.081234073237757498, 
0.49805691003214791], [-0.026859754866597078, -0.44204519227894368, 0.35398679248410975, -0.25487052063395638, 
-0.37118479186079695]], [[0.41843877890233105, -0.41732698714381067, -0.39014533603439461, -0.29121378030692424, 
-0.00078435093879056575], [-0.037710502320269804, -0.14137670974537073, -0.38691577995318882, -0.25843007190862005, 
-0.2459676702163168], [-0.11695499883209939, -0.08160009815153782, 0.34491042532183369, -0.052601755262158906, 
0.32250835941045775], [-0.017392581654241374, -0.28623891542426444, 0.22921818249575437, -0.16503711085233391, 
-0.24035445719126949]], [[0.1685345803081765, -0.16808678395932028, -0.15713883077528235, -0.11729216965199298, 
-0.00031591301511337011], [-0.015188658418393501, -0.056942295130462652, -0.15583806252393731, -0.10408787594301175, 
-0.099068394612065389], [-0.047105963015229846, -0.032866070231711078, 0.13891956650867204, -0.021186408128417322, 
0.12989668677865757], [-0.0070052098356243684, -0.1152884434025082, 0.092322203709857359, -0.066471994507144722, 
-0.096807560891447084]]], [[[-0.237473936100386, 0.23684296789587969, 0.22141673590181513, 0.16527073049388713, 
0.00044513776951372072], [0.021401604893812865, 0.080234637488012114, 0.21958388619206129, 0.14666519805797071, 
0.13959248937905985], [0.066374737045481733, 0.046309991977934166, -0.19574485069977771, 0.029852744290723956, 
-0.18303114671968732], [0.0098707027948385321, 0.16244737662512782, -0.13008675765146716, 0.093662476550452262, 
0.13640685779222486]], [[-0.35757808584515982, 0.35662800093688951, 0.33339984125409727, 0.24885759012834835, 
0.00067026939534456755], [0.03222562036748744, 0.12081388198878236, 0.33064001463220399, 0.22084217595778996, 
0.21019239403782256], [0.099944237295861271, 0.06973160321886579, -0.29474421562493036, 0.044950984246918138, 
-0.27560046449225473], [0.014862881667286663, 0.24460630475095976, -0.19587907017783757, 0.14103294715370429, 
0.20539560638300047]], [[-0.70902806253410877, 0.7071441751024683, 0.66108593577497554, 0.49345030347306046, 
0.0013290518339058898], [0.063898963828039126, 0.23955727731823004, 0.65561357994526626, 0.43789959827955738, 
0.41678254849369595], [0.19817564816323585, 0.13826815871786702, -0.58443715770147708, 0.089131606581156, 
-0.54647773761242324], [0.029471045932029764, 0.48502059048523866, -0.38840119995305028, 0.2796488969325911, 
0.40727117967128218]], [[-0.79747799126147101, 0.79535909238548297, 0.74355517358338064, 0.55500730872994031, 
0.0014948485719980308], [0.071870240417772777, 0.26944160097840825, 0.73740015156782446, 0.49252675664496282, 
0.46877539427376314], [0.22289769074768401, 0.15551685369920368, -0.65734460336669143, 0.10025060830484205, 
-0.61464981640173588], [0.033147503959505158, 0.54552600476516877, -0.43685352542333672, 0.31453457538369362, 
0.45807749992592045]], [[-0.27566804652653937, 0.27493559657766126, 0.25702828721605059, 0.19185204141310383, 
0.00051673148376652042], [0.024843731107889903, 0.093139172000482093, 0.25490065120107763, 0.17025408895811478, 
0.16204384147924006], [0.077050115058967653, 0.053758257570718178, -0.22722746544794023, 0.034654109150746494, 
-0.21246895342821839], [0.011458256859594654, 0.18857459354478276, -0.15100925579298979, 0.108726676988191, 
0.15834584897144408]]]])+(1.-msk_ref)*numpy.array([[[[0.78374449841236027, -0.78166208942434445, -0.73075029398642311, 
-0.54544944131645856, -0.0014691055516275699], [-0.070632551799833562, -0.26480150515026307, -0.72470126856483319, 
-0.48404487656237943, -0.46070254010644734], [-0.21905913485584755, -0.1528386737097901, 0.64602436944342312, 
-0.098524176945782582, 0.60406483606277028], [-0.032576665624677086, -0.5361314163658133, 0.42933040273750561, 
-0.30911792641135383, -0.45018887586543715]], [[0.1608070272912652, -0.16037976304935248, -0.14993378927216161, 
-0.11191415489800628, -0.00030142794879311162], [-0.014492236574465262, -0.054331408962646842, -0.14869266311685764, 
-0.09931529705566107, -0.094525966166430345], [-0.044946086828709382, -0.031359113619550717, 0.13254990448845566, 
-0.020214980829931194, 0.12394073677737169], [-0.0066840108846411943, -0.11000230238028597, 0.08808909782439775, 
-0.063424157910319759, -0.092368794925088121]], [[0.25727364009543341, -0.25659006408105683, -0.2398776495599452, 
-0.17905039657665645, -0.00048225172070397507], [-0.023185992051725324, -0.086924306672343257, -0.23789198357773444, 
-0.15889360322785537, -0.15123119809389254], [-0.071908818670734703, -0.050171148904176843, 0.21206533692261942, 
-0.032341756390756329, 0.19829161103182963], [-0.010693685715702209, -0.17599164183910002, 0.14093291339163105, 
-0.10147170960396128, -0.14777996025362036]], [[0.024286618406791784, -0.024222088866155438, -0.022644437794013716, 
-0.016902348237576709, -4.5524537657303756e-05], [-0.0021887564584318854, -0.0082056500838681053, -0.022456991027312662, 
-0.01499954798106656, -0.01427621733009988], [-0.0067881887879827031, -0.0047361538788586559, 0.020018956909992464, 
-0.0030530601416311233, 0.018718717893569038], [-0.0010094833821416031, -0.016613601947504227, 0.013304059783312148, 
-0.0095789241731956982, -0.013950420655296043]], [[0.12743352214667511, -0.12709493130178839, -0.11881688989338568, 
-0.088687759340826886, -0.0002388703145328296], [-0.011484552519722434, -0.043055598526555189, -0.11783334408657045, 
-0.078703638267764067, -0.074908273635874684], [-0.035618083660724664, -0.024850918316516788, 0.10504081490527493, 
-0.01601961214431067, 0.098218373233072892], [-0.0052968272807739896, -0.087172688113766633, 0.069807297526578757, 
-0.050261260144182598, -0.073198796545300496]]], [[[-0.10752570234739589, 0.10724000657605139, 0.10025517086324084, 
0.074832849725075604, 0.00020155370351078393], [0.0096904217589416242, 0.036329400565627716, 0.099425275778437988, 
0.06640846018754977, 0.063206012033926173], [0.030053783316760119, 0.020968677636373022, -0.088631211062600204, 
0.0135170088539754, -0.082874579525073433], [0.0044693504815987743, 0.073554464767556518, -0.058901916615626208, 
0.042409384962678889, 0.061763591533304643]], [[0.84334441838318952, -0.84110365241878182, -0.78632026497630036, 
-0.58692819251209416, -0.0015808238137436964], [-0.076003810472951086, -0.28493835912127136, -0.77981124087942977, 
-0.52085411217914934, -0.49573670567484834], [-0.23571750621632934, -0.16446130422267244, 0.6951513499785712, 
-0.10601645672963497, 0.65000100015637896], [-0.035053961054603892, -0.57690157752672488, 0.46197887133413784, 
-0.33262482657203868, -0.48442352890311707]], [[-0.84310809328755287, 0.84086795523888092, 0.78609991940007928, 
0.58676372132072352, 0.0015803808294411693], [0.075982512403755115, 0.28485851263921486, 0.77959271928604257, 
0.52070815651122737, 0.49559778849958114], [0.23565145258392778, 0.16441521826703945, -0.6949565520932891, 0.10598674840557479, 
-0.64981885446930565], [0.035044138103840133, 0.57673991603053187, -0.46184941390420736, 0.33253161721150076, 
0.48428778194813499]], [[-0.2885702453487024, 0.28780351425992873, 0.26905808213195226, 0.20083136714178024, 
0.00054091626842049775], [0.026006501919658587, 0.097498400900688217, 0.26683086554087598, 0.17822255731598935, 
0.16962804243040447], [0.080656321568163603, 0.056274326212876459, -0.23786248090949194, 0.036276038902495304, 
-0.2224132205103441], [0.011994542112893573, 0.197400523605883, -0.1580770007375982, 0.11381545394819925, 
0.16575697134067044]], [[0.34197439351876124, -0.34106576761813001, -0.31885121886773615, -0.23799815152412351, 
-0.00064102074215588553], [-0.030819385799020795, -0.1155419072287656, -0.31621182324311842, -0.21120525047844191, 
-0.20102019480149816], [-0.095582954571067871, -0.066688714056679144, 0.28188241497871119, -0.042989450932313884, 
0.26357404278694962], [-0.014214307714350307, -0.23393238016911611, 0.18733146375220353, -0.13487866980175114, 
-0.19643272533949541]]], [[[-0.5844608174260929, 0.58290790514177493, 0.54494151477022601, 0.40675733862528823, 
0.001095554269699891], [0.052672725672010312, 0.19747009959150452, 0.54043058250888909, 0.36096618834281424, 
0.34355913658895182], [0.16335869825160643, 0.11397619549699801, -0.4817589556378884, 0.0734723128654972, 
-0.45046852460053921], [0.024293356646948552, 0.39980861938007123, -0.32016403131140486, 0.2305181297188286, 
0.33571879473150723]], [[-0.61900205037902267, 0.61735736204510694, 0.57714718407462051, 0.43079641801241325, 
0.0011603007747075873], [0.055785647588170659, 0.20914044687204042, 0.5723696588143552, 0.38229904219363198, 
0.36386324563476824], [0.17301308513770361, 0.12071211038191382, -0.51023057908585845, 0.077814476101441751, 
-0.47709090506175361], [0.025729077342214777, 0.42343703423158108, -0.33908550570104568, 0.24414159288537116, 
0.35555954495761488]], [[-0.31823047153817546, 0.31738493323387601, 0.2967127821669342, 0.22147349456613871, 
0.00059651347266978126], [0.028679538179519487, 0.10751961642944766, 0.29425664471887758, 0.19654087476992826, 
0.18706298656500756], [0.088946451198236229, 0.062058391864264706, -0.26231079150104153, 0.040004612920256456, 
-0.24527360384575861], [0.013227381734586688, 0.21769001732342103, -0.17432468972426088, 0.12551379139765426, 
0.18279403369096722]], [[-0.8488823947633436, 0.84662691439671334, 0.79148378175515544, 0.59078236453971011, 
0.0015912045843409354], [0.076502903486464893, 0.28680945931262458, 0.78493201495326348, 0.52427439659424546, 
0.4989920520161048], [0.23726538861569055, 0.16554126965361773, -0.69971618929320634, 0.10671263331001947, 
-0.65426935138688069], [0.035284149343184031, 0.5806899079411737, -0.46501254064149344, 0.33480906873082394, 
0.4876045851863845]], [[-0.44550332788706148, 0.4443196256268705, 0.41537986995107928, 0.31004943804118251, 
0.0008350826239842832], [0.040149611190540065, 0.15052093126384694, 0.41194142673233802, 0.27514528496474749, 
0.261876817252575], [0.12451962824623712, 0.086877978608445941, -0.36721917291438377, 0.056004027837635945, 
-0.34336814519356923], [0.018517530874738959, 0.30475279974482916, -0.24404397551766557, 0.17571168308648133, 
0.25590054256465639]]], [[[0.26359050061632189, -0.26289014070470407, -0.24576738491637995, -0.18344663546442186, 
-0.00049409248625817436], [-0.023755279592317703, -0.089058566213742252, -0.24373296471649636, -0.16279493073610474, 
-0.15494438994833831], [-0.073674401719182256, -0.051403005186393909, 0.21727219431445524, -0.033135846154655882, 
0.20316028101638656], [-0.010956248647121266, -0.18031277887407715, 0.14439325062776281, -0.10396315270767979, 
-0.15140841358587115]], [[-0.82319380244542961, 0.82100657666385946, 0.76753216687755232, 0.57290430816241311, 
0.0015430520886434297], [0.074187798460228119, 0.2781301283256068, 0.7611786674297415, 0.5084089818796611, 
0.48389172307396117], [0.23008534356245425, 0.16053171566338897, -0.67854161430392446, 0.10348333164328342, 
-0.63447007325652283], [0.034216392333081186, 0.56311726607670609, -0.45094048819589283, 0.32467718977559556, 
0.47284885991929299]], [[-0.79353878824430191, 0.79143035583000543, 0.73988232641354446, 0.55226580803768721, 
0.0014874646541099511], [0.071515231914716135, 0.26811067375639663, 0.73375770759602021, 0.49009388337814913, 
0.46645984266264368], [0.22179667070006565, 0.15474866640623086, -0.65409759984649918, 0.099755412822289863, 
-0.61161370701965478], [0.032983769349848584, 0.54283133769289527, -0.43469565430430251, 0.31298090799483497, 
0.45581479137524167]], [[0.084195107833696159, -0.083971401447680033, -0.078502113796428741, -0.058595849313772097, 
-0.00015782120396242417], [-0.007587824000554833, -0.028446759531728334, -0.077852286781768698, -0.051999357776767723, 
-0.049491766924163627], [-0.023532806314437795, -0.016418958780855586, 0.069400284861542538, -0.010584130056389692, 
0.064892709440225974], [-0.0034996046296823279, -0.057594844372613481, 0.046121560824979195, -0.033207527708652602, 
-0.048362318364981026]], [[-0.38860916363588149, 0.3875766291596332, 0.36233270045020055, 0.27045376602327792, 
0.0007284362197977971], [0.035022200393117951, 0.13129826321514301, 0.35933337258932485, 0.24000713883693556, 
0.22843315539485409], [0.1086175244044152, 0.075782999793803998, -0.32032249081985659, 0.048851887417868153, 
-0.29951742079175647], [0.016152701305203677, 0.26583354873289339, -0.21287770321737673, 0.15327196438496837, 
0.22322009645061405]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.301813613463)+(1.-msk_arg0)*(0.727110642101)
      arg1=Data(numpy.array([[-0.84614696308677417, -0.33247673122998478, 0.57230573144163022, -0.70955774937999205, 
0.29481519101937215], [-0.17747847979304732, 0.35977904309928066, -0.78470207141906911, 0.79315989800765641, 
-0.48880985228402341], [-0.85970978747551619, -0.54651625396590786, -0.91193431861718222, -0.89861413456560735, 
-0.013520906220351403], [-0.46786512942225889, 0.69005135718166533, 0.59083978122721126, -0.52809667689026507, 
-0.5516605228719571]]),self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.25537867244997542, 0.10034600364489327, -0.17272966081199209, 0.21415418830105959, 
-0.088979238105345637], [0.05356542129826225, -0.1085862130460596, 0.2368337676669019, -0.23938645487164698, 
0.14752946781416357], [0.25947211748750598, 0.1649460454257213, 0.27523419194278392, 0.27121397906218569, 
0.0040807935636588098], [0.14120806532427299, -0.20826689358605585, -0.17832348934988171, 0.15938676631006105, 
0.16649865581288145]])+(1.-msk_ref)*numpy.array([[-0.61524246164179008, -0.24174736952825798, 0.4161295878665755, 
-0.51592699075938842, 0.21436326284320883], [-0.12904649140142246, 0.26159917104238201, -0.57056522700746204, 
0.57671500272906817, -0.35541884555950498], [-0.62510413559179068, -0.39737778433975512, -0.66307714796362838, 
-0.65339190038498496, -0.0098311948036663883], [-0.34018971467086101, 0.50174368540299019, 0.42960589270690025, 
-0.38398471382505661, -0.40111823700717253]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg0.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.50837565399440865, 0.027194880772694496])+(1.-msk_arg0)*numpy.array([-0.95525264517241903, 
0.39119361412161857])
      arg1=Data(numpy.array([[0.57395007622644179, 0.042614720839584175, 0.067192262504945255, 0.89561435895389252, 
-0.10709780675364455], [-0.27314154100005039, 0.054487430934573089, 0.72790883928028505, -0.48871374949030733, 
-0.68034067253660457], [0.51320482651384713, 0.11688658195513679, 0.51427970574258475, 0.39407629047254322, 
-0.36105065851902607], [-0.96414116432459473, 0.42030377683323783, 0.06595019391169199, 0.82912154649728698, 
0.49091056947243605]]),self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.29178224536175806, -0.021664286576612759, -0.034158910394315525, -0.45530853545996819, 
0.054445917549750845], [0.13885850953894119, -0.027700083335838767, -0.3700511322174258, 0.24845017201319458, 
0.34586863433979215], [-0.26090083931206409, -0.05942229254461371, -0.26144728174293858, -0.2003387918926697, 
0.1835493646497218], [0.49014589495644645, -0.21367220742391727, -0.033527472960914483, -0.42150520844141376, 
-0.24956698180831727]], [[0.015608503892457002, 0.0011589022523941512, 0.001827285567669577, 0.024356125710064319, 
-0.0029125120856824389], [-0.0074280516415664161, 0.001481779187876141, 0.019795394097617793, -0.013290512149365394, 
-0.018501783474447749], [0.013956544069015435, 0.0031787166601977292, 0.013985775281486002, 0.010716857734746538, 
-0.0098187296113277488], [-0.026219704011854205, 0.011430111099293098, 0.0017935076603645462, 0.022547861603065794, 
0.013350254406858457]]])+(1.-msk_ref)*numpy.array([[[-0.54826732851222004, -0.040707824805296992, -0.064185586492968502, 
-0.85553798544510617, 0.10230546319358352], [0.26091917954676891, -0.052049262528900436, -0.69533684416687758, 
0.46684510193274703, 0.64989722705897401], [-0.4902402680426049, -0.11165621659780717, -0.49126704926909737, 
-0.37644241887363145, 0.34489459659154342], [0.92099839754068502, -0.40149629459590852, -0.062999097183777733, 
-0.79202055046098019, -0.46894362003164314]], [[0.22452560464440025, 0.016670606660020789, 0.026285184010318055, 
0.35035861793838979, -0.041895978088456902], [-0.10685122659055797, 0.021315135031497727, 0.2847532895891271, 
-0.19118169793404066, -0.26614492652352695], [0.20076245086861011, 0.045725284437352728, 0.20118293675884424, 
0.15416012830959494, -0.14124071198704816], [-0.37716586659556356, 0.16442015348836053, 0.025799294708336355, 
0.3243470543203793, 0.19204107988242416]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.83545533635143987, 0.88593802697563584, -0.13137432742715816, 0.79834957810499785, 
-0.34659285335561085], [0.79205920222768489, -0.5442503557523124, 0.86098363592905813, -0.76446806321431415, 
-0.35901720438855866], [-0.9188798609420834, -0.017033504748293238, 0.96462600688012778, 0.7697190958071789, 
0.6421550306644519], [-0.53342932023783485, 0.97574947391618738, 0.84854160434718096, 0.72296579812545336, 
0.78585488081390209]])+(1.-msk_arg0)*numpy.array([[0.51969551986658802, -0.35447466539562233, -0.059544973375288857, 
-0.062774266876408635, -0.42472411284195855], [0.39001387745137195, 0.97309649704704704, 0.11415452203338505, 
0.11381837680652263, -0.82397904811236433], [0.9895740872665495, -0.31498857505399092, -0.40291668499781208, 
0.93169920182411015, 0.41199078249099985], [-0.12259217203566819, 0.2331181342170332, 0.30338795758460391, 
-0.12961221673806556, 0.98861933981363448]])
      arg1=Data(numpy.array([[-0.30478629701541937, -0.85322616191583101, -0.070080896897894318, -0.07976899175371166, 
0.56375603635745741], [-0.94577255650508674, 0.086139062157043034, -0.61915925584671561, -0.36178122159045034, 
-0.63253379352393613], [-0.79337541122853206, 0.54284661040204329, 0.81227036754821502, 0.26828885205022979, 
-0.91953815164783226], [0.91196283696851133, 0.6331936479996243, 0.1362363900972261, 0.53126051613995862, 
0.23306894578328841]]),self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.25463533828832702, 0.71283235008723866, 0.058549459289640879, 0.066643429836012402, 
-0.47099298897517416], [0.79015072930691843, -0.071965339147409976, 0.517279904348525, 0.30225205216948448, 
0.52845373322219225], [0.66282972104089521, -0.45352409748067812, -0.67861561312830165, -0.22414335312896641, 
0.768233055772921], [-0.76190421869954095, -0.5290050121651213, -0.11381941911198401, -0.44384443320194866, 
-0.19471869449245272]], [[-0.27002177062705074, -0.75590550245170574, -0.062087331526403451, -0.070670383168119077, 
0.49945291054613067], [-0.83789587267781962, 0.076313870772942369, -0.54853672950854215, -0.32051574165267888, 
-0.56038574103001015], [-0.70288144647478945, 0.48092845496999792, 0.71962120679644015, 0.23768729624493884, 
-0.81465381579970353], [0.80794255645898638, 0.56097033120229245, 0.12069699864501955, 0.47066389347909288, 
0.20648464197653796]], [[0.040041094779414782, 0.11209201316494784, 0.0092068306954528802, 0.010479597641186394, 
-0.07406307010946149], [0.12425023350991971, -0.011316461356087699, 0.081341630807162002, 0.047528764662221085, 
0.083098701699156036], [0.10422916104739342, -0.071316108337680997, -0.1067114732256573, -0.035246267494303277, 
0.12080370621634612], [-0.11980850434530126, -0.083185389637099372, -0.017897964120127029, -0.069793992996491958, 
-0.030619275996436303]], [[-0.24332601163444462, -0.68117274639365022, -0.055949054471653784, -0.063683540912436762, 
0.45007439378012193], [-0.75505712146912118, 0.068769083931435493, -0.4943055306850298, -0.28882788562304679, 
-0.50498308719698826], [-0.63339092483317772, 0.43338136239019942, 0.64847570523930897, 0.21418829184457513, 
-0.73411289541949643], [0.72806514614124795, 0.50550988173926459, 0.10876426455666836, 0.42413160892417934, 
0.18607049453546493]], [[0.10563675234626491, 0.29572209001606431, 0.024289538021561568, 0.027647362461219117, 
-0.19539381323758065], [0.32779800898452871, -0.029855183338385865, 0.21459617316544985, 0.1253907858815127, 
0.21923169234130982], [0.27497824755987804, -0.18814675563366581, -0.2815271043847466, -0.092986998755590466, 
0.31870535174896658], [-0.3160798018191941, -0.21946039318683805, -0.047218559174665678, -0.18413109816412282, 
-0.080780030947614101]]], [[[-0.24140879126396328, -0.67580563312684261, -0.055508219288346809, -0.063181763970951627, 
0.44652815640832944], [-0.74910785659425705, 0.068227236852748466, -0.49041078623783663, -0.28655214575388938, 
-0.50100421188062005], [-0.62840029528473251, 0.4299666531670453, 0.64336621931342752, 0.21250065412148639, 
-0.72832865481210196], [0.72232855711057531, 0.50152685569021993, 0.10790728645478857, 0.42078978058888372, 
0.18460440326115896]], [[0.16588005057907193, 0.46436864215987111, 0.038141553068120108, 0.043414302139960843, 
-0.30682442334505977], [0.51473705033866746, -0.046881215223141225, 0.33697764526191198, 0.19689955855510877, 
0.34425674215076196], [0.4317948498062657, -0.295444460830249, -0.44207843650517759, -0.14601630317271708, 
0.50045896616215646], [-0.49633609845300036, -0.3446158681839, -0.074146703776826112, -0.28913872490632958, 
-0.12684785665737114]], [[-0.26241601418568961, -0.7346137631560874, -0.060338505420318503, -0.06867979655450572, 
0.48538472195999799], [-0.81429469446167013, 0.074164322931490043, -0.53308598731803514, -0.31148771157580218, 
-0.54460124539623866], [-0.68308324621625327, 0.46738204837571612, 0.69935149440909461, 0.23099231131744, 
-0.79170730118123622], [0.78518507920532776, 0.54516936930190074, 0.11729730249175926, 0.45740661081172962, 
0.20066854836264816]], [[0.23299939017364035, 0.65226415148357819, 0.053574607519855308, 0.060980846630518552, 
-0.43097348523956391], [0.72301291451269412, -0.065850562014292111, 0.47332747713835471, 0.27657018977656017, 
0.48355188405284633], [0.60651016402373592, -0.41498889687650542, -0.62095475468596306, -0.20509825910883084, 
0.70295754984188863], [-0.69716646370074919, -0.48405632172587898, -0.1041483692769362, -0.40613169783575104, 
-0.17817376557835249]], [[0.10942352429041677, 0.30632287136220132, 0.025160247685324832, 0.028638440416311549, 
-0.20239811613022901], [0.33954861922387636, -0.030925405284273878, 0.22228882510538819, 0.12988568277568113, 
0.22709051423225335], [0.28483542216989066, -0.19489127247834664, -0.29161903656482718, -0.096320313631689122, 
0.33013001653322727], [-0.32741034823469384, -0.22732741334141818, -0.048911207908695235, -0.19073166530659069, 
-0.083675761344904745]]], [[[0.28006199021858108, 0.78401233701336637, 0.064395924796233614, 0.073298120050140775, 
-0.51802406829340053], [0.86905135520423282, -0.079151449456545175, 0.56893297091343387, 0.33243347858649008, 
0.58122256423444296], [0.72901668754454174, -0.49881081787911086, -0.74637888238007877, -0.24652522306422633, 
0.84494508891710052], [-0.8379842848179736, -0.58182889122330528, -0.12518487518779056, -0.48816458919470462, 
-0.21416236049126602]], [[0.0051915788374268593, 0.014533431880361323, 0.0011937232900749318, 0.0013587454998034117, 
-0.0096027411221737257], [0.01610982133203483, -0.0014672501242655189, 0.010546452124414738, 0.0061624021558042636, 
0.010774267375445901], [0.0135139638343403, -0.0092465803158780936, -0.013835811162530414, -0.0045698994353117308, 
0.015662957472330138], [-0.01553392331377011, -0.010785507009790718, -0.0023205831976114305, -0.0090492285242507013, 
-0.0039699809946793421]], [[-0.2940047886417646, -0.82304414553452543, -0.06760185573319373, -0.076947243988236727, 
0.54381373420606227], [-0.91231680459831188, 0.083091979564947543, -0.59725712059028868, -0.34898357514701078, 
-0.61015854746373377], [-0.76531055489025812, 0.52364395814053544, 0.78353712115508833, 0.25879840404366655, 
-0.88701041539798187], [0.87970306984800806, 0.6107950602517388, 0.13141716497125061, 0.51246771029716398, 
0.2248243664986945]], [[-0.23459983295312686, -0.65674446986888302, -0.053942604593603345, -0.061399716206117248, 
0.4339337865609012], [-0.72797919703233938, 0.066302881037197545, -0.47657870257097967, -0.27846991476261806, 
-0.48687333961872892], [-0.61067620416647439, 0.41783940212065268, 0.62522001286017692, 0.20650705261524888, 
-0.70778607464657395], [0.70195521028115226, 0.48738124220911994, 0.10486375100167097, 0.40892136412130414, 
0.17939761820904515]], [[-0.19572005390604136, -0.54790347216877311, -0.045002800496459619, -0.051224059345677121, 
0.36201877481439293], [-0.60733260502412101, 0.055314632100863098, -0.39759623092442686, -0.23231963144423851, 
-0.40618475757666528], [-0.50947001152588012, 0.34859168174881788, 0.52160350278074963, 0.17228303601524592, 
-0.59048604996854714], [0.58562152353835495, 0.4066084864477349, 0.087484883260498456, 0.34115161303266767, 
0.14966639602639906]]], [[[0.16258194723474198, 0.45513585155989855, 0.037383205193901557, 0.042551119047239863, 
-0.30072399925413462], [0.50450281191610769, -0.045949101372356069, 0.3302777009652772, 0.19298471110780743, 
0.33741207150693225], [0.42320970630504856, -0.28957029838017473, -0.43328883001058061, -0.14311313997654312, 
0.49050861116625827], [-0.48646771620628038, -0.33776405723135444, -0.072672484961219808, -0.28338993599373941, 
-0.12432580931772833]], [[-0.29739506896965828, -0.83253497862089987, -0.068381398279694944, -0.077834551738508839, 
0.55008465589286404], [-0.92283707445420604, 0.0840501445833685, -0.60414431816277081, -0.35300783663963753, 
-0.61719451626519095], [-0.77413564012427893, 0.52968229451697924, 0.79257238381287898, 0.26178270624558952, 
-0.89723886771623562], [0.88984725840313872, 0.61783836892270494, 0.13293258596560883, 0.51837716913600684, 
0.22741690123424407]], [[-0.25862385345250039, -0.72399789630304678, -0.059466556687828624, -0.067687308239851543, 
0.47837045155116459], [-0.80252736244436118, 0.073092577999698843, -0.52538238830257877, -0.30698641819104372, 
-0.53673123996060923], [-0.67321204429346304, 0.4606279337049789, 0.68924520084303675, 0.22765425294716546, 
-0.78026637845769298], [0.7738384087862672, 0.53729115393604532, 0.11560224502356863, 0.45079665069171193, 
0.19776869717845769]], [[-0.22035006847945415, -0.61685333313099611, -0.050666091559133777, -0.057670252788884858, 
0.40757633277321131], [-0.68376121115885047, 0.062275595822144654, -0.4476309655699825, -0.26155544961394145, 
-0.45730029887635321], [-0.57358328739194542, 0.39245953294901026, 0.58724369456815062, 0.19396366405065604, 
-0.66479463371287917], [0.65931794028969248, 0.45777735109401574, 0.09849425050037168, 0.38408318306366546, 
0.16850087640647313]], [[-0.23951779911476295, -0.6705119437796685, -0.055073414879026103, -0.062686851507258209, 
0.44303043275980752], [-0.74323997966936439, 0.067692802424844351, -0.48656932320824503, -0.28430753877367126, 
-0.49707976892051819], [-0.6234779392316786, 0.42659865831772847, 0.63832663287826696, 0.21083610385163193, 
-0.72262354456704303], [0.71667044655259748, 0.49759831878086463, 0.10706203210237189, 0.41749366959229928, 
0.18315836860994791]]]])+(1.-msk_ref)*numpy.array([[[[-0.15839607307564069, -0.44341781378062139, -0.036420728146067942, 
-0.041455587638678751, 0.29298148639271593], [-0.49151376042846306, 0.044766084688524818, -0.32177429134746866, 
-0.18801608003241838, -0.32872497865860706], [-0.41231364678778004, 0.28211495140070508, 0.42213327093519415, 
0.13942851444065429, -0.47787985775778163], [0.47394300065735895, 0.3290679020733861, 0.070801441576325203, 
0.27609371011994766, 0.12112488694360368]], [[0.10803902065171155, 0.30244705825190527, 0.024841902478506196, 
0.028276086660843098, -0.199837232352572], [0.33525241050750293, -0.030534115235610542, 0.21947627004286704, 
0.12824227746969438, 0.22421720481082091], [0.28123148342834819, -0.19242537058341205, -0.28792926674743269, 
-0.09510160105988083, 0.32595297862387435], [-0.32326772148765553, -0.22445110650530029, -0.048292348794421701, 
-0.18831839369661746, -0.082617036570641594]], [[0.01814849194093603, 0.050805329094378057, 0.0041729651399014809, 
0.0047498424901483973, -0.033568838175063176], [0.056316001696174268, -0.005129148162713479, 0.036867821404456345, 
0.021542253207282844, 0.037664207894353235], [0.047241517738211787, -0.032323786963255469, -0.048366617409194554, 
-0.015975252552217743, 0.0547538747574325], [-0.054302802846542902, -0.037703498911539651, -0.0081121922210847953, 
-0.031633893288896049, -0.013878084171272551]], [[0.019132736349118284, 0.053560646794038221, 0.0043992769248064958, 
0.0050074399768095358, -0.035389371879489359], [0.05937017886643358, -0.0054073164763297713, 0.038867268365520298, 
0.022710550954992061, 0.039706845163018721], [0.049803559797640319, -0.034076797994331692, -0.050989676828270179, 
-0.016841635998566435, 0.057723333334580539], [-0.057247798509228072, -0.039748267043975168, -0.0085521395102417866, 
-0.033349489421068357, -0.014630732203203361]], [[0.12945008960625967, 0.36238572467325059, 0.029765046762126927, 
0.033879814254892691, -0.23944078240122002], [0.4016924100118941, -0.036585336755688425, 0.26297186564738351, 
0.15365720838288405, 0.26865235429701234], [0.3369656676846623, -0.23056004501227215, -0.34499081124472725, 
-0.11394874467242132, 0.39055002568295988], [-0.38733260687628662, -0.26893261040380378, -0.057862879920835344, 
-0.22563915140550492, -0.098990001228817709]]], [[[-0.11887088549302922, -0.33277004375174535, -0.027332522334417589, 
-0.0311110137742516, 0.21987267767638857], [-0.36886442194964564, 0.03359542963189309, -0.24148070213268358, 
-0.14109969702158554, -0.24669695743129583], [-0.30942742040781651, 0.21171771138423517, 0.31679671558633038, 
0.10463637546508758, -0.3586326399886387], [0.35567816213764247, 0.24695430983391262, 0.053134082751796846, 
0.20719897383656244, 0.090900123258443891]], [[-0.29658647797364546, -0.83027138934919176, -0.068195475281256232, 
-0.077622926448511598, 0.54858902416856947], [-0.92032796173833031, 0.083821619643936432, -0.60250170297869532, 
-0.35204803942706875, -0.61551641874202234], [-0.772030833509745, 0.52824213501609141, 0.79041744931628544, 
0.26107094212685206, -0.89479935426962187], [0.88742784209114567, 0.6161585208208753, 0.13257115397394573, 0.5169677472751999, 
0.22679857471216608]], [[-0.034792734058120535, -0.097399624699881296, -0.0080000512890500627, -0.0091059911267299884, 
0.0643553008738212], [-0.10796421414013084, 0.0098331634689412944, -0.07067982891372819, -0.041298962431312022, 
-0.072206592869688793], [-0.090567390861833394, 0.061968395347888443, 0.09272433556934849, 0.03062638567267954, 
-0.10496943819282063], [0.10410468176635027, 0.072281918241972568, 0.015551999995102638, 0.060645790295166424, 
0.026605874106716224]], [[-0.034690281599165727, -0.097112816798119131, -0.0079764939300635985, -0.0090791771609003494, 
0.064165796973084763], [-0.10764629720956417, 0.0098042082343507975, -0.070471701485207622, -0.041177351400505935, 
-0.071993969654166545], [-0.090300701504238898, 0.061785920050883347, 0.092451294762375361, 0.030536201655642452, 
-0.10466033983222632], [0.10379812981162738, 0.072069073219517887, 0.015506204782846488, 0.060467209608445503, 
0.026527529093061313]], [[0.25113752289245761, 0.70304048071997249, 0.057745190716787707, 0.065727977894106379, 
-0.46452316220541723], [0.77929677083985871, -0.070976782441452099, 0.5101742542625366, 0.29810014659102763, 
0.52119459308675564], [0.6537247161398414, -0.44729423331009915, -0.66929376426225851, -0.22106439293150729, 
0.75768017089778372], [-0.75143827031916521, -0.52173829934952587, -0.1122559310305771, -0.43774753438868647, 
-0.19204392809106624]]], [[[-0.3016086216803851, -0.84433050040979962, -0.069350239582554929, -0.078937327206852134, 
0.55787836511943856], [-0.93591201436527227, 0.085240983812052432, -0.61270395547714962, -0.3580093221455472, 
-0.62593905139169714], [-0.78510374842619801, 0.53718693901434222, 0.80380170758018954, 0.26549169589139648, 
-0.90995112712367354], [0.9024547920141277, 0.62659202628220501, 0.13481600138295211, 0.5257216403599555, 
0.23063898929367455]], [[0.096004201392869387, 0.26875649294065335, 0.022074681852373382, 0.025126321045995187, 
-0.1775767105703214], [0.29790754989870738, -0.02713282044533414, 0.19502809173064634, 0.11395695147006808, 
0.19924091829559995], [0.24990419026574937, -0.17099048028342859, -0.2558558856325937, -0.084507923210172864, 
0.2896440120953313], [-0.28725787451890639, -0.19944876491663996, -0.042912906387224892, -0.16734099296137331, 
-0.073414055121613878]], [[0.12280348442621132, 0.34377905671253306, 0.02823676265977303, 0.032140257723023313, 
-0.22714671331665276], [0.38106754322893543, -0.034706865373136261, 0.24946959485147086, 0.14576769049768312, 
0.25485841923575486], [0.31966419065097607, -0.21872195672549011, -0.32727728381448118, -0.10809805488994705, 
0.37049726379095999], [-0.36744504311255272, -0.25512428561368011, -0.054891914674043092, -0.21405372603333878, 
-0.093907367010937354]], [[-0.28396914965619241, -0.79495013403242876, -0.065294315702885894, -0.074320705947247184, 
0.5252510490977671], [-0.88117553600293741, 0.080255695457594409, -0.57687018447439498, -0.33707127539077408, 
-0.58933123055302783], [-0.73918723738849845, 0.50576975362450738, 0.75679165311004848, 0.24996450931350586, 
-0.85673296193710291], [0.84967504729681309, 0.58994601644134648, 0.12693133591298367, 0.49497499884826424, 
0.21715015075627661]], [[-0.12556914499991692, -0.35152131408949577, -0.028872683550634565, -0.032864089331129782, 
0.23226229055293343], [-0.38964957561304403, 0.035488499621121036, -0.25508790630283351, -0.14905052857359943, 
-0.26059809254592697], [-0.32686335648116172, 0.22364779979212476, 0.33464790432044117, 0.11053253408978626, 
-0.37884124262771812], [0.37572028280536912, 0.26086994650769596, 0.056128136959905282, 0.21887443575107401, 
0.096022257347609413]]], [[[0.037364414157828554, 0.10459884842691844, 0.0085913693689205848, 0.0097790539601828163, 
-0.069112076995279825], [0.11594431195368531, -0.010559974726947335, 0.075904078010236853, 0.044351545756490682, 
0.077543691634060205], [0.097261614902197202, -0.066548745051386637, -0.099577988637946202, -0.032890113105793699, 
0.1127281792801714], [-0.11179950499977977, -0.077624584627462273, -0.016701514972317543, -0.065128380590387683, 
-0.028572428297636714]], [[-0.071051212895153079, -0.19890249093097878, -0.016337127929093392, -0.018595598525999169, 
0.13142175534924042], [-0.22047673376613944, 0.020080577453254922, -0.14433725050619306, -0.084337763371924834, 
-0.1474550977755221], [-0.18495019559926684, 0.12654738898296505, 0.18935495256262369, 0.062542996621179237, 
-0.214361018253522], [0.21259507502937181, 0.1476089218197493, 0.031759173071929249, 0.1238464603057252, 
0.054332597784931061]], [[-0.092468492151282541, -0.25885854262139452, -0.021261700175549361, -0.02420095148674169, 
0.17103679244648071], [-0.28693600425764765, 0.02613355413607853, -0.18784546205093827, -0.10976006591078974, 
-0.19190313572046855], [-0.24070054561046958, 0.16469312441160111, 0.2464330478169485, 0.081395606866237188, 
-0.27897680174955763], [0.27667854250093776, 0.19210332762215063, 0.041332480140296786, 0.16117804293704455, 
0.070710311437588644]], [[0.03950402758755496, 0.11058853422482251, 0.0090833403979279047, 0.010339035848159039, 
-0.073069669571755536], [0.12258367757865166, -0.011164674793912362, 0.080250603664183892, 0.046891266104543569, 
0.081984107140375212], [0.10283114575480438, -0.070359552522953867, -0.10528016292856741, -0.034773512840341185, 
0.11918337821029902], [-0.11820152488222384, -0.082069632341693691, -0.017657900520893319, -0.068857853162308885, 
-0.030208582715776026]], [[-0.30131762773962623, -0.84351588490495, -0.06928333002474367, -0.078861167965153675, 
0.55734012047966086], [-0.93500904042591215, 0.085158742761861514, -0.61211281475468116, -0.35766391244572121, 
-0.62533514136344748], [-0.78434627527312217, 0.53666865759573723, 0.80302619451569457, 0.26523554779325603, 
-0.90907320041552964], [0.90158409781837889, 0.62598748625957545, 0.13468593003651244, 0.52521442073533664, 
0.23041646731133436]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.464030061953)+(1.-msk_arg0)*(-0.96738565521)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-0.67402898847849047, 0.20486008013328139, 0.86910684630548163, 0.11617789766533293, 
-0.67091974768759255], [0.63735266421821568, 0.52482421330068418, 0.96549957417820953, 0.93314176044328634, 
-0.66620391933821188], [-0.56446554507968827, 0.97270940295746322, 0.4273768968805165, 0.0795646430337138, 
-0.56522717749374385], [0.64133192382437154, -0.69241677052592632, 0.75734947668207875, -0.69971855592277477, 
-0.31147593247856209]])+(1.-msk_arg1)*numpy.array([[0.95121411016144797, 0.1130560179342579, 0.52012252957284932, 
-0.68225588101818579, 0.17730561916900478], [0.59947361921565534, 0.7649078291107767, 0.45338418670272795, 
-0.85303367434739097, -0.50029340792348398], [0.35754933425934743, 0.33821264868230383, -0.74573645227047036, 
-0.75507994872497042, -0.53190481570081594], [-0.93781246725424849, -0.38341219731292941, 0.42257095557524416, 
-0.18386027017029849, -0.38407080653131875]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.31276971328169517, 0.095061235675913722, 0.4032917037347844, 0.053910037051197068, 
-0.31132693208486845], [0.29575079626299683, 0.2435342122122757, 0.44802082722137121, 0.43300582890929579, 
-0.30913864596374629], [-0.26192898185358071, 0.4513664045164778, 0.19831572793668567, 0.036920386236191136, 
-0.26228240218986021], [0.29759729234456783, -0.32130219692434248, 0.35143292458474851, -0.32469044485440851, 
-0.14453419624485092]])+(1.-msk_ref)*numpy.array([[-0.92019088520370329, -0.10936876998478624, -0.5031590740604085, 
0.66000455247977807, -0.17152291257225483], [-0.57992217990615635, -0.73996086143972717, -0.43859735851535431, 
0.82521253997490041, 0.48397666622139479], [-0.34588809699244344, -0.32718206474590183, 0.72141474649378612, 
0.73045351093337663, 0.51455708864618532], [0.90722632809902926, 0.37090745971314409, -0.40878908073195058, 
0.17786378792581542, 0.37154458882340313]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.38164549218084232, -0.84943064650584477])+(1.-msk_arg0)*numpy.array([-0.063913606660673583, 
-0.89479132834762409])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.22046330966882177, 0.83542118723512249, 0.49831593535934582, 0.6272392679616996, 
-0.31521766907302196], [0.40162162896491904, 0.06623229899035965, 0.95859730892263073, 0.25588908389287979, 
-0.8272136207389138], [0.81517961958285157, 0.77730439461900702, -0.079977269227189174, 0.80967752989262998, 
-0.21695203317986378], [-0.28088513635915779, -0.40055183374451619, 0.96377618778594387, 0.97555828700940062, 
-0.27852718947653399]])+(1.-msk_arg1)*numpy.array([[0.18744921804733261, 0.84141914602805357, -0.67806681231531796, 
-0.27641832837831171, -0.7576319052308631], [0.35459885597183805, 0.60938573262227713, 0.59102751490644856, 
0.58222541360655056, 0.7558014619438469], [0.06759826978510608, -0.45978302887178413, 0.20446940512042167, 0.11211057267987212, 
-0.37033917552328299], [0.53135465885908939, 0.33701500564107301, 0.54425303922014434, -0.18130551567932396, 
-0.078814758820267894]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.08413882832637494, 0.31883473018065195, 0.19018003041177434, 0.23938303913639408, 
-0.12030140245747134], [0.15327708425678815, 0.025277258346444516, 0.36584434176700836, 0.097658915366002963, 
-0.31570234942559938], [0.31110962713148921, 0.2966547182587026, -0.030522964277490346, 0.30900977940364144, 
-0.082798765482563541], [-0.10719854611207379, -0.15286880173336481, 0.36782083753974248, 0.37231742259680212, 
-0.10629864631351854]], [[-0.18726829166280554, -0.70963235917781053, -0.42328482713645388, -0.53279625689855925, 
0.26775554843076249], [-0.34114971994240167, -0.056259744550949606, -0.8142619318569132, -0.21736002996491724, 
0.7026606006626962], [-0.69243855128065024, -0.66026617445305746, 0.067935143505423304, -0.68776490767795218, 
0.18428570580472917], [0.23859244297144178, 0.34024100309670602, -0.81866103027795278, -0.82866910643852965, 
0.2365895306265082]]])+(1.-msk_ref)*numpy.array([[[-0.011980555591128053, -0.053778132335996882, 0.043337695531978013, 
0.017666892313772321, 0.048422987584502107], [-0.022663691802908902, -0.038948040019446621, -0.037774700113366144, 
-0.037212126073097061, -0.048305997352241087], [-0.0043204492259873656, 0.029386391656564335, -0.013068377133008548, 
-0.0071653910447642046, 0.023669712395433262], [-0.033960792663636233, -0.021539844509288229, -0.034785174672592456, 
0.011587889414538899, 0.0050373354942944565]], [[-0.16772793481429618, -0.75289455537156558, 0.60672830370006248, 
0.24733672322925929, 0.67792245888006519], [-0.31729198136558878, -0.54527306916917739, -0.5288462951531363, 
-0.52097025123875018, -0.67628459409981101], [-0.060486345615016132, 0.41140986715587774, -0.18295745061415059, 
-0.10031556825003562, 0.3313762828056423], [-0.47545154102422327, -0.30155810457065774, -0.48699289992102451, 
0.16223060321145327, 0.070522762738185127]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank2_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.7404773332238368, 0.54735446305705127, -0.60976156515517843, -0.91988988534917771, 
0.94982879508028195], [-0.47925259590889246, 0.018461915276561269, 0.074865997893243552, -0.91906868677879183, 
-0.66380093527647133], [0.32921485725414845, 0.2327164935228383, -0.99951906447055583, -0.29839110112065903, 
-0.97521805679670415], [0.42585346098043475, 0.84140569875823989, -0.25614418347587686, 0.93757162920473247, 
-0.24299196943566526]])+(1.-msk_arg0)*numpy.array([[0.74735966835600798, 0.55356556314205907, -0.79664761259132577, 
-0.083573228323402304, -0.09886577387565576], [-0.11423197215038416, 0.50110483461189959, -0.32401081610725013, 
-0.95250271200920489, -0.08323616314293858], [0.47358512315705692, -0.86022323801341516, -0.11039972820750443, 
-0.083964682778386868, -0.78651382635546296], [-0.020880118242885715, 0.52612289356883424, -0.90535873207328499, 
-0.43573562176556124, 0.029851727443945597]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-0.94170691718388344, 0.88462212402533158, -0.10385925507555438, -0.97727211719508178, 
0.67739494200640826], [0.27883639324142595, -0.36261004110559969, 0.90290987220864816, -0.69147390917696439, 
-0.98681627347326661], [0.064480872009119894, 0.44687978524198879, -0.86518312423678867, -0.41416292585597159, 
0.16212940218186911], [0.78165062167133437, 0.89987109000799292, -0.91765215528596289, 0.6706855143282453, 
-0.49690520527620263]])+(1.-msk_arg1)*numpy.array([[-0.55087645570753607, 0.069507565325803045, -0.60607347319922145, 
0.50137395826357256, 0.765268821355368], [-0.43076688393051321, 0.5987114326134837, -0.58866918847603333, -0.4089799468708637, 
-0.46204382982823478], [0.44673372515774679, -0.8896304761646654, 0.50531756290906094, 0.55178853719531307, 
-0.42163121911561885], [0.25546186385324576, -0.33184726578369128, -0.22570142190403009, 0.8324772605226114, 
-0.93173636328846476]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.69731262671476257, -0.65504263130908369, 0.076905424228960739, 0.72364785117462704, 
-0.50159560019622074], [-0.20647202887316415, 0.26850451623806032, -0.66858429431453503, 0.51202075626122012, 
0.73071508256336892], [-0.04774662414926064, -0.33090435164762877, 0.6406484925851248, 0.30667825885801148, 
-0.12005314736480535], [-0.57879456784794381, -0.66633414497434584, 0.67950062077325601, -0.49662742108163649, 
0.36794704126796574]], [[-0.51544748401229556, 0.48420186770427359, -0.056847826795385394, -0.53491425496794165, 
0.37077514475947998], [0.1526223443034255, -0.1984762243484508, 0.49421174829167541, -0.37848133027551756, 
-0.54013829150292014], [0.035293893076002267, 0.24460164490217914, -0.47356184441264954, -0.22669392590003265, 
0.088742251877017689], [0.42783995632312355, 0.4925484572918884, -0.502281002729694, 0.36710270957527896, 
-0.27198328182420972]], [[0.57421668373950274, -0.53940857091658456, 0.063329381930720949, 0.59590297576338802, 
-0.4130494000660288], [-0.1700237155651167, 0.22110566620553404, -0.55055973687200743, 0.42163421312371552, 
0.60172263543365967], [-0.039317957438851685, -0.2724901172853651, 0.5275554159804714, 0.25254063389918535, 
-0.09886027803208991], [-0.47662050647483106, -0.54870680428117025, 0.55954901447519156, -0.4089582489436967, 
0.30299369570297252]], [[0.8662666680808101, -0.81375494424700834, 0.095539078243902717, 0.89898273584153199, 
-0.62312875553838776], [-0.25649877781003355, 0.33356130913909071, -0.83057765882665402, 0.63607985503474551, 
0.90776230866602603], [-0.059315301959684298, -0.4110801944111182, 0.79587320496022285, 0.3809842863815297, 
-0.14914119718481031], [-0.71903250075235725, -0.82778231381649214, 0.84413893591643019, -0.61695682088076387, 
0.45709807231093563]], [[-0.89446034646753492, 0.84023956616434048, -0.098648511106349479, -0.92824119754096057, 
0.64340922155942426], [0.2648468354170353, -0.34441745842734328, 0.8576097959860316, -0.65678182998300838, 
-0.93730651199872683], [0.061245788966148226, 0.42445928796213339, -0.82177584441762286, -0.39338387283270165, 
0.15399517472149118], [0.7424342681558368, 0.85472347314987185, -0.87161244095808987, 0.63703641395219646, 
-0.47197487239661573]]], [[[0.45131548464573656, -0.4239574493375784, 0.04977481760412325, 0.46836019907512233, 
-0.32464328441212481], [-0.13363306529482616, 0.17378180350248884, -0.43272190012776102, 0.33139066597632993, 
0.47293426074720257], [-0.03090262529683975, -0.21416829713643151, 0.41464125822704678, 0.19848865734569654, 
-0.077700936868817627], [-0.37460808952978658, -0.43126555586969523, 0.43978717756218783, -0.32142777378030224, 
0.23814310954926121]], [[-0.017385713320400557, 0.016331818705527346, -0.0019174407678916509, -0.018042315029801255, 
0.012506008028093443], [0.0051478538680451271, -0.0066944758573219808, 0.016669445563086825, -0.012765932727177739, 
-0.018218518434395363], [0.0011904403959911625, 0.0082502567339454921, -0.015972937538370174, -0.0076462408478456742, 
0.0029932192869211952], [0.014430767553167622, 0.016613343823554404, -0.016941616344243293, 0.012382139142744985, 
-0.0091738218002915383]], [[-0.070501828077941503, 0.066228118073597117, -0.0077755267716802991, -0.073164452267052663, 
0.0507138483011456], [0.020875364828972225, -0.027147162573480783, 0.06759724859056146, -0.051767884227675497, 
-0.073878985050868026], [0.0048274248279892773, 0.033456101060459861, -0.064772797956381301, -0.031006720734592753, 
0.01213797948218065], [0.058519053795298634, 0.067369747128729174, -0.068700944324369309, 0.050211540302727384, 
-0.037201304051349943]], [[0.86549333970669629, -0.81302849382342701, 0.095453789172113335, 0.8981802013760134, 
-0.62257247978042551], [-0.25626979776253211, 0.33326353429172723, -0.82983619053040902, 0.63551201764907028, 
0.9069519365530162], [-0.059262350359773175, -0.41071321737034316, 0.79516271781547765, 0.38064417637890996, 
-0.14900805675152104], [-0.71839061037929952, -0.82704334096384602, 0.84338536127839792, -0.61640605489521894, 
0.45669001446674551]], [[0.6251059323829844, -0.58721299329427368, 0.068941870656270596, 0.64871414541371253, 
-0.44965539605540483], [-0.1850918586227765, 0.24070088442653678, -0.59935241764245983, 0.45900102763094675, 
0.6550495652775965], [-0.042802463147096223, -0.29663921939978083, 0.57430936705379976, 0.27492173754003379, 
-0.10762164880413988], [-0.51886041372486702, -0.5973352711755634, 0.60913835893729185, -0.44520167168747043, 
0.3298461400060903]]], [[[-0.3100239083159364, 0.29123074628486112, -0.034192009834220827, -0.3217325005608383, 
0.22300847913732186], [0.091797083398237653, -0.11937661292150091, 0.29725134469253151, -0.22764348430466233, 
-0.32487457860757218], [0.021228061074105423, 0.14711946470820586, -0.2848311387443126, -0.13634858851563417, 
0.053375407996004466], [0.25733099783614471, 0.29625093244411638, -0.30210472331142996, 0.22079963586199841, 
-0.1635885762238484]], [[-0.21915073169323523, 0.20586615879590053, -0.024169761661077063, -0.22742734033127973, 
0.15764097563383772], [0.06488982770169989, -0.084385337282267422, 0.21012201942755063, -0.1609173835061927, 
-0.22964842291397289], [0.015005762433257314, 0.10399629664775466, -0.20134238292751963, -0.096382543852360975, 
0.037730185972718586], [0.18190299183529962, 0.20941484468923452, -0.21355279185182438, 0.15607958115103057, 
-0.11563803698512404]], [[0.94125401686908639, -0.88419667781575539, 0.10380930546972694, 0.97680211231198755, 
-0.67706915871133155], [-0.27870229091301407, 0.36243564905349879, -0.90247563077121717, 0.69114135480635752, 
0.98634167846631959], [-0.06444986086680117, -0.44666486487587553, 0.86476702693286767, 0.41396374018994891, 
-0.1620514283919923], [-0.78127469811576045, -0.89943831002888841, 0.91721082376081486, -0.67036295783532129, 
0.49666622590821957]], [[0.28099696395144025, -0.26396336966361489, 0.030990677483566067, 0.29160930314435818, 
-0.20212862263885711], [-0.083202298411822181, 0.10819960944290732, -0.26942027098105203, 0.20632966115552098, 
0.29445719444547341], [-0.019240518400021566, -0.13334495118692066, 0.25816294511202731, 0.12358253148951723, 
-0.048377970841082099], [-0.23323758969215713, -0.2685135254141327, 0.27381923706152445, -0.20012658912608067, 
0.1482720913549532]], [[0.91836958984808159, -0.86269946879135684, 0.10128542091513537, 0.95305341509258856, 
-0.66060777902740553], [-0.27192628558110504, 0.35362385966197596, -0.88053401103787832, 0.67433784203317992, 
0.96236104863196403], [-0.062882910701290901, -0.4358052357854208, 0.84374220519150256, 0.40389916375049806, 
-0.15811152054541372], [-0.76227980036025444, -0.87757053576512689, 0.89491095169328416, -0.65406462400488941, 
0.48459092870162573]]], [[[-0.40102914991197242, 0.37671939317605085, -0.044228823228774618, -0.41617471342720264, 
0.28847098050406983], [0.11874344310916275, -0.15441874099107733, 0.38450729403345485, -0.29446655740068106, 
-0.42023912541040576], [0.027459402512120146, 0.19030530318749433, -0.3684412278381019, -0.17637271538554869, 
0.069043367045837814], [0.33286862251624616, 0.38321321811614012, -0.39078534630468265, 0.28561374750612623, 
-0.21160880144606428]], [[-0.79235756667857338, 0.74432609640253244, -0.087387769089357109, -0.82228232864547235, 
0.56996396451419928], [0.23461453029452936, -0.3051021550132112, 0.75971351194143066, -0.58181008772413545, 
-0.83031283612777629], [0.054254573169374154, 0.37600719796246779, -0.72797001120229221, -0.34847904602960089, 
0.13641660293209129], [0.65768528751218169, 0.75715666328051423, -0.77211775293539042, 0.56431861383038673, 
-0.4180988714620299]], [[0.24121274937565101, -0.22659081164316444, 0.026602944087740697, 0.25032256849267565, 
-0.17351077431092041], [-0.071422320270183554, 0.092880452899147981, -0.23127511196919251, 0.17711701986100617, 
0.25276724860951749], [-0.016516400310588538, -0.11446565770268442, 0.22161162491474037, 0.10608542446935797, 
-0.04152850333930691], [-0.20021526025141548, -0.23049674558364464, 0.23505126203060153, -0.1717921934367069, 
0.12727937807038589]], [[-0.88291768857745967, 0.82939660605298104, -0.097375490989177402, -0.91626261109495111, 
0.63510627939199349], [0.26142909149293519, -0.33997288700537209, 0.84654267991169907, -0.64830631957961171, 
-0.92521094124607339], [0.060455436222132369, 0.41898180830799242, -0.81117115135112638, -0.38830740915098211, 
0.15200792774564434], [0.73285344682928488, 0.84369360393303239, -0.86036462627469434, 0.62881571035274686, 
-0.4658842228711213]], [[0.22882721843770076, -0.21495607212327664, 0.025236964934930071, 0.23746927643179519, 
-0.16460153104389538], [-0.067755004344071709, 0.088111328025397201, -0.21939984807088425, 0.16802260700428889, 
0.23978842976243309], [-0.015668334080425103, -0.108588199116938, 0.21023255128079912, 0.10063826502097994, 
-0.039396142739599418], [-0.18993482397052963, -0.21866144839926099, 0.22298210446981903, -0.16297119399859242, 
0.120743974452898]]]])+(1.-msk_ref)*numpy.array([[[[-0.41170284524271727, 0.05194715097012572, -0.45295486992954404, 
0.37470667517020256, 0.57193105253134091], [-0.32193779551305934, 0.44745277771896358, -0.4399476094708486, 
-0.30565511745766644, -0.34531292342636932], [0.33387076867733767, -0.66487393762582181, 0.37765396623018199, 
0.41238449816093592, -0.31511016808678821], [0.19092189384696942, -0.24800926250094754, -0.16868013982167537, 
0.62215992933809694, -0.69634217946249999]], [[-0.30494623542544375, 0.038476994542211623, -0.33550140349699065, 
0.27754335755093779, 0.42362646604864407], [-0.23845771268594454, 0.33142603135427207, -0.32586699082311432, 
-0.22639721460337905, -0.2557715528551805], [0.24729640614149795, -0.49246879552643114, 0.27972640127732717, 
0.30545113232785648, -0.23340052324801044], [0.141414890525242, -0.18369921856070162, -0.12494053471826788, 
0.46083074352415798, -0.51577716464371315]], [[0.43885441327217983, -0.05537303597383661, 0.48282698547909264, 
-0.39941836686613807, -0.60964957952333165], [0.34316940966664811, -0.47696203342266419, 0.46896190360550516, 
0.32581289827240084, 0.36808611394521601], [-0.35588935561094848, 0.70872199492506505, -0.40256003009197044, 
-0.43958102081190609, 0.33589150410242791], [-0.20351308394681855, 0.26436533203153678, 0.17980449891831313, 
-0.66319102213190551, 0.7422655493782796]], [[0.046038523810832531, -0.0058089716271771388, 0.050651516756435984, 
-0.041901440289369526, -0.063955985935913146], [0.036000579144885317, -0.050036247257637965, 0.049196984495459478, 
0.034179774479531633, 0.038614494485654308], [-0.037334979612372421, 0.074349290907966686, -0.042231020060824159, 
-0.046114749405260065, 0.035237082143424081], [-0.021349772675729222, 0.027733547311837199, 0.018862596465702059, 
-0.06957281216769666, 0.077868215826323389]], [[0.054462827103403957, -0.0068719192361482145, 0.05991992295334754, 
-0.049568724384828837, -0.075658894246209418], [0.042588101339794973, -0.059192069113534591, 0.058199234875437295, 
0.040434118947012515, 0.045680320800440231], [-0.044166675454075141, 0.087954005489387768, -0.049958611909964673, 
-0.05455300074553069, 0.041684896768001842], [-0.025256434865568552, 0.032808336740225062, 0.022314145741377817, 
-0.082303508595453864, 0.092116836604603208]]], [[[0.062927703946685593, -0.0079399862665381418, 0.06923296811158007, 
-0.057272936037292291, -0.087418166688623705], [0.049207350688458155, -0.068391987696420076, 0.067244842343783487, 
0.0467185859010181, 0.052780177901195754], [-0.051031274450857103, 0.10162424377737506, -0.05772342177332785, 
-0.063031892813796217, 0.048163765679747894], [-0.029181912517169201, 0.037907567623183758, 0.025782318541243271, 
-0.095095519239847048, 0.10643408230266813]], [[-0.27604685522891431, 0.03483057702686234, -0.30370634755015541, 
0.25124091443438096, 0.38347990615892502], [-0.21585936812828316, 0.30001719342003319, -0.29498497633240384, 
-0.20494182863630764, -0.23153239692952626], [0.22386042946073051, -0.44579813262420015, 0.25321707378803315, 
0.27650390367199934, -0.21128144232214577], [0.12801317503582832, -0.1662902692369477, -0.11310007369488957, 
0.41715837995235044, -0.46689759622755894]], [[0.1784899299880682, -0.022521202966841444, 0.19637436067223532, 
-0.16245058539190252, -0.2479553753487862], [0.13957312961430268, -0.19398897989383573, 0.19073518417531221, 
0.13251392635712836, 0.14970719837996574], [-0.14474655887099352, 0.28824989661599476, -0.16372835595149154, 
-0.17878545425527914, 0.13661307540194648], [-0.082772406991369382, 0.10752210340953336, 0.073129701907691566, 
-0.26973163657265919, 0.30189265946589677]], [[0.52471131804344673, -0.066206144477984369, 0.5772866268990966, 
-0.47756005497684278, -0.72892062775707578], [0.41030662518756822, -0.57027426327525954, 0.56070899849967948, 
0.38955450855187818, 0.4400980009785132], [-0.42551508475872857, 0.84737544123288411, -0.48131634909676252, 
-0.52558007813412777, 0.40160487967537428], [-0.24332811813514285, 0.31608542063180539, 0.21498121646792243, 
-0.79293684833378075, 0.88748141290985649]], [[0.045852842538876254, -0.0057855430471270026, 0.050447230491817811, 
-0.041732444585647607, -0.063698040462539726], [0.035855382627415486, -0.049834442480558405, 0.048998564609212369, 
0.034041921579933562, 0.038458755598771099], [-0.037184401228682898, 0.074049427450972219, -0.042060695105290723, 
-0.045928760702392511, 0.035094964940463734], [-0.021263665376487929, 0.027621693153309427, 0.018786520375197058, 
-0.069292213069646669, 0.077554159940886935]]], [[[-0.26088689412057647, 0.032917748885167616, -0.28702738044727855, 
0.23744324777199513, 0.36241992900983777], [-0.20400478777821374, 0.28354082754979465, -0.27878497012318698, 
-0.19368681850760458, -0.2188170840531628], [0.21156644624724233, -0.42131575861871423, 0.23931088026371147, 
0.26131884234429464, -0.1996782728317304], [0.12098293825487071, -0.15715792823550204, -0.10688883568914295, 
0.39424884595005028, -0.4412564803578759]], [[0.47387672847409035, -0.059792022910991276, 0.52135848558947107, 
-0.43129352983309327, -0.65830202345702438], [0.37055568372365505, -0.51502548719842156, 0.50638691542958281, 
0.35181405417980888, 0.39746083939896348], [-0.384290731584992, 0.76528080884178484, -0.43468591019068004, 
-0.47466132216483797, 0.36269697255518141], [-0.2197542317127813, 0.28546272949834528, 0.1941536079745167, 
-0.71611628461929822, 0.80150127140284688]], [[0.060816610986025338, -0.0076736163203340157, 0.066910346714972271, 
-0.055351548722619083, -0.084485469883309891], [0.047556546906722265, -0.066097579435254211, 0.064988918411886282, 
0.045151274976862957, 0.051009513232991549], [-0.049319281838541229, 0.098214962773691808, -0.05578692160363885, 
-0.060917304534379012, 0.046547971994163072], [-0.028202920336780832, 0.036635847948923002, 0.024917375634252208, 
-0.091905263300624152, 0.10286344126809512]], [[0.04625416685356535, -0.0058361806732790547, 0.050888766917567781, 
-0.042097705358945046, -0.064255553825293479], [0.03616920476065974, -0.050270615515184705, 0.049427421671800573, 
0.034339871501733586, 0.038795363601238697], [-0.037509855519277276, 0.074697540721151395, -0.04242882887200685, 
-0.046330749486354586, 0.035402131562507461], [-0.021449774360413235, 0.027863450402402672, 0.018950948292802743, 
-0.06989868910000159, 0.07823294817660377]], [[0.43327194902766991, -0.054668661165049655, 0.47668516645846482, 
-0.39433755034886664, -0.60189450887474572], [0.33880411014740752, -0.47089481974759201, 0.4629964558858502, 
0.32166838291605698, 0.3634038605421373], [-0.35136225153584916, 0.69970666985070351, -0.39743924992822288, 
-0.43398931372856936, 0.33161878345754398], [-0.20092428802711465, 0.26100246277712935, 0.17751728895560739, 
-0.65475487552755274, 0.73282353224453411]]], [[[0.011502365532395148, -0.0014513261827778707, 0.012654885784276169, 
-0.010468747532447042, -0.01597890347749387], [0.0089944634715885419, -0.012501165506337042, 0.012291482261323153, 
0.0085395496496327508, 0.009647529800209307], [-0.0093278530043785623, 0.018575589534792936, -0.010551090463748032, 
-0.011521409901707079, 0.0088037097100261766], [-0.0053340739238037435, 0.0069290101481417968, 0.0047126723769435842, 
-0.017382223634225702, 0.019454765436659464]], [[-0.28982871487579265, 0.036569521394136266, -0.31886912943488771, 
0.26378431768169075, 0.40262544664949751], [-0.22663631942715176, 0.31499579133934813, -0.3097123367958281, 
-0.21517371305932689, -0.24309183670485693], [0.23503684013477805, -0.46805496032677357, 0.26585913836886654, 
0.29030858182731245, -0.22182983702006456], [0.13440433500695725, -0.17459244369702165, -0.11874668517474857, 
0.43798534513641257, -0.49020783149662961]], [[0.49874080946839999, -0.062929281212870072, 0.54871391123889923, 
-0.45392329114807217, -0.69284280979751323], [0.38999855985448939, -0.54204862350872363, 0.53295679008927122, 
0.37027356614240464, 0.41831541593557531], [-0.40445427898319303, 0.80543471991419424, -0.45749366804970981, 
-0.49956657040772129, 0.38172750594103011], [-0.23128462915125272, 0.30044081979190917, 0.20434075316217024, 
-0.75369055706659316, 0.84355565249341813]], [[0.24003649494373189, -0.030286922194649155, 0.26408780168007595, 
-0.21846649344103833, -0.33345488569107951], [0.18770047600547551, -0.26087989834798625, 0.25650413485483275, 
0.17820713143942199, 0.20132895547314705], [-0.19465779749525614, 0.38764368867320281, -0.22018486246323776, 
-0.24043392133790925, 0.18371974141711578], [-0.11131383408348319, 0.14459767468745818, 0.098346149406723815, 
-0.36273999671951118, 0.40599072357908206]], [[-0.016444613811068137, 0.002074920895398116, -0.018092340132948626, 
0.014966878749576324, 0.022844596276449941], [-0.012859135610971429, 0.017872570503951916, -0.017572792169035189, 
-0.012208757904028374, -0.013792806475189245], [0.013335773403427559, -0.026557006500295133, 0.015084602160600121, 
0.016471841019047924, -0.012586420234897958], [0.0076259779320694302, -0.0099062141311933253, -0.0067375773303900788, 
0.024850884284403487, -0.027813939966500327]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5, 4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.993549884155)+(1.-msk_arg0)*(0.929529899002)
      arg1=numpy.array([[[-0.96404073390044864, 0.68299636489748128], [0.36400286263553872, -0.2902214898897415]], 
[[0.52744858859312793, -0.045387940631246293], [0.40639487510984895, 0.6685781294180575]], [[-0.20069163885522623, 
0.19403923621031405], [0.20120512920242328, -0.87559023882949072]], [[-0.39214221655530723, -0.65080180395854148], 
[0.5640793644347486, -0.98168633861320709]], [[-0.54996581116145982, 0.084797595420068594], [0.079502859459630315, 
0.053027222376161642]], [[0.71554985948411787, -0.18034726142179469], [0.3513773812994283, 0.62946438984268482]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.95782255948772366, -0.6785909592223428], [-0.36165500200371536, 0.28834952765931393]], 
[[-0.52404648409454735, 0.04509518315621968], [-0.40377358108667383, -0.66426572303203835]], [[0.19939715453553536, 
-0.19278766065832884], [-0.19990733281050782, 0.86994258035649974]], [[0.38961285383090466, 0.64660405693103029], 
[-0.5604409871885061, 0.97535434800593401]], [[0.54641846796881122, -0.084250641106252092], [-0.078990056806126058, 
-0.052685190648909572]], [[-0.71093447999774384, 0.17918400069333898], [-0.34911095648481871, 
-0.62540427160804857]]])+(1.-msk_ref)*numpy.array([[[-0.89610468601611992, 0.63486554208176271], [0.33835154414198393, 
-0.26976955218536774]], [[0.49027923328362022, -0.042189447870862755], [0.37775618721571325, 0.62146336111278955]], 
[[-0.18654887879560722, 0.18036527163696259], [0.18702618342617572, -0.8138873062661518]], [[-0.3645079149490027, 
-0.60493973510378218], [0.52432863465204005, -0.91250680318259625]], [[-0.51120966490336317, 0.078821900306413159], 
[0.073900284923865675, 0.049290388659660331]], [[0.66512498861703517, -0.16763817169465478], [0.32661578175077993, 
0.58510597071571013]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.56879413112408872, -0.28005683913711432])+(1.-msk_arg0)*numpy.array([0.78833884726990999, 
0.66356112587536176])
      arg1=numpy.array([[[-0.85252097033383101, 0.58175331209550607], [-0.70869360301507678, -0.2927221987776345]], 
[[0.36754488753156189, -0.66559884167391159], [-0.43601741806713212, -0.82616620732624724]], [[0.61801257801317044, 
0.21090349649113005], [0.41668879754804911, 0.49086430937885894]], [[-0.097196959600152955, 0.49753981495824773], 
[-0.29445462587602544, -0.60285511181456664]], [[0.77183435706312098, -0.44132925471347328], [-0.47567793664135172, 
0.59066187435301054]], [[-0.97049765916666608, -0.62489178685474744], [-0.052994325480985038, 0.76185763449074395]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.48490892458609641, 0.33089786968192419], [-0.40310076216016044, -0.1664986687144574]], 
[[0.20905737495261564, -0.37858871482711243], [-0.24800414846446295, -0.46991849006021652]], [[0.3515219273347594, 
0.11996067103770461], [0.23701014255048389, 0.27920073835297393]], [[-0.055285060183672151, 0.2829977267488164], 
[-0.1674840630806225, -0.3429004495182818]], [[0.43901485249743755, -0.25102548997439167], [-0.27056281866681697, 
0.33596500761074627]], [[-0.5520133728036658, -0.35543478095062531], [-0.030142861316464039, 0.43334015125041625]]], 
[[[0.23875432824979831, -0.16292399374301461], [0.1984744903770953, 0.081978853734930385]], [[-0.1029334594430954, 
0.18640550773252029], [0.12210965991260675, 0.23137349662568665]], [[-0.17307864914534779, -0.059064966590271363], 
[-0.11669654754515159, -0.13746990692986583]], [[0.027220673279356636, -0.1393394279220716], [0.082464031792141235, 
0.16883369707243914]], [[-0.21615749037652454, 0.12359727609379374], [0.13321685938304151, -0.16541889753010749]], 
[[0.271794506816185, 0.17500521862928395], [0.014841423286408105, 
-0.21336344098795673]]]])+(1.-msk_ref)*numpy.array([[[[-0.67207539902639746, 0.45861873545282344], [-0.55869069806846483, 
-0.23076428075467384]], [[0.2897499129565802, -0.52471742358939877], [-0.34372946874864535, -0.6512989155369272]], 
[[0.48720332334920813, 0.16626341930901098], [0.32849196632931393, 0.38696740382167011]], [[-0.076624139089324597, 
0.39222996419506934], [-0.23213002033639851, -0.47525410391866818]], [[0.60846700733045289, -0.34791699592730801], 
[-0.37499539624357248, 0.46564170115373676]], [[-0.76508100590559558, -0.49262647091750589], [-0.041777485461526161, 
0.60060196935821353]]], [[[-0.56569977490707279, 0.38602888275581471], [-0.47026152511735103, -0.19423907178959859]], 
[[0.24388849938017643, -0.44166551676247745], [-0.28932420883389448, -0.54821177869358217]], [[0.41008912207155424, 
0.13994736158270465], [0.27649848764063417, 0.32571847378346752]], [[-0.06449612394393954, 0.330148079781514], 
[-0.19538864306550388, -0.40003121673539094]], [[0.51215927496209057, -0.29284893713940663], [-0.31564138719180435, 
0.39194025835733515]], [[-0.64398451937603607, -0.41465389763560279], [-0.035164974281167806, 0.50553910969941784]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.564569713)+(1.-msk_arg0)*(0.765058255003)
      arg1=Data(numpy.array([[[-0.061916195813281893, -0.77770650262277941], [0.34046883403604578, 0.80949355749984009]], 
[[-0.64459103147321128, 0.51540888259634965], [-0.11655602739516113, -0.67199689966441034]], [[-0.88499197249500594, 
0.25313378268983855], [0.99236121472568417, -0.6940495946852514]], [[0.22304924598053755, 0.25241118750820246], 
[-0.11946085671392392, 0.16792236720876286]], [[-0.24099441005664635, 0.6916611269224997], [-0.24961596509577566, 
0.96472588150588923]], [[0.31899818052162288, 0.50443582458100011], [-0.98115542257236155, 
-0.52015867465733923]]]),self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.034956008900342971, -0.43906953698380813], [0.19221839191710138, 0.45701554543285866]], 
[[-0.3639165736410655, 0.29098424492496039], [-0.065804002934881048, -0.37938909678028065]], [[-0.49963966391861803, 
0.14291166704375177], [0.56025708618979631, -0.39183938047906863]], [[0.12592684878805024, 0.14250371168944045], 
[-0.067443981589688326, 0.094803882661295552]], [[-0.13605814492023302, 0.39049092391974266], [-0.14092561377428611, 
0.5446550140452433]], [[0.18009671122454585, 0.28478918871050452], [-0.55393063532985976, 
-0.29366583366564192]]])+(1.-msk_ref)*numpy.array([[[-0.04736949672535657, -0.5949907798013595], [0.26047849205065005, 
0.61930972853729993]], [[-0.49314968972971907, 0.39431782033240143], [-0.089172150929067273, -0.51411677542493028]], 
[[-0.67707041416902203, 0.19366209006709081], [0.75921413927105907, -0.53098837179569647]], [[0.17064566690968808, 
0.19310926265835446], [-0.091394514578762576, 0.12847039323277168]], [[-0.1843747628235051, 0.52916105481699383], 
[-0.19097075467715691, 0.73807149946148609]], [[0.24405219133912365, 0.38592279171512733], [-0.75064105548030768, 
-0.39795168795821101]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.25980353350036678, 0.38094246378925556])+(1.-msk_arg0)*numpy.array([-0.20352129609671588, 
0.1685951958778884])
      arg1=Data(numpy.array([[[-0.81679769816057579, -0.55786887196221646], [-0.50302600787280327, 0.8965691928576569]], 
[[0.64790141094910392, -0.79264082526829949], [0.8095029626250736, -0.64555155689440324]], [[0.6861970523964751, 
-0.33083887691935732], [-0.46426092136451613, -0.99503807327607685]], [[0.23706478599730141, 0.31436258162329533], 
[-0.55669207740676829, -0.94500743138959731]], [[-0.20834312393451926, 0.029866861841089998], [0.10057659973846467, 
-0.74860703974958365]], [[-0.62776180242682456, -0.2236049291303901], [0.55635192604811978, 
0.57047326936893739]]]),self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.21220692813708361, -0.14493630416564754], [-0.1306879342879376, 0.23293184433199107]], 
[[0.16832707592445043, -0.20593088720135103], [0.21031173006900947, -0.16771657553782901]], [[0.17827641889014056, 
-0.085953109242941972], [-0.12061662783663721, -0.25851440740452164]], [[0.061590269070607176, 0.081672509506029597], 
[-0.14463056878193811, -0.24551626985912281]], [[-0.054128279778692945, 0.0077595162408824513], [0.026130155999505186, 
-0.19449075413019137]], [[-0.16309473446704814, -0.058093350696174439], [0.14454219625703627, 0.14821097114955648]]], 
[[[-0.31115292755468243, -0.21251594255661949], [-0.19162396678913915, 0.34154127728474004]], [[0.24681315977948659, 
-0.30195054887765482], [0.30837405302709719, -0.24591800058634375]], [[0.26140159578483813, -0.12603057689093025], 
[-0.1768566992256686, -0.37905225519790253]], [[0.090308043655484616, 0.11975405636672908], [-0.21206765153929327, 
-0.35999345921270909]], [[-0.079366742945165991, 0.011377555935398125], [0.038313897703916529, -0.28517621013218758]], 
[[-0.23914112768925841, -0.085180612618352691], [0.21193807344266846, 
0.21731749275931467]]]])+(1.-msk_ref)*numpy.array([[[[0.1662357261784545, 0.11353819587376314], [0.10237650509262973, 
-0.18247092417077676]], [[-0.13186173489925257, 0.16131928809777482], [-0.16475109214758632, 0.13138348955640178]], 
[[-0.13965571348147668, 0.067332757029809454], [0.094486984443161809, 0.20251143833872612]], [[-0.048247732505061362, 
-0.063979480056282706], [0.11329869312059876, 0.19232913725743914]], [[0.042402262615992066, -0.0060785424322401822], 
[-0.020469479935772944, 0.15235747499696095]], [[0.1277628956699178, 0.045508364990231288], [-0.11322946507521756, 
-0.11610345917049707]]], [[[-0.13770816791399065, -0.094054011742646534], [-0.084807768328987496, 0.15115725868791696]], 
[[0.10923306528852444, -0.13363543519692006], [0.13647831054750525, -0.10883689118388773]], [[0.11568952645961338, 
-0.055777845258239658], [-0.078272160975899549, -0.16775863886993686]], [[0.039967984030964729, 0.053000021025458158], 
[-0.093855609834062714, -0.15932371300118933]], [[-0.035125649789551454, 0.0050354094223563992], [0.016956731533638431, 
-0.12621155050214725]], [[-0.10583762404480676, -0.03769871682599947], [0.093798261949123232, 0.096179052592355388]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.989567049022)+(1.-msk_arg0)*(0.111704893141)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-0.98564278032184371, -0.98926209581241942], [0.71884820439034591, -0.35985766245982154]], 
[[-0.31829775311751174, -0.31295355476148701], [-0.23289595230410742, 0.96397526515271847]], [[-0.98050373305327909, 
0.16502651443126082], [-0.15672073533714692, 0.11515718288678323]], [[0.68514762048745004, -0.080742832957046984], 
[0.37115655840142603, -0.018109035859751987]], [[0.120811315682819, -0.54008732129415926], [0.96170319243592362, 
-0.90135016688370628]], [[-0.039383107820948116, -0.0037324704980368217], [0.039849656968649283, 
0.0067319961117462412]]])+(1.-msk_arg1)*numpy.array([[[-0.93341969370184841, -0.6562498098736802], [0.0040611019140541771, 
0.75590407496308654]], [[0.16399251424059869, 0.35107616902065164], [-0.85195584055986484, 0.78978808695479863]], 
[[0.82449515840869503, -0.64524111675619156], [-0.58553323716380201, 0.865742756902522]], [[-0.60338001261042762, 
-0.92648566542986877], [0.0044011356577715599, 0.71778707833593858]], [[-0.44437597093742087, 0.64750491748878591], 
[-0.54543489906039855, -0.19085528281537423]], [[0.31818415050754689, -0.38866553388085512], [0.99116002770573752, 
0.58200948995953605]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.97535961751305766, -0.97894117286254678], [0.7113484963134139, -0.35610328510836853]], 
[[-0.31497696826287164, -0.30968852566631128], [-0.2304661602507751, 0.95391815846750416]], [[-0.97027418567271895, 
0.16330480089615126], [-0.15508567558815925, 0.11395575364297619]], [[0.67799950895030248, -0.079900446938992037], 
[0.36728430022251024, -0.017920105176370764]], [[0.11955089714872857, -0.53445261674733002], [0.95166979017398179, 
-0.89194642477871666]], [[-0.03897222578769613, -0.0036935298163044698], [0.039433907451010559, 
0.0066617615263292037]]])+(1.-msk_ref)*numpy.array([[[-0.10426754714068533, -0.073306314885751844], [0.00045364495534419943, 
0.084438183918610507]], [[0.018318766279172703, 0.039216925944809342], [-0.095167636130604596, 0.088223193857333626]], 
[[0.092100143565328751, -0.072076589997440529], [-0.06540692768789598, 0.096707702147405247]], [[-0.06740049983207301, 
-0.10349298225352706], [0.00049162838835048949, 0.080180328883518462]], [[-0.049638970348000057, 0.072329467616367535], 
[-0.060927747114922944, -0.021319468972289861]], [[0.035542726531610641, -0.043415841929757053], [0.11071742498051637, 
0.065013307882987495]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank1_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.46339937678895926, -0.07398819985425642])+(1.-msk_arg0)*numpy.array([0.19125387973739794, 
0.363847965390933])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.53932222036196209, -0.79546889428276879], [0.40819649651641421, -0.5403822480074254]], 
[[0.0075571699341703269, -0.69508479063866702], [0.87061594540981035, 0.38284731713654407]], [[0.40134777280270062, 
-0.10155047091049663], [0.12057619184612833, 0.50213788884001476]], [[0.071581753019605454, -0.64541201057082165], 
[0.45596858567208742, 0.050327024187448721]], [[0.27646171390120755, 0.27886795358673111], [0.27909761715969683, 
-0.99945859082854316]], [[0.14330865380306212, 0.65405782838481552], [0.24232918771382317, 
0.66576254667772328]]])+(1.-msk_arg1)*numpy.array([[[0.57991223054867791, -0.57792205794002527], [-0.70572588770907418, 
0.84975079195812264]], [[-0.88071566625166353, 0.72019098358391775], [-0.96333620472007286, 0.51504226327966207]], 
[[-0.13724807986659671, 0.81420229896465002], [0.45193448201464936, 0.97654411668289964]], [[0.28266570566132843, 
-0.90883007344716837], [-0.71593930216139778, -0.11859113719293135]], [[-0.41717200473919158, 0.3619607436530321], 
[-0.050599521071579057, -0.49442202374371336]], [[-0.88738130429442186, 0.27396039024978158], [-0.4578514852002773, 
0.2201967143529151]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.24992158080417098, -0.36861978986563759], [0.18915800209314293, -0.25041279695445773]], 
[[0.0035019878377827896, -0.32210185879744252], [0.40344288652543669, 0.17741120816639958]], [[0.18598430779240829, 
-0.04705842493254947], [0.055874932157081864, 0.23269038475058654]], [[0.033170939738746373, -0.29908352347062794], 
[0.21129555843578848, 0.023321511644106615]], [[0.12811218592782714, 0.1292272358985036], [0.12933366185508705, 
-0.46314848811631831]], [[0.066409140860803711, 0.30308999005746357], [0.11229519456436038, 0.30851394921988734]]], 
[[[-0.039903480225982176, 0.058855311528037871], [-0.030201723964063738, 0.039981909763265752]], [[-0.00055914139942197193, 
0.051428072405427676], [-0.064415306565283451, -0.028326183813964512]], [[-0.029694999225186914, 0.0075135365370196773], 
[-0.0089212153799765066, -0.037152278473889404]], [[-0.0052962050483325914, 0.047752872826451409], [-0.033736294843969047, 
-0.0037236059236509527]], [[-0.020454904540172805, -0.020632937882922565], [-0.020649930277258394, 0.073948141964275738]], 
[[-0.010603149318425405, -0.048392561322776681], [-0.017929500371089969, 
-0.049258572359070107]]]])+(1.-msk_ref)*numpy.array([[[[0.11091046399960303, -0.11052983576685112], [-0.13497281405547967, 
0.16251813577191745]], [[-0.16844028811613795, 0.13773931976231693], [-0.18424178664421417, 0.098503831080965734]], 
[[-0.026249227760994875, 0.1557193485680981], [0.086434223072412983, 0.18676785105033478]], [[0.054060912876438431, 
-0.17381727756879528], [-0.13692616919485256, -0.022681015090618151]], [[-0.079785764424198546, 0.06922639653627613], 
[-0.0096773547177937142, -0.094560130268601064]], [[-0.16971511725274069, 0.052395987529642335], [-0.08756587288808286, 
0.04211347592542259]]], [[[0.21099988519045412, -0.2102757649360191], [-0.25677692836665672, 0.30918009674329694]], 
[[-0.32044660325358776, 0.26204002406990329], [-0.35050791807482184, 0.1873970795846463]], [[-0.04993743461327349, 
0.29624584989490804], [0.16443544177103536, 0.35531358996955892]], [[0.10284734189066667, -0.33067597310984442], 
[-0.26049305843482901, -0.043149143981045072]], [[-0.15178718514241152, 0.1316986801295448], [-0.01841053279164968, 
-0.17989444738361768]], [[-0.32287188209347778, 0.099679930590089033], [-0.16658833134133777, 0.08011812650307662]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2, 6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.43804651728)+(1.-msk_arg0)*(-0.543337177571)
      arg1=numpy.array([[[[0.60795094604023503, -0.097794445632283677, 0.75426723610529667, -0.034864365796523078], 
[0.85978429070534101, 0.12902059279871558, -0.31201948276004643, 0.076558135122647863], [-0.97390792210779908, 
0.49232230873386418, -0.97840440866285516, 0.36414936327706937]], [[0.79385832462423922, 0.42654756978458575, 
-0.68994913906275612, 0.98855386129816658], [-0.20652839986652571, 0.33057772983488531, -0.5920830737384537, 
0.20621684716524569], [-0.031916346185909106, 0.48720143271352656, -0.74521372086186055, 0.26511369448021682]]], 
[[[0.92814304130352876, -0.60386390960393133, 0.8231201717346297, -0.14016735785423751], [-0.40984787457581096, 
0.67837081242748476, -0.2357430543063852, -0.81200518114201992], [0.41040668039544603, 0.81899606955074167, 
-0.22180340249305619, -0.25792851582212095]], [[-0.77493277192916898, 0.73469674814970154, 0.013542845859844999, 
0.5913689784999816], [-0.85270304587659984, 0.049194424212611887, -0.72045050859561766, -0.7655798577204973], 
[0.16786575391896652, 0.92315757404538923, 0.11199468205897722, -0.88181526809288391]]], [[[0.49210255594399932, 
-0.052030715984012987, 0.65375080354347515, -0.78565820813980913], [0.29470788038440965, 0.55718988716903617, 
0.10815871903784302, 0.75227607016471398], [-0.44501413908595944, 0.67574867634473756, -0.026177234674506833, 
-0.4513298929006202]], [[-0.92238036282832137, 0.34283069911981556, -0.94496013493330766, -0.86792383195402123], 
[-0.47685078769718392, -0.13354947626753133, 0.20909536400421036, 0.71714871813267655], [-0.62375289750212426, 
0.83122191743468754, 0.37425198819303973, 0.39703242187001364]]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.26631079459014878, 0.042838516318573112, -0.33040413587451362, 0.015272214014351066], 
[-0.37662551415573142, -0.056517021332908673, 0.13667904774661854, -0.033536024459945503], [0.42661697343095134, 
-0.21566007272023363, 0.4285866437063911, -0.15951436035333519]], [[-0.34774687431556989, -0.1868476773984856, 
0.30222981746693656, -0.43303257608558993], [0.090469046280991255, -0.14480842324457782, 0.25935992839170596, 
-0.090332571705246292], [0.013980844291047782, -0.21341689081410087, 0.3264382750529829, -0.11613213055035512]]], 
[[[-0.4065698267808957, 0.26452048251322852, -0.3605649245314631, 0.061399822944421076], [0.17953243407264038, 
-0.29715797180842296, 0.10326642391191725, 0.35569604161276785], [-0.17977721701576746, -0.35875837593290316, 
0.097160207982989369, 0.11298468806313997]], [[0.33945660186989079, -0.32183135178409039, -0.0059323964629681451, 
-0.25904712145948688], [0.37352359952049263, -0.021549446195941083, 0.31559083616308403, 0.33535959037436136], 
[-0.073533008874824168, -0.40438596021145301, -0.049058880429842147, 0.38627610707262416]]], [[[-0.2155638107759707, 
0.022791873928393921, -0.28637326266137414, 0.34415484184827305], [-0.12909576061743061, -0.24407508953816315, 
-0.047378550188018537, -0.32953191256891434], [0.19493689376706644, -0.29600935422954072, 0.011466846481195113, 
0.19770348772957794]], [[0.40404550554462532, -0.1501757937661832, 0.41393649607619598, 0.38019101185197457], 
[0.20888282681308795, 0.05850088296359144, -0.091593495981487277, -0.31414449835000358], [0.27323278439426069, 
-0.3641138660192636, -0.16393978001316453, -0.1739186696474963]]]])+(1.-msk_ref)*numpy.array([[[[-0.33032235112318914, 
0.053135358071976641, -0.40982143119981596, 0.018943106109689686], [-0.46715276983182102, -0.07010168473980595, 
0.16953178511004208, -0.041596881057647361], [0.52916038161219858, -0.26749701368275169, 0.53160348992600925, 
-0.19785588725728065]], [[-0.43133274149266626, -0.23175915266657404, 0.37487501788597705, -0.53711806487477098], 
[0.11221455787175624, -0.17961517069635238, 0.32170074617268041, -0.11204527970637809], [0.017341337455034401, 
-0.26471465135916988, 0.40490231978035029, -0.14404612649433127]]], [[[-0.50429462044412798, 0.32810171228125756, 
-0.44723179091214327, 0.076158136604121532], [0.22268558740554051, -0.36858408257097225, 0.12808796575882486, 
0.44119260329482518], [-0.22298920738239134, -0.44499101287153464, 0.12051403468624665, 0.14014215180189726]], 
[[0.42104978510735341, -0.39918805751033343, -0.00735833164576881, -0.32131275168129203], [0.46330526625288276, 
-0.026729159603916548, 0.39144754592001552, 0.415967999099149], [-0.091207704945178214, -0.50158583073521723, 
-0.060850874452898815, 0.4791230189047016]]], [[[-0.26737761382214315, 0.028270222369757795, -0.35520711643215874, 
0.42687731334626167], [-0.16012574793603021, -0.30274198066559083, -0.058766653131728595, -0.408739556717584], 
[0.24179272631020376, -0.36715937855256503, 0.014223064804663206, 0.24522431016209553]], [[0.50116354298615895, 
-0.18627266444449192, 0.51343197263188112, 0.4715752852006026], [0.25909076110994977, 0.072562395501300767, 
-0.11360928492125208, -0.38965356040894988], [0.33890813883060772, -0.45163377055421156, -0.20334501896518359, 
-0.21572247550307652]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.647480014887)+(1.-msk_arg0)*(-0.378983139168)
      arg1=Data(numpy.array([[[[-0.15966497293401405, -0.82379668881748769, 0.42255757211308609, 0.27970108322437981], 
[-0.11267007778659455, 0.0049557419911394618, 0.85704943783706944, 0.58893235535539623], [0.99687634124774305, 
0.58590098057034923, 0.11077319930465057, 0.31718343280306582]], [[-0.9128103218342134, -0.46964610822504937, 
-0.57260883670011609, 0.69637833743595912], [0.16045131981896832, -0.71296302378250265, 0.054788118352184778, 
-0.59976311458090792], [-0.50177368977017611, -0.022496865498684349, 0.43022221754422185, -0.90687847956880363]]], 
[[[-0.8375205296921322, 0.47187921735539762, 0.036701311258400704, 0.15982997815519395], [0.40454750391093253, 
0.057429996683073803, 0.049686735148374073, -0.048948156089927863], [-0.51397104354091527, -0.91196540959389893, 
0.36981298229808446, 0.09035240445060122]], [[0.69669588663419968, 0.99953429884768585, 0.98393221607933246, 
-0.51244956485403348], [-0.37301000438945131, 0.99808559202134495, -0.080812184802454778, -0.068534716172923504], 
[0.094817978807470338, 0.57847078543819253, -0.37507654657691658, 0.90495886400539605]]], [[[0.24152699525662746, 
0.9003824620926717, 0.96762490705552207, -0.47904707822731774], [-0.93822028424107917, -0.25496865217469478, 
0.78629828304891158, 0.7349206929295633], [0.99985514559762789, 0.0022368160793040381, -0.49421642858409776, 
0.25409309561466964]], [[-0.51028035342680056, -0.30059759007213782, -0.36242607685938144, 0.097490979178828097], 
[-0.17210992271799341, 0.15143026193152109, -0.55920023737619973, 0.867565538988045], [0.16060374897895957, 
0.67533415472816549, 0.48603174128372562, -0.54273022379048674]]]]),self.functionspace)
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.10337987905231617, 0.5333918923397607, -0.27359758308257631, -0.18110086153015112], 
[0.072951623642631885, -0.0032087438982012296, -0.55492238277000738, -0.38132193021319988], [-0.6454575082720132, 
-0.37935917562224825, -0.071723432734903156, -0.20536993379337451]], [[0.59102644077061428, 0.30408646914537751, 
0.37075277811126389, -0.450891056290317], [-0.10388902294509304, 0.46162930925288037, -0.035474211686326758, 
0.38833463035777632], [0.32488843612251322, 0.014566270808008602, -0.27856028782043496, 0.58718569145229682]]], 
[[[0.54227780503358813, -0.30553236267834061, -0.02376336555997741, -0.10348671663538224], [-0.26193642385492233, 
-0.037184775107341556, -0.032171168013576924, 0.031692952833818636], [0.33278597892357864, 0.59047937698067687, 
-0.2394465152839278, -0.058501376178790178]], [[-0.4510966630499213, -0.64717848269839429, -0.63707644591526602, 
0.33180085188074548], [0.24151652319524144, -0.64624047398090756, 0.052324274618978983, 0.044374859047950146], 
[-0.061392746329856709, -0.37454827276746294, 0.24285456796154695, -0.58594277873872358]]], [[[-0.15638390248447684, 
-0.58297964996014195, -0.62651778922575541, 0.31017340934240245], [0.60747888360810065, 0.16508710670589877, 
-0.50911242401446821, -0.47584646119911239], [-0.64738622455682338, -0.0014482937083282164, 0.31999526053724303, 
-0.1645202013313789]], [[0.33039633083354675, 0.19463093209503271, 0.23466364164050435, -0.063123460650097524], 
[0.11143773532372042, -0.098048068249828368, 0.36207097802139493, -0.56173134809979874], [-0.10398771777987346, 
-0.43726536855738107, -0.3146958390821491, 0.35140697337972138]]]])+(1.-msk_ref)*numpy.array([[[[0.060510332657683796, 
0.31220505516413888, -0.16014219515856606, -0.10600199454902588], [0.042700059769850397, -0.0018781426567080065, 
-0.32480728637354089, -0.22319543279010878], [-0.37779932516823894, -0.22204659285807737, -0.041981174808143305, 
-0.12020717305574337]], [[0.34593972123355332, 0.17798795639309697, 0.21700909444786562, -0.26391564836997394], 
[-0.060808344868618629, 0.27020096486370138, -0.02076377308221114, 0.22730010792096444], [0.19016376810094046, 
0.0085259327081284829, -0.16304696654466649, 0.34369165303075977]]], [[[0.31740615946025186, -0.17883426710142078, 
-0.013909178152285363, -0.060572866854385642], [-0.1533166829746867, -0.021765000425351008, -0.018830434861532772, 
0.018550525851439188], [0.19478635952251611, 0.34561951374039762, -0.14015288493635569, -0.034242037870052831]], 
[[-0.26403599416196338, -0.37880664628324029, -0.37289371997813298, 0.19420974475358466], [0.14136450240453091, 
-0.37825761082245973, 0.030626455479447413, 0.025973501877192745], [-0.035934415258006579, -0.21923067418226289, 
0.14214768704995931, -0.34296415109854406]]], [[[-0.091534658856137124, -0.34122977193556603, -0.36671352481290898, 
0.18155076551577951], [0.3555696685526446, 0.096628820190563638, -0.29799379163217377, -0.27852255124586361], 
[-0.37892824179172513, -0.00084771557947578569, 0.18729969353312909, -0.096296999016926302]], [[0.19338765019737314, 
0.11392141831183186, 0.1373533723244599, -0.036947437329740596], [0.065226758793602652, -0.05738951603181891, 
0.21192746138424376, -0.32879271139954436], [-0.060866112950172836, -0.25594025794615238, -0.18419783504692674, 
0.20568560393339297]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_outer_expandedData_rank0_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.690473237624)+(1.-msk_arg0)*(-0.0562045307337)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-0.56201639410078919, -0.67905807272448215, -0.25380207692191359, -0.32222609823072079], 
[0.76016816954232169, 0.24391957505617157, 0.26418043996577079, -0.32148058834963256], [-0.69353153778537813, 
-0.85059153048984104, -0.41192354690135802, -0.09464631974832094]], [[-0.64890091661083549, -0.087129640274469011, 
0.92245871263029744, 0.17058762013903395], [-0.29767982521749548, -0.47635452775846954, 0.36511868715211371, 
0.65030298443566781], [-0.52341553066884439, -0.136229235578611, 0.13799323750383441, -0.74164324953122218]]], 
[[[-0.81201740098239839, -0.11022566256500266, 0.82203378705157415, 0.13325227536636453], [0.18849371161493056, 
-0.50191772584346173, -0.64447769012620082, 0.058216229392509922], [0.67428521460060353, 0.6029321949871751, 
0.50664817337072376, 0.1680297553997907]], [[-0.20918113073753042, -0.8713521135486364, 0.91086493006721159, 
-0.47669574703485496], [-0.17870237370162179, 0.19863142254470101, -0.44726017245596061, 0.21973053356979233], 
[-0.13665198792278255, -0.58338985283344003, -0.83782387884143894, 0.37637197375960674]]], [[[0.48442932734435407, 
-0.66084657381684431, 0.013136421713676016, -0.30866551598460745], [0.97879855698655449, -0.77731723293622768, 
0.67126837966669139, -0.26330714312312908], [0.92670271865249032, -0.58770621430601544, 0.40426437257369874, 
0.14919210195014121]], [[-0.63536521155437176, -0.82420572388288837, -0.53044375853468684, -0.53196527767767821], 
[-0.2730105313227722, -0.13733715172520444, -0.33175551206416021, -0.019053105080914845], [-0.66132172779628196, 
0.58168822478577753, 0.36299207595924909, -0.016604553536709643]]]])+(1.-msk_arg1)*numpy.array([[[[-0.93858479936100658, 
0.85293162859941574, -0.23719969973251875, -0.59966352839920201], [0.76964952963950251, 0.099205810203436107, 
0.10815561430526488, -0.2269339495561149], [0.7272309695157666, 0.98459525877631671, -0.38356325840308836, 
-0.79663843013422331]], [[-0.61086241061772206, 0.50866661913662514, 0.8750602262829843, -0.41188300050386295], 
[0.60240578511221887, -0.66351955655808426, -0.42029065852688063, 0.12740508973692011], [-0.65658994729083653, 
-0.29334456722385105, -0.87068164399583092, -0.25083390343440404]]], [[[0.3684013559470114, -0.098457429179994449, 
-0.85367203809283088, -0.46399893153072269], [-0.80864851862858034, 0.43887697186794949, 0.57510730528110399, 
0.85667058370089966], [0.025836965582203275, -0.23640860192184587, 0.21613477422384664, 0.67145006285814479]], 
[[-0.87221722248074096, 0.35929053249580289, 0.79083599288546269, 0.1352443609608609], [0.13264053413890675, 
-0.062105862078114304, 0.5600587428281163, -0.11056941015246968], [-0.15919859652552848, 0.73179978310420735, 
-0.80962177892350429, -0.72481111775677354]]], [[[-0.98213931444948832, -0.99397587509615715, -0.32204725603144935, 
0.40306261750230998], [-0.089799448848834951, 0.65825162639669088, -0.52160276190906951, 0.84004216473803495], 
[-0.5391586472258616, -0.25865381238936069, -0.67372728245366553, -0.51690007321233233]], [[0.68035296315998606, 
-0.39722085397897633, 0.94591470622274665, -0.26720530940111842], [0.23930428363536183, -0.65515207918738083, 
-0.81974978522324471, -0.0056555485824274498], [0.922210890681332, 0.74667199058644851, -0.90112295821245314, 
0.27807862829333208]]]])
      res=outer(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.38805727923252009, 0.46887142600876536, 0.17524354176796114, 0.22248849729230466], 
[-0.52487577716257261, -0.16841993870889735, -0.18240952370009017, 0.22197374267102901], [0.47886496628899961, 
0.58731068795284702, 0.28442218508252926, 0.065350750825816498]], [[0.4480487167894443, 0.06016068481332433, 
-0.63693305388427934, -0.11778618637596644], [0.20553995269326114, 0.32891005303822701, -0.2521046820349328, 
-0.44901680709982467], [0.36140441608358453, 0.094062641349001844, -0.095280637469485757, 0.51208481566578368]]], 
[[[0.5606762838633168, 0.076107870100504432, -0.56759233038179224, -0.092007129992974279], [-0.13014986333051973, 
0.34656075718399437, 0.44499459727785451, -0.040196748390905958], [-0.46557589520725107, -0.41630854474052059, 
-0.34982700460355326, -0.11602004922805698]], [[0.14443397259018523, 0.60164531495241469, -0.62892785730163714, 
0.32914565581673255], [0.1233892065408471, -0.1371496814182942, 0.30882117933592157, -0.15171805291877657], 
[0.094354540528795092, 0.40281508048287579, 0.57849496618231977, -0.25987477527271896]]], [[[-0.33448548605145734, 
0.45629687339602332, -0.0090703476314356775, 0.21312527816476468], [-0.67583420862417465, 0.53671674648638157, 
-0.46349285142305563, 0.18180653560174456], [-0.63986342646291849, 0.4057954125636003, -0.27913373018698395, 
-0.10301315366143916]], [[0.43870267469558466, 0.5690919946376245, 0.36625721933287175, 0.36730778758163984], 
[0.18850646546787436, 0.094827627797756095, 0.22906830251453819, 0.013155659152008956], [0.45662495450257551, 
-0.4016401518555745, -0.2506363139194282, 0.011464999839792421]]]])+(1.-msk_ref)*numpy.array([[[[0.052752718201846856, 
-0.047938621933340218, 0.013331697813635078, 0.033703807211777563], [-0.043257790642782727, -0.0055758160085382568, 
-0.0060788355482398722, 0.012754716142341149], [-0.040873675376629927, -0.055338714482125205, 0.021557992945225302, 
0.044774689130106383]], [[0.034333235131611191, -0.028589368628459531, -0.049182349381939501, 0.023149690760498068], 
[-0.033857934463483952, 0.037292805308963943, 0.023622239234251004, -0.00716074328174548], [0.036903329871930554, 
0.016487293744089832, 0.048936253219211259, 0.014098001834626889]]], [[[-0.02070582533265173, 0.0055337536043057302, 
0.047980236301468374, 0.026078842207611346], [0.045449710518001646, -0.024666874253654842, -0.032323636214833418, 
-0.048148768150253408], [-0.001452154526129874, 0.01328723453242177, -0.012147753560480321, -0.037738535694039309]], 
[[0.049022559687360356, -0.020193755775979184, -0.044448565867428194, -0.0076013458421810825], [-0.0074549989775413907, 
0.0034906308339108208, -0.031477838823946828, 0.0062145018111189141], [0.0089476824111771568, -0.041130463400377883, 
0.045504412156159565, 0.040737668744070664]]], [[[0.05520067928372778, 0.055865947620374501, 0.01810051489931563, 
-0.022653945273004492], [0.0050471358826915131, -0.036996723766305049, 0.029316438462488603, -0.047214175665602633], 
[0.030303158758333136, 0.0145375161478203, 0.037866525752783084, 0.029052126051101926]], [[-0.038238919027672999, 
0.022325611695518416, -0.053164692177332502, 0.015018149024436576], [-0.013449984964284022, 0.036822515169918905, 
0.046073651997504191, 0.00031786745411684157], [-0.05183243034822968, -0.041966348842891112, 0.050647192999672894, 
-0.015629278810291056]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      if not arg0.isLazy():
        arg0=arg0*1j
        arg1=arg1*1j
        ref=-ref
        res=outer(arg0,arg1)
        self.assertTrue(isinstance(res,Data),"wrong type of result.")
        self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
        self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")        
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_float_rank0(self):
      arg0=0.190833616561
      arg1=0.400555937829
      res=maximum(arg0,arg1)
      ref=0.400555937829
      if isinstance(res,numpy.ndarray):
         self.assertEqual(res.shape,(),"wrong shape of result.")
      else: 
         self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_array_rank0(self):
      arg0=-0.607520185637
      arg1=numpy.array(-0.822580587394)
      res=maximum(arg0,arg1)
      ref=numpy.array(-0.607520185637)
      if isinstance(res,numpy.ndarray):
         self.assertEqual(res.shape,(),"wrong shape of result.")
      else: 
         self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      res=maximum(5.0,-1e20)
      self.assertTrue(Lsup(res-5.0)<=self.RES_TOL*5,"Wrong large result")
      self.assertTrue(Lsup(maximum(-numpy.inf,-5.)+5)<self.RES_TOL*5,"Wrong large result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_constData_rank0(self):
      arg0=0.66510948385
      arg1=Data(0.447127953255,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(0.66510948385,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_expandedData_rank0(self):
      arg0=-0.626379875424
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.808319082591)+(1.-msk_arg1)*(-0.732167960529)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.626379875424)+(1.-msk_ref)*(-0.626379875424)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_array_rank1(self):
      arg0=-0.34068193517
      arg1=numpy.array([-0.28498591942826468, -0.9244388369769494])
      res=maximum(arg0,arg1)
      ref=numpy.array([-0.28498591942826468, -0.34068193517048506])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
      arg1=numpy.array([-1e16, -numpy.inf])
      arg0=5.
      res=maximum(arg0,arg1)
      ref=numpy.array([5.,5.])
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*5,"wrong large result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_constData_rank1(self):
      arg0=0.133219946847
      arg1=Data(numpy.array([0.73105505422959327, -0.57302577924904785]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.73105505422959327, 0.13321994684664973]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_expandedData_rank1(self):
      arg0=-0.989795574687
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-0.12744166984605543, 0.1251383582008796])+(1.-msk_arg1)*numpy.array([-0.75971755196106439, 
-0.044294049795981616])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.12744166984605543, 0.1251383582008796])+(1.-msk_ref)*numpy.array([-0.75971755196106439, 
-0.044294049795981616])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_array_rank2(self):
      arg0=-0.918302739699
      arg1=numpy.array([[0.19183086796477156, 0.41569819970182253, -0.57007179868513158, 0.18371653862222859, 
-0.95803703880242908], [-0.99708943520039761, -0.33564912838654082, 0.22798896799780199, -0.32560020734492867, 
-0.83553942115001578], [0.90329116559065703, -0.4744523361429227, -0.48030437204982679, -0.25285363526704185, 
0.4839054158389946], [-0.75326661018214147, -0.89189805199917815, -0.29494941406721664, 0.33826278929980447, 
0.52219823679588995]])
      res=maximum(arg0,arg1)
      ref=numpy.array([[0.19183086796477156, 0.41569819970182253, -0.57007179868513158, 0.18371653862222859, 
-0.91830273969888498], [-0.91830273969888498, -0.33564912838654082, 0.22798896799780199, -0.32560020734492867, 
-0.83553942115001578], [0.90329116559065703, -0.4744523361429227, -0.48030437204982679, -0.25285363526704185, 
0.4839054158389946], [-0.75326661018214147, -0.89189805199917815, -0.29494941406721664, 0.33826278929980447, 
0.52219823679588995]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_constData_rank2(self):
      arg0=-0.711360717248
      arg1=Data(numpy.array([[-0.49649726218321621, -0.1862706537495149, 0.6559357060844726, -0.60255638663804834, 
-0.72898391055117684], [0.877046256493895, -0.62092354207162237, -0.42516414792107171, -0.06686623727516694, 
0.93735063523149598], [0.64862166841681268, 0.45648573520660918, -0.63173946887266208, 0.070770417261089902, 
0.27864121106526873], [-0.25951227165524937, 0.85373113390321809, 0.5672699499488485, 0.22967921637760402, 
0.88510133022334969]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[-0.49649726218321621, -0.1862706537495149, 0.6559357060844726, -0.60255638663804834, 
-0.71136071724766237], [0.877046256493895, -0.62092354207162237, -0.42516414792107171, -0.06686623727516694, 
0.93735063523149598], [0.64862166841681268, 0.45648573520660918, -0.63173946887266208, 0.070770417261089902, 
0.27864121106526873], [-0.25951227165524937, 0.85373113390321809, 0.5672699499488485, 0.22967921637760402, 
0.88510133022334969]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_expandedData_rank2(self):
      arg0=0.959635515709
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-0.0064259026939066377, -0.7701474037173317, -0.19596310110827386, -0.73984521228212596, 
-0.61313716616198133], [0.43189572076819838, 0.088536215133996565, 0.067634491583320466, -0.70571296121285232, 
-0.21462215163400677], [-0.29878468443140771, 0.49093472354971013, 0.86357875479621882, 0.021213198562558766, 
-0.68952176426127321], [-0.64172570311955179, -0.10403523776207346, -0.1560226988237976, 0.18544285896364587, 
-0.99033629867964823]])+(1.-msk_arg1)*numpy.array([[0.59197813276731015, 0.69989892013318467, 0.37778322838437939, 
-0.96988982584154027, -0.12556098215489531], [-0.57102119315921684, -0.97261687381790707, 0.59394598916500385, 
-0.95926160849480113, -0.74824572871782435], [0.3774569754088497, -0.71634815703797106, -0.64965181170462061, 
-0.88835061435822782, 0.049571630549325674], [0.056229799936159397, 0.69426319270729175, 0.32265901320206658, 
-0.26876155184504102, 0.5465577920078144]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 
0.95963551570949646], [0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 
0.95963551570949646], [0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 
0.95963551570949646], [0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 
0.95963551570949646]])+(1.-msk_ref)*numpy.array([[0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 
0.95963551570949646, 0.95963551570949646], [0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 
0.95963551570949646], [0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 
0.95963551570949646], [0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 0.95963551570949646, 
0.95963551570949646]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_array_rank3(self):
      arg0=0.417103350623
      arg1=numpy.array([[[-0.45176533427754739, -0.55230637452486797], [-0.2772690226072374, -0.43884388551060138]], 
[[0.43650997405556069, -0.91050440106967101], [-0.0097638249506593056, -0.99719694387315783]], [[-0.88346341044578236, 
-0.42616345838712633], [0.47323800588958997, 0.31718901666579002]], [[0.72726350287088914, 0.18527925932004718], 
[0.95182455621773032, 0.5402040160782311]], [[-0.60062553156664, 0.18455913185890904], [-0.041559661168358408, 
0.65568567006769984]], [[0.46551889127092827, -0.88399518601736693], [-0.45129696041660194, -0.97227033198060986]]])
      res=maximum(arg0,arg1)
      ref=numpy.array([[[0.41710335062270154, 0.41710335062270154], [0.41710335062270154, 0.41710335062270154]], 
[[0.43650997405556069, 0.41710335062270154], [0.41710335062270154, 0.41710335062270154]], [[0.41710335062270154, 
0.41710335062270154], [0.47323800588958997, 0.41710335062270154]], [[0.72726350287088914, 0.41710335062270154], 
[0.95182455621773032, 0.5402040160782311]], [[0.41710335062270154, 0.41710335062270154], [0.41710335062270154, 
0.65568567006769984]], [[0.46551889127092827, 0.41710335062270154], [0.41710335062270154, 0.41710335062270154]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_constData_rank3(self):
      arg0=-0.143403966016
      arg1=Data(numpy.array([[[0.95668360633256144, -0.1800322431671213], [-0.42783838761912918, -0.89612594522855216]], 
[[0.60799134259365384, 0.37935166549606292], [-0.16348477876382117, 0.055255205348395009]], [[-0.84148244802025407, 
-0.08342603479235744], [-0.073128933987135269, 0.91858564423034572]], [[-0.44722942409971833, 0.12566700748073134], 
[-0.84487161060035754, 0.69960515111291777]], [[0.28894841908227131, -0.28216896408608538], [-0.4241999876693332, 
0.41787923950246331]], [[-0.74860509777465034, 0.95471157706396692], [0.3975080494679184, 
-0.47708712009131093]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.95668360633256144, -0.14340396601550798], [-0.14340396601550798, -0.14340396601550798]], 
[[0.60799134259365384, 0.37935166549606292], [-0.14340396601550798, 0.055255205348395009]], [[-0.14340396601550798, 
-0.08342603479235744], [-0.073128933987135269, 0.91858564423034572]], [[-0.14340396601550798, 0.12566700748073134], 
[-0.14340396601550798, 0.69960515111291777]], [[0.28894841908227131, -0.14340396601550798], [-0.14340396601550798, 
0.41787923950246331]], [[-0.14340396601550798, 0.95471157706396692], [0.3975080494679184, 
-0.14340396601550798]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_expandedData_rank3(self):
      arg0=-0.801969488354
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-0.9889605718250738, -0.37340300276544558], [-0.23838119100250488, 0.20477150811741951]], 
[[-0.20467295181293843, 0.35301722647248601], [-0.095474828781975996, -0.8510749109672644]], [[-0.70425456344214243, 
-0.46755265678910085], [0.66251795702833483, -0.92448552384625948]], [[-0.66717945104318677, -0.78744474882359006], 
[-0.16257558572897457, -0.26356489046284248]], [[-0.27934849406098761, 0.93892978662484139], [-0.18468624033742786, 
-0.70668348468539377]], [[-0.41995260750527241, -0.046679843759030781], [0.13620169917808034, 
0.46624408802032424]]])+(1.-msk_arg1)*numpy.array([[[-0.56084457456829484, 0.032990993182650197], [0.42149270935344574, 
0.1750548418363671]], [[-0.78410663312115148, -0.74664526632471984], [0.88029830669181575, -0.60878924835930359]], 
[[0.054733122499459386, 0.19985918328880037], [0.60155025489305403, 0.19928445576334952]], [[-0.086860661257179173, 
-0.14114143266705215], [0.57480201913819329, 0.38294532940281845]], [[0.39095132070017913, 0.096587125583033284], 
[0.50661117336330563, -0.043716480204347041]], [[0.74636288223324332, 0.82758235657560153], [-0.88560450972740812, 
-0.97248750693241459]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.80196948835439708, -0.37340300276544558], [-0.23838119100250488, 0.20477150811741951]], 
[[-0.20467295181293843, 0.35301722647248601], [-0.095474828781975996, -0.80196948835439708]], [[-0.70425456344214243, 
-0.46755265678910085], [0.66251795702833483, -0.80196948835439708]], [[-0.66717945104318677, -0.78744474882359006], 
[-0.16257558572897457, -0.26356489046284248]], [[-0.27934849406098761, 0.93892978662484139], [-0.18468624033742786, 
-0.70668348468539377]], [[-0.41995260750527241, -0.046679843759030781], [0.13620169917808034, 
0.46624408802032424]]])+(1.-msk_ref)*numpy.array([[[-0.56084457456829484, 0.032990993182650197], [0.42149270935344574, 
0.1750548418363671]], [[-0.78410663312115148, -0.74664526632471984], [0.88029830669181575, -0.60878924835930359]], 
[[0.054733122499459386, 0.19985918328880037], [0.60155025489305403, 0.19928445576334952]], [[-0.086860661257179173, 
-0.14114143266705215], [0.57480201913819329, 0.38294532940281845]], [[0.39095132070017913, 0.096587125583033284], 
[0.50661117336330563, -0.043716480204347041]], [[0.74636288223324332, 0.82758235657560153], [-0.80196948835439708, 
-0.80196948835439708]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_array_rank4(self):
      arg0=0.270905406087
      arg1=numpy.array([[[[-0.95360288151815653, -0.64078885487275894, 0.96879725589600185, 0.37349635624215671], 
[-0.23784772076621397, 0.65332164004684334, -0.2721768925317285, 0.22325285780446613], [0.065576671186904578, 
-0.0042823363828001781, 0.15081972618097739, -0.96500711704114939]], [[0.52806648530122047, -0.25468561493618047, 
0.55916471457602035, 0.48281251134172609], [0.55407748333919904, -0.11022873429516356, -0.43913854875915792, 
-0.12104124446879827], [-0.17051966474671065, 0.390161734069155, -0.40998699913604164, -0.63105941814244515]]], 
[[[-0.94813299631664916, 0.33923647049967887, -0.6464524838010095, 0.099997153482365375], [-0.19694004792045505, 
-0.21476580508277232, 0.50450751987287878, 0.0044718532670784406], [-0.35156116891903766, 0.93361770074048556, 
0.64503353549824793, 0.31429197499208672]], [[0.67485265775424352, -0.0069832167465539996, -0.27581854606767786, 
-0.049974746554111604], [0.88520110268538477, 0.59559353742053656, -0.88876809828391945, 0.14398781431221419], 
[0.49156586086617393, 0.34186756327969836, -0.39488984681038053, -0.49978462106847665]]], [[[0.4711536016262412, 
-0.84357230970236885, -0.60522103581380593, 0.74011510067186914], [0.058929816244633448, -0.51830971900097267, 
-0.92512610450551414, 0.79105987068184391], [-0.08685194216427039, 0.9276912197047904, 0.75649609887119307, 
-0.2819853018873526]], [[-0.23960953208469071, -0.026232616293173194, -0.60625185640538937, 0.41226223746307045], 
[-0.48188248446866555, -0.57075257849297079, -0.98430513472426018, 0.0026142817361640702], [-0.25427236937723752, 
-0.83104284375176407, -0.76917008485860205, 0.63675334398029215]]]])
      res=maximum(arg0,arg1)
      ref=numpy.array([[[[0.27090540608696001, 0.27090540608696001, 0.96879725589600185, 0.37349635624215671], 
[0.27090540608696001, 0.65332164004684334, 0.27090540608696001, 0.27090540608696001], [0.27090540608696001, 
0.27090540608696001, 0.27090540608696001, 0.27090540608696001]], [[0.52806648530122047, 0.27090540608696001, 
0.55916471457602035, 0.48281251134172609], [0.55407748333919904, 0.27090540608696001, 0.27090540608696001, 
0.27090540608696001], [0.27090540608696001, 0.390161734069155, 0.27090540608696001, 0.27090540608696001]]], 
[[[0.27090540608696001, 0.33923647049967887, 0.27090540608696001, 0.27090540608696001], [0.27090540608696001, 
0.27090540608696001, 0.50450751987287878, 0.27090540608696001], [0.27090540608696001, 0.93361770074048556, 0.64503353549824793, 
0.31429197499208672]], [[0.67485265775424352, 0.27090540608696001, 0.27090540608696001, 0.27090540608696001], 
[0.88520110268538477, 0.59559353742053656, 0.27090540608696001, 0.27090540608696001], [0.49156586086617393, 
0.34186756327969836, 0.27090540608696001, 0.27090540608696001]]], [[[0.4711536016262412, 0.27090540608696001, 
0.27090540608696001, 0.74011510067186914], [0.27090540608696001, 0.27090540608696001, 0.27090540608696001, 
0.79105987068184391], [0.27090540608696001, 0.9276912197047904, 0.75649609887119307, 0.27090540608696001]], 
[[0.27090540608696001, 0.27090540608696001, 0.27090540608696001, 0.41226223746307045], [0.27090540608696001, 
0.27090540608696001, 0.27090540608696001, 0.27090540608696001], [0.27090540608696001, 0.27090540608696001, 0.27090540608696001, 
0.63675334398029215]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_constData_rank4(self):
      arg0=-0.309869238816
      arg1=Data(numpy.array([[[[-0.82719032260804259, 0.55581165940832689, 0.28993332944093009, -0.86289948135541761], 
[0.48402181856866533, -0.32199675516574411, 0.12059109046218897, -0.7225649933438465], [-0.83246155788795284, 
0.81949583430522099, 0.917821601053878, -0.38511162370320817]], [[0.040807935475384971, 0.064255792480005569, 
-0.31189093961899572, -0.84263132870438207], [-0.019662050520152574, 0.44586496426970701, -0.91681580233687932, 
-0.18241240078470145], [-0.5433480154993422, -0.59905565233432823, -0.92180233099065312, 0.86950034082755967]]], 
[[[0.59960822306615458, 0.13652846774954908, 0.43957232336106244, -0.43425247806437417], [-0.96917998667692262, 
0.55216812010001926, 0.71335421685640177, 0.58559993798062027], [0.70518632845671458, -0.51213306047690121, 
0.49991004029291486, -0.88896060694390422]], [[-0.01327086420919299, 0.94285928205783565, 0.36084019612625773, 
0.75948720217972943], [0.15534832801909948, -0.087517839294278144, -0.82737480559591559, -0.17830159519382138], 
[-0.53216202742506535, 0.14079322663297011, 0.51872828089506218, -0.87286763824554892]]], [[[0.63517519655793753, 
0.83248640824649534, -0.96421638668998866, -0.21027115956215714], [-0.65484094298333928, -0.20529300129283179, 
0.30264230449111329, 0.57118632679487069], [0.94473501747170197, -0.073741122006270921, -0.16445760290671507, 
-0.94122715459958095]], [[-0.47320172714174435, 0.52414108821599092, -0.58822269419300688, 0.65154104107711608], 
[0.79073955181970002, 0.66222126451513996, 0.50406761329019711, 0.80692979137230991], [-0.44007779482511467, 
0.83184141637396403, -0.59153030329212752, 0.67780297389993294]]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[-0.30986923881604533, 0.55581165940832689, 0.28993332944093009, -0.30986923881604533], 
[0.48402181856866533, -0.30986923881604533, 0.12059109046218897, -0.30986923881604533], [-0.30986923881604533, 
0.81949583430522099, 0.917821601053878, -0.30986923881604533]], [[0.040807935475384971, 0.064255792480005569, 
-0.30986923881604533, -0.30986923881604533], [-0.019662050520152574, 0.44586496426970701, -0.30986923881604533, 
-0.18241240078470145], [-0.30986923881604533, -0.30986923881604533, -0.30986923881604533, 0.86950034082755967]]], 
[[[0.59960822306615458, 0.13652846774954908, 0.43957232336106244, -0.30986923881604533], [-0.30986923881604533, 
0.55216812010001926, 0.71335421685640177, 0.58559993798062027], [0.70518632845671458, -0.30986923881604533, 
0.49991004029291486, -0.30986923881604533]], [[-0.01327086420919299, 0.94285928205783565, 0.36084019612625773, 
0.75948720217972943], [0.15534832801909948, -0.087517839294278144, -0.30986923881604533, -0.17830159519382138], 
[-0.30986923881604533, 0.14079322663297011, 0.51872828089506218, -0.30986923881604533]]], [[[0.63517519655793753, 
0.83248640824649534, -0.30986923881604533, -0.21027115956215714], [-0.30986923881604533, -0.20529300129283179, 
0.30264230449111329, 0.57118632679487069], [0.94473501747170197, -0.073741122006270921, -0.16445760290671507, 
-0.30986923881604533]], [[-0.30986923881604533, 0.52414108821599092, -0.30986923881604533, 0.65154104107711608], 
[0.79073955181970002, 0.66222126451513996, 0.50406761329019711, 0.80692979137230991], [-0.30986923881604533, 
0.83184141637396403, -0.30986923881604533, 0.67780297389993294]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_float_rank0_expandedData_rank4(self):
      arg0=-0.246649004468
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-0.59710738457463131, 0.081437393836007343, -0.56546756227867667, 0.71443872053420754], 
[-0.57457972849842576, 0.99714958865565517, -0.39999443875374929, -0.68094907708339836], [0.34526888607366546, 
-0.29241451792195905, 0.96907276723625935, -0.2966708791455217]], [[-0.76911685555342713, 0.21001220920747699, 
0.7216150128564951, 0.57784624578115795], [0.59505769173662104, -0.11354294660473241, -0.66879018252488964, 
0.78630853516684396], [0.3616307103083698, -0.79300540303062417, 0.74450579924491178, -0.054555970573139767]]], 
[[[0.96283128026153286, -0.87620458144557722, 0.72154059555131611, 0.0082452389660727654], [-0.046764826200485832, 
-0.53421181085747382, -0.19205291797994817, 0.15732232154864123], [-0.64350186911984397, 0.86535072837345894, 
-0.089737099310619683, -0.68915912536904322]], [[-0.50600653327166056, 0.87271444603618842, 0.42777020885243378, 
0.63522958101972282], [-0.62649776548253322, 0.4567028350414013, 0.97523051451214271, 0.81579778175168283], 
[0.20125503319935079, -0.46379240558699131, -0.31245213377702896, 0.2704000376928466]]], [[[-0.52356266147432984, 
0.17857675536580886, 0.3664267659935363, -0.88731256350709842], [-0.37314717051082713, 0.54211837452172662, 
-0.72485428615154301, -0.60715112600284726], [-0.99925424756131687, -0.16075147245083277, -0.0025301737170317917, 
0.14075502002330875]], [[-0.72510015564658348, -0.070820851711875132, -0.05397335487409638, 0.31296807923888648], 
[0.66626010161325921, -0.7189380142049282, -0.62806927909760879, 0.23068004843743117], [-0.70574206651459992, 
-0.10745983979108953, -0.16531954799426085, -0.43129597709381007]]]])+(1.-msk_arg1)*numpy.array([[[[0.87166570761445938, 
0.7318590182568474, -0.80737962090297621, 0.53766528032150029], [-0.29770111995100423, -0.041729104475766832, 
0.95558043277951188, 0.87068423363682523], [0.26116387820103548, 0.76929801350975846, -0.34600777484521661, 
0.20956877787614303]], [[-0.27623529169912397, 0.74714116143240372, 0.67467631548270068, 0.80485965465015274], 
[-0.80196951310483411, 0.54024415368197976, -0.45263989703849994, -0.77389469864887639], [-0.11922685978923897, 
0.39196315394661263, 0.17728969359335789, -0.38647232675584942]]], [[[0.96589541331221995, -0.80607305323690448, 
-0.7174848711693449, 0.37455766619796327], [0.91224675515514875, -0.8324612503371378, -0.95542285313177433, 
-0.11861636937415732], [0.47029221874868399, -0.40274109861783103, 0.9175933763798334, -0.43517134542523084]], 
[[-0.47437396776048946, -0.62388433551176026, -0.93178844269694805, 0.16109167169456717], [0.41540675061264087, 
0.56983043087032437, 0.79225905558275778, -0.70471106926370264], [0.57876248922509843, -0.92178848481156983, 
-0.20102218593254717, -0.51116400848044763]]], [[[0.43214438484520512, -0.44676797608176422, -0.96373941940887264, 
-0.97610643146800369], [0.1482565940064382, -0.15856386047336768, -0.97338314492097022, -0.77655581838335452], 
[0.14918854112868329, 0.35190145831475594, 0.76106715747610876, 0.87042812467135966]], [[0.5937340223957881, 
0.93249676000364778, 0.62934483950694187, -0.79475089188750059], [-0.7844570636858641, 0.063504498696464129, 
0.80077845206462461, -0.17147808526676922], [0.77647533980748817, -0.41826398787923291, -0.93662968561496873, 
0.5899829279078268]]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.24664900446763838, 0.081437393836007343, -0.24664900446763838, 0.71443872053420754], 
[-0.24664900446763838, 0.99714958865565517, -0.24664900446763838, -0.24664900446763838], [0.34526888607366546, 
-0.24664900446763838, 0.96907276723625935, -0.24664900446763838]], [[-0.24664900446763838, 0.21001220920747699, 
0.7216150128564951, 0.57784624578115795], [0.59505769173662104, -0.11354294660473241, -0.24664900446763838, 
0.78630853516684396], [0.3616307103083698, -0.24664900446763838, 0.74450579924491178, -0.054555970573139767]]], 
[[[0.96283128026153286, -0.24664900446763838, 0.72154059555131611, 0.0082452389660727654], [-0.046764826200485832, 
-0.24664900446763838, -0.19205291797994817, 0.15732232154864123], [-0.24664900446763838, 0.86535072837345894, 
-0.089737099310619683, -0.24664900446763838]], [[-0.24664900446763838, 0.87271444603618842, 0.42777020885243378, 
0.63522958101972282], [-0.24664900446763838, 0.4567028350414013, 0.97523051451214271, 0.81579778175168283], 
[0.20125503319935079, -0.24664900446763838, -0.24664900446763838, 0.2704000376928466]]], [[[-0.24664900446763838, 
0.17857675536580886, 0.3664267659935363, -0.24664900446763838], [-0.24664900446763838, 0.54211837452172662, 
-0.24664900446763838, -0.24664900446763838], [-0.24664900446763838, -0.16075147245083277, -0.0025301737170317917, 
0.14075502002330875]], [[-0.24664900446763838, -0.070820851711875132, -0.05397335487409638, 0.31296807923888648], 
[0.66626010161325921, -0.24664900446763838, -0.24664900446763838, 0.23068004843743117], [-0.24664900446763838, 
-0.10745983979108953, -0.16531954799426085, -0.24664900446763838]]]])+(1.-msk_ref)*numpy.array([[[[0.87166570761445938, 
0.7318590182568474, -0.24664900446763838, 0.53766528032150029], [-0.24664900446763838, -0.041729104475766832, 
0.95558043277951188, 0.87068423363682523], [0.26116387820103548, 0.76929801350975846, -0.24664900446763838, 
0.20956877787614303]], [[-0.24664900446763838, 0.74714116143240372, 0.67467631548270068, 0.80485965465015274], 
[-0.24664900446763838, 0.54024415368197976, -0.24664900446763838, -0.24664900446763838], [-0.11922685978923897, 
0.39196315394661263, 0.17728969359335789, -0.24664900446763838]]], [[[0.96589541331221995, -0.24664900446763838, 
-0.24664900446763838, 0.37455766619796327], [0.91224675515514875, -0.24664900446763838, -0.24664900446763838, 
-0.11861636937415732], [0.47029221874868399, -0.24664900446763838, 0.9175933763798334, -0.24664900446763838]], 
[[-0.24664900446763838, -0.24664900446763838, -0.24664900446763838, 0.16109167169456717], [0.41540675061264087, 
0.56983043087032437, 0.79225905558275778, -0.24664900446763838], [0.57876248922509843, -0.24664900446763838, 
-0.20102218593254717, -0.24664900446763838]]], [[[0.43214438484520512, -0.24664900446763838, -0.24664900446763838, 
-0.24664900446763838], [0.1482565940064382, -0.15856386047336768, -0.24664900446763838, -0.24664900446763838], 
[0.14918854112868329, 0.35190145831475594, 0.76106715747610876, 0.87042812467135966]], [[0.5937340223957881, 
0.93249676000364778, 0.62934483950694187, -0.24664900446763838], [-0.24664900446763838, 0.063504498696464129, 
0.80077845206462461, -0.17147808526676922], [0.77647533980748817, -0.24664900446763838, -0.24664900446763838, 
0.5899829279078268]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_float_rank0(self):
      arg0=numpy.array(-0.532218032885)
      arg1=-0.367677151081
      res=maximum(arg0,arg1)
      ref=numpy.array(-0.367677151081)
      if isinstance(res,numpy.ndarray):
          self.assertEqual(res.shape,(),"wrong shape of result.")
      else:
          self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank1_float_rank0(self):
      arg0=numpy.array([0.53113072385818016, -0.91297296718973731])
      arg1=0.192754148616
      res=maximum(arg0,arg1)
      ref=numpy.array([0.53113072385818016, 0.19275414861572537])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank2_float_rank0(self):
      arg0=numpy.array([[-0.83224495764259587, 0.67546682831068661, 0.79825688470140466, -0.51659524388186018, 
-0.57330212318607887], [0.43871883762893171, -0.51383061657367102, 0.046022377444661977, -0.97181480474218085, 
-0.064914882677348462], [0.54107834535109811, 0.52030930437850165, -0.46403217495914517, -0.40333646983529814, 
0.14402492053142035], [-0.06466356036920895, -0.72560803049513556, -0.71248570650624976, 0.29239946340327383, 
0.23126053673271052]])
      arg1=0.436559137295
      res=maximum(arg0,arg1)
      ref=numpy.array([[0.43655913729458451, 0.67546682831068661, 0.79825688470140466, 0.43655913729458451, 
0.43655913729458451], [0.43871883762893171, 0.43655913729458451, 0.43655913729458451, 0.43655913729458451, 
0.43655913729458451], [0.54107834535109811, 0.52030930437850165, 0.43655913729458451, 0.43655913729458451, 
0.43655913729458451], [0.43655913729458451, 0.43655913729458451, 0.43655913729458451, 0.43655913729458451, 
0.43655913729458451]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank3_float_rank0(self):
      arg0=numpy.array([[[0.14602536321923187, 0.3890404886953438], [0.96763189489480217, 0.31878780255153472]], 
[[-0.80681415429363024, 0.81150006186927182], [0.085441963781010877, 0.28294027876384797]], [[-0.90676678758245544, 
0.035304671448607161], [-0.24178422686744017, -0.020920873167083132]], [[-0.85375981355743735, -0.44307345060683789], 
[0.0374669536660035, -0.0058344936858338858]], [[0.31755388514905247, 0.33655876978208488], [-0.60668072237969684, 
-0.046438959267768309]], [[-0.74363599042485151, -0.8956664888069259], [-0.53199684552981585, 0.75892708027917499]]])
      arg1=-0.149468998389
      res=maximum(arg0,arg1)
      ref=numpy.array([[[0.14602536321923187, 0.3890404886953438], [0.96763189489480217, 0.31878780255153472]], 
[[-0.1494689983892894, 0.81150006186927182], [0.085441963781010877, 0.28294027876384797]], [[-0.1494689983892894, 
0.035304671448607161], [-0.1494689983892894, -0.020920873167083132]], [[-0.1494689983892894, -0.1494689983892894], 
[0.0374669536660035, -0.0058344936858338858]], [[0.31755388514905247, 0.33655876978208488], [-0.1494689983892894, 
-0.046438959267768309]], [[-0.1494689983892894, -0.1494689983892894], [-0.1494689983892894, 0.75892708027917499]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank4_float_rank0(self):
      arg0=numpy.array([[[[0.2890956567711751, 0.0047389611017556454, 0.87892527211444582, -0.65150693309569196], 
[0.41621480995390114, -0.43854307594957254, -0.59815785643830077, -0.67517272581107557], [0.29083357354248873, 
-0.34488587447865737, -0.3534529402795854, -0.76014085074946847]], [[-0.19051880582497072, 0.85284977603728396, 
0.31270253036076934, 0.37421013369344447], [0.71318255965443345, 0.92162734459590911, 0.67934988220751147, 
-0.42337007079808697], [0.42833893416405266, 0.30720360302238192, 0.27000152473334049, 0.95393479511777279]]], 
[[[-0.098035752002608278, 0.13176090528152407, 0.62934363744184929, -0.053289126771334905], [0.81346047837084501, 
-0.45301649520463716, -0.96631411831608749, -0.071951542993987649], [0.99690192197172367, -0.025819634205427322, 
0.87468585164061108, 0.60501309254424518]], [[-0.88303485657993463, 0.46312758028512135, -0.77275723545859076, 
-0.067439349816685201], [0.14089517815207553, 0.56655437330646286, 0.35573146108388021, -0.9370989113947783], 
[-0.33072965554665301, -0.81929016865935234, -0.981755182164328, -0.80123180974034836]]], [[[0.32559736893252644, 
-0.15298445424344309, 0.5756357845038651, 0.54846826957121642], [0.49314026154864776, 0.40147202599848408, 0.65052530484204629, 
0.78030715854096466], [0.11772129008710874, 0.97339684272768734, -0.46213970918704739, 0.64480146515364178]], 
[[-0.489873441461109, 0.67676986710802556, 0.57320598982638238, -0.081510204873377567], [0.14129642797645747, 
-0.88117171552134099, 0.62120016125873723, 0.62570138942740905], [-0.4571545505832082, 0.18504471982789994, 0.269612515446229, 
-0.92783580872646998]]]])
      arg1=0.44439427692
      res=maximum(arg0,arg1)
      ref=numpy.array([[[[0.44439427692047384, 0.44439427692047384, 0.87892527211444582, 0.44439427692047384], 
[0.44439427692047384, 0.44439427692047384, 0.44439427692047384, 0.44439427692047384], [0.44439427692047384, 
0.44439427692047384, 0.44439427692047384, 0.44439427692047384]], [[0.44439427692047384, 0.85284977603728396, 
0.44439427692047384, 0.44439427692047384], [0.71318255965443345, 0.92162734459590911, 0.67934988220751147, 
0.44439427692047384], [0.44439427692047384, 0.44439427692047384, 0.44439427692047384, 0.95393479511777279]]], 
[[[0.44439427692047384, 0.44439427692047384, 0.62934363744184929, 0.44439427692047384], [0.81346047837084501, 
0.44439427692047384, 0.44439427692047384, 0.44439427692047384], [0.99690192197172367, 0.44439427692047384, 0.87468585164061108, 
0.60501309254424518]], [[0.44439427692047384, 0.46312758028512135, 0.44439427692047384, 0.44439427692047384], 
[0.44439427692047384, 0.56655437330646286, 0.44439427692047384, 0.44439427692047384], [0.44439427692047384, 
0.44439427692047384, 0.44439427692047384, 0.44439427692047384]]], [[[0.44439427692047384, 0.44439427692047384, 
0.5756357845038651, 0.54846826957121642], [0.49314026154864776, 0.44439427692047384, 0.65052530484204629, 0.78030715854096466], 
[0.44439427692047384, 0.97339684272768734, 0.44439427692047384, 0.64480146515364178]], [[0.44439427692047384, 
0.67676986710802556, 0.57320598982638238, 0.44439427692047384], [0.44439427692047384, 0.44439427692047384, 0.62120016125873723, 
0.62570138942740905], [0.44439427692047384, 0.44439427692047384, 0.44439427692047384, 0.44439427692047384]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_array_rank0(self):
      arg0=numpy.array(-0.595911728088)
      arg1=numpy.array(-0.259958577128)
      res=maximum(arg0,arg1)
      ref=numpy.array(-0.259958577128)
      if isinstance(res,numpy.ndarray):
          self.assertEqual(res.shape,(),"wrong shape of result.")
      else:
          self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank1_array_rank0(self):
      arg0=numpy.array([0.70448785703212291, 0.83900014998124073])
      arg1=numpy.array(-0.728326601622)
      res=maximum(arg0,arg1)
      ref=numpy.array([0.70448785703212291, 0.83900014998124073])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank2_array_rank0(self):
      arg0=numpy.array([[0.58907572104688044, -0.30767970415015888, -0.78138846520643046, -0.15168954291983616, 
0.31446212443091293], [0.074998472598995392, -0.9250742963204186, 0.45334932675665041, -0.95916572344077733, 
-0.99342314865250936], [-0.79251531920902263, 0.44321789574157977, 0.62908133704628444, 0.24472622296386093, 
0.29465980693370675], [-0.5523390670941164, -0.096556923965434915, -0.97293609728291464, 0.4811104330547229, 
0.11676047360942654]])
      arg1=numpy.array(-0.637835089534)
      res=maximum(arg0,arg1)
      ref=numpy.array([[0.58907572104688044, -0.30767970415015888, -0.63783508953404722, -0.15168954291983616, 
0.31446212443091293], [0.074998472598995392, -0.63783508953404722, 0.45334932675665041, -0.63783508953404722, 
-0.63783508953404722], [-0.63783508953404722, 0.44321789574157977, 0.62908133704628444, 0.24472622296386093, 
0.29465980693370675], [-0.5523390670941164, -0.096556923965434915, -0.63783508953404722, 0.4811104330547229, 
0.11676047360942654]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank3_array_rank0(self):
      arg0=numpy.array([[[0.26078403166817066, -0.88883099092846662], [0.2962776357981165, 0.33765471443325268]], 
[[0.61994569524479037, -0.86070803156387798], [0.40887497854892496, -0.95032693825538606]], [[-0.6230603384583091, 
0.84641044210513128], [0.072929215696905469, 0.63605426925408093]], [[0.82476138834838242, -0.87239621000692358], 
[0.9136576981759601, 0.012874827321107896]], [[0.6527601185704035, 0.45834972014119924], [-0.20281376602514722, 
0.78801993661046477]], [[-0.84382527553821407, 0.69859016456466971], [0.95609346718482136, -0.9642394043877307]]])
      arg1=numpy.array(-0.0639685246813)
      res=maximum(arg0,arg1)
      ref=numpy.array([[[0.26078403166817066, -0.063968524681262906], [0.2962776357981165, 0.33765471443325268]], 
[[0.61994569524479037, -0.063968524681262906], [0.40887497854892496, -0.063968524681262906]], [[-0.063968524681262906, 
0.84641044210513128], [0.072929215696905469, 0.63605426925408093]], [[0.82476138834838242, -0.063968524681262906], 
[0.9136576981759601, 0.012874827321107896]], [[0.6527601185704035, 0.45834972014119924], [-0.063968524681262906, 
0.78801993661046477]], [[-0.063968524681262906, 0.69859016456466971], [0.95609346718482136, -0.063968524681262906]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank4_array_rank0(self):
      arg0=numpy.array([[[[-0.49178507725843357, -0.95412613874160401, -0.89953408681402758, -0.4182783638072558], 
[-0.12851206262547121, 0.97771672278447719, 0.095498689842355322, -0.63677823186616767], [-0.31171543678709579, 
-0.24066261014019164, 0.8884558662985651, -0.86534369235026132]], [[0.41073749486304023, -0.75082798331554312, 
-0.021514401962046348, -0.4552145355922177], [0.025376418086104691, 0.94693831924338046, 0.72528013847070816, 
0.5590032330140231], [-0.63961957547864201, 0.38553731067491559, 0.47766722193193556, 0.22722302846704578]]], 
[[[-0.8552086040279212, -0.028628848798257334, 0.39615293686569841, 0.60472409304678898], [-0.76179395489150892, 
0.34743550408059365, 0.84900306720985297, -0.32754549604906047], [0.96774618220062791, -0.17510456094746396, 
-0.83577780367198806, -0.28595508510654843]], [[-0.72033121788336163, -0.96380753221847604, 0.64121075422201179, 
0.50807412144193731], [-0.22520904548844145, -0.79626857735444134, 0.6727388150821505, -0.97429723888611686], 
[-0.3912584771763985, 0.017073154831940274, -0.78645964556472814, 0.063579858350630403]]], [[[0.49275761913585359, 
0.68061912084321108, 0.27488873542534376, -0.91020336944963454], [0.38754903931740148, -0.095266564605377857, 
-0.34983002073831537, -0.98686346316121343], [-0.57870251452427079, 0.019782802975875313, -0.99030465294393522, 
0.36519674435393079]], [[0.36201122081142789, 0.89247761722577512, 0.88862029406919207, 0.65118385314611027], 
[0.62444020668337985, 0.29010577239708102, 0.50090883083435256, -0.86322012454452723], [-0.25882379276404111, 
0.50333245341748101, -0.98127424266544994, 0.32881151993132085]]]])
      arg1=numpy.array(0.54009693064)
      res=maximum(arg0,arg1)
      ref=numpy.array([[[[0.54009693064018083, 0.54009693064018083, 0.54009693064018083, 0.54009693064018083], 
[0.54009693064018083, 0.97771672278447719, 0.54009693064018083, 0.54009693064018083], [0.54009693064018083, 
0.54009693064018083, 0.8884558662985651, 0.54009693064018083]], [[0.54009693064018083, 0.54009693064018083, 
0.54009693064018083, 0.54009693064018083], [0.54009693064018083, 0.94693831924338046, 0.72528013847070816, 0.5590032330140231], 
[0.54009693064018083, 0.54009693064018083, 0.54009693064018083, 0.54009693064018083]]], [[[0.54009693064018083, 
0.54009693064018083, 0.54009693064018083, 0.60472409304678898], [0.54009693064018083, 0.54009693064018083, 0.84900306720985297, 
0.54009693064018083], [0.96774618220062791, 0.54009693064018083, 0.54009693064018083, 0.54009693064018083]], 
[[0.54009693064018083, 0.54009693064018083, 0.64121075422201179, 0.54009693064018083], [0.54009693064018083, 
0.54009693064018083, 0.6727388150821505, 0.54009693064018083], [0.54009693064018083, 0.54009693064018083, 0.54009693064018083, 
0.54009693064018083]]], [[[0.54009693064018083, 0.68061912084321108, 0.54009693064018083, 0.54009693064018083], 
[0.54009693064018083, 0.54009693064018083, 0.54009693064018083, 0.54009693064018083], [0.54009693064018083, 
0.54009693064018083, 0.54009693064018083, 0.54009693064018083]], [[0.54009693064018083, 0.89247761722577512, 
0.88862029406919207, 0.65118385314611027], [0.62444020668337985, 0.54009693064018083, 0.54009693064018083, 
0.54009693064018083], [0.54009693064018083, 0.54009693064018083, 0.54009693064018083, 0.54009693064018083]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_constData_rank0(self):
      arg0=numpy.array(0.275719067126)
      arg1=Data(0.379978099496,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(0.379978099496,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank1_constData_rank0(self):
      arg0=numpy.array([0.23412985120029073, 0.96842799591312634])
      arg1=Data(0.0282351777389,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.23412985120029073, 0.96842799591312634]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank2_constData_rank0(self):
      arg0=numpy.array([[0.75083228363058563, 0.14021129209291994, 0.66454891543379668, 0.52438137383900618, 
0.59989233019891586], [-0.64926806752121502, -0.31973316364062088, -0.15677459149134809, 0.94726185071047286, 
0.77006499756531577], [0.96436511217108922, 0.10798486901631477, 0.49614119848238003, -0.58809677000944571, 
0.40684393221640747], [-0.33426541098798523, -0.99641768740611458, 0.019371391555169426, 0.41032148071109376, 
0.55944773037045037]])
      arg1=Data(0.0504551040829,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.75083228363058563, 0.14021129209291994, 0.66454891543379668, 0.52438137383900618, 
0.59989233019891586], [0.050455104082914204, 0.050455104082914204, 0.050455104082914204, 0.94726185071047286, 
0.77006499756531577], [0.96436511217108922, 0.10798486901631477, 0.49614119848238003, 0.050455104082914204, 
0.40684393221640747], [0.050455104082914204, 0.050455104082914204, 0.050455104082914204, 0.41032148071109376, 
0.55944773037045037]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank3_constData_rank0(self):
      arg0=numpy.array([[[0.68078035361717415, 0.77513835622840355], [0.88778377916351281, -0.59991037059023289]], 
[[-0.31479263876172614, 0.52689690629824115], [-0.32445682623377969, 0.70068635269972779]], [[0.7068855302553303, 
-0.90632512202385662], [-0.69310089072096459, 0.21580652641517606]], [[0.18682929631820056, 0.98487526483307875], 
[-0.80732797686141189, -0.74387425497735205]], [[-0.94959049904561454, -0.26401122319745207], [0.97666196551554485, 
0.13397037446463989]], [[-0.32237452231000963, 0.23366086287779164], [-0.66314084754224023, 0.061657994330439836]]])
      arg1=Data(0.887177842883,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.88717784288295465, 0.88717784288295465], [0.88778377916351281, 0.88717784288295465]], 
[[0.88717784288295465, 0.88717784288295465], [0.88717784288295465, 0.88717784288295465]], [[0.88717784288295465, 
0.88717784288295465], [0.88717784288295465, 0.88717784288295465]], [[0.88717784288295465, 0.98487526483307875], 
[0.88717784288295465, 0.88717784288295465]], [[0.88717784288295465, 0.88717784288295465], [0.97666196551554485, 
0.88717784288295465]], [[0.88717784288295465, 0.88717784288295465], [0.88717784288295465, 
0.88717784288295465]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank4_constData_rank0(self):
      arg0=numpy.array([[[[-0.44596394506288406, -0.99451216053143821, 0.80947207669238974, 0.53381678739762761], 
[0.24988481018214626, 0.97340528016730477, -0.88181308417483706, -0.10501736634476666], [0.67231055089947356, 
-0.41705694674616267, -0.99130432976066873, -0.76183835148666756]], [[0.87564340019720688, -0.56545607723354552, 
0.35986228734119252, -0.95148055206041304], [0.54888663455682729, 0.97420887940931578, 0.058263938512163049, 
0.76370352854831602], [-0.82673099109694004, 0.9659484007501713, -0.92707003031757185, -0.29800942664995955]]], 
[[[0.22672462889085043, 0.77216678282924223, -0.68765485895438716, -0.41035411530512622], [-0.10574234135684346, 
0.41868370912220354, -0.42894043779354374, 0.14689038755897377], [0.80637358889951183, 0.0048710336014621269, 
0.95227700237370771, 0.67742226677379058]], [[0.33943135111692935, 0.051854083713816834, -0.11097498559618657, 
-0.28695360058894792], [0.25170868190713924, 0.51759300366308403, 0.045719489002853608, 0.85414177693285209], 
[-0.62313668843495651, -0.65158905094111397, -0.35965515542399928, -0.91881459559837508]]], [[[0.50411396804110731, 
-0.35071960896573962, 0.85463428495185267, 0.047161400398034514], [0.96936117001305488, -0.23708858547871281, 
-0.97958490805427312, 0.19911749296919523], [0.3558068103726566, 0.83166317419688163, 0.094348126909065133, 
0.53229454234478757]], [[-0.95064691521898426, 0.40080396732915236, 0.33234699751344388, -0.9871206815396758], 
[0.98903106268798591, 0.091842667447183279, 0.29485522713344281, -0.33705972294401088], [0.29805334379644233, 
-0.9604786924125317, -0.73602864043687255, -0.40470873942057994]]]])
      arg1=Data(-0.176207534638,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[-0.17620753463766059, -0.17620753463766059, 0.80947207669238974, 0.53381678739762761], 
[0.24988481018214626, 0.97340528016730477, -0.17620753463766059, -0.10501736634476666], [0.67231055089947356, 
-0.17620753463766059, -0.17620753463766059, -0.17620753463766059]], [[0.87564340019720688, -0.17620753463766059, 
0.35986228734119252, -0.17620753463766059], [0.54888663455682729, 0.97420887940931578, 0.058263938512163049, 
0.76370352854831602], [-0.17620753463766059, 0.9659484007501713, -0.17620753463766059, -0.17620753463766059]]], 
[[[0.22672462889085043, 0.77216678282924223, -0.17620753463766059, -0.17620753463766059], [-0.10574234135684346, 
0.41868370912220354, -0.17620753463766059, 0.14689038755897377], [0.80637358889951183, 0.0048710336014621269, 
0.95227700237370771, 0.67742226677379058]], [[0.33943135111692935, 0.051854083713816834, -0.11097498559618657, 
-0.17620753463766059], [0.25170868190713924, 0.51759300366308403, 0.045719489002853608, 0.85414177693285209], 
[-0.17620753463766059, -0.17620753463766059, -0.17620753463766059, -0.17620753463766059]]], [[[0.50411396804110731, 
-0.17620753463766059, 0.85463428495185267, 0.047161400398034514], [0.96936117001305488, -0.17620753463766059, 
-0.17620753463766059, 0.19911749296919523], [0.3558068103726566, 0.83166317419688163, 0.094348126909065133, 
0.53229454234478757]], [[-0.17620753463766059, 0.40080396732915236, 0.33234699751344388, -0.17620753463766059], 
[0.98903106268798591, 0.091842667447183279, 0.29485522713344281, -0.17620753463766059], [0.29805334379644233, 
-0.17620753463766059, -0.17620753463766059, -0.17620753463766059]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_expandedData_rank0(self):
      arg0=numpy.array(-0.239808661002)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.499507234748)+(1.-msk_arg1)*(-0.671148207596)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.239808661002)+(1.-msk_ref)*(-0.239808661002)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank1_expandedData_rank0(self):
      arg0=numpy.array([-0.097106554962706371, -0.53299957388591124])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.48432167856)+(1.-msk_arg1)*(-0.25985897402)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.097106554962706371, -0.48432167856004193])+(1.-msk_ref)*numpy.array([-0.097106554962706371, 
-0.25985897401999281])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank2_expandedData_rank0(self):
      arg0=numpy.array([[-0.13167949633164855, -0.0083337422827065577, -0.38573640576223611, -0.069870464900080531, 
0.77840356126391153], [0.62218334866236424, 0.68483355583508021, -0.48863487618037715, -0.0082838053084859364, 
-0.33346342674536089], [-0.89360989325312534, -0.9751359066636649, -0.0084300747210315219, -0.96657583767552402, 
-0.10309752389903437], [-0.22842150326198207, -0.49737293664485316, -0.47025074030093283, -0.67836806796837679, 
0.99564770637844346]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.0781036984237)+(1.-msk_arg1)*(-0.0843238721184)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.078103698423652057, 0.078103698423652057, 0.078103698423652057, 0.078103698423652057, 
0.77840356126391153], [0.62218334866236424, 0.68483355583508021, 0.078103698423652057, 0.078103698423652057, 
0.078103698423652057], [0.078103698423652057, 0.078103698423652057, 0.078103698423652057, 0.078103698423652057, 
0.078103698423652057], [0.078103698423652057, 0.078103698423652057, 0.078103698423652057, 0.078103698423652057, 
0.99564770637844346]])+(1.-msk_ref)*numpy.array([[-0.084323872118388099, -0.0083337422827065577, -0.084323872118388099, 
-0.069870464900080531, 0.77840356126391153], [0.62218334866236424, 0.68483355583508021, -0.084323872118388099, 
-0.0082838053084859364, -0.084323872118388099], [-0.084323872118388099, -0.084323872118388099, -0.0084300747210315219, 
-0.084323872118388099, -0.084323872118388099], [-0.084323872118388099, -0.084323872118388099, -0.084323872118388099, 
-0.084323872118388099, 0.99564770637844346]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank3_expandedData_rank0(self):
      arg0=numpy.array([[[-0.70161399966991178, 0.33853877023786727], [-0.76494633110216448, -0.93684954048568492]], 
[[-0.65137354058954, 0.26553439129603973], [0.64552443308190122, -0.60830143039855877]], [[0.8023162924896885, 
-0.32038038872912455], [-0.47547406933615965, 0.27454091657539204]], [[-0.9672523328014897, -0.71919678369683115], 
[-0.35423678746927645, 0.81975223582448642]], [[-0.81403739453676183, 0.0020767234774008436], [0.94253026200138423, 
0.98275512168780477]], [[-0.24210704831235974, 0.71604514999739655], [-0.55274691970980117, 0.99831912241667875]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.0102246658782)+(1.-msk_arg1)*(0.652230128332)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.010224665878218353, 0.33853877023786727], [0.010224665878218353, 0.010224665878218353]], 
[[0.010224665878218353, 0.26553439129603973], [0.64552443308190122, 0.010224665878218353]], [[0.8023162924896885, 
0.010224665878218353], [0.010224665878218353, 0.27454091657539204]], [[0.010224665878218353, 0.010224665878218353], 
[0.010224665878218353, 0.81975223582448642]], [[0.010224665878218353, 0.010224665878218353], [0.94253026200138423, 
0.98275512168780477]], [[0.010224665878218353, 0.71604514999739655], [0.010224665878218353, 
0.99831912241667875]]])+(1.-msk_ref)*numpy.array([[[0.65223012833196758, 0.65223012833196758], [0.65223012833196758, 
0.65223012833196758]], [[0.65223012833196758, 0.65223012833196758], [0.65223012833196758, 0.65223012833196758]], 
[[0.8023162924896885, 0.65223012833196758], [0.65223012833196758, 0.65223012833196758]], [[0.65223012833196758, 
0.65223012833196758], [0.65223012833196758, 0.81975223582448642]], [[0.65223012833196758, 0.65223012833196758], 
[0.94253026200138423, 0.98275512168780477]], [[0.65223012833196758, 0.71604514999739655], [0.65223012833196758, 
0.99831912241667875]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank4_expandedData_rank0(self):
      arg0=numpy.array([[[[0.11857521935136139, 0.093929496673029522, -0.86185255457483856, 0.66859958438889944], 
[-0.71414593797980697, 0.11258374469683963, 0.94835492072493288, -0.68730039028716172], [-0.69653609424045704, 
0.079954029214261757, -0.091180257555399713, -0.1442422632927951]], [[0.26155223948894291, -0.27259910943079357, 
0.74661493838533688, 0.53585345987130495], [-0.60022580287227467, -0.48790336997422057, 0.94579083650142803, 
-0.8863996264681393], [-0.48285648955978755, 0.43421403658329649, 0.12057557891417603, -0.48016460023246776]]], 
[[[0.82622400614598757, 0.89451467594773959, -0.42386713423727307, 0.4292564116088935], [0.15325746129096651, 
-0.40650587605155497, 0.48852383833911195, -0.12133478623760663], [0.79062049417854996, -0.18313225818980827, 
-0.97553885725628553, -0.8652842476948277]], [[-0.05398981985994844, 0.25155972652674774, 0.58065613688539863, 
0.49996739942398416], [-0.96741564092273324, 0.25704723590663892, 0.21718381354462557, -0.87988046847383972], 
[0.17224341707309865, 0.2585605917314151, -0.61262203743428256, 0.088679962718663852]]], [[[0.95856449568455826, 
-0.21525011988362075, -0.87880681141864181, 0.74102075636768272], [0.41959007487547573, 0.80453495515335471, 
0.48817109722463581, -0.11806636970967643], [0.90075904019500075, 0.47717994916605289, 0.35995395169862054, 
0.40418049684201307]], [[0.28919635447948355, -0.28199908853065292, -0.051524269447178916, 0.39147412124693748], 
[-0.21236681046692141, -0.35585295155544783, -0.4325141024321022, 0.50366801462727206], [-0.69662947289754285, 
-0.10168687807009102, -0.74795243165887348, -0.26529343675798978]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.933797791441)+(1.-msk_arg1)*(-0.798477769938)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.11857521935136139, 0.093929496673029522, -0.86185255457483856, 0.66859958438889944], 
[-0.71414593797980697, 0.11258374469683963, 0.94835492072493288, -0.68730039028716172], [-0.69653609424045704, 
0.079954029214261757, -0.091180257555399713, -0.1442422632927951]], [[0.26155223948894291, -0.27259910943079357, 
0.74661493838533688, 0.53585345987130495], [-0.60022580287227467, -0.48790336997422057, 0.94579083650142803, 
-0.8863996264681393], [-0.48285648955978755, 0.43421403658329649, 0.12057557891417603, -0.48016460023246776]]], 
[[[0.82622400614598757, 0.89451467594773959, -0.42386713423727307, 0.4292564116088935], [0.15325746129096651, 
-0.40650587605155497, 0.48852383833911195, -0.12133478623760663], [0.79062049417854996, -0.18313225818980827, 
-0.93379779144110442, -0.8652842476948277]], [[-0.05398981985994844, 0.25155972652674774, 0.58065613688539863, 
0.49996739942398416], [-0.93379779144110442, 0.25704723590663892, 0.21718381354462557, -0.87988046847383972], 
[0.17224341707309865, 0.2585605917314151, -0.61262203743428256, 0.088679962718663852]]], [[[0.95856449568455826, 
-0.21525011988362075, -0.87880681141864181, 0.74102075636768272], [0.41959007487547573, 0.80453495515335471, 
0.48817109722463581, -0.11806636970967643], [0.90075904019500075, 0.47717994916605289, 0.35995395169862054, 
0.40418049684201307]], [[0.28919635447948355, -0.28199908853065292, -0.051524269447178916, 0.39147412124693748], 
[-0.21236681046692141, -0.35585295155544783, -0.4325141024321022, 0.50366801462727206], [-0.69662947289754285, 
-0.10168687807009102, -0.74795243165887348, -0.26529343675798978]]]])+(1.-msk_ref)*numpy.array([[[[0.11857521935136139, 
0.093929496673029522, -0.79847776993817776, 0.66859958438889944], [-0.71414593797980697, 0.11258374469683963, 
0.94835492072493288, -0.68730039028716172], [-0.69653609424045704, 0.079954029214261757, -0.091180257555399713, 
-0.1442422632927951]], [[0.26155223948894291, -0.27259910943079357, 0.74661493838533688, 0.53585345987130495], 
[-0.60022580287227467, -0.48790336997422057, 0.94579083650142803, -0.79847776993817776], [-0.48285648955978755, 
0.43421403658329649, 0.12057557891417603, -0.48016460023246776]]], [[[0.82622400614598757, 0.89451467594773959, 
-0.42386713423727307, 0.4292564116088935], [0.15325746129096651, -0.40650587605155497, 0.48852383833911195, 
-0.12133478623760663], [0.79062049417854996, -0.18313225818980827, -0.79847776993817776, -0.79847776993817776]], 
[[-0.05398981985994844, 0.25155972652674774, 0.58065613688539863, 0.49996739942398416], [-0.79847776993817776, 
0.25704723590663892, 0.21718381354462557, -0.79847776993817776], [0.17224341707309865, 0.2585605917314151, 
-0.61262203743428256, 0.088679962718663852]]], [[[0.95856449568455826, -0.21525011988362075, -0.79847776993817776, 
0.74102075636768272], [0.41959007487547573, 0.80453495515335471, 0.48817109722463581, -0.11806636970967643], 
[0.90075904019500075, 0.47717994916605289, 0.35995395169862054, 0.40418049684201307]], [[0.28919635447948355, 
-0.28199908853065292, -0.051524269447178916, 0.39147412124693748], [-0.21236681046692141, -0.35585295155544783, 
-0.4325141024321022, 0.50366801462727206], [-0.69662947289754285, -0.10168687807009102, -0.74795243165887348, 
-0.26529343675798978]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_array_rank1(self):
      arg0=numpy.array(0.174025779864)
      arg1=numpy.array([-0.80468496847716819, -0.55702002599440315])
      res=maximum(arg0,arg1)
      ref=numpy.array([0.17402577986423351, 0.17402577986423351])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank1_array_rank1(self):
      arg0=numpy.array([0.40031346790971001, 0.42660221126313203])
      arg1=numpy.array([-0.60356690978921446, 0.38041619399718063])
      res=maximum(arg0,arg1)
      ref=numpy.array([0.40031346790971001, 0.42660221126313203])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_constData_rank1(self):
      arg0=numpy.array(-0.232441816441)
      arg1=Data(numpy.array([0.7042628437324705, -0.052385211928024367]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.7042628437324705, -0.052385211928024367]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank1_constData_rank1(self):
      arg0=numpy.array([0.43161703003093366, -0.69479066934103839])
      arg1=Data(numpy.array([0.0397394354531031, 0.99784298068228838]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.43161703003093366, 0.99784298068228838]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_expandedData_rank1(self):
      arg0=numpy.array(0.531197916018)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.10695530878160886, -0.091570793259940553])+(1.-msk_arg1)*numpy.array([0.19175738089028238, 
-0.89275588159527008])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.53119791601751842, 0.53119791601751842])+(1.-msk_ref)*numpy.array([0.53119791601751842, 
0.53119791601751842])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank1_expandedData_rank1(self):
      arg0=numpy.array([0.16613419849800959, 0.79547150938095146])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-0.76819288363166582, 0.064091176986368348])+(1.-msk_arg1)*numpy.array([-0.37616391703349095, 
-0.52746213013335952])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.16613419849800959, 0.79547150938095146])+(1.-msk_ref)*numpy.array([0.16613419849800959, 
0.79547150938095146])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_array_rank2(self):
      arg0=numpy.array(0.386947484668)
      arg1=numpy.array([[0.87213326940203939, -0.38189636586633791, -0.72473189333280996, -0.21432483890478471, 
0.88028312911208029], [0.2064306399504563, 0.30698465546256481, -0.43583358859083776, -0.70245985617074291, 
-0.31221616245670569], [0.33731000862177352, -0.22831708789872907, -0.776015520692938, 0.90598610188778328, 
-0.45012002694087161], [-0.97538226851945575, 0.79531267578818832, -0.52616844351177017, 0.30697695091792299, 
-0.82183015670659132]])
      res=maximum(arg0,arg1)
      ref=numpy.array([[0.87213326940203939, 0.38694748466821971, 0.38694748466821971, 0.38694748466821971, 
0.88028312911208029], [0.38694748466821971, 0.38694748466821971, 0.38694748466821971, 0.38694748466821971, 
0.38694748466821971], [0.38694748466821971, 0.38694748466821971, 0.38694748466821971, 0.90598610188778328, 
0.38694748466821971], [0.38694748466821971, 0.79531267578818832, 0.38694748466821971, 0.38694748466821971, 
0.38694748466821971]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank2_array_rank2(self):
      arg0=numpy.array([[-0.83417152417968299, 0.83688796349341943, -0.27671759181162559, -0.67424349443288167, 
-0.42278111327171874], [0.99320754959522506, -0.82405717801872003, -0.76061490742008808, 0.5386983552951472, 
-0.90805452020755628], [-0.54835914387487272, -0.87502977528875259, 0.6410277115607339, 0.33439192634626114, 
0.053433089463401418], [0.32291094963350808, -0.30495569379510146, 0.5766949489189841, 0.12613166199354442, 
-0.36970324165010404]])
      arg1=numpy.array([[0.71480188355203733, -0.80922817795802859, -0.19167944208480336, -0.40332523244781227, 
0.40664695590425337], [-0.24805869736740882, 0.76973514026691303, 0.080936864295739319, 0.95644003803365796, 
-0.62680571189350576], [0.094574552319768079, 0.45665785511001311, -0.40597207626956666, 0.74668608783219925, 
-0.24298652271704668], [0.51320676722505087, 0.46491077097585576, -0.12305996770430627, -0.35470356431074013, 
-0.8913618557905405]])
      res=maximum(arg0,arg1)
      ref=numpy.array([[0.71480188355203733, 0.83688796349341943, -0.19167944208480336, -0.40332523244781227, 
0.40664695590425337], [0.99320754959522506, 0.76973514026691303, 0.080936864295739319, 0.95644003803365796, 
-0.62680571189350576], [0.094574552319768079, 0.45665785511001311, 0.6410277115607339, 0.74668608783219925, 
0.053433089463401418], [0.51320676722505087, 0.46491077097585576, 0.5766949489189841, 0.12613166199354442, 
-0.36970324165010404]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_constData_rank2(self):
      arg0=numpy.array(-0.304885788971)
      arg1=Data(numpy.array([[-0.22172548151150773, -0.93904306211217503, 0.25957365759321482, 0.96193812818412838, 
-0.3025989293684388], [0.15322972106506172, -0.17624868576712505, -0.96966588216584459, -0.34755630024821937, 
-0.80746601605652324], [-0.29018139429836909, 0.43809370308475382, 0.11270049169675955, -0.082055163241442131, 
0.54761620786381182], [0.48410554687630114, 0.36722666263652659, -0.013480413011818015, 0.05569938806142205, 
0.87738343744750513]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[-0.22172548151150773, -0.30488578897068686, 0.25957365759321482, 0.96193812818412838, 
-0.3025989293684388], [0.15322972106506172, -0.17624868576712505, -0.30488578897068686, -0.30488578897068686, 
-0.30488578897068686], [-0.29018139429836909, 0.43809370308475382, 0.11270049169675955, -0.082055163241442131, 
0.54761620786381182], [0.48410554687630114, 0.36722666263652659, -0.013480413011818015, 0.05569938806142205, 
0.87738343744750513]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank2_constData_rank2(self):
      arg0=numpy.array([[-0.076997964172731859, -0.67651472726038619, 0.79667022354874106, -0.84019816840575978, 
0.94360680273325759], [0.47038093139447756, -0.59450503868551907, -0.023296743257425589, -0.34991504516580862, 
0.10421432583738932], [0.66486132511251816, -0.83990036625902253, -0.4366608776718115, 0.13311527467714845, 
0.3382164722060923], [0.52398650977835115, -0.70218769667138736, 0.062518838539082466, 0.40994775955849438, 
-0.058568911794933065]])
      arg1=Data(numpy.array([[0.40801392810258319, 0.55970232701195477, 0.53761617047533083, -0.97909149343422142, 
0.98483751837359579], [0.17961521617799492, -0.22784069507161275, 0.1643289876635754, -0.37631266069934499, 
0.11329288954069172], [0.14297194831055426, 0.17614343920854547, -0.22505241632107609, -0.40785463004531297, 
-0.3909325837682216], [-0.047415577352520666, 0.92663082365757532, 0.078178606632032466, 0.91281457271967104, 
0.81139969205422791]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.40801392810258319, 0.55970232701195477, 0.79667022354874106, -0.84019816840575978, 
0.98483751837359579], [0.47038093139447756, -0.22784069507161275, 0.1643289876635754, -0.34991504516580862, 
0.11329288954069172], [0.66486132511251816, 0.17614343920854547, -0.22505241632107609, 0.13311527467714845, 
0.3382164722060923], [0.52398650977835115, 0.92663082365757532, 0.078178606632032466, 0.91281457271967104, 
0.81139969205422791]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_expandedData_rank2(self):
      arg0=numpy.array(-0.8135239605)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.61388846009494302, 0.28085229667463008, 0.96966170418990538, 0.4756731086027064, 
0.089044479963028111], [-0.22360355015236388, 0.6233791983511312, 0.68326777995318166, 0.54220296868399953, 
-0.75022793841884283], [-0.35635379209712803, -0.23282610888785338, 0.96092420087535979, -0.42869505675150421, 
0.65750802195711344], [-0.42871508206319087, -0.56475764881159818, -0.96280950832962331, 0.40149040331234409, 
-0.26763005910436588]])+(1.-msk_arg1)*numpy.array([[0.39944561956393798, -0.43585902666808818, -0.58310182711372915, 
0.0015142487710770869, 0.46901395376714472], [-0.87900855153850288, 0.38639011731442729, -0.95185702388174454, 
0.53002944784207284, -0.73959003057720962], [0.53242706460536771, -0.50214820452948183, -0.54088517606287101, 
-0.68854716037107155, 0.39914930452741304], [0.017706305731783445, -0.37994166987594036, -0.78515827077432565, 
0.61126257987463384, 0.453222031968322]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.61388846009494302, 0.28085229667463008, 0.96966170418990538, 0.4756731086027064, 
0.089044479963028111], [-0.22360355015236388, 0.6233791983511312, 0.68326777995318166, 0.54220296868399953, 
-0.75022793841884283], [-0.35635379209712803, -0.23282610888785338, 0.96092420087535979, -0.42869505675150421, 
0.65750802195711344], [-0.42871508206319087, -0.56475764881159818, -0.81352396050002107, 0.40149040331234409, 
-0.26763005910436588]])+(1.-msk_ref)*numpy.array([[0.39944561956393798, -0.43585902666808818, -0.58310182711372915, 
0.0015142487710770869, 0.46901395376714472], [-0.81352396050002107, 0.38639011731442729, -0.81352396050002107, 
0.53002944784207284, -0.73959003057720962], [0.53242706460536771, -0.50214820452948183, -0.54088517606287101, 
-0.68854716037107155, 0.39914930452741304], [0.017706305731783445, -0.37994166987594036, -0.78515827077432565, 
0.61126257987463384, 0.453222031968322]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank2_expandedData_rank2(self):
      arg0=numpy.array([[-0.78653766932808744, -0.079549862437257701, -0.97814835106959053, 0.040719482339229174, 
-0.97962562375604301], [-0.40079581080394244, -0.09401787400557815, 0.62166778688084201, -0.75199001510197849, 
0.20816228185381469], [0.0095398820105889737, -0.68544916461350902, 0.60254682946002269, -0.5420118338827502, 
0.088366583114217079], [-0.14450048156635842, 0.54026628193149651, -0.61281491161900647, 0.94473957477532911, 
0.041380925722077633]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.5466488143349244, -0.99031514955758526, -0.0376062612628143, -0.96730469425974897, 
0.35607274289209556], [-0.8471312962686266, -0.54896276227621144, -0.63471248523310386, -0.24615168151838396, 
-0.25723265379342197], [0.049145120793863128, 0.41925950625254549, 0.053557161047377511, -0.3343699299117846, 
0.084647011122490401], [0.14946012103736361, 0.80747938071502534, 0.91500285776142665, -0.39265567276306723, 
0.92295370825448098]])+(1.-msk_arg1)*numpy.array([[0.54379962366779711, -0.78061481958224044, 0.050683010869727596, 
-0.93957509919839777, -0.83910628271728838], [0.85987841725590952, -0.55992543465972155, -0.49874145776952972, 
-0.054873430290065972, -0.44738464167107428], [-0.42793766792957233, 0.13909111370769045, 0.39594342406841165, 
0.69552748643520901, -0.29658339184532245], [-0.053113369145769695, -0.88956129876827283, -0.56416742238479545, 
-0.71325747423089059, 0.21741601018126056]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.5466488143349244, -0.079549862437257701, -0.0376062612628143, 0.040719482339229174, 
0.35607274289209556], [-0.40079581080394244, -0.09401787400557815, 0.62166778688084201, -0.24615168151838396, 
0.20816228185381469], [0.049145120793863128, 0.41925950625254549, 0.60254682946002269, -0.3343699299117846, 
0.088366583114217079], [0.14946012103736361, 0.80747938071502534, 0.91500285776142665, 0.94473957477532911, 
0.92295370825448098]])+(1.-msk_ref)*numpy.array([[0.54379962366779711, -0.079549862437257701, 0.050683010869727596, 
0.040719482339229174, -0.83910628271728838], [0.85987841725590952, -0.09401787400557815, 0.62166778688084201, 
-0.054873430290065972, 0.20816228185381469], [0.0095398820105889737, 0.13909111370769045, 0.60254682946002269, 
0.69552748643520901, 0.088366583114217079], [-0.053113369145769695, 0.54026628193149651, -0.56416742238479545, 
0.94473957477532911, 0.21741601018126056]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_array_rank3(self):
      arg0=numpy.array(0.206495354305)
      arg1=numpy.array([[[-0.28919465172476477, 0.73115741235130649], [0.66849766546201494, 0.32380315894628775]], 
[[0.34229286100657319, 0.89978338241535871], [-0.94517862805279029, -0.76245816228020713]], [[-0.33058799974889852, 
0.91202625598008713], [-0.056699425352694677, -0.96047331967606175]], [[-0.18669649213118422, 0.89626981818651652], 
[0.12998045639700662, 0.48782120229237735]], [[-0.15134137439915119, -0.88244397684457598], [-0.52188376295396921, 
-0.81844180102396336]], [[-0.94590032799150481, -0.89329016632804104], [0.96506970958692784, -0.7503920073950836]]])
      res=maximum(arg0,arg1)
      ref=numpy.array([[[0.20649535430469368, 0.73115741235130649], [0.66849766546201494, 0.32380315894628775]], 
[[0.34229286100657319, 0.89978338241535871], [0.20649535430469368, 0.20649535430469368]], [[0.20649535430469368, 
0.91202625598008713], [0.20649535430469368, 0.20649535430469368]], [[0.20649535430469368, 0.89626981818651652], 
[0.20649535430469368, 0.48782120229237735]], [[0.20649535430469368, 0.20649535430469368], [0.20649535430469368, 
0.20649535430469368]], [[0.20649535430469368, 0.20649535430469368], [0.96506970958692784, 0.20649535430469368]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank3_array_rank3(self):
      arg0=numpy.array([[[0.39235233572087225, 0.74304758249838065], [-0.66779588135394619, 0.9432836579138042]], 
[[0.49351107181892084, -0.54694467231291211], [0.99103217822194889, -0.54216696018531252]], [[-0.8175841098544685, 
0.8326990222625672], [0.055153398295502587, -0.069716551616386457]], [[0.77626178147117608, 0.11964332061737948], 
[0.63436115410214011, 0.45842546157944164]], [[0.74397226514682946, -0.77625932725898372], [0.31696360097990373, 
0.0024321261945943196]], [[-0.65200916938862652, -0.66937271743063431], [0.32954705135201956, -0.16907363126350994]]])
      arg1=numpy.array([[[-0.50482866899770862, -0.44765629013986286], [0.06780136207618459, 0.15558669698070049]], 
[[0.044218703621608846, 0.2249976385020942], [0.65123878442216898, 0.65553773857321596]], [[-0.17880284508873401, 
-0.12255713511715682], [-0.10992299072827882, 0.7685617836322296]], [[0.91958169502841858, -0.44956276558073194], 
[0.31549165750027508, -0.68010957917212078]], [[-0.8629768256097865, 0.026136425252384576], [-0.71708710148747556, 
0.30781358079440957]], [[0.41046623722164766, -0.91807050647819977], [-0.80193690343803214, 0.39991444698935408]]])
      res=maximum(arg0,arg1)
      ref=numpy.array([[[0.39235233572087225, 0.74304758249838065], [0.06780136207618459, 0.9432836579138042]], 
[[0.49351107181892084, 0.2249976385020942], [0.99103217822194889, 0.65553773857321596]], [[-0.17880284508873401, 
0.8326990222625672], [0.055153398295502587, 0.7685617836322296]], [[0.91958169502841858, 0.11964332061737948], 
[0.63436115410214011, 0.45842546157944164]], [[0.74397226514682946, 0.026136425252384576], [0.31696360097990373, 
0.30781358079440957]], [[0.41046623722164766, -0.66937271743063431], [0.32954705135201956, 0.39991444698935408]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_constData_rank3(self):
      arg0=numpy.array(0.378728231479)
      arg1=Data(numpy.array([[[-0.67114944879927396, -0.16379812287341133], [-0.9257424893497439, -0.39645982000198798]], 
[[0.9478088084230436, -0.18879856392144134], [-0.16336027795938834, -0.77606729529959839]], [[0.1844003363969049, 
0.78778704269394706], [-0.28578747268227223, 0.64431707984038233]], [[0.68238252707515246, 0.06834327476896096], 
[0.55758618402198645, -0.27105782153341762]], [[0.83653411523399557, -0.36510045139192182], [-0.013797618027125758, 
0.11305598817590523]], [[0.9030275523114546, 0.86927059946280005], [0.79084757115115578, 
-0.51570522549785514]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.37872823147870549, 0.37872823147870549], [0.37872823147870549, 0.37872823147870549]], 
[[0.9478088084230436, 0.37872823147870549], [0.37872823147870549, 0.37872823147870549]], [[0.37872823147870549, 
0.78778704269394706], [0.37872823147870549, 0.64431707984038233]], [[0.68238252707515246, 0.37872823147870549], 
[0.55758618402198645, 0.37872823147870549]], [[0.83653411523399557, 0.37872823147870549], [0.37872823147870549, 
0.37872823147870549]], [[0.9030275523114546, 0.86927059946280005], [0.79084757115115578, 
0.37872823147870549]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank3_constData_rank3(self):
      arg0=numpy.array([[[0.18144247762408483, -0.78353539631767855], [0.4162761253699141, 0.86266675194423259]], 
[[0.012556149953617091, 0.4804386420113258], [-0.76805986046045316, 0.14356514697520395]], [[0.49194772346347282, 
-0.092267796433379345], [-0.26102733986309756, 0.41571439368388496]], [[0.10264134639034794, -0.080200667814620541], 
[-0.63587157924917848, 0.18913827649625636]], [[-0.36058395425965362, 0.77598983861399407], [-0.73989677505891915, 
-0.1411309114240975]], [[0.34868774774129974, 0.19384418036852469], [-0.637183036828326, -0.19633048544319909]]])
      arg1=Data(numpy.array([[[0.83338557285258363, 0.075765655696657497], [-0.034150038711153075, 0.48867409214454072]], 
[[-0.016093860376023894, 0.2815160832699517], [-0.61553678302662829, -0.14883913383260139]], [[-0.94569473686322292, 
0.50994618929453117], [-0.42494790481694533, -0.1667068481277989]], [[0.19933276666126143, 0.57718400900736966], 
[-0.51588024821082401, -0.65798936279151676]], [[-0.90857839351892578, 0.80942308915069616], [-0.75446107420460651, 
0.50588344731088331]], [[0.38538650065235491, 0.81707429971303891], [-0.97815096479177388, 
-0.71051070913370973]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.83338557285258363, 0.075765655696657497], [0.4162761253699141, 0.86266675194423259]], 
[[0.012556149953617091, 0.4804386420113258], [-0.61553678302662829, 0.14356514697520395]], [[0.49194772346347282, 
0.50994618929453117], [-0.26102733986309756, 0.41571439368388496]], [[0.19933276666126143, 0.57718400900736966], 
[-0.51588024821082401, 0.18913827649625636]], [[-0.36058395425965362, 0.80942308915069616], [-0.73989677505891915, 
0.50588344731088331]], [[0.38538650065235491, 0.81707429971303891], [-0.637183036828326, 
-0.19633048544319909]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_expandedData_rank3(self):
      arg0=numpy.array(-0.609061899567)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-0.02221934064502995, -0.24823677577161973], [0.21094272664364166, -0.55842446834946635]], 
[[0.42482246714205973, 0.06867811566792148], [-0.09860813606919594, -0.38907891589509846]], [[0.040756602755701365, 
-0.56155860925355627], [-0.63505984155662887, 0.1755095121350585]], [[-0.66997146937465057, 0.29725734526569592], 
[-0.2396248349932053, -0.32831714430302616]], [[0.4825884611716833, 0.52633576472674903], [0.21070943365647499, 
0.27856846579068084]], [[-0.0073100047866563322, -0.25318043364058052], [-0.34499905875957904, 
-0.1738541124162174]]])+(1.-msk_arg1)*numpy.array([[[-0.54275150799677596, 0.048651829704920102], [0.50592847793838391, 
-0.6263705525173533]], [[0.67364333189053371, 0.56674993507685767], [0.65430400632474917, 0.80805941287995409]], 
[[0.064865726857806116, 0.56140841973052491], [-0.22773339570038464, 0.11192902757904855]], [[0.39747464964694235, 
-0.071536903509030747], [0.0085715500683487544, -0.69734624746017571]], [[-0.39876917998500661, 0.63390636634158959], 
[-0.39861821315858226, -0.35254308646570043]], [[0.14082903951981685, -0.99563200824916698], [0.086951946864466167, 
0.57962945075783878]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.02221934064502995, -0.24823677577161973], [0.21094272664364166, -0.55842446834946635]], 
[[0.42482246714205973, 0.06867811566792148], [-0.09860813606919594, -0.38907891589509846]], [[0.040756602755701365, 
-0.56155860925355627], [-0.60906189956736645, 0.1755095121350585]], [[-0.60906189956736645, 0.29725734526569592], 
[-0.2396248349932053, -0.32831714430302616]], [[0.4825884611716833, 0.52633576472674903], [0.21070943365647499, 
0.27856846579068084]], [[-0.0073100047866563322, -0.25318043364058052], [-0.34499905875957904, 
-0.1738541124162174]]])+(1.-msk_ref)*numpy.array([[[-0.54275150799677596, 0.048651829704920102], [0.50592847793838391, 
-0.60906189956736645]], [[0.67364333189053371, 0.56674993507685767], [0.65430400632474917, 0.80805941287995409]], 
[[0.064865726857806116, 0.56140841973052491], [-0.22773339570038464, 0.11192902757904855]], [[0.39747464964694235, 
-0.071536903509030747], [0.0085715500683487544, -0.60906189956736645]], [[-0.39876917998500661, 0.63390636634158959], 
[-0.39861821315858226, -0.35254308646570043]], [[0.14082903951981685, -0.60906189956736645], [0.086951946864466167, 
0.57962945075783878]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank3_expandedData_rank3(self):
      arg0=numpy.array([[[0.38841144894036916, 0.92715868690960357], [-0.77447087138717952, 0.36220771166763188]], 
[[-0.16230793101547336, 0.49331094494024885], [-0.46865610524207124, -0.034121204544792283]], [[0.60386974342368327, 
0.2626947818589247], [0.93855252670546929, 0.3634155608194618]], [[0.57612389585726764, 0.92358799548671633], 
[-0.299282269070823, 0.63505715881940072]], [[-0.42338854353484456, -0.42092239793600861], [0.95171309029484008, 
0.66415309416591728]], [[-0.54765368059109276, 0.23554498111650468], [0.76618949130819791, -0.3948153453377552]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.9266725179072941, 0.53090206757564506], [-0.59713986747580439, 0.076267888409365936]], 
[[0.22324371017712408, 0.87148261272246397], [-0.67623763534981807, 0.52405697820419905]], [[-0.097525537051566191, 
0.89706368891826105], [-0.3810691076053736, 0.013402757258889952]], [[0.5113244637433414, -0.55238850920540195], 
[-0.99749026942641228, 0.67337322595687987]], [[-0.7550449944751183, -0.80888632434803931], [0.5666662381666081, 
0.79044507419751109]], [[0.54280750281353374, -0.52462690319031169], [-0.75601693543054327, 
-0.63628825071666228]]])+(1.-msk_arg1)*numpy.array([[[0.33073885193792507, 0.46494205056802484], [-0.0093783968795420058, 
-0.99272021120250731]], [[-0.50160691421559145, 0.7830226406524603], [0.7158546638991472, 0.10681849654155018]], 
[[-0.65696144464899331, 0.25217540916152914], [-0.93176795276651858, 0.2468866135972112]], [[0.63556079393859766, 
0.91078215014392927], [0.11430987826908967, 0.4340935122722549]], [[0.50475199331880161, 0.74420618264664129], 
[-0.068579135846449146, 0.7950673667181396]], [[0.41364534013008858, -0.49894363315644497], [-0.46943702047349745, 
0.72811686317254676]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.9266725179072941, 0.92715868690960357], [-0.59713986747580439, 0.36220771166763188]], 
[[0.22324371017712408, 0.87148261272246397], [-0.46865610524207124, 0.52405697820419905]], [[0.60386974342368327, 
0.89706368891826105], [0.93855252670546929, 0.3634155608194618]], [[0.57612389585726764, 0.92358799548671633], 
[-0.299282269070823, 0.67337322595687987]], [[-0.42338854353484456, -0.42092239793600861], [0.95171309029484008, 
0.79044507419751109]], [[0.54280750281353374, 0.23554498111650468], [0.76618949130819791, 
-0.3948153453377552]]])+(1.-msk_ref)*numpy.array([[[0.38841144894036916, 0.92715868690960357], [-0.0093783968795420058, 
0.36220771166763188]], [[-0.16230793101547336, 0.7830226406524603], [0.7158546638991472, 0.10681849654155018]], 
[[0.60386974342368327, 0.2626947818589247], [0.93855252670546929, 0.3634155608194618]], [[0.63556079393859766, 
0.92358799548671633], [0.11430987826908967, 0.63505715881940072]], [[0.50475199331880161, 0.74420618264664129], 
[0.95171309029484008, 0.7950673667181396]], [[0.41364534013008858, 0.23554498111650468], [0.76618949130819791, 
0.72811686317254676]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_array_rank4(self):
      arg0=numpy.array(0.129171289856)
      arg1=numpy.array([[[[0.3673768442593579, 0.11571490160281228, -0.64517552668077371, 0.21138864531232571], 
[-0.17300389051526821, 0.38561202294913177, -0.7799425045145012, 0.91789518329451791], [0.44295986488382333, 
-0.56752817407673462, -0.67717388658505007, -0.13952018204273786]], [[-0.76623182406287338, 0.85573861149727271, 
-0.95164332684139752, 0.86720827147905721], [0.74534759751714708, 0.50799031619007984, -0.41362920053940533, 
-0.36619696053715045], [0.6430492090475719, 0.082871936976270044, -0.68180628674605592, -0.92951630862957102]]], 
[[[0.29463597351945991, -0.58173079797975968, -0.7230223183212805, 0.61668827498944934], [0.57368054758243181, 
0.3467677103973561, -0.86059141654013738, -0.14937151897206813], [-0.75121283127075067, -0.69342076558479571, 
-0.29776459669994493, -0.29380111352128369]], [[-0.61904618645811826, 0.037206696855934807, -0.30720321824196484, 
-0.42828827933323077], [0.94459053163673512, 0.75017970241763399, -0.024543351104895805, 0.7317992761434462], 
[-0.68772557726122208, -0.26921105423467973, -0.33303964038314393, 0.88689290884788208]]], [[[0.19931943871727253, 
0.50620323919947507, 0.026887179304865505, 0.9726645366564568], [0.39022216572404478, 0.20642036994395507, 0.58810510782489067, 
0.022525598659116186], [0.29380513762421545, -0.51206092926883473, 0.23740794951394628, 0.83348785598584829]], 
[[0.82449928560496843, -0.88047679800160905, 0.0437070552148493, -0.33527652724059176], [-0.15159772508509795, 
-0.45358363740648411, -0.51667738655965545, 0.5776497411681254], [0.9659629370206908, -0.41135044415686139, 
-0.37504198978467462, -0.36072733216551578]]]])
      res=maximum(arg0,arg1)
      ref=numpy.array([[[[0.3673768442593579, 0.1291712898560109, 0.1291712898560109, 0.21138864531232571], 
[0.1291712898560109, 0.38561202294913177, 0.1291712898560109, 0.91789518329451791], [0.44295986488382333, 0.1291712898560109, 
0.1291712898560109, 0.1291712898560109]], [[0.1291712898560109, 0.85573861149727271, 0.1291712898560109, 0.86720827147905721], 
[0.74534759751714708, 0.50799031619007984, 0.1291712898560109, 0.1291712898560109], [0.6430492090475719, 0.1291712898560109, 
0.1291712898560109, 0.1291712898560109]]], [[[0.29463597351945991, 0.1291712898560109, 0.1291712898560109, 
0.61668827498944934], [0.57368054758243181, 0.3467677103973561, 0.1291712898560109, 0.1291712898560109], [0.1291712898560109, 
0.1291712898560109, 0.1291712898560109, 0.1291712898560109]], [[0.1291712898560109, 0.1291712898560109, 0.1291712898560109, 
0.1291712898560109], [0.94459053163673512, 0.75017970241763399, 0.1291712898560109, 0.7317992761434462], [0.1291712898560109, 
0.1291712898560109, 0.1291712898560109, 0.88689290884788208]]], [[[0.19931943871727253, 0.50620323919947507, 
0.1291712898560109, 0.9726645366564568], [0.39022216572404478, 0.20642036994395507, 0.58810510782489067, 0.1291712898560109], 
[0.29380513762421545, 0.1291712898560109, 0.23740794951394628, 0.83348785598584829]], [[0.82449928560496843, 
0.1291712898560109, 0.1291712898560109, 0.1291712898560109], [0.1291712898560109, 0.1291712898560109, 0.1291712898560109, 
0.5776497411681254], [0.9659629370206908, 0.1291712898560109, 0.1291712898560109, 0.1291712898560109]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank4_array_rank4(self):
      arg0=numpy.array([[[[-0.65015954456679181, 0.40240408318498599, 0.84464712425699462, -0.43969720683132807], 
[0.060269909487150608, -0.74687444707146433, -0.6437215821030513, 0.50694044142420136], [-0.95651303413075461, 
0.30077769940421639, -0.88469041671271231, 0.35428303244641701]], [[-0.64479714996431903, -0.02806358507098361, 
0.17146951356917417, 0.17856182601303572], [0.82338647213827665, 0.62775989519430597, 0.98131295897678883, 
-0.27925295851008336], [-0.077888587859346714, 0.39299547141403646, -0.76703658182228662, 0.69803559132588267]]], 
[[[-0.60438450369130425, 0.50974341740755325, 0.2628982550897998, 0.62253269815816936], [0.098454268427629721, 
-0.11354020704871637, 0.97520314368136374, 0.82424110975204568], [0.15014068269747205, -0.42103180617954483, 
0.90299092501087008, -0.23139020793721432]], [[0.76941820515528692, -0.1765839158818503, -0.5913087365745302, 
0.7161045583038077], [0.85854815816198493, 0.024439202303835561, -0.75033601915149672, 0.47128628281539831], 
[-0.41886779770164062, -0.18310623967189343, -0.6230995563003221, 0.9482826576576675]]], [[[0.094084288889770606, 
-0.1009961699861317, -0.24821425388692853, -0.70296524988401554], [0.67010687729768903, 0.57353374648941413, 
0.81190592473146639, 0.99708169066638397], [-0.37503932645378679, -0.24423377176761285, -0.18549806495578736, 
-0.47985036549736715]], [[0.62433786823811555, 0.22620666330362238, -0.18049352036474309, 0.99051652854626893], 
[-0.3286392033550134, 0.83921438988823871, 0.68465908457587843, 0.44769792179940149], [0.11216755461815131, 
0.21844427393884125, -0.42328115351224671, -0.85880777265173469]]]])
      arg1=numpy.array([[[[-0.73093615882935814, -0.70919589053874077, 0.78870394532867283, -0.50159810974536545], 
[-0.85519344422155075, -0.56155628934300195, -0.62373278444802516, -0.9772447843857226], [0.87755710000772535, 
-0.013588596244209494, -0.90114581352274459, 0.55410820192830745]], [[-0.72956039379052307, -0.35348269136869503, 
-0.42184790113450843, -0.89870920570023705], [-0.14489448657028414, -0.91755892231285974, 0.89607768435234414, 
0.4058334012935787], [-0.55720951018176401, 0.90451728855028346, -0.54876588909958501, 0.70078916216098674]]], 
[[[0.84626531438079744, 0.45264260263683553, 0.12965000086402689, -0.021003962192429393], [-0.27005492904656037, 
0.55590537183241318, -0.24978960230861946, 0.92321598749374068], [0.016189016341156437, -0.35579544797087137, 
0.62297910324803074, 0.10682842803458792]], [[-0.5264656932595686, 0.044463460549419409, -0.73796856857767978, 
0.16294987868437039], [0.78510668328973954, 0.756487852598553, 0.44081375563129299, -0.60856363687850457], 
[0.60854054966272164, 0.83451336734693471, 0.71676784780975211, 0.7477674423454328]]], [[[0.94511009490633557, 
-0.45500546126530228, -0.27748406592529928, -0.11960260520462684], [0.83727559558880982, 0.6973251313608686, 
0.43086624118575867, 0.0099152218565310424], [0.18361709779587665, 0.53065002315173349, 0.45664732903855709, 
0.71620430423422099]], [[-0.60506520502917138, 0.69610912235182143, -0.10835353752278154, -0.63873781536516083], 
[-0.80308848611516837, 0.54896014232641921, 0.48999568478728261, 0.075051256341975181], [0.00034354126399538387, 
0.14982509954067424, 0.33039570291905651, -0.95215296115442083]]]])
      res=maximum(arg0,arg1)
      ref=numpy.array([[[[-0.65015954456679181, 0.40240408318498599, 0.84464712425699462, -0.43969720683132807], 
[0.060269909487150608, -0.56155628934300195, -0.62373278444802516, 0.50694044142420136], [0.87755710000772535, 
0.30077769940421639, -0.88469041671271231, 0.55410820192830745]], [[-0.64479714996431903, -0.02806358507098361, 
0.17146951356917417, 0.17856182601303572], [0.82338647213827665, 0.62775989519430597, 0.98131295897678883, 0.4058334012935787], 
[-0.077888587859346714, 0.90451728855028346, -0.54876588909958501, 0.70078916216098674]]], [[[0.84626531438079744, 
0.50974341740755325, 0.2628982550897998, 0.62253269815816936], [0.098454268427629721, 0.55590537183241318, 0.97520314368136374, 
0.92321598749374068], [0.15014068269747205, -0.35579544797087137, 0.90299092501087008, 0.10682842803458792]], 
[[0.76941820515528692, 0.044463460549419409, -0.5913087365745302, 0.7161045583038077], [0.85854815816198493, 0.756487852598553, 
0.44081375563129299, 0.47128628281539831], [0.60854054966272164, 0.83451336734693471, 0.71676784780975211, 
0.9482826576576675]]], [[[0.94511009490633557, -0.1009961699861317, -0.24821425388692853, -0.11960260520462684], 
[0.83727559558880982, 0.6973251313608686, 0.81190592473146639, 0.99708169066638397], [0.18361709779587665, 0.53065002315173349, 
0.45664732903855709, 0.71620430423422099]], [[0.62433786823811555, 0.69610912235182143, -0.10835353752278154, 
0.99051652854626893], [-0.3286392033550134, 0.83921438988823871, 0.68465908457587843, 0.44769792179940149], 
[0.11216755461815131, 0.21844427393884125, 0.33039570291905651, -0.85880777265173469]]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_constData_rank4(self):
      arg0=numpy.array(-0.943242637062)
      arg1=Data(numpy.array([[[[0.46673989197448695, 0.6549649376118345, 0.9984111355226466, -0.70351419152946271], 
[0.71307312699624026, 0.5419548297460679, -0.17409109440608583, -0.97014228996198115], [-0.54280492332372599, 
0.86380903975785861, -0.34701233365455697, 0.48567075553942263]], [[0.54700266237459316, -0.35248705368749378, 
0.23377808128753208, 0.96721598543765519], [0.88430869613619123, -0.1219865051849538, -0.058637092978300664, 
-0.61754559413262333], [0.74079280717308227, -0.78739161180686512, 0.85037530898787361, 0.16508890280036592]]], 
[[[0.44817727199203761, 0.8926307678886285, -0.76111914551638038, 0.14362232256439045], [-0.48467388532021149, 
-0.8094428609146751, 0.53535210524492927, -0.56637649678780466], [0.98969180896200037, -0.3271162647098742, 
0.69094018879758989, -0.61586242247168044]], [[-0.36044332609306107, -0.98997760997627204, 0.90220389283531444, 
-0.60156374176429561], [0.40405941425843839, 0.4234467099813175, -0.95866754182894565, 0.8344078032898532], 
[-0.48131027576552698, -0.59963997992966767, -0.83658860482593256, -0.75913735933586257]]], [[[0.88365651759965269, 
0.89012413369555121, -0.4898888357384561, 0.67718373616908689], [0.32819567260296134, -0.89006474342304842, 
-0.58497923437075872, -0.025361716234509002], [-0.27583713562824563, -0.1438801449075906, 0.75117793183554871, 
-0.44679974659724797]], [[0.71819263148139534, -0.73238371283246728, -0.92822578406489709, -0.13704547005331813], 
[0.13259671454727084, -0.45302324436441288, -0.28335401748666711, 0.32261970225996106], [0.90169925985201749, 
0.89781229478225999, -0.54573322813790259, -0.22215533008165655]]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.46673989197448695, 0.6549649376118345, 0.9984111355226466, -0.70351419152946271], 
[0.71307312699624026, 0.5419548297460679, -0.17409109440608583, -0.94324263706204037], [-0.54280492332372599, 
0.86380903975785861, -0.34701233365455697, 0.48567075553942263]], [[0.54700266237459316, -0.35248705368749378, 
0.23377808128753208, 0.96721598543765519], [0.88430869613619123, -0.1219865051849538, -0.058637092978300664, 
-0.61754559413262333], [0.74079280717308227, -0.78739161180686512, 0.85037530898787361, 0.16508890280036592]]], 
[[[0.44817727199203761, 0.8926307678886285, -0.76111914551638038, 0.14362232256439045], [-0.48467388532021149, 
-0.8094428609146751, 0.53535210524492927, -0.56637649678780466], [0.98969180896200037, -0.3271162647098742, 
0.69094018879758989, -0.61586242247168044]], [[-0.36044332609306107, -0.94324263706204037, 0.90220389283531444, 
-0.60156374176429561], [0.40405941425843839, 0.4234467099813175, -0.94324263706204037, 0.8344078032898532], 
[-0.48131027576552698, -0.59963997992966767, -0.83658860482593256, -0.75913735933586257]]], [[[0.88365651759965269, 
0.89012413369555121, -0.4898888357384561, 0.67718373616908689], [0.32819567260296134, -0.89006474342304842, 
-0.58497923437075872, -0.025361716234509002], [-0.27583713562824563, -0.1438801449075906, 0.75117793183554871, 
-0.44679974659724797]], [[0.71819263148139534, -0.73238371283246728, -0.92822578406489709, -0.13704547005331813], 
[0.13259671454727084, -0.45302324436441288, -0.28335401748666711, 0.32261970225996106], [0.90169925985201749, 
0.89781229478225999, -0.54573322813790259, -0.22215533008165655]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank4_constData_rank4(self):
      arg0=numpy.array([[[[0.6710452933322606, 0.73655023524342855, -0.96637183491308676, 0.47953689956820522], 
[0.98053583083570528, 0.1461687898181756, 0.012703818765450459, 0.48064837271561767], [0.52408663541340594, 
-0.6101131637614976, 0.71951147562469586, 0.97773169491505763]], [[0.85269224639001395, -0.64939328460945966, 
-0.40530815940692455, -0.043873643895695258], [0.013474207600136179, -0.83199072645059813, -0.24875392233750837, 
-0.14945963740023349], [-0.19859755265586743, -0.83519278737823521, -0.16490322565926241, 0.19988995509904828]]], 
[[[0.54261691324742345, -0.83216349466983819, -0.82229431953325571, -0.95593602291977064], [-0.038265980497722341, 
0.38578299235089331, -0.43729585014806571, -0.94893987277299208], [-0.27213639583191407, 0.1045917845853499, 
0.0051298010833620289, -0.87753324258171328]], [[0.47867910844244244, -0.7935953174942354, 0.48633491846960841, 
-0.16964491089477818], [-0.73725496052678019, -0.65544040345490262, 0.82473847295242608, -0.86310822408155241], 
[-0.22876295798043222, 0.087254884598592541, -0.66897239866347769, -0.88155102312679112]]], [[[0.80518718751035201, 
0.9208930345840538, -0.84320779895756859, -0.22169550184009501], [0.34144334236516105, -0.65813010675475092, 
-0.67816684120052328, 0.03057411294794643], [0.83964290874070091, -0.65242174738552761, 0.65986798996685425, 
0.55354844330886732]], [[0.41987573705519776, 0.86448705917430524, 0.89887374333241499, 0.50981938714297903], 
[-0.66180921979411456, 0.59715491866752002, -0.94380294634555173, 0.61085742396555731], [-0.061851722069981907, 
-0.82539430034923944, -0.05820899628782894, -0.12313347830576449]]]])
      arg1=Data(numpy.array([[[[0.74351019640477412, -0.98281371332345957, 0.047498607093644729, 0.15927637373867798], 
[-0.56718099478511252, -0.40521930963777275, -0.15323791375707096, -0.13397472408265321], [0.32026529445003682, 
0.94984367824314786, -0.21650883898062334, 0.36878669403603381]], [[0.76460386081962728, 0.26035938179375706, 
-0.17228362074739612, -0.76176095860293325], [0.79650097864796598, -0.92064005521421444, 0.63695414023301522, 
0.29763045353273676], [0.028233556083444356, 0.65356505734241077, 0.64710795344625938, 0.47088770373999034]]], 
[[[0.052592817909343559, 0.90142884270306523, -0.55744523632845322, -0.10133937858296571], [-0.38270754560875409, 
0.84777761145662178, -0.19647228139194683, 0.2624309028567815], [-0.36151988444214433, 0.79453458152737544, 
0.035112284546496175, 0.090453586873261216]], [[-0.042959217360417634, -0.40165574292355521, 0.87853656547492331, 
-0.74381165790643089], [-0.035497537775224242, -0.26990065440269873, 0.52801706764535417, 0.44592510638569061], 
[-0.46487843397425954, 0.88572704975088068, -0.70820902545471909, -0.83719918740365951]]], [[[-0.17444138264473819, 
0.44713858293412612, 0.42308639570713846, 0.11624103502896643], [-0.53171945486995664, -0.43730113574911877, 
0.023341839389971808, -0.9174119628732611], [-0.34478808240190162, -0.64263237508691695, 0.09480577060325146, 
-0.54979877472252747]], [[-0.83762097768252852, -0.28250723065451577, -0.70302270429800839, 0.20350206223682488], 
[-0.51785671081588647, -0.99895978214481795, -0.09100830119876635, -0.10432462166690804], [-0.2140802658780443, 
-0.32749288111075736, -0.34840221305540831, -0.33613300029766058]]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.74351019640477412, 0.73655023524342855, 0.047498607093644729, 0.47953689956820522], 
[0.98053583083570528, 0.1461687898181756, 0.012703818765450459, 0.48064837271561767], [0.52408663541340594, 
0.94984367824314786, 0.71951147562469586, 0.97773169491505763]], [[0.85269224639001395, 0.26035938179375706, 
-0.17228362074739612, -0.043873643895695258], [0.79650097864796598, -0.83199072645059813, 0.63695414023301522, 
0.29763045353273676], [0.028233556083444356, 0.65356505734241077, 0.64710795344625938, 0.47088770373999034]]], 
[[[0.54261691324742345, 0.90142884270306523, -0.55744523632845322, -0.10133937858296571], [-0.038265980497722341, 
0.84777761145662178, -0.19647228139194683, 0.2624309028567815], [-0.27213639583191407, 0.79453458152737544, 
0.035112284546496175, 0.090453586873261216]], [[0.47867910844244244, -0.40165574292355521, 0.87853656547492331, 
-0.16964491089477818], [-0.035497537775224242, -0.26990065440269873, 0.82473847295242608, 0.44592510638569061], 
[-0.22876295798043222, 0.88572704975088068, -0.66897239866347769, -0.83719918740365951]]], [[[0.80518718751035201, 
0.9208930345840538, 0.42308639570713846, 0.11624103502896643], [0.34144334236516105, -0.43730113574911877, 
0.023341839389971808, 0.03057411294794643], [0.83964290874070091, -0.64263237508691695, 0.65986798996685425, 
0.55354844330886732]], [[0.41987573705519776, 0.86448705917430524, 0.89887374333241499, 0.50981938714297903], 
[-0.51785671081588647, 0.59715491866752002, -0.09100830119876635, 0.61085742396555731], [-0.061851722069981907, 
-0.32749288111075736, -0.05820899628782894, -0.12313347830576449]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank0_expandedData_rank4(self):
      arg0=numpy.array(0.816751245349)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[0.40706796204856954, -0.53751083415836831, -0.45864908744463273, -0.16181548971941506], 
[0.24911417445318973, -0.90143034728419491, 0.49231704616202521, -0.02953522449634538], [-0.54604157728790303, 
0.96956023052870233, -0.35875149576930676, -0.65119882270566221]], [[-0.63683028644751638, 0.517070292194729, 
0.88993945395848617, 0.45862552604255513], [-0.3937255872490002, 0.62928624666236899, 0.15644492847277935, 
0.39426821818667213], [0.18687916068352117, -0.90059464422537161, 0.63405011651666454, -0.91397232745661849]]], 
[[[-0.59152682775898779, -0.29881608198483178, -0.12022357576247278, -0.14636655539718824], [0.10683837535157203, 
0.4534643980627564, 0.56557067866858457, -0.0058893033942237771], [0.2438289343318536, -0.44513011643495815, 
-0.82351570680276276, -0.74628626035713075]], [[0.24049586413674917, 0.93536796522792276, 0.40179585898884218, 
0.11001812792063603], [0.24888872211708435, 0.13170738221634015, -0.23386082553392895, 0.26287786471668606], 
[-0.12817923648221208, -0.81433650068603169, -0.85580088180765967, 0.37425524564540913]]], [[[-0.0015262470861385768, 
-0.13657548471526271, -0.17763500899807361, -0.29008714552601167], [-0.29605361382858919, -0.30804118243930012, 
-0.69034132273448701, 0.17159914379615593], [-0.055083234234476253, 0.27179546429245183, 0.73841704903963601, 
0.56943419450749744]], [[-0.13221678354751099, -0.25975064422559724, 0.98055444051148144, 0.041900636495320498], 
[-0.67581584371274128, -0.044493982042192126, 0.38493910887288241, -0.053928330783549638], [0.24767507371629205, 
0.02740918244429702, -0.40707529813083276, -0.53224165365256848]]]])+(1.-msk_arg1)*numpy.array([[[[-0.74622638578077849, 
0.075132461697541508, -0.33313454924071673, -0.67257843324720423], [-0.89086147690801054, -0.37521216146728076, 
-0.3769720211792944, -0.25074902554592482], [0.03327796652289905, -0.831487975870429, -0.46430521993783014, 
0.35666688446285999]], [[-0.3002282511139418, -0.9131223259696819, -0.35725566311385237, -0.08122060507717177], 
[0.49125425698890024, 0.04278065414817589, -0.10736165776646334, -0.58357962728965052], [-0.78389654631628258, 
0.014252969571323604, -0.58349722415645489, 0.48015407778285857]]], [[[0.8629968548019411, -0.39330356017360191, 
-0.43953900522134193, -0.93850862679670422], [-0.67563892257080771, -0.30985847916271125, -0.73029188466858708, 
0.45768180905096179], [0.89312962196038459, 0.24375888513531274, 0.79974754704490114, 0.062990767856061503]], 
[[0.19388446284468785, 0.2228516825113267, 0.58972079924045051, 0.16046525287644209], [0.16738918455502771, 
-0.87793601312414005, 0.55708100133014371, 0.79422379620746852], [0.48900265257984232, 0.20469753181175121, 
-0.076793936351922776, -0.66050327071342485]]], [[[0.37333906028974995, -0.13276337459811982, 0.55257492514565332, 
-0.87250909004301769], [-0.92159637562262575, 0.035611817197426232, 0.49669980616950649, 0.81933503740201186], 
[-0.96340332935349826, 0.85175864132768342, -0.56016253718759845, 0.56499606586394324]], [[-0.33109596385421125, 
0.97878513835337344, -0.16451086770378209, 0.10562009554922169], [0.64934854776565909, 0.39252253014845206, 
0.086082586528108962, -0.87179727093305726], [-0.64734193154791497, -0.11793903610185485, -0.80594955530268586, 
-0.7145408832676321]]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574], 
[0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 
0.96956023052870233, 0.81675124534906574, 0.81675124534906574]], [[0.81675124534906574, 0.81675124534906574, 
0.88993945395848617, 0.81675124534906574], [0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 
0.81675124534906574], [0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574]]], 
[[[0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 
0.81675124534906574, 0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 
0.81675124534906574]], [[0.81675124534906574, 0.93536796522792276, 0.81675124534906574, 0.81675124534906574], 
[0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 
0.81675124534906574, 0.81675124534906574, 0.81675124534906574]]], [[[0.81675124534906574, 0.81675124534906574, 
0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 
0.81675124534906574], [0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574]], 
[[0.81675124534906574, 0.81675124534906574, 0.98055444051148144, 0.81675124534906574], [0.81675124534906574, 
0.81675124534906574, 0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 
0.81675124534906574]]]])+(1.-msk_ref)*numpy.array([[[[0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 
0.81675124534906574], [0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574], 
[0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574]], [[0.81675124534906574, 
0.81675124534906574, 0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 
0.81675124534906574], [0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574]]], 
[[[0.8629968548019411, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 
0.81675124534906574, 0.81675124534906574, 0.81675124534906574], [0.89312962196038459, 0.81675124534906574, 0.81675124534906574, 
0.81675124534906574]], [[0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574], 
[0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 
0.81675124534906574, 0.81675124534906574, 0.81675124534906574]]], [[[0.81675124534906574, 0.81675124534906574, 
0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 
0.81933503740201186], [0.81675124534906574, 0.85175864132768342, 0.81675124534906574, 0.81675124534906574]], 
[[0.81675124534906574, 0.97878513835337344, 0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 
0.81675124534906574, 0.81675124534906574, 0.81675124534906574], [0.81675124534906574, 0.81675124534906574, 0.81675124534906574, 
0.81675124534906574]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_array_rank4_expandedData_rank4(self):
      arg0=numpy.array([[[[0.42000806204653474, 0.43445211169835551, 0.67259254968505977, -0.44236725657295772], 
[0.57797003911910583, -0.56843078081208565, 0.73401747796917305, 0.2724166496453182], [-0.63816568661349526, 
0.47652891249952689, 0.22262872392584887, 0.57248495666865962]], [[0.67126178307885032, 0.55524712080063265, 
-0.57633405842088048, -0.51172125052107154], [0.64644288887632562, -0.30158732813994793, 0.066495417813510427, 
0.55087047201075623], [-0.70640833320964092, 0.79392204123529808, -0.30151094639813847, 0.63477282920213551]]], 
[[[-0.33887822678892188, -0.17257825839362329, -0.86588501606225043, 0.79230278052527692], [-0.39941731689877535, 
-0.079106652140733758, 0.34561234826387599, -0.3250977623978315], [-0.072143230372351264, -0.35332740106055804, 
0.36569821506455535, -0.40358096819303957]], [[-0.46326111929557134, -0.65734587522945453, 0.36365229912540875, 
0.29629501392799629], [0.76965355270422631, 0.86916618870287166, -0.30399370154030803, 0.052406314916716434], 
[-0.76844408403720332, -0.80603282987458247, 0.87923305663161355, -0.47575486213082696]]], [[[0.064241833879999932, 
0.75416840871509683, -0.16640916907322456, 0.51810523833466604], [-0.52612642670199117, -0.54881438921402426, 
-0.60230840326054902, -0.27590086084382826], [0.56589884837057913, 0.63901913471463057, 0.14143396068435887, 
0.4572178777562419]], [[-0.77613236188204149, -0.10681746398394076, 0.56066764451409656, -0.2652991439307546], 
[-0.47167240379595765, 0.13624117786721168, 0.90830564372286182, -0.50408842514981544], [-0.10789679373755079, 
0.89455801147160074, 0.64339672677130522, -0.52438730821821777]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[0.58994058399593396, -0.84790900163215799, -0.2810393046159938, 0.95132339628179641], 
[-0.96691100251741302, -0.27351720946556934, -0.59482024886374374, 0.067410044082623743], [-0.47797733987261082, 
-0.15881452904936122, 0.048733333513018007, -0.2097978517034238]], [[-0.022424892496763604, 0.95039754974581636, 
-0.26755918731750183, 0.11642407168693469], [0.42843959365869888, 0.87961951831310414, 0.87592903677448142, 
-0.14801753665349993], [-0.44794994006371702, 0.90952976942348851, 0.86095504356940866, 0.40092005641620676]]], 
[[[-0.87293977911397436, -0.26515845603721533, 0.51886214635551764, -0.84860714710587803], [-0.96644396225751872, 
-0.7631489360116015, 0.32411191551565888, -0.61020033012558517], [0.62767467887627681, 0.37469207688125628, 
0.97311127999171076, -0.48576153689962909]], [[0.80653641318766311, 0.7667470510008485, -0.1244805386140686, 
-0.58255916955809517], [-0.68905687561103779, 0.73624509848196351, 0.14977570034343146, 0.040371627889322026], 
[0.59631748792445949, -0.36976261889718165, -0.32387905826980279, -0.70956255815582425]]], [[[-0.47168955139676272, 
-0.62259149336107811, -0.71438890780775055, -0.70887906389104938], [-0.22646044424514478, 0.22578442905780038, 
0.82289907678613572, 0.78508276520394693], [0.66158100390938746, 0.81084818404365477, 0.41404741540900059, 
0.29086117891310748]], [[-0.48563213325230881, -0.66773681864261558, 0.13251259057679676, 0.38312110170916869], 
[0.4985513501920098, -0.054491390265769768, -0.034239376662770438, 0.15118777726900667], [0.637203043607808, 
-0.63220275381972035, -0.83486043539732502, -0.89196454166383954]]]])+(1.-msk_arg1)*numpy.array([[[[0.77783291145936362, 
0.14573546502013346, 0.16427843684067489, 0.070174330346382652], [-0.75910085997762344, -0.13628190232977233, 
0.050999095952770856, 0.11715267966221887], [0.24272200394062349, 0.5320807594862087, 0.56874924645624914, 
-0.97874608588909484]], [[-0.98167918496033835, -0.92884768363832904, 0.020720960458232351, 0.60484296418798422], 
[0.43656063194574424, 0.24417269636492955, 0.88220623197013071, 0.0051165565172857086], [0.59919244135295746, 
-0.24769803256244494, 0.46747145925858957, -0.32539081032403749]]], [[[0.36547407180069946, 0.69858798920753662, 
-0.11815619917568543, 0.7202806361724019], [0.89566175611285326, -0.56471605346156917, -0.96295426503805337, 
-0.79642127774673965], [0.0082356467622470841, 0.86729114253916273, -0.61543952427247706, 0.84048187872968816]], 
[[-0.70083513366236239, -0.63770108259677039, 0.8010185543717967, -0.171777821609852], [0.28780672248819994, 
0.77948501197327458, 0.80566125829149815, -0.48883690350583775], [-0.31426640827493268, 0.010369391352117008, 
-0.17008192331534411, -0.24571331409278607]]], [[[-0.81997880316304705, -0.65143657046299919, 0.53349064661197132, 
0.56466068808179881], [0.9458545705591126, -0.62032388429602836, -0.20671123052978779, -0.87582148391296899], 
[0.025494758959417885, -0.16261424915626144, -0.44378234236454994, -0.22992948225644549]], [[0.81002927637991107, 
-0.68316174531148488, -0.85225788243989031, -0.88222373281789257], [-0.26541667769836907, 0.17943188950419575, 
0.63512022836563475, -0.38654668208377108], [-0.31161290145238318, 0.35849193061546902, -0.032927010158966707, 
0.33048499755046112]]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.58994058399593396, 0.43445211169835551, 0.67259254968505977, 0.95132339628179641], 
[0.57797003911910583, -0.27351720946556934, 0.73401747796917305, 0.2724166496453182], [-0.47797733987261082, 
0.47652891249952689, 0.22262872392584887, 0.57248495666865962]], [[0.67126178307885032, 0.95039754974581636, 
-0.26755918731750183, 0.11642407168693469], [0.64644288887632562, 0.87961951831310414, 0.87592903677448142, 
0.55087047201075623], [-0.44794994006371702, 0.90952976942348851, 0.86095504356940866, 0.63477282920213551]]], 
[[[-0.33887822678892188, -0.17257825839362329, 0.51886214635551764, 0.79230278052527692], [-0.39941731689877535, 
-0.079106652140733758, 0.34561234826387599, -0.3250977623978315], [0.62767467887627681, 0.37469207688125628, 
0.97311127999171076, -0.40358096819303957]], [[0.80653641318766311, 0.7667470510008485, 0.36365229912540875, 
0.29629501392799629], [0.76965355270422631, 0.86916618870287166, 0.14977570034343146, 0.052406314916716434], 
[0.59631748792445949, -0.36976261889718165, 0.87923305663161355, -0.47575486213082696]]], [[[0.064241833879999932, 
0.75416840871509683, -0.16640916907322456, 0.51810523833466604], [-0.22646044424514478, 0.22578442905780038, 
0.82289907678613572, 0.78508276520394693], [0.66158100390938746, 0.81084818404365477, 0.41404741540900059, 
0.4572178777562419]], [[-0.48563213325230881, -0.10681746398394076, 0.56066764451409656, 0.38312110170916869], 
[0.4985513501920098, 0.13624117786721168, 0.90830564372286182, 0.15118777726900667], [0.637203043607808, 0.89455801147160074, 
0.64339672677130522, -0.52438730821821777]]]])+(1.-msk_ref)*numpy.array([[[[0.77783291145936362, 0.43445211169835551, 
0.67259254968505977, 0.070174330346382652], [0.57797003911910583, -0.13628190232977233, 0.73401747796917305, 
0.2724166496453182], [0.24272200394062349, 0.5320807594862087, 0.56874924645624914, 0.57248495666865962]], 
[[0.67126178307885032, 0.55524712080063265, 0.020720960458232351, 0.60484296418798422], [0.64644288887632562, 
0.24417269636492955, 0.88220623197013071, 0.55087047201075623], [0.59919244135295746, 0.79392204123529808, 0.46747145925858957, 
0.63477282920213551]]], [[[0.36547407180069946, 0.69858798920753662, -0.11815619917568543, 0.79230278052527692], 
[0.89566175611285326, -0.079106652140733758, 0.34561234826387599, -0.3250977623978315], [0.0082356467622470841, 
0.86729114253916273, 0.36569821506455535, 0.84048187872968816]], [[-0.46326111929557134, -0.63770108259677039, 
0.8010185543717967, 0.29629501392799629], [0.76965355270422631, 0.86916618870287166, 0.80566125829149815, 
0.052406314916716434], [-0.31426640827493268, 0.010369391352117008, 0.87923305663161355, -0.24571331409278607]]], 
[[[0.064241833879999932, 0.75416840871509683, 0.53349064661197132, 0.56466068808179881], [0.9458545705591126, 
-0.54881438921402426, -0.20671123052978779, -0.27590086084382826], [0.56589884837057913, 0.63901913471463057, 
0.14143396068435887, 0.4572178777562419]], [[0.81002927637991107, -0.10681746398394076, 0.56066764451409656, 
-0.2652991439307546], [-0.26541667769836907, 0.17943188950419575, 0.90830564372286182, -0.38654668208377108], 
[-0.10789679373755079, 0.89455801147160074, 0.64339672677130522, 0.33048499755046112]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_float_rank0(self):
      arg0=Data(-0.103724211678,self.functionspace)
      arg1=0.328283764361
      res=maximum(arg0,arg1)
      ref=Data(0.328283764361,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank1_float_rank0(self):
      arg0=Data(numpy.array([-0.994662543839125, -0.74842601409030385]),self.functionspace)
      arg1=0.83441404241
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.83441404241015316, 0.83441404241015316]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank2_float_rank0(self):
      arg0=Data(numpy.array([[-0.93123334581207584, -0.6375770434441641, 0.14522839584946268, -0.73025618312998342, 
-0.43938340547287202], [-0.94178326076177643, 0.37494459873539854, -0.41021077539181738, -0.20418492678279287, 
0.66837201836494731], [-0.8482840255895383, -0.37013598496106792, 0.9491945264624071, -0.84517384298961939, 
-0.98444834611902565], [-0.62650653009151203, -0.80707547020964321, -0.54928564740444141, 0.62760912325779228, 
-0.068170353902772929]]),self.functionspace)
      arg1=0.416476250509
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.41647625050863057, 0.41647625050863057, 0.41647625050863057, 0.41647625050863057, 
0.41647625050863057], [0.41647625050863057, 0.41647625050863057, 0.41647625050863057, 0.41647625050863057, 
0.66837201836494731], [0.41647625050863057, 0.41647625050863057, 0.9491945264624071, 0.41647625050863057, 0.41647625050863057], 
[0.41647625050863057, 0.41647625050863057, 0.41647625050863057, 0.62760912325779228, 0.41647625050863057]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank3_float_rank0(self):
      arg0=Data(numpy.array([[[-0.30986069585890852, -0.71233438667691962], [-0.71108485594464965, 0.46803401869143135]], 
[[0.047347209617666586, 0.40377859059535814], [-0.076869071518825383, 0.66840655201248778]], [[-0.73984483745703389, 
0.15706212028006727], [-0.75038361207139048, 0.71534375261818672]], [[-0.16981297613502533, 0.95326334909603228], 
[-0.578228141354864, 0.69445686433215403]], [[0.85239986273571344, 0.70437733963137261], [-0.64356702754466366, 
0.98497620890820303]], [[-0.73348032264680829, -0.90903872026049592], [-0.30876237687286801, 
0.85358440828017468]]]),self.functionspace)
      arg1=-0.707243668032
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[-0.30986069585890852, -0.70724366803240968], [-0.70724366803240968, 0.46803401869143135]], 
[[0.047347209617666586, 0.40377859059535814], [-0.076869071518825383, 0.66840655201248778]], [[-0.70724366803240968, 
0.15706212028006727], [-0.70724366803240968, 0.71534375261818672]], [[-0.16981297613502533, 0.95326334909603228], 
[-0.578228141354864, 0.69445686433215403]], [[0.85239986273571344, 0.70437733963137261], [-0.64356702754466366, 
0.98497620890820303]], [[-0.70724366803240968, -0.70724366803240968], [-0.30876237687286801, 
0.85358440828017468]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank4_float_rank0(self):
      arg0=Data(numpy.array([[[[-0.44038422283869894, -0.99307493506995592, -0.64116938861589978, -0.52885072305198189], 
[-0.60664143188537656, -0.53692428608899201, 0.9427489146021919, -0.66511035190756762], [-0.95640611039066736, 
0.98227091107399023, -0.35080572865502968, -0.062671325929018584]], [[-0.74010294573905111, -0.60760114920218666, 
0.45863579037285751, -0.45688569472448681], [-0.33875049551517833, 0.35019781316828325, 0.45173979734058101, 
0.8762043904815493], [-0.34188391703990084, 0.19998649935419754, -0.63879709781309413, 0.18545222105657011]]], 
[[[0.011754072910120206, -0.90550908100131511, -0.93322868838339446, -0.11741827984316866], [0.77558036495515026, 
-0.85219854425588326, -0.061841571000625484, -0.066269008250732009], [0.96157185984128013, -0.95342288705795242, 
0.45172913312384599, 0.018267106882194906]], [[0.1591168098755007, -0.27340675847513096, -0.14469177879585371, 
0.694740264377673], [-0.80538537088324014, 0.20415359879847728, -0.25347422230372807, -0.93986986572358044], 
[0.82523363323342358, 0.95118360314535688, -0.084226421580063437, 0.23148605050984217]]], [[[0.19213259368793234, 
0.79589318105724693, -0.54294778796310839, -0.20471748468873829], [0.18053098336615636, -0.84012326762801948, 
-0.62575984457545952, -0.14368177341561639], [0.62597131428981112, 0.38509874618804685, 0.47371812469111485, 
-0.86705010549302908]], [[-0.36502653007689423, -0.79613146258086531, 0.15329391158627304, 0.68148041836662387], 
[0.73324139002880684, 0.38529515075962784, -0.9317611817520175, -0.6261241009923757], [0.80092298112981131, 
0.34865093765311261, 0.89442927947537765, 0.92170715451548602]]]]),self.functionspace)
      arg1=0.635778225458
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.63577822545793317, 0.63577822545793317, 0.63577822545793317, 0.63577822545793317], 
[0.63577822545793317, 0.63577822545793317, 0.9427489146021919, 0.63577822545793317], [0.63577822545793317, 0.98227091107399023, 
0.63577822545793317, 0.63577822545793317]], [[0.63577822545793317, 0.63577822545793317, 0.63577822545793317, 
0.63577822545793317], [0.63577822545793317, 0.63577822545793317, 0.63577822545793317, 0.8762043904815493], 
[0.63577822545793317, 0.63577822545793317, 0.63577822545793317, 0.63577822545793317]]], [[[0.63577822545793317, 
0.63577822545793317, 0.63577822545793317, 0.63577822545793317], [0.77558036495515026, 0.63577822545793317, 0.63577822545793317, 
0.63577822545793317], [0.96157185984128013, 0.63577822545793317, 0.63577822545793317, 0.63577822545793317]], 
[[0.63577822545793317, 0.63577822545793317, 0.63577822545793317, 0.694740264377673], [0.63577822545793317, 0.63577822545793317, 
0.63577822545793317, 0.63577822545793317], [0.82523363323342358, 0.95118360314535688, 0.63577822545793317, 
0.63577822545793317]]], [[[0.63577822545793317, 0.79589318105724693, 0.63577822545793317, 0.63577822545793317], 
[0.63577822545793317, 0.63577822545793317, 0.63577822545793317, 0.63577822545793317], [0.63577822545793317, 
0.63577822545793317, 0.63577822545793317, 0.63577822545793317]], [[0.63577822545793317, 0.63577822545793317, 
0.63577822545793317, 0.68148041836662387], [0.73324139002880684, 0.63577822545793317, 0.63577822545793317, 
0.63577822545793317], [0.80092298112981131, 0.63577822545793317, 0.89442927947537765, 
0.92170715451548602]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_array_rank0(self):
      arg0=Data(0.131131173778,self.functionspace)
      arg1=numpy.array(-0.576805781524)
      res=maximum(arg0,arg1)
      ref=Data(0.131131173778,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank1_array_rank0(self):
      arg0=Data(numpy.array([0.14808740913781926, 0.35567967234737252]),self.functionspace)
      arg1=numpy.array(-0.935584836565)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.14808740913781926, 0.35567967234737252]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank2_array_rank0(self):
      arg0=Data(numpy.array([[0.11219561462947825, -0.18427729976238316, -0.26893116897649416, 0.40378445902375137, 
-0.64010126735322159], [-0.54548479863007193, -0.6771466146796592, -0.65941588238170357, -0.080554716485880862, 
-0.51142929986801766], [0.1802163583855485, 0.4654264085214439, -0.88158680167000036, -0.67093289247607557, 
0.47213709052956609], [-0.15939930729103469, -0.25730020832439515, -0.97874531075582616, 0.10703962128016098, 
0.32041153354414065]]),self.functionspace)
      arg1=numpy.array(0.896240426157)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.89624042615692456, 0.89624042615692456, 0.89624042615692456, 0.89624042615692456, 
0.89624042615692456], [0.89624042615692456, 0.89624042615692456, 0.89624042615692456, 0.89624042615692456, 
0.89624042615692456], [0.89624042615692456, 0.89624042615692456, 0.89624042615692456, 0.89624042615692456, 
0.89624042615692456], [0.89624042615692456, 0.89624042615692456, 0.89624042615692456, 0.89624042615692456, 
0.89624042615692456]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank3_array_rank0(self):
      arg0=Data(numpy.array([[[0.11547252472671099, 0.27715619160036087], [-0.0011872311418676951, -0.38096891774653074]], 
[[0.37877267210850607, -0.8255429528666729], [-0.63956098319827714, -0.70194003551721074]], [[0.44309695940748495, 
-0.88721518381663578], [-0.53272604278013991, 0.79625941492663554]], [[-0.1703558707341577, -0.95114537821544309], 
[-0.32281622748150207, 0.31750595998728737]], [[0.30447422840521154, -0.82721331108465446], [0.65676818386047175, 
0.40338692981661328]], [[0.39404755664916813, -0.78343567350875953], [-0.30036954834309015, 
-0.83068548144181831]]]),self.functionspace)
      arg1=numpy.array(0.597874377242)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.59787437724227765, 0.59787437724227765], [0.59787437724227765, 0.59787437724227765]], 
[[0.59787437724227765, 0.59787437724227765], [0.59787437724227765, 0.59787437724227765]], [[0.59787437724227765, 
0.59787437724227765], [0.59787437724227765, 0.79625941492663554]], [[0.59787437724227765, 0.59787437724227765], 
[0.59787437724227765, 0.59787437724227765]], [[0.59787437724227765, 0.59787437724227765], [0.65676818386047175, 
0.59787437724227765]], [[0.59787437724227765, 0.59787437724227765], [0.59787437724227765, 
0.59787437724227765]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank4_array_rank0(self):
      arg0=Data(numpy.array([[[[-0.12333178650474741, -0.41601255505495294, 0.78421542610699757, -0.89389135345287363], 
[-0.89386090823390885, 0.56081437970269388, -0.39255299041023939, 0.32411039481285564], [0.8551210553300661, 
-0.90707027924543415, 0.16036883016043224, -0.10713182442360014]], [[0.62413380378897632, -0.33807365498251962, 
-0.14322028563106537, -0.28390387123110661], [0.10848574295659086, -0.3237100322279467, 0.40626470900918554, 
-0.86322529775615497], [-0.53241603186354824, 0.96450022170256, 0.7410412312711554, -0.90628249882883471]]], 
[[[0.56961012500639319, 0.46258567990715616, 0.2691574282574225, -0.70084123183330149], [0.32178078403959587, 
-0.16491236638222273, -0.67544384124087165, 0.49044876181851271], [0.87426425567196486, 0.99113280536628512, 
-0.33592835344070004, -0.48167757670424649]], [[0.37582959179360564, 0.079659313583228863, 0.13142551769766975, 
-0.50982437173132999], [-0.80549764829567927, -0.99459313338638555, -0.24095207228821169, -0.91245577244012499], 
[0.36688639267496503, 0.20893052084134811, -0.95250384603251925, -0.46421785538004756]]], [[[-0.69408638083069896, 
-0.60590039199066448, 0.65199154172599783, 0.5600386130626831], [-0.45945308229643844, -0.33990278016795927, 
-0.12156326670116702, 0.76710181803628164], [-0.30308185765731799, -0.47607829009357494, -0.66299346892125444, 
0.85153168985207617]], [[0.97196327159277796, -0.76370007931813078, 0.50139581647673981, 0.63066011940195899], 
[-0.98705858372661903, 0.30251550651882297, -0.12436255937028684, 0.47348321672077809], [-0.11280446521491339, 
0.3669717233616796, -0.58417675366941624, -0.8354338127149179]]]]),self.functionspace)
      arg1=numpy.array(-0.987163870919)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[-0.12333178650474741, -0.41601255505495294, 0.78421542610699757, -0.89389135345287363], 
[-0.89386090823390885, 0.56081437970269388, -0.39255299041023939, 0.32411039481285564], [0.8551210553300661, 
-0.90707027924543415, 0.16036883016043224, -0.10713182442360014]], [[0.62413380378897632, -0.33807365498251962, 
-0.14322028563106537, -0.28390387123110661], [0.10848574295659086, -0.3237100322279467, 0.40626470900918554, 
-0.86322529775615497], [-0.53241603186354824, 0.96450022170256, 0.7410412312711554, -0.90628249882883471]]], 
[[[0.56961012500639319, 0.46258567990715616, 0.2691574282574225, -0.70084123183330149], [0.32178078403959587, 
-0.16491236638222273, -0.67544384124087165, 0.49044876181851271], [0.87426425567196486, 0.99113280536628512, 
-0.33592835344070004, -0.48167757670424649]], [[0.37582959179360564, 0.079659313583228863, 0.13142551769766975, 
-0.50982437173132999], [-0.80549764829567927, -0.98716387091897451, -0.24095207228821169, -0.91245577244012499], 
[0.36688639267496503, 0.20893052084134811, -0.95250384603251925, -0.46421785538004756]]], [[[-0.69408638083069896, 
-0.60590039199066448, 0.65199154172599783, 0.5600386130626831], [-0.45945308229643844, -0.33990278016795927, 
-0.12156326670116702, 0.76710181803628164], [-0.30308185765731799, -0.47607829009357494, -0.66299346892125444, 
0.85153168985207617]], [[0.97196327159277796, -0.76370007931813078, 0.50139581647673981, 0.63066011940195899], 
[-0.98705858372661903, 0.30251550651882297, -0.12436255937028684, 0.47348321672077809], [-0.11280446521491339, 
0.3669717233616796, -0.58417675366941624, -0.8354338127149179]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_constData_rank0(self):
      arg0=Data(0.585764561052,self.functionspace)
      arg1=Data(0.562102768003,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(0.585764561052,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank1_constData_rank0(self):
      arg0=Data(numpy.array([0.36817249063478297, 0.84922379654348235]),self.functionspace)
      arg1=Data(-0.000326831113467,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.36817249063478297, 0.84922379654348235]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank2_constData_rank0(self):
      arg0=Data(numpy.array([[0.62061167912589932, -0.38731372384862972, -0.095757334081436607, -0.71510399904747191, 
-0.7960986251434321], [-0.62309855289829441, -0.87302083840022671, 0.19339618209313958, -0.46839248025963864, 
-0.50779431412333742], [-0.21582349308100612, 0.38788705235820564, 0.013156532615211658, 0.48974126868597168, 
-0.8747876083639794], [-0.03351051938462235, -0.01330223402014763, 0.41055032605572195, -0.83673851035555735, 
0.75259533240721854]]),self.functionspace)
      arg1=Data(0.834120422885,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.8341204228848349, 0.8341204228848349, 0.8341204228848349, 0.8341204228848349, 
0.8341204228848349], [0.8341204228848349, 0.8341204228848349, 0.8341204228848349, 0.8341204228848349, 0.8341204228848349], 
[0.8341204228848349, 0.8341204228848349, 0.8341204228848349, 0.8341204228848349, 0.8341204228848349], [0.8341204228848349, 
0.8341204228848349, 0.8341204228848349, 0.8341204228848349, 0.8341204228848349]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank3_constData_rank0(self):
      arg0=Data(numpy.array([[[-0.34639348942196935, 0.54454548312732642], [0.54767137350158723, -0.1261592719343172]], 
[[0.50901738204046398, -0.84585881665715412], [-0.13374339923115808, -0.2770246163740675]], [[-0.54418624743208754, 
0.12594979383328964], [0.36093139391063178, 0.82920727027033436]], [[0.27913464865114546, 0.80222365962388098], 
[0.20763542419244296, 0.90359922967678674]], [[0.84504710858705456, 0.65071402913489296], [-0.38961429927310487, 
-0.71287474316420352]], [[0.21051157156997924, 0.12062151771734198], [0.15420074047028498, 
-0.77490659965239894]]]),self.functionspace)
      arg1=Data(-0.719008102783,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[-0.34639348942196935, 0.54454548312732642], [0.54767137350158723, -0.1261592719343172]], 
[[0.50901738204046398, -0.71900810278299088], [-0.13374339923115808, -0.2770246163740675]], [[-0.54418624743208754, 
0.12594979383328964], [0.36093139391063178, 0.82920727027033436]], [[0.27913464865114546, 0.80222365962388098], 
[0.20763542419244296, 0.90359922967678674]], [[0.84504710858705456, 0.65071402913489296], [-0.38961429927310487, 
-0.71287474316420352]], [[0.21051157156997924, 0.12062151771734198], [0.15420074047028498, 
-0.71900810278299088]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank4_constData_rank0(self):
      arg0=Data(numpy.array([[[[0.24615935787720122, -0.78756263814918848, 0.14312411275206283, -0.61209069712580444], 
[0.66751834472618388, -0.97270217915095181, 0.44734491429921719, 0.56094524052401296], [-0.52616981543309271, 
-0.59090561139819875, 0.72854643636544969, 0.28084180850612128]], [[-0.13831139862668707, -0.41869220979295729, 
0.45665100088168176, -0.17425379986376388], [-0.71060410977750221, -0.23490189062085109, 0.55707427536980814, 
-0.47235867785724728], [-0.7825415636663855, -0.50535912385584014, 0.18080270846310542, 0.81347475179814932]]], 
[[[0.11676945660942573, 0.0090598274192139616, -0.45702967566484753, 0.10306553976340282], [0.99323093060600787, 
0.29434293524837996, 0.54768497876439315, 0.64082304875839924], [0.11154113644780606, -0.92134481397307355, 
0.47282662127503539, 0.88748076184699709]], [[0.22357060873765233, -0.64706904541581722, 0.19015911386848727, 
-0.17771475973870299], [0.26760950005587758, 0.53275086600982924, -0.67459888863079587, -0.47669911211071847], 
[0.13841053267986525, -0.55447231310576828, -0.60380591334225908, 0.45443682452695322]]], [[[0.90802807850631129, 
0.5777144475067848, 0.29643536414101779, -0.70268788049900732], [-0.40417344419730084, 0.60820239131834386, 
0.98877684418314415, 0.02773308630434923], [-0.99593792390185087, -0.99197717949359321, -0.57269874716477709, 
0.23365613209137837]], [[0.74741771939310353, -0.57587639566704985, 0.74979270157039379, 0.88416116732103767], 
[0.3503774673176725, 0.4043599318959008, 0.92900873297982289, -0.060730954440895157], [-0.82062848810850908, 
0.36166456925688828, -0.98165811447801321, 0.10663254098157515]]]]),self.functionspace)
      arg1=Data(-0.218709768625,self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.24615935787720122, -0.21870976862466218, 0.14312411275206283, -0.21870976862466218], 
[0.66751834472618388, -0.21870976862466218, 0.44734491429921719, 0.56094524052401296], [-0.21870976862466218, 
-0.21870976862466218, 0.72854643636544969, 0.28084180850612128]], [[-0.13831139862668707, -0.21870976862466218, 
0.45665100088168176, -0.17425379986376388], [-0.21870976862466218, -0.21870976862466218, 0.55707427536980814, 
-0.21870976862466218], [-0.21870976862466218, -0.21870976862466218, 0.18080270846310542, 0.81347475179814932]]], 
[[[0.11676945660942573, 0.0090598274192139616, -0.21870976862466218, 0.10306553976340282], [0.99323093060600787, 
0.29434293524837996, 0.54768497876439315, 0.64082304875839924], [0.11154113644780606, -0.21870976862466218, 
0.47282662127503539, 0.88748076184699709]], [[0.22357060873765233, -0.21870976862466218, 0.19015911386848727, 
-0.17771475973870299], [0.26760950005587758, 0.53275086600982924, -0.21870976862466218, -0.21870976862466218], 
[0.13841053267986525, -0.21870976862466218, -0.21870976862466218, 0.45443682452695322]]], [[[0.90802807850631129, 
0.5777144475067848, 0.29643536414101779, -0.21870976862466218], [-0.21870976862466218, 0.60820239131834386, 
0.98877684418314415, 0.02773308630434923], [-0.21870976862466218, -0.21870976862466218, -0.21870976862466218, 
0.23365613209137837]], [[0.74741771939310353, -0.21870976862466218, 0.74979270157039379, 0.88416116732103767], 
[0.3503774673176725, 0.4043599318959008, 0.92900873297982289, -0.060730954440895157], [-0.21870976862466218, 
0.36166456925688828, -0.21870976862466218, 0.10663254098157515]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_expandedData_rank0(self):
      arg0=Data(-0.0924750298109,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.719276974378)+(1.-msk_arg1)*(-0.901780055402)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.719276974378)+(1.-msk_ref)*(-0.0924750298109)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank1_expandedData_rank0(self):
      arg0=Data(numpy.array([0.19003810057983306, 0.86747179360557602]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.979758050952)+(1.-msk_arg1)*(0.498784837635)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.19003810057983306, 0.86747179360557602])+(1.-msk_ref)*numpy.array([0.49878483763491466, 
0.86747179360557602])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank2_expandedData_rank0(self):
      arg0=Data(numpy.array([[-0.21116738853853634, 0.39376908967419455, -0.90448828983244067, 0.39775680151484716, 
-0.62546967898667782], [0.71684884432650064, -0.2950221732323155, 0.061549355711304932, 0.92168965316615159, 
0.54752726203495428], [-0.12485263858702367, 0.16626287261463624, -0.79329781788494325, -0.22207097268419607, 
0.19795438035557633], [0.76599397431182026, 0.56414018544113698, 0.95805026779533731, -0.69615823883217742, 
0.21821142022539242]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.243791427144)+(1.-msk_arg1)*(-0.440291613911)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.21116738853853634, 0.39376908967419455, -0.24379142714392676, 0.39775680151484716, 
-0.24379142714392676], [0.71684884432650064, -0.24379142714392676, 0.061549355711304932, 0.92168965316615159, 
0.54752726203495428], [-0.12485263858702367, 0.16626287261463624, -0.24379142714392676, -0.22207097268419607, 
0.19795438035557633], [0.76599397431182026, 0.56414018544113698, 0.95805026779533731, -0.24379142714392676, 
0.21821142022539242]])+(1.-msk_ref)*numpy.array([[-0.21116738853853634, 0.39376908967419455, -0.44029161391063454, 
0.39775680151484716, -0.44029161391063454], [0.71684884432650064, -0.2950221732323155, 0.061549355711304932, 
0.92168965316615159, 0.54752726203495428], [-0.12485263858702367, 0.16626287261463624, -0.44029161391063454, 
-0.22207097268419607, 0.19795438035557633], [0.76599397431182026, 0.56414018544113698, 0.95805026779533731, 
-0.44029161391063454, 0.21821142022539242]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank3_expandedData_rank0(self):
      arg0=Data(numpy.array([[[0.71382052722853828, 0.59639246293382753], [0.79091055029858603, 0.6098069915009563]], 
[[-0.81877756445176075, -0.12572711561586747], [0.6629503717900096, 0.088668188470614639]], [[-0.25907968164793926, 
0.078844106245125056], [0.53171494813158082, -0.48562454486265927]], [[-0.16221141560120533, -0.22108938742601714], 
[-0.19849691519790702, 0.1366342465866397]], [[0.97032753659399651, 0.10811990970766572], [-0.53083812183678813, 
0.48681350528164535]], [[0.94501075888711217, 0.15734516046100699], [0.67336660412428007, 
0.54871928770743117]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.154574403253)+(1.-msk_arg1)*(0.444360735115)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.71382052722853828, 0.59639246293382753], [0.79091055029858603, 0.6098069915009563]], 
[[0.15457440325278737, 0.15457440325278737], [0.6629503717900096, 0.15457440325278737]], [[0.15457440325278737, 
0.15457440325278737], [0.53171494813158082, 0.15457440325278737]], [[0.15457440325278737, 0.15457440325278737], 
[0.15457440325278737, 0.15457440325278737]], [[0.97032753659399651, 0.15457440325278737], [0.15457440325278737, 
0.48681350528164535]], [[0.94501075888711217, 0.15734516046100699], [0.67336660412428007, 
0.54871928770743117]]])+(1.-msk_ref)*numpy.array([[[0.71382052722853828, 0.59639246293382753], [0.79091055029858603, 
0.6098069915009563]], [[0.4443607351150709, 0.4443607351150709], [0.6629503717900096, 0.4443607351150709]], 
[[0.4443607351150709, 0.4443607351150709], [0.53171494813158082, 0.4443607351150709]], [[0.4443607351150709, 
0.4443607351150709], [0.4443607351150709, 0.4443607351150709]], [[0.97032753659399651, 0.4443607351150709], 
[0.4443607351150709, 0.48681350528164535]], [[0.94501075888711217, 0.4443607351150709], [0.67336660412428007, 
0.54871928770743117]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank4_expandedData_rank0(self):
      arg0=Data(numpy.array([[[[0.17987454394191782, -0.47366581235947391, 0.57343836836209627, -0.38076754396458123], 
[0.57156205056895448, 0.22221196833792156, -0.63457115192030189, -0.90743873572672884], [0.50190781994193601, 
-0.72382204778829151, -0.16048375541143778, 0.24666469799623414]], [[0.1797380572435916, 0.16543463798169644, 
-0.38942568372144026, -0.60094923326812366], [0.58369106606630172, -0.72979782894371525, -0.71685913494409648, 
-0.43487862431086666], [0.94570907221553924, -0.71041646506714295, -0.57201057001577316, -0.27780439634546106]]], 
[[[0.21885917868948757, 0.691429511541092, -0.86074155940358898, -0.74572948424544871], [-0.67104106531707486, 
0.65026368776100796, -0.77242606184409479, -0.17309140373321386], [-0.71424293657762106, 0.0160043137660697, 
-0.86608565479945288, 0.21568143007726359]], [[0.17793601295093553, 0.8241368231005648, -0.56689484436349358, 
-0.64591475368907791], [-0.83488920100836284, 0.89807531811166474, 0.12481763857472217, 0.19555968134032309], 
[0.90280696977547814, -0.3114337513979637, -0.4016548633642858, 0.84998160623596397]]], [[[-0.41113212039641711, 
0.69589842214715314, 0.40122617777186043, 0.66162309921520612], [-0.64259349002452848, -0.61545179085033141, 
0.35720959723129897, -0.74561290819100612], [0.25049896732519583, -0.84957718142891347, -0.69000881268819692, 
0.55532528673635673]], [[0.65030548208217898, 0.61753581199201957, 0.55386412523479889, -0.59819873675646185], 
[0.59881331123370507, -0.73382421864055347, -0.25155846699294737, -0.83893004279709227], [-0.68286287600890794, 
-0.029611557116543841, 0.092485711989719332, 0.54588443381266383]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.647690225694)+(1.-msk_arg1)*(-0.116678653334)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.17987454394191782, -0.47366581235947391, 0.57343836836209627, -0.38076754396458123], 
[0.57156205056895448, 0.22221196833792156, -0.63457115192030189, -0.64769022569387036], [0.50190781994193601, 
-0.64769022569387036, -0.16048375541143778, 0.24666469799623414]], [[0.1797380572435916, 0.16543463798169644, 
-0.38942568372144026, -0.60094923326812366], [0.58369106606630172, -0.64769022569387036, -0.64769022569387036, 
-0.43487862431086666], [0.94570907221553924, -0.64769022569387036, -0.57201057001577316, -0.27780439634546106]]], 
[[[0.21885917868948757, 0.691429511541092, -0.64769022569387036, -0.64769022569387036], [-0.64769022569387036, 
0.65026368776100796, -0.64769022569387036, -0.17309140373321386], [-0.64769022569387036, 0.0160043137660697, 
-0.64769022569387036, 0.21568143007726359]], [[0.17793601295093553, 0.8241368231005648, -0.56689484436349358, 
-0.64591475368907791], [-0.64769022569387036, 0.89807531811166474, 0.12481763857472217, 0.19555968134032309], 
[0.90280696977547814, -0.3114337513979637, -0.4016548633642858, 0.84998160623596397]]], [[[-0.41113212039641711, 
0.69589842214715314, 0.40122617777186043, 0.66162309921520612], [-0.64259349002452848, -0.61545179085033141, 
0.35720959723129897, -0.64769022569387036], [0.25049896732519583, -0.64769022569387036, -0.64769022569387036, 
0.55532528673635673]], [[0.65030548208217898, 0.61753581199201957, 0.55386412523479889, -0.59819873675646185], 
[0.59881331123370507, -0.64769022569387036, -0.25155846699294737, -0.64769022569387036], [-0.64769022569387036, 
-0.029611557116543841, 0.092485711989719332, 0.54588443381266383]]]])+(1.-msk_ref)*numpy.array([[[[0.17987454394191782, 
-0.11667865333375405, 0.57343836836209627, -0.11667865333375405], [0.57156205056895448, 0.22221196833792156, 
-0.11667865333375405, -0.11667865333375405], [0.50190781994193601, -0.11667865333375405, -0.11667865333375405, 
0.24666469799623414]], [[0.1797380572435916, 0.16543463798169644, -0.11667865333375405, -0.11667865333375405], 
[0.58369106606630172, -0.11667865333375405, -0.11667865333375405, -0.11667865333375405], [0.94570907221553924, 
-0.11667865333375405, -0.11667865333375405, -0.11667865333375405]]], [[[0.21885917868948757, 0.691429511541092, 
-0.11667865333375405, -0.11667865333375405], [-0.11667865333375405, 0.65026368776100796, -0.11667865333375405, 
-0.11667865333375405], [-0.11667865333375405, 0.0160043137660697, -0.11667865333375405, 0.21568143007726359]], 
[[0.17793601295093553, 0.8241368231005648, -0.11667865333375405, -0.11667865333375405], [-0.11667865333375405, 
0.89807531811166474, 0.12481763857472217, 0.19555968134032309], [0.90280696977547814, -0.11667865333375405, 
-0.11667865333375405, 0.84998160623596397]]], [[[-0.11667865333375405, 0.69589842214715314, 0.40122617777186043, 
0.66162309921520612], [-0.11667865333375405, -0.11667865333375405, 0.35720959723129897, -0.11667865333375405], 
[0.25049896732519583, -0.11667865333375405, -0.11667865333375405, 0.55532528673635673]], [[0.65030548208217898, 
0.61753581199201957, 0.55386412523479889, -0.11667865333375405], [0.59881331123370507, -0.11667865333375405, 
-0.11667865333375405, -0.11667865333375405], [-0.11667865333375405, -0.029611557116543841, 0.092485711989719332, 
0.54588443381266383]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_array_rank1(self):
      arg0=Data(-0.977109354346,self.functionspace)
      arg1=numpy.array([0.70276044813264082, -0.31317697172843362])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.70276044813264082, -0.31317697172843362]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank1_array_rank1(self):
      arg0=Data(numpy.array([-0.62917253954533403, -0.91478760916921553]),self.functionspace)
      arg1=numpy.array([0.33852418316707711, -0.28298537896949161])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.33852418316707711, -0.28298537896949161]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_constData_rank1(self):
      arg0=Data(-0.159726189832,self.functionspace)
      arg1=Data(numpy.array([0.1189612003237861, -0.63087634684158944]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([0.1189612003237861, -0.15972618983177433]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank1_constData_rank1(self):
      arg0=Data(numpy.array([-0.740289439487964, 0.50792620718681647]),self.functionspace)
      arg1=Data(numpy.array([-0.98869116031302773, -0.030691151152568041]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([-0.740289439487964, 0.50792620718681647]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_expandedData_rank1(self):
      arg0=Data(0.0787324456408,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-0.48016870047192817, -0.99662347188520961])+(1.-msk_arg1)*numpy.array([0.58774285667709059, 
0.67262794612436538])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.078732445640770532, 0.078732445640770532])+(1.-msk_ref)*numpy.array([0.58774285667709059, 
0.67262794612436538])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank1_expandedData_rank1(self):
      arg0=Data(numpy.array([0.17322811204814759, 0.25536817066436801]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.91972158516709168, -0.98465374807727835])+(1.-msk_arg1)*numpy.array([-0.19155853001762169, 
-0.43931509419045467])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.91972158516709168, 0.25536817066436801])+(1.-msk_ref)*numpy.array([0.17322811204814759, 
0.25536817066436801])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_array_rank2(self):
      arg0=Data(-0.590060805237,self.functionspace)
      arg1=numpy.array([[-0.46148245190505666, 0.72542064767267411, -0.49628838214418636, 0.36379029454198486, 
-0.35637937725143498], [0.77716266154028713, -0.0294929702278659, 0.77367789791539621, 0.26018292128939491, 
-0.0013933338625782277], [-0.71102579607495486, 0.058714555627548126, 0.62191425731141292, -0.9329697856968644, 
0.48541403689676299], [-0.78897231411198532, -0.72015354884882865, -0.62803607199062306, -0.96795550516592099, 
0.86073050375759519]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[-0.46148245190505666, 0.72542064767267411, -0.49628838214418636, 0.36379029454198486, 
-0.35637937725143498], [0.77716266154028713, -0.0294929702278659, 0.77367789791539621, 0.26018292128939491, 
-0.0013933338625782277], [-0.59006080523717053, 0.058714555627548126, 0.62191425731141292, -0.59006080523717053, 
0.48541403689676299], [-0.59006080523717053, -0.59006080523717053, -0.59006080523717053, -0.59006080523717053, 
0.86073050375759519]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank2_array_rank2(self):
      arg0=Data(numpy.array([[-0.73760404253705514, 0.50224248181499931, 0.11399712803883166, -0.13254212815072308, 
0.74172163669326041], [-0.87041848231795704, 0.84343778201486463, -0.75554993046891106, 0.9855020616978345, 
-0.54320886383165745], [0.015954104044625605, -0.72612141326347879, -0.34799202711515242, -0.59679759609203797, 
-0.2575746318844927], [-0.2477036236422252, -0.021811926970866979, 0.81844582541975552, -0.14903149292622531, 
0.80053174022124529]]),self.functionspace)
      arg1=numpy.array([[-0.86954510950167463, 0.1017003127120133, -0.16951929662470433, 0.70137020593450439, 
0.77429037367257458], [0.39247727833147628, -0.71486192220367606, -0.59117959653327024, 0.50948617730426826, 
-0.88092334234319924], [-0.95167080948153315, 0.021566922221860629, -0.37806539314914223, 0.3191199630251873, 
0.21890366409298223], [0.8635367262140623, 0.52741696809771943, 0.40373578351947836, 0.40740580857358721, 0.50991869479490881]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[-0.73760404253705514, 0.50224248181499931, 0.11399712803883166, 0.70137020593450439, 
0.77429037367257458], [0.39247727833147628, 0.84343778201486463, -0.59117959653327024, 0.9855020616978345, 
-0.54320886383165745], [0.015954104044625605, 0.021566922221860629, -0.34799202711515242, 0.3191199630251873, 
0.21890366409298223], [0.8635367262140623, 0.52741696809771943, 0.81844582541975552, 0.40740580857358721, 
0.80053174022124529]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_constData_rank2(self):
      arg0=Data(0.471113579692,self.functionspace)
      arg1=Data(numpy.array([[0.20278512451424202, 0.46932554124551484, 0.51627101397671771, -0.15806811678484656, 
0.90125732162226746], [-0.64525760065047266, -0.99646917035423299, -0.0079700168467822241, -0.1148252168651378, 
-0.60869273468514651], [-0.81920269598799123, 0.18492463625508537, 0.67703260540557975, 0.38678293776089578, 
-0.58337450118932721], [-0.85103881319765606, -0.18597068284684348, 0.55532783770806615, -0.3391026116648197, 
0.59931507372202009]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.47111357969230805, 0.47111357969230805, 0.51627101397671771, 0.47111357969230805, 
0.90125732162226746], [0.47111357969230805, 0.47111357969230805, 0.47111357969230805, 0.47111357969230805, 
0.47111357969230805], [0.47111357969230805, 0.47111357969230805, 0.67703260540557975, 0.47111357969230805, 
0.47111357969230805], [0.47111357969230805, 0.47111357969230805, 0.55532783770806615, 0.47111357969230805, 
0.59931507372202009]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank2_constData_rank2(self):
      arg0=Data(numpy.array([[-0.17406452878186696, -0.34759214930565996, -0.13708153516362609, 0.93110590449668984, 
-0.47649961659515139], [-0.35702537789588118, -0.4057682901706241, 0.56892562984164674, 0.20451757373433055, 
-0.47729926624352892], [-0.55493234840963379, 0.020855045362278224, 0.95156666561071002, 0.14975165550539549, 
-0.17074783729098764], [-0.2364495010997254, -0.32021782391836329, -0.45843843696539488, -0.47665234638234, 
-0.41919606212785676]]),self.functionspace)
      arg1=Data(numpy.array([[0.64408124666207822, -0.67629542279010635, -0.4931164628479503, -0.96447891179315248, 
-0.45618690487636582], [0.18569576708094893, -0.085193091641874163, 0.08793086582533971, 0.64570776148383136, 
-0.5516531478806932], [0.87675126968759676, -0.39508497185268077, -0.97669171907295138, -0.68655710586099006, 
-0.028857220969970232], [-0.3670380390017618, 0.76258006170069592, -0.31454790037894464, 0.6627659629911522, 
0.82734991520477563]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[0.64408124666207822, -0.34759214930565996, -0.13708153516362609, 0.93110590449668984, 
-0.45618690487636582], [0.18569576708094893, -0.085193091641874163, 0.56892562984164674, 0.64570776148383136, 
-0.47729926624352892], [0.87675126968759676, 0.020855045362278224, 0.95156666561071002, 0.14975165550539549, 
-0.028857220969970232], [-0.2364495010997254, 0.76258006170069592, -0.31454790037894464, 0.6627659629911522, 
0.82734991520477563]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_expandedData_rank2(self):
      arg0=Data(-0.0165871765665,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-0.73128249610105489, 0.8419999761644652, 0.32706544558055151, 0.98621609382436559, 
0.11814390137662678], [0.19794947116873596, 0.31254355907321152, -0.87988550364485918, -0.11555651603083028, 
-0.71600368849223273], [-0.1893391600405403, 0.90957761312463936, -0.52243082625432802, -0.61123719298866308, 
0.33907244894623134], [0.69675393273476094, 0.80610078097181703, -0.025854445517779379, 0.94898341046945389, 
0.46113499488993592]])+(1.-msk_arg1)*numpy.array([[0.23505303855840864, 0.99481465040860417, -0.086492983784499877, 
-0.65370229345838227, 0.10003757203241559], [-0.55539504622389524, -0.97706587559470259, 0.76222581775594289, 
0.77064743585593676, 0.37822774452875341], [0.63726139395335868, 0.82448429030660098, -0.98743591219817395, 
-0.11783271519582783, 0.82622085914926746], [0.79063087780255503, -0.51677954263556924, -0.11079135634696624, 
0.55532802164644224, 0.97336607909484085]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.016587176566458695, 0.8419999761644652, 0.32706544558055151, 0.98621609382436559, 
0.11814390137662678], [0.19794947116873596, 0.31254355907321152, -0.016587176566458695, -0.016587176566458695, 
-0.016587176566458695], [-0.016587176566458695, 0.90957761312463936, -0.016587176566458695, -0.016587176566458695, 
0.33907244894623134], [0.69675393273476094, 0.80610078097181703, -0.016587176566458695, 0.94898341046945389, 
0.46113499488993592]])+(1.-msk_ref)*numpy.array([[0.23505303855840864, 0.99481465040860417, -0.016587176566458695, 
-0.016587176566458695, 0.10003757203241559], [-0.016587176566458695, -0.016587176566458695, 0.76222581775594289, 
0.77064743585593676, 0.37822774452875341], [0.63726139395335868, 0.82448429030660098, -0.016587176566458695, 
-0.016587176566458695, 0.82622085914926746], [0.79063087780255503, -0.016587176566458695, -0.016587176566458695, 
0.55532802164644224, 0.97336607909484085]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank2_expandedData_rank2(self):
      arg0=Data(numpy.array([[-0.58944550721294209, 0.11789616185996188, 0.72555020927912528, 0.95067825171691411, 
0.44929807483372386], [-0.90050975945004685, -0.14027235979416108, -0.50271571837565965, 0.71764496858912841, 
0.13936247694569537], [-0.55363895921566852, -0.29692903194729126, -0.12968512147399403, 0.16630251248490757, 
-0.15779583568993605], [-0.91804262372165168, -0.0058087569025269126, 0.57431208620955165, -0.71561067889580943, 
-0.22286757830799497]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-0.99165981962421723, 0.056234396987206559, -0.96929836288649107, 0.85741514845807765, 
0.31290702472003784], [0.66879661516796074, 0.74410544323631544, 0.61758123318413549, 0.0023178475410641219, 
0.035637036084378515], [0.80734776741943692, -0.93425673113467789, -0.3545773036750044, 0.17713706129968121, 
-0.78562855023306311], [0.95764289578713813, 0.82592763875748232, 0.56453388909804225, 0.33669515016119256, 
0.68270953107335797]])+(1.-msk_arg1)*numpy.array([[0.46121428483489479, 0.77301745076266992, -0.75239767323970663, 
-0.84007824075301141, 0.54854476434581101], [-0.18306960903719038, 0.22306205318244277, 0.95307177391042264, 
-0.83728247335587391, -0.64823533348017937], [0.90595429846142239, -0.55842375993133087, 0.69860419850019628, 
-0.58794473533051539, -0.90321846168159303], [0.19112532041271257, -0.16066829098287294, 0.62710659839874872, 
-0.58348891958458182, -0.43880626183960802]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.58944550721294209, 0.11789616185996188, 0.72555020927912528, 0.95067825171691411, 
0.44929807483372386], [0.66879661516796074, 0.74410544323631544, 0.61758123318413549, 0.71764496858912841, 
0.13936247694569537], [0.80734776741943692, -0.29692903194729126, -0.12968512147399403, 0.17713706129968121, 
-0.15779583568993605], [0.95764289578713813, 0.82592763875748232, 0.57431208620955165, 0.33669515016119256, 
0.68270953107335797]])+(1.-msk_ref)*numpy.array([[0.46121428483489479, 0.77301745076266992, 0.72555020927912528, 
0.95067825171691411, 0.54854476434581101], [-0.18306960903719038, 0.22306205318244277, 0.95307177391042264, 
0.71764496858912841, 0.13936247694569537], [0.90595429846142239, -0.29692903194729126, 0.69860419850019628, 
0.16630251248490757, -0.15779583568993605], [0.19112532041271257, -0.0058087569025269126, 0.62710659839874872, 
-0.58348891958458182, -0.22286757830799497]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_array_rank3(self):
      arg0=Data(0.516978268542,self.functionspace)
      arg1=numpy.array([[[0.1203754261590837, -0.30872063802497229], [-0.54367292398629163, -0.28894544783644749]], 
[[0.98579636264447745, -0.1617667296901939], [-0.71399942465432775, 0.53397315922387301]], [[-0.30699998747323254, 
0.077194107501915799], [0.79875258103307534, 0.80801781194196387]], [[0.15460773887563417, 0.90896321404086278], 
[0.20719915824628354, 0.51262923866513121]], [[-0.42605851894450963, 0.11121377720978609], [0.83365318051571125, 
-0.090277192445190391]], [[0.11506747491337666, 0.6855450860006127], [-0.55520344405285305, -0.29122646515827966]]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.51697826854212692, 0.51697826854212692], [0.51697826854212692, 0.51697826854212692]], 
[[0.98579636264447745, 0.51697826854212692], [0.51697826854212692, 0.53397315922387301]], [[0.51697826854212692, 
0.51697826854212692], [0.79875258103307534, 0.80801781194196387]], [[0.51697826854212692, 0.90896321404086278], 
[0.51697826854212692, 0.51697826854212692]], [[0.51697826854212692, 0.51697826854212692], [0.83365318051571125, 
0.51697826854212692]], [[0.51697826854212692, 0.6855450860006127], [0.51697826854212692, 
0.51697826854212692]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank3_array_rank3(self):
      arg0=Data(numpy.array([[[-0.019064560845273526, -0.66082512842015606], [0.7775245397942645, -0.97863728880489664]], 
[[0.14796990911746843, -0.14940640351421752], [-0.86829509897198331, 0.37449746403230755]], [[0.37150654719541598, 
-0.28378634170515826], [-0.50000172789557618, 0.12870803836646894]], [[0.011474562798716059, 0.62211999362561543], 
[0.20936671603779877, 0.1596384101994337]], [[0.48413175379125017, -0.70455876802613626], [0.95196006961287805, 
0.20450770288369013]], [[-0.69630908911115341, -0.40070777001767066], [0.39061204191308385, 
-0.7494107073434304]]]),self.functionspace)
      arg1=numpy.array([[[-0.78977223364983429, -0.39796958371743218], [-0.72434391726505232, -0.52658036061654578]], 
[[0.30677953915987843, -0.27325740398129361], [-0.44704945638883409, 0.35675189799215068]], [[-0.85496603242765823, 
-0.49015497061116231], [0.38100286965101793, 0.71258135816204082]], [[-0.69502829292158985, -0.66075782445418518], 
[0.62742938866730125, -0.0042722353662665302]], [[0.47028792653016804, -0.43167155331296869], [0.46101712703457975, 
-0.26196273464811326]], [[-0.79525334450742657, 0.58902924204184304], [0.071218600732356085, -0.44930940165988953]]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[-0.019064560845273526, -0.39796958371743218], [0.7775245397942645, -0.52658036061654578]], 
[[0.30677953915987843, -0.14940640351421752], [-0.44704945638883409, 0.37449746403230755]], [[0.37150654719541598, 
-0.28378634170515826], [0.38100286965101793, 0.71258135816204082]], [[0.011474562798716059, 0.62211999362561543], 
[0.62742938866730125, 0.1596384101994337]], [[0.48413175379125017, -0.43167155331296869], [0.95196006961287805, 
0.20450770288369013]], [[-0.69630908911115341, 0.58902924204184304], [0.39061204191308385, 
-0.44930940165988953]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_constData_rank3(self):
      arg0=Data(-0.520111358301,self.functionspace)
      arg1=Data(numpy.array([[[-0.53755593069331176, 0.37344778982644522], [0.90325038776102318, -0.89380296608704612]], 
[[0.59616188236635415, 0.051917694202589626], [-0.04448028945702287, 0.41100328136339304]], [[0.33171935562963939, 
-0.24899622688538758], [0.62960041822469925, 0.33958207874297885]], [[0.77867517548268683, 0.42072327881397564], 
[-0.74938980192739946, 0.81416985659114172]], [[0.3116774006710874, -0.33738332950437711], [0.78527047728394561, 
-0.74961547941494744]], [[-0.20009722385279827, -0.6571543996337208], [0.17776822203863141, 
0.44451143669045345]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[-0.52011135830068911, 0.37344778982644522], [0.90325038776102318, -0.52011135830068911]], 
[[0.59616188236635415, 0.051917694202589626], [-0.04448028945702287, 0.41100328136339304]], [[0.33171935562963939, 
-0.24899622688538758], [0.62960041822469925, 0.33958207874297885]], [[0.77867517548268683, 0.42072327881397564], 
[-0.52011135830068911, 0.81416985659114172]], [[0.3116774006710874, -0.33738332950437711], [0.78527047728394561, 
-0.52011135830068911]], [[-0.20009722385279827, -0.52011135830068911], [0.17776822203863141, 
0.44451143669045345]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank3_constData_rank3(self):
      arg0=Data(numpy.array([[[0.22644209225706202, -0.82408753542802948], [-0.3229055658381601, 0.2590293226229694]], 
[[-0.26905946802022762, 0.2903861382640851], [0.12277829585192546, -0.77850587791871839]], [[-0.89889146053925439, 
0.38226490597490859], [0.90580586744844571, 0.55103010631847082]], [[-0.039650312436662727, -0.88676021840684105], 
[-0.51601766497181489, -0.25197490301551628]], [[0.61158062010002601, 0.29776204958735475], [0.71009302472890323, 
-0.66290156645868592]], [[0.35864937589582269, 0.39150421091712273], [0.62113215392768795, 
-0.60699036535846251]]]),self.functionspace)
      arg1=Data(numpy.array([[[0.19115277784536788, -0.12179826773639668], [0.93712678010940254, 0.88203929581384566]], 
[[-0.88435454431013327, 0.9789660191397036], [0.59264976401288116, -0.95487963435415701]], [[-0.63201606726111259, 
-0.55289610271892187], [-0.28138775032557017, 0.10882077764575948]], [[-0.49427255052853813, -0.052753620845127358], 
[0.041666359004336684, 0.79096925989660916]], [[-0.41422835506377953, 0.44371821729329164], [0.29584910796715769, 
0.32801776168641106]], [[0.48887697683026987, 0.060894815731677765], [0.03438210218260096, 
0.012054779031575036]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[0.22644209225706202, -0.12179826773639668], [0.93712678010940254, 0.88203929581384566]], 
[[-0.26905946802022762, 0.9789660191397036], [0.59264976401288116, -0.77850587791871839]], [[-0.63201606726111259, 
0.38226490597490859], [0.90580586744844571, 0.55103010631847082]], [[-0.039650312436662727, -0.052753620845127358], 
[0.041666359004336684, 0.79096925989660916]], [[0.61158062010002601, 0.44371821729329164], [0.71009302472890323, 
0.32801776168641106]], [[0.48887697683026987, 0.39150421091712273], [0.62113215392768795, 
0.012054779031575036]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_expandedData_rank3(self):
      arg0=Data(-0.417160613179,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-0.59344857070579238, 0.57386185515692323], [0.22052027505720551, -0.02253631100635034]], 
[[-0.79319250300435051, -0.80838786337089852], [0.72156951045421569, 0.79754573800662776]], [[0.44658460292787949, 
-0.6036139053961691], [-0.0074518374538019572, 0.10259587076510246]], [[-0.67150694427957869, -0.24769554924870607], 
[0.34642765365536365, 0.40804680401868687]], [[-0.48546098050896314, -0.37273512567714118], [-0.93149112098916875, 
0.56189739537971306]], [[0.38968682790960374, 0.6967969004356549], [0.31719546261205389, 
0.66755203749869896]]])+(1.-msk_arg1)*numpy.array([[[-0.32798712079017855, 0.91458585203417941], [-0.1858316961301949, 
0.10471165389886661]], [[0.25739129223381751, -0.20580522632299081], [-0.77836146736583722, 0.71646905903017921]], 
[[-0.12213192671753959, 0.2681111999511725], [-0.88598687196668746, -0.10299574975782755]], [[-0.47951968494409059, 
0.38762496133793989], [-0.20961640133930182, -0.46499685779521527]], [[-0.63002780928919999, 0.99970479651183686], 
[-0.28892538074111696, -0.86014943997554583]], [[-0.87521791528084747, -0.2290352438643557], [0.61458437038138758, 
-0.054295041010556266]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.41716061317925801, 0.57386185515692323], [0.22052027505720551, -0.02253631100635034]], 
[[-0.41716061317925801, -0.41716061317925801], [0.72156951045421569, 0.79754573800662776]], [[0.44658460292787949, 
-0.41716061317925801], [-0.0074518374538019572, 0.10259587076510246]], [[-0.41716061317925801, -0.24769554924870607], 
[0.34642765365536365, 0.40804680401868687]], [[-0.41716061317925801, -0.37273512567714118], [-0.41716061317925801, 
0.56189739537971306]], [[0.38968682790960374, 0.6967969004356549], [0.31719546261205389, 
0.66755203749869896]]])+(1.-msk_ref)*numpy.array([[[-0.32798712079017855, 0.91458585203417941], [-0.1858316961301949, 
0.10471165389886661]], [[0.25739129223381751, -0.20580522632299081], [-0.41716061317925801, 0.71646905903017921]], 
[[-0.12213192671753959, 0.2681111999511725], [-0.41716061317925801, -0.10299574975782755]], [[-0.41716061317925801, 
0.38762496133793989], [-0.20961640133930182, -0.41716061317925801]], [[-0.41716061317925801, 0.99970479651183686], 
[-0.28892538074111696, -0.41716061317925801]], [[-0.41716061317925801, -0.2290352438643557], [0.61458437038138758, 
-0.054295041010556266]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank3_expandedData_rank3(self):
      arg0=Data(numpy.array([[[-0.83383467576998993, -0.83006279778742731], [0.61582380340209775, -0.4064088511087891]], 
[[-0.4159938969754351, -0.073787477564773329], [-0.021547740656737036, 0.5133460168591264]], [[0.91656606912919014, 
0.53167536065528687], [-0.61521902289133901, 0.4664937027114231]], [[0.97834019232877312, -0.93183928685658612], 
[-0.89297362621765086, 0.98615396264285726]], [[0.34755706158837762, 0.58475703376154442], [-0.15101139759741611, 
0.40395116766294437]], [[0.62168225352120166, 0.50496809316394398], [0.72198090495722789, 
0.85904449954970996]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.19214098611675801, 0.07906975802244931], [0.70387607748340963, 0.73265073793599189]], 
[[-0.6142258277704653, 0.6255979194324861], [0.46214070326839796, -0.28059147251841399]], [[-0.57130450989061998, 
0.5709484739777515], [0.48615540360013054, -0.60718602833024926]], [[0.48182675680934151, -0.66948554719719966], 
[0.8085115047175595, -0.95171986323783053]], [[0.5870376913405424, 0.59265808147737098], [0.49569538437672511, 
-0.32465377675760454]], [[0.41879820042914684, 0.82573065216442831], [0.14668313470902383, 
0.78600006019282054]]])+(1.-msk_arg1)*numpy.array([[[0.63268720723055161, 0.78130831328720141], [-0.18520848313978489, 
-0.35202194720092805]], [[0.58179955935294925, 0.1226309312208731], [0.29154247836357272, -0.9060564605671757]], 
[[0.71741017447493505, 0.96466082998871294], [-0.81353604274588265, -0.098340216808310954]], [[-0.88020377622640522, 
-0.84721483807877607], [0.6939301069991286, -0.30953811263549613]], [[0.99593228905602715, 0.32599642612508051], 
[-0.40377432683868597, 0.0021172786571095337]], [[0.69285022412981334, 0.70887735927613194], [0.17846982139165757, 
0.33981526774046644]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.19214098611675801, 0.07906975802244931], [0.70387607748340963, 0.73265073793599189]], 
[[-0.4159938969754351, 0.6255979194324861], [0.46214070326839796, 0.5133460168591264]], [[0.91656606912919014, 
0.5709484739777515], [0.48615540360013054, 0.4664937027114231]], [[0.97834019232877312, -0.66948554719719966], 
[0.8085115047175595, 0.98615396264285726]], [[0.5870376913405424, 0.59265808147737098], [0.49569538437672511, 
0.40395116766294437]], [[0.62168225352120166, 0.82573065216442831], [0.72198090495722789, 
0.85904449954970996]]])+(1.-msk_ref)*numpy.array([[[0.63268720723055161, 0.78130831328720141], [0.61582380340209775, 
-0.35202194720092805]], [[0.58179955935294925, 0.1226309312208731], [0.29154247836357272, 0.5133460168591264]], 
[[0.91656606912919014, 0.96466082998871294], [-0.61521902289133901, 0.4664937027114231]], [[0.97834019232877312, 
-0.84721483807877607], [0.6939301069991286, 0.98615396264285726]], [[0.99593228905602715, 0.58475703376154442], 
[-0.15101139759741611, 0.40395116766294437]], [[0.69285022412981334, 0.70887735927613194], [0.72198090495722789, 
0.85904449954970996]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_array_rank4(self):
      arg0=Data(-0.142135287883,self.functionspace)
      arg1=numpy.array([[[[-0.3214758956274788, -0.095246696795476593, 0.88428285109896976, 0.16221904765171113], 
[0.2694361196789572, -0.88687321541037512, 0.02982831555263199, 0.36044697686186544], [0.089838925108686407, 
0.1168644540354129, -0.41060659412622824, 0.51404357016969393]], [[-0.7228284299326766, -0.74820755507542192, 
-0.53859530054671856, -0.10221620366876616], [0.59211782258097068, -0.9868578737653857, 0.51866928562956849, 
-0.47068533837586557], [0.54008589927595518, 0.30880228975280732, 0.011323333079816589, -0.23519250089433008]]], 
[[[-0.88835060064517424, 0.97805478155238479, 0.044980281282704704, 0.54172810913588032], [0.50306412490394248, 
-0.9089797473431025, -0.65257701723518702, -0.81985356457976932], [-0.55532857771618849, 0.90847532179586454, 
0.73479685498652625, -0.95360986864558894]], [[-0.070414261238669784, -0.55257468908581853, -0.8087953736998108, 
-0.88978135385971924], [0.30925435765744647, 0.011985479496297913, -0.94329973754759822, -0.78626227567625295], 
[-0.80368619190217405, -0.0018149953637613958, -0.64327784734883697, -0.70628255506304627]]], [[[0.21889092381937481, 
0.95982372912792258, -0.51545817953418194, -0.79191496743321532], [0.24699184622858739, -0.97716194806902634, 
-0.74023402440265662, -0.68483112762279252], [-0.80837443342334714, 0.39602906627714529, -0.93722656251851855, 
-0.36568401963574959]], [[-0.062719970600047636, -0.75215102861789918, 0.80346816982468328, -0.03123166762679852], 
[0.87032572315546886, -0.26645188225830663, 0.16577077009238161, -0.42846478915247221], [-0.83350849887379552, 
-0.09057758663591553, -0.72415237677358313, -0.11083229941883377]]]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[-0.14213528788334617, -0.095246696795476593, 0.88428285109896976, 0.16221904765171113], 
[0.2694361196789572, -0.14213528788334617, 0.02982831555263199, 0.36044697686186544], [0.089838925108686407, 
0.1168644540354129, -0.14213528788334617, 0.51404357016969393]], [[-0.14213528788334617, -0.14213528788334617, 
-0.14213528788334617, -0.10221620366876616], [0.59211782258097068, -0.14213528788334617, 0.51866928562956849, 
-0.14213528788334617], [0.54008589927595518, 0.30880228975280732, 0.011323333079816589, -0.14213528788334617]]], 
[[[-0.14213528788334617, 0.97805478155238479, 0.044980281282704704, 0.54172810913588032], [0.50306412490394248, 
-0.14213528788334617, -0.14213528788334617, -0.14213528788334617], [-0.14213528788334617, 0.90847532179586454, 
0.73479685498652625, -0.14213528788334617]], [[-0.070414261238669784, -0.14213528788334617, -0.14213528788334617, 
-0.14213528788334617], [0.30925435765744647, 0.011985479496297913, -0.14213528788334617, -0.14213528788334617], 
[-0.14213528788334617, -0.0018149953637613958, -0.14213528788334617, -0.14213528788334617]]], [[[0.21889092381937481, 
0.95982372912792258, -0.14213528788334617, -0.14213528788334617], [0.24699184622858739, -0.14213528788334617, 
-0.14213528788334617, -0.14213528788334617], [-0.14213528788334617, 0.39602906627714529, -0.14213528788334617, 
-0.14213528788334617]], [[-0.062719970600047636, -0.14213528788334617, 0.80346816982468328, -0.03123166762679852], 
[0.87032572315546886, -0.14213528788334617, 0.16577077009238161, -0.14213528788334617], [-0.14213528788334617, 
-0.09057758663591553, -0.14213528788334617, -0.11083229941883377]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank4_array_rank4(self):
      arg0=Data(numpy.array([[[[0.46361503964213124, 0.75212103924315676, 0.1517014593635575, 0.63148557170685327], 
[-0.9875146867756015, 0.42928007237587473, 0.85118517914010172, -0.15509137168645615], [0.15514320286516092, 
0.0005342941540116275, -0.89094877398399008, 0.908865463056445]], [[0.45054003146551436, -0.23123539736983156, 
-0.076540616512914772, 0.10502324913621375], [-0.0071973112790566507, -0.81620273555925849, -0.79541946133102126, 
0.26480060384338566], [-0.45249656775626268, 0.010265849576416963, -0.9086687652789831, -0.71275837340643822]]], 
[[[0.67609301986855086, 0.37746787067053145, -0.42950164086299925, -0.034365339088483715], [0.3373674099598567, 
0.18896844133041868, 0.93458147614245779, -0.40631016965933098], [-0.19339720965663387, 0.66853942853358883, 
0.56086919503548471, 0.49829638104085539]], [[-0.67730793306659365, -0.93402109411008638, -0.53909789907854089, 
-0.11494460733501222], [0.47127259923995135, 0.40243458962316581, 0.65232695613202951, 0.87100864021002966], 
[-0.62331578035775403, -0.78832076452515865, 0.98453152930704513, -0.32270285556445955]]], [[[0.030889419318346212, 
-0.74436621353272381, -0.41393005003246119, -0.27459836901005485], [0.2727885358054869, 0.39114640764062014, 
-0.62734280749087201, -0.68278841829167125], [0.45687670732427632, -0.41786887648402415, 0.14358785951974951, 
0.20168949679051473]], [[-0.61804283685818384, -0.74947293782095592, 0.47882556396076903, -0.38116235218691275], 
[-0.16868935807808083, -0.31479531035901021, -0.629767051514182, 0.40666969603452707], [0.47141425542113868, 
0.42602189213064201, -0.42510981826215244, 0.35951635388005387]]]]),self.functionspace)
      arg1=numpy.array([[[[-0.075878674569879179, -0.39617760413616954, -0.37441304528755759, 0.059028750333119939], 
[-0.45199615428664996, -0.67982487852801032, -0.43878514017544568, -0.2306154287295592], [-0.028641567153798952, 
0.20315619172000665, 0.25175605070668317, -0.62452580082432019]], [[-0.5510023141107756, 0.47087051768615695, 
-0.20141109488796705, -0.75776728595767517], [0.13680386381556775, -0.43979686709904531, -0.97978303421397572, 
0.84162201515236612], [-0.26616111363933515, -0.32677302519872753, -0.74540501454095254, -0.31553251505895985]]], 
[[[0.0019165847734914365, -0.34429378460591642, -0.6920165333737931, 0.54536804846096953], [-0.21558916120957838, 
0.60934895052910298, -0.72555820711554286, 0.81566556247952771], [0.28566419298828816, -0.94399443344238465, 
-0.11385943971564827, 0.0098538097357632193]], [[0.23053049901245104, -0.63018177830499922, -0.58754298214345324, 
0.32371305451881383], [0.34351012374192247, 0.50137166714297887, 0.95947548444362751, 0.42833198450872101], 
[-0.72876678207075196, -0.016030958882900848, 0.31136367053725356, 0.4736018523880543]]], [[[-0.49061444452775582, 
0.81114387212475636, 0.38995779155236199, 0.68167382208379479], [-0.99234885343352053, 0.54948405195911665, 
0.67748560641952316, 0.12361771392509757], [0.74957930366266456, 0.29970297400899271, 0.3159611266551301, 
-0.42046308727640236]], [[0.20950673516815188, -0.088341400215833366, 0.9638973190702973, 0.68199731178218692], 
[-0.57396477728598105, -0.89652900611840525, 0.019727515752113067, 0.57108671455067683], [0.70179048884165618, 
-0.72010473438276934, 0.82364341343487069, 0.81816644420876461]]]])
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.46361503964213124, 0.75212103924315676, 0.1517014593635575, 0.63148557170685327], 
[-0.45199615428664996, 0.42928007237587473, 0.85118517914010172, -0.15509137168645615], [0.15514320286516092, 
0.20315619172000665, 0.25175605070668317, 0.908865463056445]], [[0.45054003146551436, 0.47087051768615695, 
-0.076540616512914772, 0.10502324913621375], [0.13680386381556775, -0.43979686709904531, -0.79541946133102126, 
0.84162201515236612], [-0.26616111363933515, 0.010265849576416963, -0.74540501454095254, -0.31553251505895985]]], 
[[[0.67609301986855086, 0.37746787067053145, -0.42950164086299925, 0.54536804846096953], [0.3373674099598567, 
0.60934895052910298, 0.93458147614245779, 0.81566556247952771], [0.28566419298828816, 0.66853942853358883, 0.56086919503548471, 
0.49829638104085539]], [[0.23053049901245104, -0.63018177830499922, -0.53909789907854089, 0.32371305451881383], 
[0.47127259923995135, 0.50137166714297887, 0.95947548444362751, 0.87100864021002966], [-0.62331578035775403, 
-0.016030958882900848, 0.98453152930704513, 0.4736018523880543]]], [[[0.030889419318346212, 0.81114387212475636, 
0.38995779155236199, 0.68167382208379479], [0.2727885358054869, 0.54948405195911665, 0.67748560641952316, 0.12361771392509757], 
[0.74957930366266456, 0.29970297400899271, 0.3159611266551301, 0.20168949679051473]], [[0.20950673516815188, 
-0.088341400215833366, 0.9638973190702973, 0.68199731178218692], [-0.16868935807808083, -0.31479531035901021, 
0.019727515752113067, 0.57108671455067683], [0.70179048884165618, 0.42602189213064201, 0.82364341343487069, 
0.81816644420876461]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_constData_rank4(self):
      arg0=Data(-0.893686645203,self.functionspace)
      arg1=Data(numpy.array([[[[0.61638843267588261, -0.97285254339522864, 0.39019707596035991, -0.38329856435815723], 
[-0.39669633259426185, -0.43633677524387782, -0.062118904024617283, 0.52486356486440955], [0.59806243722015462, 
-0.95562790660740671, 0.89716362287428741, 0.25616048508546663]], [[0.24435399713825645, 0.58840030186856374, 
-0.11669001468268436, -0.10473414559607241], [0.53850954948156304, -0.21927908342820301, 0.5308446295654845, 
-0.14536822578932429], [0.90594574080059331, 0.77607397603230743, 0.34830152058308195, 0.53599148614980741]]], 
[[[0.84897407852148055, -0.54152647293785039, 0.93603932280166657, 0.8311250335877951], [-0.048354616437206177, 
0.82001605920812848, 0.92696057539180776, -0.08827000636161686], [0.020618916247708174, 0.98375130765872387, 
0.8200458168780016, -0.053631253048684435]], [[-0.37789023552666512, 0.052215678410592714, -0.18675922076149987, 
0.43345944086720611], [-0.30827748070532701, -0.66668820782501603, 0.084841064566959279, -0.95519159649105756], 
[-0.6725560045726664, -0.53620416688473016, -0.25767276940639294, 0.58749559487682013]]], [[[-0.19257594002810241, 
0.72549845431588622, 0.39704918849500959, 0.61577341276098552], [0.70773259021309665, -0.22540767589709043, 
-0.031031950353477278, -0.33370198061247858], [-0.8280003728298464, -0.81439343811994092, 0.95962098528552975, 
0.50268932526386201]], [[-0.37240249982818407, -0.92063840650310702, 0.12378723541570058, -0.75766552223525885], 
[-0.8507935520807457, 0.40017389307358187, -0.23436141671947053, 0.5452149596180087], [0.49123509401676135, 
-0.2193283129711836, 0.29665843719288354, -0.61641191860539335]]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.61638843267588261, -0.89368664520320795, 0.39019707596035991, -0.38329856435815723], 
[-0.39669633259426185, -0.43633677524387782, -0.062118904024617283, 0.52486356486440955], [0.59806243722015462, 
-0.89368664520320795, 0.89716362287428741, 0.25616048508546663]], [[0.24435399713825645, 0.58840030186856374, 
-0.11669001468268436, -0.10473414559607241], [0.53850954948156304, -0.21927908342820301, 0.5308446295654845, 
-0.14536822578932429], [0.90594574080059331, 0.77607397603230743, 0.34830152058308195, 0.53599148614980741]]], 
[[[0.84897407852148055, -0.54152647293785039, 0.93603932280166657, 0.8311250335877951], [-0.048354616437206177, 
0.82001605920812848, 0.92696057539180776, -0.08827000636161686], [0.020618916247708174, 0.98375130765872387, 
0.8200458168780016, -0.053631253048684435]], [[-0.37789023552666512, 0.052215678410592714, -0.18675922076149987, 
0.43345944086720611], [-0.30827748070532701, -0.66668820782501603, 0.084841064566959279, -0.89368664520320795], 
[-0.6725560045726664, -0.53620416688473016, -0.25767276940639294, 0.58749559487682013]]], [[[-0.19257594002810241, 
0.72549845431588622, 0.39704918849500959, 0.61577341276098552], [0.70773259021309665, -0.22540767589709043, 
-0.031031950353477278, -0.33370198061247858], [-0.8280003728298464, -0.81439343811994092, 0.95962098528552975, 
0.50268932526386201]], [[-0.37240249982818407, -0.89368664520320795, 0.12378723541570058, -0.75766552223525885], 
[-0.8507935520807457, 0.40017389307358187, -0.23436141671947053, 0.5452149596180087], [0.49123509401676135, 
-0.2193283129711836, 0.29665843719288354, -0.61641191860539335]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank4_constData_rank4(self):
      arg0=Data(numpy.array([[[[0.96485565509648352, 0.53935464926562759, 0.53077936146728266, -0.73952245406800854], 
[-0.60881766722021902, 0.66032495037649741, 0.93304470628387515, -0.29318048118189055], [-0.66535631878242762, 
0.86695831494535347, -0.14571574174128665, -0.33755901373168684]], [[0.10130701439391632, -0.17559510038495674, 
0.48296274631000302, -0.57408428494447028], [-0.27410670377127633, 0.17219315446797312, -0.60634684633721503, 
0.20140988897507128], [0.063883370598835532, 0.062627983781888386, -0.2846290129863831, 0.15747009011324331]]], 
[[[-0.33786498745323823, -0.77773342159826253, -0.43944390559155377, 0.30699880066796359], [0.45721299355990963, 
-0.93225225636245979, 0.24951116488168035, -0.60399840561802765], [-0.98061260494043778, -0.75645258975794905, 
0.27958959667741956, 0.087975846205211594]], [[0.66543448707063324, 0.85514480449431862, 0.82014945665654015, 
0.81505258529464553], [0.36193943466925615, -0.43856446384821646, -0.28386557764570841, -0.44311355588507073], 
[0.8203258501802293, 0.88410934322225532, -0.42070109791391741, 0.095773374554660284]]], [[[-0.27073826415250957, 
0.66740636068176817, -0.48896464867053013, -0.94994837491554485], [0.6540939867398663, 0.3623199797026595, 
-0.46325457322592967, -0.018711908974897273], [-0.35561156732560661, 0.41410104364264733, -0.17250699705434713, 
-0.25548148839396423]], [[-0.52110139543812939, -0.48803425309908843, 0.96222057216100421, -0.53953717021125525], 
[-0.50454727047867465, 0.66698964265585015, -0.15287504292958864, 0.50355653533991096], [0.83323461461179282, 
0.62448988217093437, 0.4654771267287694, 0.72161921224557446]]]]),self.functionspace)
      arg1=Data(numpy.array([[[[-0.57504027263496882, -0.70962427655760041, 0.5000686248515589, 0.22836735474708636], 
[-0.2456640208529155, 0.55145194339939985, -0.18188128973631157, -0.010052160520559017], [-0.67026258406375816, 
-0.097417557973243252, 0.78510475463922025, -0.96873477555278131]], [[-0.10002520261013537, 0.63100857099364571, 
-0.37013228408685617, -0.59175793402432819], [0.77337490554492994, 0.46196627477364838, -0.64798814992967224, 
0.60516599121325076], [-0.0017246851860892587, -0.30784264595534161, -0.65209147103284981, -0.88757120178936799]]], 
[[[0.80040151821525596, 0.43450571239861646, 0.0078842302106048034, -0.55464057489099483], [0.84844302837406849, 
0.48335761669393862, 0.018983703352744907, 0.89616530091553726], [-0.57373660097290591, -0.84825568142179031, 
-0.35092918722312461, 0.92340153576769501]], [[-0.85244829995021609, 0.095947521825416882, -0.18971802853754727, 
-0.65909953891892914], [0.11749664922727288, 0.13980571669569475, -0.79988329726892982, 0.055016274898991968], 
[0.35743533014846873, 0.27824733541877245, 0.42999566458196914, 0.3568350967844669]]], [[[0.99608799667364067, 
-0.5403793610139318, -0.12575108132177171, 0.41721666984548333], [0.875733275967848, 0.9523625165119125, -0.20675560052583641, 
-0.41564394921784187], [-0.84876383654236665, 0.62662285663526829, 0.49660771467477982, 0.5918409282574939]], 
[[-0.4729461618005697, -0.64161728948155772, 0.65333699389309863, 0.49984898391509858], [0.43829686077623897, 
0.33192534406508289, 0.61100656789543994, 0.073960447209926006], [0.78875551306588365, -0.25000292972836191, 
0.99289944129181062, -0.54605870017388525]]]]),self.functionspace)
      res=maximum(arg0,arg1)
      ref=Data(numpy.array([[[[0.96485565509648352, 0.53935464926562759, 0.53077936146728266, 0.22836735474708636], 
[-0.2456640208529155, 0.66032495037649741, 0.93304470628387515, -0.010052160520559017], [-0.66535631878242762, 
0.86695831494535347, 0.78510475463922025, -0.33755901373168684]], [[0.10130701439391632, 0.63100857099364571, 
0.48296274631000302, -0.57408428494447028], [0.77337490554492994, 0.46196627477364838, -0.60634684633721503, 
0.60516599121325076], [0.063883370598835532, 0.062627983781888386, -0.2846290129863831, 0.15747009011324331]]], 
[[[0.80040151821525596, 0.43450571239861646, 0.0078842302106048034, 0.30699880066796359], [0.84844302837406849, 
0.48335761669393862, 0.24951116488168035, 0.89616530091553726], [-0.57373660097290591, -0.75645258975794905, 
0.27958959667741956, 0.92340153576769501]], [[0.66543448707063324, 0.85514480449431862, 0.82014945665654015, 
0.81505258529464553], [0.36193943466925615, 0.13980571669569475, -0.28386557764570841, 0.055016274898991968], 
[0.8203258501802293, 0.88410934322225532, 0.42999566458196914, 0.3568350967844669]]], [[[0.99608799667364067, 
0.66740636068176817, -0.12575108132177171, 0.41721666984548333], [0.875733275967848, 0.9523625165119125, -0.20675560052583641, 
-0.018711908974897273], [-0.35561156732560661, 0.62662285663526829, 0.49660771467477982, 0.5918409282574939]], 
[[-0.4729461618005697, -0.48803425309908843, 0.96222057216100421, 0.49984898391509858], [0.43829686077623897, 
0.66698964265585015, 0.61100656789543994, 0.50355653533991096], [0.83323461461179282, 0.62448988217093437, 0.99289944129181062, 
0.72161921224557446]]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank0_expandedData_rank4(self):
      arg0=Data(0.216711670991,self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-0.81879161478097595, 0.23338466982083461, -0.037340658437786667, 0.53960130549007368], 
[-0.8433547141558082, 0.51986550271704757, -0.22844534933352723, -0.89465705240856908], [-0.11648963460000861, 
-0.96923924737633538, 0.072867751072468012, 0.65625904184925732]], [[0.27849706772004201, -0.90647830830724296, 
0.96202673895146806, 0.22494056179803157], [-0.2684069854476685, -0.42191439566230504, -0.95625432477352734, 
0.72510262095158207], [-0.036924511893698497, -0.93553684683379412, -0.76943122405185593, 0.42971341646797767]]], 
[[[0.89494738055097778, 0.46650780268669378, 0.30405300775028121, -0.084542513395697849], [-0.027722135654347779, 
-0.59483396322557391, -0.028304510159490759, -0.88731646326202407], [0.34624637943920167, -0.39682309028436302, 
0.89367998074843014, 0.56073508085183299]], [[0.63023798830300204, 0.18652798267686688, -0.3328850324391277, 
-0.21182328663126193], [-0.53239281368561087, 0.22479105625018425, 0.90626113618872939, -7.4251614887721473e-05], 
[-0.98623780619390233, -0.5666180903186695, -0.30927062686357099, -0.79727648923625138]]], [[[0.59536350392532866, 
0.22068473036479674, -0.66623026326771462, 0.10843352912766102], [0.52432284313983835, -0.3431109607082834, 
-0.23886295117126921, -0.8119295310297503], [-0.54397735011730486, -0.7068782977204644, 0.71740443446430424, 
-0.31144890281105586]], [[-0.32351391151939302, -0.35563858193171405, 0.65601036166328086, -0.032917953499258745], 
[0.45870081001448093, 0.3006509634165655, -0.15553712418631416, 0.85594927731050774], [-0.84241444057987591, 
-0.64337027261577129, 0.32134231645586109, -0.39606743840399128]]]])+(1.-msk_arg1)*numpy.array([[[[0.55712979668912488, 
-0.80066523551532121, -0.48656514655750849, 0.76097556013292111], [-0.13438565652155154, -0.94821563688317712, 
-0.31012625194007182, -0.84105859136182715], [-0.20208116248301411, -0.50024164072503563, 0.87481453005108034, 
0.13755575289800137]], [[-0.085598442928114249, -0.45841913232395637, -0.18238598324276789, 0.62303900657350142], 
[-0.79318456738858467, 0.34558408974177413, -0.8550879149504329, 0.8287523579134235], [-0.40328058936497357, 
-0.83102127169123086, -0.21653057906368112, -0.76688016358972666]]], [[[0.68068340217757584, 0.57860355650918649, 
-0.19514598498967861, 0.35797229868730107], [-0.940274604582233, -0.52273338186671348, -0.37314192021830439, 
0.10804775393755039], [-0.20074283030723383, 0.95964123909155385, -0.98950080490821501, -0.58835700500290256]], 
[[0.99348893772020896, 0.85220955503331552, 0.41443782029517151, 0.83659124171927268], [-0.65635663807364564, 
0.70812110348599355, 0.13053831014253792, 0.91230146888102226], [0.19243965515044836, 0.96567248884233092, 
-0.28016453566744892, -0.65082014990966219]]], [[[-0.013133698627046275, -0.9481821696335877, 0.63769890160711773, 
-0.35875665803034429], [-0.65944938772191586, 0.01066509243901459, 0.17762038374459466, -0.21321783232210256], 
[0.99670034374764271, 0.67813675209818114, 0.37840221867825585, 0.45475310270530489]], [[0.29249236353371622, 
-0.51219658392539991, 0.22458795099500506, -0.29060307474253499], [-0.86387736284336181, -0.17064143853256808, 
-0.45438045421185458, 0.9948613071200707], [-0.38789172342813538, 0.052527231540111297, 0.81849101524424928, 
0.43381945218064333]]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.21671167099079991, 0.23338466982083461, 0.21671167099079991, 0.53960130549007368], 
[0.21671167099079991, 0.51986550271704757, 0.21671167099079991, 0.21671167099079991], [0.21671167099079991, 
0.21671167099079991, 0.21671167099079991, 0.65625904184925732]], [[0.27849706772004201, 0.21671167099079991, 
0.96202673895146806, 0.22494056179803157], [0.21671167099079991, 0.21671167099079991, 0.21671167099079991, 
0.72510262095158207], [0.21671167099079991, 0.21671167099079991, 0.21671167099079991, 0.42971341646797767]]], 
[[[0.89494738055097778, 0.46650780268669378, 0.30405300775028121, 0.21671167099079991], [0.21671167099079991, 
0.21671167099079991, 0.21671167099079991, 0.21671167099079991], [0.34624637943920167, 0.21671167099079991, 0.89367998074843014, 
0.56073508085183299]], [[0.63023798830300204, 0.21671167099079991, 0.21671167099079991, 0.21671167099079991], 
[0.21671167099079991, 0.22479105625018425, 0.90626113618872939, 0.21671167099079991], [0.21671167099079991, 
0.21671167099079991, 0.21671167099079991, 0.21671167099079991]]], [[[0.59536350392532866, 0.22068473036479674, 
0.21671167099079991, 0.21671167099079991], [0.52432284313983835, 0.21671167099079991, 0.21671167099079991, 
0.21671167099079991], [0.21671167099079991, 0.21671167099079991, 0.71740443446430424, 0.21671167099079991]], 
[[0.21671167099079991, 0.21671167099079991, 0.65601036166328086, 0.21671167099079991], [0.45870081001448093, 
0.3006509634165655, 0.21671167099079991, 0.85594927731050774], [0.21671167099079991, 0.21671167099079991, 0.32134231645586109, 
0.21671167099079991]]]])+(1.-msk_ref)*numpy.array([[[[0.55712979668912488, 0.21671167099079991, 0.21671167099079991, 
0.76097556013292111], [0.21671167099079991, 0.21671167099079991, 0.21671167099079991, 0.21671167099079991], 
[0.21671167099079991, 0.21671167099079991, 0.87481453005108034, 0.21671167099079991]], [[0.21671167099079991, 
0.21671167099079991, 0.21671167099079991, 0.62303900657350142], [0.21671167099079991, 0.34558408974177413, 0.21671167099079991, 
0.8287523579134235], [0.21671167099079991, 0.21671167099079991, 0.21671167099079991, 0.21671167099079991]]], 
[[[0.68068340217757584, 0.57860355650918649, 0.21671167099079991, 0.35797229868730107], [0.21671167099079991, 
0.21671167099079991, 0.21671167099079991, 0.21671167099079991], [0.21671167099079991, 0.95964123909155385, 0.21671167099079991, 
0.21671167099079991]], [[0.99348893772020896, 0.85220955503331552, 0.41443782029517151, 0.83659124171927268], 
[0.21671167099079991, 0.70812110348599355, 0.21671167099079991, 0.91230146888102226], [0.21671167099079991, 
0.96567248884233092, 0.21671167099079991, 0.21671167099079991]]], [[[0.21671167099079991, 0.21671167099079991, 
0.63769890160711773, 0.21671167099079991], [0.21671167099079991, 0.21671167099079991, 0.21671167099079991, 
0.21671167099079991], [0.99670034374764271, 0.67813675209818114, 0.37840221867825585, 0.45475310270530489]], 
[[0.29249236353371622, 0.21671167099079991, 0.22458795099500506, 0.21671167099079991], [0.21671167099079991, 
0.21671167099079991, 0.21671167099079991, 0.9948613071200707], [0.21671167099079991, 0.21671167099079991, 0.81849101524424928, 
0.43381945218064333]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_constData_rank4_expandedData_rank4(self):
      arg0=Data(numpy.array([[[[0.9662251484946478, -0.26481381091852141, -0.066210160157844644, 0.40117951882974068], 
[0.2455027289465177, 0.67324997630420214, 0.79365479453824861, -0.037015541405008623], [-0.86030987461175457, 
-0.45616271741243231, 0.39179913013670342, -0.10731696095517629]], [[-0.19449522639452876, -0.67751174669485992, 
0.28984158743095256, 0.67531831661784203], [-0.49980904983410723, -0.02577831866914404, 0.94104429684279367, 
0.74763465415851904], [-0.28985388443946092, -0.90882129286702029, -0.90352359388616366, 0.0516463980384263]]], 
[[[-0.99639029481638874, 0.23581131279953027, 0.73436613098786441, 0.89956373143685453], [0.70388018442763145, 
0.079895063328745408, -0.75367452457404527, 0.56811816966845319], [0.90284242548995652, 0.66397516482541619, 
-0.99647484543635345, 0.79709159927353346]], [[0.70723973142155505, 0.27467701034793879, -0.79995723813418396, 
-0.24824137685387848], [0.56082965044735578, 0.73559725370917817, 0.62475940100976302, 0.030745279822806282], 
[-0.36016041397070708, 0.7839330226446426, -0.69945965257975473, -0.89855265013928132]]], [[[0.33431193949654969, 
-0.46895691087357716, 0.94874414920913108, 0.35408330328067761], [-0.56725159045543427, -0.11646626979600483, 
-0.27664126953599744, -0.62696292984368651], [0.15060075336069922, -0.16895232031523566, -0.83212543422584218, 
0.37516376063681012]], [[-0.21699302716045343, -0.24013090348080812, 0.73993829114859677, -0.63476480682303671], 
[-0.34968439152048214, 0.46819693714489508, 0.62165805854076095, -0.3791178743428667], [-0.41700973768159155, 
0.79899832669176507, -0.86119824926464239, 0.46575159472502037]]]]),self.functionspace)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-0.31931488193208657, -0.28905671753527695, 0.85561932051536327, 0.43258332189744197], 
[-0.022093948889682302, -0.87293800937103039, 0.56614871319438853, 0.86910428719314292], [0.68092117827772292, 
0.53186005878844611, 0.28162238085357139, 0.97361457365289072]], [[-0.79666566403785954, 0.37011155709186694, 
-0.68233116747040423, 0.89387098751104865], [0.42703403901311821, 0.89550507893657039, 0.84501584710121325, 
-0.66998942788818261], [0.53860906470280701, -0.094095450408768766, -0.35365080444144259, 0.4361075442378417]]], 
[[[-0.16099982532347235, -0.8651116694760872, 0.52385238626653319, -0.28446906365453173], [0.14671592161598701, 
0.71409402259173094, 0.3258739797599115, 0.70362092527950693], [-0.32717756310317503, -0.95828422715748962, 
0.68727489580226919, 0.15249900882725109]], [[0.22017885162589113, 0.80937795601648843, 0.35945120041216438, 
0.93900043111170084], [0.79190649660945756, 0.57203937052107467, 0.54292126003639929, 0.84571132339638178], 
[0.23760803956899457, 0.52713270420645153, -0.24975442892820454, -0.72715475618178438]]], [[[-0.84787512935758658, 
-0.62122949761588964, 0.68568216243118529, 0.34087206419913829], [-0.66077267614956314, 0.88586283683410505, 
0.48865077882965502, -0.2823801328248261], [0.24010264814531568, -0.77577498913372622, -0.55288042007565297, 
-0.34642896895972042]], [[-0.42554067083401659, 0.87743453278083261, 0.50397543167721515, 0.97556159346990001], 
[0.84530378175248244, 0.66561045498275107, 0.80033289693830301, 0.92226186452120906], [-0.4442108477506006, 
-0.52692217686024989, -0.85590042003205369, 0.43455207703783549]]]])+(1.-msk_arg1)*numpy.array([[[[0.21605231242464118, 
0.32170152972234489, 0.69540488109778908, 0.2547128534596883], [0.34796901405025005, -0.31607291081984523, 
-0.98038121857023519, -0.93906508361402574], [-0.28276602806278417, -0.62275848697078584, -0.26878568593014385, 
-0.22758564658522573]], [[0.85908277675070277, 0.70033299040044406, 0.93445045187257891, -0.75943811515314663], 
[0.50550467560364476, -0.89661957212631349, -0.20586932595053264, 0.74555292517790739], [-0.46974519139411886, 
-0.064153765304233668, -0.5695552407484461, -0.11199926962703399]]], [[[-0.046076882407277919, -0.4751273456616687, 
0.54211989560690821, -0.95839866060661283], [-0.34429450468712863, 0.9945935413627196, 0.12045306605821482, 
0.70763311498764381], [-0.93229388462703189, -0.8159573239034752, 0.43375216165599206, -0.58156200439008288]], 
[[-0.94359110518590672, 0.21676492225881594, -0.57979669038590131, 0.035244336548687416], [-0.13786684734995425, 
0.75386861643073622, 0.022898320967623453, 0.66532275876665459], [-0.30607908894667135, -0.066287086564079223, 
-0.64210438699958949, 0.58486901207821607]]], [[[0.69742093261447113, 0.77935504242461695, 0.28045816177452632, 
0.35225109286857292], [-0.96809103221557291, 0.77688458560421636, 0.92713086412266987, 0.49029829063110997], 
[-0.1905974439471938, 0.14508130554930188, 0.24154428575645048, 0.45490680510213011]], [[-0.021068579578790381, 
-0.055644755704836957, -0.71431963870922144, -0.65498904995738161], [0.87855331454880914, 0.53311774880814933, 
0.18160214068170566, -0.33338460760283883], [-0.49933663079121393, -0.30118428496778038, 0.86142651573184215, 
-0.99665689200339136]]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.9662251484946478, -0.26481381091852141, 0.85561932051536327, 0.43258332189744197], 
[0.2455027289465177, 0.67324997630420214, 0.79365479453824861, 0.86910428719314292], [0.68092117827772292, 0.53186005878844611, 
0.39179913013670342, 0.97361457365289072]], [[-0.19449522639452876, 0.37011155709186694, 0.28984158743095256, 
0.89387098751104865], [0.42703403901311821, 0.89550507893657039, 0.94104429684279367, 0.74763465415851904], 
[0.53860906470280701, -0.094095450408768766, -0.35365080444144259, 0.4361075442378417]]], [[[-0.16099982532347235, 
0.23581131279953027, 0.73436613098786441, 0.89956373143685453], [0.70388018442763145, 0.71409402259173094, 0.3258739797599115, 
0.70362092527950693], [0.90284242548995652, 0.66397516482541619, 0.68727489580226919, 0.79709159927353346]], 
[[0.70723973142155505, 0.80937795601648843, 0.35945120041216438, 0.93900043111170084], [0.79190649660945756, 
0.73559725370917817, 0.62475940100976302, 0.84571132339638178], [0.23760803956899457, 0.7839330226446426, -0.24975442892820454, 
-0.72715475618178438]]], [[[0.33431193949654969, -0.46895691087357716, 0.94874414920913108, 0.35408330328067761], 
[-0.56725159045543427, 0.88586283683410505, 0.48865077882965502, -0.2823801328248261], [0.24010264814531568, 
-0.16895232031523566, -0.55288042007565297, 0.37516376063681012]], [[-0.21699302716045343, 0.87743453278083261, 
0.73993829114859677, 0.97556159346990001], [0.84530378175248244, 0.66561045498275107, 0.80033289693830301, 
0.92226186452120906], [-0.41700973768159155, 0.79899832669176507, -0.85590042003205369, 
0.46575159472502037]]]])+(1.-msk_ref)*numpy.array([[[[0.9662251484946478, 0.32170152972234489, 0.69540488109778908, 
0.40117951882974068], [0.34796901405025005, 0.67324997630420214, 0.79365479453824861, -0.037015541405008623], 
[-0.28276602806278417, -0.45616271741243231, 0.39179913013670342, -0.10731696095517629]], [[0.85908277675070277, 
0.70033299040044406, 0.93445045187257891, 0.67531831661784203], [0.50550467560364476, -0.02577831866914404, 
0.94104429684279367, 0.74763465415851904], [-0.28985388443946092, -0.064153765304233668, -0.5695552407484461, 
0.0516463980384263]]], [[[-0.046076882407277919, 0.23581131279953027, 0.73436613098786441, 0.89956373143685453], 
[0.70388018442763145, 0.9945935413627196, 0.12045306605821482, 0.70763311498764381], [0.90284242548995652, 0.66397516482541619, 
0.43375216165599206, 0.79709159927353346]], [[0.70723973142155505, 0.27467701034793879, -0.57979669038590131, 
0.035244336548687416], [0.56082965044735578, 0.75386861643073622, 0.62475940100976302, 0.66532275876665459], 
[-0.30607908894667135, 0.7839330226446426, -0.64210438699958949, 0.58486901207821607]]], [[[0.69742093261447113, 
0.77935504242461695, 0.94874414920913108, 0.35408330328067761], [-0.56725159045543427, 0.77688458560421636, 
0.92713086412266987, 0.49029829063110997], [0.15060075336069922, 0.14508130554930188, 0.24154428575645048, 
0.45490680510213011]], [[-0.021068579578790381, -0.055644755704836957, 0.73993829114859677, -0.63476480682303671], 
[0.87855331454880914, 0.53311774880814933, 0.62165805854076095, -0.33338460760283883], [-0.41700973768159155, 
0.79899832669176507, 0.86142651573184215, 0.46575159472502037]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.318576793935)+(1.-msk_arg0)*(-0.920315318998)
      arg1=0.264453770493
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.318576793935)+(1.-msk_ref)*(0.264453770493)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank1_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.61984078319308633, -0.74126411397009218])+(1.-msk_arg0)*numpy.array([-0.36354960922197699, 
-0.45840304611368676])
      arg1=0.798333407454
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.79833340745371917, 0.79833340745371917])+(1.-msk_ref)*numpy.array([0.79833340745371917, 
0.79833340745371917])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank2_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[0.56027127038237556, -0.38092190445101282, -0.77591749903008322, -0.36171347397940457, 
-0.3091925404043383], [0.34776167092224908, -0.37902110057843474, 0.8566813057073317, -0.87799841344488661, 
0.61149184516634381], [0.11264186928512365, 0.34551824833197631, 0.4058993843871157, 0.42214558457038209, 
-0.88397115089856348], [0.57017035572765851, -0.17472794481465992, -0.37739320194928472, 0.24459442745468318, 
-0.0032651849035065794]])+(1.-msk_arg0)*numpy.array([[-0.63041245210130414, -0.40539069241410175, 0.95484833881413089, 
0.76528251808922398, -0.45472655311970289], [-0.34559052794751488, -0.91180569915010246, -0.08113468153701886, 
0.31746466254664774, 0.69806708955802943], [0.76702933573803689, -0.24228609093935383, 0.35224880600670572, 
0.47231857975553981, -0.88896833667212327], [-0.087726728409511789, -0.23153089968162344, 0.39249534318815371, 
0.23502532808353349, 0.68373987182917317]])
      arg1=0.989825243681
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 
0.98982524368093427], [0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 
0.98982524368093427], [0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 
0.98982524368093427], [0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 
0.98982524368093427]])+(1.-msk_ref)*numpy.array([[0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 
0.98982524368093427, 0.98982524368093427], [0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 
0.98982524368093427], [0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 
0.98982524368093427], [0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 0.98982524368093427, 
0.98982524368093427]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank3_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.028875141511675473, 0.59607526974242653], [-0.85743668105869553, 0.70579283933075554]], 
[[0.53533801908700807, -0.63679610683946919], [-0.079395571307905088, 0.48032940038802163]], [[0.62755828198212904, 
-0.43131717689352711], [-0.69602024353252068, -0.39729826147508174]], [[0.871577527593145, 0.13860228070825675], 
[-0.41566762577178307, 0.94629933788596232]], [[0.67599858677777047, 0.69484194011263689], [0.60029405821428594, 
0.28493844302972837]], [[-0.71484527552962907, 0.22995449367240717], [0.791550071610712, 
-0.87117943761246108]]])+(1.-msk_arg0)*numpy.array([[[0.78242950070530659, 0.014091229458666366], [0.25389636301837526, 
-0.32650460454438401]], [[-0.2924625700857808, 0.6366577134210214], [-0.93500357448199378, -0.040378324895707918]], 
[[-0.50017267851444713, -0.27455430871359354], [-0.54888413525358515, 0.39328658448761655]], [[0.32330904571084806, 
-0.58124263920494301], [0.15299156800247538, -0.28958015194809961]], [[-0.2894735013468257, 0.39047784429864718], 
[0.41819146976323429, 0.016153713585879137]], [[0.8615981317548056, -0.37210375853443756], [-0.84988813841877109, 
0.90373148295773231]]])
      arg1=0.832320572018
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.8323205720176674, 0.8323205720176674], [0.8323205720176674, 0.8323205720176674]], 
[[0.8323205720176674, 0.8323205720176674], [0.8323205720176674, 0.8323205720176674]], [[0.8323205720176674, 
0.8323205720176674], [0.8323205720176674, 0.8323205720176674]], [[0.871577527593145, 0.8323205720176674], [0.8323205720176674, 
0.94629933788596232]], [[0.8323205720176674, 0.8323205720176674], [0.8323205720176674, 0.8323205720176674]], 
[[0.8323205720176674, 0.8323205720176674], [0.8323205720176674, 
0.8323205720176674]]])+(1.-msk_ref)*numpy.array([[[0.8323205720176674, 0.8323205720176674], [0.8323205720176674, 
0.8323205720176674]], [[0.8323205720176674, 0.8323205720176674], [0.8323205720176674, 0.8323205720176674]], 
[[0.8323205720176674, 0.8323205720176674], [0.8323205720176674, 0.8323205720176674]], [[0.8323205720176674, 
0.8323205720176674], [0.8323205720176674, 0.8323205720176674]], [[0.8323205720176674, 0.8323205720176674], [0.8323205720176674, 
0.8323205720176674]], [[0.8615981317548056, 0.8323205720176674], [0.8323205720176674, 0.90373148295773231]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank4_float_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-0.42251039851841288, -0.5086459170688129, -0.45492102664037537, 0.31392311416403995], 
[0.65608273573528852, -0.85710626049138106, 0.21505967960738959, 0.70718137746639131], [0.83790464665837794, 
-0.60856345438902792, -0.60079149562707634, -0.1076802420164078]], [[0.46732380623178282, -0.35500225168117239, 
0.25444444559611989, -0.48744596442284194], [0.90027866484871244, -0.31706597272192649, 0.59522974780728655, 
-0.47446522005210379], [-0.8633196459105501, 0.66961511233335314, 0.061156594494290673, 0.79799072926268422]]], 
[[[-0.76276876249566472, -0.099532420785620968, 0.10129392224865463, 0.39167447951002199], [-0.98828984273881293, 
-0.57023782466024131, -0.065377529996557548, -0.40442289015670507], [-0.82339803606408557, 0.014662422844692102, 
0.21649789675243114, 0.77505805175113651]], [[-0.76191062946085264, 0.42107637324614444, 0.92357976930849417, 
-0.83786675093267315], [0.52554082210275488, -0.8888682988825547, 0.40287390701474046, -0.52290176387004972], 
[-0.34201346434522772, 0.38682001181478975, -0.819612138653941, -0.94002052598078389]]], [[[-0.51734417834945168, 
-0.94482365146869829, 0.64319147138071275, -0.35722472189352694], [-0.94118399117998175, -0.28909676588864341, 
-0.77521699548798018, 0.57822873832654831], [0.89952167291261254, 0.6107886611155009, -0.013437891648716782, 
-0.474928738027242]], [[0.26286809808854472, 0.25318532614933087, -0.74577774138101782, -0.5726071651639919], 
[0.97839123557228058, 0.9460156616470814, -0.35659193985301241, 0.12569654130562768], [-0.99682390578515179, 
-0.60347225844308694, 0.39905112336897774, -0.6498254841152844]]]])+(1.-msk_arg0)*numpy.array([[[[-0.43200273953647184, 
0.38040962014976043, -0.6572991923365028, 0.45207958622632516], [-0.58836374817160375, 0.26862299998111294, 
0.21139361003683721, 0.3086343700658376], [-0.77309660882283571, -0.70430093120909287, -0.56474331239969677, 
-0.017208266779719983]], [[0.67436506707034893, -0.30858059192041742, -0.031394042016385626, -0.63378689685176881], 
[-0.66569084269765399, 0.97185177189397032, -0.58300442478147785, 0.70372679431572527], [-0.022365773489303686, 
0.85035019395735212, -0.97846447525858959, -0.64044194234297014]]], [[[-0.97775916672490459, -0.41628654841289214, 
-0.48245585793355605, -0.055238920243368517], [0.78673608724885491, -0.19279316198073482, -0.98976200752193, 
-0.47776310463081839], [0.84717802695622835, -0.68427094271230504, -0.4804198631320109, 0.96098954130826031]], 
[[0.53394958997155872, -0.22737455212877689, 0.30471343662030992, 0.18229541027562624], [0.37038417314131489, 
0.87665381096600825, 0.44054927441029701, 0.35339162806138336], [-0.17989219181739258, 0.71059747268101714, 
-0.98051690316527695, 0.27992594956317007]]], [[[0.52813647387794704, 0.024570478697439002, 0.93464353314847237, 
-0.84326079071506377], [-0.53538275657360623, 0.93198072028257628, -0.74578726744677581, -0.35281011313225541], 
[0.56547473195301534, 0.11350722177621453, 0.12064243147442166, -0.56343163545267116]], [[-0.048739251574649201, 
0.24929112934527575, 0.99164564508467357, -0.44909041546971151], [0.74190021533591577, 0.65438363932926258, 0.4220458996941816, 
-0.69453814291522642], [-0.30803415118241451, 0.24508007551774624, 0.53629014177782874, 0.31168691811329174]]]])
      arg1=0.734113764334
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], 
[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], [0.83790464665837794, 0.7341137643336908, 
0.7341137643336908, 0.7341137643336908]], [[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], 
[0.90027866484871244, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], [0.7341137643336908, 0.7341137643336908, 
0.7341137643336908, 0.79799072926268422]]], [[[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], 
[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], [0.7341137643336908, 0.7341137643336908, 
0.7341137643336908, 0.77505805175113651]], [[0.7341137643336908, 0.7341137643336908, 0.92357976930849417, 0.7341137643336908], 
[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], [0.7341137643336908, 0.7341137643336908, 
0.7341137643336908, 0.7341137643336908]]], [[[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], 
[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], [0.89952167291261254, 0.7341137643336908, 
0.7341137643336908, 0.7341137643336908]], [[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], 
[0.97839123557228058, 0.9460156616470814, 0.7341137643336908, 0.7341137643336908], [0.7341137643336908, 0.7341137643336908, 
0.7341137643336908, 0.7341137643336908]]]])+(1.-msk_ref)*numpy.array([[[[0.7341137643336908, 0.7341137643336908, 
0.7341137643336908, 0.7341137643336908], [0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], 
[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908]], [[0.7341137643336908, 0.7341137643336908, 
0.7341137643336908, 0.7341137643336908], [0.7341137643336908, 0.97185177189397032, 0.7341137643336908, 0.7341137643336908], 
[0.7341137643336908, 0.85035019395735212, 0.7341137643336908, 0.7341137643336908]]], [[[0.7341137643336908, 0.7341137643336908, 
0.7341137643336908, 0.7341137643336908], [0.78673608724885491, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], 
[0.84717802695622835, 0.7341137643336908, 0.7341137643336908, 0.96098954130826031]], [[0.7341137643336908, 0.7341137643336908, 
0.7341137643336908, 0.7341137643336908], [0.7341137643336908, 0.87665381096600825, 0.7341137643336908, 0.7341137643336908], 
[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908]]], [[[0.7341137643336908, 0.7341137643336908, 
0.93464353314847237, 0.7341137643336908], [0.7341137643336908, 0.93198072028257628, 0.7341137643336908, 0.7341137643336908], 
[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908]], [[0.7341137643336908, 0.7341137643336908, 
0.99164564508467357, 0.7341137643336908], [0.74190021533591577, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908], 
[0.7341137643336908, 0.7341137643336908, 0.7341137643336908, 0.7341137643336908]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.0285167629509)+(1.-msk_arg0)*(-0.0753791701623)
      arg1=numpy.array(0.369649061202)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.369649061202)+(1.-msk_ref)*(0.369649061202)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank1_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.32342379681372324, 0.54353331547928652])+(1.-msk_arg0)*numpy.array([0.22862832886262274, 
0.31483883730737539])
      arg1=numpy.array(0.626949968399)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.62694996839906447, 0.62694996839906447])+(1.-msk_ref)*numpy.array([0.62694996839906447, 
0.62694996839906447])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank2_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.71394651499733919, -0.92287830402130533, -0.90858139421470385, -0.13825598900369229, 
-0.080541309303345132], [-0.36949666654546265, 0.3898727048518853, -0.28621012865918916, -0.25638266490918404, 
0.20407905334345466], [0.15975716776790327, 0.86409710321773558, 0.71725217347095049, 0.20875761848553376, 
-0.24238991057422754], [-0.78556146630181956, -0.076076120078814924, 0.5830295620999455, 0.62973766057396441, 
-0.44550377230379912]])+(1.-msk_arg0)*numpy.array([[-0.88705425678644301, 0.55511600368349057, 0.50879851258581632, 
0.65698430346681214, -0.92944433639452151], [0.72904334195690801, -0.57898988094606052, 0.80587168656123565, 
0.75367205014945649, -0.51428464418517805], [-0.32501367205410969, -0.97880582883393408, 0.053009572718297759, 
0.094411264210482848, -0.63030483500737655], [-0.088813446479829139, -0.75746914620099148, -0.95633858232383417, 
0.67276534899551677, 0.15202939959842365]])
      arg1=numpy.array(-0.940581531438)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.71394651499733919, -0.92287830402130533, -0.90858139421470385, -0.13825598900369229, 
-0.080541309303345132], [-0.36949666654546265, 0.3898727048518853, -0.28621012865918916, -0.25638266490918404, 
0.20407905334345466], [0.15975716776790327, 0.86409710321773558, 0.71725217347095049, 0.20875761848553376, 
-0.24238991057422754], [-0.78556146630181956, -0.076076120078814924, 0.5830295620999455, 0.62973766057396441, 
-0.44550377230379912]])+(1.-msk_ref)*numpy.array([[-0.88705425678644301, 0.55511600368349057, 0.50879851258581632, 
0.65698430346681214, -0.92944433639452151], [0.72904334195690801, -0.57898988094606052, 0.80587168656123565, 
0.75367205014945649, -0.51428464418517805], [-0.32501367205410969, -0.94058153143782453, 0.053009572718297759, 
0.094411264210482848, -0.63030483500737655], [-0.088813446479829139, -0.75746914620099148, -0.94058153143782453, 
0.67276534899551677, 0.15202939959842365]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank3_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.18937114215158379, 0.97884605242301004], [-0.10174593321344649, -0.022379247683969705]], 
[[0.37143236733782325, 0.9436323979261656], [-0.1396616687095853, -0.36032181421557552]], [[-0.082984451143639948, 
0.68652743342457567], [-0.935628845473649, 0.58410565742224385]], [[-0.12311403457235026, 0.051753925365314313], 
[-0.19479735727282055, -0.97762777864901618]], [[-0.43740194607254024, 0.047952734299117505], [-0.83030948972383301, 
0.2720270466764092]], [[-0.13916673117721445, -0.38789957921110485], [-0.82523648008893313, 
-0.90290874072044436]]])+(1.-msk_arg0)*numpy.array([[[0.45196786956134583, 0.61219599413960424], [-0.065721595612748152, 
-0.26282718706162234]], [[-0.77537708674852834, 0.44532451113123472], [-0.069741033742395819, 0.85312396785377786]], 
[[0.77152934839087872, -0.89912864802471115], [-0.59570931290004303, -0.20558993214725563]], [[0.47614495941699153, 
-0.87515809599830097], [0.5635701134536959, 0.79401765404455427]], [[-0.22008420609503343, -0.65942311245997409], 
[0.9655251213411864, -0.30378791706072894]], [[0.44247117863110397, 0.10977233527612396], [0.050396835055231071, 
-0.17185541796448978]]])
      arg1=numpy.array(0.879188127312)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.87918812731204588, 0.97884605242301004], [0.87918812731204588, 0.87918812731204588]], 
[[0.87918812731204588, 0.9436323979261656], [0.87918812731204588, 0.87918812731204588]], [[0.87918812731204588, 
0.87918812731204588], [0.87918812731204588, 0.87918812731204588]], [[0.87918812731204588, 0.87918812731204588], 
[0.87918812731204588, 0.87918812731204588]], [[0.87918812731204588, 0.87918812731204588], [0.87918812731204588, 
0.87918812731204588]], [[0.87918812731204588, 0.87918812731204588], [0.87918812731204588, 
0.87918812731204588]]])+(1.-msk_ref)*numpy.array([[[0.87918812731204588, 0.87918812731204588], [0.87918812731204588, 
0.87918812731204588]], [[0.87918812731204588, 0.87918812731204588], [0.87918812731204588, 0.87918812731204588]], 
[[0.87918812731204588, 0.87918812731204588], [0.87918812731204588, 0.87918812731204588]], [[0.87918812731204588, 
0.87918812731204588], [0.87918812731204588, 0.87918812731204588]], [[0.87918812731204588, 0.87918812731204588], 
[0.9655251213411864, 0.87918812731204588]], [[0.87918812731204588, 0.87918812731204588], [0.87918812731204588, 
0.87918812731204588]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank4_array_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[0.44580962530016999, -0.69348488146163345, -0.34481050040016448, 0.59519412601278909], 
[-0.29649588803121163, -0.90433699382125621, 0.58229774413551061, -0.36778950106866248], [-0.367202105893363, 
-0.11391186768016848, 0.091423849403935442, -0.80223166648482014]], [[-0.80093892706918934, -0.29174863765813419, 
-0.16211197355096374, 0.075768040268325176], [0.68362107850802523, 0.30569303924697744, -0.27076611907986936, 
-0.32562966437337582], [0.32515793722254704, 0.077836894356146047, -0.37435906797035168, -0.35187117123705081]]], 
[[[-0.51098052012853468, -0.13010630276624302, 0.87102220930593499, -0.64012790280688536], [-0.21267317186303125, 
-0.88402284628937533, 0.02431388699609216, 0.06052877315768912], [-0.28034820201167854, 0.95297399082866185, 
-0.30163718369713011, -0.50183508057815374]], [[-0.43680686484138076, -0.82955436525397341, -0.30888908734602927, 
0.73714563499555674], [0.85936269560114997, -0.37602998325975867, -0.16401661856146643, 0.42565242322259933], 
[-0.27441411269419902, 0.29164819210771387, -0.85938658660708112, -0.99892556232036434]]], [[[-0.81554459002181479, 
-0.35510738606811576, -0.6024526618358339, -0.96050342654364496], [-0.56351464899865644, -0.15196137810022159, 
0.10412313747497359, 0.2755721298497178], [0.69743953672566561, -0.36580894827046495, 0.95677556962500332, 
-0.99552242481327524]], [[-0.11222970781965746, -0.33762110851060645, 0.27106365544031208, -0.29815607031304237], 
[-0.056603466192955532, -0.93774716471967889, 0.74223460094575988, 0.42675210408069519], [-0.73797087885832946, 
-0.14895862461116716, -0.026765745522702833, 0.86420158258370283]]]])+(1.-msk_arg0)*numpy.array([[[[0.19372975479304544, 
-0.66808663318684403, 0.77537567177849098, 0.81919260999807397], [-0.10116256869273443, -0.57520810468712269, 
0.15753839520629942, 0.79473404582834428], [-0.73645830163893855, 0.87616863458367211, -0.96733449165902963, 
-0.3516939203286118]], [[0.48878397140623631, 0.45609994336875803, 0.91436485388943489, -0.17781328406697527], 
[0.65055833166980692, 0.10117618648410831, -0.09041980822458906, 0.59283939012926501], [0.78272644881934017, 
0.25217910521330689, -0.84476748260357493, -0.42169419337471803]]], [[[-0.45223518020548759, 0.38375427581487509, 
-0.033829350614732911, 0.70903023968037648], [-0.80644295317701453, 0.54199751717049516, -0.3972649595244353, 
-0.26862061395744208], [0.163939148868391, -0.19776011299147966, -0.29279137734597294, 0.38651062526683488]], 
[[0.73784058524755514, -0.62987052734714788, 0.96978901651430682, 0.68202116917743494], [-0.9741599743353393, 
-0.70746264080763765, -0.73637201007369524, -0.36338439617661566], [0.5758662884613357, 0.58540767089470402, 
-0.40111460235649776, 0.061425771822538389]]], [[[0.90108401417416362, 0.72943001699832255, -0.75909447019101606, 
0.97176132609079291], [-0.31728686169672304, -0.40901210547887934, 0.96695071353736539, 0.032603047050764777], 
[-0.12055510695784455, 0.768561520375171, 0.948776815267379, -0.50392752129221652]], [[-0.87599954417448522, 
0.95279874663930531, 0.90744334941927685, -0.50378095934118838], [0.9824024122546311, 0.69765742163802824, 
-0.38287651073484019, 0.40060850054520558], [0.91703259523128899, -0.39809207284495951, -0.84284247078113972, 
0.36091797506547008]]]])
      arg1=numpy.array(-0.694396194523)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.44580962530016999, -0.69348488146163345, -0.34481050040016448, 0.59519412601278909], 
[-0.29649588803121163, -0.6943961945233823, 0.58229774413551061, -0.36778950106866248], [-0.367202105893363, 
-0.11391186768016848, 0.091423849403935442, -0.6943961945233823]], [[-0.6943961945233823, -0.29174863765813419, 
-0.16211197355096374, 0.075768040268325176], [0.68362107850802523, 0.30569303924697744, -0.27076611907986936, 
-0.32562966437337582], [0.32515793722254704, 0.077836894356146047, -0.37435906797035168, -0.35187117123705081]]], 
[[[-0.51098052012853468, -0.13010630276624302, 0.87102220930593499, -0.64012790280688536], [-0.21267317186303125, 
-0.6943961945233823, 0.02431388699609216, 0.06052877315768912], [-0.28034820201167854, 0.95297399082866185, 
-0.30163718369713011, -0.50183508057815374]], [[-0.43680686484138076, -0.6943961945233823, -0.30888908734602927, 
0.73714563499555674], [0.85936269560114997, -0.37602998325975867, -0.16401661856146643, 0.42565242322259933], 
[-0.27441411269419902, 0.29164819210771387, -0.6943961945233823, -0.6943961945233823]]], [[[-0.6943961945233823, 
-0.35510738606811576, -0.6024526618358339, -0.6943961945233823], [-0.56351464899865644, -0.15196137810022159, 
0.10412313747497359, 0.2755721298497178], [0.69743953672566561, -0.36580894827046495, 0.95677556962500332, 
-0.6943961945233823]], [[-0.11222970781965746, -0.33762110851060645, 0.27106365544031208, -0.29815607031304237], 
[-0.056603466192955532, -0.6943961945233823, 0.74223460094575988, 0.42675210408069519], [-0.6943961945233823, 
-0.14895862461116716, -0.026765745522702833, 0.86420158258370283]]]])+(1.-msk_ref)*numpy.array([[[[0.19372975479304544, 
-0.66808663318684403, 0.77537567177849098, 0.81919260999807397], [-0.10116256869273443, -0.57520810468712269, 
0.15753839520629942, 0.79473404582834428], [-0.6943961945233823, 0.87616863458367211, -0.6943961945233823, 
-0.3516939203286118]], [[0.48878397140623631, 0.45609994336875803, 0.91436485388943489, -0.17781328406697527], 
[0.65055833166980692, 0.10117618648410831, -0.09041980822458906, 0.59283939012926501], [0.78272644881934017, 
0.25217910521330689, -0.6943961945233823, -0.42169419337471803]]], [[[-0.45223518020548759, 0.38375427581487509, 
-0.033829350614732911, 0.70903023968037648], [-0.6943961945233823, 0.54199751717049516, -0.3972649595244353, 
-0.26862061395744208], [0.163939148868391, -0.19776011299147966, -0.29279137734597294, 0.38651062526683488]], 
[[0.73784058524755514, -0.62987052734714788, 0.96978901651430682, 0.68202116917743494], [-0.6943961945233823, 
-0.6943961945233823, -0.6943961945233823, -0.36338439617661566], [0.5758662884613357, 0.58540767089470402, 
-0.40111460235649776, 0.061425771822538389]]], [[[0.90108401417416362, 0.72943001699832255, -0.6943961945233823, 
0.97176132609079291], [-0.31728686169672304, -0.40901210547887934, 0.96695071353736539, 0.032603047050764777], 
[-0.12055510695784455, 0.768561520375171, 0.948776815267379, -0.50392752129221652]], [[-0.6943961945233823, 
0.95279874663930531, 0.90744334941927685, -0.50378095934118838], [0.9824024122546311, 0.69765742163802824, 
-0.38287651073484019, 0.40060850054520558], [0.91703259523128899, -0.39809207284495951, -0.6943961945233823, 
0.36091797506547008]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.88000228503)+(1.-msk_arg0)*(-0.20984850813)
      arg1=Data(0.659456392175,self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.88000228503)+(1.-msk_ref)*(0.659456392175)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank1_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.71272356771458245, -0.3611606406674901])+(1.-msk_arg0)*numpy.array([-0.82598998478380836, 
-0.42990113071867553])
      arg1=Data(0.178686493024,self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.17868649302435347, 0.17868649302435347])+(1.-msk_ref)*numpy.array([0.17868649302435347, 
0.17868649302435347])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank2_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[0.63593420649217469, 0.071252699353494764, 0.52730188526412869, 0.90707321993763812, 
-0.49826390881410676], [0.65603487673248218, -0.1221632218420734, -0.65505386558118928, -0.092624138241382603, 
-0.60221631581080404], [0.59133951441423194, 0.62499728002156107, 0.91980036751900829, -0.76085845352133918, 
-0.85482925516349528], [-0.080754965852170235, 0.044026814393698155, -0.39390549886182535, -0.39036051478539835, 
-0.22353088797836307]])+(1.-msk_arg0)*numpy.array([[-0.72561188067886895, 0.44063138647043476, -0.71568806092968162, 
0.79302562670403565, -0.7203898051760802], [-0.47457497511194724, 0.49128310742058168, 0.24651099047400171, 
-0.8094160968572508, -0.24832575278907898], [-0.38354631430621122, 0.21584585668083234, 0.88500663492033116, 
-0.17117462898833979, 0.91967256531048047], [0.53756830959719037, -0.037911466866273003, 0.014786652808508993, 
-0.43313218614069782, 0.83859088561946526]])
      arg1=Data(0.24971708422,self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.63593420649217469, 0.24971708421996541, 0.52730188526412869, 0.90707321993763812, 
0.24971708421996541], [0.65603487673248218, 0.24971708421996541, 0.24971708421996541, 0.24971708421996541, 
0.24971708421996541], [0.59133951441423194, 0.62499728002156107, 0.91980036751900829, 0.24971708421996541, 
0.24971708421996541], [0.24971708421996541, 0.24971708421996541, 0.24971708421996541, 0.24971708421996541, 
0.24971708421996541]])+(1.-msk_ref)*numpy.array([[0.24971708421996541, 0.44063138647043476, 0.24971708421996541, 
0.79302562670403565, 0.24971708421996541], [0.24971708421996541, 0.49128310742058168, 0.24971708421996541, 0.24971708421996541, 
0.24971708421996541], [0.24971708421996541, 0.24971708421996541, 0.88500663492033116, 0.24971708421996541, 
0.91967256531048047], [0.53756830959719037, 0.24971708421996541, 0.24971708421996541, 0.24971708421996541, 
0.83859088561946526]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank3_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.22561961094336969, -0.69644978225088083], [-0.44227121250995638, -0.50735454965999582]], 
[[0.16161200783821461, -0.4347630322634457], [0.26562304460337183, -0.15583347531179226]], [[0.59080340657647357, 
-0.30898005352956903], [-0.96090859496814618, 0.64957222823764948]], [[-0.78194949614475751, -0.96330346961526092], 
[-0.75472165908116562, -0.97849212182888756]], [[-0.75432107675432469, 0.082043851417474389], [0.47974376178170175, 
0.6709257227867087]], [[0.54824661624696036, -0.54157677822383921], [-0.659240045201267, 
0.18082823678245341]]])+(1.-msk_arg0)*numpy.array([[[-0.54709583559254926, -0.15868154997696871], [0.026457093040474833, 
0.56251013754601731]], [[-0.37622675052380505, -0.23721419821220846], [0.52430722837144028, -0.02821186532212705]], 
[[-0.7164360240992329, 0.87804574359157361], [-0.36055956747943774, 0.8665138503873846]], [[-0.87103016729055427, 
-0.36978512045411382], [0.16675457478226341, 0.23700377551418117]], [[0.32530875953749505, 0.73065087014702934], 
[-0.31548638071883328, -0.89169514477602374]], [[-0.58353491789348433, 0.3017692417364648], [0.42105486881407872, 
-0.24670377062806792]]])
      arg1=Data(-0.751837483947,self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.22561961094336969, -0.69644978225088083], [-0.44227121250995638, -0.50735454965999582]], 
[[0.16161200783821461, -0.4347630322634457], [0.26562304460337183, -0.15583347531179226]], [[0.59080340657647357, 
-0.30898005352956903], [-0.75183748394658645, 0.64957222823764948]], [[-0.75183748394658645, -0.75183748394658645], 
[-0.75183748394658645, -0.75183748394658645]], [[-0.75183748394658645, 0.082043851417474389], [0.47974376178170175, 
0.6709257227867087]], [[0.54824661624696036, -0.54157677822383921], [-0.659240045201267, 
0.18082823678245341]]])+(1.-msk_ref)*numpy.array([[[-0.54709583559254926, -0.15868154997696871], [0.026457093040474833, 
0.56251013754601731]], [[-0.37622675052380505, -0.23721419821220846], [0.52430722837144028, -0.02821186532212705]], 
[[-0.7164360240992329, 0.87804574359157361], [-0.36055956747943774, 0.8665138503873846]], [[-0.75183748394658645, 
-0.36978512045411382], [0.16675457478226341, 0.23700377551418117]], [[0.32530875953749505, 0.73065087014702934], 
[-0.31548638071883328, -0.75183748394658645]], [[-0.58353491789348433, 0.3017692417364648], [0.42105486881407872, 
-0.24670377062806792]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank4_constData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-0.46836650421512216, 0.55392434050869133, 0.64462355173707331, 0.47549964468485606], 
[-0.69289889981348418, -0.20461909725159777, -0.95905690844675684, 0.01420826477124737], [-0.77419156074874329, 
0.93441370846146876, 0.67855558163879071, 0.16075772354444062]], [[-0.0065968567345067175, 0.20050959624333031, 
0.72598290161102197, -0.97695441137457473], [0.34654853789303552, 0.48359989540054027, 0.45795932416242602, 
-0.13036567808708344], [-0.45679097037796268, 0.15641582786074903, 0.31786075279898829, -0.23870739994752288]]], 
[[[0.95042836806476449, 0.78075242319855431, -0.36462778215860348, -0.81076625675969294], [0.11060510010106617, 
-0.72824020802919986, -0.56633417520061813, -0.45562654021780635], [-0.42709566814202682, 0.79610178616158822, 
-0.85661543848488786, 0.070560494507247107]], [[0.20045514586255098, -0.19237619847201781, 0.49739688886609956, 
-0.96963620011390494], [0.17787451059406711, 0.95809663437507897, -0.68662219002349167, 0.20731333335407776], 
[-0.40982045954261337, -0.16017268023574838, -0.24073312330589358, -0.043607865804729862]]], [[[-0.89903746166893161, 
0.21373862072105831, -0.01480721421090192, -0.12849469109133538], [0.89888673761286042, 0.87560850881946917, 
-0.81952018477772159, 0.37617546473368257], [0.92421029785719599, 0.26088854799923022, -0.75492467331593938, 
0.8355004322022761]], [[-0.98145328336063553, -0.93694991169090569, 0.36611389422804419, 0.49453177652277347], 
[0.35699505986429214, -0.73480516245106497, 0.15750510738097434, 0.60146580130477645], [0.8080086070813568, 
-0.57380389192843828, 0.45232139162778928, -0.87668421492511817]]]])+(1.-msk_arg0)*numpy.array([[[[0.79049818853845744, 
-0.31643284915831793, 0.62698204144556136, 0.15055585522974413], [0.18428359666870175, -0.90655403355361042, 
-0.61799856818968246, -0.96209670683248194], [-0.22150547598602555, 0.11608945933719728, -0.93550584238462808, 
-0.51437923919154094]], [[0.11433307037970808, -0.6697957728739139, 0.026080474663143338, 0.89687571470160266], 
[0.74220257866685402, 0.80630222035973476, -0.88765397198480867, -0.30073269119267354], [-0.61692766321637516, 
0.70275565435456011, -0.770480369255093, 0.45855506259202561]]], [[[-0.25841786485345164, -0.9569035410607365, 
-0.2354046937481602, -0.76468796690158114], [-0.12764743888647057, -0.7570280529729656, -0.31857315603959679, 
0.15543454155327607], [-0.46592327565824387, -0.60197786685351118, 0.24325043755979214, -0.70586610361238722]], 
[[0.25903971611354648, -0.516785599192078, -0.4534446250373132, 0.19387204679020309], [-0.43563421185323126, 
0.55496223606378137, -0.13284603494736147, -0.58152191275770404], [-0.19132818776329441, 0.16638009745376592, 
0.92524151693459689, -0.22952254139222439]]], [[[0.3341414793432731, -0.40904633866616935, 0.86990541483951533, 
-0.38690851826482731], [0.34494363122687943, 0.41021617144572242, 0.33061965307913654, -0.43580755194659915], 
[-0.69103379639840234, 0.50350152095540768, -0.69843540817247174, 0.26288005317296004]], [[-0.2886477174481421, 
-0.78908077047169134, 0.11673976034770406, 0.33019842460207882], [-0.93492125398225467, -0.2334918373016639, 
0.98554713583062536, 0.4728612818618716], [-0.93118935725322149, 0.039886713080873326, 0.023210062083922844, 
-0.17367472025873631]]]])
      arg1=Data(0.515617958919,self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.51561795891922024, 0.55392434050869133, 0.64462355173707331, 0.51561795891922024], 
[0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 0.51561795891922024], [0.51561795891922024, 
0.93441370846146876, 0.67855558163879071, 0.51561795891922024]], [[0.51561795891922024, 0.51561795891922024, 
0.72598290161102197, 0.51561795891922024], [0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 
0.51561795891922024], [0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 0.51561795891922024]]], 
[[[0.95042836806476449, 0.78075242319855431, 0.51561795891922024, 0.51561795891922024], [0.51561795891922024, 
0.51561795891922024, 0.51561795891922024, 0.51561795891922024], [0.51561795891922024, 0.79610178616158822, 0.51561795891922024, 
0.51561795891922024]], [[0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 0.51561795891922024], 
[0.51561795891922024, 0.95809663437507897, 0.51561795891922024, 0.51561795891922024], [0.51561795891922024, 
0.51561795891922024, 0.51561795891922024, 0.51561795891922024]]], [[[0.51561795891922024, 0.51561795891922024, 
0.51561795891922024, 0.51561795891922024], [0.89888673761286042, 0.87560850881946917, 0.51561795891922024, 
0.51561795891922024], [0.92421029785719599, 0.51561795891922024, 0.51561795891922024, 0.8355004322022761]], 
[[0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 0.51561795891922024], [0.51561795891922024, 
0.51561795891922024, 0.51561795891922024, 0.60146580130477645], [0.8080086070813568, 0.51561795891922024, 0.51561795891922024, 
0.51561795891922024]]]])+(1.-msk_ref)*numpy.array([[[[0.79049818853845744, 0.51561795891922024, 0.62698204144556136, 
0.51561795891922024], [0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 0.51561795891922024], 
[0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 0.51561795891922024]], [[0.51561795891922024, 
0.51561795891922024, 0.51561795891922024, 0.89687571470160266], [0.74220257866685402, 0.80630222035973476, 0.51561795891922024, 
0.51561795891922024], [0.51561795891922024, 0.70275565435456011, 0.51561795891922024, 0.51561795891922024]]], 
[[[0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 0.51561795891922024], [0.51561795891922024, 
0.51561795891922024, 0.51561795891922024, 0.51561795891922024], [0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 
0.51561795891922024]], [[0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 0.51561795891922024], 
[0.51561795891922024, 0.55496223606378137, 0.51561795891922024, 0.51561795891922024], [0.51561795891922024, 
0.51561795891922024, 0.92524151693459689, 0.51561795891922024]]], [[[0.51561795891922024, 0.51561795891922024, 
0.86990541483951533, 0.51561795891922024], [0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 
0.51561795891922024], [0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 0.51561795891922024]], 
[[0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 0.51561795891922024], [0.51561795891922024, 
0.51561795891922024, 0.98554713583062536, 0.51561795891922024], [0.51561795891922024, 0.51561795891922024, 0.51561795891922024, 
0.51561795891922024]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.802743243229)+(1.-msk_arg0)*(0.99702688738)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.596467110586)+(1.-msk_arg1)*(0.684381079372)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(0.802743243229)+(1.-msk_ref)*(0.99702688738)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank1_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.30614519824934328, -0.51813576976876141])+(1.-msk_arg0)*numpy.array([0.5500576508538344, 
0.12364229321815245])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(0.321480996125)+(1.-msk_arg1)*(0.0905449773774)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.32148099612549919, 0.32148099612549919])+(1.-msk_ref)*numpy.array([0.5500576508538344, 
0.12364229321815245])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank2_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[0.35552309152320305, -0.11101213305383895, -0.74227520152636406, 0.46252596749984454, 
0.42113714908898525], [0.93097659800329025, 0.83327502155685007, 0.52757195622578923, 0.061911816792792118, 
-0.70771644366942166], [0.073915923799455552, -0.5867632562644447, -0.47160386240000851, -0.37172839238353261, 
0.91218541682748122], [0.8577488749130433, -0.56376354937296624, -0.81071810619576512, 0.81701750260161798, 
-0.06159461532096655]])+(1.-msk_arg0)*numpy.array([[-0.14196838411601753, 0.54924551192337701, 0.19495347583316969, 
-0.68202363369328567, 0.39461470689048062], [-0.3688063281037306, 0.48318268985280555, -0.77231524148262065, 
0.98493877845621736, -0.42013126010835244], [-0.9491470463541829, 0.10891550790333016, 0.93107340408360417, 
-0.70621836778754332, 0.71341896718143105], [-0.45972826679918621, -0.21350972161348403, 0.96659767157606402, 
-0.12881865656272806, 0.4043650376973722]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.67091545012)+(1.-msk_arg1)*(-0.0757810945746)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.35552309152320305, -0.11101213305383895, -0.67091545012013643, 0.46252596749984454, 
0.42113714908898525], [0.93097659800329025, 0.83327502155685007, 0.52757195622578923, 0.061911816792792118, 
-0.67091545012013643], [0.073915923799455552, -0.5867632562644447, -0.47160386240000851, -0.37172839238353261, 
0.91218541682748122], [0.8577488749130433, -0.56376354937296624, -0.67091545012013643, 0.81701750260161798, 
-0.06159461532096655]])+(1.-msk_ref)*numpy.array([[-0.075781094574565167, 0.54924551192337701, 0.19495347583316969, 
-0.075781094574565167, 0.39461470689048062], [-0.075781094574565167, 0.48318268985280555, -0.075781094574565167, 
0.98493877845621736, -0.075781094574565167], [-0.075781094574565167, 0.10891550790333016, 0.93107340408360417, 
-0.075781094574565167, 0.71341896718143105], [-0.075781094574565167, -0.075781094574565167, 0.96659767157606402, 
-0.075781094574565167, 0.4043650376973722]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank3_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.73259259492432927, 0.43516685226353369], [0.34658816662510294, 0.99694890396299152]], 
[[-0.82513842361582301, -0.60640821327530237], [-0.46651350541260106, -0.44412570768110937]], [[-0.095664979228247793, 
0.53884106143660015], [-0.15097890528281876, -0.0053807492881323693]], [[0.83781924943391406, -0.99066586932757672], 
[-0.88389801870266527, 0.065863228211280322]], [[-0.1940964066265598, -0.44657666544066421], [-0.7618618185275754, 
-0.022638072362826334]], [[0.64174101237204484, -0.75430513988969916], [0.97134520182564943, 
0.4795408665156784]]])+(1.-msk_arg0)*numpy.array([[[0.26927998366552952, 0.75523312878691362], [0.076434487275322427, 
-0.6819391552501628]], [[-0.48721751558695492, -0.18534329149598827], [-0.40788258318505388, -0.70764982356182471]], 
[[-0.82449663467055356, 0.16419130511566893], [0.25011445773188701, 0.1035128720077596]], [[-0.0022868899113452912, 
-0.47461046292673736], [-0.48883758713916237, 0.93798235093853721]], [[0.18720827904762127, -0.0030792936071819543], 
[-0.47477258458581706, -0.62011335136235024]], [[-0.63547863145971673, -0.81616381306744112], [-0.87756446408192956, 
0.25672900864254466]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.043916106452)+(1.-msk_arg1)*(0.158271071418)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.043916106452033032, 0.43516685226353369], [0.34658816662510294, 0.99694890396299152]], 
[[-0.043916106452033032, -0.043916106452033032], [-0.043916106452033032, -0.043916106452033032]], [[-0.043916106452033032, 
0.53884106143660015], [-0.043916106452033032, -0.0053807492881323693]], [[0.83781924943391406, -0.043916106452033032], 
[-0.043916106452033032, 0.065863228211280322]], [[-0.043916106452033032, -0.043916106452033032], [-0.043916106452033032, 
-0.022638072362826334]], [[0.64174101237204484, -0.043916106452033032], [0.97134520182564943, 
0.4795408665156784]]])+(1.-msk_ref)*numpy.array([[[0.26927998366552952, 0.75523312878691362], [0.15827107141755392, 
0.15827107141755392]], [[0.15827107141755392, 0.15827107141755392], [0.15827107141755392, 0.15827107141755392]], 
[[0.15827107141755392, 0.16419130511566893], [0.25011445773188701, 0.15827107141755392]], [[0.15827107141755392, 
0.15827107141755392], [0.15827107141755392, 0.93798235093853721]], [[0.18720827904762127, 0.15827107141755392], 
[0.15827107141755392, 0.15827107141755392]], [[0.15827107141755392, 0.15827107141755392], [0.15827107141755392, 
0.25672900864254466]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank4_expandedData_rank0(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-0.97239764595471057, -0.43270302318573983, -0.17542209868837455, -0.71317358004463216], 
[0.72878255833928107, 0.25393968296339708, 0.7062928811864444, 0.70798565970935257], [-0.8868456954486732, 
-0.074127186200250605, -0.26518707102168726, -0.91047990906076226]], [[-0.5485021908717751, 0.4625407566939097, 
-0.013117973326893795, 0.98963345359662513], [-0.56450486895185792, 0.40832041648188855, -0.48341521803155008, 
-0.99848277869207203], [0.30951894134880442, 0.60735662998233275, 0.61717782998775084, 0.19184298106071851]]], 
[[[0.6509286048782652, -0.86979300962840544, -0.5411735404074498, 0.88956340514970922], [-0.96087197906551292, 
-0.21534927161261552, -0.26855994833000074, 0.87089013542946114], [-0.62559526136686361, 0.64612780085586707, 
-0.71008042900676327, -0.12675921354333219]], [[0.14266331123609066, 0.12940917183095224, 0.53440833399954135, 
0.5541645660247565], [0.045871400215263813, 0.0028821188154224409, 0.61511017352453567, 0.23973103956187747], 
[-0.5445038209174482, 0.91013574526764818, -0.23848811565899952, -0.50332896814919081]]], [[[0.68813990143394066, 
-0.056178984894786232, -0.90169036885976483, -0.53603412124753502], [0.72461066752157022, -0.78363334606207768, 
-0.30673442696718567, -0.57239754291823752], [0.47800787094455832, 0.22707910699571876, 0.15163871045515576, 
0.34159974670530202]], [[0.22466552974210119, 0.61881183505470827, 0.1234877438382167, -0.12282347214627087], 
[0.61057990868408396, 0.8011211522833841, 0.04817598302759829, -0.69530242501742423], [-0.89443503352680165, 
-0.28642491039227203, 0.85109648851133368, 0.76098655847489716]]]])+(1.-msk_arg0)*numpy.array([[[[-0.78310919117047861, 
0.85334677407078319, -0.62426072225732399, 0.46836843422702801], [0.19400083911486732, 0.51787351106685575, 
0.77734214592705597, -0.95555304880082459], [-0.57736565407128349, -0.962078230900135, 0.71908577835422061, 
0.52860269178335706]], [[-0.23053457594726123, -0.266981616166589, 0.14706660164143481, 0.87362580720893002], 
[-0.13519752223577863, 0.68403053579261708, 0.81667447656236591, -0.63546890273087198], [-0.6941496632135975, 
-0.97716830228976104, -0.76264707972380985, 0.95414050486223734]]], [[[0.81051389375652416, -0.20260132938999731, 
-0.15605191092737858, 0.00027979793612331605], [0.35802562761073653, -0.90719191248436526, -0.43781707733763953, 
-0.011859015060990297], [-0.00026721880675317777, 0.03389014993537498, -0.86094631324460957, 0.63209497268625059]], 
[[0.84985041427879127, 0.63435047168217107, 0.63306132652062419, 0.15142825356810752], [-0.95169088382223621, 
-0.53852352077320687, 0.40757365469216311, 0.72549198576227814], [-0.40725622434408404, 0.2979281144336543, 
0.66349993369849813, 0.90003860925876911]]], [[[-0.18468813635358217, -0.15991413502959206, -0.37936629946621503, 
0.68121986886073937], [0.62629515765129828, -0.69478206811435461, -0.055458611202817476, 0.51186245363263772], 
[0.66956258449967776, 0.95206087000905248, 0.84125023317845504, -0.80845902992651286]], [[-0.24067863353083996, 
0.34352122106482885, 0.4111960087158808, 0.49747272331840375], [-0.71349227438459728, -0.017613099557210354, 
-0.66930930912049247, -0.042255829832726244], [0.69452252329293929, -0.54976031507306566, -0.76373909421180697, 
0.13579444083187586]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.0564655883036)+(1.-msk_arg1)*(0.980742034114)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.056465588303555192, -0.056465588303555192, -0.056465588303555192, -0.056465588303555192], 
[0.72878255833928107, 0.25393968296339708, 0.7062928811864444, 0.70798565970935257], [-0.056465588303555192, 
-0.056465588303555192, -0.056465588303555192, -0.056465588303555192]], [[-0.056465588303555192, 0.4625407566939097, 
-0.013117973326893795, 0.98963345359662513], [-0.056465588303555192, 0.40832041648188855, -0.056465588303555192, 
-0.056465588303555192], [0.30951894134880442, 0.60735662998233275, 0.61717782998775084, 0.19184298106071851]]], 
[[[0.6509286048782652, -0.056465588303555192, -0.056465588303555192, 0.88956340514970922], [-0.056465588303555192, 
-0.056465588303555192, -0.056465588303555192, 0.87089013542946114], [-0.056465588303555192, 0.64612780085586707, 
-0.056465588303555192, -0.056465588303555192]], [[0.14266331123609066, 0.12940917183095224, 0.53440833399954135, 
0.5541645660247565], [0.045871400215263813, 0.0028821188154224409, 0.61511017352453567, 0.23973103956187747], 
[-0.056465588303555192, 0.91013574526764818, -0.056465588303555192, -0.056465588303555192]]], [[[0.68813990143394066, 
-0.056178984894786232, -0.056465588303555192, -0.056465588303555192], [0.72461066752157022, -0.056465588303555192, 
-0.056465588303555192, -0.056465588303555192], [0.47800787094455832, 0.22707910699571876, 0.15163871045515576, 
0.34159974670530202]], [[0.22466552974210119, 0.61881183505470827, 0.1234877438382167, -0.056465588303555192], 
[0.61057990868408396, 0.8011211522833841, 0.04817598302759829, -0.056465588303555192], [-0.056465588303555192, 
-0.056465588303555192, 0.85109648851133368, 0.76098655847489716]]]])+(1.-msk_ref)*numpy.array([[[[0.98074203411363214, 
0.98074203411363214, 0.98074203411363214, 0.98074203411363214], [0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 
0.98074203411363214], [0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 0.98074203411363214]], 
[[0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 0.98074203411363214], [0.98074203411363214, 
0.98074203411363214, 0.98074203411363214, 0.98074203411363214], [0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 
0.98074203411363214]]], [[[0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 0.98074203411363214], 
[0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 0.98074203411363214], [0.98074203411363214, 
0.98074203411363214, 0.98074203411363214, 0.98074203411363214]], [[0.98074203411363214, 0.98074203411363214, 
0.98074203411363214, 0.98074203411363214], [0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 
0.98074203411363214], [0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 0.98074203411363214]]], 
[[[0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 0.98074203411363214], [0.98074203411363214, 
0.98074203411363214, 0.98074203411363214, 0.98074203411363214], [0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 
0.98074203411363214]], [[0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 0.98074203411363214], 
[0.98074203411363214, 0.98074203411363214, 0.98074203411363214, 0.98074203411363214], [0.98074203411363214, 
0.98074203411363214, 0.98074203411363214, 0.98074203411363214]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.91472696079)+(1.-msk_arg0)*(0.0113171772217)
      arg1=numpy.array([0.14371887572797215, 0.37905569309172993])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.14371887572797215, 0.37905569309172993])+(1.-msk_ref)*numpy.array([0.14371887572797215, 
0.37905569309172993])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank1_array_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.81630127664035768, -0.15929336428417074])+(1.-msk_arg0)*numpy.array([0.68019737932345881, 
-0.093124132522008241])
      arg1=numpy.array([-0.18398194604258977, -0.032707152944865703])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.18398194604258977, -0.032707152944865703])+(1.-msk_ref)*numpy.array([0.68019737932345881, 
-0.032707152944865703])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.548457814582)+(1.-msk_arg0)*(0.436923684914)
      arg1=Data(numpy.array([0.32727602859016058, -0.69475369769015094]),self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.54845781458158971, 0.54845781458158971])+(1.-msk_ref)*numpy.array([0.43692368491449796, 
0.43692368491449796])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank1_constData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([0.99653128761363829, -0.95662233086782145])+(1.-msk_arg0)*numpy.array([-0.2807142353642802, 
-0.065215727549697844])
      arg1=Data(numpy.array([-0.79471881002574074, -0.20025122987772725]),self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.99653128761363829, -0.20025122987772725])+(1.-msk_ref)*numpy.array([-0.2807142353642802, 
-0.065215727549697844])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.29012502378)+(1.-msk_arg0)*(0.41584476555)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-0.51903312957260583, 0.43793002132024195])+(1.-msk_arg1)*numpy.array([-0.78491146092485953, 
-0.011835549572744242])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.29012502378044203, 0.43793002132024195])+(1.-msk_ref)*numpy.array([0.41584476555003325, 
0.41584476555003325])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank1_expandedData_rank1(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([-0.82921401986738741, 0.80635465212532376])+(1.-msk_arg0)*numpy.array([0.88940997358724871, 
-0.066276300855594661])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([-0.48900030716550646, 0.89815654363021591])+(1.-msk_arg1)*numpy.array([-0.28630310397292424, 
0.90329307144140514])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([-0.48900030716550646, 0.89815654363021591])+(1.-msk_ref)*numpy.array([0.88940997358724871, 
0.90329307144140514])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.738866997633)+(1.-msk_arg0)*(-0.433750876359)
      arg1=numpy.array([[-0.72541252238085341, -0.27328120068358186, -0.25768712583393794, 0.21782093726317253, 
0.21367510956324054], [0.98361068092360515, 0.35248978957852106, 0.42339790030821889, -0.94369935325387355, 
-0.37125013968130149], [0.40578636976051574, 0.092530880501699109, 0.53629223276669968, -0.79074006834632415, 
-0.15815695657904016], [-0.919681786301078, 0.94894235245350478, 0.5214898825250871, -0.66979099258767993, 
-0.38568393578428029]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.72541252238085341, -0.27328120068358186, -0.25768712583393794, 0.21782093726317253, 
0.21367510956324054], [0.98361068092360515, 0.35248978957852106, 0.42339790030821889, -0.73886699763279085, 
-0.37125013968130149], [0.40578636976051574, 0.092530880501699109, 0.53629223276669968, -0.73886699763279085, 
-0.15815695657904016], [-0.73886699763279085, 0.94894235245350478, 0.5214898825250871, -0.66979099258767993, 
-0.38568393578428029]])+(1.-msk_ref)*numpy.array([[-0.43375087635858489, -0.27328120068358186, -0.25768712583393794, 
0.21782093726317253, 0.21367510956324054], [0.98361068092360515, 0.35248978957852106, 0.42339790030821889, 
-0.43375087635858489, -0.37125013968130149], [0.40578636976051574, 0.092530880501699109, 0.53629223276669968, 
-0.43375087635858489, -0.15815695657904016], [-0.43375087635858489, 0.94894235245350478, 0.5214898825250871, 
-0.43375087635858489, -0.38568393578428029]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank2_array_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[0.59602224313129293, -0.11901786006980619, 0.47394054830502341, 0.71692130476721272, 
0.09678329807748387], [-0.7741902289056608, 0.45979801568792888, -0.98432358068659065, 0.43810958790745702, 
-0.2181908164969264], [0.68635344198849357, -0.63580672602393418, 0.56914340993295376, 0.72582816628325864, 
0.28625749548694635], [-0.96371518108689647, 0.96240821379092734, 0.28345092970524055, 0.30653314343609073, 
-0.46516206234910218]])+(1.-msk_arg0)*numpy.array([[-0.34558335496233084, 0.12813098391466982, 0.41371341344934742, 
-0.31603088740499929, 0.84942205473057886], [-0.047845479330066443, -0.9688999068869768, 0.13558507922573537, 
-0.0021241484260634547, -0.0031519468474074941], [-0.24144821576740583, 0.75686073650684005, 0.42949597298606812, 
-0.3727904968363922, -0.17834896282439594], [-0.94049628031166632, 0.38349145105884608, 0.023352370386710009, 
-0.42358610500863669, 0.37437019047424891]])
      arg1=numpy.array([[-0.18542847783609706, 0.22742315441537819, -0.89799604135669098, -0.029715367528657666, 
0.26637098196110909], [0.08744313945977189, -0.3111398797675895, 0.33915365506764372, 0.17136460418220367, 
0.74584076818085498], [0.31715314583046261, 0.67587032025034177, -0.10233387490482748, 0.66906369472410776, 
0.47404774150329287], [0.20628235268347828, 0.6926711444575866, -0.17570076094413967, -0.73720093961838984, 
0.18922410282454538]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.59602224313129293, 0.22742315441537819, 0.47394054830502341, 0.71692130476721272, 
0.26637098196110909], [0.08744313945977189, 0.45979801568792888, 0.33915365506764372, 0.43810958790745702, 
0.74584076818085498], [0.68635344198849357, 0.67587032025034177, 0.56914340993295376, 0.72582816628325864, 
0.47404774150329287], [0.20628235268347828, 0.96240821379092734, 0.28345092970524055, 0.30653314343609073, 
0.18922410282454538]])+(1.-msk_ref)*numpy.array([[-0.18542847783609706, 0.22742315441537819, 0.41371341344934742, 
-0.029715367528657666, 0.84942205473057886], [0.08744313945977189, -0.3111398797675895, 0.33915365506764372, 
0.17136460418220367, 0.74584076818085498], [0.31715314583046261, 0.75686073650684005, 0.42949597298606812, 0.66906369472410776, 
0.47404774150329287], [0.20628235268347828, 0.6926711444575866, 0.023352370386710009, -0.42358610500863669, 
0.37437019047424891]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.968391321824)+(1.-msk_arg0)*(-0.7099346671)
      arg1=Data(numpy.array([[0.29701179879797723, 0.70347702823080427, -0.79471835800766311, -0.68731376369202835, 
0.90311857651371241], [0.7103320021427153, 0.054717441367131858, -0.99745778238441463, 0.04873028739048979, 
-0.61006905413485013], [-0.93842733155605274, 0.50853628427338804, -0.30426091903544128, 0.58360725848963546, 
-0.31705926426858344], [0.69399270037069161, 0.86826926031660534, -0.47801614714304264, -0.70843001046225007, 
0.41841801765930908]]),self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.9683913218243918, 0.9683913218243918, 0.9683913218243918, 0.9683913218243918, 
0.9683913218243918], [0.9683913218243918, 0.9683913218243918, 0.9683913218243918, 0.9683913218243918, 0.9683913218243918], 
[0.9683913218243918, 0.9683913218243918, 0.9683913218243918, 0.9683913218243918, 0.9683913218243918], [0.9683913218243918, 
0.9683913218243918, 0.9683913218243918, 0.9683913218243918, 
0.9683913218243918]])+(1.-msk_ref)*numpy.array([[0.29701179879797723, 0.70347702823080427, -0.7099346670995712, 
-0.68731376369202835, 0.90311857651371241], [0.7103320021427153, 0.054717441367131858, -0.7099346670995712, 
0.04873028739048979, -0.61006905413485013], [-0.7099346670995712, 0.50853628427338804, -0.30426091903544128, 
0.58360725848963546, -0.31705926426858344], [0.69399270037069161, 0.86826926031660534, -0.47801614714304264, 
-0.70843001046225007, 0.41841801765930908]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank2_constData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[0.17637613197723212, 0.81368899697847774, -0.70066741436084534, 0.52296323461967109, 
0.66661088326806484], [-0.67485549317596027, 0.59758445609776101, 0.45040429582876418, 0.4217012917259817, 
0.31279429738001019], [-0.58077835069237804, 0.29166102420102469, 0.2526616156848065, -0.99562234283945816, 
0.65468685772365487], [0.19825255328895963, -0.76510565916121398, -0.50483448612496407, -0.45238980645974602, 
-0.53086113668897883]])+(1.-msk_arg0)*numpy.array([[-0.22161254255821672, 0.68759782267067937, 0.42154665952894632, 
-0.985440698773, 0.65635042945475552], [-0.98478600172882391, -0.5752667375492353, 0.65357545821292073, -0.56915537584382703, 
0.81813057385681121], [-0.40228961090376347, 0.33922609331886067, 0.39610970863817063, -0.73836203087494967, 
0.024995750541199069], [0.88777276882702072, 0.17246350399093902, -0.61218504009487229, 0.31396178043500922, 
0.11512731297688328]])
      arg1=Data(numpy.array([[-0.13197862175472586, 0.14177033935387584, 0.014728422246227613, 0.79013786976409039, 
-0.18503758822783389], [-0.55156124342170476, 0.42996511802016557, -0.48720125682385684, -0.15886085390437277, 
0.30830039080251459], [-0.33140838037629905, 0.11411784911565048, -0.73051867757577305, 0.254789079889016, 
-0.43020531322659283], [0.95938676389248312, 0.30942507644975104, -0.50054369053870018, 0.38636057061586304, 
-0.98930316764198034]]),self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.17637613197723212, 0.81368899697847774, 0.014728422246227613, 0.79013786976409039, 
0.66661088326806484], [-0.55156124342170476, 0.59758445609776101, 0.45040429582876418, 0.4217012917259817, 
0.31279429738001019], [-0.33140838037629905, 0.29166102420102469, 0.2526616156848065, 0.254789079889016, 0.65468685772365487], 
[0.95938676389248312, 0.30942507644975104, -0.50054369053870018, 0.38636057061586304, 
-0.53086113668897883]])+(1.-msk_ref)*numpy.array([[-0.13197862175472586, 0.68759782267067937, 0.42154665952894632, 
0.79013786976409039, 0.65635042945475552], [-0.55156124342170476, 0.42996511802016557, 0.65357545821292073, 
-0.15886085390437277, 0.81813057385681121], [-0.33140838037629905, 0.33922609331886067, 0.39610970863817063, 0.254789079889016, 
0.024995750541199069], [0.95938676389248312, 0.30942507644975104, -0.50054369053870018, 0.38636057061586304, 
0.11512731297688328]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.886606128808)+(1.-msk_arg0)*(0.700768026575)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[-0.64305532973646962, -0.60638128218256138, -0.61260373793718603, 0.70274481583329318, 
0.67216573607284502], [0.056357513607530052, -0.48959189340698983, 0.77594105159309468, -0.33512945900811486, 
-0.25598250131831435], [-0.30715706143639299, 0.74424305314625716, 0.24844080090239506, 0.6650072844121715, 
-0.11577214960345117], [0.22799447655880756, 0.12096394541929056, -0.61783357756637769, -0.7264377493442018, 
-0.47973788724609601]])+(1.-msk_arg1)*numpy.array([[0.40953857832882945, -0.99145479187770569, -0.44359836716615697, 
-0.30133864079518013, 0.9989647760984417], [0.23049902934564859, 0.076896464188720426, 0.89322879768241381, 
0.23892993150226904, 0.19317941564711605], [0.61204111604612521, -0.071423367004127769, 0.74464158577928607, 
-0.26337579380661036, -0.42258481952201143], [0.43093362538307511, 0.39300817159427748, -0.86217742540170916, 
-0.60481644540941271, 0.29651542402341824]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[-0.64305532973646962, -0.60638128218256138, -0.61260373793718603, 0.70274481583329318, 
0.67216573607284502], [0.056357513607530052, -0.48959189340698983, 0.77594105159309468, -0.33512945900811486, 
-0.25598250131831435], [-0.30715706143639299, 0.74424305314625716, 0.24844080090239506, 0.6650072844121715, 
-0.11577214960345117], [0.22799447655880756, 0.12096394541929056, -0.61783357756637769, -0.7264377493442018, 
-0.47973788724609601]])+(1.-msk_ref)*numpy.array([[0.70076802657454795, 0.70076802657454795, 0.70076802657454795, 
0.70076802657454795, 0.9989647760984417], [0.70076802657454795, 0.70076802657454795, 0.89322879768241381, 0.70076802657454795, 
0.70076802657454795], [0.70076802657454795, 0.70076802657454795, 0.74464158577928607, 0.70076802657454795, 
0.70076802657454795], [0.70076802657454795, 0.70076802657454795, 0.70076802657454795, 0.70076802657454795, 
0.70076802657454795]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank2_expandedData_rank2(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[-0.30612019189686168, 0.95135935967120422, -0.075353866244783019, -0.56798070953046609, 
0.42529437925846336], [-0.7968326902743923, 0.25500403452815013, -0.34667163196195383, -0.15250832494823841, 
-0.98608167456642448], [0.75976092051693511, -0.71954763745402794, -0.85716569577642621, 0.5049444481642722, 
-0.5589267595072247], [-0.44267531322074372, 0.92213529484491441, 0.55119130907482328, -0.68170498176248873, 
0.90555976103882951]])+(1.-msk_arg0)*numpy.array([[0.59519177758892727, 0.25976913209471464, -0.27191012388999081, 
0.8616827024136533, 0.48496769325691225], [0.42123832877010292, 0.97608441953561509, -0.033557590270044457, 
-0.80335344330470848, 0.2498103208196969], [-0.28440764569673171, 0.48513587003581704, -0.26670168419543239, 
-0.023258098131885507, -0.28285209082596752], [-0.77293657173661967, -0.87258589181928414, 0.16777292094842156, 
0.98554552171112531, -0.58041025857394812]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.53980428075498432, -0.8620985675765529, 0.10318753714451057, -0.34594469649839743, 
-0.26253653354933437], [0.5568000522030927, 0.95471851543455877, 0.76494137771653703, -0.89909752018530487, 
0.065218938426729212], [-0.94889860454446717, -0.27132057498615136, 0.25050587653834056, -0.022689614047457729, 
-0.83525701623221682], [-0.72343888167391657, -0.94000414827540379, -0.12680345659866288, -0.3726106505034219, 
0.12322122882315356]])+(1.-msk_arg1)*numpy.array([[0.42724124999901059, -0.22595557355760132, -0.80352821333452384, 
-0.92811140531522573, 0.63931956037221016], [-0.6877750864294403, -0.0056332903472737783, 0.21419300884938419, 
-0.69188811767615754, -0.9138285400178312], [-0.54256535713927923, 0.73733527645626329, -0.098012441663765548, 
0.80054498308143573, 0.11936239919805236], [0.46067331791290411, -0.85622216822493269, -0.85577286610605174, 
-0.64462140755589581, -0.91167565524099992]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.53980428075498432, 0.95135935967120422, 0.10318753714451057, -0.34594469649839743, 
0.42529437925846336], [0.5568000522030927, 0.95471851543455877, 0.76494137771653703, -0.15250832494823841, 
0.065218938426729212], [0.75976092051693511, -0.27132057498615136, 0.25050587653834056, 0.5049444481642722, 
-0.5589267595072247], [-0.44267531322074372, 0.92213529484491441, 0.55119130907482328, -0.3726106505034219, 
0.90555976103882951]])+(1.-msk_ref)*numpy.array([[0.59519177758892727, 0.25976913209471464, -0.27191012388999081, 
0.8616827024136533, 0.63931956037221016], [0.42123832877010292, 0.97608441953561509, 0.21419300884938419, -0.69188811767615754, 
0.2498103208196969], [-0.28440764569673171, 0.73733527645626329, -0.098012441663765548, 0.80054498308143573, 
0.11936239919805236], [0.46067331791290411, -0.85622216822493269, 0.16777292094842156, 0.98554552171112531, 
-0.58041025857394812]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.718530103778)+(1.-msk_arg0)*(0.793689594671)
      arg1=numpy.array([[[-0.0024451589941487661, 0.37962439196941866], [0.63153612360410327, -0.47899847783370531]], 
[[-0.90520992078403451, -0.95918575086250835], [-0.28115016997012687, -0.51161918860583588]], [[-0.61627320654983975, 
-0.54619724559098004], [0.47194607858515991, 0.70563469194352724]], [[0.77071388199460955, 0.071543453256624145], 
[0.320642130841571, -0.23511721513687456]], [[0.565685956805559, 0.39919171271086351], [0.39452154791888394, 
-0.8334230961358382]], [[0.71553572930335529, -0.72729092650994009], [-0.57115890968262772, -0.19116184636450217]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.71853010377798143, 0.71853010377798143], [0.71853010377798143, 0.71853010377798143]], 
[[0.71853010377798143, 0.71853010377798143], [0.71853010377798143, 0.71853010377798143]], [[0.71853010377798143, 
0.71853010377798143], [0.71853010377798143, 0.71853010377798143]], [[0.77071388199460955, 0.71853010377798143], 
[0.71853010377798143, 0.71853010377798143]], [[0.71853010377798143, 0.71853010377798143], [0.71853010377798143, 
0.71853010377798143]], [[0.71853010377798143, 0.71853010377798143], [0.71853010377798143, 
0.71853010377798143]]])+(1.-msk_ref)*numpy.array([[[0.79368959467057976, 0.79368959467057976], [0.79368959467057976, 
0.79368959467057976]], [[0.79368959467057976, 0.79368959467057976], [0.79368959467057976, 0.79368959467057976]], 
[[0.79368959467057976, 0.79368959467057976], [0.79368959467057976, 0.79368959467057976]], [[0.79368959467057976, 
0.79368959467057976], [0.79368959467057976, 0.79368959467057976]], [[0.79368959467057976, 0.79368959467057976], 
[0.79368959467057976, 0.79368959467057976]], [[0.79368959467057976, 0.79368959467057976], [0.79368959467057976, 
0.79368959467057976]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank3_array_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.23289574950911729, -0.39979931374519451], [0.15897379012575996, -0.50180377622002981]], 
[[-0.82406624777113491, 0.68978178267237378], [-0.49214098282971119, 0.91663604312644487]], [[0.51961880703793417, 
0.33271371791298732], [-0.029330613495592406, -0.33267726604768266]], [[-0.44169585496973585, -0.19253039780834591], 
[-0.86233079085192776, 0.22210644263950541]], [[0.58157093465620235, 0.4651068757348702], [-0.80136699806451972, 
0.4791116027753044]], [[-0.34358341468322973, 0.97285610062438943], [0.54582634844290179, 
-0.3844491351792334]]])+(1.-msk_arg0)*numpy.array([[[0.43243149290797156, 0.42118817946076303], [-0.18453847901320564, 
-0.66339724486601703]], [[0.44682412774501268, 0.06259230513593339], [-0.85030268678048393, 0.090377810940876158]], 
[[0.56249810268779799, 0.23332735750156597], [-0.6347229076098917, 0.17946284083046082]], [[-0.44724133701655133, 
-0.98766139526109908], [-0.19160273520718185, 0.37747814250070011]], [[-0.376248630682541, 0.97158262633526449], 
[0.2983965823325585, -0.33246515396523035]], [[0.79962535603610885, 0.42934691003447378], [0.76478449568109652, 
0.91637336177037865]]])
      arg1=numpy.array([[[0.088752584503619891, -0.39447101553511321], [-0.22430239048448097, -0.72298239865701297]], 
[[-0.39702422374427049, -0.82719309363155413], [-0.88688183257968389, 0.23290068315861734]], [[0.19723100073754174, 
0.17510622396992837], [-0.75586102931009358, 0.086122900287766813]], [[0.2038452039966705, 0.69127428790485257], 
[-0.35550974805881208, 0.95332996260066616]], [[0.86443048779177034, -0.38168990310894912], [0.38182538983725678, 
0.73503462747021775]], [[-0.70625227491884734, -0.12131105742004333], [0.85069828133096981, -0.6433556681341337]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.088752584503619891, -0.39447101553511321], [0.15897379012575996, -0.50180377622002981]], 
[[-0.39702422374427049, 0.68978178267237378], [-0.49214098282971119, 0.91663604312644487]], [[0.51961880703793417, 
0.33271371791298732], [-0.029330613495592406, 0.086122900287766813]], [[0.2038452039966705, 0.69127428790485257], 
[-0.35550974805881208, 0.95332996260066616]], [[0.86443048779177034, 0.4651068757348702], [0.38182538983725678, 
0.73503462747021775]], [[-0.34358341468322973, 0.97285610062438943], [0.85069828133096981, 
-0.3844491351792334]]])+(1.-msk_ref)*numpy.array([[[0.43243149290797156, 0.42118817946076303], [-0.18453847901320564, 
-0.66339724486601703]], [[0.44682412774501268, 0.06259230513593339], [-0.85030268678048393, 0.23290068315861734]], 
[[0.56249810268779799, 0.23332735750156597], [-0.6347229076098917, 0.17946284083046082]], [[0.2038452039966705, 
0.69127428790485257], [-0.19160273520718185, 0.95332996260066616]], [[0.86443048779177034, 0.97158262633526449], 
[0.38182538983725678, 0.73503462747021775]], [[0.79962535603610885, 0.42934691003447378], [0.85069828133096981, 
0.91637336177037865]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.989176914801)+(1.-msk_arg0)*(-0.989986362534)
      arg1=Data(numpy.array([[[-0.49388719096314349, -0.79623559567830759], [-0.83658541823517818, 0.42072406054838662]], 
[[-0.48192235106585324, -0.14003155480478036], [0.47407230666875866, 0.95145307875088792]], [[0.81276956345717855, 
0.54577402893520022], [-0.90122971458194412, -0.58274583646026556]], [[0.85522050991554255, 0.75964838308411031], 
[0.5778018835429104, -0.76867690348681994]], [[0.15025246865549424, 0.055620041059527336], [-0.54156080230040549, 
-0.63085875592520013]], [[-0.335005199709838, 0.74217379824147822], [-0.61569565476952914, 
-0.23786048005871696]]]),self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.49388719096314349, -0.79623559567830759], [-0.83658541823517818, 0.42072406054838662]], 
[[-0.48192235106585324, -0.14003155480478036], [0.47407230666875866, 0.95145307875088792]], [[0.81276956345717855, 
0.54577402893520022], [-0.90122971458194412, -0.58274583646026556]], [[0.85522050991554255, 0.75964838308411031], 
[0.5778018835429104, -0.76867690348681994]], [[0.15025246865549424, 0.055620041059527336], [-0.54156080230040549, 
-0.63085875592520013]], [[-0.335005199709838, 0.74217379824147822], [-0.61569565476952914, 
-0.23786048005871696]]])+(1.-msk_ref)*numpy.array([[[-0.49388719096314349, -0.79623559567830759], [-0.83658541823517818, 
0.42072406054838662]], [[-0.48192235106585324, -0.14003155480478036], [0.47407230666875866, 0.95145307875088792]], 
[[0.81276956345717855, 0.54577402893520022], [-0.90122971458194412, -0.58274583646026556]], [[0.85522050991554255, 
0.75964838308411031], [0.5778018835429104, -0.76867690348681994]], [[0.15025246865549424, 0.055620041059527336], 
[-0.54156080230040549, -0.63085875592520013]], [[-0.335005199709838, 0.74217379824147822], [-0.61569565476952914, 
-0.23786048005871696]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank3_constData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[0.9515053377069429, 0.47807174372932359], [-0.75564278236571569, -0.28735791360983409]], 
[[0.38398978657266181, 0.81107114048034323], [-0.83362651172912039, -0.7692532587204759]], [[-0.85019914335870261, 
-0.74233731831744643], [0.094460048711824784, 0.88107054995270251]], [[-0.16449632437481898, -0.44322035699164086], 
[-0.93906847021333362, 0.63056982121743488]], [[-0.15478314379855163, -0.51272224940380906], [-0.90580658840006745, 
-0.088880576181244342]], [[-0.57864595898321425, -0.98961581993571057], [0.69127717551907675, 
0.12303841555427852]]])+(1.-msk_arg0)*numpy.array([[[0.54991250710512474, 0.75252556510707502], [0.4524255659499381, 
0.034482370597328282]], [[0.048275902648243552, -0.42803062132180703], [-0.42953608825084499, -0.52649686005029883]], 
[[0.98434263591531024, 0.045527551414420664], [0.25896487569318261, -0.82768236289669472]], [[0.82974872968024926, 
-0.85936448047482017], [-0.48527758532198395, -0.76833200375575417]], [[0.89349009195765672, -0.19880662446189135], 
[0.2966368207713177, -0.18896189651548845]], [[0.1803928604184375, -0.82241408231310964], [0.22841835028060076, 
0.39317537222153964]]])
      arg1=Data(numpy.array([[[0.68617253474215634, 0.15087223754864798], [-0.10726012723616618, -0.92665009980729041]], 
[[-0.041620269299473822, 0.015752588101144216], [-0.14427171039873188, -0.42796473013430236]], [[0.11932262149106454, 
0.64034918365669058], [-0.48270730652849725, 0.053510918018123022]], [[0.33635000832920325, -0.86051716060430916], 
[-0.70908675709144253, -0.11679411206479129]], [[-0.071763389013920875, 0.16572479797356277], [0.77137834529942251, 
0.26120416887974818]], [[0.61310060082682982, 0.015920493203621211], [-0.46407909287571503, 
-0.60491137260032635]]]),self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.9515053377069429, 0.47807174372932359], [-0.10726012723616618, -0.28735791360983409]], 
[[0.38398978657266181, 0.81107114048034323], [-0.14427171039873188, -0.42796473013430236]], [[0.11932262149106454, 
0.64034918365669058], [0.094460048711824784, 0.88107054995270251]], [[0.33635000832920325, -0.44322035699164086], 
[-0.70908675709144253, 0.63056982121743488]], [[-0.071763389013920875, 0.16572479797356277], [0.77137834529942251, 
0.26120416887974818]], [[0.61310060082682982, 0.015920493203621211], [0.69127717551907675, 
0.12303841555427852]]])+(1.-msk_ref)*numpy.array([[[0.68617253474215634, 0.75252556510707502], [0.4524255659499381, 
0.034482370597328282]], [[0.048275902648243552, 0.015752588101144216], [-0.14427171039873188, -0.42796473013430236]], 
[[0.98434263591531024, 0.64034918365669058], [0.25896487569318261, 0.053510918018123022]], [[0.82974872968024926, 
-0.85936448047482017], [-0.48527758532198395, -0.11679411206479129]], [[0.89349009195765672, 0.16572479797356277], 
[0.77137834529942251, 0.26120416887974818]], [[0.61310060082682982, 0.015920493203621211], [0.22841835028060076, 
0.39317537222153964]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.293076478313)+(1.-msk_arg0)*(0.415695253254)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-0.42476097366268384, -0.99781990217086425], [-0.42797078478453088, 0.56788496043163539]], 
[[0.88356209487076054, -0.21592478018874428], [0.59903633153983038, 0.99453929697875876]], [[-0.4031451864299096, 
-0.78793554032137436], [0.99404205252643041, -0.46900675116702062]], [[-0.35961855333700887, -0.11711382975673645], 
[0.18349118086242, -0.035040226820079123]], [[-0.67627961022952809, 0.4515888728838664], [0.93565887741312981, 
0.31169707414443426]], [[0.96128404783088417, -0.8324773748092742], [0.23613968220346138, 
-0.11039468454161838]]])+(1.-msk_arg1)*numpy.array([[[-0.88023059794213743, 0.927763549181158], [-0.83970906479816909, 
0.79761854094125706]], [[-0.40619318341717858, 0.69914276566700284], [0.44376855556648342, -0.91825223070551631]], 
[[0.39649333256863528, -0.092930299765284108], [-0.072283891404024025, 0.13436467337237956]], [[0.19751892281774874, 
-0.82813284163388401], [0.6214385672795204, 0.47316992672533109]], [[-0.97549578931282155, -0.50190475490773157], 
[0.6647972184776747, -0.63181839238397086]], [[-0.16594757687004513, -0.34293676516292471], [0.14039762775702891, 
-0.044223647723118154]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.29307647831298689, -0.29307647831298689], [-0.29307647831298689, 0.56788496043163539]], 
[[0.88356209487076054, -0.21592478018874428], [0.59903633153983038, 0.99453929697875876]], [[-0.29307647831298689, 
-0.29307647831298689], [0.99404205252643041, -0.29307647831298689]], [[-0.29307647831298689, -0.11711382975673645], 
[0.18349118086242, -0.035040226820079123]], [[-0.29307647831298689, 0.4515888728838664], [0.93565887741312981, 
0.31169707414443426]], [[0.96128404783088417, -0.29307647831298689], [0.23613968220346138, 
-0.11039468454161838]]])+(1.-msk_ref)*numpy.array([[[0.41569525325398415, 0.927763549181158], [0.41569525325398415, 
0.79761854094125706]], [[0.41569525325398415, 0.69914276566700284], [0.44376855556648342, 0.41569525325398415]], 
[[0.41569525325398415, 0.41569525325398415], [0.41569525325398415, 0.41569525325398415]], [[0.41569525325398415, 
0.41569525325398415], [0.6214385672795204, 0.47316992672533109]], [[0.41569525325398415, 0.41569525325398415], 
[0.6647972184776747, 0.41569525325398415]], [[0.41569525325398415, 0.41569525325398415], [0.41569525325398415, 
0.41569525325398415]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank3_expandedData_rank3(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[-0.36130298057679111, -0.72982256164520121], [-0.98427828362689751, -0.71392419224338233]], 
[[0.0028648066420755569, 0.40593075742981033], [-0.39308624937033798, 0.41173487980367285]], [[0.040429985666669666, 
0.9956986678939499], [-0.3766616780070684, -0.14800180944610708]], [[-0.34619453852173332, -0.17504597230775909], 
[0.88567660560918915, -0.74291128943136053]], [[0.03679824663741571, -0.10882953030274356], [0.45259132917204359, 
0.6036332348120923]], [[-0.74384731565805584, -0.98281302250512592], [-0.97516438572854058, 
-0.43418945070204695]]])+(1.-msk_arg0)*numpy.array([[[-0.56133979475379858, 0.5901398064050758], [0.26863333535059408, 
0.21622822005148534]], [[-0.014772805904820441, -0.61818138105374354], [0.3201607812914451, -0.16980551408521727]], 
[[0.65927338717041262, 0.82573021065854024], [-0.9382351948264771, 0.46463417799389251]], [[0.55336804661738448, 
-0.16893054714418709], [-0.45486072472527739, -0.36714074972175692]], [[-0.45180298993654455, -0.029952989308233535], 
[0.71588478817998147, -0.59014771697946466]], [[0.86344487271841497, -0.66727303798019189], [-0.36584595815879517, 
0.33547655315459779]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[0.64224348205869819, 0.049037997241984765], [-0.98116819130903643, 0.94181846056675522]], 
[[-0.16946566364672533, 0.3705330855361475], [-0.18152511908793034, 0.018261630850941124]], [[0.029393061582448254, 
-0.32804788747238334], [-0.08265934486633153, -0.62626548403291404]], [[-0.81665972655100605, 0.18713502632711143], 
[0.069041181401869434, -0.012080898549561292]], [[-0.49967653078908558, -0.88780398744026323], [-0.96151630421477496, 
0.17188541642458732]], [[0.8559946449708169, 0.78065817299993023], [-0.37293727862758685, 
-0.60616899821341486]]])+(1.-msk_arg1)*numpy.array([[[-0.49032583607212854, -0.8136795614509551], [0.89577823564665859, 
0.79840425740211107]], [[0.41626566605101467, -0.24782501100599741], [-0.59737466766066505, 0.053224551388946217]], 
[[-0.49418240106620037, -0.014671319634904867], [0.43041860621440775, -0.6139043809851934]], [[-0.70080342153969033, 
0.3930154267575332], [-0.31944992543072104, -0.82077676018890178]], [[-0.25102242325231439, 0.045813485089310158], 
[0.89982702067370801, 0.27597754182222012]], [[-0.13163387797868897, 0.50410986652020684], [0.4718132434424589, 
0.55301018967858151]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[0.64224348205869819, 0.049037997241984765], [-0.98116819130903643, 0.94181846056675522]], 
[[0.0028648066420755569, 0.40593075742981033], [-0.18152511908793034, 0.41173487980367285]], [[0.040429985666669666, 
0.9956986678939499], [-0.08265934486633153, -0.14800180944610708]], [[-0.34619453852173332, 0.18713502632711143], 
[0.88567660560918915, -0.012080898549561292]], [[0.03679824663741571, -0.10882953030274356], [0.45259132917204359, 
0.6036332348120923]], [[0.8559946449708169, 0.78065817299993023], [-0.37293727862758685, 
-0.43418945070204695]]])+(1.-msk_ref)*numpy.array([[[-0.49032583607212854, 0.5901398064050758], [0.89577823564665859, 
0.79840425740211107]], [[0.41626566605101467, -0.24782501100599741], [0.3201607812914451, 0.053224551388946217]], 
[[0.65927338717041262, 0.82573021065854024], [0.43041860621440775, 0.46463417799389251]], [[0.55336804661738448, 
0.3930154267575332], [-0.31944992543072104, -0.36714074972175692]], [[-0.25102242325231439, 0.045813485089310158], 
[0.89982702067370801, 0.27597754182222012]], [[0.86344487271841497, 0.50410986652020684], [0.4718132434424589, 
0.55301018967858151]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.93122841147)+(1.-msk_arg0)*(0.690664190259)
      arg1=numpy.array([[[[-0.14129653506270778, -0.057224607125864191, 0.9211833065741597, 0.6473574724937774], 
[0.79342813565602577, 0.58188727876918134, 0.54967922043088868, -0.86099657295389598], [0.72149005255943988, 
0.49970954592644179, 0.20671785867262726, -0.70256925121758917]], [[0.30291193430769958, 0.38342882633653308, 
-0.8482847466432486, 0.80705958071323347], [-0.29500788390136612, -0.45682358930611389, 0.42671080439327636, 
0.82874473369979196], [0.33378375281988304, 0.57146717744436648, 0.71657597252409011, -0.35470560156724984]]], 
[[[0.64905652148048087, 0.066757786624187521, -0.033149691345921273, -0.09095042959743993], [0.23199263733313469, 
0.37983826394739206, 0.64154881030545563, -0.51234130039277082], [0.69621394755055732, -0.82794221977727456, 
-0.39538559469645329, -0.1618575046203059]], [[-0.29009671679704652, -0.9022342552920628, 0.81807598072470888, 
-0.13568667169335469], [-0.5503101579327343, 0.5357357653203636, -0.15952645784198238, 0.70807762475485791], 
[-0.64948567730623208, 0.75852757462222153, -0.3563619745115234, -0.069412219258663477]]], [[[0.80762645906831954, 
0.34639538845074114, -0.36919251697493238, 0.32457680007810064], [-0.9161626068958062, 0.90565137289941, -0.68875897981177348, 
-0.23023391707486063], [0.12566730654437941, 0.28534019901624386, 0.28644025853864852, -0.36938545264538325]], 
[[-0.59831408015871768, -0.17176820348331701, 0.54554002636113474, 0.58717693841655438], [-0.67460631161913742, 
-0.53038034535930567, 0.41207107058237002, 0.86424360099095976], [0.84577384964306712, 0.76585183685560199, 
-0.67847345259712255, -0.46920326976603643]]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.14129653506270778, -0.057224607125864191, 0.9211833065741597, 0.6473574724937774], 
[0.79342813565602577, 0.58188727876918134, 0.54967922043088868, -0.86099657295389598], [0.72149005255943988, 
0.49970954592644179, 0.20671785867262726, -0.70256925121758917]], [[0.30291193430769958, 0.38342882633653308, 
-0.8482847466432486, 0.80705958071323347], [-0.29500788390136612, -0.45682358930611389, 0.42671080439327636, 
0.82874473369979196], [0.33378375281988304, 0.57146717744436648, 0.71657597252409011, -0.35470560156724984]]], 
[[[0.64905652148048087, 0.066757786624187521, -0.033149691345921273, -0.09095042959743993], [0.23199263733313469, 
0.37983826394739206, 0.64154881030545563, -0.51234130039277082], [0.69621394755055732, -0.82794221977727456, 
-0.39538559469645329, -0.1618575046203059]], [[-0.29009671679704652, -0.9022342552920628, 0.81807598072470888, 
-0.13568667169335469], [-0.5503101579327343, 0.5357357653203636, -0.15952645784198238, 0.70807762475485791], 
[-0.64948567730623208, 0.75852757462222153, -0.3563619745115234, -0.069412219258663477]]], [[[0.80762645906831954, 
0.34639538845074114, -0.36919251697493238, 0.32457680007810064], [-0.9161626068958062, 0.90565137289941, -0.68875897981177348, 
-0.23023391707486063], [0.12566730654437941, 0.28534019901624386, 0.28644025853864852, -0.36938545264538325]], 
[[-0.59831408015871768, -0.17176820348331701, 0.54554002636113474, 0.58717693841655438], [-0.67460631161913742, 
-0.53038034535930567, 0.41207107058237002, 0.86424360099095976], [0.84577384964306712, 0.76585183685560199, 
-0.67847345259712255, -0.46920326976603643]]]])+(1.-msk_ref)*numpy.array([[[[0.69066419025918635, 0.69066419025918635, 
0.9211833065741597, 0.69066419025918635], [0.79342813565602577, 0.69066419025918635, 0.69066419025918635, 0.69066419025918635], 
[0.72149005255943988, 0.69066419025918635, 0.69066419025918635, 0.69066419025918635]], [[0.69066419025918635, 
0.69066419025918635, 0.69066419025918635, 0.80705958071323347], [0.69066419025918635, 0.69066419025918635, 0.69066419025918635, 
0.82874473369979196], [0.69066419025918635, 0.69066419025918635, 0.71657597252409011, 0.69066419025918635]]], 
[[[0.69066419025918635, 0.69066419025918635, 0.69066419025918635, 0.69066419025918635], [0.69066419025918635, 
0.69066419025918635, 0.69066419025918635, 0.69066419025918635], [0.69621394755055732, 0.69066419025918635, 0.69066419025918635, 
0.69066419025918635]], [[0.69066419025918635, 0.69066419025918635, 0.81807598072470888, 0.69066419025918635], 
[0.69066419025918635, 0.69066419025918635, 0.69066419025918635, 0.70807762475485791], [0.69066419025918635, 
0.75852757462222153, 0.69066419025918635, 0.69066419025918635]]], [[[0.80762645906831954, 0.69066419025918635, 
0.69066419025918635, 0.69066419025918635], [0.69066419025918635, 0.90565137289941, 0.69066419025918635, 0.69066419025918635], 
[0.69066419025918635, 0.69066419025918635, 0.69066419025918635, 0.69066419025918635]], [[0.69066419025918635, 
0.69066419025918635, 0.69066419025918635, 0.69066419025918635], [0.69066419025918635, 0.69066419025918635, 0.69066419025918635, 
0.86424360099095976], [0.84577384964306712, 0.76585183685560199, 0.69066419025918635, 0.69066419025918635]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank4_array_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-0.44772485229861014, 0.33612133273895872, 0.85463791292335456, 0.041150956342052902], 
[0.86196525725452178, -0.39230412793663172, -0.55689630426568115, 0.24500138369530022], [0.250440568048236, 
0.56208355979267344, -0.83931525120237538, -0.89962845216257148]], [[0.93199744061363954, 0.036161545737686263, 
0.17600123827465097, -0.15644558799055064], [0.51769117284229615, 0.33689820823374972, 0.060403722261924919, 
-0.67401337318561683], [-0.5870077618091023, 0.35549507870374053, -0.91444103477854144, 0.23111635010592591]]], 
[[[0.30427728778811924, 0.11748924311521702, 0.10481933285039213, -0.33502229131967587], [0.26515755662112572, 
0.099878676762299445, 0.69816433138192102, -0.98773857616342631], [-0.092028032496626722, -0.70240638229622876, 
0.71611949446200152, 0.25170319313794276]], [[0.037527545049968802, 0.49361154856116607, 0.62722682321215739, 
-0.87892648689079045], [-0.95341541684169084, -0.010773509314439877, -0.58116207689853905, -0.58140329890782971], 
[0.27136771093638901, 0.082457582086216341, -0.21110397496607991, -0.29970993820392677]]], [[[-0.22265001693402553, 
0.94823954643701369, -0.7266026099756171, -0.32353607416408758], [0.53071688333832867, -0.28644943651030919, 
-0.7169950181827689, 0.58843920390628113], [-0.68278304053745997, -0.73191962597309357, 0.83951460129804634, 
-0.28603981175880899]], [[-0.62530509820592028, 0.19163503831394335, 0.71630459577755268, -0.48358514127015995], 
[0.35290552000196262, 0.42122549901315964, 0.48162879235316169, 0.92685704379160105], [0.21890865144110738, 
-0.69229869309205228, -0.51436410927581733, 0.77525889246393453]]]])+(1.-msk_arg0)*numpy.array([[[[-0.47994795124062661, 
0.74856862291517623, 0.5821395617608589, -0.31324499648561299], [0.13609834355782402, 0.8651786428977688, -0.22241596237712269, 
0.59595179741831994], [0.52442180802467986, -0.67755617307904803, 0.041047440148369585, -0.63982185998775787]], 
[[0.89633393878486145, 0.60528407122761974, 0.087008352545255407, 0.61818598246202194], [-0.82215369326968912, 
-0.17257825389866843, -0.97707914343112701, 0.10612032693135687], [0.54832826138702373, 0.14418882636852293, 
-0.59559456960762192, -0.20767572929860978]]], [[[-0.80347236426212421, -0.91815707719608963, 0.91255277503919396, 
0.045503228874384849], [0.75467835024620356, 0.25562178283718717, 0.24903527543586712, 0.56502967349932165], 
[-0.1335888822357898, -0.68279551726839105, 0.43491768196300673, 0.46168510826508036]], [[0.4356763541452715, 
0.98085251814408414, -0.62755096901235152, -0.78345634103039363], [-0.60500930133859709, 0.56402299324965721, 
0.59561490314477927, 0.82072216033311784], [0.95001617573414032, 0.20468480993367022, 0.30565824220619442, 
-0.070107462971154888]]], [[[0.23393156511221869, -0.37563984619602975, 0.36378548060529137, 0.43451806931870163], 
[-0.1642002012922239, -0.54728511822572523, 0.36670201211269826, 0.18918610945368441], [-0.33406116215906256, 
-0.25576174770698823, -0.02020947518693661, 0.87728898366655694]], [[-0.59338050831235423, -0.14657407934408662, 
0.69570242811712601, 0.36366022997929148], [0.94456195322361602, -0.86212596260553642, 0.5145071925072755, 
-0.61650036986407497], [-0.47495276384029639, -0.9072767739797154, 0.60749045264710033, 0.80404847388366707]]]])
      arg1=numpy.array([[[[0.12235724577495133, 0.030550102850180449, 0.82361823715408589, -0.47577063319692625], 
[0.24762255252735432, -0.18117195942957376, 0.78421570529513662, -0.33120166549504804], [-0.31661878366910501, 
0.050408417654134619, 0.93543377163183949, 0.7138338863269178]], [[0.60609599680976656, -0.18646992543326979, 
0.032245229491007832, -0.16135181789862751], [-0.2197064057048792, 0.89130327750849525, 0.87528272382646333, 
-0.4054146695562737], [0.34864798718456469, 0.6865288809229968, -0.27667529617816933, 0.0036869563701744656]]], 
[[[-0.95770828799863805, 0.096015202531622768, -0.60118964454981816, -0.77935740289018307], [0.72274943044452389, 
0.0073194320184981621, -0.059288992988969946, 0.79201968761767816], [-0.12866704041475518, 0.98722308476263909, 
-0.4127435552309402, 0.90334700524787515]], [[0.46963073206058481, 0.79351354921673289, 0.044478338561507069, 
-0.86621149843666823], [-0.86768077600190563, 0.87895872758421212, -0.68813225933821531, 0.051058982021920718], 
[-0.6400145105837145, -0.6791315301293348, 0.35602453634758469, -0.40925802590739879]]], [[[0.6864899883695792, 
0.44223216849695901, -0.89165346027070291, 0.40133713829912065], [0.78750618231061065, -0.013479137470628899, 
0.6577875955603778, -0.87184650021107624], [-0.69474956775331775, 0.93518928901459675, 0.90966513308817243, 
0.77851041589827696]], [[-0.50347347741991966, 0.98138043351913606, -0.88658887741951498, 0.89654400325398464], 
[-0.78549232237218436, -0.34691444129238169, 0.30748561536051167, 0.62508666703593296], [0.44494445035343788, 
-0.87171691786051886, 0.31674335839547196, 0.8800806202145921]]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.12235724577495133, 0.33612133273895872, 0.85463791292335456, 0.041150956342052902], 
[0.86196525725452178, -0.18117195942957376, 0.78421570529513662, 0.24500138369530022], [0.250440568048236, 0.56208355979267344, 
0.93543377163183949, 0.7138338863269178]], [[0.93199744061363954, 0.036161545737686263, 0.17600123827465097, 
-0.15644558799055064], [0.51769117284229615, 0.89130327750849525, 0.87528272382646333, -0.4054146695562737], 
[0.34864798718456469, 0.6865288809229968, -0.27667529617816933, 0.23111635010592591]]], [[[0.30427728778811924, 
0.11748924311521702, 0.10481933285039213, -0.33502229131967587], [0.72274943044452389, 0.099878676762299445, 
0.69816433138192102, 0.79201968761767816], [-0.092028032496626722, 0.98722308476263909, 0.71611949446200152, 
0.90334700524787515]], [[0.46963073206058481, 0.79351354921673289, 0.62722682321215739, -0.86621149843666823], 
[-0.86768077600190563, 0.87895872758421212, -0.58116207689853905, 0.051058982021920718], [0.27136771093638901, 
0.082457582086216341, 0.35602453634758469, -0.29970993820392677]]], [[[0.6864899883695792, 0.94823954643701369, 
-0.7266026099756171, 0.40133713829912065], [0.78750618231061065, -0.013479137470628899, 0.6577875955603778, 
0.58843920390628113], [-0.68278304053745997, 0.93518928901459675, 0.90966513308817243, 0.77851041589827696]], 
[[-0.50347347741991966, 0.98138043351913606, 0.71630459577755268, 0.89654400325398464], [0.35290552000196262, 
0.42122549901315964, 0.48162879235316169, 0.92685704379160105], [0.44494445035343788, -0.69229869309205228, 
0.31674335839547196, 0.8800806202145921]]]])+(1.-msk_ref)*numpy.array([[[[0.12235724577495133, 0.74856862291517623, 
0.82361823715408589, -0.31324499648561299], [0.24762255252735432, 0.8651786428977688, 0.78421570529513662, 
0.59595179741831994], [0.52442180802467986, 0.050408417654134619, 0.93543377163183949, 0.7138338863269178]], 
[[0.89633393878486145, 0.60528407122761974, 0.087008352545255407, 0.61818598246202194], [-0.2197064057048792, 
0.89130327750849525, 0.87528272382646333, 0.10612032693135687], [0.54832826138702373, 0.6865288809229968, -0.27667529617816933, 
0.0036869563701744656]]], [[[-0.80347236426212421, 0.096015202531622768, 0.91255277503919396, 0.045503228874384849], 
[0.75467835024620356, 0.25562178283718717, 0.24903527543586712, 0.79201968761767816], [-0.12866704041475518, 
0.98722308476263909, 0.43491768196300673, 0.90334700524787515]], [[0.46963073206058481, 0.98085251814408414, 
0.044478338561507069, -0.78345634103039363], [-0.60500930133859709, 0.87895872758421212, 0.59561490314477927, 
0.82072216033311784], [0.95001617573414032, 0.20468480993367022, 0.35602453634758469, -0.070107462971154888]]], 
[[[0.6864899883695792, 0.44223216849695901, 0.36378548060529137, 0.43451806931870163], [0.78750618231061065, 
-0.013479137470628899, 0.6577875955603778, 0.18918610945368441], [-0.33406116215906256, 0.93518928901459675, 
0.90966513308817243, 0.87728898366655694]], [[-0.50347347741991966, 0.98138043351913606, 0.69570242811712601, 
0.89654400325398464], [0.94456195322361602, -0.34691444129238169, 0.5145071925072755, 0.62508666703593296], 
[0.44494445035343788, -0.87171691786051886, 0.60749045264710033, 0.8800806202145921]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(0.0881956698806)+(1.-msk_arg0)*(-0.731128324845)
      arg1=Data(numpy.array([[[[-0.64189758046957146, -0.53815399874264136, 0.033693340777282366, 0.99925795797181571], 
[-0.2419875717107054, -0.50747686174550699, -0.81027767900895475, -0.84877637041813614], [-0.20071783395542631, 
-0.8069228315827619, 0.66582338261558349, 0.64096730572944427]], [[0.81185199455197243, -0.90778080246183657, 
0.017632995172836674, -0.32090381670185386], [0.13011082833732179, 0.6578464151231187, -0.6959246201621212, 
-0.6545688299236001], [0.14663475574733353, 0.91800376327950861, 0.56376151063798741, -0.96804446803081667]]], 
[[[-0.0021218076221873883, -0.10724622444534382, 0.38123000588899658, 0.12283867161727002], [0.3563313122155094, 
-0.042825459741448135, -0.39224174048110716, -0.93819757083393496], [-0.56950271363830596, 0.037025945267049831, 
-0.10711115625494805, -0.12244688771221512]], [[0.55557589199957658, 0.95005362115253589, 0.084514241828244518, 
-0.67046064450650178], [0.01747628454518857, 0.18620488628187792, 0.66507212829733886, 0.76247784353876535], 
[0.37735898180876082, 0.32486295087694095, 0.47501482765124581, 0.52248847470936344]]], [[[0.0036882590128266557, 
0.8282751649691007, -0.80873751745324807, 0.47849609936161519], [0.39166314221529275, 0.48831283571527884, 
-0.86334428729172585, 0.79143886820146481], [0.83249453703468279, -0.98930182274409639, -0.083316120352270762, 
0.294221269622853]], [[0.18930386704305269, -0.99776611169212037, 0.14587140405481969, -0.95686905824011759], 
[-0.15749539134547641, -0.56249207552408076, 0.45265834859712295, -0.46177078880637601], [0.66396367131331657, 
0.96630757924567945, -0.86681340389547001, 0.6623848566435715]]]]),self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.088195669880560379, 0.088195669880560379, 0.088195669880560379, 0.99925795797181571], 
[0.088195669880560379, 0.088195669880560379, 0.088195669880560379, 0.088195669880560379], [0.088195669880560379, 
0.088195669880560379, 0.66582338261558349, 0.64096730572944427]], [[0.81185199455197243, 0.088195669880560379, 
0.088195669880560379, 0.088195669880560379], [0.13011082833732179, 0.6578464151231187, 0.088195669880560379, 
0.088195669880560379], [0.14663475574733353, 0.91800376327950861, 0.56376151063798741, 0.088195669880560379]]], 
[[[0.088195669880560379, 0.088195669880560379, 0.38123000588899658, 0.12283867161727002], [0.3563313122155094, 
0.088195669880560379, 0.088195669880560379, 0.088195669880560379], [0.088195669880560379, 0.088195669880560379, 
0.088195669880560379, 0.088195669880560379]], [[0.55557589199957658, 0.95005362115253589, 0.088195669880560379, 
0.088195669880560379], [0.088195669880560379, 0.18620488628187792, 0.66507212829733886, 0.76247784353876535], 
[0.37735898180876082, 0.32486295087694095, 0.47501482765124581, 0.52248847470936344]]], [[[0.088195669880560379, 
0.8282751649691007, 0.088195669880560379, 0.47849609936161519], [0.39166314221529275, 0.48831283571527884, 
0.088195669880560379, 0.79143886820146481], [0.83249453703468279, 0.088195669880560379, 0.088195669880560379, 
0.294221269622853]], [[0.18930386704305269, 0.088195669880560379, 0.14587140405481969, 0.088195669880560379], 
[0.088195669880560379, 0.088195669880560379, 0.45265834859712295, 0.088195669880560379], [0.66396367131331657, 
0.96630757924567945, 0.088195669880560379, 0.6623848566435715]]]])+(1.-msk_ref)*numpy.array([[[[-0.64189758046957146, 
-0.53815399874264136, 0.033693340777282366, 0.99925795797181571], [-0.2419875717107054, -0.50747686174550699, 
-0.73112832484494983, -0.73112832484494983], [-0.20071783395542631, -0.73112832484494983, 0.66582338261558349, 
0.64096730572944427]], [[0.81185199455197243, -0.73112832484494983, 0.017632995172836674, -0.32090381670185386], 
[0.13011082833732179, 0.6578464151231187, -0.6959246201621212, -0.6545688299236001], [0.14663475574733353, 0.91800376327950861, 
0.56376151063798741, -0.73112832484494983]]], [[[-0.0021218076221873883, -0.10724622444534382, 0.38123000588899658, 
0.12283867161727002], [0.3563313122155094, -0.042825459741448135, -0.39224174048110716, -0.73112832484494983], 
[-0.56950271363830596, 0.037025945267049831, -0.10711115625494805, -0.12244688771221512]], [[0.55557589199957658, 
0.95005362115253589, 0.084514241828244518, -0.67046064450650178], [0.01747628454518857, 0.18620488628187792, 
0.66507212829733886, 0.76247784353876535], [0.37735898180876082, 0.32486295087694095, 0.47501482765124581, 
0.52248847470936344]]], [[[0.0036882590128266557, 0.8282751649691007, -0.73112832484494983, 0.47849609936161519], 
[0.39166314221529275, 0.48831283571527884, -0.73112832484494983, 0.79143886820146481], [0.83249453703468279, 
-0.73112832484494983, -0.083316120352270762, 0.294221269622853]], [[0.18930386704305269, -0.73112832484494983, 
0.14587140405481969, -0.73112832484494983], [-0.15749539134547641, -0.56249207552408076, 0.45265834859712295, 
-0.46177078880637601], [0.66396367131331657, 0.96630757924567945, -0.73112832484494983, 0.6623848566435715]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank4_constData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[-0.10531364581559632, -0.26135514113778791, 0.75823435654818261, 0.41913136584038457], 
[0.43115964993714972, 0.43325049456887954, 0.99798790086677469, 0.12148464100767264], [-0.792314005480238, 0.94449170983715258, 
-0.51282378136184414, 0.78279449919260413]], [[-0.256730957467878, -0.54685067746437355, 0.13799031452254362, 
0.54420562233980974], [0.79271302319406733, 0.61289153297761811, 0.22488152217118107, 0.38057207678407945], 
[-0.58355124854227669, 0.33339550351240987, -0.45440247482861396, -0.32858088081907244]]], [[[0.42804301050211202, 
0.71528000472301367, -0.29947441761599003, -0.65907060023841413], [0.21382369966318948, -0.23392820079386456, 
0.1149254861227258, 0.26438763174771629], [0.54292796867386128, 0.93595972935560612, 0.63111118015829226, 
-0.48720925111434954]], [[-0.64752714499820963, 0.23062867134246989, -0.44111125359638725, 0.87281600565752915], 
[0.17646081091253429, 0.16503304920470341, -0.76446719475275016, -0.9496673356542491], [0.81402347205529568, 
-0.7279756234241106, 0.32595508977473875, 0.78908310868157816]]], [[[-0.7994446077113897, 0.54116981041814904, 
-0.29869702018833433, -0.60523743501453087], [-0.46193843131786938, -0.87737364655870431, -0.53149381910526228, 
0.32118149490277959], [0.88124482945730831, 0.76961174010593836, -0.51624564499591163, 0.46112358500778217]], 
[[-0.72956420221936336, -0.49297722281654144, -0.93821464396887477, 0.6257868729024445], [-0.22970749405216329, 
-0.44455529448948572, 0.85363272062066464, 0.84754506932043783], [-0.66852456480381584, 0.79050538702573658, 
0.32041213027833426, 0.036833079887540254]]]])+(1.-msk_arg0)*numpy.array([[[[-0.19914272243996511, -0.22066958304650086, 
-0.051623157097562489, -0.30649365550154917], [0.70526559900670605, 0.79190691746271002, -0.1840028427567888, 
0.11449103902153879], [-0.5924052950405061, 0.3079650735182351, -0.9438123347143772, -0.065661960754274995]], 
[[-0.0036167079524380696, 0.84054224471953476, -0.53140279065734641, 0.35467617149678676], [0.29362960794600701, 
-0.14052271099644154, 0.1293556777292062, 0.45589620449734247], [0.45309220884092505, 0.070251984236403331, 
-0.79210333449212667, 0.10845805646166129]]], [[[0.30350245072854709, -0.61792951212226188, -0.98924055757045237, 
0.75886074238090151], [0.19835957943587212, -0.32993508643372804, -0.81430829784396175, 0.67280893395841113], 
[-0.23121853491576294, 0.53086442636291209, -0.63883554946929699, 0.17420740192224771]], [[0.89221420493213976, 
-0.2045207552720627, -0.20771044164770736, 0.42648308785419431], [0.87113955270066956, 0.24540728624391916, 
-0.45094960000767226, -0.16188150623057007], [-0.45960021467049605, -0.95157979026703687, -0.82635297970091592, 
-0.97341062097802844]]], [[[0.48800175644884614, -0.50576269437461963, -0.17133398261855226, -0.72602653879188628], 
[-0.85605984598767382, -0.58287114441973764, 0.005984177454671169, 0.26177073655897831], [0.64181206014940528, 
0.87931494110685526, 0.78403977073172282, 0.024419280675760158]], [[-0.68551675053970906, 0.14152572212171521, 
-0.77906301525452615, -0.12878542896599199], [0.33305465516919974, 0.0082682574899826111, -0.39785724636599062, 
0.047898579856563428], [-0.84404774609808642, 0.67721243023535527, -0.59693854720693174, -0.37505322327010027]]]])
      arg1=Data(numpy.array([[[[-0.82759551572398471, 0.058473671021340712, -0.10837091814903643, 0.57740785960292684], 
[0.27913511962597259, 0.0011338106694573913, 0.0009601979643438785, -0.04802408596147334], [0.92688524933994465, 
0.78439774647519878, -0.31909062602360128, 0.72602592282456158]], [[0.94140041851607315, 0.94570499934150676, 
0.26198355391860817, 0.3065174789316123], [-0.48677124302537544, -0.85999326805316434, -0.99216619030997077, 
-0.083870140182316755], [-0.26968066595677942, -0.028588804387562616, 0.57648293580256471, 0.70249263700215847]]], 
[[[-0.59432588710527101, -0.015338113227458816, -0.67625911560986385, 0.87073185203582937], [0.26418569454285712, 
0.76837073130429445, 0.32369932417835279, 0.14645995660195821], [-0.50777857923006087, 0.27528983484000102, 
0.57195763489050599, 0.75155872411421054]], [[0.30020184045055109, -0.83179186646902536, 0.66761313528171162, 
0.93305194275955561], [-0.44456663805664576, 0.55650037021524135, 0.62058565622477313, -0.61150812573877777], 
[-0.31226647099785221, 0.14513696705524182, 0.68032486017657678, -0.22086700354642508]]], [[[0.66284368284982187, 
0.61236796493579271, 0.43440212926366417, 0.083118805833193532], [-0.85956397424801834, -0.10084123824854152, 
0.67438359152521032, 0.85048470938007048], [0.66201451496804764, -0.61354777895468482, -0.80926632689236633, 
0.8510794761665228]], [[-0.73695063130269212, 0.44757527421684307, 0.68262028916181294, 0.10862811437247544], 
[0.65035504442237335, -0.49416276490023092, 0.99800064674598987, -0.91779706704154007], [-0.69460250652998123, 
0.81518826983336323, 0.33775413898516593, -0.57214687266614961]]]]),self.functionspace)
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[-0.10531364581559632, 0.058473671021340712, 0.75823435654818261, 0.57740785960292684], 
[0.43115964993714972, 0.43325049456887954, 0.99798790086677469, 0.12148464100767264], [0.92688524933994465, 
0.94449170983715258, -0.31909062602360128, 0.78279449919260413]], [[0.94140041851607315, 0.94570499934150676, 
0.26198355391860817, 0.54420562233980974], [0.79271302319406733, 0.61289153297761811, 0.22488152217118107, 
0.38057207678407945], [-0.26968066595677942, 0.33339550351240987, 0.57648293580256471, 0.70249263700215847]]], 
[[[0.42804301050211202, 0.71528000472301367, -0.29947441761599003, 0.87073185203582937], [0.26418569454285712, 
0.76837073130429445, 0.32369932417835279, 0.26438763174771629], [0.54292796867386128, 0.93595972935560612, 0.63111118015829226, 
0.75155872411421054]], [[0.30020184045055109, 0.23062867134246989, 0.66761313528171162, 0.93305194275955561], 
[0.17646081091253429, 0.55650037021524135, 0.62058565622477313, -0.61150812573877777], [0.81402347205529568, 
0.14513696705524182, 0.68032486017657678, 0.78908310868157816]]], [[[0.66284368284982187, 0.61236796493579271, 
0.43440212926366417, 0.083118805833193532], [-0.46193843131786938, -0.10084123824854152, 0.67438359152521032, 
0.85048470938007048], [0.88124482945730831, 0.76961174010593836, -0.51624564499591163, 0.8510794761665228]], 
[[-0.72956420221936336, 0.44757527421684307, 0.68262028916181294, 0.6257868729024445], [0.65035504442237335, 
-0.44455529448948572, 0.99800064674598987, 0.84754506932043783], [-0.66852456480381584, 0.81518826983336323, 
0.33775413898516593, 0.036833079887540254]]]])+(1.-msk_ref)*numpy.array([[[[-0.19914272243996511, 0.058473671021340712, 
-0.051623157097562489, 0.57740785960292684], [0.70526559900670605, 0.79190691746271002, 0.0009601979643438785, 
0.11449103902153879], [0.92688524933994465, 0.78439774647519878, -0.31909062602360128, 0.72602592282456158]], 
[[0.94140041851607315, 0.94570499934150676, 0.26198355391860817, 0.35467617149678676], [0.29362960794600701, 
-0.14052271099644154, 0.1293556777292062, 0.45589620449734247], [0.45309220884092505, 0.070251984236403331, 
0.57648293580256471, 0.70249263700215847]]], [[[0.30350245072854709, -0.015338113227458816, -0.67625911560986385, 
0.87073185203582937], [0.26418569454285712, 0.76837073130429445, 0.32369932417835279, 0.67280893395841113], 
[-0.23121853491576294, 0.53086442636291209, 0.57195763489050599, 0.75155872411421054]], [[0.89221420493213976, 
-0.2045207552720627, 0.66761313528171162, 0.93305194275955561], [0.87113955270066956, 0.55650037021524135, 0.62058565622477313, 
-0.16188150623057007], [-0.31226647099785221, 0.14513696705524182, 0.68032486017657678, -0.22086700354642508]]], 
[[[0.66284368284982187, 0.61236796493579271, 0.43440212926366417, 0.083118805833193532], [-0.85605984598767382, 
-0.10084123824854152, 0.67438359152521032, 0.85048470938007048], [0.66201451496804764, 0.87931494110685526, 
0.78403977073172282, 0.8510794761665228]], [[-0.68551675053970906, 0.44757527421684307, 0.68262028916181294, 
0.10862811437247544], [0.65035504442237335, 0.0082682574899826111, 0.99800064674598987, 0.047898579856563428], 
[-0.69460250652998123, 0.81518826983336323, 0.33775413898516593, -0.37505322327010027]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank0_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*(-0.430656784848)+(1.-msk_arg0)*(-0.700783789461)
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[0.68835738514383604, -0.76863401076143667, 0.92032350741511615, -0.1469001502609133], 
[0.06288453409673278, 0.38073779065461522, -0.90058807506723526, 0.38778286232709025], [0.96668256007501041, 
0.057601266285067343, 0.79648117729052714, 0.49143175520310778]], [[0.98909339096938509, 0.7602637867878832, 
-0.912321941323607, -0.54315061194825787], [0.80423923526891117, 0.47661901140330509, 0.99031475342531228, 
-0.89450136434961736], [-0.36910767598659655, 0.55360242758997402, -0.24636032220916326, 0.013256286221418945]]], 
[[[0.15603828628588978, -0.70571626749096739, -0.22412221738086946, -0.44778376773528183], [0.83165578331141399, 
0.65163198519388876, 0.6039897710150941, -0.3822861567244713], [0.31330097532750978, -0.81103293228454709, 0.17174182343570354, 
-0.035823123408156521]], [[0.62310089072276864, 0.54271156487897554, 0.44186419167763291, 0.22536200856774924], 
[0.81691779953483756, 0.97875287193551719, -0.10259001428076586, 0.99985268669145078], [-0.6545383935440594, 
-0.91966644139989562, 0.54410275610147973, -0.25257465340702723]]], [[[0.22718370645623831, -0.8974233572367345, 
-0.75452612517607376, -0.58709992502677699], [-0.063069442929713171, -0.2742728542659314, 0.0030934546784433525, 
0.88865344687736392], [-0.28059615259505599, -0.38818143127085736, -0.26036533554867081, -0.91372878625635412]], 
[[-0.19770833412921429, -0.2520416533993588, -0.32329826656736227, -0.2921042427795626], [-0.77757026759743275, 
0.19691427600790279, -0.82662487726058442, 0.76440912595714972], [-0.8785601665435594, 0.6685997731242812, 0.17858501472408173, 
-0.072430031752663382]]]])+(1.-msk_arg1)*numpy.array([[[[0.218050144488529, -0.97758051946393643, 0.850873231285731, 
-0.62549319560829364], [0.64439251587171076, 0.02043882543426645, -0.40088447997559573, 0.32825901856835582], 
[0.40086412906080149, 0.44806775103789498, 0.7575462104983206, -0.71189398852218688]], [[-0.47405190774543682, 
-0.13001057892794665, -0.51504190270399475, 0.35420972217429236], [-0.83399490871466253, 0.0058345121935481359, 
0.14938712239679286, 0.94294724195590929], [0.52941947608322826, 0.92556345140725682, -0.88319470923943211, 
0.76396950935303654]]], [[[0.89036261550337015, 0.87671711452257739, 0.25741308722544365, 0.62226771243047474], 
[-0.015121664468682905, 0.93979205811046596, -0.71174182978124589, 0.97248180642636251], [-0.64050451914857787, 
-0.61134599587804939, -0.9143086087295571, -0.10806644675596133]], [[0.21434619581078618, 0.85410384206731016, 
-0.051783648108333358, 0.80705842847270559], [0.059772809014316586, 0.63262662785196522, 0.16149018829346895, 
-0.34656541265527929], [-0.88512172547595824, -0.86523726523371947, -0.82269875427618322, -0.72165361398578187]]], 
[[[0.93903641427310136, 0.28209257258291887, -0.51351863703745515, 0.68881576173912262], [-0.19081765518314353, 
-0.044663078887927421, 0.065339976778740816, 0.97860023971151544], [0.065270643177932852, 0.84667221338805354, 
0.015860276910488746, -0.23037023584540139]], [[0.56220085723088831, 0.2868468697603257, 0.73939367184777383, 
0.48844672936175515], [-0.54331103170726092, -0.55778174945812209, -0.95425197901750436, 0.59928943529546808], 
[0.79804805060136541, 0.64738539292254482, 0.47278529075708753, -0.84128123298524726]]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.68835738514383604, -0.43065678484805492, 0.92032350741511615, -0.1469001502609133], 
[0.06288453409673278, 0.38073779065461522, -0.43065678484805492, 0.38778286232709025], [0.96668256007501041, 
0.057601266285067343, 0.79648117729052714, 0.49143175520310778]], [[0.98909339096938509, 0.7602637867878832, 
-0.43065678484805492, -0.43065678484805492], [0.80423923526891117, 0.47661901140330509, 0.99031475342531228, 
-0.43065678484805492], [-0.36910767598659655, 0.55360242758997402, -0.24636032220916326, 0.013256286221418945]]], 
[[[0.15603828628588978, -0.43065678484805492, -0.22412221738086946, -0.43065678484805492], [0.83165578331141399, 
0.65163198519388876, 0.6039897710150941, -0.3822861567244713], [0.31330097532750978, -0.43065678484805492, 0.17174182343570354, 
-0.035823123408156521]], [[0.62310089072276864, 0.54271156487897554, 0.44186419167763291, 0.22536200856774924], 
[0.81691779953483756, 0.97875287193551719, -0.10259001428076586, 0.99985268669145078], [-0.43065678484805492, 
-0.43065678484805492, 0.54410275610147973, -0.25257465340702723]]], [[[0.22718370645623831, -0.43065678484805492, 
-0.43065678484805492, -0.43065678484805492], [-0.063069442929713171, -0.2742728542659314, 0.0030934546784433525, 
0.88865344687736392], [-0.28059615259505599, -0.38818143127085736, -0.26036533554867081, -0.43065678484805492]], 
[[-0.19770833412921429, -0.2520416533993588, -0.32329826656736227, -0.2921042427795626], [-0.43065678484805492, 
0.19691427600790279, -0.43065678484805492, 0.76440912595714972], [-0.43065678484805492, 0.6685997731242812, 
0.17858501472408173, -0.072430031752663382]]]])+(1.-msk_ref)*numpy.array([[[[0.218050144488529, -0.70078378946110842, 
0.850873231285731, -0.62549319560829364], [0.64439251587171076, 0.02043882543426645, -0.40088447997559573, 
0.32825901856835582], [0.40086412906080149, 0.44806775103789498, 0.7575462104983206, -0.70078378946110842]], 
[[-0.47405190774543682, -0.13001057892794665, -0.51504190270399475, 0.35420972217429236], [-0.70078378946110842, 
0.0058345121935481359, 0.14938712239679286, 0.94294724195590929], [0.52941947608322826, 0.92556345140725682, 
-0.70078378946110842, 0.76396950935303654]]], [[[0.89036261550337015, 0.87671711452257739, 0.25741308722544365, 
0.62226771243047474], [-0.015121664468682905, 0.93979205811046596, -0.70078378946110842, 0.97248180642636251], 
[-0.64050451914857787, -0.61134599587804939, -0.70078378946110842, -0.10806644675596133]], [[0.21434619581078618, 
0.85410384206731016, -0.051783648108333358, 0.80705842847270559], [0.059772809014316586, 0.63262662785196522, 
0.16149018829346895, -0.34656541265527929], [-0.70078378946110842, -0.70078378946110842, -0.70078378946110842, 
-0.70078378946110842]]], [[[0.93903641427310136, 0.28209257258291887, -0.51351863703745515, 0.68881576173912262], 
[-0.19081765518314353, -0.044663078887927421, 0.065339976778740816, 0.97860023971151544], [0.065270643177932852, 
0.84667221338805354, 0.015860276910488746, -0.23037023584540139]], [[0.56220085723088831, 0.2868468697603257, 
0.73939367184777383, 0.48844672936175515], [-0.54331103170726092, -0.55778174945812209, -0.70078378946110842, 
0.59928943529546808], [0.79804805060136541, 0.64738539292254482, 0.47278529075708753, -0.70078378946110842]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_maximum_expandedData_rank4_expandedData_rank4(self):
      msk_arg0=whereNegative(self.functionspace.getX()[0]-0.5)
      arg0=msk_arg0*numpy.array([[[[0.56285787359423467, 0.37775535632450929, -0.3363819504506349, -0.39614224287220079], 
[-0.65956942938945873, -0.050567052731754325, -0.54905395376446253, -0.94001597388873392], [-0.78068340465775754, 
-0.89097864337441868, -0.83984842944522442, 0.74631234877918584]], [[0.08573606948206014, 0.55431246693940839, 
-0.8641772485747512, -0.42863009943253139], [0.80323187807206664, -0.74856933876086118, -0.01644211276713814, 
-0.91644648328349509], [0.40224351184954954, -0.91717203862964691, -0.84862948625343493, -0.25037544559678815]]], 
[[[0.82728992326268469, -0.95042129406079257, -0.93015204068307433, -0.28209813227511105], [-0.51972916795137514, 
0.74506459052200502, -0.58415751918737313, -0.72089184286465779], [0.70832242911523857, 0.32910818999400115, 
0.51617654973055238, 0.88129235546352791]], [[0.58177700099759, 0.63909107674454901, 0.86508533718829983, 
-0.45384546311830021], [0.34909094815239139, -0.85407855551520706, -0.83626668667446791, -0.19144948150684948], 
[-0.54080937187207279, -0.29228298836204991, 0.073340450796397505, -0.39919385201564572]]], [[[-0.35454323448815428, 
0.78785581682131256, -0.37407581640252885, -0.9491936944611421], [-0.90501286696617278, 0.69422301523334684, 
0.080299003664048518, -0.10604345961307726], [0.37557752598770766, -0.11425319383355337, 0.43620438053394817, 
-0.63938084921875493]], [[-0.6542747934342823, -0.77225386259596918, -0.30885873461055691, -0.93491787154442219], 
[0.44906982794974248, 0.47078393816377861, -0.53577916133628367, 0.57388498777390184], [0.53885977209128866, 
0.79843035696797116, 0.29270123507386536, 0.76740426807290363]]]])+(1.-msk_arg0)*numpy.array([[[[-0.89333258741389199, 
0.57101135857695096, 0.82736606705903304, -0.55893684971502733], [-0.92021692928061039, -0.55348474011111159, 
0.29566092180536008, -0.13507617727452992], [0.13917734283523964, -0.83620085111886411, 0.23398407194295245, 
0.31146881246574765]], [[-0.029507824602079902, 0.1574243100965087, 0.31486576471582972, -0.95685787951979218], 
[-0.48899322354200181, -0.94149220476950357, -0.24568012939396544, -0.15845295863431397], [0.29158080714097045, 
0.23100712186933658, 0.7981595934007133, 0.38286456647335521]]], [[[0.61716197962362473, -0.45981943122067626, 
0.77169576646882287, -0.61914888379614541], [0.85941618082643645, 0.3953325518632147, -0.071709454511042603, 
-0.16346196683480119], [-0.27195889282127017, 0.640204102093195, 0.36381437568430219, -0.23698452461448793]], 
[[-0.19651750243903776, -0.85806312069582069, -0.16697127876879536, -0.13043639826490505], [0.30397572395834271, 
-0.72931842874315134, -0.042718535847001471, 0.29178523953544122], [-0.26391565154853391, -0.45797052486847045, 
-0.31300470368371225, -0.25426548079345146]]], [[[0.25564008776185876, -0.33973920939409541, -0.42791398656409929, 
-0.52664151938223003], [-0.10999100193161926, -0.29629666691577783, -0.40782973073033224, -0.7964414558802948], 
[0.76976134456727685, -0.91549509332347023, 0.72186262412866609, -0.56773143000653792]], [[-0.42862609106683847, 
-0.35583358749489524, 0.87176384608629998, -0.6521142543462386], [0.067116251967670904, 0.48376703343805683, 
-0.38447080115823962, -0.52461174025121959], [0.95864693409401736, -0.18283658347090248, 0.65543503798526181, 
0.34294213784697214]]]])
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[[-0.91334699106125039, 0.08034469059439342, 0.95336683070903305, -0.30302709745042145], 
[0.67698272338736065, -0.081495809822069276, -0.96880332310388173, -0.65951169314669], [0.14645181982635735, 
0.95081750997466497, 0.070403435387265789, 0.86128470750091801]], [[0.31458850366898372, 0.37355625042636609, 
0.59835306604159055, -0.55972405824577054], [0.58982641275618297, -0.92890562173951086, -0.30083832088441054, 
0.89547691587223865], [-0.63027121919601004, -0.64536171128018927, 0.24414107153983289, 0.87244828606233771]]], 
[[[-0.89466900572920127, 0.5598293778988026, -0.89619884347773127, -0.88632252361082808], [-0.43917654443748066, 
-0.26084937566420874, 0.085241904688784231, 0.91415146709412731], [0.21564731418211958, 0.48337780597622526, 
0.19274924619906364, 0.68742605559501802]], [[-0.52271461311172263, 0.86796787718622181, 0.60388060730113802, 
0.59164210921511518], [0.06016452677678541, -0.69422396793023222, -0.94949047918679574, -0.023652022509516479], 
[0.47311564439251907, -0.83556354001059763, 0.30169446600775252, -0.41514884541574859]]], [[[0.94784819259638442, 
-0.96878672195987425, 0.41736597890415839, -0.54187648571449287], [0.087381032350053456, 0.45323520162423536, 
0.04915039358163531, 0.44902577072276673], [-0.36295093024222025, -0.3956861841715571, -0.26069456297899429, 
0.96853061131817464]], [[0.11413660171644269, -0.36860481482212748, -0.65069198361079228, 0.46860889975506259], 
[-0.96249981724552258, -0.64237067775573453, -0.20746509600546581, -0.41040214652163631], [-0.38498880402659741, 
0.97133351909870091, -0.20699459203726933, 0.52388252445004002]]]])+(1.-msk_arg1)*numpy.array([[[[0.59326071032227223, 
-0.80869719454086364, 0.94238621327406347, -0.84308193696275091], [0.70828574326935856, 0.91353233044236459, 
-0.15648092096926169, 0.18059659161430264], [-0.37080639325253117, -0.096042972458373566, -0.50397349081083131, 
0.90885407233921245]], [[-0.36356977188789052, 0.07834805962528657, -0.50427040211387508, -0.17978606506745876], 
[0.0037874599835847622, 0.5356158232271877, -0.95158338448852087, 0.77887517249080407], [-0.97788888695396725, 
0.025164090720837473, 0.61062564247754847, 0.50036684617541782]]], [[[-0.51188738745996476, -0.56019687946102437, 
0.5290236210334176, -0.59831305037402949], [0.16156648393526041, 0.013553898362512085, 0.47562501938041502, 
0.88759773681313248], [0.8561694563853337, 0.60259960885555874, -0.082171173450925172, 0.63858906210461575]], 
[[-0.46713128615175425, -0.055061913628714487, -0.26470592863123388, 0.12914283139830651], [0.30407054362431873, 
-0.51481475425475165, -0.61454326667132175, -0.83597993953409167], [-0.361452844057218, -0.14442685496647223, 
0.78922625506892952, -0.60792243592735828]]], [[[-0.49703903944354133, 0.095413976577233273, 0.16265112247299029, 
0.61673952395802845], [-0.88844924670122549, -0.36852319019343116, 0.90935427930023738, -0.29773383810366583], 
[-0.80667363405005532, -0.16023550925636854, 0.40614983587173814, -0.81931798357574137]], [[0.75510295592173726, 
-0.43123287953825495, 0.70147960136393661, 0.14788978483969961], [-0.23220064658101736, 0.5689017497000739, 
0.012773766156810806, 0.088171873737747708], [-0.67756995292205335, -0.43998855003188519, 0.96781348177706006, 
0.8176918424069477]]]])
      res=maximum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[[0.56285787359423467, 0.37775535632450929, 0.95336683070903305, -0.30302709745042145], 
[0.67698272338736065, -0.050567052731754325, -0.54905395376446253, -0.65951169314669], [0.14645181982635735, 
0.95081750997466497, 0.070403435387265789, 0.86128470750091801]], [[0.31458850366898372, 0.55431246693940839, 
0.59835306604159055, -0.42863009943253139], [0.80323187807206664, -0.74856933876086118, -0.01644211276713814, 
0.89547691587223865], [0.40224351184954954, -0.64536171128018927, 0.24414107153983289, 0.87244828606233771]]], 
[[[0.82728992326268469, 0.5598293778988026, -0.89619884347773127, -0.28209813227511105], [-0.43917654443748066, 
0.74506459052200502, 0.085241904688784231, 0.91415146709412731], [0.70832242911523857, 0.48337780597622526, 
0.51617654973055238, 0.88129235546352791]], [[0.58177700099759, 0.86796787718622181, 0.86508533718829983, 0.59164210921511518], 
[0.34909094815239139, -0.69422396793023222, -0.83626668667446791, -0.023652022509516479], [0.47311564439251907, 
-0.29228298836204991, 0.30169446600775252, -0.39919385201564572]]], [[[0.94784819259638442, 0.78785581682131256, 
0.41736597890415839, -0.54187648571449287], [0.087381032350053456, 0.69422301523334684, 0.080299003664048518, 
0.44902577072276673], [0.37557752598770766, -0.11425319383355337, 0.43620438053394817, 0.96853061131817464]], 
[[0.11413660171644269, -0.36860481482212748, -0.30885873461055691, 0.46860889975506259], [0.44906982794974248, 
0.47078393816377861, -0.20746509600546581, 0.57388498777390184], [0.53885977209128866, 0.97133351909870091, 
0.29270123507386536, 0.76740426807290363]]]])+(1.-msk_ref)*numpy.array([[[[0.59326071032227223, 0.57101135857695096, 
0.94238621327406347, -0.55893684971502733], [0.70828574326935856, 0.91353233044236459, 0.29566092180536008, 
0.18059659161430264], [0.13917734283523964, -0.096042972458373566, 0.23398407194295245, 0.90885407233921245]], 
[[-0.029507824602079902, 0.1574243100965087, 0.31486576471582972, -0.17978606506745876], [0.0037874599835847622, 
0.5356158232271877, -0.24568012939396544, 0.77887517249080407], [0.29158080714097045, 0.23100712186933658, 0.7981595934007133, 
0.50036684617541782]]], [[[0.61716197962362473, -0.45981943122067626, 0.77169576646882287, -0.59831305037402949], 
[0.85941618082643645, 0.3953325518632147, 0.47562501938041502, 0.88759773681313248], [0.8561694563853337, 0.640204102093195, 
0.36381437568430219, 0.63858906210461575]], [[-0.19651750243903776, -0.055061913628714487, -0.16697127876879536, 
0.12914283139830651], [0.30407054362431873, -0.51481475425475165, -0.042718535847001471, 0.29178523953544122], 
[-0.26391565154853391, -0.14442685496647223, 0.78922625506892952, -0.25426548079345146]]], [[[0.25564008776185876, 
0.095413976577233273, 0.16265112247299029, 0.61673952395802845], [-0.10999100193161926, -0.29629666691577783, 
0.90935427930023738, -0.29773383810366583], [0.76976134456727685, -0.16023550925636854, 0.72186262412866609, 
-0.56773143000653792]], [[0.75510295592173726, -0.35583358749489524, 0.87176384608629998, 0.14788978483969961], 
[0.067116251967670904, 0.5689017497000739, 0.012773766156810806, 0.088171873737747708], [0.95864693409401736, 
-0.18283658347090248, 0.96781348177706006, 0.8176918424069477]]]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(3, 2, 3, 4),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_float_rank0(self):
      arg0=0.576130683834
      arg1=0.412116338185
      res=minimum(arg0,arg1)
      ref=0.412116338185
      self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_array_rank0(self):
      arg0=0.289003350398
      arg1=numpy.array(-0.445379036457)
      res=minimum(arg0,arg1)
      ref=numpy.array(-0.445379036457)
      if isinstance(res,numpy.ndarray):
         self.assertEqual(res.shape,(),"wrong shape of result.")
      else:
         self.assertTrue(isinstance(res,float),"wrong type of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_constData_rank0(self):
      arg0=0.371238578658
      arg1=Data(-0.645135030501,self.functionspace)
      res=minimum(arg0,arg1)
      ref=Data(-0.645135030501,self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_expandedData_rank0(self):
      arg0=-0.911968300013
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*(-0.806502526946)+(1.-msk_arg1)*(-0.438461765499)
      res=minimum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*(-0.911968300013)+(1.-msk_ref)*(-0.911968300013)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_array_rank1(self):
      arg0=-0.819071116627
      arg1=numpy.array([0.31932661296889409, -0.069005867155198297])
      res=minimum(arg0,arg1)
      ref=numpy.array([-0.81907111662704235, -0.81907111662704235])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_constData_rank1(self):
      arg0=-0.968249647669
      arg1=Data(numpy.array([-0.52214145626118125, -0.58128985653909826]),self.functionspace)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([-0.96824964766889465, -0.96824964766889465]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_expandedData_rank1(self):
      arg0=0.925021455714
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([0.51802917099944845, -0.73552775153957839])+(1.-msk_arg1)*numpy.array([0.94684306680258179, 
0.90506150125803431])
      res=minimum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([0.51802917099944845, -0.73552775153957839])+(1.-msk_ref)*numpy.array([0.92502145571355165, 
0.90506150125803431])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(2,),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_array_rank2(self):
      arg0=0.478671800384
      arg1=numpy.array([[0.90441638908409461, 0.65105243799339574, 0.55757907837341425, 0.96796203872288622, 
0.65429378508429648], [-0.9839204249726301, -0.074489762658563885, 0.60404519273294666, -0.49171021040278884, 
-0.96236949107971359], [-0.34775021026631392, 0.03117292968711971, -0.12859207137734652, 0.82761350470691464, 
-0.47178241252823083], [0.34614973425604645, 0.54814360606236878, 0.73720876600461183, 0.76149194774572737, 
0.22429834323918452]])
      res=minimum(arg0,arg1)
      ref=numpy.array([[0.47867180038398538, 0.47867180038398538, 0.47867180038398538, 0.47867180038398538, 
0.47867180038398538], [-0.9839204249726301, -0.074489762658563885, 0.47867180038398538, -0.49171021040278884, 
-0.96236949107971359], [-0.34775021026631392, 0.03117292968711971, -0.12859207137734652, 0.47867180038398538, 
-0.47178241252823083], [0.34614973425604645, 0.47867180038398538, 0.47867180038398538, 0.47867180038398538, 
0.22429834323918452]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_constData_rank2(self):
      arg0=-0.720838996247
      arg1=Data(numpy.array([[-0.3194234988515039, 0.61804206873773371, 0.38173572485452434, 0.83348152044490842, 
0.62513540816554869], [-0.8843675267723512, 0.47081586165207501, -0.8954880719672702, 0.87135071329587555, 
0.45445544130117765], [0.45909558655893501, -0.2196943692231903, -0.84367859877127671, -0.8839183602265086, 
0.57583179655742933], [0.70220506269758887, -0.96222604760788122, -0.73678170743805582, -0.49608249020194473, 
-0.80887113413205292]]),self.functionspace)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[-0.7208389962473174, -0.7208389962473174, -0.7208389962473174, -0.7208389962473174, 
-0.7208389962473174], [-0.8843675267723512, -0.7208389962473174, -0.8954880719672702, -0.7208389962473174, 
-0.7208389962473174], [-0.7208389962473174, -0.7208389962473174, -0.84367859877127671, -0.8839183602265086, 
-0.7208389962473174], [-0.7208389962473174, -0.96222604760788122, -0.73678170743805582, -0.7208389962473174, 
-0.80887113413205292]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_expandedData_rank2(self):
      arg0=0.495699222644
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[0.49930708441927374, 0.7877563342214875, 0.18718421840595245, 0.28640212239203855, 
-0.098555309394831481], [0.13795575274883176, -0.40917614030441851, 0.32903723135509, -0.38623631617861931, 
0.029402757771769261], [-0.96910292054156155, -0.56109509853717054, -0.10246680739939507, 0.3287515340044147, 
0.17887298858214229], [-0.81089405837278794, 0.042371551758132675, -0.51600300197393367, -0.55464369981571227, 
-0.47783840203259986]])+(1.-msk_arg1)*numpy.array([[-0.051685226236429349, -0.13272660137191084, 0.78576887405219753, 
-0.7748999659405067, 0.14209476128681842], [0.84845388952803136, -0.26771507129818128, 0.04001228027127457, 
-0.65756325918272207, 0.83341994005464581], [0.69877677820696116, -0.34424091482753227, -0.33461693696891515, 
-0.77592869887258065, 0.065093052535560192], [0.93191889936701977, -0.88228320287439876, 0.96359621418279051, 
-0.61215698455214462, 0.57613852010423794]])
      res=minimum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[0.4956992226436987, 0.4956992226436987, 0.18718421840595245, 0.28640212239203855, 
-0.098555309394831481], [0.13795575274883176, -0.40917614030441851, 0.32903723135509, -0.38623631617861931, 
0.029402757771769261], [-0.96910292054156155, -0.56109509853717054, -0.10246680739939507, 0.3287515340044147, 
0.17887298858214229], [-0.81089405837278794, 0.042371551758132675, -0.51600300197393367, -0.55464369981571227, 
-0.47783840203259986]])+(1.-msk_ref)*numpy.array([[-0.051685226236429349, -0.13272660137191084, 0.4956992226436987, 
-0.7748999659405067, 0.14209476128681842], [0.4956992226436987, -0.26771507129818128, 0.04001228027127457, 
-0.65756325918272207, 0.4956992226436987], [0.4956992226436987, -0.34424091482753227, -0.33461693696891515, 
-0.77592869887258065, 0.065093052535560192], [0.4956992226436987, -0.88228320287439876, 0.4956992226436987, 
-0.61215698455214462, 0.4956992226436987]])
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(4, 5),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_array_rank3(self):
      arg0=0.637928830253
      arg1=numpy.array([[[-0.35653583772730757, 0.61571708422109794], [0.07350669790974762, 0.61876388475252808]], 
[[-0.053463270495726967, 0.83419890725283885], [-0.96099905287190679, -0.80575372226068542]], [[0.24300609133706175, 
0.2756588695642721], [-0.87654405373062816, -0.3200825567458685]], [[-0.34218532069750118, -0.4170463384581613], 
[0.53608434420038287, -0.26824412326699676]], [[0.56718420178173146, -0.75344847265520709], [0.4436615786631195, 
-0.52972537223344784]], [[-0.75515269605520174, 0.50931856772376993], [-0.84961959248176089, 0.02703055814068378]]])
      res=minimum(arg0,arg1)
      ref=numpy.array([[[-0.35653583772730757, 0.61571708422109794], [0.07350669790974762, 0.61876388475252808]], 
[[-0.053463270495726967, 0.63792883025270131], [-0.96099905287190679, -0.80575372226068542]], [[0.24300609133706175, 
0.2756588695642721], [-0.87654405373062816, -0.3200825567458685]], [[-0.34218532069750118, -0.4170463384581613], 
[0.53608434420038287, -0.26824412326699676]], [[0.56718420178173146, -0.75344847265520709], [0.4436615786631195, 
-0.52972537223344784]], [[-0.75515269605520174, 0.50931856772376993], [-0.84961959248176089, 0.02703055814068378]]])
      self.assertTrue(isinstance(res,numpy.ndarray),"wrong type of result.")
      self.assertEqual(res.shape,(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_constData_rank3(self):
      arg0=-0.310043459727
      arg1=Data(numpy.array([[[0.38697328105644235, 0.70221233643550196], [0.34544746758392653, -0.11055874729875126]], 
[[0.42778446857838048, -0.8880788764748333], [0.73132093952186783, 0.8033154117025938]], [[-0.5661335107684673, 
0.49843795591687612], [0.22595639372635068, 0.044169583330319506]], [[-0.9124201688421103, 0.67544031532457383], 
[0.25484397741761766, 0.047857857156642147]], [[0.91698387844722751, -0.63314665729291408], [-0.56451465364624975, 
0.97099413120842981]], [[-0.96623694702119445, -0.36559666676379576], [0.48002112676354947, 
-0.49901854944236645]]]),self.functionspace)
      res=minimum(arg0,arg1)
      ref=Data(numpy.array([[[-0.31004345972712621, -0.31004345972712621], [-0.31004345972712621, -0.31004345972712621]], 
[[-0.31004345972712621, -0.8880788764748333], [-0.31004345972712621, -0.31004345972712621]], [[-0.5661335107684673, 
-0.31004345972712621], [-0.31004345972712621, -0.31004345972712621]], [[-0.9124201688421103, -0.31004345972712621], 
[-0.31004345972712621, -0.31004345972712621]], [[-0.31004345972712621, -0.63314665729291408], [-0.56451465364624975, 
-0.31004345972712621]], [[-0.96623694702119445, -0.36559666676379576], [-0.31004345972712621, 
-0.49901854944236645]]]),self.functionspace)
      self.assertTrue(isinstance(res,Data),"wrong type of result.")
      self.assertEqual(res.getShape(),(6, 2, 2),"wrong shape of result.")
      self.assertTrue(Lsup(res-ref)<=self.RES_TOL*Lsup(ref),"wrong result")
   #+++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++++
   def test_minimum_float_rank0_expandedData_rank3(self):
      arg0=0.180877844009
      msk_arg1=whereNegative(self.functionspace.getX()[0]-0.5)
      arg1=msk_arg1*numpy.array([[[-0.72765769710886885, -0.77950756222280115], [-0.76657464297493383, 0.26360833458842103]], 
[[-0.63889942876567685, -0.96415887226926067], [0.40819049952211262, 0.53747956489837434]], [[-0.17320920691384134, 
-0.66907140584029845], [0.12483196580198208, -0.70507661237760755]], [[0.071507590554227729, 0.062086896243514822], 
[-0.37700203793464682, -0.06698451825887064]], [[0.99908694441096602, -0.49471711706132759], [-0.25036019181025893, 
0.69865642814709528]], [[-0.47164774628773198, 0.19617691072048715], [-0.50986442038570967, 
0.18436333118518289]]])+(1.-msk_arg1)*numpy.array([[[0.10851375096232618, -0.13477534514489853], [0.18640776317948937, 
-0.18353749086282534]], [[0.64663017185804406, 0.25612468253827858], [-0.64860384050774189, -0.26232530692787082]], 
[[0.51631302366268272, 0.74589200206144768], [0.2374091871954056, 0.055785511654019571]], [[-0.8453176382142269, 
-0.10030284644757259], [-0.56959526008388495, 0.13512355223289974]], [[0.99581050667376392, 0.48594409950935535], 
[-0.92853699670508494, -0.8971336427464387]], [[0.69034433856458577, 0.14940275380447643], [-0.23670374220161006, 
0.79042782527452959]]])
      res=minimum(arg0,arg1)
      msk_ref=whereNegative(self.functionspace.getX()[0]-0.5)
      ref=msk_ref*numpy.array([[[-0.72765769710886885, -0.77950756222280115], [-0.76657464297493383, 0.18087784400928597]], 
[[-0.63889942876567685, -0.96415887226926067], [0.18087784400928597, 0.180