"""Collection of bulk systems.

From here::

   http://molmod.ugent.be/DeltaCodesDFT

   Data extracted to csv file with

   * Wien2K ref:

     pdftotext -layout -f 13 -l 14 SupplMat.pdf - | sed -n '/text file/,$p' \
     | grep -E -v "text" | sed '/^$/d' | tr '\f' ' ' \
     | sed 's/,WIEN2k//g' | sed 's/V0/formula,V0/g' | sed -e 's/^[ \t]*//' \
     | sed -e 's/\s\+/,/g'

"""

import os
import glob
import pprint
import urllib
import tarfile
import zipfile

import numpy as np

from numpy import array

from ase import Atoms
import ase.io

from ase.tasks.io import read_json
from ase.tasks.bulk import BulkTask

from ase.utils.eos import EquationOfState
from ase.test import NotAvailable

try:
    import urllib2
except ImportError:
    raise NotAvailable



class FullEquationOfState(EquationOfState):

    """Fit equation of state for bulk systems.

    Based on eosfit.py from http://molmod.ugent.be/DeltaCodesDFT

    """

    def __init__(self, volumes, energies, eos='birch'):
        assert eos == 'birch', eos + ' eos not available.'
        self.v = np.array(volumes)
        self.e = np.array(energies)
        self.eos_string = 'birch'

        self.v0 = None

    def fit(self):
        """Calculate volume (v0), energy (e0), bulk modulus (B0), and
        bulk modulus pressure derivative (B1).

        Returns v0, e0, B0, B1, fit residuals.

        Notice that the ASE units for the bulk modulus is
        eV/Angstrom^3 - to get the value in GPa, do this::

          v0, e0, B0, B1, R = eos.fit()
          print(B0 / kJ * 1.0e24, 'GPa')

        """

        fitdata = np.polyfit(self.v**(-2./3.), self.e, 3, full=True)
        ssr = fitdata[1]
        sst = np.sum((self.e - np.average(self.e))**2.)
        residuals0 = ssr/sst
        deriv0 = np.poly1d(fitdata[0])
        deriv1 = np.polyder(deriv0, 1)
        deriv2 = np.polyder(deriv1, 1)
        deriv3 = np.polyder(deriv2, 1)

        self.v0 = None
        for x in np.roots(deriv1):
            if x > 0 and deriv2(x) > 0:
                self.v0 = x**(-3./2.)
                break

        if self.v0 is None:
            raise ValueError('No minimum!')

        derivV2 = 4./9. * x**5. * deriv2(x)
        derivV3 = (-20./9. * x**(13./2.) * deriv2(x) -
                   8./27. * x**(15./2.) * deriv3(x))
        bulk_modulus0 = derivV2 / x**(3./2.)
        bulk_deriv0 = -1 - x**(-3./2.) * derivV3 / derivV2

        self.e0 = deriv0(x)
        self.B0 = bulk_modulus0
        self.B1 = bulk_deriv0

        return self.v0, self.e0, self.B0, self.B1, residuals0


class DeltaCodesDFTCollection:

    # https://molmod.ugent.be/sites/default/files/Delta_v3-0_0.zip
    wienref = """
    formula,V0,B0,B1
    H,17.3883,10.284,2.71
    He,17.7708,0.847,7.71
    Li,20.2191,13.839,3.34
    Be,7.9099,122.903,3.04
    B,7.2405,237.290,3.47
    C,11.6366,208.991,3.58
    N,28.8848,54.2195,3.7244
    O,18.5590,51.378,3.89
    F,19.1666,34.325,3.93
    Ne,24.2492,1.406,14.44
    Na,37.4686,7.472,3.77
    Mg,22.9355,35.933,4.07
    Al,16.4796,78.077,4.57
    Si,20.4530,88.545,4.31
    P,21.4709,68.208,4.35
    S,17.1840,83.407,4.26
    Cl,38.8889,19.081,4.34
    Ar,52.3852,0.743,7.26
    K,73.6793,3.574,4.59
    Ca,42.1991,17.114,3.31
    Sc,24.6196,54.393,3.42
    Ti,17.3900,112.213,3.58
    V,13.4520,181.674,3.75
    Cr,11.7730,183.899,7.16
    Mn,11.4473,118.632,-0.21
    Fe,11.3436,197.652,5.80
    Co,10.8599,217.295,4.37
    Ni,10.8876,200.368,5.00
    Cu,11.9511,141.335,4.86
    Zn,15.1820,74.780,5.26
    Ga,20.3069,49.223,5.38
    Ge,23.9148,59.128,4.99
    As,22.5890,68.285,4.22
    Se,29.7437,47.070,4.44
    Br,39.4470,22.415,4.87
    Kr,65.6576,0.671,9.86
    Rb,90.8087,2.787,5.80
    Sr,54.5272,11.256,3.49
    Y,32.8442,41.593,3.02
    Zr,23.3850,93.684,3.21
    Nb,18.1368,171.270,3.55
    Mo,15.7862,258.928,4.33
    Tc,14.4366,299.149,4.46
    Ru,13.7619,312.502,4.95
    Rh,14.0396,257.824,5.32
    Pd,15.3101,168.629,5.56
    Ag,17.8471,90.148,5.42
    Cd,22.6287,46.403,6.92
    In,27.4710,34.937,4.78
    Sn,36.8166,36.030,4.64
    Sb,31.7296,50.367,4.52
    Te,34.9765,44.787,4.69
    I,50.2333,18.654,5.05
    Xe,86.6814,0.548,6.34
    Cs,117.080,1.982,2.14
    Ba,63.1401,8.677,3.77
    Lu,29.0544,46.384,2.94
    Hf,22.5325,107.004,3.50
    Ta,18.2856,195.147,3.71
    W,16.1394,301.622,4.28
    Re,14.9580,362.850,4.52
    Os,14.2802,397.259,4.84
    Ir,14.5004,347.680,5.18
    Pt,15.6420,248.711,5.46
    Au,17.9745,139.109,5.76
    Hg,29.5220,8.204,8.87
    Tl,31.3902,26.865,5.49
    Pb,32.0028,39.544,4.53
    Bi,36.9047,42.630,4.70
    Po,37.5869,45.458,4.93
    Rn,92.6852,0.564,8.62
    """

    # http://molmod.ugent.be/sites/default/files/Delta_v1-1.zip
    atoms = {
 'Ag': {'cell': array([[  4.16424000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.54986159e-16,   4.16424000e+00,   0.00000000e+00],
       [  2.54986159e-16,   2.54986159e-16,   4.16424000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.54986159e-16,   2.08212000e+00,   2.08212000e+00],
       [  2.08212000e+00,   1.27493080e-16,   2.08212000e+00],
       [  2.08212000e+00,   2.08212000e+00,   0.00000000e+00]]),
        'symbols': ['Ag', 'Ag', 'Ag', 'Ag']},
 'Al': {'cell': array([[  4.04021000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.47391512e-16,   4.04021000e+00,   0.00000000e+00],
       [  2.47391512e-16,   2.47391512e-16,   4.04021000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.47391512e-16,   2.02010500e+00,   2.02010500e+00],
       [  2.02010500e+00,   1.23695756e-16,   2.02010500e+00],
       [  2.02010500e+00,   2.02010500e+00,   0.00000000e+00]]),
        'symbols': ['Al', 'Al', 'Al', 'Al']},
 'Ar': {'cell': array([[  5.95059000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.64368550e-16,   5.95059000e+00,   0.00000000e+00],
       [  3.64368550e-16,   3.64368550e-16,   5.95059000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.64368550e-16,   2.97529500e+00,   2.97529500e+00],
       [  2.97529500e+00,   1.82184275e-16,   2.97529500e+00],
       [  2.97529500e+00,   2.97529500e+00,   0.00000000e+00]]),
        'symbols': ['Ar', 'Ar', 'Ar', 'Ar']},
 'As': {'cell': array([[  3.82055000e+00,   0.00000000e+00,   0.00000000e+00],
       [ -1.91027500e+00,   3.30869336e+00,   0.00000000e+00],
       [  6.60674292e-16,   1.14432144e-15,   1.07896300e+01]]),
        'magmoms': None,
        'positions': array([[  1.49688974e-16,   2.59268909e-16,   2.44460647e+00],
       [  5.10985318e-16,   8.85052532e-16,   8.34502353e+00],
       [  1.91029410e+00,   1.10288676e+00,   6.04111384e+00],
       [  1.91029410e+00,   1.10288676e+00,   1.15190090e+00],
       [ -1.91027500e-05,   2.20580660e+00,   9.63772910e+00],
       [ -1.91027500e-05,   2.20580660e+00,   4.74851616e+00]]),
        'symbols': ['As', 'As', 'As', 'As', 'As', 'As']},
 'Au': {'cell': array([[  4.17410000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.55589910e-16,   4.17410000e+00,   0.00000000e+00],
       [  2.55589910e-16,   2.55589910e-16,   4.17410000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.55589910e-16,   2.08705000e+00,   2.08705000e+00],
       [  2.08705000e+00,   1.27794955e-16,   2.08705000e+00],
       [  2.08705000e+00,   2.08705000e+00,   0.00000000e+00]]),
        'symbols': ['Au', 'Au', 'Au', 'Au']},
 'B': {'cell': array([[ 4.90067   ,  0.        ,  0.        ],
       [-2.450335  ,  4.24410472,  0.        ],
       [ 2.45033259, -1.41470018,  4.18412387]]),
       'magmoms': None,
       'positions': array([[  6.47799802e+00,  -9.10670757e-01,   2.82281917e+00],
       [ -1.57733043e+00,   3.74007529e+00,   1.36130470e+00],
       [  4.90064387e+00,   1.82138677e+00,   2.82281917e+00],
       [  2.37195308e-05,   1.00801776e+00,   1.36130470e+00],
       [  3.32328972e+00,  -9.10670757e-01,   2.82281917e+00],
       [  1.57737787e+00,   3.74007529e+00,   1.36130470e+00],
       [  3.45323244e+00,   8.35678467e-01,   3.04353170e-01],
       [  1.44743515e+00,   1.99372607e+00,   3.87977070e+00],
       [  2.45035933e+00,   2.57270564e+00,   3.04353170e-01],
       [  2.45030826e+00,   2.56698890e-01,   3.87977070e+00],
       [  1.44748622e+00,   8.35678467e-01,   3.04353170e-01],
       [  3.45318137e+00,   1.99372607e+00,   3.87977070e+00]]),
       'symbols': ['B',
                   'B',
                   'B',
                   'B',
                   'B',
                   'B',
                   'B',
                   'B',
                   'B',
                   'B',
                   'B',
                   'B']},
 'Ba': {'cell': array([[  5.02792000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.07871307e-16,   5.02792000e+00,   0.00000000e+00],
       [  3.07871307e-16,   3.07871307e-16,   5.02792000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.     ,  0.     ,  0.     ],
       [ 2.51396,  2.51396,  2.51396]]),
        'symbols': ['Ba', 'Ba']},
 'Be': {'cell': array([[  2.26276000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.13138000e+00,   1.95960764e+00,   0.00000000e+00],
       [  2.18792948e-16,   1.26320167e-16,   3.57316000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.13136869,  0.65319602,  1.78658   ]]),
        'symbols': ['Be', 'Be']},
 'Bi': {'cell': array([[  4.58866000e+00,   0.00000000e+00,   0.00000000e+00],
       [ -2.29433000e+00,   3.97389613e+00,   0.00000000e+00],
       [  7.45946449e-16,   1.29201715e-15,   1.21822300e+01]]),
        'magmoms': None,
        'positions': array([[  1.74357523e-16,   3.01996088e-16,   2.84747444e+00],
       [  5.71588926e-16,   9.90021061e-16,   9.33475556e+00],
       [  2.29435294e+00,   1.32461880e+00,   6.90817717e+00],
       [  2.29435294e+00,   1.32461880e+00,   1.21322829e+00],
       [ -2.29433000e-05,   2.64927733e+00,   1.09690017e+01],
       [ -2.29433000e-05,   2.64927733e+00,   5.27405283e+00]]),
        'symbols': ['Bi', 'Bi', 'Bi', 'Bi', 'Bi', 'Bi']},
 'Br': {'cell': array([[  8.22860000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.58848083e-16,   4.22731000e+00,   0.00000000e+00],
       [  5.53125810e-16,   5.53125810e-16,   9.03323000e+00]]),
        'magmoms': None,
        'positions': array([[  5.85581169e-16,   1.58836946e+00,   7.97489677e+00],
       [  2.43923148e-16,   5.25285541e-01,   3.45828177e+00],
       [  5.68050745e-16,   3.70202446e+00,   5.57494823e+00],
       [  2.26392724e-16,   2.63894054e+00,   1.05833323e+00],
       [  4.11430000e+00,   3.70202446e+00,   7.97489677e+00],
       [  4.11430000e+00,   2.63894054e+00,   3.45828177e+00],
       [  4.11430000e+00,   1.58836946e+00,   5.57494823e+00],
       [  4.11430000e+00,   5.25285541e-01,   1.05833323e+00]]),
        'symbols': ['Br', 'Br', 'Br', 'Br', 'Br', 'Br', 'Br', 'Br']},
 'C': {'cell': array([[  2.46857000e+00,   0.00000000e+00,   0.00000000e+00],
       [ -1.23428500e+00,   2.13784433e+00,   0.00000000e+00],
       [  5.41342259e-16,   9.37632296e-16,   8.84079000e+00]]),
       'magmoms': None,
       'positions': array([[  1.35335565e-16,   2.34408074e-16,   2.21019750e+00],
       [  4.06006694e-16,   7.03224222e-16,   6.63059250e+00],
       [ -1.23428500e-05,   1.42523668e+00,   2.21019750e+00],
       [  1.23429734e+00,   7.12607651e-01,   6.63059250e+00]]),
       'symbols': ['C', 'C', 'C', 'C']},
 'Ca': {'cell': array([[  5.52507000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.38312965e-16,   5.52507000e+00,   0.00000000e+00],
       [  3.38312965e-16,   3.38312965e-16,   5.52507000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.38312965e-16,   2.76253500e+00,   2.76253500e+00],
       [  2.76253500e+00,   1.69156482e-16,   2.76253500e+00],
       [  2.76253500e+00,   2.76253500e+00,   0.00000000e+00]]),
        'symbols': ['Ca', 'Ca', 'Ca', 'Ca']},
 'Cd': {'cell': array([[  3.03716000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.51858000e+00,   2.63025772e+00,   0.00000000e+00],
       [  3.53347953e-16,   2.04005536e-16,   5.77061000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.51856481,  0.8767438 ,  2.885305  ]]),
        'symbols': ['Cd', 'Cd']},
 'Cl': {'cell': array([[  7.77875000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.66324552e-16,   4.34941000e+00,   0.00000000e+00],
       [  5.52777398e-16,   5.52777398e-16,   9.02754000e+00]]),
        'magmoms': None,
        'positions': array([[  6.05538240e-16,   1.77812580e+00,   8.11106414e+00],
       [  2.44554210e-16,   3.96579204e-01,   3.59729414e+00],
       [  5.74547740e-16,   3.95283080e+00,   5.43024586e+00],
       [  2.13563710e-16,   2.57128420e+00,   9.16475861e-01],
       [  3.88937500e+00,   3.95283080e+00,   8.11106414e+00],
       [  3.88937500e+00,   2.57128420e+00,   3.59729414e+00],
       [  3.88937500e+00,   1.77812580e+00,   5.43024586e+00],
       [  3.88937500e+00,   3.96579204e-01,   9.16475861e-01]]),
        'symbols': ['Cl', 'Cl', 'Cl', 'Cl', 'Cl', 'Cl', 'Cl', 'Cl']},
 'Co': {'cell': array([[  2.49680000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.24840000e+00,   2.16229223e+00,   0.00000000e+00],
       [  2.46815928e-16,   1.42499243e-16,   4.03081000e+00]]),
        'magmoms': [1.2, 1.2],
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.24838752,  0.72075687,  2.015405  ]]),
        'symbols': ['Co', 'Co']},
 'Cr': {'cell': array([[  2.87100000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.75798048e-16,   2.87100000e+00,   0.00000000e+00],
       [  1.75798048e-16,   1.75798048e-16,   2.87100000e+00]]),
        'magmoms': [1.5, -1.5],
        'positions': array([[ 0.    ,  0.    ,  0.    ],
       [ 1.4355,  1.4355,  1.4355]]),
        'symbols': ['Cr', 'Cr']},
 'Cs': {'cell': array([[  6.16153000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.77284900e-16,   6.16153000e+00,   0.00000000e+00],
       [  3.77284900e-16,   3.77284900e-16,   6.16153000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.      ,  0.      ,  0.      ],
       [ 3.080765,  3.080765,  3.080765]]),
        'symbols': ['Cs', 'Cs']},
 'Cu': {'cell': array([[  3.63689000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.22695285e-16,   3.63689000e+00,   0.00000000e+00],
       [  2.22695285e-16,   2.22695285e-16,   3.63689000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.22695285e-16,   1.81844500e+00,   1.81844500e+00],
       [  1.81844500e+00,   1.11347642e-16,   1.81844500e+00],
       [  1.81844500e+00,   1.81844500e+00,   0.00000000e+00]]),
        'symbols': ['Cu', 'Cu', 'Cu', 'Cu']},
 'F': {'cell': array([[  6.16310000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.25644234e-16,   3.68505000e+00,   0.00000000e+00],
       [ -8.17043395e-01,   4.74089763e-16,   6.87706484e+00]]),
       'magmoms': None,
       'positions': array([[ 1.49468221,  1.21717201,  0.64733811],
       [ 3.8513744 ,  2.46787799,  6.22972672],
       [ 4.2598961 ,  1.21717202,  2.79119431],
       [ 1.08616051,  2.46787799,  4.08587053],
       [ 4.57623221,  3.05969702,  0.64733811],
       [ 0.7698244 ,  0.62535299,  6.22972672],
       [ 1.1783461 ,  3.05969702,  2.79119431],
       [ 4.16771051,  0.62535299,  4.08587053]]),
       'symbols': ['F', 'F', 'F', 'F', 'F', 'F', 'F', 'F']},
 'Fe': {'cell': array([[  2.83351000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.73502448e-16,   2.83351000e+00,   0.00000000e+00],
       [  1.73502448e-16,   1.73502448e-16,   2.83351000e+00]]),
        'magmoms': [2.3, 2.3],
        'positions': array([[ 0.      ,  0.      ,  0.      ],
       [ 1.416755,  1.416755,  1.416755]]),
        'symbols': ['Fe', 'Fe']},
 'Ga': {'cell': array([[  4.56689000e+00,   0.00000000e+00,   0.00000000e+00],
       [  4.74752701e-16,   7.75330000e+00,   0.00000000e+00],
       [  2.81655293e-16,   2.81655293e-16,   4.59978000e+00]]),
        'magmoms': None,
        'positions': array([[  9.71312273e-17,   1.21277119e+00,   3.73502136e-01],
       [  6.59276767e-16,   6.54052881e+00,   4.22627786e+00],
       [  3.26813589e-16,   2.66387881e+00,   2.67339214e+00],
       [  4.29594405e-16,   5.08942119e+00,   1.92638786e+00],
       [  2.28344500e+00,   5.08942119e+00,   3.73502136e-01],
       [  2.28344500e+00,   2.66387881e+00,   4.22627786e+00],
       [  2.28344500e+00,   6.54052881e+00,   2.67339214e+00],
       [  2.28344500e+00,   1.21277119e+00,   1.92638786e+00]]),
        'symbols': ['Ga', 'Ga', 'Ga', 'Ga', 'Ga', 'Ga', 'Ga', 'Ga']},
 'Ge': {'cell': array([[ 4.07419   ,  0.        ,  0.        ],
       [ 2.037095  ,  3.52835204,  0.        ],
       [ 2.037095  ,  1.17611735,  3.32656221]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 2.037095  ,  1.17611735,  0.83164055]]),
        'symbols': ['Ge', 'Ge']},
 'H': {'cell': array([[  4.00759000e+00,   0.00000000e+00,   0.00000000e+00],
       [ -2.00379500e+00,   3.47067475e+00,   0.00000000e+00],
       [  3.06349683e-16,   5.30613216e-16,   5.00307000e+00]]),
       'magmoms': None,
       'positions': array([[ -2.00379500e-05,   2.31379473e+00,   8.75437189e-01],
       [  2.00381504e+00,   1.15688001e+00,   4.12763281e+00],
       [  2.00381504e+00,   1.15688001e+00,   3.37697219e+00],
       [ -2.00379500e-05,   2.31379473e+00,   1.62609781e+00]]),
       'symbols': ['H', 'H', 'H', 'H']},
 'He': {'cell': array([[  2.92661000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.46330500e+00,   2.53451861e+00,   0.00000000e+00],
       [  2.92637313e-16,   1.68954231e-16,   4.77913000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.46329037,  0.84483109,  2.389565  ]]),
        'symbols': ['He', 'He']},
 'Hf': {'cell': array([[  3.20273000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.60136500e+00,   2.77364554e+00,   0.00000000e+00],
       [  3.10137516e-16,   1.79057978e-16,   5.06493000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.60134899,  0.92453927,  2.532465  ]]),
        'symbols': ['Hf', 'Hf']},
 'Hg': {'cell': array([[  4.10305000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.51239352e-16,   4.10305000e+00,   0.00000000e+00],
       [  2.17271324e-16,   2.17271324e-16,   3.54831000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.      ,  0.      ,  0.      ],
       [ 2.051525,  2.051525,  1.774155]]),
        'symbols': ['Hg', 'Hg']},
 'I': {'cell': array([[  8.57701000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.78865547e-16,   4.55422000e+00,   0.00000000e+00],
       [  6.23578516e-16,   6.23578516e-16,   1.01838100e+01]]),
       'magmoms': None,
       'positions': array([[  6.49392323e-16,   1.65819150e+00,   8.94718995e+00],
       [  2.73965947e-16,   6.18918498e-01,   3.85528495e+00],
       [  6.28478117e-16,   3.93530150e+00,   6.32852505e+00],
       [  2.53051741e-16,   2.89602850e+00,   1.23662005e+00],
       [  4.28850500e+00,   3.93530150e+00,   8.94718995e+00],
       [  4.28850500e+00,   2.89602850e+00,   3.85528495e+00],
       [  4.28850500e+00,   1.65819150e+00,   6.32852505e+00],
       [  4.28850500e+00,   6.18918498e-01,   1.23662005e+00]]),
       'symbols': ['I', 'I', 'I', 'I', 'I', 'I', 'I', 'I']},
 'In': {'cell': array([[  3.29841000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.01969362e-16,   3.29841000e+00,   0.00000000e+00],
       [  3.09992395e-16,   3.09992395e-16,   5.06256000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.      ,  0.      ,  0.      ],
       [ 1.649205,  1.649205,  2.53128 ]]),
        'symbols': ['In', 'In']},
 'Ir': {'cell': array([[  3.87675000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.37382474e-16,   3.87675000e+00,   0.00000000e+00],
       [  2.37382474e-16,   2.37382474e-16,   3.87675000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.37382474e-16,   1.93837500e+00,   1.93837500e+00],
       [  1.93837500e+00,   1.18691237e-16,   1.93837500e+00],
       [  1.93837500e+00,   1.93837500e+00,   0.00000000e+00]]),
        'symbols': ['Ir', 'Ir', 'Ir', 'Ir']},
 'K': {'cell': array([[  5.28589000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.23667413e-16,   5.28589000e+00,   0.00000000e+00],
       [  3.23667413e-16,   3.23667413e-16,   5.28589000e+00]]),
       'magmoms': None,
       'positions': array([[ 0.      ,  0.      ,  0.      ],
       [ 2.642945,  2.642945,  2.642945]]),
       'symbols': ['K', 'K']},
 'Kr': {'cell': array([[  6.42930000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.93681083e-16,   6.42930000e+00,   0.00000000e+00],
       [  3.93681083e-16,   3.93681083e-16,   6.42930000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.93681083e-16,   3.21465000e+00,   3.21465000e+00],
       [  3.21465000e+00,   1.96840542e-16,   3.21465000e+00],
       [  3.21465000e+00,   3.21465000e+00,   0.00000000e+00]]),
        'symbols': ['Kr', 'Kr', 'Kr', 'Kr']},
 'Li': {'cell': array([[ 7.64004   ,  0.        ,  0.        ],
       [ 7.02117921,  3.01218421,  0.        ],
       [ 7.02117921,  1.44251886,  2.64431335]]),
        'magmoms': None,
        'positions': array([[  0.        ,   0.        ,   0.        ],
       [  4.81826258,   0.98992412,   0.58761931],
       [ 16.86413584,   3.46477895,   2.05669403]]),
        'symbols': ['Li', 'Li', 'Li']},
 'Lu': {'cell': array([[  3.52388000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.76194000e+00,   3.05176960e+00,   0.00000000e+00],
       [  3.35568531e-16,   1.93740582e-16,   5.48025000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.76192238,  1.01724636,  2.740125  ]]),
        'symbols': ['Lu', 'Lu']},
 'Mg': {'cell': array([[  3.19405000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.59702500e+00,   2.76612844e+00,   0.00000000e+00],
       [  3.16692438e-16,   1.82842464e-16,   5.17198000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.59700903,  0.92203359,  2.58599   ]]),
        'symbols': ['Mg', 'Mg']},
 'Mn': {'cell': array([[  3.59552000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.20162103e-16,   3.59552000e+00,   0.00000000e+00],
       [  2.20162103e-16,   2.20162103e-16,   3.59552000e+00]]),
        'magmoms': [2.0, 1.9, -2.0, -1.9],
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.20162103e-16,   1.79776000e+00,   1.79776000e+00],
       [  1.79776000e+00,   1.10081051e-16,   1.79776000e+00],
       [  1.79776000e+00,   1.79776000e+00,   0.00000000e+00]]),
        'symbols': ['Mn', 'Mn', 'Mn', 'Mn']},
 'Mo': {'cell': array([[  3.16930000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.94063655e-16,   3.16930000e+00,   0.00000000e+00],
       [  1.94063655e-16,   1.94063655e-16,   3.16930000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.     ,  0.     ,  0.     ],
       [ 1.58465,  1.58465,  1.58465]]),
        'symbols': ['Mo', 'Mo']},
 'N': {'cell': array([[  6.19019000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.79039818e-16,   6.19019000e+00,   0.00000000e+00],
       [  3.79039818e-16,   3.79039818e-16,   6.19019000e+00]]),
       'magmoms': None,
       'positions': array([[ 0.32120896,  0.32120896,  0.32120896],
       [ 5.86898104,  5.86898104,  5.86898104],
       [ 2.77388604,  5.86898104,  3.41630396],
       [ 3.41630396,  0.32120896,  2.77388604],
       [ 5.86898104,  3.41630396,  2.77388604],
       [ 0.32120896,  2.77388604,  3.41630396],
       [ 3.41630396,  2.77388604,  5.86898104],
       [ 2.77388604,  3.41630396,  0.32120896]]),
       'symbols': ['N', 'N', 'N', 'N', 'N', 'N', 'N', 'N']},
 'Na': {'cell': array([[ 9.41813   ,  0.        ,  0.        ],
       [ 8.67443415,  3.66815551,  0.        ],
       [ 8.67443415,  1.75868789,  3.21906535]]),
        'magmoms': None,
        'positions': array([[  0.        ,   0.        ,   0.        ],
       [  5.94816236,   1.20595314,   0.7153407 ],
       [ 20.81883594,   4.22089025,   2.50372465]]),
        'symbols': ['Na', 'Na', 'Na']},
 'Nb': {'cell': array([[  3.32239000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.03437714e-16,   3.32239000e+00,   0.00000000e+00],
       [  2.03437714e-16,   2.03437714e-16,   3.32239000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.      ,  0.      ,  0.      ],
       [ 1.661195,  1.661195,  1.661195]]),
        'symbols': ['Nb', 'Nb']},
 'Ne': {'cell': array([[  4.62262000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.83053839e-16,   4.62262000e+00,   0.00000000e+00],
       [  2.83053839e-16,   2.83053839e-16,   4.62262000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.83053839e-16,   2.31131000e+00,   2.31131000e+00],
       [  2.31131000e+00,   1.41526920e-16,   2.31131000e+00],
       [  2.31131000e+00,   2.31131000e+00,   0.00000000e+00]]),
        'symbols': ['Ne', 'Ne', 'Ne', 'Ne']},
 'Ni': {'cell': array([[  3.52414000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.15791339e-16,   3.52414000e+00,   0.00000000e+00],
       [  2.15791339e-16,   2.15791339e-16,   3.52414000e+00]]),
        'magmoms': [0.6, 0.6, 0.6, 0.6],
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.15791339e-16,   1.76207000e+00,   1.76207000e+00],
       [  1.76207000e+00,   1.07895669e-16,   1.76207000e+00],
       [  1.76207000e+00,   1.76207000e+00,   0.00000000e+00]]),
        'symbols': ['Ni', 'Ni', 'Ni', 'Ni']},
 'O': {'cell': array([[  4.27163000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.61879696e-16,   4.27682000e+00,   0.00000000e+00],
       [  1.80666882e+00,   1.69295858e-16,   4.19933056e+00]]),
       'magmoms': [1.5, 1.5, -1.5, -1.5],
       'positions': array([[  4.27100416e+00,   2.48306235e-17,   6.15915813e-01],
       [  1.80729466e+00,   1.44465234e-16,   3.58341475e+00],
       [  2.13518916e+00,   2.13841000e+00,   6.15915813e-01],
       [  3.94310966e+00,   2.13841000e+00,   3.58341475e+00]]),
       'symbols': ['O', 'O', 'O', 'O']},
 'Os': {'cell': array([[  2.75895000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.37947500e+00,   2.38932079e+00,   0.00000000e+00],
       [  2.66785631e-16,   1.54028756e-16,   4.35694000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.37946121,  0.7964323 ,  2.17847   ]]),
        'symbols': ['Os', 'Os']},
 'P': {'cell': array([[  3.30466000e+00,   0.00000000e+00,   0.00000000e+00],
       [  6.92962717e-16,   1.13169400e+01,   0.00000000e+00],
       [  2.80032023e-16,   2.80032023e-16,   4.57327000e+00]]),
       'magmoms': None,
       'positions': array([[  8.92565136e-17,   1.05892608e+00,   3.98743411e-01],
       [  8.83738227e-16,   1.02580139e+01,   4.17452659e+00],
       [  4.46072841e-16,   4.59954392e+00,   2.68537841e+00],
       [  5.26921900e-16,   6.71739608e+00,   1.88789159e+00],
       [  1.65233000e+00,   6.71739608e+00,   3.98743411e-01],
       [  1.65233000e+00,   4.59954392e+00,   4.17452659e+00],
       [  1.65233000e+00,   1.02580139e+01,   2.68537841e+00],
       [  1.65233000e+00,   1.05892608e+00,   1.88789159e+00]]),
       'symbols': ['P', 'P', 'P', 'P', 'P', 'P', 'P', 'P']},
 'Pb': {'cell': array([[  5.04386000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.08847350e-16,   5.04386000e+00,   0.00000000e+00],
       [  3.08847350e-16,   3.08847350e-16,   5.04386000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.08847350e-16,   2.52193000e+00,   2.52193000e+00],
       [  2.52193000e+00,   1.54423675e-16,   2.52193000e+00],
       [  2.52193000e+00,   2.52193000e+00,   0.00000000e+00]]),
        'symbols': ['Pb', 'Pb', 'Pb', 'Pb']},
 'Pd': {'cell': array([[  3.95315000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.42060625e-16,   3.95315000e+00,   0.00000000e+00],
       [  2.42060625e-16,   2.42060625e-16,   3.95315000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.42060625e-16,   1.97657500e+00,   1.97657500e+00],
       [  1.97657500e+00,   1.21030312e-16,   1.97657500e+00],
       [  1.97657500e+00,   1.97657500e+00,   0.00000000e+00]]),
        'symbols': ['Pd', 'Pd', 'Pd', 'Pd']},
 'Po': {'cell': array([[  3.34818000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.05016896e-16,   3.34818000e+00,   0.00000000e+00],
       [  2.05016896e-16,   2.05016896e-16,   3.34818000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.,  0.,  0.]]),
        'symbols': ['Po']},
 'Pt': {'cell': array([[  3.97675000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.43505708e-16,   3.97675000e+00,   0.00000000e+00],
       [  2.43505708e-16,   2.43505708e-16,   3.97675000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.43505708e-16,   1.98837500e+00,   1.98837500e+00],
       [  1.98837500e+00,   1.21752854e-16,   1.98837500e+00],
       [  1.98837500e+00,   1.98837500e+00,   0.00000000e+00]]),
        'symbols': ['Pt', 'Pt', 'Pt', 'Pt']},
 'Rb': {'cell': array([[  5.67197000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.47307995e-16,   5.67197000e+00,   0.00000000e+00],
       [  3.47307995e-16,   3.47307995e-16,   5.67197000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.      ,  0.      ,  0.      ],
       [ 2.835985,  2.835985,  2.835985]]),
        'symbols': ['Rb', 'Rb']},
 'Re': {'cell': array([[  2.77447000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.38723500e+00,   2.40276150e+00,   0.00000000e+00],
       [  2.74316597e-16,   1.58376761e-16,   4.47993000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.38722113,  0.80091249,  2.239965  ]]),
        'symbols': ['Re', 'Re']},
 'Rh': {'cell': array([[  3.84210000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.35260773e-16,   3.84210000e+00,   0.00000000e+00],
       [  2.35260773e-16,   2.35260773e-16,   3.84210000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.35260773e-16,   1.92105000e+00,   1.92105000e+00],
       [  1.92105000e+00,   1.17630387e-16,   1.92105000e+00],
       [  1.92105000e+00,   1.92105000e+00,   0.00000000e+00]]),
        'symbols': ['Rh', 'Rh', 'Rh', 'Rh']},
 'Rn': {'cell': array([[  7.19417000e+00,   0.00000000e+00,   0.00000000e+00],
       [  4.40515863e-16,   7.19417000e+00,   0.00000000e+00],
       [  4.40515863e-16,   4.40515863e-16,   7.19417000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  4.40515863e-16,   3.59708500e+00,   3.59708500e+00],
       [  3.59708500e+00,   2.20257932e-16,   3.59708500e+00],
       [  3.59708500e+00,   3.59708500e+00,   0.00000000e+00]]),
        'symbols': ['Rn', 'Rn', 'Rn', 'Rn']},
 'Ru': {'cell': array([[  2.72661000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.36330500e+00,   2.36131353e+00,   0.00000000e+00],
       [  2.63329678e-16,   1.52033460e-16,   4.30050000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.36329137,  0.78709664,  2.15025   ]]),
        'symbols': ['Ru', 'Ru']},
 'S': {'cell': array([[ 2.58153   ,  0.        ,  0.        ],
       [-0.06556819,  2.58069718,  0.        ],
       [-0.06556819, -0.06725525,  2.57982067]]),
       'magmoms': None,
       'positions': array([[ 0.,  0.,  0.]]),
       'symbols': ['S']},
 'Sb': {'cell': array([[  4.38047000e+00,   0.00000000e+00,   0.00000000e+00],
       [ -2.19023500e+00,   3.79359830e+00,   0.00000000e+00],
       [  7.02827247e-16,   1.21733250e-15,   1.14780400e+01]]),
        'magmoms': None,
        'positions': array([[  1.63983653e-16,   2.84028019e-16,   2.67805629e+00],
       [  5.38843594e-16,   9.33304482e-16,   8.79998371e+00],
       [  2.19025690e+00,   1.26452012e+00,   6.50414615e+00],
       [  2.19025690e+00,   1.26452012e+00,   1.14791878e+00],
       [ -2.19023500e-05,   2.52907818e+00,   1.03301212e+01],
       [ -2.19023500e-05,   2.52907818e+00,   4.97389385e+00]]),
        'symbols': ['Sb', 'Sb', 'Sb', 'Sb', 'Sb', 'Sb']},
 'Sc': {'cell': array([[  3.32145000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.66072500e+00,   2.87646008e+00,   0.00000000e+00],
       [  3.16076440e-16,   1.82486818e-16,   5.16192000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.66070839,  0.95881044,  2.58096   ]]),
        'symbols': ['Sc', 'Sc']},
 'Se': {'cell': array([[  4.52464000e+00,   0.00000000e+00,   0.00000000e+00],
       [ -2.26232000e+00,   3.91845318e+00,   0.00000000e+00],
       [  3.09209846e-16,   5.35567163e-16,   5.04978000e+00]]),
        'magmoms': None,
        'positions': array([[  9.91031899e-01,   1.78520602e-16,   1.68324317e+00],
       [ -4.95515950e-01,   8.58258801e-01,   3.36653683e+00],
       [  1.76680405e+00,   3.06019438e+00,   0.00000000e+00]]),
        'symbols': ['Se', 'Se', 'Se']},
 'Si': {'cell': array([[ 3.86709   ,  0.        ,  0.        ],
       [ 1.933545  ,  3.34899818,  0.        ],
       [ 1.933545  ,  1.11633273,  3.15746576]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.933545  ,  1.11633273,  0.78936644]]),
        'symbols': ['Si', 'Si']},
 'Sn': {'cell': array([[ 4.70647   ,  0.        ,  0.        ],
       [ 2.353235  ,  4.07592258,  0.        ],
       [ 2.353235  ,  1.35864086,  3.84281666]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 2.353235  ,  1.35864086,  0.96070417]]),
        'symbols': ['Sn', 'Sn']},
 'Sr': {'cell': array([[  6.01969000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.68599705e-16,   6.01969000e+00,   0.00000000e+00],
       [  3.68599705e-16,   3.68599705e-16,   6.01969000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  3.68599705e-16,   3.00984500e+00,   3.00984500e+00],
       [  3.00984500e+00,   1.84299852e-16,   3.00984500e+00],
       [  3.00984500e+00,   3.00984500e+00,   0.00000000e+00]]),
        'symbols': ['Sr', 'Sr', 'Sr', 'Sr']},
 'Ta': {'cell': array([[  3.32169000e+00,   0.00000000e+00,   0.00000000e+00],
       [  2.03394851e-16,   3.32169000e+00,   0.00000000e+00],
       [  2.03394851e-16,   2.03394851e-16,   3.32169000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.      ,  0.      ,  0.      ],
       [ 1.660845,  1.660845,  1.660845]]),
        'symbols': ['Ta', 'Ta']},
 'Tc': {'cell': array([[  2.76187000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.38093500e+00,   2.39184958e+00,   0.00000000e+00],
       [  2.70588772e-16,   1.56224500e-16,   4.41905000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.38092119,  0.79727522,  2.209525  ]]),
        'symbols': ['Tc', 'Tc']},
 'Te': {'cell': array([[  4.50788000e+00,   0.00000000e+00,   0.00000000e+00],
       [ -2.25394000e+00,   3.90393860e+00,   0.00000000e+00],
       [  3.65156610e-16,   6.32469801e-16,   5.96346000e+00]]),
        'magmoms': None,
        'positions': array([[  1.21559492e+00,   2.10821159e-16,   1.98780012e+00],
       [ -6.07797460e-01,   1.05273608e+00,   3.97565988e+00],
       [  1.64614254e+00,   2.85120252e+00,   0.00000000e+00]]),
        'symbols': ['Te', 'Te', 'Te']},
 'Ti': {'cell': array([[  2.93664000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.46832000e+00,   2.54320484e+00,   0.00000000e+00],
       [  2.84848559e-16,   1.64457392e-16,   4.65193000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.46830532,  0.84772647,  2.325965  ]]),
        'symbols': ['Ti', 'Ti']},
 'Tl': {'cell': array([[  3.59255000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.79627500e+00,   3.11123956e+00,   0.00000000e+00],
       [  3.45469800e-16,   1.99457082e-16,   5.64195000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.79625704,  1.03706948,  2.820975  ]]),
        'symbols': ['Tl', 'Tl']},
 'V': {'cell': array([[  2.99894000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.83632114e-16,   2.99894000e+00,   0.00000000e+00],
       [  1.83632114e-16,   1.83632114e-16,   2.99894000e+00]]),
       'magmoms': None,
       'positions': array([[ 0.     ,  0.     ,  0.     ],
       [ 1.49947,  1.49947,  1.49947]]),
       'symbols': ['V', 'V']},
 'W': {'cell': array([[  3.18968000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.95311570e-16,   3.18968000e+00,   0.00000000e+00],
       [  1.95311570e-16,   1.95311570e-16,   3.18968000e+00]]),
       'magmoms': None,
       'positions': array([[ 0.     ,  0.     ,  0.     ],
       [ 1.59484,  1.59484,  1.59484]]),
       'symbols': ['W', 'W']},
 'Xe': {'cell': array([[  7.05482000e+00,   0.00000000e+00,   0.00000000e+00],
       [  4.31983137e-16,   7.05482000e+00,   0.00000000e+00],
       [  4.31983137e-16,   4.31983137e-16,   7.05482000e+00]]),
        'magmoms': None,
        'positions': array([[  0.00000000e+00,   0.00000000e+00,   0.00000000e+00],
       [  4.31983137e-16,   3.52741000e+00,   3.52741000e+00],
       [  3.52741000e+00,   2.15991568e-16,   3.52741000e+00],
       [  3.52741000e+00,   3.52741000e+00,   0.00000000e+00]]),
        'symbols': ['Xe', 'Xe', 'Xe', 'Xe']},
 'Y': {'cell': array([[  3.66051000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.83025500e+00,   3.17009465e+00,   0.00000000e+00],
       [  3.47431685e-16,   2.00589777e-16,   5.67399000e+00]]),
       'magmoms': None,
       'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.8302367 ,  1.05668765,  2.836995  ]]),
       'symbols': ['Y', 'Y']},
 'Zn': {'cell': array([[  2.66169000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.33084500e+00,   2.30509116e+00,   0.00000000e+00],
       [  3.06404792e-16,   1.76902889e-16,   5.00397000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.33083169,  0.76835604,  2.501985  ]]),
        'symbols': ['Zn', 'Zn']},
 'Zr': {'cell': array([[  3.23603000e+00,   0.00000000e+00,   0.00000000e+00],
       [  1.61801500e+00,   2.80248419e+00,   0.00000000e+00],
       [  3.17767065e-16,   1.83462901e-16,   5.18953000e+00]]),
        'magmoms': None,
        'positions': array([[ 0.        ,  0.        ,  0.        ],
       [ 1.61799882,  0.93415205,  2.594765  ]]),
        'symbols': ['Zr', 'Zr']}}

    names = [r.split(',')[0] for r in wienref.split()][1:]
    ref = {}
    for name in names:
        for l in wienref.split():
            if l.split(',')[0] == name:
                ref[name] = [float(v) for v in l.split(',')[1:]]
    data = {}
    for name in names:
        d = atoms[name]
        a = Atoms(symbols=d['symbols'],
                  positions=d['positions'],
                  cell=d['cell'],
                  magmoms=d['magmoms'])
        a.set_pbc((1, 1, 1))
        data[name] = a

    def __getitem__(self, name):
        # must return a copy
        return self.data[name].copy()

    def keys(self):
        return self.names

class DeltaCodesDFTTask(BulkTask):
    def __init__(self, **kwargs):
        BulkTask.__init__(self,
                          collection=DeltaCodesDFTCollection(),
                          **kwargs)

        self.summary_keys = ['energy', 'fitted energy', 'volume',
                             'volume error [%]', 'B', 'B error [%]']

    def analyse(self, atomsfile=None):
        try:
            BulkTask.analyse(self)
        except ValueError: # allow fit outside of range
            pass

        for name, data in self.data.items():
            if 'strains' in data:
                atoms = self.create_system(name)
                # full equation of state
                # use relaxed volume if present
                if 'relaxed volume' in data:
                    volume = data['relaxed volume']
                else:
                    volume = atoms.get_volume()
                volumes = data['strains']**3 * volume
                energies = data['energies']
                eos = FullEquationOfState(volumes, energies)
                try:
                    v, e, B0, B1, R = eos.fit()
                except ValueError:
                    pass
                else:
                    data['dcdft fitted energy'] = e / len(atoms)
                    data['dcdft volume'] = v / len(atoms)
                    data['dcdft B0'] = B0
                    data['dcdft B1'] = B1
                    data['dcdft R'] = R

        if atomsfile:
            # MDTMP: TODO
            atomdata = read_json(atomsfile)
            for name, data in self.data.items():
                atoms = self.create_system(name)
                e = -data['energy']
                for atom in atoms:
                    e += atomdata[atom.symbol]['energy']
                e /= len(atoms)
                data['cohesive energy'] = e
                if self.collection.xc == 'PBE':
                    eref = self.collection.data[name][7]
                else:
                    eref = self.collection.data[name][9]
                data['cohesive energy error [%]'] = (e / eref - 1) * 100

            self.summary_keys += ['cohesive energy',
                                  'cohesive energy error [%]']

if __name__ == '__main__':
    # download and create the project databases
    src = 'https://molmod.ugent.be/sites/default/files/Delta_v3-0.zip'
    name = os.path.basename(src)
    dir = 'Delta'
    if not os.path.exists(dir): os.makedirs(dir)
    os.chdir(dir)
    try:
        resp = urllib.request.urlopen(src)
        urllib.urlretrieve(src, filename=name)
        z = zipfile.ZipFile(name)
        try:  # new in 2.6
            z.extractall()
        except AttributeError:
            # http://stackoverflow.com/questions/7806563/how-to-unzip-a-zip-file-with-python-2-4
            for f in z.namelist():
                fd = open(f, "w")
                fd.write(z.read(f))
                fd.close()
        t = tarfile.open('CIFs.tar.gz', 'r:gz')
        t.extractall()
        # extract structures
        data = {}
        for f in glob.glob('*.cif'):
            a = ase.io.read(f)
            s = os.path.splitext(f)[0]
            # Fe, Co, Ni
            M = {'Fe': 2.3, 'Co': 1.2, 'Ni': 0.6}.get(s)
            if M is not None:
                magmoms = [M] * len(a)
            else:
                magmoms = None
            # antiferromagnetic Cr, O
            if s in ['Cr', 'O']:
                magmoms = [1.5 for i in range(len(a) / 2)]
                magmoms += [-1.5 for i in range(len(a) / 2)]
            # ferrimagnetic Mn
            elif s in ['Mn']:
                magmoms = [2.0, 1.9, -2.0, -1.9]
            d = {'symbols': a.get_chemical_symbols(),
                 'positions': a.get_positions(),
                 'cell': a.get_cell(),
                 'magmoms': magmoms}
            data[s] = d
        pprint.pprint(data)
    # AttributeError if unzip not found
    except (urllib2.HTTPError, AttributeError):
        pass
    os.chdir('..')
    # run with emt
    from ase.calculators.emt import parameters
    from ase.tasks.main import run
    task = DeltaCodesDFTTask(calcfactory='emt', fit=(13, 0.06))
    task.run(parameters.keys())
