///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

/**
 * \file StandardSplineControllers.h
 * \brief Contains the definition of the spline interpolation controller classes.
 */

#ifndef __OVITO_STD_SPLINE_CONTROLLERS_H
#define __OVITO_STD_SPLINE_CONTROLLERS_H

#include <core/Core.h>
#include "StandardKeyedControllers.h"

namespace Core {

template<typename ValueType, typename TangentType, typename NullValue>
class SplineControllerKey
{
public:

	/// Constructs a key with the default value.
	SplineControllerKey() : _value(NullValue()), _inSpeed(1), _outSpeed(1), _inTangent(NullValue()), _outTangent(NullValue()) {}

	/// Constructs a key from a given value.
	SplineControllerKey(const ValueType& value) : _value(value), _inSpeed(1), _outSpeed(1),
	_inTangent(NullValue()), _outTangent(NullValue()) {}

	/// Cast operator that converts the key to the stored value.
	operator const ValueType&() const { return _value; }

	/// Sets the internal key value.
	SplineControllerKey& operator=(const ValueType& v) { _value =  v; return *this; }

	/// Adds another value to the key's internal value.
	SplineControllerKey& operator+=(const ValueType& v) { _value +=  v; return *this; }

	/// Saves the key information to an output stream.
	void saveToStream(SaveStream& stream) const {
		stream << _value;
		stream << _inSpeed;
		stream << _outSpeed;
		stream << _inTangent;
		stream << _outTangent;
	}

	/// Loads the key information from an input stream.
	void loadFromStream(LoadStream& stream) {
		stream >> _value;
		stream >> _inSpeed;
		stream >> _outSpeed;
		stream >> _inTangent;
		stream >> _outTangent;
	}

	/// \brief Returns the incoming derivative of animation time.
	FloatType inSpeed() const { return _inSpeed; }
	/// \brief Returns the outgoing derivative of animation time.
	FloatType outSpeed() const { return _outSpeed; }

	/// \brief Returns the tangent that defines the incoming direction.
	const TangentType& inTangent() const { return _inTangent; }
	/// \brief Returns the tangent that defines the outgoing direction.
	const TangentType& outTangent() const { return _outTangent; }

	/// \brief Sets the tangent that defines the incoming direction.
	void setInTangent(const TangentType& t) { _inTangent = t; }
	/// \brief Sets the tangent that defines the outgoing direction.
	void setOutTangent(const TangentType& t) { _outTangent = t; }

	/// \brief Returns the point that defines the incoming direction.
	ValueType inPoint() const { return _value + _inTangent; }
	/// \brief Returns the point that defines the outgoing direction.
	ValueType outPoint() const { return _value + _outTangent; }

private:

	/// The stored value of the key.
	ValueType _value;

	/// The incoming speed factor.
	FloatType _inSpeed;
	/// The outgoing speed factor.
	FloatType _outSpeed;

	/// The tangent that defines the incoming derivative.
	TangentType _inTangent;
	/// The tangent that defines the outgoing derivative.
	TangentType _outTangent;
};

/// \brief Writes a spline controller key to a binary output stream.
template<typename ValueType, typename TangentType, typename NullValue>
inline SaveStream& operator<<(SaveStream& stream, const SplineControllerKey<ValueType, TangentType, NullValue>& k) { k.saveToStream(stream); return stream; }

/// \brief Reads a spline controller key from a binary input stream.
template<typename ValueType, typename TangentType, typename NullValue>
inline LoadStream& operator>>(LoadStream& stream, SplineControllerKey<ValueType, TangentType, NullValue>& k) { k.loadFromStream(stream); return stream; }

/**
 * \brief Implementation of the key interpolator concept that performs smooth spline interpolation.
 *
 * This class calculates a value in the [0,1] range based on the input time and the times of the two input keys.
 * The returned value specifies the weighting of the two key values.
 *
 * This class is used with the spline interpolation controllers.
 *
 * \author Alexander Stukowski
 */
template<typename ValueType, typename KeyType>
struct SplineKeyInterpolator {
	ValueType operator()(const TimeTicks& time,
						 const pair<TimeTicks, KeyType>& key1,
						 const pair<TimeTicks, KeyType>& key2) const {
		OVITO_ASSERT(key2.first > key1.first);
		FloatType t = (FloatType)(time - key1.first) / (key2.first - key1.first);
		SplineValueInterpolator<ValueType> valueInterpolator;
		return valueInterpolator(t, key1.second, key2.second, key1.second.outPoint(), key2.second.inPoint());
	}
};

/******************************** Spline controller base *******************************/

template<class BaseClass>
class SplineControllerBase : public BaseClass
{
public:
	SplineControllerBase(bool isLoading = false) : BaseClass(isLoading) {}

protected:

	/// The list of controller keys.
	using BaseClass::keys;
	typedef typename BaseClass::ValueType ValueType;

	/// This is called each time the value or time positions of one or more keys has been changed.
	virtual void updateKeys() {
		if(keys.size() < 2) return;

		// Update tangent of first key.
		typename BaseClass::KeyArray::iterator key1 = keys.begin();
		typename BaseClass::KeyArray::iterator key2 = key1;
		++key2;
		typename BaseClass::KeyArray::iterator key3 = key2;
		++key3;

		// Update the tangent vector of the first key.
		key1->second.setOutTangent(((ValueType)key2->second - (ValueType)key1->second) / 3.0);

		// Update the tangent vectors for inner keys.
		while(key3 != keys.end()) {
			ValueType tangentL = ((ValueType)key2->second - (ValueType)key1->second);
			ValueType tangentR = ((ValueType)key3->second - (ValueType)key2->second);
			ValueType avgTangent = ((ValueType)key3->second - (ValueType)key1->second);
			key2->second.setOutTangent(avgTangent * (Length(tangentR) / Length(avgTangent) / 6.0));
			key2->second.setInTangent(-avgTangent * (Length(tangentL) / Length(avgTangent) / 6.0));
			key1 = key2;
			key2 = key3;
			++key3;
		}

		// Update the tangent vector of the last key.
		key2->second.setInTangent(((ValueType)key1->second - (ValueType)key2->second) / 3.0);
	}
};

/******************************** Position controller *************************************/

/// The key type used by the spline position controller.
typedef SplineControllerKey<Vector3,		// Value type
							Vector3,		// Tangent type
							Vector3			// Null value
> SplinePositionControllerKey;

/// The base class of the spline position controller.
typedef SplineControllerBase< KeyedPositionController<
								SplinePositionControllerKey,		// Key type
								SplineKeyInterpolator<Vector3, SplinePositionControllerKey>	// Key interpolator
> > SplinePositionControllerBase;

/**
 * \brief A keyed controller that interpolates between position values using
 *        a spline interpolation scheme.
 *
 * \author Alexander Stukowski
 */
class CORE_DLLEXPORT SplinePositionController : public SplinePositionControllerBase {
public:
	SplinePositionController(bool isLoading = false) : SplinePositionControllerBase(isLoading) {}
private:
	Q_OBJECT
	DECLARE_SERIALIZABLE_PLUGIN_CLASS(SplinePositionController)
};

};

#endif // __OVITO_STD_SPLINE_CONTROLLERS_H
