///////////////////////////////////////////////////////////////////////////////
// 
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <core/plugins/branding/Branding.h>
#include <core/gui/properties/PropertiesPanel.h>

#include <QNetworkAccessManager>
#include <QNetworkRequest>
#include <QNetworkReply>

namespace Core {

/******************************************************************************
* Lets the branding display the about panel in the command panel of the main window.
* This is called by the system when no scene node is currently selected.
******************************************************************************/
void Branding::showAboutPanel(PropertiesPanel* container)
{
	if(_aboutRollout == NULL) {
		QWidget* rollout = new QWidget();
		QVBoxLayout* layout = new QVBoxLayout(rollout);
		layout->setContentsMargins(8,8,8,8);

		_aboutLabel = new QTextBrowser();
		_aboutLabel->setOpenExternalLinks(true);
		_aboutLabel->setMinimumSize(0, 600);
		_aboutLabel->setFrameStyle(QFrame::NoFrame | QFrame::Plain);
		_aboutLabel->viewport()->setAutoFillBackground(false);
		layout->addWidget(_aboutLabel);

		// Retrieve cached news page from settings store.
		QSettings settings;
		settings.beginGroup("news");
		QByteArray newsPage = settings.value("cached_webpage").toByteArray();
		if(newsPage.isEmpty()) {
			QResource res("/core/other/about_panel.html");
			newsPage = QByteArray((const char *)res.data(), (int)res.size());
		}
		_aboutLabel->setHtml(QString::fromUtf8(newsPage.constData()));
		settings.endGroup();

		_aboutRollout = container->addRollout(rollout, productName());

		// Retrieve/generate unique installation id.
		settings.beginGroup("installation");
		QByteArray id;
		if(settings.contains("id")) {
			id = settings.value("id").toByteArray();
		}
		else {
			id = QByteArray(16, '0');
			qsrand(time(NULL));
			for(int i = 0; i < id.size(); i++)
				id[i] = qrand() * 0xFF / RAND_MAX;
			settings.setValue("id", id);
		}

		// Fetch newest web page from web server.
		QNetworkAccessManager* networkAccessManager = new QNetworkAccessManager(_aboutRollout);
		QString urlString = QString("http://www.ovito.org/appnews/v%1.%2.%3/?ovito=%4")
				.arg(OVITO_PROJ_VERSION_MAJOR)
				.arg(OVITO_PROJ_VERSION_MINOR)
				.arg(OVITO_PROJ_VERSION_REVISION)
				.arg(QString(id.toHex()));
		QNetworkReply* networkReply = networkAccessManager->get(QNetworkRequest(QUrl(urlString)));
		connect(networkAccessManager, SIGNAL(finished(QNetworkReply*)), this, SLOT(onWebRequestFinished(QNetworkReply*)));
	}

	_aboutRollout->show();
}

/******************************************************************************
* Lets the branding hide its about panel.
******************************************************************************/
void Branding::hideAboutPanel(PropertiesPanel* container)
{
	if(_aboutRollout != NULL)
		_aboutRollout->hide();
}

/******************************************************************************
* Is called by the system when fetching the news webpage from the webserver is
* finished.
******************************************************************************/
void Branding::onWebRequestFinished(QNetworkReply* reply)
{
	if(reply->error() == QNetworkReply::NoError) {
		QByteArray page = reply->readAll();
		reply->close();
		if(page.startsWith("<html><!--OVITO-->")) {
			_aboutLabel->setHtml(QString::fromUtf8(page.constData()));

			QSettings settings;
			settings.beginGroup("news");
			settings.setValue("cached_webpage", page);

			VerboseLogger() << logdate << "Successfully fetched news page from webserver." << endl;
		}
		else {
			VerboseLogger() << logdate << "News page fetched from webserver is invalid." << endl;
		}
	}
	else {
		VerboseLogger() << logdate << "Failed to fetch news page from webserver: " << reply->errorString() << endl;
	}
	reply->deleteLater();
}

};
