///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#include <core/Core.h>
#include <atomviz/atoms/AtomsObject.h>
#include "XYZParserSettingsDialog.h"

namespace AtomViz {

/******************************************************************************
* Constructor.
******************************************************************************/
XYZParserSettingsDialog::XYZParserSettingsDialog(XYZParser* parser, QWidget* parent)
	: QDialog(parent)
{
	setWindowTitle(tr("XYZ File Import Settings"));
	this->parser = parser;

	QVBoxLayout* layout1 = new QVBoxLayout(this);

	// Time steps group
	QGroupBox* timeStepGroupBox = new QGroupBox(tr("Time Steps"), this);
	layout1->addWidget(timeStepGroupBox);

	QGridLayout* layout2 = new QGridLayout(timeStepGroupBox);
	layout2->setColumnStretch(0, 5);
	layout2->setColumnStretch(1, 95);
	timeStepButtonGroup = new QButtonGroup(this);

	QRadioButton* radioBtn = new QRadioButton(tr("Load file containing a single snapshot"));
	timeStepButtonGroup->addButton(radioBtn, 0);
	layout2->addWidget(radioBtn, 0, 0, 1, 2);
	radioBtn->setChecked(!parser->movieFileEnabled() && !parser->useWildcardFilename());

	radioBtn = new QRadioButton(tr("Load file containing a sequence of snapshots"));
	timeStepButtonGroup->addButton(radioBtn, 1);
	layout2->addWidget(radioBtn, 1, 0, 1, 2);
	radioBtn->setChecked(parser->movieFileEnabled() && !parser->useWildcardFilename());

	radioBtn = new QRadioButton(tr("Use following wild-card name to load multiple files:"));
	timeStepButtonGroup->addButton(radioBtn, 2);
	layout2->addWidget(radioBtn, 2, 0, 1, 2);
	radioBtn->setChecked(parser->useWildcardFilename());

	wildcardTextbox = new QLineEdit(parser->wildcardFilename(), timeStepGroupBox);
	layout2->addWidget(wildcardTextbox, 3, 1, 1, 1);
	wildcardTextbox->setEnabled(radioBtn->isChecked());
	connect(radioBtn, SIGNAL(toggled(bool)), wildcardTextbox, SLOT(setEnabled(bool)));

	// Column to data channel mapping
	QGroupBox* mappingGroupBox = new QGroupBox(tr("Data Columns"), this);
	layout1->addWidget(mappingGroupBox);

	QVBoxLayout* layout3 = new QVBoxLayout(mappingGroupBox);
	mappingEditor = new ColumnChannelMappingEditor(mappingGroupBox);
	mappingEditor->setMapping(parser->columnMapping());
	layout3->addWidget(mappingEditor);

	// Ok and cancel buttons
	QDialogButtonBox* buttonBox = new QDialogButtonBox(QDialogButtonBox::Ok | QDialogButtonBox::Cancel, Qt::Horizontal, this);
	connect(buttonBox, SIGNAL(accepted()), this, SLOT(onOk()));
	connect(buttonBox, SIGNAL(rejected()), this, SLOT(reject()));
	layout1->addWidget(buttonBox);
}

/******************************************************************************
* This is called when the user has pressed the OK button.
******************************************************************************/
void XYZParserSettingsDialog::onOk()
{
	try {
		// Write settings back to the parser.
		parser->setColumnMapping(mappingEditor->mapping());
		parser->setMovieFileEnabled(timeStepButtonGroup->checkedId() == 1);
		parser->setUseWildcardFilename(timeStepButtonGroup->checkedId() == 2);
		parser->setWildcardFilename(wildcardTextbox->text());

		// Close dialog box.
		accept();
	}
	catch(const Exception& ex) {
		ex.showError();
		return;
	}
}

};	// End of namespace AtomViz
