///////////////////////////////////////////////////////////////////////////////
//
//  Copyright (2008) Alexander Stukowski
//
//  This file is part of OVITO (Open Visualization Tool).
//
//  OVITO is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 2 of the License, or
//  (at your option) any later version.
//
//  OVITO is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program.  If not, see <http://www.gnu.org/licenses/>.
//
///////////////////////////////////////////////////////////////////////////////

#ifndef __ATOMS_RENDERER_H
#define __ATOMS_RENDERER_H

#include <atomviz/AtomViz.h>

#include <core/Core.h>
#include <core/viewport/Window3D.h>
#include <core/viewport/OpenGLShader.h>

namespace AtomViz {

/**
 * \brief This helper class is used to render the atoms stored in an AtomsObject
 *        via OpenGL.
 * It uses the best/fastest rendering method available on the installed
 * graphics hardware.
 * This class manages an internal buffer, so the atoms data can be specified
 * once and then rendered many times.
 *
 * \author Alexander Stukowski
 */
class ATOMVIZ_DLLEXPORT AtomsRenderer
{
public:

	/// The OpenGL rendering methods for atoms implemented by this class.
	enum RenderingMethod {
		QUAD_GEOMETRY_IMPOSTERS,			// Fallback method that is compatible with any graphics card. Uses real quad facets with static billboard texture maps.
		POINT_SPRITE_IMPOSTERS,				// Uses the point sprite extension to generate static texture mapped quad facets.
		SHADED_POINT_SPRITE_IMPOSTERS,		// Optimal real-time rendering method. Uses point sprites in conjunction with hardware shader mapping.
		SHADED_RAYTRACED_SPHERES,			// High quality rendering method. For each atom, sends box geometry to the graphics card and uses hardware shader to raytrace a sphere.

		DEFAULT_RENDERING_METHOD,			// The user-defined default rendering method.
		DEFAULT_HQ_RENDERING_METHOD,		// The user-defined default high-quality rendering method.
	};

public:

	/// Default constructor.
	AtomsRenderer();

	/// Destructor.
	~AtomsRenderer();

	/// Returns true if the atoms renderer has been initialized.
	/// Other wise Prepare has to be called.
	bool isInitialized() const { return _window != NULL; }

	/// Initializes the internal buffers of this helper class if they have not been initialized yet.
	/// It will be associated with the given rendering window.
	/// Returns the actual rendering method that will be used.
	RenderingMethod prepare(Window3D* win, bool flatAtoms = false, RenderingMethod renderingMethod = DEFAULT_RENDERING_METHOD);

	/// Return true if the  internal rendering buffer is filled. Otherwise
	/// The buffers have to be filled using beginAtoms()/specifyAtom()/endAtoms() before
	/// they can be rendered using render().
	bool isFilled() const { return hasBeenFilled; }

	/// Allocates a memory buffer for the specified number of atoms.
	/// After all atoms have been specified, endAtoms() must be called.
	void beginAtoms(GLuint numAtoms);

	/// Stores a single atom in the internal buffer. This function may only be called
	/// N times between a call to beginAtoms() and endAtoms();
	void specifyAtom(const Point3& pos, GLubyte r, GLubyte g, GLubyte b, FloatType radius);

	/// This method must be called after beginAtoms() has been called and all atom
	/// data has been written to the supplied buffer.
	void endAtoms();

	/// Renders the buffered atoms in the given rendering window.
	void render(Window3D* win);

	/// Render the buffered atoms to an offscreen buffer.
	/// This function is used by the ambient lighting modifier.
	void renderOffscreen(bool isPerspective, const Matrix4& projMatrix, QSize windowSize);

	/// Returns the local bounding box of all atoms stored in the buffer.
	const Box3& boundingBox() const { return _boundingBox; }

	/// Calculates the lighting for the atoms using an ambient occlusion algorithm.
	bool calculateAmbientOcclusion(Window3D* glcontext);

	/// Release all cached OpenGL resources.
	void reset();

private:

	/// This structure holds all properties of a single atom
	/// that are relevant for OpenGL rendering.
	struct OpenGLAtom {
		float x, y, z;		// The position of the atom.
		GLubyte r, g, b, a;	// The color of the atom.
		float radius;		// The radius of the atom.
	};

	/// The rendering window this class is associated with.
	QPointer<Window3D> _window;

	/// This array is used if no vertex buffer objects are available.
	QVector<OpenGLAtom> internalArray;

	/// The rendering method requested by the caller.
	RenderingMethod _requestedRenderingMethod;

	/// The rendering method currently in use.
	RenderingMethod _renderingMethod;

	/// Enables or disables the shading of atoms.
	bool _flatAtoms;

	/// Indicates whether atomic coordinates have transfered.
	bool hasBeenFilled;

	/// The number of atoms in the buffer.
	GLuint numAtoms;

	/// A pointer to the atom in the internal array that will be set
	/// on the next call to specifyAtom().
	OpenGLAtom* currentAtom;

	/// If all atoms in the buffer have the same radius then this field contains
	/// the uniform rdaius. If there are at least two atoms with differnt radii then
	/// this field is zero.
	FloatType uniformRadius;

	/// The radius of the biggest atom.
	FloatType maxRadius;

	/// The bounding box that has been computed for the atoms.
	Box3 _boundingBox;

	/// The identifiers of OpenGL textures that are used for billboard rendering of shaded and flat atoms.
	GLuint textureID[2];

	/// The identifier of the OpenGL vertex buffer object.
	GLuint vboVerticesID;

	/// The OpenGL shader programs that are used to render the atoms.
	OpenGLShader* flatImposterShader;
	OpenGLShader* shadedImposterShader;
	OpenGLShader* raytracedSphereShader;

	/// Specifies the number of atoms per chunk if the whole atoms array
	/// is rendered not at once but chunk by chunk. Some graphics cards
	/// can only render a limited amount of point sprites at once so the
	/// array has to be split up into chunks.
	GLuint chunkRenderSize;

	/// Renders the atoms stored in the internal atoms array
	/// using the GL_ARB_point_parameters OpenGL extension.
	void renderInternalArrayPointSprites(bool isPerspective, const Matrix4& projMatrix, float windowHeight);

	/// Renders the atoms stored in the internal atoms array
	/// using the custom vertex and fragment shaders.
	void renderInternalImpostersWithShader(bool isPerspective, const Matrix4& projMatrix, float windowHeight);

	/// Renders the atoms stored in the internal atoms array
	/// using the custom vertex and fragment shaders.
	void renderInternalRaytracedWithShaders(bool isPerspective);

	/// Renders the atoms stored in the internal atoms array
	/// without the GL_ARB_point_parameters OpenGL extension.
	void renderInternalArrayQuads();

	/// Creates and activates the texture used for billboard rendering of atoms.
	void initializeBillboardTextures(bool withAlpha = true);

public:			// Global settings

	/// Returns the default rendering method for atoms set by the user.
	static RenderingMethod defaultRenderingMethod();

	/// Returns the default high-quality rendering method for atoms set by the user.
	static RenderingMethod defaultHQRenderingMethod();

	/// Changes the default rendering method for atoms.
	static void setDefaultRenderingMethod(RenderingMethod method);

	/// Changes the default high-quality rendering method for atoms.
	static void setDefaultHQRenderingMethod(RenderingMethod method);

private:

	static RenderingMethod _defaultRenderingMethod;
	static RenderingMethod _defaultHQRenderingMethod;

	/// \brief Indicates that the render settings have been loaded from the application settings store.
	static bool _settingsLoaded;

	/// \brief Loads the render settings from the application settings store.
	static void loadRenderSettings();
};

};	// End of namespace AtomViz

#endif // __ATOMS_RENDERER_H
