/*

Copyright 2010, Google Inc.
All rights reserved.

Redistribution and use in source and binary forms, with or without
modification, are permitted provided that the following conditions are
met:

    * Redistributions of source code must retain the above copyright
notice, this list of conditions and the following disclaimer.
    * Redistributions in binary form must reproduce the above
copyright notice, this list of conditions and the following disclaimer
in the documentation and/or other materials provided with the
distribution.
    * Neither the name of Google Inc. nor the names of its
contributors may be used to endorse or promote products derived from
this software without specific prior written permission.

THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS
"AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT
LIMITED TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
OWNER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT
LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,           
DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY           
THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
(INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE
OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

*/

package com.google.refine.expr.functions.math;

import java.util.List;
import java.util.Properties;

import com.google.refine.expr.EvalError;
import com.google.refine.expr.ExpressionUtils;
import com.google.refine.grel.ControlFunctionRegistry;
import com.google.refine.grel.Function;

public class Sum implements Function {

    @Override
    public Object call(Properties bindings, Object[] args) {
        if (args.length == 1) {
            Object v = args[0];
            
            if (v != null && (v.getClass().isArray() || v instanceof List<?>)) {
                int length = v.getClass().isArray() ? 
                        ((Object[]) v).length :
                        ExpressionUtils.toObjectList(v).size();
                
                double total = 0;
                
                if (v.getClass().isArray()) {
                    Object[] a = (Object[]) v;
                    for (int i = 0; i < length; i++) {
                        Object n = a[length - i - 1];
                        if (n instanceof Number) {
                            total += ((Number) n).doubleValue();
                        }
                    }
                } else {
                    List<Object> a = ExpressionUtils.toObjectList(v);
                    for (int i = 0; i < length; i++) {
                        Object n = a.get(length - i - 1);
                        if (n instanceof Number) {
                            total += ((Number) n).doubleValue();
                        }
                    }
                }
                return total;
            }
        }
        return new EvalError(ControlFunctionRegistry.getFunctionName(this) + " expects an array of numbers");
    }

    @Override
    public String getDescription() {
        return "Return the sum of the numbers in the array. Ignores non-number items. Returns 0 if the array does not contain numbers.";
    }
    
    @Override
    public String getParams() {
        return "array a";
    }
    
    @Override
    public String getReturns() {
        return "number";
    }
}
