/*
 * This file is part of the Ubuntu TV Media Scanner
 * Copyright (C) 2012-2013 Canonical Ltd.
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU Lesser General Public License version 3 as
 * published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 *
 * Contact: Jim Hodapp <jim.hodapp@canonical.com>
 * Authored by: Mathias Hasselmann <mathias@openismus.com>
 */
#ifndef MEDIASCANNER_WRITABLEMEDIAINDEX_H
#define MEDIASCANNER_WRITABLEMEDIAINDEX_H

// C++ Standard Library
#include <string>

// Media Scanner Library
#include "mediascanner/declarations.h"
#include "mediascanner/mediaindex.h"

namespace mediascanner {

/**
 * @brief A writable instance of the media index.
 *
 * Use set_commit_policy() to control when changes 
 * are written back to the disk.
 *
 * @see MediaIndex for a read-only instance.
 */
class WritableMediaIndex : public MediaIndex {
    friend class CommitPolicy;

    class Private;

public:
    explicit WritableMediaIndex(MediaRootManagerPtr root_manager);
    ~WritableMediaIndex();

    /**
     * @brief Changes the commit_policy() of this media index.
     */
    void set_commit_policy(CommitPolicyPtr policy);

    /**
     * @brief The commit policy of this media index.
     */
    CommitPolicyPtr commit_policy() const;

    /**
     * @brief Changes the time to wait for a write lock in milliseconds.
     */
    void set_write_lock_timeout(int64_t timeout);

    /**
     * @brief The time to wait for a write lock in milliseconds.
     */
    int64_t write_lock_timeout() const;

    bool Open(const FileSystemPath &path);
    void Close();

    /**
     * @brief Inserts new properties for the media referenced by @p url.
     * @param url The URL of the media to modify.
     * @param metadata The new property values to attach.
     */
    bool Insert(const std::wstring &url, const MediaInfo &metadata);

    /**
     * @brief Removes a media information from the index.
     * @param url The URL of the media to remove.
     */
    bool Delete(const std::wstring &url);

    /**
     * @brief Commits all pending changes so that other readers will see the
     * changes, and so that the updates will survive a system crash or power
     * loss.
     *
     * This call can be very expensive, therefore we should only call it if
     * when really necessary. It's usually more efficient to just rely on the
     * configured commit_policy().
     */
    void CommitPendingChanges();

protected:
    bool ReadParams();
    bool FlushParams(Wrapper<GKeyFile> params);
    using MediaIndex::FlushParams;

    Lucene::IndexReaderPtr OpenIndex();
    Lucene::IndexReaderPtr OpenChildIndex(const MediaRoot &root);

    Lucene::IndexWriterPtr find_index_writer(const std::wstring &url);

private:
    bool ReadParamsUnlocked();

    // TODO(M5): figure out how to allocate in one go with parent class pimpl
    Private *const d;
};

} // namespace mediascanner

#endif // MEDIASCANNER_WRITABLEMEDIAINDEX_H
