// license:BSD-3-Clause
// copyright-holders:Miodrag Milanovic
/********************************************************************************************************************
PINBALL
Stern SAM system.

An evolution of the Whitestar system, it has now been superseded by SPIKE.

Major components: AT91R40008, 2x XC95144XL, AT43USB380, PCM1755, OPA2353, serial interface, USB interface.

Game                                   NUM  Start game                   End ball
---------------------------------------------------------------------------------------------
World Poker Tour                      5088
Simpsons Kooky Carnival (Redemption)  5090
Pirates of the Caribbean              5092
Family Guy                            5093
Spider-Man                            5094
Wheel of Fortune                      5098
Spider-Man                              A0 (this game has 2 numbers)
Good Morning America (unemulated custom adaption of Family Guy)
CSI                                     A2
Batman                                  A3
Indiana Jones                           A4
Shrek                                   A5
Black Spider-Man                        A6
24                                      A7
NBA                                     A9
James Cameron's Avatar                  B1
Big Buck Hunter Pro                     B2
Iron Man                                B3
The Rolling Stones                      B8
Disney Tron Legacy                      B9
Transformers                            C4
AC/DC (Pro)                             C0
AC/DC (Premium)                         C7
AC/DC (Luci)                            C8
X-Men                                   D2
The Avengers (Pro)                      D3
The Avengers (LE/Premium)               D4
Metallica                               E3
Star Trek (Pro)                         E4
Star Trek (LE/Premium)                  E6
Mustang (Pro)                           F1
Mustang (LE/Premium)                    F2
The Walking Dead (LE)                   F6
The Walking Dead (Pro)                  F8

Status:
- Skeletons

ToDo:
- Everything

*********************************************************************************************************************/
#include "emu.h"
#include "cpu/arm7/arm7.h"
#include "cpu/arm7/arm7core.h"

namespace {

class sam_state : public driver_device
{
public:
	sam_state(const machine_config &mconfig, device_type type, const char *tag)
		: driver_device(mconfig, type, tag) { }

	void sam(machine_config &config);

private:
	void sam_map(address_map &map);
};

void sam_state::sam_map(address_map &map)
{
	map(0x00000000, 0x000fffff).rom().region("boot", 0);
	map(0x00300000, 0x003fffff).ram();
}

static INPUT_PORTS_START( sam )
INPUT_PORTS_END

void sam_state::sam(machine_config &config)
{
	arm7_cpu_device &maincpu(ARM7(config, "maincpu", XTAL(40'000'000))); // AT91R40008
	maincpu.set_addrmap(AS_PROGRAM, &sam_state::sam_map);
}

#define ROM_LOAD_BOOT_EPROM \
	ROM_REGION32_LE( 0x100000, "boot", ROMREGION_ERASEFF ) \
	ROM_LOAD( "boot_eprom_01_02.bin", 0x0000, 0x100000, CRC(92c93cba) SHA1(aed7ba2f988df8c95e2ad08f70409152d5caa49a)) \
	ROM_LOAD( "boot_eprom_01_06.bin", 0x0000, 0x0e8ac8, CRC(fe7bcece) SHA1(775590bbd52c24950db86cc231566ba3780030d8)) \
	ROM_LOAD( "boot_eprom_02_10.bin", 0x0000, 0x0f0304, CRC(0f3fd4a4) SHA1(115d0b73c40fcdb2d202a0a9065472d216ca89e0)) \
	ROM_LOAD( "boot_eprom_02_30.bin", 0x0000, 0x0f0624, CRC(a4258c49) SHA1(d865edf7d1c6d2c922980dd192222dc24bc092a0)) \
	ROM_LOAD( "boot_eprom_03_10.bin", 0x0000, 0x100000, CRC(de017f82) SHA1(e4a9a818fa3f1754374cd00b52b8a087d6c442a9))
/*-------------------------------------------------------------------
/ 24
/-------------------------------------------------------------------*/
ROM_START(24_130)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "24_v1-3_a.bin",   0x0000, 0x1c08280, CRC(955a5c12) SHA1(66e33fb438c831679aeb3ba68af7b4a3c59966ef))
ROM_END
ROM_START(24_140)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "24_v1-4_a.bin",   0x0000, 0x1c08280, CRC(bab92fb1) SHA1(07c8d9c28730411dd0f23d5960a223beb4c587b2))
ROM_END
ROM_START(24_144)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "24_v1-44_e.bin",  0x0000, 0x1ca8e50, CRC(29c47da0) SHA1(8d38e35a0df843a71cac6cd4dd6aa460347a208c))
ROM_END
ROM_START(24_150)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "24_v1-5_e.bin",   0x0000, 0x1ca8e50, CRC(9d7d87cc) SHA1(df6b2f60b87226fdda33bdbbe03ea87d690fc563))
ROM_END
/*-------------------------------------------------------------------
/ Batman - The Dark Knight
/-------------------------------------------------------------------*/
ROM_START(bdk_130)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "bat_v1-3_e.bin",  0x0000, 0x1ba1e94, CRC(83a32958) SHA1(0326891bc142c8b92bd4f6d29bd4301bacbed0e7))
ROM_END
ROM_START(bdk_150)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "bat_v1-5_e.bin",  0x0000, 0x18ee5e8, CRC(ed11b88c) SHA1(534224de597cbd3632b902397d945ab725e24912))
ROM_END
ROM_START(bdk_160)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "bat_v1-6_e.bin",  0x0000, 0x1b02f70, CRC(5554ea47) SHA1(0ece4779ad9a3d6c8428306774e2bf36a20d680d))
ROM_END
ROM_START(bdk_200)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "bat_v2-0_e.bin",  0x0000, 0x1b04378, CRC(07b716a9) SHA1(4cde06308bb967435c7c1bf078a2cda36088e3ec))
ROM_END
ROM_START(bdk_210)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "bat_v2-1_e.bin",  0x0000, 0x1b96d94, CRC(ac84fef1) SHA1(bde3250f3d95a12a5f3b74ac9d11ba0bd331e9cd))
ROM_END
ROM_START(bdk_220)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "bat_v2-2_e.bin",  0x0000, 0x1b96d94, CRC(6e415ce7) SHA1(30a3938817da20ccb87c7e878cdd8a13ada097ab))
ROM_END
ROM_START(bdk_290)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "bat_v2-9_e.bin",  0x0000, 0x1d3d2d4, CRC(09ce777e) SHA1(79b6d3f91aa4d42318c698a44444bf875ad573f2))
ROM_END
ROM_START(bdk_294)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "bdk_v2-94_e.bin", 0x0000, 0x1c00844, CRC(e087ec82) SHA1(aad2c43e6de9a520954eb50b6c824a138cd6f47f))
ROM_END
/*-------------------------------------------------------------------
/ Big Buck Hunter Pro
/-------------------------------------------------------------------*/
ROM_START(bbh_140)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "bbh_v1-4_a.bin",  0x0000, 0x1bb8fa4, CRC(302e29f0) SHA1(0c500c0a5588f8476a71599be70b515ba3e19cab))
ROM_END
ROM_START(bbh_150)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "bbh_v1-5_a.bin",  0x0000, 0x1bb8fa4, CRC(18bad072) SHA1(16e499046107baceda6f6c934d70ba2108915973))
ROM_END
ROM_START(bbh_160)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "bbh_v1-6_a.bin",  0x0000, 0x1bb8fa4, CRC(75077f85) SHA1(c58a2ae5c1332390f0d1191ee8ff920ceec23352))
ROM_END
/*-------------------------------------------------------------------
/ C.S.I.
/-------------------------------------------------------------------*/
ROM_START(csi_102)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "csi102a.bin",     0x0000, 0x1e21fc0, CRC(770f4ab6) SHA1(7670022926fcf5bb8f8848374cf1a6237803100a))
ROM_END
ROM_START(csi_103)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "csi103a.bin",     0x0000, 0x1e61c88, CRC(371bc874) SHA1(547588b85b4d6e79123178db3f3e51354e8d2229))
ROM_END
ROM_START(csi_104)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "csi104a.bin",     0x0000, 0x1e21fc0, CRC(15694586) SHA1(3a6b70d43f9922d7a459e1dc4c235bcf03e7858e))
ROM_END
ROM_START(csi_200)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "csi200a.bin",     0x0000, 0x1e21fc0, CRC(ecb25112) SHA1(385bede7955e06c1e1b7cd06e988a64b0e6ea54f))
ROM_END
ROM_START(csi_210)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "csi_v2-1_a.bin",  0x0000, 0x1e21fc0, CRC(afebb31f) SHA1(9b8179baa2f6e61852b57aaad9a28def0c014861))
ROM_END
ROM_START(csi_230)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "csi_v2-3_e.bin",  0x0000, 0x1e21fc0, CRC(c25ccc67) SHA1(51a21fca06db4b05bda2c7d5a09d655c97ba19c6))
ROM_END
ROM_START(csi_240)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "csi_v2-4_a.bin",  0x0000, 0x1e21fc0, CRC(2be97fa3) SHA1(5aa231bde81f7787cc06567c8b3d28c750588071))
ROM_END
/*-------------------------------------------------------------------
/ Family Guy
/-------------------------------------------------------------------*/
ROM_START(fg_1000af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg1000af.bin",    0x0000, 0x1ce5514, CRC(27cabf5d) SHA1(dde359c1fed728c8f91901f5ce351b5adef399f3))
ROM_END
ROM_START(fg_1000ag)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg1000ag.bin",    0x0000, 0x1c53678, CRC(130e0bd6) SHA1(ced815270d419704d94d5acdc5335460a64484ae))
ROM_END
ROM_START(fg_1000ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg1000ai.bin",    0x0000, 0x1d9f8b8, CRC(2137e62a) SHA1(ac892d2536c5dde97194ffb69c74d0517000357a))
ROM_END
ROM_START(fg_1000al)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg1000al.bin",    0x0000, 0x1e5f448, CRC(0f570f24) SHA1(8861bf3e6add7a5372d81199c135808d09b5e600))
ROM_END
ROM_START(fg_110af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg110af.bin",     0x0000, 0x1ce5514, CRC(31304627) SHA1(f36d6924f1f291f675f162ff056b6ea2f03f4351))
ROM_END
ROM_START(fg_110ag)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg110ag.bin",     0x0000, 0x1c53678, CRC(d2735578) SHA1(a38b8f690ffcdb96875d3c8293e6602d7142be11))
ROM_END
ROM_START(fg_110ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg110ai.bin",     0x0000, 0x1d9f8b8, CRC(4fa2c59e) SHA1(7fce5c1fd306eccc567ae7d155c782649c022074))
ROM_END
ROM_START(fg_110al)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg110al.bin",     0x0000, 0x1e5f448, CRC(d9b724a8) SHA1(33ac12fd4bbed11e38ade68426547ed97612cbd3))
ROM_END
ROM_START(fg_120af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg120af.bin",     0x0000, 0x1ce5514, CRC(ba6a3a2e) SHA1(78eb2e26abe00d7ce5fa998b6ec1381ac0f1db31))
ROM_END
ROM_START(fg_120ag)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg120ag.bin",     0x0000, 0x1c53678, CRC(d9734f94) SHA1(d56ddf5961e5ac4c3565f9d92d6fb7e0e0af4bcb))
ROM_END
ROM_START(fg_120ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg120ai.bin",     0x0000, 0x1d9f8b8, CRC(078b0c9a) SHA1(f1472d2c4a06d674bf652dd481cce5d6ca125e0c))
ROM_END
ROM_START(fg_120al)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg120al.bin",     0x0000, 0x1e5f448, CRC(d10cff88) SHA1(e312a3b24f1b69db9f88a5313db168d9f2a71450))
ROM_END
ROM_START(fg_300ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg300ai.bin",     0x0000, 0x1fc0290, CRC(e2cffa79) SHA1(59dff445118ed8a3a76b6e93950802d1fec87619))
ROM_END
ROM_START(fg_400a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg400a.bin",      0x0000, 0x13e789c, CRC(af6c2dd4) SHA1(e3164e982c90a5300144e63e4a74dd225fe1b272))
ROM_END
ROM_START(fg_400ag)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg400ag.bin",     0x0000, 0x1971684, CRC(3b4ae199) SHA1(4ef674badce2c90334fa7a8b6b90c32dcabc2334))
ROM_END
ROM_START(fg_700af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg700af.bin",     0x0000, 0x1a4d3d4, CRC(bbeda480) SHA1(792c396dee1b5abe113484e1fd4c4b449d8e7d95))
ROM_END
ROM_START(fg_700al)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg700al.bin",     0x0000, 0x1bc6cb4, CRC(b74dc3bc) SHA1(b24bab06b9f451cf9f068c555d3f70ffdbf40da7))
ROM_END
ROM_START(fg_800al)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "fg800al.bin",     0x0000, 0x1bce8f8, CRC(25288f43) SHA1(5a2ed2e0b264895938466ca1104ba4ed9be86b3a))
ROM_END
/*-------------------------------------------------------------------
/ Iron Man 2
/-------------------------------------------------------------------*/
ROM_START(im2_100)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "im2_v1-00_e.bin", 0x0000, 0x1b8fe44, CRC(b27d12bf) SHA1(dfb497f2edaf4321823b243cced9d9e2b7bac628))
ROM_END
ROM_START(im2_110)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "im2_v1-1_e.bin",  0x0000, 0x1b8fe44, CRC(3140cb7c) SHA1(20b0e84b61069e09f189d79e6b4d5abf0369a893))
ROM_END
ROM_START(im2_120)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "im2_v1-2_e.bin",  0x0000, 0x1b8fe44, CRC(71df27ad) SHA1(9e1745522d28af6bdcada56f2cf0b489656fc885))
ROM_END
ROM_START(im2_140)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "im2_v1-4_e.bin",  0x0000, 0x1cb8870, CRC(9cbfd6ef) SHA1(904c058a00c268593a62988127f8a18d974eda5e))
ROM_END
/*-------------------------------------------------------------------
/ Indiana Jones
/-------------------------------------------------------------------*/
ROM_START(indy_113a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0113a.bin",   0x0000, 0x1c6d98c, CRC(aa2bdf3e) SHA1(71fd1c970fe589cec5124237684facaae92cbf09))
ROM_END
ROM_START(indy_113f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0113f.bin",   0x0000, 0x1c6d98c, CRC(cb7b7c31) SHA1(3a2f718a9a533941c5476f8348dacf7e3523ddd0))
ROM_END
ROM_START(indy_113g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0113g.bin",   0x0000, 0x1bff3f4, CRC(30a33bfd) SHA1(c37b6035c313cce85d325ab87039f5a872d28f5a))
ROM_END
ROM_START(indy_113i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0113i.bin",   0x0000, 0x1c81fa4, CRC(fcb37e0f) SHA1(7b23a56baa9985e2322aee954befa13dc2d55119))
ROM_END
ROM_START(indy_113l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0113l.bin",   0x0000, 0x1d02988, CRC(e4ff8120) SHA1(f5537cf920633a621b4c7a740bfc07cefe3a99d0))
ROM_END
ROM_START(indy_114a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0114a.bin",   0x0000, 0x1c6d9e4, CRC(00e5b850) SHA1(3ad57120d11aff4ca8917dea28c2c26ae254e2b5))
ROM_END
ROM_START(indy_114f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0114f.bin",   0x0000, 0x1c6d9e4, CRC(a7c2a5e4) SHA1(c0463b055096a3112a31680dc509f421c1a5c1cf))
ROM_END
ROM_START(indy_114g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0114g.bin",   0x0000, 0x1c34974, CRC(7176b0be) SHA1(505132887bca0fa9d6ca8597101357f26501a0ad))
ROM_END
ROM_START(indy_114i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0114i.bin",   0x0000, 0x1c875f8, CRC(dac0563e) SHA1(30dbaed1b1a180f7ca68a4caef469c2997bf0355))
ROM_END
ROM_START(indy_114l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0114l.bin",   0x0000, 0x1d0b290, CRC(e9b3a81a) SHA1(574377e7a398083f3498d91640ad7dc5250acbd7))
ROM_END
ROM_START(indy_116a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0116a.bin",   0x0000, 0x1c6d9e4, CRC(80293485) SHA1(043c857a8dfa79cb7ae876c55a10227bdff8e873))
ROM_END
ROM_START(indy_116f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0116f.bin",   0x0000, 0x1c6d9e4, CRC(56821942) SHA1(484f4359b6d1ecb45c29bef7532a8136028504f4))
ROM_END
ROM_START(indy_116g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0116g.bin",   0x0000, 0x1c34974, CRC(2b7b81be) SHA1(a70ed07daec7f13165a0256bc011a72136e25210))
ROM_END
ROM_START(indy_116i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0116i.bin",   0x0000, 0x1c96b38, CRC(7b07c207) SHA1(67969e85cf96949f8b85d88acfb69be55f32ea52))
ROM_END
ROM_START(indy_116l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "indy0116l.bin",   0x0000, 0x1d14fd8, CRC(833ae2fa) SHA1(cb931e473164ddfa2559f3a58f2fcac5d456dc96))
ROM_END
ROM_START(indy_210)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "ind_v2-1_a.bin", 0x0000, 0x1c6d9e4, CRC(b96e6fd2) SHA1(f59cbdefc5ab6b21662981b3eb681fd8bd7ade54))
ROM_END
ROM_START(indy_210f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "ind_v2-1_f.bin", 0x0000, 0x1c6d9e4, CRC(d1d37248) SHA1(fd6819e0e86b83d658790ff30871596542f98c8e))
ROM_END

/*-------------------------------------------------------------------
/ NBA
/-------------------------------------------------------------------*/
ROM_START(nba_500)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "nba_v5-0_a.bin",  0x0000, 0x19112d0, CRC(01b0c27a) SHA1(d7f4f6b24630b55559a48cde4475422905811106))
ROM_END
ROM_START(nba_600)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "nba_v6-0a.bin",   0x0000, 0x19112d0, CRC(af2fbcf4) SHA1(47df1992a1eb6c4cd5ec246912eab9f5636499a7))
ROM_END
ROM_START(nba_700)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "nba_v7-0_a.bin",  0x0000, 0x19112d0, CRC(15ece43b) SHA1(90cc8b4c52a61da9701fcaba0a21144fe576eaf4))
ROM_END
ROM_START(nba_801)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "nba_v8-01_a.bin", 0x0000, 0x19112d0, CRC(0f8b146e) SHA1(090d73a9bff0a0b0c17ced1557d5e63e5c986e95))
ROM_END
ROM_START(nba_802)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "nba_v8-02_e.bin", 0x0000, 0x19112d0, CRC(ba681dac) SHA1(184f3315a54b1a5295b19222c718ac38fa60d340))
ROM_END
/*-------------------------------------------------------------------
/ Pirates of the Caribbean
/-------------------------------------------------------------------*/
ROM_START(potc_110af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0110af.bin",  0x0000, 0x1ac6550, CRC(9d87bb49) SHA1(9db04259a0b2733d6f5966a2f3e0fc1c7002cef1))
ROM_END
ROM_START(potc_110ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0110ai.bin",  0x0000, 0x1b178e8, CRC(027916d9) SHA1(0ddc0fa86da55ea0494f2095c838b41b53f568de))
ROM_END
ROM_START(potc_110gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0110gf.bin",  0x0000, 0x1b60464, CRC(ce29b69c) SHA1(ecc9ad8f77ab30538536631d513d25654f5a2f3c))
ROM_END
ROM_START(potc_111as)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0111as.bin",  0x0000, 0x1c829b4, CRC(09903169) SHA1(e284b1dc2642337633867bac9739fdda692acb2f))
ROM_END
ROM_START(potc_113af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0113af.bin",  0x0000, 0x1ac6550, CRC(1c52b3f5) SHA1(2079f06f1f1514614fa7cb240559b4e72925c70c))
ROM_END
ROM_START(potc_113ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0113ai.bin",  0x0000, 0x1b178e8, CRC(e8b487d1) SHA1(037435b40347a8e1197876fbf7a79e03befa11f4))
ROM_END
ROM_START(potc_113as)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0113as.bin",  0x0000, 0x1c829b4, CRC(2c819a02) SHA1(98a79b50e6c80bd58b2571fefc2f5f61030bc25d))
ROM_END
ROM_START(potc_113gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0113gf.bin",  0x0000, 0x1b60464, CRC(a508a2f8) SHA1(45e46af267c7caec86e4c92526c4cda85a1bb168))
ROM_END
ROM_START(potc_115af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0115af.bin",  0x0000, 0x1ac6564, CRC(008e93b2) SHA1(5a272670cb3e5e59071500124a0086ef86e2b528))
ROM_END
ROM_START(potc_115ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0115ai.bin",  0x0000, 0x1b178fc, CRC(88b66285) SHA1(1d65e4f7a31e51167b91f82d96c3951442b16264))
ROM_END
ROM_START(potc_115as)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0115as.bin",  0x0000, 0x1c829c8, CRC(9c107d0e) SHA1(5213246ee78c6cc082b9f895b1d1abfa52016ede))
ROM_END
ROM_START(potc_115gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0115gf.bin",  0x0000, 0x1b60478, CRC(09a8454c) SHA1(1af420b314d339231d3b7772ffa44175a01ebd30))
ROM_END
ROM_START(potc_300af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0300af.bin",  0x0000, 0x1ad2b40, CRC(b6fc0c4b) SHA1(5c0d6b46dd6c4f14e03298500558f376ee342de0))
ROM_END
ROM_START(potc_300ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0300ai.bin",  0x0000, 0x1b213a8, CRC(2d3eb95e) SHA1(fea9409ffea3554ff0ec1c9ef6642465ec4120e7))
ROM_END
ROM_START(potc_300as)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0300as.bin",  0x0000, 0x1c88124, CRC(e5e7049d) SHA1(570125f9eb6d7a04ba97890095c15769f0e0dbd6))
ROM_END
ROM_START(potc_300gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0300gf.bin",  0x0000, 0x1b67104, CRC(52772953) SHA1(e820ca5f347ab637bee07a9d7426058b9fd6557c))
ROM_END
ROM_START(potc_400af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0400af.bin",  0x0000, 0x1ad2b40, CRC(03cfed21) SHA1(947fff6bf3ed69cb346ae9f159e378902901033f))
ROM_END
ROM_START(potc_400ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0400ai.bin",  0x0000, 0x1b213a8, CRC(5382440b) SHA1(01d8258b98e256fc54565afd9915fd5079201973))
ROM_END
ROM_START(potc_400as)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0400as.bin",  0x0000, 0x1c88124, CRC(f739474d) SHA1(43bf3fbd23498e2cbeac3d87f5da727e7c05eb86))
ROM_END
ROM_START(potc_400gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0400gf.bin",  0x0000, 0x1b67104, CRC(778d02e7) SHA1(6524e56ebf6c5c0effc4cb0521e3a463540ceac4))
ROM_END
ROM_START(potc_600)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0600af.bin",  0x0000, 0x1ad2b40, CRC(39a51873) SHA1(9597d356a3283c5a4e488a399196a51bf5ed16ca))
ROM_END
ROM_START(potc_600ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0600ai.bin",  0x0000, 0x1b24cc8, CRC(2d7aebae) SHA1(9e383507d225859b4df276b21525f500ba98d600))
ROM_END
ROM_START(potc_600as)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0600as.bin",  0x0000, 0x1c92990, CRC(5d5e1aaa) SHA1(9c7a416ae6587a86c8d2c6350621f09580226971))
ROM_END
ROM_START(potc_600gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "potc0600gf.bin",  0x0000, 0x1b67104, CRC(44eb2610) SHA1(ec1e1f7f2cd135942531e0e3f540afadb5d2f527))
ROM_END
/*-------------------------------------------------------------------
/ Shrek
/-------------------------------------------------------------------*/
ROM_START(sh_141)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "sh0141e.bin",     0x0000, 0x1c55290, CRC(f4f847ce) SHA1(d28f9186bb04036e9ff56d540e70a50f0816051b))
ROM_END
/*-------------------------------------------------------------------
/ The Simpsons Kooky Carnival Redemption
/-------------------------------------------------------------------*/
ROM_START(skc_090)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "skc090nj.bin",    0x0000, 0x053b7cc, CRC(3a9142e0) SHA1(57d75763fb52c891d1bb16e85ae170c38e6dd818))
ROM_END
ROM_START(skc_103)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "skc0103.bin",     0x0000, 0x053a860, CRC(69f5bb8a) SHA1(436db9872d5809c7ed5fe607c4167cdc0e1b5294))
ROM_END
ROM_START(skc_105)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "skc0105.bin",     0x0000, 0x053dd14, CRC(a09ffa33) SHA1(fab75f338a5d6c82632cd0804ddac1ab78466636))
ROM_END
/*-------------------------------------------------------------------
/ Spider-Man
/-------------------------------------------------------------------*/
ROM_START(sman_102ef)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "sman_102af.bin", 0x0000, 0xfbb834, CRC(1e77651c) SHA1(fbce7dbe4ce70cd8bd1c01279a774f410f5aaeff))
ROM_END
ROM_START(sman_130ef)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1_30_ef.bin", 0x0000, 0x17916c8, CRC(6aa6a03a) SHA1(f56442e84b8789f49127bf4ba97dd05c77ea7c36))
ROM_END
ROM_START(sman_130ei)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1_30_ei.bin", 0x0000, 0x17b7960, CRC(92aab158) SHA1(51662102da54e7e7c0f63689fffbf70653ee8f11))
ROM_END
ROM_START(sman_130es)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1_30_es.bin", 0x0000, 0x180aaa0, CRC(33004e72) SHA1(3bc30200945d896aefbff51c7b427595885a23c4))
ROM_END
ROM_START(sman_130gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1_30_gf.bin", 0x0000, 0x17aec84, CRC(2838d2f3) SHA1(2192f1fbc393c5e0dcd59198d098bb2531d8b6de))
ROM_END
ROM_START(sman_140e)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1_4_e.bin",   0x0000, 0x16ce3c0, CRC(48c2565d) SHA1(78f5d3242cfaa85fa0fd3937b6042f067dff535b))
ROM_END
ROM_START(sman_140ef)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1_4_ef.bin",  0x0000, 0x1a50398, CRC(d181fa71) SHA1(66af219d9266b6b24e6857ad1a6b4fe539058052))
ROM_END
ROM_START(sman_140ei)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1_4_ei.bin",  0x0000, 0x1a70f78, CRC(0de6937e) SHA1(f2e60b545ef278e1b7981bf0a3dc2c622205e8e1))
ROM_END
ROM_START(sman_140es)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1_4_es.bin",  0x0000, 0x1adc768, CRC(fd372e14) SHA1(70f3e4d210a4da4b6122089c477b5b3f51d3593f))
ROM_END
ROM_START(sman_140gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1_4_gf.bin",  0x0000, 0x1a70f78, CRC(f1124c86) SHA1(755f15dd566f86695c7143512d81e16af71c8853))
ROM_END
ROM_START(sman_142e)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1_42_e-beta.bin", 0x0000, 0x16e8d60, CRC(307b0163) SHA1(015c8c86763c645b43bd71a3cdb8975fcd36a99f))
ROM_END
ROM_START(sman_160e)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-6_e.bin",   0x0000, 0x1725778, CRC(05425962) SHA1(a37f61239a7116e5c14a345c288f781fa6248cf8))
ROM_END
ROM_START(sman_160ef)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-6_ef.bin",  0x0000, 0x1b0121c, CRC(d0b552e9) SHA1(2550baba3c4be5308779d502a2d2d01e1c2539ef))
ROM_END
ROM_START(sman_160ei)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-6_ei.bin",  0x0000, 0x1b26d28, CRC(b776f59b) SHA1(62600474b8a5e1e2d40319817505c8b5fd3df2fa))
ROM_END
ROM_START(sman_160es)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-6_es.bin",  0x0000, 0x1bb15bc, CRC(776937d9) SHA1(631cadd665f895feac90c3cbc14eb8e321d19b4e))
ROM_END
ROM_START(sman_160gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-6_gf.bin",  0x0000, 0x1b24430, CRC(1498f877) SHA1(e625a7e683035665a0a1a97e5de0947628c3f7ea))
ROM_END
ROM_START(sman_170e)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-7_e.bin",   0x0000, 0x1877484, CRC(45c9e5f5) SHA1(8af3215ecc247186c83e235c60c3a2990364baad))
ROM_END
ROM_START(sman_170ef)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-7_ef.bin",  0x0000, 0x1c6f32c, CRC(b38f3948) SHA1(8daae4bc8b1eaca2bd43198365474f5da09b4788))
ROM_END
ROM_START(sman_170ei)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-7_ei.bin",  0x0000, 0x1c90f74, CRC(ba176624) SHA1(56c847995b5a3e2286e231c1d69f82cf5492cd5d))
ROM_END
ROM_START(sman_170es)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-7_es.bin",  0x0000, 0x1d24e70, CRC(0455f3a9) SHA1(134ff31605798989b396220f8580d1c079678084))
ROM_END
ROM_START(sman_170gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-7_gf.bin",  0x0000, 0x1c99c74, CRC(152aa803) SHA1(e18f9dcc5380126262cf1e32e99b6cc2c4aa23cb))
ROM_END
ROM_START(sman_190e)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-9_e.bin",   0x0000, 0x1652310, CRC(7822a6d1) SHA1(6a21dfc44e8fa5e138fe6474c467ef6d6544d78c))
ROM_END
ROM_START(sman_190ef)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-9_ef.bin",  0x0000, 0x18b5c34, CRC(dac27fde) SHA1(93a236afc4be6514a8fc57e45eb5698bd999eef6))
ROM_END
ROM_START(sman_190ei)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-9_ei.bin",  0x0000, 0x18cd02c, CRC(95c769ac) SHA1(e713677fea9e28b2438a30bf5d81448d3ca140e4))
ROM_END
ROM_START(sman_190es)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-9_es.bin",  0x0000, 0x1925dd0, CRC(4df8168c) SHA1(8ebfda5378037c231075017713515a3681a0e38c))
ROM_END
ROM_START(sman_190gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-9_gf.bin",  0x0000, 0x18cd02c, CRC(a4a874a4) SHA1(1e46720462f1279c417d955c500e829e878ce31f))
ROM_END
ROM_START(sman_192e)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-92_e.bin",  0x0000, 0x1920870, CRC(a44054fa) SHA1(a0910693d13cc61dba7a2bbe9185a24b33ef20ec))
ROM_END
ROM_START(sman_192ef)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-92_ef.bin", 0x0000, 0x1b81624, CRC(c9f8a7dd) SHA1(e63e98965d08b8a645c92fb34ce7fc6e1ad05ddc))
ROM_END
ROM_START(sman_192ei)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-92_ei.bin", 0x0000, 0x1b99f88, CRC(f02acad4) SHA1(da103d5ddbcbdcc19cca6c17b557dcc71942970a))
ROM_END
ROM_START(sman_192es)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-92_es.bin", 0x0000, 0x1bf19a0, CRC(501f9986) SHA1(d93f973f9eddfd85903544f0ce49c1bf17b36eb9))
ROM_END
ROM_START(sman_192gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_1-92_gf.bin", 0x0000, 0x1b9a1b4, CRC(32597e1d) SHA1(47a28cdba11b32661dbae95e3be1a41fc475fa5e))
ROM_END
ROM_START(sman_200e)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_2-00_e.bin",  0x0000, 0x168e8a8, CRC(3b13348c) SHA1(4b5c6445d7805c0a39054bd51522751030b73162))
ROM_END
ROM_START(sman_210e)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_v2-1_e.bin",  0x0000, 0x168e8a8, CRC(f983df18) SHA1(a0d46e1a58f016102773861a4f1b026755f776c8))
ROM_END
ROM_START(sman_210f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_v2-1_ef.bin", 0x0000, 0x192b160, CRC(2e86ac24) SHA1(aa223db6289a876e77080e16f29cbfc62183fa67))
ROM_END
ROM_START(sman_210ei)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_v2-1_ei.bin", 0x0000, 0x193cc7c, CRC(aadd1ea7) SHA1(a41b0067f7490c6df5d85e80b208c9993f806366))
ROM_END
ROM_START(sman_210es)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_v2-1_es.bin", 0x0000, 0x19a5d5c, CRC(8c441caa) SHA1(e40ac748284f65de5c444ac89d3b02dd987facd0))
ROM_END
ROM_START(sman_210gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_v2-1_gf.bin", 0x0000, 0x1941c04, CRC(2995cb97) SHA1(0093d3f20aebbf6129854757cc10aff63fc18a4a))
ROM_END
ROM_START(sman_220e)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_v2-2_e.bin",  0x0000, 0x18775b8, CRC(44f31e8e) SHA1(4c07d01c95c5fab1955b11e4f7c65f369a91dfd7))
ROM_END
ROM_START(sman_230e)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_v2-3_e.bin",  0x0000, 0x18775b8, CRC(a86f1768) SHA1(72662dcf05717d3b2b335077ceddabe562738468))
ROM_END
ROM_START(sman_240)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "spd_v2-4_e.bin",  0x0000, 0x18775b8, CRC(dc5ee57e) SHA1(7453db81b161cdbf7be690da15ea8a78e4a4e57d))
ROM_END
/*-------------------------------------------------------------------
/ Wheel Of Fortune
/-------------------------------------------------------------------*/
ROM_START(wof_100a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0100a.bin",    0x0000, 0x1c7df60, CRC(f3b80429) SHA1(ab1c9752ea74b5950b51aabc6dbca4f405705240))
ROM_END
ROM_START(wof_200a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0200a.bin",    0x0000, 0x1c7dfd0, CRC(2e56b65f) SHA1(908662261548f4b80433d58359e9ff1013bf315b))
ROM_END
ROM_START(wof_200f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0200f.bin",    0x0000, 0x1e76ba4, CRC(d48d4885) SHA1(25cabea55f30d86b8d6398f94e1d180377c34de6))
ROM_END
ROM_START(wof_200g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0200g.bin",    0x0000, 0x1cdec2c, CRC(81f61e6c) SHA1(395be7e0ccb9a806738fc6338b8e6dbea561986d))
ROM_END
ROM_START(wof_200i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0200i.bin",    0x0000, 0x1d45ee8, CRC(3e48eef7) SHA1(806a0313852405cd9913406201dd9e434b9b160a))
ROM_END
ROM_START(wof_300a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0300a.bin",    0x0000, 0x1c7dfd0, CRC(7a8483b8) SHA1(e361eea5a01d6ba22782d34538edd05f3b068472))
ROM_END
ROM_START(wof_300f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0300f.bin",    0x0000, 0x1e76ba4, CRC(fd5c2bec) SHA1(77f6e4177df8a17f43198843f8a0a3cf5caf1704))
ROM_END
ROM_START(wof_300g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0300g.bin",    0x0000, 0x1cdec2c, CRC(54b50069) SHA1(909b98a7f5fdfa0164c7dc52e9c830eecada2a64))
ROM_END
ROM_START(wof_300i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0300i.bin",    0x0000, 0x1d45ee8, CRC(7528800b) SHA1(d55024935861aa8895f9604e92f0d74cb2f3827d))
ROM_END
ROM_START(wof_300l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0300l.bin",    0x0000, 0x1b080b0, CRC(12e1b3a5) SHA1(6b62e40e7b124477dc8508e39722c3444d4b39a4))
ROM_END
ROM_START(wof_400a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0400a.bin",    0x0000, 0x1c7dfd0, CRC(974e6dd0) SHA1(ce4d7537e8f42ab6c3e84eac19688e2155115345))
ROM_END
ROM_START(wof_400f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0400f.bin",    0x0000, 0x1e76ba4, CRC(91a793c0) SHA1(6c390ab435dc20889bccfdd11bbfc411efd1e4f9))
ROM_END
ROM_START(wof_400g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0400g.bin",    0x0000, 0x1cdec2c, CRC(ee97a6f3) SHA1(17a3093f7e5d052c23b669ee8717a21a80b61813))
ROM_END
ROM_START(wof_400i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0400i.bin",    0x0000, 0x1d45ee8, CRC(35053d2e) SHA1(3b8d176c7b34e7eaf20f9dcf27649841c5122609))
ROM_END
ROM_START(wof_400l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0401l.bin",    0x0000, 0x1b080b0, CRC(4db936f4) SHA1(4af1d4642529164cb5bc0b9adbc229b131098007))
ROM_END
ROM_START(wof_500a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0500a.bin",    0x0000, 0x1c7dfd0, CRC(6613e864) SHA1(b6e6dcfa782720e7d0ce36f8ea33a0d05763d6bd))
ROM_END
ROM_START(wof_500f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0500f.bin",    0x0000, 0x1e76ba4, CRC(3aef1035) SHA1(4fa0a40fea403beef0b3ce695ff52dec3d90f7bf))
ROM_END
ROM_START(wof_500g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0500g.bin",    0x0000, 0x1cdec2c, CRC(658f8622) SHA1(31926717b5914f91b70eeba182eb219a4fd51299))
ROM_END
ROM_START(wof_500i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0500i.bin",    0x0000, 0x1d45ee8, CRC(27fb48bc) SHA1(9a9846c84a1fc543ec2236a28991d0cd70e86b52))
ROM_END
ROM_START(wof_500l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wof0500l.bin",    0x0000, 0x1b080b0, CRC(b8e09fcd) SHA1(522983ce75b24733a0827a2eeea3d44419c7998e))
ROM_END
/*-------------------------------------------------------------------
/ World Poker Tour
/-------------------------------------------------------------------*/
ROM_START(wpt_103a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0103a.bin",    0x0000, 0x19bb1dc, CRC(cd5f80bc) SHA1(4aaab2bf6b744e1a3c3509dc9dd2416ff3320cdb))
ROM_END
ROM_START(wpt_105a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0105a.bin",    0x0000, 0x19bb198, CRC(51608819) SHA1(a14aa47bdbce1dc958504d866ac963b06cd93bef))
ROM_END
ROM_START(wpt_106a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0106a.bin",    0x0000, 0x19bb198, CRC(72fd2e58) SHA1(3e910b964d0dc67fd538c027b474b3587b216ce5))
ROM_END
ROM_START(wpt_106f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0106f.bin",    0x0000, 0x1aa3fdc, CRC(efa3eeb9) SHA1(a5260511b6325917a9076bac6c92f1a8472142b8))
ROM_END
ROM_START(wpt_106g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0106g.bin",    0x0000, 0x1a33de8, CRC(9b486bc4) SHA1(c2c3c426201db99303131c5efb4275291ab721d7))
ROM_END
ROM_START(wpt_106i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0106i.bin",    0x0000, 0x1b2c6ec, CRC(177146f0) SHA1(279380fcc3924a8bb8e3002a66c317473d3fc773))
ROM_END
ROM_START(wpt_106l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0106l.bin",    0x0000, 0x1c706d8, CRC(e38034a1) SHA1(c391887a90f9387f86dc94e22bb7fca57c8e91be))
ROM_END
ROM_START(wpt_108a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0108a.bin",    0x0000, 0x19bb198, CRC(bca1f1f7) SHA1(cba81c9645f91f4b0b62ec1eed514069248c19b7))
ROM_END
ROM_START(wpt_108f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0108f.bin",    0x0000, 0x1aa3fdc, CRC(b1a8f235) SHA1(ea7b553f2340eb82c34f7e95f4dee6fdd3026f14))
ROM_END
ROM_START(wpt_108g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0108g.bin",    0x0000, 0x1a33de8, CRC(b77ccfae) SHA1(730de2c5e9fa85e25ce799577748c9cf7b83c5e0))
ROM_END
ROM_START(wpt_108i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0108i.bin",    0x0000, 0x1b2c6ec, CRC(748362f2) SHA1(174733a2d0f45c46dca8bc6d6bc35d39e36e465d))
ROM_END
ROM_START(wpt_108l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0108l.bin",    0x0000, 0x1c706d8, CRC(6440224a) SHA1(e1748f0204464d134c5f5083b5c12723186c0422))
ROM_END
ROM_START(wpt_109a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0109a.bin",    0x0000, 0x19bb198, CRC(6702e90c) SHA1(5d208894ef293c8a7157ab27eac9a8bca012dc43))
ROM_END
ROM_START(wpt_109f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0109f.bin",    0x0000, 0x1aa3fdc, CRC(44f64903) SHA1(f3bcb8acbc8a6cad6f8573f78de53ce8336e7879))
ROM_END
ROM_START(wpt_109f2)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0109f-2.bin",  0x0000, 0x1aa3fdc, CRC(656f3957) SHA1(8c68b00fe528f6467a9c34663bbaa9bc308fc971))
ROM_END
ROM_START(wpt_109g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0109g.bin",    0x0000, 0x1a33de8, CRC(0699b279) SHA1(e645361f02865aa5560a4bbae45e085df0c4ae22))
ROM_END
ROM_START(wpt_109i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0109i.bin",    0x0000, 0x1b2c6ec, CRC(87e5f39f) SHA1(9c79bb0f9ebb5f4f4b9ef959f56812a3fe2fda11))
ROM_END
ROM_START(wpt_109l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0109l.bin",    0x0000, 0x1c706d8, CRC(a724e6c4) SHA1(161c9e6319a305953ac169cdeceeca304ab582e6))
ROM_END
ROM_START(wpt_111a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0111a.bin",    0x0000, 0x19bb19c, CRC(423138a9) SHA1(8df7b9358cacb9399c7886b9905441dc727693a6))
ROM_END
ROM_START(wpt_111af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0111af.bin",   0x0000, 0x1a46cf0, CRC(e3a53741) SHA1(395ffe5e25248504d61bb1c96b914e712e7360c3))
ROM_END
ROM_START(wpt_111ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0111ai.bin",   0x0000, 0x1a8c8b8, CRC(a1e819c5) SHA1(f4e2dc6473f31e7019495d0f37b9b60f2c252f70))
ROM_END
ROM_START(wpt_111al)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0111al.bin",   0x0000, 0x1b2ebb0, CRC(fbe2e2cf) SHA1(ed837d6ecc1f312c84a2fd235ade86227c2df843))
ROM_END
ROM_START(wpt_111f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0111f.bin",    0x0000, 0x1aa3fe0, CRC(25573be5) SHA1(20a33f387fbf150adda835d2f91ec456077e4c41))
ROM_END
ROM_START(wpt_111g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0111g.bin",    0x0000, 0x1a33dec, CRC(96782b8e) SHA1(4b89f0d44894f0157397a65a93346e637d71c4f2))
ROM_END
ROM_START(wpt_111gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0111gf.bin",   0x0000, 0x1a74b80, CRC(c1488680) SHA1(fc652273e55d32b0c6e8e12c8ece666edac42962))
ROM_END
ROM_START(wpt_111i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0111i.bin",    0x0000, 0x1b2c6f0, CRC(4d718e63) SHA1(3ae6cefd6f96a31634f1399d1ce5d2c60955a93c))
ROM_END
ROM_START(wpt_111l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0111l.bin",    0x0000, 0x1c706dc, CRC(61f4e257) SHA1(10b11e1340593c9555ff88b0ac971433583fbf13))
ROM_END
ROM_START(wpt_112a)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0112a.bin",    0x0000, 0x19bb1b0, CRC(b98b4bf9) SHA1(75257a2759978d5fc699f78e809543d1cc8c456b))
ROM_END
ROM_START(wpt_112af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0112af.bin",   0x0000, 0x1a46d04, CRC(8fe1e3c8) SHA1(837bfc2cf7f4601f99d110428f5de5dd69d2186f))
ROM_END
ROM_START(wpt_112ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0112ai.bin",   0x0000, 0x1a8c8cc, CRC(ac878dfb) SHA1(13db57c77f5d75e87b21d3cfd7aba5dcbcbef59b))
ROM_END
ROM_START(wpt_112al)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0112al.bin",   0x0000, 0x1b2ebc4, CRC(2c0dc704) SHA1(d5735977463ee92d87aba3a41d368b92a76b2908))
ROM_END
ROM_START(wpt_112f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0112f.bin",    0x0000, 0x1aa3ff4, CRC(1f7e081c) SHA1(512d44353f619f974d98294c55378f5a1ab2d04b))
ROM_END
ROM_START(wpt_112g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0112g.bin",    0x0000, 0x1a33e00, CRC(2fbac57d) SHA1(fb19e7a4a5384fc8c91e166617dad29a47b2d8b0))
ROM_END
ROM_START(wpt_112gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0112gf.bin",   0x0000, 0x1a74b94, CRC(a6b933b3) SHA1(72a36a427527c3c5cb455a74afbbb43f2bee6480))
ROM_END
ROM_START(wpt_112i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0112i.bin",    0x0000, 0x1b2c704, CRC(0ba02986) SHA1(db1cbe0611d40c334205d0a8b9f5c6147b259549))
ROM_END
ROM_START(wpt_112l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt0112l.bin",    0x0000, 0x1c706f0, CRC(203c3a05) SHA1(6173f6a6110e2a226beb566371b2821b0a5b8609))
ROM_END
ROM_START(wpt_1400)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt1400a.bin",    0x0000, 0x19bb1ec, CRC(4b287770) SHA1(e19b60a08de9067a2b4c4dd71783fc812b3c7648))
ROM_END
ROM_START(wpt_1400af)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt1400af.bin",   0x0000, 0x1a46d40, CRC(bed3e3f1) SHA1(43b9cd6deccc8e516e2f5e99295b751ccadbac29))
ROM_END
ROM_START(wpt_1400ai)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt1400ai.bin",   0x0000, 0x1a8c908, CRC(12a62641) SHA1(680283a7493921904f7fe9fae10d965db839f986))
ROM_END
ROM_START(wpt_1400al)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt1400al.bin",   0x0000, 0x1b2ec00, CRC(2f03204b) SHA1(c7a0b645258dc1aca6a297641bc5cc10c255d5a7))
ROM_END
ROM_START(wpt_1400f)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt1400f.bin",    0x0000, 0x1aa4030, CRC(3c9ce123) SHA1(5e9f6c9e5d4cdba36b7eacc24b602ea4dde92514))
ROM_END
ROM_START(wpt_1400g)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt1400g.bin",    0x0000, 0x1a33e3c, CRC(5f8216da) SHA1(79b79acf7c457e6d70af458712bf946094d08d2a))
ROM_END
ROM_START(wpt_1400gf)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt1400gf.bin",   0x0000, 0x1a74bd0, CRC(7be526fa) SHA1(a42e5c2c1fde9ab97d7dcfe64b8c0055372729f3))
ROM_END
ROM_START(wpt_1400i)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt1400i.bin",    0x0000, 0x1b2c740, CRC(9f19ed03) SHA1(4ef570be084b1e5196a19b7f516f621025c174bc))
ROM_END
ROM_START(wpt_1400l)
	ROM_LOAD_BOOT_EPROM
	ROM_REGION( 0x2000000, "flash", ROMREGION_ERASEFF )
	ROM_LOAD( "wpt1400l.bin",    0x0000, 0x1c7072c, CRC(00eff09c) SHA1(847203d4d2ce8d11a5403374f2d5b6dda8458bc9))
ROM_END

} // Anonymous namespace

GAME( 2009, 24_130       ,24_150,   sam, sam, sam_state, empty_init, ROT0, "Stern", "24 v1.30",                                              MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2009, 24_140       ,24_150,   sam, sam, sam_state, empty_init, ROT0, "Stern", "24 v1.40",                                              MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2009, 24_144       ,24_150,   sam, sam, sam_state, empty_init, ROT0, "Stern", "24 v1.44",                                              MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2009, 24_150       ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "24 v1.50",                                              MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, bdk_130      ,bdk_294,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Batman - The Dark Knight v1.30",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, bdk_150      ,bdk_294,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Batman - The Dark Knight v1.50",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, bdk_160      ,bdk_294,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Batman - The Dark Knight v1.60",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, bdk_200      ,bdk_294,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Batman - The Dark Knight v2.00",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, bdk_210      ,bdk_294,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Batman - The Dark Knight v2.10",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, bdk_220      ,bdk_294,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Batman - The Dark Knight v2.20",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, bdk_290      ,bdk_294,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Batman - The Dark Knight v2.90",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, bdk_294      ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "Batman - The Dark Knight v2.94",                        MACHINE_IS_SKELETON_MECHANICAL)
// Missing version 1.30 (26-Jan-2010), 1.20 (25-Jan-2010), 1.10 (21-Jan-2010), 1.00 (15-Jan-2010)
GAME( 2010, bbh_140      ,bbh_160,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Big Buck Hunter Pro v1.40",                             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2010, bbh_150      ,bbh_160,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Big Buck Hunter Pro v1.50",                             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2010, bbh_160      ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "Big Buck Hunter Pro v1.60",                             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, csi_102      ,csi_240,  sam, sam, sam_state, empty_init, ROT0, "Stern", "C.S.I. v1.02",                                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, csi_103      ,csi_240,  sam, sam, sam_state, empty_init, ROT0, "Stern", "C.S.I. v1.03",                                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, csi_104      ,csi_240,  sam, sam, sam_state, empty_init, ROT0, "Stern", "C.S.I. v1.04",                                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, csi_200      ,csi_240,  sam, sam, sam_state, empty_init, ROT0, "Stern", "C.S.I. v2.00",                                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, csi_210      ,csi_240,  sam, sam, sam_state, empty_init, ROT0, "Stern", "C.S.I. v2.10",                                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, csi_230      ,csi_240,  sam, sam, sam_state, empty_init, ROT0, "Stern", "C.S.I. v2.30",                                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, csi_240      ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "C.S.I. v2.40",                                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_1000af    ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v10.0 (English,French)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_1000ag    ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v10.0 (English,German)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_1000ai    ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v10.0 (English,Italian)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_1000al    ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v10.0 (English,Spanish)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_110af     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v11.0 (English,French)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_110ag     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v11.0 (English,German)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_110ai     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v11.0 (English,Italian)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_110al     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v11.0 (English,Spanish)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_120af     ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v12.0 (English,French)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_120ag     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v12.0 (English,German)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_120ai     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v12.0 (English,Italian)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_120al     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v12.0 (English,Spanish)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_300ai     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v3.0 (English,Italian)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_400a      ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v4.0 (English)",                             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_400ag     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v4.0 (English,German)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_700af     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v7.0 (English,French)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_700al     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v7.0 (English,Spanish)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, fg_800al     ,fg_120af, sam, sam, sam_state, empty_init, ROT0, "Stern", "Family Guy v8.0 (English,Spanish)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2010, im2_100      ,im2_140,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Iron Man 2 v1.00",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2010, im2_110      ,im2_140,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Iron Man 2 v1.10",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2010, im2_120      ,im2_140,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Iron Man 2 v1.20",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2010, im2_140      ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "Iron Man 2 v1.40",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_113a    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.13",                                   MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_113f    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.13 (French)",                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_113g    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.13 (German)",                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_113i    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.13 (Italian)",                         MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_113l    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.13 (Spanish)",                         MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_114a    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.14",                                   MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_114f    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.14 (French)",                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_114g    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.14 (German)",                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_114i    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.14 (Italian)",                         MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_114l    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.14 (Spanish)",                         MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_116a    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.16",                                   MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_116f    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.16 (French)",                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_116g    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.16 (German)",                          MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_116i    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.16 (Italian)",                         MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_116l    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v1.16 (Spanish)",                         MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_210     ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v2.10",                                   MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, indy_210f    ,indy_210, sam, sam, sam_state, empty_init, ROT0, "Stern", "Indiana Jones v2.10 (French)",                          MACHINE_IS_SKELETON_MECHANICAL)
// Missing versions 4.00 (21-May-2009), 3.00 (20-May-2009), 2.00 (19-May-2009) and 1.00 (14-May-2009) of NBA
GAME( 2009, nba_500      ,nba_802,  sam, sam, sam_state, empty_init, ROT0, "Stern", "NBA v5.00",                                             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2009, nba_600      ,nba_802,  sam, sam, sam_state, empty_init, ROT0, "Stern", "NBA v6.00",                                             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2009, nba_700      ,nba_802,  sam, sam, sam_state, empty_init, ROT0, "Stern", "NBA v7.00",                                             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2009, nba_801      ,nba_802,  sam, sam, sam_state, empty_init, ROT0, "Stern", "NBA v8.01",                                             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2009, nba_802      ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "NBA v8.02",                                             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, potc_110af   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.10 (English,French)",       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, potc_110ai   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.10 (English,Italian)",      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, potc_110gf   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.10 (German,French)",        MACHINE_IS_SKELETON_MECHANICAL)
// Missing version 1.11  for other languages ??? August 17, 2006
GAME( 2006, potc_111as   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.11 (English,Spanish)",      MACHINE_IS_SKELETON_MECHANICAL)
// Missing version 1.12  August 24, 2006
GAME( 2006, potc_113af   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.13 (English,French)",       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, potc_113ai   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.13 (English,Italian)",      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, potc_113as   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.13 (English,Spanish)",      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, potc_113gf   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.13 (German,French)",        MACHINE_IS_SKELETON_MECHANICAL)
// Missing version 1.14  October 11, 2006
GAME( 2006, potc_115af   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.15 (English,French)",       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, potc_115ai   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.15 (English,Italian)",      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, potc_115as   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.15 (English,Spanish)",      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, potc_115gf   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v1.15 (German,French)",        MACHINE_IS_SKELETON_MECHANICAL)
// Missing version 2.00  April 11, 2007
GAME( 2007, potc_300af   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v3.00 (English,French)",       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, potc_300ai   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v3.00 (English,Italian)",      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, potc_300as   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v3.00 (English,Spanish)",      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, potc_300gf   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v3.00 (German,French)",        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, potc_400af   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v4.00 (English,French)",       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, potc_400ai   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v4.00 (English,Italian)",      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, potc_400as   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v4.00 (English,Spanish)",      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, potc_400gf   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v4.00 (German,French)",        MACHINE_IS_SKELETON_MECHANICAL)
// Missing version 5.00 September 27, 2007
GAME( 2007, potc_600     ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v6.00 (English,French)",       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, potc_600ai   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v6.00 (English,Italian)",      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, potc_600as   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v6.00 (English,Spanish)",      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, potc_600gf   ,potc_600, sam, sam, sam_state, empty_init, ROT0, "Stern", "Pirates of the Caribbean v6.00 (German,French)",        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2008, sh_141       ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "Shrek v1.41",                                           MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, skc_090      ,skc_105,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Simpsons Kooky Carnival (Redemption) v0.90 New Jersey", MACHINE_IS_SKELETON_MECHANICAL)
// Missing version 1.00
// Missing version 1.01  April 10, 2006
// Missing version 1.02  April 12, 2006
GAME( 2006, skc_103      ,skc_105,  sam, sam, sam_state, empty_init, ROT0, "Stern", "Simpsons Kooky Carnival (Redemption) v1.03",            MACHINE_IS_SKELETON_MECHANICAL)
// Missing version 1.04  May 12, 2006
GAME( 2006, skc_105      ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "Simpsons Kooky Carnival (Redemption) v1.05",            MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_102ef   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.02 (English,French)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_130ef   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.30 (English,French)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_130ei   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.30 (English,Italian)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_130es   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.30 (English,Spanish)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_130gf   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.30 (German,French)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_140e    ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.40",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_140ef   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.40 (English,French)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_140ei   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.40 (English,Italian)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_140es   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.40 (English,Spanish)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_140gf   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.40 (German,French)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_142e    ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.42",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_160e    ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.60",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_160ef   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.60 (English,French)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_160ei   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.60 (English,Italian)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_160es   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.60 (English,Spanish)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_160gf   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.60 (German,French)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_170e    ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.70",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_170ef   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.70 (English,French)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_170ei   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.70 (English,Italian)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_170es   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.70 (English,Spanish)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_170gf   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.70 (German,French)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_190e    ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.90",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_190ef   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.90 (English,French)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_190ei   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.90 (English,Italian)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_190es   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.90 (English,Spanish)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_190gf   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.90 (German,French)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_192e    ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.92",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_192ef   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.92 (English,French)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_192ei   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.92 (English,Italian)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_192es   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.92 (English,Spanish)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_192gf   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v1.92 (German,French)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_200e    ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v2.00",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_210e    ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v2.10",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_210f    ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v2.10 (French)",                             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_210ei   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v2.10 (English,Italian)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_210es   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v2.10 (English,Spanish)",                    MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_210gf   ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v2.10 (German,French)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_220e    ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v2.20",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_230e    ,sman_240, sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v2.30",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, sman_240     ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "Spider-Man v2.40",                                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_100a     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v1.0",                                 MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_200a     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v2.0",                                 MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_200f     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v2.0 (French)",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_200g     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v2.0 (German)",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_200i     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v2.0 (Italian)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_300a     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v3.0",                                 MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_300f     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v3.0 (French)",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_300g     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v3.0 (German)",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_300i     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v3.0 (Italian)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_300l     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v3.0 (Spanish)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_400a     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v4.0",                                 MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_400f     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v4.0 (French)",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_400g     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v4.0 (German)",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_400i     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v4.0 (Italian)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_400l     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v4.0 (Spanish)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_500a     ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v5.0",                                 MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_500f     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v5.0 (French)",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_500g     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v5.0 (German)",                        MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_500i     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v5.0 (Italian)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2007, wof_500l     ,wof_500a, sam, sam, sam_state, empty_init, ROT0, "Stern", "Wheel of Fortune v5.0 (Spanish)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_103a     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.03",                                MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_105a     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.05",                                MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_106a     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.06",                                MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_106f     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.06 (French)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_106g     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.06 (German)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_106i     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.06 (Italian)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_106l     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.06 (Spanish)",                      MACHINE_IS_SKELETON_MECHANICAL)
// Missing version 1.07 should be 5 different files per language
GAME( 2006, wpt_108a     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.08",                                MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_108f     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.08 (French)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_108g     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.08 (German)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_108i     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.08 (Italian)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_108l     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.08 (Spanish)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_109a     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.09",                                MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_109f     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.09 (French)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_109f2    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.09 (French)(alt)",                  MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_109g     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.09 (German)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_109i     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.09 (Italian)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_109l     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.09 (Spanish)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_111a     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.11",                                MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_111af    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.11 (English,French)",               MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_111ai    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.11 (English,Italian)",              MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_111al    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.11 (English,Spanish)",              MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_111f     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.11 (French)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_111g     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.11 (German)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_111gf    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.11 (German,French)",                MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_111i     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.11 (Italian)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_111l     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.11 (Spanish)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_112a     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.12",                                MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_112af    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.12 (English,French)",               MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_112ai    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.12 (English,Italian)",              MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_112al    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.12 (English,Spanish)",              MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_112f     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.12 (French)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_112g     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.12 (German)",                       MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_112gf    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.12 (German,French)",                MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_112i     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.12 (Italian)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_112l     ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v1.12 (Spanish)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_1400     ,0,        sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v14.00",                               MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_1400af   ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v14.00 (English,French)",              MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_1400ai   ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v14.00 (English,Italian)",             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_1400al   ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v14.00 (English,Spanish)",             MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_1400f    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v14.00 (French)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_1400g    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v14.00 (German)",                      MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_1400gf   ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v14.00 (German,French)",               MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_1400i    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v14.00 (Italian)",                     MACHINE_IS_SKELETON_MECHANICAL)
GAME( 2006, wpt_1400l    ,wpt_1400, sam, sam, sam_state, empty_init, ROT0, "Stern", "World Poker Tour v14.00 (Spanish)",                     MACHINE_IS_SKELETON_MECHANICAL)
