from twisted.trial import unittest
import os, sets, email

from MailPing import maildir, fileutil

class TestMaildir(unittest.TestCase):
    def setUp(self):
        self.maildir = self.mktemp()
        os.mkdir(self.maildir)
        os.mkdir(os.path.join(self.maildir, 'cur'))
        os.mkdir(os.path.join(self.maildir, 'new'))
        os.mkdir(os.path.join(self.maildir, 'tmp'))

    def testEmpty(self):
        def _fail(*a, **kw):
            raise FailTest
        maildir.process(self.maildir, _fail)

    def testCallback(self):
        fileutil.writeFile(os.path.join(self.maildir, 'new',
                                        '42.P34Q0.testhost'),
                           '''foo''')
        fileutil.writeFile(os.path.join(self.maildir, 'new',
                                        '42.P34Q1.testhost'),
                           '''bar''')
        fileutil.writeFile(os.path.join(self.maildir, 'new',
                                        '100.P43Q4.testhost'),
                           '''bar''')

        seen = sets.Set()
        def _store(path, filename):
            seen.add(filename)
        maildir.process(os.path.join(self.maildir),
                        _store)
        self.assertEquals(seen,
                          sets.Set(['42.P34Q0.testhost',
                                    '42.P34Q1.testhost',
                                    '100.P43Q4.testhost']))

    def testTimeFromFilename_Simple(self):
        self.assertEquals(maildir.getTimeFromFilename('42.P34Q0.testhost'), 42)

    def testTimeFromFilename_Microseconds_OK(self):
        self.assertApproximates(maildir.getTimeFromFilename('42.M123P34Q0.testhost'), 42.000123, 0.000001)
        self.assertApproximates(maildir.getTimeFromFilename('42.P34M123Q0.testhost'), 42.000123, 0.000001)
        self.assertApproximates(maildir.getTimeFromFilename('42.P34Q0M123.testhost'), 42.000123, 0.000001)

    def testTimeFromFilename_Microseconds_Nonnumeric(self):
        self.assertEquals(maildir.getTimeFromFilename('42.Ma123P34Q0.testhost'), 42)
        self.failUnless(isinstance(maildir.getTimeFromFilename('42.Ma123P34Q0.testhost'), int))
        self.assertEquals(maildir.getTimeFromFilename('42.P34Ma123Q0.testhost'), 42)
        self.failUnless(isinstance(maildir.getTimeFromFilename('42.P34Ma123Q0.testhost'), int))
        self.assertEquals(maildir.getTimeFromFilename('42.P34Q0Ma123.testhost'), 42)
        self.failUnless(isinstance(maildir.getTimeFromFilename('42.P34Q0Ma123.testhost'), int))

    def testTimeFromFilename_Microseconds_Truncated(self):
        self.assertEquals(maildir.getTimeFromFilename('42.MP34Q0.testhost'), 42)
        self.failUnless(isinstance(maildir.getTimeFromFilename('42.MP34Q0.testhost'), int))
        self.assertEquals(maildir.getTimeFromFilename('42.P34MQ0.testhost'), 42)
        self.failUnless(isinstance(maildir.getTimeFromFilename('42.P34MQ0.testhost'), int))
        self.assertEquals(maildir.getTimeFromFilename('42.P34Q0M.testhost'), 42)
        self.failUnless(isinstance(maildir.getTimeFromFilename('42.P34Q0M.testhost'), int))

    def testTimeFromFilename_Fail_NoPeriods(self):
        self.assertEquals(maildir.getTimeFromFilename('foo'), None)

    def testTimeFromFilename_Fail_OnePeriod(self):
        self.assertEquals(maildir.getTimeFromFilename('foo.bar'), None)

    def testTimeFromFilename_Fail_Nonnumeric(self):
        self.assertEquals(maildir.getTimeFromFilename('foo.delivery.host'), None)
