/*
 * libspe - A wrapper library to adapt the JSRE SPU usage model to SPUFS 
 * Copyright (C) 2005 IBM Corp. 
 *
 * This library is free software; you can redistribute it and/or modify it
 * under the terms of the GNU Lesser General Public License as published by 
 * the Free Software Foundation; either version 2.1 of the License, 
 * or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful, but 
 *  WITHOUT ANY WARRANTY; without even the implied warranty of MERCHANTABILITY 
 *  or FITNESS FOR A PARTICULAR PURPOSE. See the GNU Lesser General Public 
 *  License for more details.
 *
 *   You should have received a copy of the GNU Lesser General Public License 
 *   along with this library; if not, write to the Free Software Foundation, 
 *   Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */
 /** \file
  * libspe.h contains the public API funtions
  */
#ifndef _libspe_h_
#define _libspe_h_

#ifdef __cplusplus
extern "C"
{
#endif

#include "cbea_map.h"

typedef void *speid_t;
typedef void *spe_gid_t;

/* Flags for spe_create_thread
 */

#define SPE_USER_REGS		  0x0002	/* 128b user data for r3-5.   */

#define SPE_CFG_SIGNOTIFY1_OR     0x00000010
#define SPE_CFG_SIGNOTIFY2_OR     0x00000020
#define SPE_MAP_PS		  0x00000040
#define SPE_ISOLATE		  0x00000080

#define SPE_DEF_GRP		  0x00000000

#define MAX_THREADS_PER_GROUP 16

enum ps_area { SPE_MSSYNC_AREA, SPE_MFC_COMMAND_AREA, SPE_CONTROL_AREA, SPE_SIG_NOTIFY_1_AREA, SPE_SIG_NOTIFY_2_AREA };


/** spe user context
 */
struct spe_ucontext {				
	unsigned int 	gprs[128][4];
	unsigned int 	fpcr[4];
	unsigned int 	decr;
	unsigned int 	decr_status;
	unsigned int 	npc;
	unsigned int 	tag_mask;
	unsigned int 	event_mask;
	unsigned int 	srr0;
	unsigned int 	_reserved[2];
	void *       	ls;
};

/** spe_program_handle per CESOF spec
 */
typedef struct spe_program_handle {
	unsigned int	handle_size;
	void		*elf_image;
	void		*toe_shadow;
} spe_program_handle_t;

/* SIGSPE maps to SIGURG 
*/

#define SIGSPE SIGURG

/* spe_events  
 */

#define SPE_EVENT_MAILBOX		0x0001	/*Interrupting mailbox data   */
#define SPE_EVENT_STOP			0x0002	/*SPE 'stop-and-signal data'  */
#define SPE_EVENT_TAG_GROUP		0x0004	/*Tag group complete data     */
#define SPE_EVENT_DMA_ALIGNMENT		0x0008	/*A DMA alignment error       */
#define SPE_EVENT_SPE_ERROR		0x0010	/*An illegal instruction error*/
#define SPE_EVENT_SPE_DATA_SEGMENT	0x0020	/*A DMA segmentation error    */
#define SPE_EVENT_SPE_DATA_STORAGE	0x0040	/*A DMA storage error         */
#define SPE_EVENT_SPE_TRAPPED		0x0080	/*SPE 'halt' instruction      */
#define SPE_EVENT_THREAD_EXIT		0x0100	/*A thread has exited         */
#define SPE_EVENT_ERR			0x0200	/*An error occurred           */
#define SPE_EVENT_NVAL			0x0400	/*Invalid request - deprecated*/
#define SPE_EVENT_INVALID_DMA_CMD	0x0800	/*Invalid MFC/DMA command     */

struct spe_event {
	spe_gid_t	gid;		/* input, SPE group id 		*/
	int		events;		/* input, requested event mask 	*/
	int		revents;	/* output, returned events 	*/
	speid_t		speid;		/* output, returned speid 	*/
	unsigned long 	data;		/* output, returned data	*/
};

/* Signal Targets 
 */

#define SPE_SIG_NOTIFY_REG_1		0x0001
#define SPE_SIG_NOTIFY_REG_2		0x0002

/* APIs for SPE threads.
 */
/**
 * The spe_create_group function allocates a new SPE thread group. SPE thread groups 
 * define the scheduling policies and priorities for a set of SPE threads. Each SPE 
 * thread belongs to exactly one group. 
 * As an application creates SPE threads, the new threads are added to the designated 
 * SPE group. However the total number of SPE threads in a group cannot exceed the 
 * group maximum, which is dependent upon scheduling policy, priority, and 
 * availability of system resources. The spe_group_max function returns the maximum 
 * allowable number of SPE threads for a group.
 * All runnable threads in an SPE group may be gang scheduled for execution. 
 * Gang scheduling permits low-latency interaction among SPE threads in shared-memory 
 * parallel applications.
 * 
 * @param policy Defines the scheduling class for SPE threads in a group. 
 * Accepted values are: 
 * SCHED_RR which indicates real-time round-robin scheduling.
 * SCHED_FIFO which indicates real-time FIFO scheduling.
 * SCHED_OTHER which is used for low priority tasks suitable for filling 
 * otherwise idle SPE cycles. 
 * The real-time scheduling policies SCHED_RR and SCHED_FIFO are available 
 * only to processes with super-user privileges.
 * 
 * @param priority Defines the SPE group's scheduling priority within the policy class.
 * For the real-time policies SCHED_RR and SCHED_FIFO, priority is a value in the range
 * of 1 to 99. For interactive scheduling (SCHED_OTHER) the priority is a value in the 
 * range 0 to 39. The priority for an SPE thread group can be modified with 
 * spe_set_priority, or queried with spe_get_priority.
 * 
 * @param use_events A non-zero value for this parameter allows the application to 
 * receive events for SPE threads in the group. SPE events are conceptually similar 
 * to Linux signals, but differ as follows: SPE events are queued, ensuring that 
 * if multiple events are generated, each is delivered; SPE events are delivered 
 * in the order received; SPE events have associated data, including the type of 
 * event and the SPE thread id. The spe_get_event function can be called to wait 
 * for SPE events.
 * 
 * @return On success, a positive non-zero identifier for a new SPE group is returned. 
 * On error, zero is returned and errno is set to indicate the error.
 * Possible errors include:
 * @retval ENOMEM The SPE group could not be allocated due to lack of system resources.
 * @retval ENOMEM The total number of SPE groups in the system has reached the system 
 * maximum value.
 * @retval EINVAL The requested scheduling policy or priority was invalid.
 * @retval EPERM The process does not have sufficient privileges to create an SPE 
 * group with the requested scheduling policy or priority.
 * @retval ENOSYS The SPE group could not be allocated due to lack of implementation 
 * support for the specified scheduling priority or policy.
 * 
 * @sa spe_create_thread(), spe_group_defaults(), spe_group_max(), spe_get_priority(), 
 * spe_set_priority(), spe_get_policy()
 * 
 */
extern spe_gid_t spe_create_group (int policy, int priority, int spe_events);

/**
 * The spe_group_max function returns the maximum number of SPE threads that may be 
 * created for an SPE group, as indicated by gid.
 * The total number of SPE threads in a group cannot exceed the group maximum, 
 * which is dependent upon the group’s scheduling policy, priority, and availability 
 * of system resources.
 * @param spe_gid This is the identifier of the SPE group.
 * @return On success, the maximum number of SPE threads allowed for the SPE group 
 * is return. On error, -1 is returned and errno is set appropriately.
 * Possible errors include:
 * @retval EPERM The calling process does not have privileges to query the SPE group.
 * @retval ESRCH The specified SPE group could not be found.
 * @sa spe_create_group, spe_create_thread
 */
extern int spe_group_max (spe_gid_t spe_gid);

/**
 * The spe_count_physical_spes returns the total number of physical SPEs available
 * to applications in the system under control of the currently running OS.
 *
 * @return Value On success, the number of physical SPEs is returned. On error, -1 is 
 * returned and errno is set appropriately.
 * Possible errors include:
 * @retval  EIO An I/O error occurred
 */
  extern int spe_count_physical_spes(void);

/**
 * spe_create_thread creates a new SPE thread of control that can be executed independently of the calling task. As an application creates SPE threads, the threads are added to the designated SPE group. The total number of SPE threads in a group cannot exceed the group maximum. The spe_group_max function returns the number of SPE threads allowed for the group.
 * @param gid Identifier of the SPE group that the new thread will belong to. SPE group identifiers are returned by spe_create_group. The new SPE thread inherits scheduling attributes from the designated SPE group. If gid is equal to SPE_DEF_GRP (0), then a new group is created with default scheduling attributes, as set by calling spe_group_defaults.
 * 
 * @param handle
 * Indicates the program to be executed on the SPE. This is an opaque pointer to an 
 * SPE ELF image which has already been loaded and mapped into system memory. This 
 * pointer is normally provided as a symbol reference to an SPE ELF executable image 
 * which has been embedded into a PPE ELF object and linked with the calling PPE 
 * program. This pointer can also be established dynamically by loading a shared 
 * library containing an embedded SPE ELF executable, using dlopen(2) and dlsym(2), 
 * or by using the spe_open_image function to load and map a raw SPE ELF executable.
 * @param argp
 * An (optional) pointer to application specific data, and is passed as the second 
 * parameter to the SPE program.
 * @param envp
 * An (optional) pointer to environment specific data, and is passed as the third 
 * parameter to the SPE program.
 * @param mask
 * The processor affinity mask for the new thread. Each bit in the mask enables (1) 
 * or disables (0) thread execution on a cpu. At least one bit in the affinity mask 
 * must be enabled. If equal to -1, the new thread can be scheduled for execution 
 * on any processor. The affinity mask for an SPE thread can be changed by calling 
 * spe_set_affinity, or queried by calling spe_get_affinity.
 * A bit-wise OR of modifiers that are applied when the new thread is created. The 
 * following values are accepted:\n
 * 0 No modifiers are applied\n
 * SPE_CFG_SIGNOTIFY1_OR Configure the Signal Notification 1 Register to be in 
 * 'logical OR' mode instead of the default 'Overwrite' mode.\n
 * SPE_CFG_SIGNOTIFY2_OR Configure the Signal Notification 1 Register to be in 
 * 'logical OR' mode instead of the default 'Overwrite' mode.\n
 * SPE_MAP_PS Request permission for memory-mapped access to the SPE thread’s 
 * problem state area(s). Direct access to problem state is a real-time feature, 
 * and may only be available to programs running with privileged authority 
 * (or in Linux, to processes with access to CAP_RAW_IO; see capget(2) for details).
 * 
 * @param flags SPE_USER_REGS
 * Specifies that the SPE setup registers r3, r4, and r5 are initialized with the 48 bytes pointed to by argp.
 * 
 * @return
 * On success, a positive non-zero identifier of the newly created SPE thread is returned. On error, 0 is returned and errno is set to indicate the error.
 * Possible errors include:
 * @retval ENOMEM The SPE thread could not be allocated due to lack of system resources
 * @retval EINVAL The value passed for mask or flags was invalid. 
 * SPE Thread Management Facilities 5
 * @retval EPERM The process does not have permission to add threads to the designated SPE group, or to use the SPU_MAP_PS setting.
 * @retval ESRCH The SPE group could not be found.
 * @sa spe_create_group
 * spe_get_group spe_get_ls
 * spe_get_ps_area
 * spe_get_threads
 * spe_group_defaults
 * spe_group_max
 * spe_open_image, spe_close_image
 */
extern speid_t spe_create_thread (spe_gid_t gid, spe_program_handle_t *handle,
				  void *argp, void *envp,
				  unsigned long mask, int flags);
/**
 * spe_recycle
 */				 
extern int spe_recycle (speid_t speid, spe_program_handle_t *handle);
/**
 * spe_wait suspends execution of the current process until the SPE thread specified 
 * by speid has exited. If the SPE thread has already exited by the time of the 
 * call (a so-called “zombie” SPE thread), then the function returns immediately. 
 * Any system resources used by the SPE thread are freed.
 * @param speid Wait for the SPE thread identified.
 * @param options This parameter is an logical OR of zero or more of the following
 * constants:\n
 * WNOHANG Return immediately if the SPE thread has not exited.\n
 * WUNTRACED Return if the SPE thread is stopped and its status has not been reported.
 * @param status If this value is non-NULL, spe_wait stores the SPE thread’s exit c
 * ode at the address indicated by status. This status can be evaluated with the 
 * following macros. Note: these macros take the status buffer, an int, as a 
 * parameter - not a pointer to the buffer!\n
 * WIFEXITED(status) Is non-zero if the SPE thread exited normally.\n
 * WEXITSTATUS(status) Evaluates to the least significant eight bits of the return 
 * code of the SPE thread which terminated, which may have been set as the argument 
 * to a call to exit() or as the argument for a return statement in the main program. 
 * This macro can only be evaluated if WIFEXITED returned non-zero.\n
 * WIFSIGNALED(status) Returns true if the SPE thread exited because of a signal 
 * which was not caught.
 * WTERMSIG(status) Returns the number of the signal that caused the SPE thread to 
 * terminate. This macro can only be evaluated if WIFSIGNALED returned non-zero.
 * WIFSTOPPED(status) Returns true if the SPE thread which caused the return is 
 * currently stopped; this is only possible if the call was done using WUNTRACED.
 * WSTOPSIG(status) Returns the number of the signal which caused the SPE thread 
 * to stop. This macro can only be evaluated if WIFSTOPPED returned non-zero.
 * @return On success, 0 is returned. A 0 value is returned if WNOHANG was used and 
 * the SPE thread was available. On failure, -1 is returned and errno is set 
 * appropriately.
 * Possible errors include:
 * @retval ESRCH The specified SPE thread could not be found.
 * @retval EINVAL The options parameter is invalid.
 * @retval EFAULT status points to an address that was not contained in the calling 
 * process’s address space.
 * @retval EPERM The calling process does not have permission to wait on the 
 * specified SPE thread.
 * @retval EAGAIN The wait queue was active at the time spe_wait was called, 
 * prohibiting additional waits, so try again.
 * 
 * @sa spe_create_thread
 */
extern int spe_wait (speid_t speid, int *status, int options);

/**
 * spe_get_event polls or waits for events that may be generated by threads in an SPE 
 * group.
 * @param pevents This specifies an array of SPE event structures
 * @param nevents This specifies the number of spe_event structures in the pevents 
 * array.
 * @param timeout This specifies the timeout value in milliseconds. A negative value 
 * means an infinite timeout. If none of the events requested (and no error) had 
 * occurred any of the SPE groups, the operating system waits for timeout 
 * milliseconds for one of these events to occur.
 * 
 * @return On success, a positive number is returned, where the number returned 
 * indicates the number of structures which have non-zero revents fields (in other 
 * words, those with events or errors reported). A value of 0 indicates that the 
 * call timed out and no events have been selected. On error, -1 is returned and 
 * errno is set appropriately.\n
 * Possible errors include:
 * @retval EFAULT The array given as a parameter was not contained in the calling 
 * program’s address space.
 * @retval EINVAL No SPE groups have yet been created.
 * @retval EINTR A signal occurred before any requested event.
 * @retval EPERM The current process does not have permission to get SPE events.
 * 
 * @sa spe_create_group, poll
 */
extern int spe_get_event(struct spe_event *pevents, int nevents, int timeout);
/**
 * The spe_kill can be used to send a control signal to an SPE thread.
 * @param speid The signal is delivered to the SPE thread identified.
 * @param sig This indicates the type of control signal to be delivered. 
 * It may be one of the following values:\n
 * SIGKILL Kill the specified SPE thread.\n
 * SIGSTOP Stop execution of the specified SPE thread.\n
 * SIGCONT Resume execution of the specified SPE thread.\n
 * 
 * @return On success, 0 is returned. On error, -1 is returned and errno is set 
 * appropriately.
 * Possible errors include:
 * @retval ENOSYS The spe_kill operation is not supported by the implementation or 
 * environment.
 * @retval EPERM The calling process does not have permission to perform the kill 
 * action for the receiving SPE thread.
 * @retval ESRCH The SPE thread does not exist. Note that a existing SPE thread 
 * might be a zombie, an SPE thread which is already committed termination but 
 * yet had spe_wait called for it.
 * @sa spe_create_thread, spe_wait, kill (2)
 */
extern int spe_kill (speid_t speid, int sig);

/**
 * The scheduling priority for the SPE thread group, as indicated by gid, is 
 * obtained by calling the spe_get_priority.
 * @param gid The identifier of a specific SPE group.
 * @return On success, spe_get_priority returns a priority value of 0 to 99. On 
 * failure, spe_get_priority returns -1 and sets errno appropriately.
 * Possible errors include:
 * @retval ESRCH The specified SPE thread group could not be found.
 * @sa spe_create_group
 */
extern int spe_get_priority(spe_gid_t gid);

/**
 * The scheduling priority for the SPE thread group, as indicated by gid is set by 
 * calling the spe_set_priority function.
 * For the real-time policies SCHED_RR and SCHED_FIFO, priority is a value in the 
 * range of 1 to 99. Only the super-user may modify real-time priorities. 
 * For the interactive policy SCHED_OTHER, priority is a value in the range 0 to 39. 
 * Only the super-user may raise interactive priorities.
 * 
 * @param gid The identifier of a specific SPE group.
 * @param priority Specified the SPE thread group’s scheduling priority within the 
 * group’s scheduling policy class.
 * @return On success, spe_set_priority returns zero. On failure, spe_set_priority 
 * returns -1 and sets errno appropriately.
 * Possible errors include:
 * @retval EINVAL The specified priority value is invalid.
 * @retval EPERM The current process does not have permission to set the specified 
 * SPE thread group priority.
 * @retval ESRCH The specified SPE thread group could not be found.
 * @sa spe_create_group
 */
extern int spe_set_priority(spe_gid_t gid, int priority);

/**
 * The scheduling policy class for an SPE group is queried by calling the 
 * spe_get_policy function.
 * 
 * @param gid The identifier of a specific SPE group.
 * @return On success, spe_get_policy returns a scheduling policy class value 
 * of SCHED_RR, SCHED_FIFO, or SCHED_OTHER. On failure, spe_get_policy returns -1 
 * and sets errno appropriately.
 * SPE thread group priority.
 * @retval ESRCH The specified SPE thread group could not be found.
 * @sa spe_create_group
 */
extern int spe_get_policy(spe_gid_t gid);

/**
 * The spe_get_group function returns the SPE group identifier for the SPE thread, 
 * as indicated by speid.
 * @param speid The identifier of a specific SPE thread.
 * @return The SPE group identifier for an SPE thread, or 0 on failure.
 * Possible errors include:
 * @retval ESRCH The specified SPE thread could not be found.
 * 
 * @sa spe_create_group, spe_get_threads
 */
extern spe_gid_t spe_get_group (speid_t speid);

/**
 * The spe_get_ls function returns the address of the local storage 
 * for the SPE thread indicated by speid.
 * 
 * @param speid The identifier of a specific SPE thread.
 * @return On success, a non-NULL pointer is returned. On failure, NULL is returned 
 * and errno is set appropriately. Possible errors include:
 * @retval ENOSYS Access to the local store of an SPE thread is not supported by 
 * the operating system.
 * @retval ESRCH The specified SPE thread could not be found.
 * @sa spe_create_group, spe_get_ps_area
 */
extern void *spe_get_ls(speid_t speid);

/**
 * spe_group_defaults changes the application defaults for SPE groups. When an 
 * application calls spe_create_thread and designates an SPE group id equal to 
 * SPE_DEF_GRP (0), then a new group is created and the thread is added to the 
 * new group. The group is created with default settings for memory access 
 * privileges and scheduling attributes. By calling spe_group_defaults, the 
 * application can override the settings for these attributes.
 * The initial attribute values for SPE group 0 are defined as follows: 
 * the policy is set to SCHED_OTHER; the priority is set to 0; and spe_events are 
 * disabled.
 * 
 * @param policy This defines the scheduling class. Accepted values are:\n
 * SCHED_RR which indicates real-time round-robin scheduling.\n
 * SCHED_FIFO which indicates real-time FIFO scheduling.\n
 * SCHED_OTHER which is used for low priority tasks suitable 
 * for filling otherwise idle SPE cycles.\n
 * @param priority This defines the default scheduling priority. For the real-time 
 * policies SCHED_RR and SCHED_FIFO, priority is a value in the range of 1 to 99. 
 * For interactive scheduling
 * (SCHED_OTHER) the priority is a value in the range 0 to 99.
 * @param spe_events A non-zero value for this parameter registers the application’s 
 * interest i<n SPE events for the group.
 * @return On success, 0 is returned. On failure, -1 is returned and errno is set 
 * appropriately.
 * Possible errors include:
 * @retval EINVAL The specified policy or priority value is invalid.
 * @sa spe_create_group, spe_create_thread
 */
extern int spe_group_defaults(int policy, int priority, int spe_events);
/**
 * spe_get_threads returns a list of SPE threads in a group, as indicated by gid, 
 * to the array pointed to by spe_ids.
 * The storage for the spe_ids array must be allocated and managed by the application.
 * Further, the spe_ids array must be large enough to accommodate the current number 
 * of SPE threads in the group. The number of SPE threads in a group can be obtained 
 * by setting the spe_ids parameter to NULL.
 * 
 * @param gid This is the identifier of the SPE group.
 * @param spe_ids This is a pointer to an array of speid_t values that are filled in 
 * with the ids of the SPE threads in the group specified by gid.
 * 
 * @return On success, the number of SPE threads in the group is returned. 
 * On failure, -1 is returned and errno is set appropriately.
 * Possible errors include:
 * @retval EFAULT The spe_ids array was contained within the calling program’s 
 * address space. 
 * @retval EPERM The current process does not have permission to query SPE threads 
 * for this group.
 * @retval ESRCH The specified SPE thread group could not be found.
 * @sa spe_create_group, spe_create_thread
 */
extern int spe_get_threads(spe_gid_t gid, speid_t *spe_ids);

/**
 * The spe_destroy_group function marks groups to be destroyed, and destroys them 
 * if they do not contain any SPE threads. If a SPE group cannot be destroyed
 * immediately it will be destroyed after the last SPE thread has ended.
 * @param spe_gid The identifier of a specific SPE group.
 * @return 0 on success, -1 on failure.
 * Possible errors include:
 * @retval ESRCH The specified SPE group could not be found.
 * 
 * @sa spe_create_group, spe_get_group
 */
extern int spe_destroy_group(spe_gid_t spe_gid);

/* Currently without implementation or support
 */
/**
 * The spe_set_affinity function sets the processor affinity mask for an SPE thread.
 * @param speid Identifier of a specific SPE thread.
 * @param mask The affinity bitmap is represented by the value specified by mask. 
 * The least significant bit corresponds to the first cpu on the system, while the 
 * most significant bit corresponds to the last cpu on the system. A set bit 
 * corresponds to a legally schedulable processor while an unset bit corresponds 
 * to an illegally schedulable processor. In other words, a thread is bound to and 
 * will only run on a cpu whose corresponding bit is set. Usually, all bits in the 
 * mask are set.
 * @return On success, spe_get_affinity and spe_set_affinity return 0. On failure, 
 * -1 is returned and errno is set appropriately. spe_get_affinity returns the 
 * affinity mask in the memory pointed to by the mask parameter. 
 * Possible errors include:
 * @retval EFAULT The supplied memory address for mask was invalid.
 * @retval EINVAL The mask is invalid or cannot be applied.
 * @retval ENOSYS The affinity setting operation is not supported by the i
 * mplementation or environment.
 * @retval ESRCH The specified SPE thread could not be found.
 * @sa spe_create_thread, sched_setaffinity
 */
extern int spe_get_affinity( speid_t speid, unsigned long *mask);

/**
 * The spe_set_affinity function sets the processor affinity mask for an SPE thread.
 * @param speid Identifier of a specific SPE thread.
 * @param mask The affinity bitmap is represented by the value specified by mask. 
 * The least significant bit corresponds to the first cpu on the system, while the 
 * most significant bit corresponds to the last cpu on the system. A set bit 
 * corresponds to a legally schedulable processor while an unset bit corresponds 
 * to an illegally schedulable processor. In other words, a thread is bound to and 
 * will only run on a cpu whose corresponding bit is set. Usually, all bits in the 
 * mask are set.
 * @return On success, spe_get_affinity and spe_set_affinity return 0. On failure, 
 * -1 is returned and errno is set appropriately. spe_get_affinity returns the 
 * affinity mask in the memory pointed to by the mask parameter. 
 * Possible errors include:
 * @retval EFAULT The supplied memory address for mask was invalid.
 * @retval EINVAL The mask is invalid or cannot be applied.
 * @retval ENOSYS The affinity setting operation is not supported by the i
 * mplementation or environment.
 * @retval ESRCH The specified SPE thread could not be found.
 * @sa spe_create_thread, sched_setaffinity
 */
extern int spe_set_affinity(speid_t speid, unsigned long mask);

/**
 * The spe_get_context call returns the SPE user context for an SPE thread.
 * @param speid Specifies the SPE thread
 * @param uc Points to the SPE user context structure.
 * 
 * @return On failure, -1 is returned and errno is set appropriately.
 * 
 * @sa spe_kill, spe_create_thread, spe_wait, getcontext, setcontect
 */
extern int spe_get_context(speid_t speid, struct spe_ucontext *uc);

/**
 * The spe_set_context call sets the SPE user context for an SPE thread.
 * @param speid Specifies the SPE thread
 * @param uc Points to the SPE user context structure.
 * 
 * @return On failure, -1 is returned and errno is set appropriately.
 * 
 * @sa spe_kill, spe_create_thread, spe_wait, getcontext, setcontect
 */
extern int spe_set_context(speid_t speid, struct spe_ucontext *uc);

/**
 * spe_open_image maps an SPE ELF executable indicated by filename into system 
 * memory and returns the mapped address appropriate for use by the spe_create_thread 
 * API. It is often more convenient/appropriate to use the loading methodologies 
 * where SPE ELF objects are converted to PPE static or shared libraries with 
 * symbols which point to the SPE ELF objects after these special libraries are 
 * loaded. These libraries are then linked with the associated PPE code to provide 
 * a direct symbol reference to the SPE ELF object. The symbols in this scheme 
 * are equivalent to the address returned from the spe_open_image function.
 * SPE ELF objects loaded using this function are not shared with other processes, 
 * but SPE ELF objects loaded using the other scheme, mentioned above, can be 
 * shared if so desired.
 * 
 * @param filename Specifies the filename of an SPE ELF executable to be loaded 
 * and mapped into system memory.
 * 
 * @return On success, spe_open_image returns the address at which the specified 
 * SPE ELF object has been mapped. On failure, NULL is returned and errno is set 
 * appropriately.
 * Possible errors include:
 * @retval EACCES The calling process does not have permission to access the 
 * specified file.
 * @retval EFAULT The filename parameter points to an address that was not 
 * contained in the calling process’s address space.
 * 
 * A number of other errno values could be returned by the open(2), fstat(2), 
 * mmap(2), munmap(2), or close(2) system calls which may be utilized by the 
 * spe_open_image or spe_close_image functions.
 * @sa spe_create_thread
 */
extern spe_program_handle_t *spe_open_image(const char *filename);

/**
 * spe_close_image unmaps an SPE ELF object that was previously mapped using 
 * spe_open_image.
 * @param program_handle handle to open file
 * 
 * @return On success, spe_close_image returns 0. On failure, -1 is returned and errno is 
 * set appropriately.
 * @retval EINVAL From spe_close_image, this indicates that the file, specified by 
 * filename, was not previously mapped by a call to spe_open_image.
 */
extern int spe_close_image(spe_program_handle_t *program_handle);

/* ===========================================================================
 * mfc function wrappers in spe library are defined here.
 */

/**
 * The spe_write_in_mbox function places the 32-bit message specified by data into 
 * the SPU inbound mailbox for the SPE thread specified by the speid parameter.
 * If the mailbox is full, then spe_write_in_mbox can overwrite the last entry in 
 * the mailbox. The spe_stat_in_mbox function can be called to ensure that space is 
 * available prior to writing to the inbound mailbox.
 * 
 * @param speid Specifies the SPE thread whose outbound mailbox is to be read.
 * @param data The 32-bit message to be written into the SPE’s inbound mailbox.
 * @return On success, spe_write_in_mbox returns 0. On failure, -1 is returned.
 * @sa spe_read_out_mbox, spe_stat_in_mbox. Spe_stat_out_mbox, 
 * spe_stat_out_intr_mbox, write (2)
 */
extern int spe_write_in_mbox (speid_t speid ,unsigned int data);

/**
 * The spe_stat_in_mbox function fetches the status of the SPU inbound mailbox for 
 * the SPE thread specified by the speid parameter. A 0 value is return if the 
 * mailbox is full. A non-zero value specifies the number of available (32-bit) 
 * mailbox entries.
 * 
 * @param speid Specifies the SPE thread whose mailbox status is to be read.
 * @return On success, returns the current status of the mailbox, respectively. 
 * On failure, -1 is returned.
 * @sa spe_read_out_mbox, spe_write_in_mbox, read (2)
 */
extern int spe_stat_in_mbox (speid_t speid);

/**
 * The spe_read_out_mbox function returns the contents of the SPU outbound mailbox 
 * for the SPE thread specified by the speid parameter. This read is non-blocking 
 * and returns -1 if no mailbox data is available.
 * The spe_stat_out_mbox function can be called to ensure that data is available 
 * prior to reading the outbound mailbox.
 * @param speid Specifies the SPE thread whose outbound mailbox is to be read.
 * @return On success, spe_read_out_mbox returns the next 32-bit mailbox message. 
 * On failure, -1 is returned.
 * @sa spe_stat_in_mbox, spe_stat_out_mbox, spe_stat_out_intr_mbox,
 * spe_write_in_mbox, read (2)
 */
extern unsigned int spe_read_out_mbox (speid_t speid);

/**
 * The spe_stat_out_mbox function fetches the status of the SPU outbound mailbox 
 * for the SPE thread specified by the speid parameter. A 0 value is return if 
 * the mailbox is empty. A non-zero value specifies the number of 32-bit unread 
 * mailbox entries.
 * 
 * @param speid Specifies the SPE thread whose mailbox status is to be read.
 * @return On success, returns the current status of the mailbox, respectively. 
 * On failure, -1 is returned.
 * @sa spe_read_out_mbox, spe_write_in_mbox, read (2)
 */
extern int spe_stat_out_mbox (speid_t speid);

/**
 * The spe_stat_out_intr_mbox function fetches the status of the SPU outbound 
 * interrupt mailbox for the SPE thread specified by the speid parameter. A 0 
 * value is return if the mailbox is empty. A non-zero value specifies the number 
 * of 32-bit unread mailbox entries.
 * 
 * @param speid Specifies the SPE thread whose mailbox status is to be read.
 * @return On success, returns the current status of the mailbox, respectively. 
 * On failure, -1 is returned.
 * @sa spe_read_out_mbox, spe_write_in_mbox, read (2)
 */
extern int spe_stat_out_intr_mbox (speid_t speid);

/**
 * The spe_read_out_intr_mbox function reads the contents of the SPE outbound interrupting
 * mailbox for the SPE context.
 */
extern unsigned int spe_read_out_intr_mbox(speid_t speid);

/**
 * The spe_write_signal function writes data to the signal notification register 
 * specified by signal_reg for the SPE thread specified by the speid parameter.
 * 
 * @param speid Specifies the SPE thread whose signal register is to be written to.
 * @param signal_reg Specified the signal notification register to be written. 
 * Valid signal notification registers are:\n
 * SPE_SIG_NOTIFY_REG_1 SPE signal notification register 1\n
 * SPE_SIG_NOTIFY_REG_2 SPE signal notification register 2\n
 * @param data The 32-bit data to be written to the specified signal notification 
 * register.
 * @return On success, spe_write_signal returns 0. On failure, -1 is returned.
 * @sa spe_get_ps_area, spe_write_in_mbox
 */ 
extern int spe_write_signal (speid_t speid, unsigned int signal_reg, unsigned int data);

/**
 * The spe_mfc_put function places a put DMA command on the proxy command queue 
 * of the SPE thread specified by speid. The put command transfers size bytes of 
 * data starting at the local store address specified by ls to the effective 
 * address specified by ea. The DMA is identified by the tag id specified by 
 * tag and performed according transfer class and replacement class specified 
 * by tid and rid respectively.
 * 
 * @param spe Specifies the SPE thread whose proxy command queue the get command 
 * is to be placed into.
 * @param src Specifies the starting local store source address.
 * @param dst Specifies the starting effective address source address.
 * @param size Specifies the size, in bytes, to be transferred.
 * @param tag Specifies the tag id used to identify the DMA command.
 * @param tid Specifies the transfer class identifier of the DMA command.
 * @param rid Specifies the replacement class identifier of the DMA command.
 * @return On success, return 0. On failure, -1 is returned.
 */
extern int spe_mfc_put (speid_t spe, unsigned src, void *dst,
			unsigned size, unsigned short tag,
			unsigned char tid, unsigned char rid);

/**
 * The spe_mfc_puttf function is identical to spe_mfc_put except that it places a 
 * putf (put with fence) DMA command on the proxy command queue. The fence form 
 * ensures that this command is locally ordered with respect to all previously 
 * issued commands with the same tag group and command queue.
 * 
 * @param spe Specifies the SPE thread whose proxy command queue the get command 
 * is to be placed into.
 * @param src Specifies the starting local store source address.
 * @param dst Specifies the starting effective address source address.
 * @param size Specifies the size, in bytes, to be transferred.
 * @param tag Specifies the tag id used to identify the DMA command.
 * @param tid Specifies the transfer class identifier of the DMA command.
 * @param rid Specifies the replacement class identifier of the DMA command.
 * @return On success, return 0. On failure, -1 is returned.
 */
extern int spe_mfc_putf(speid_t spe, unsigned src, void *dst,
			unsigned size, unsigned short tag,
			unsigned char tid, unsigned char rid);

/**
 * The spe_mfc_putb function is identical to spe_mfc_put except that it places a 
 * putb (put with barrier) DMA command on the proxy command queue. The barrier 
 * form ensures that this command and all sequence commands with the same tag 
 * identifier as this command are locally ordered with respect to all previously i
 * ssued commands with the same tag group and command queue.
 * 
 * @param spe Specifies the SPE thread whose proxy command queue the get command 
 * is to be placed into.
 * @param src Specifies the starting local store source address.
 * @param dst Specifies the starting effective address source address.
 * @param size Specifies the size, in bytes, to be transferred.
 * @param tag Specifies the tag id used to identify the DMA command.
 * @param tid Specifies the transfer class identifier of the DMA command.
 * @param rid Specifies the replacement class identifier of the DMA command.
 * @return On success, return 0. On failure, -1 is returned.
 */
extern int spe_mfc_putb(speid_t spe, unsigned src, void *dst,
			unsigned size, unsigned short tag,
			unsigned char tid, unsigned char rid);

/* FUNCTION:	spe_mfc_get (speid, dst, src, size, tag, tid, rid)
 *		spe_mfc_getf(speid, dst, src, size, tag, tid, rid)
 *		spe_mfc_getb(speid, dst, src, size, tag, tid, rid)
 *
 * initiate data transfer from process memory to local storage
 * optionally including a fence (getf) or barrier (getb)
 */
/**
 * The spe_mfc_get function places a get DMA command on the proxy command queue 
 * of the SPE thread specified by speid. The get command transfers size bytes of 
 * data starting at the effective address specified by ea to the local store 
 * address specified by ls. The DMA is identified by the tag id specified by tag 
 * and performed according to the transfer class and replacement class specified 
 * by tid and rid respectively.
 * @param spe Specifies the SPE thread whose proxy command queue the get command 
 * is to be placed into.
 * @param dst Specifies the starting local store destination address.
 * @param src Specifies the starting effective address source address.
 * @param size Specifies the size, in bytes, to be transferred.
 * @param tag Specifies the tag id used to identify the DMA command.
 * @param tid Specifies the transfer class identifier of the DMA command.
 * @param rid Specifies the replacement class identifier of the DMA command.
 * @return On success, return 0. On failure, -1 is returned.
 */
extern int spe_mfc_get (speid_t spe, unsigned dst, void *src,
			unsigned size, unsigned short tag,
			unsigned char tid, unsigned char rid);
/**
 * The spe_mfc_getf function is identical to spe_mfc_get except that it places a 
 * getf (get with fence) DMA command on the proxy command queue. The fence form 
 * ensure that this command is locally ordered with respect to all previously 
 * issued commands with the same tag group and command queue.
 * 
 * @param spe Specifies the SPE thread whose proxy command queue the get command 
 * is to be placed into.
 * @param dst Specifies the starting local store destination address.
 * @param src Specifies the starting effective address source address.
 * @param size Specifies the size, in bytes, to be transferred.
 * @param tag Specifies the tag id used to identify the DMA command.
 * @param tid Specifies the transfer class identifier of the DMA command.
 * @param rid Specifies the replacement class identifier of the DMA command.
 * @return On success, return 0. On failure, -1 is returned.
 */
extern int spe_mfc_getf(speid_t spe, unsigned dst, void *src,
			unsigned size, unsigned short tag,
			unsigned char tid, unsigned char rid);

/**
 * The spe_mfc_getb function is identical to spe_mfc_get except that it places a 
 * getb (get with barrier) DMA command on the proxy command queue. The barrier 
 * form ensures that this command and all sequence commands with the same tag 
 * identifier as this command are locally ordered with respect to all previously 
 * issued commands with the same tag group and command queue.
 * 
 * @param spe Specifies the SPE thread whose proxy command queue the get command 
 * is to be placed into.
 * @param dst Specifies the starting local store destination address.
 * @param src Specifies the starting effective address source address.
 * @param size Specifies the size, in bytes, to be transferred.
 * @param tag Specifies the tag id used to identify the DMA command.
 * @param tid Specifies the transfer class identifier of the DMA command.
 * @param rid Specifies the replacement class identifier of the DMA command.
 * @return On success, return 0. On failure, -1 is returned.
 */
extern int spe_mfc_getb(speid_t spe, unsigned dst, void *src,
			unsigned size, unsigned short tag,
			unsigned char tid, unsigned char rid);

/* FUNCTION:	spe_mfc_read_tag_status_all(speid, mask);
 *              spe_mfc_read_tag_status_any(speid, mask);
 *              spe_mfc_read_tag_status_immediate(speid, mask);
 *
 * read the mfc tag status register.
 */

/**
 * The spe_mfc_read_tag_status_all function suspends execution until all DMA 
 * commands in the tag groups enabled by the mask parameter have no outstanding 
 * DMAs in the proxy command queue of the SPE thread specified by speid. The 
 * masked tag status is returned.
 * 
 * @param speid Specifies the SPE thread whose proxy command queue status is to be 
 * read.
 * @param mask Specifies the tag groups to be included in the query or wait operation.
 * @return On success, returns the current tag status. On failure, -1 is returned.
 */
extern int spe_mfc_read_tag_status_all(speid_t speid, unsigned int mask);

/**
 * The spe_mfc_read_tag_status_any function suspends execution until any DMA 
 * commands in the tag groups enabled by the mask parameter have no outstanding 
 * DMAs in the proxy command queue of the SPE thread specified by speid. The masked 
 * tag status is returned.
 * 
 * @param speid Specifies the SPE thread whose proxy command queue status is to be 
 * read.
 * @param mask Specifies the tag groups to be included in the query or wait operation.
 * @return On success, returns the current tag status. On failure, -1 is returned.
 */
extern int spe_mfc_read_tag_status_any(speid_t speid, unsigned int mask);

/**
 * The spe_mfc_read_tag_status_immediate function returns the tag status for the 
 * tag groups specified by the mask parameter for the proxy command queue of the 
 * SPE thread specified by the speid.
 * 
 * @param speid Specifies the SPE thread whose proxy command queue status is to be 
 * read.
 * @param mask Specifies the tag groups to be included in the query or wait operation.
 * @return On success, returns the current tag status. On failure, -1 is returned.
 */
extern int spe_mfc_read_tag_status_immediate(speid_t speid, unsigned int mask);

/**
 * 
 * C Specification
 * #include <libspe.h>
 * void *spe_get_ps_area (speid_t speid, enum ps_area area)
 * Description
 * The spe_get_ps_area function returns a pointer to the problem state area 
 * specified by ps_area for the SPE thread indicated by speid. In order to 
 * obtain a problem state area pointer the specified SPE thread must have been 
 * created with the SPE_MAP_PS flag set with sufficient privileges.
 * The problem state pointer can be used to directly access problem state 
 * features without having to make library system calls. Problem state features 
 * include multi-source synchronization, proxy DMAs, mailboxes, and signal notifiers. 
 * In addition, these pointers, along with local store pointers (see spe_get_ls), 
 * can be used to perform SPE to SPE communications via mailboxes, DMA’s and signal
 *  notification.
 * 
 * @param speid The identifier of a specific SPE thread.
 * 
 * @param area The problem state area pointer to be granted access and returned. 
 * Possible problem state areas include:\n
 * - SPE_MSSYNC_AREA Return a pointer to the specified SPE’s MFC multisource 
 * synchronization register problem state area as defined by the following structure: 
 * typedef struct spe_mssync_area {\n
 * unsigned int MFC_MSSync;\n
 * } spe_mssync_area_t;\n
 * \n
 * - SPE_MFC_COMMAND_AREA Return a pointer to the specified SPE’s MFC command parameter 
 * and command queue control area as defined by the following structure:\n
 * typedef struct spe_mfc_command_area {\n
 * unsigned char reserved_0_3[4];\n
 * unsigned int MFC_LSA;\n
 * unsigned int MFC_EAH;\n
 * unsigned int MFC_EAL;\n
 * unsigned int MFC_Size_Tag;\n
 * union {\n
 * unsigned int MFC_ClassID_CMD;\n
 * unsigned int MFC_CMDStatus;\n
 * };\n
 * unsigned char reserved_18_103[236];\n
 * unsigned int MFC_QStatus;\n
 * unsigned char reserved_108_203[252];\n
 * unsigned int Prxy_QueryType;\n
 * unsigned char reserved_208_21B[20];\n
 * unsigned int Prxy_QueryMask;\n
 * unsigned char reserved_220_22B[12];\n
 * unsigned int Prxy_TagStatus;\n
 * } spe_mfc_command_area_t;\n
 * \n
 * Note: The MFC_EAH and MFC_EAL registers can be simultaneously written using a 64-bit 
 * store. Likewise, MFC_Size_Tag and MFC_ClassID_CMD registers can be simultaneously 
 * written using a 64-bit store.
 * - SPE_CONTROL_AREA Return a pointer to the specified SPE’s SPU control area as 
 * defined by the following structure:\n
 * typedef struct spe_spu_control_area {\n
 * unsigned char reserved_0_3[4];\n
 * unsigned int SPU_Out_Mbox;\n
 * unsigned char reserved_8_B[4];\n
 * unsigned int SPU_In_Mbox;\n
 * unsigned char reserved_10_13[4];\n
 * unsigned int SPU_Mbox_Stat;\n
 * unsigned char reserved_18_1B[4];\n
 * unsigned int SPU_RunCntl;\n
 * unsigned char reserved_20_23[4];\n
 * unsigned int SPU_Status;\n
 * unsigned char reserved_28_33[12];\n
 * unsigned int SPU_NPC;\n
 * } spe_spu_control_area_t;\n
 * Note: Explicit programmer manipulation of the SPU run control is highly discouraged.
 * \n
 * - SPE_SIG_NOTIFY_1_AREA Return a pointer to the specified SPE’s signal 
 * notification area 1 as defined by the following structure:
 * \n
 * typedef struct spe_sig_notify_1_area {\n
 * unsigned char reserved_0_B[12];\n
 * unsigned int SPU_Sig_Notify_1;\n
 * } spe_sig_notify_1_area_t;\n
 * \n
 * - SPE_SIG_NOTIFY_2_AREA Return a pointer to the specified SPE’s signal notification 
 * area 2 as defined by the following structure:
 * typedef struct spe_sig_notify_2_area {\n
 * unsigned char reserved_0_B[12];\n
 * unsigned int SPU_Sig_Notify_2;\n
 * } spe_sig_notify_2_area_t;\n
 * \n
 * 
 * @return On success, a non-NULL pointer to the requested problem state area 
 * is returned. On failure, NULL is returned and errno is set appropriately.
 * 
 * @retval EACCES Permission for direct access to the specified problem state 
 * area is denied or the SPE thread was not created with memory-mapped problem 
 * state access.
 * @retval EINVAL The specified problem state area is invalid.
 * @retval ENOSYS Access to the specified problem area for the specified SPE 
 * thread is not supported by the operating system.
 * @retval ESRCH The specified SPE thread could not be found.
 * @retval EIO I/O error 
 * 
 * @sa spe_create_thread, spe_get_ls
 */
extern void *spe_get_ps_area (speid_t speid, enum ps_area area);

/**
 * The spe_set_app_data function associates application specific data with an 
 * SPE thread. Any association with previously associated data is lost on success. 
 * On failure, any previous association is unchanged. 
 *
 * @param speid The identifier of the SPE thread.
 *
 * @param data Application specific data to associate with the SPE thread.
 *
 * @return On success, spe_set_app_data and spe_get_app_data return 0. On failure, 
 * -1 is returned and errno is set appropriately.
 * Possible errors include:
 * @retval EFAULT The address of p_data was invalid.
 * @retval ESRCH The specified SPE thread could not be found.
 */
extern int spe_set_app_data( speid_t speid, void* data);

/**
 * The spe_get_app_data function returns the application specific data associated 
 * with the specified SPE thread as set by spe_set_app_data. If no application 
 * data has been associated with the specified thread, then the memory pointed to 
 * by p_data is set to NULL.
 *
 * @param speid The identifier of the SPE thread.
 *
 * @param p_data A pointer to memory to receive the associated data.
 *
 * @return On success, spe_set_app_data and spe_get_app_data return 0. On failure, 
 * -1 is returned and errno is set appropriately.
 * Possible errors include:
 * @retval EFAULT The address of p_data was invalid.
 * @retval ESRCH The specified SPE thread could not be found.
 */
extern int spe_get_app_data( speid_t speid, void** p_data);


/* INOFFICIAL FUNCTION:	__spe_get_context_fd(speid)
 *
 * returns the file descriptor for the spe context.
 */
extern int __spe_get_context_fd(speid_t speid);

#ifdef __cplusplus
}
#endif

#endif

