/**
 * \file InsetMathBox.cpp
 * This file is part of LyX, the document processor.
 * Licence details can be found in the file COPYING.
 *
 * \author André Pönitz
 * \author Ling Li (InsetMathMakebox)
 *
 * Full author contact details are available in file CREDITS.
 */

#include <config.h>

#include "InsetMathBox.h"

#include "LaTeXFeatures.h"
#include "MathData.h"
#include "MathStream.h"
#include "MathSupport.h"
#include "MetricsInfo.h"

#include "support/gettext.h"
#include "support/lstrings.h"

#include "frontends/Painter.h"

#include <algorithm>
#include <iostream>
#include <ostream>

using namespace lyx::support;

namespace lyx {

/////////////////////////////////////////////////////////////////////
//
// InsetMathBox
//
/////////////////////////////////////////////////////////////////////

InsetMathBox::InsetMathBox(Buffer * buf, docstring const & name)
	: InsetMathNest(buf, 1), name_(name)
{}


void InsetMathBox::write(TeXMathStream & os) const
{
	ModeSpecifier specifier(os, TEXT_MODE);
	os << '\\' << name_ << '{' << cell(0) << '}';
}


void InsetMathBox::normalize(NormalStream & os) const
{
	os << '[' << name_ << ' ';
	//text_->write(buffer(), os);
	os << "] ";
}


namespace {
void splitAndWrapInMText(MathMLStream & ms, MathData const & cell,
						 const std::string & attributes)
{
	// First, generate the inset into a string of its own.
	docstring inset_contents;
	{
		odocstringstream ostmp;
		MathMLStream mstmp(ostmp, ms.xmlns());

		SetMode textmode(mstmp, true);
		mstmp << cell;

		inset_contents = ostmp.str();
	}

	// No tags are allowed within <m:mtext>: split the string if there are tags.
	std::vector<docstring> parts;
	while (true) {
		std::size_t angle_pos = inset_contents.find('<');
		if (angle_pos == docstring::npos)
			break;

		// String structure:
		// - prefix: pure text, no tag
		// - tag to split: something like <m:mn>1</m:mn> or more complicated
		//   (like nested tags), with or without name space
		// - rest to be taken care of in the next iteration

		// Push the part before the tag.
		parts.emplace_back(inset_contents.substr(0, angle_pos));
		inset_contents = inset_contents.substr(angle_pos);
		// Now, inset_contents starts with the tag to isolate, so that
		//     inset_contents[0] == '<'

		// Push the tag, up to its end. Process: find the tag name (either
		// before > or the first attribute of the tag), then the matching end
		// tag, then proceed with pushing.
		const std::size_t tag_name_end =
				std::min(inset_contents.find(' ', 1), inset_contents.find('>', 1));
		const std::size_t tag_name_length = tag_name_end - 1;
		const docstring tag_name = inset_contents.substr(1, tag_name_length);

		const std::size_t end_tag_start =
				inset_contents.find(tag_name, tag_name_end + 1);
		const std::size_t end_tag = inset_contents.find('>', end_tag_start);

		parts.emplace_back(inset_contents.substr(0, end_tag + 1));
		inset_contents = inset_contents.substr(end_tag + 1);
	}
	parts.emplace_back(inset_contents);

	// Finally, output the complete inset: escape the test in <m:mtext>, leave
	// the other tags untouched.
	ms << MTag("mrow", attributes);
	for (std::size_t i = 0; i < parts.size(); i += 2) {
		ms << MTag("mtext")
		   << parts[i]
		   << ETag("mtext");
		if (parts.size() > i + 1)
			ms << parts[i + 1];
	}
	ms << ETag("mrow");
}
}


void InsetMathBox::mathmlize(MathMLStream & ms) const
{
	// FIXME XHTML
	// Need to do something special for tags here.
	// Probably will have to involve deferring them, which
	// means returning something from this routine.
	splitAndWrapInMText(ms, cell(0), "class='mathbox'");
}


void InsetMathBox::htmlize(HtmlStream & ms) const
{
	SetHTMLMode textmode(ms, true);
	ms << MTag("span", "class='mathbox'")
	   << cell(0)
	   << ETag("span");
}


void InsetMathBox::metrics(MetricsInfo & mi, Dimension & dim) const
{
	Changer dummy = mi.base.changeFontSet("text");
	cell(0).metrics(mi, dim);
}


void InsetMathBox::draw(PainterInfo & pi, int x, int y) const
{
	Changer dummy = pi.base.changeFontSet("text");
	cell(0).draw(pi, x, y);
}


void InsetMathBox::infoize(odocstream & os) const
{
	os << bformat(_("Box: %1$s"), name_);
}


void InsetMathBox::validate(LaTeXFeatures & features) const
{
	// FIXME XHTML
	// It'd be better to be able to get this from an InsetLayout, but at present
	// InsetLayouts do not seem really to work for things that aren't InsetTexts.
	if (features.runparams().math_flavor == OutputParams::MathAsMathML)
		features.addCSSSnippet("mtext.mathbox { font-style: normal; }");
	else if (features.runparams().math_flavor == OutputParams::MathAsHTML)
		features.addCSSSnippet("span.mathbox { font-style: normal; }");

	if (name_ == "tag" || name_ == "tag*")
		features.require("amsmath");

	InsetMathNest::validate(features);
}



/////////////////////////////////////////////////////////////////////
//
// InsetMathFBox
//
/////////////////////////////////////////////////////////////////////


InsetMathFBox::InsetMathFBox(Buffer * buf)
	: InsetMathNest(buf, 1)
{}


void InsetMathFBox::metrics(MetricsInfo & mi, Dimension & dim) const
{
	Changer dummy = mi.base.changeFontSet("text");
	cell(0).metrics(mi, dim);
	// 1 pixel space, 1 frame, 1 space
	dim.wid += 2 * 3;
	dim.asc += 3;
	dim.des += 3;
}


void InsetMathFBox::draw(PainterInfo & pi, int x, int y) const
{
	Dimension const dim = dimension(*pi.base.bv);
	pi.pain.rectangle(x + 1, y - dim.ascent() + 1,
		dim.width() - 2, dim.height() - 2, Color_foreground);
	Changer dummy = pi.base.changeFontSet("text");
	cell(0).draw(pi, x + 3, y);
}


void InsetMathFBox::write(TeXMathStream & os) const
{
	ModeSpecifier specifier(os, TEXT_MODE);
	os << "\\fbox{" << cell(0) << '}';
}


void InsetMathFBox::normalize(NormalStream & os) const
{
	os << "[fbox " << cell(0) << ']';
}


void InsetMathFBox::mathmlize(MathMLStream & ms) const
{
	splitAndWrapInMText(ms, cell(0), "class='fbox'");
}


void InsetMathFBox::htmlize(HtmlStream & ms) const
{
	SetHTMLMode textmode(ms, true);
	ms << MTag("span", "class='fbox'")
	   << cell(0)
	   << ETag("span");
}


void InsetMathFBox::infoize(odocstream & os) const
{
	os << "FBox: ";
}


void InsetMathFBox::validate(LaTeXFeatures & features) const
{
	// FIXME XHTML
	// It'd be better to be able to get this from an InsetLayout, but at present
	// InsetLayouts do not seem really to work for things that aren't InsetTexts.
	if (features.runparams().math_flavor == OutputParams::MathAsMathML)
		features.addCSSSnippet(
			"mtext.fbox { border: 1px solid black; font-style: normal; padding: 0.5ex; }");
	else if (features.runparams().math_flavor == OutputParams::MathAsHTML)
		features.addCSSSnippet(
			"span.fbox { border: 1px solid black; font-style: normal; padding: 0.5ex; }");

	cell(0).validate(features);
	InsetMathNest::validate(features);
}



/////////////////////////////////////////////////////////////////////
//
// InsetMathMakebox
//
/////////////////////////////////////////////////////////////////////


InsetMathMakebox::InsetMathMakebox(Buffer * buf, bool framebox)
	: InsetMathNest(buf, 3), framebox_(framebox)
{}


void InsetMathMakebox::metrics(MetricsInfo & mi, Dimension & dim) const
{
	Changer dummy = mi.base.changeFontSet("text");

	Dimension wdim;
	static docstring bracket = from_ascii("[");
	metricsStrRedBlack(mi, wdim, bracket);
	int w = wdim.wid;

	Dimension dim0;
	Dimension dim1;
	Dimension dim2;
	cell(0).metrics(mi, dim0);
	cell(1).metrics(mi, dim1);
	cell(2).metrics(mi, dim2);

	dim.wid = w + dim0.wid + w + w + dim1.wid + w + 2 + dim2.wid;
	dim.asc = std::max(std::max(wdim.asc, dim0.asc), std::max(dim1.asc, dim2.asc));
	dim.des = std::max(std::max(wdim.des, dim0.des), std::max(dim1.des, dim2.des));

	if (framebox_) {
		dim.wid += 4;
		dim.asc += 3;
		dim.des += 2;
	} else {
		dim.asc += 1;
		dim.des += 1;
	}
}


void InsetMathMakebox::draw(PainterInfo & pi, int x, int y) const
{
	Changer dummy = pi.base.changeFontSet("text");
	BufferView const & bv = *pi.base.bv;
	int w = mathed_char_width(pi.base.font, '[');

	if (framebox_) {
		Dimension const dim = dimension(*pi.base.bv);
		pi.pain.rectangle(x + 1, y - dim.ascent() + 1,
				  dim.width() - 2, dim.height() - 2, Color_foreground);
		x += 2;
	}

	drawStrBlack(pi, x, y, from_ascii("["));
	x += w;
	cell(0).draw(pi, x, y);
	x += cell(0).dimension(bv).wid;
	drawStrBlack(pi, x, y, from_ascii("]"));
	x += w;

	drawStrBlack(pi, x, y, from_ascii("["));
	x += w;
	cell(1).draw(pi, x, y);
	x += cell(1).dimension(bv).wid;
	drawStrBlack(pi, x, y, from_ascii("]"));
	x += w + 2;

	cell(2).draw(pi, x, y);
}


void InsetMathMakebox::write(TeXMathStream & os) const
{
	ModeSpecifier specifier(os, TEXT_MODE);
	os << (framebox_ ? "\\framebox" : "\\makebox");
	if (!cell(0).empty() || !os.latex()) {
		os << '[' << cell(0) << ']';
		if (!cell(1).empty() || !os.latex())
			os << '[' << cell(1) << ']';
	}
	os << '{' << cell(2) << '}';
}


void InsetMathMakebox::normalize(NormalStream & os) const
{
	os << (framebox_ ? "[framebox " : "[makebox ")
	   << cell(0) << ' ' << cell(1) << ' ' << cell(2) << ']';
}


void InsetMathMakebox::infoize(odocstream & os) const
{
	os << (framebox_ ? "Framebox" : "Makebox")
	   << " (width: " << cell(0)
	   << " pos: " << cell(1) << ")";
}


void InsetMathMakebox::mathmlize(MathMLStream & ms) const
{
	// FIXME We could do something with the other arguments.
	std::string const cssclass = framebox_ ? "framebox" : "makebox";
	splitAndWrapInMText(ms, cell(2), "class='" + cssclass + "'");
}


void InsetMathMakebox::htmlize(HtmlStream & ms) const
{
	// FIXME We could do something with the other arguments.
	SetHTMLMode textmode(ms, true);
	std::string const cssclass = framebox_ ? "framebox" : "makebox";
	ms << MTag("span", "class='" + cssclass + "'")
	   << cell(2)
	   << ETag("span");
}


void InsetMathMakebox::validate(LaTeXFeatures & features) const
{
	// FIXME XHTML
	// It'd be better to be able to get this from an InsetLayout, but at present
	// InsetLayouts do not seem really to work for things that aren't InsetTexts.
	if (features.runparams().math_flavor == OutputParams::MathAsMathML)
		features.addCSSSnippet("mtext.framebox { border: 1px solid black; }");
	else if (features.runparams().math_flavor == OutputParams::MathAsHTML)
		features.addCSSSnippet("span.framebox { border: 1px solid black; }");
	InsetMathNest::validate(features);
}


/////////////////////////////////////////////////////////////////////
//
// InsetMathBoxed
//
/////////////////////////////////////////////////////////////////////

InsetMathBoxed::InsetMathBoxed(Buffer * buf)
	: InsetMathNest(buf, 1)
{}


void InsetMathBoxed::metrics(MetricsInfo & mi, Dimension & dim) const
{
	cell(0).metrics(mi, dim);
	// 1 pixel space, 1 frame, 1 space
	dim.wid += 2 * 3;
	dim.asc += 3;
	dim.des += 3;
}


void InsetMathBoxed::draw(PainterInfo & pi, int x, int y) const
{
	Dimension const dim = dimension(*pi.base.bv);
	pi.pain.rectangle(x + 1, y - dim.ascent() + 1,
		dim.width() - 2, dim.height() - 2, Color_foreground);
	cell(0).draw(pi, x + 3, y);
}


void InsetMathBoxed::write(TeXMathStream & os) const
{
	ModeSpecifier specifier(os, MATH_MODE);
	os << "\\boxed{" << cell(0) << '}';
}


void InsetMathBoxed::normalize(NormalStream & os) const
{
	os << "[boxed " << cell(0) << ']';
}


void InsetMathBoxed::infoize(odocstream & os) const
{
	os << "Boxed: ";
}


void InsetMathBoxed::mathmlize(MathMLStream & ms) const
{
	splitAndWrapInMText(ms, cell(0), "class='boxed'");
}


void InsetMathBoxed::htmlize(HtmlStream & ms) const
{
	ms << MTag("span", "class='boxed'")
	   << cell(0)
	   << ETag("span");
}


void InsetMathBoxed::validate(LaTeXFeatures & features) const
{
	features.require("amsmath");

	// FIXME XHTML
	// It'd be better to be able to get this from an InsetLayout, but at present
	// InsetLayouts do not seem really to work for things that aren't InsetTexts.
	if (features.runparams().math_flavor == OutputParams::MathAsMathML)
		features.addCSSSnippet("mtext.boxed { border: 1px solid black; }");
	else if (features.runparams().math_flavor == OutputParams::MathAsHTML)
		features.addCSSSnippet("span.boxed { border: 1px solid black; }");

	InsetMathNest::validate(features);
}


} // namespace lyx
