/*
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010. 
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.  
 * 
 * Licensed under the Apache License, Version 2.0 (the "License"); 
 * you may not use this file except in compliance with the License. 
 * You may obtain a copy of the License at 
 * 
 *     http://www.apache.org/licenses/LICENSE-2.0 
 * 
 * Unless required by applicable law or agreed to in writing, software 
 * distributed under the License is distributed on an "AS IS" BASIS, 
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. 
 * See the License for the specific language governing permissions and 
 * limitations under the License.
 *
 *
 *  Authors:
 *  2009-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     Mischa Sall\'e <msalle@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl> 
 *
 *  2007-2009
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 *  2003-2007
 *     Martijn Steenbakkers <martijn@nikhef.nl>
 *     Gerben Venekamp <venekamp@nikhef.nl>
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     David Groep <davidg@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *
 */


/*!
    \file   _lcmaps_pluginmanager.h
    \brief  API of the PluginManager
    \author Martijn Steenbakkers for the EU DataGrid.

    This header contains the declarations of the LCMAPS library functions:
    -# lcmaps_startPluginManager(): start the PluginManager --> load plugins, start evaluation manager
    -# lcmaps_runPluginManager(): run the PluginManager --> run evaluation manager --> run plugins
    -# lcmaps_stopPluginManager(): stop the PluginManager
    -# lcmaps_runPlugin(): run the specified plugin. (used by Evaluation Manager)
    -# lcmaps_resetCredentialData(): reset the Credential Data. (used by Evaluation Manager)
*/

#ifndef _LCMAPS_PLUGINMANAGER_H
#define _LCMAPS_PLUGINMANAGER_H

/******************************************************************************
                             Include header files
******************************************************************************/
#include "lcmaps_types.h"
#include "grid_credential_handling/_lcmaps_credential.h"

/******************************************************************************
 *                 Define constants
 *****************************************************************************/
#define LCMAPS_NORMAL_MODE (unsigned int) 0
#define LCMAPS_VERIFICATION_MODE (unsigned int) 1

/******************************************************************************
 *                 Module definition
 *****************************************************************************/

/******************************************************************************
Function:   lcmaps_startPluginManager
Description:
    Initialize PluginManager module: 
    read from LCMAPS config file, the plugins to be loaded
    Initialize evaluation manager (get policy)

Parameters:
Returns:
    0: succes
    1: failure
******************************************************************************/
int lcmaps_startPluginManager(void);

/******************************************************************************
Function:   lcmaps_stopPluginManager
Description:
    Terminate PluginManager module.

Parameters:
Returns:
    0: termination succeeded
    1: termination failed
******************************************************************************/
int lcmaps_stopPluginManager(void);

/******************************************************************************
Function:   lcmaps_runPluginManager
Description:
    Run PluginManager:
    Contact Evaluation Manager with list of possible policies --> runs plugins
    The plugin manager can be run in verification mode or normal mode, depending on
    "verification switch".

Parameters:
    request     : RSL request
    lcmaps_cred : user credential
    npols       : number of policies to be considered for evaluation
    policynames : the names of the policies to be considered for evaluation
    mode        : Mode of running lcmaps: LCMAPS_NORMAL_MODE or
                  LCMAPS_VERIFICATION_MODE

Returns:
    0: user mapping or mapping verification succeeded
    1: user mapping or mapping verification failed
******************************************************************************/
int lcmaps_runPluginManager(
        lcmaps_request_t request,
        lcmaps_cred_id_t lcmaps_cred,
	char *requested_username,
        int npols,
        char ** policynames,
        unsigned int mode
);

/******************************************************************************
Function:   runPlugin
Description:
    Run a plugin for the Evaluation Manager
    the result (succes or not will be passed to the Evaluation Manager)

Parameters:
    pluginname: the name of the plugin module
Returns:
    0: plugin run succeeded
    1: plugin run failed
******************************************************************************/
int lcmaps_runPlugin(
        const char * pluginname
);

/******************************************************************************
Function:   lcmaps_resetCredentialData
Description:
    Reset the Credential data.
    Wrapper around lcmaps_cleanCredentialData() to be used
    by the Evaluation Manager, before a new policy is
    evaluated

Parameters:
    none
Returns:
    the return value of lcmaps_cleanCredentialData()
******************************************************************************/
int lcmaps_resetCredentialData(void);

#endif /* _LCMAPS_PLUGINMANAGER_H */

/******************************************************************************
CVS Information:
    $Source: /srv/home/dennisvd/svn/mw-security/lcmaps/src/pluginmanager/_lcmaps_pluginmanager.h,v $
    $Date: 2014-03-13 11:56:06 +0100 (Thu, 13 Mar 2014) $
    $Revision: 17647 $
    $Author: msalle $
******************************************************************************/
