// SPDX-License-Identifier: GPL-2.0-or-later
// SPDX-FileCopyrightText: 2011 Konrad Twardowski

#pragma once

#include "config.h"

#include <QDialog>
#include <QDialogButtonBox>
#include <QVBoxLayout>

class UDialog: public QDialog {
public:
	enum class WindowSize { FIXED, LARGE };

	explicit UDialog(QWidget *parent, const QString &windowTitle, const bool simple);
	virtual ~UDialog() = default;
	QPushButton *acceptButton() { return m_acceptButton; }
	QDialogButtonBox *buttonBox() { return m_dialogButtonBox; }
	QVBoxLayout *mainLayout() { return m_mainLayout; }
	QVBoxLayout *rootLayout() { return m_rootLayout; }

	void moveToCenterOfScreen();
	void setWindowSize(const WindowSize windowSize);

	// messages

	static void error(QWidget *parent, const QString &text, const Qt::TextFormat format = Qt::AutoText);
	static void info(QWidget *parent, const QString &text, const Qt::TextFormat format = Qt::AutoText);
	static void warning(QWidget *parent, const QString &text, const Qt::TextFormat format = Qt::AutoText);

	// misc

	#ifdef KS_PURE_QT
	static void about(QWidget *parent);
	#endif // KS_PURE_QT
	static void largeWidget(QWidget *parent, QWidget *widget, const QString &windowTitle, const QList<QPushButton *> &actionButtonList);
	static void plainText(QWidget *parent, const QString &text, const QString &windowTitle);
	static void systemInfo(QWidget *parent);

private:
	Q_DISABLE_COPY(UDialog)
	QDialogButtonBox *m_dialogButtonBox;
	QPushButton *m_acceptButton;
	QVBoxLayout *m_mainLayout;
	QVBoxLayout *m_rootLayout;
};

class UMessageBuilder final {
public:
	enum class Type { Error, Info, Question, Warning };
	explicit UMessageBuilder(const Type value)
		: m_type(value) {
	}
	virtual ~UMessageBuilder() = default;

	bool exec(QWidget *parent);
	void showKey(const QString &key) {
		// NOTE: the group name is compatible with KF
		m_showVar = std::make_shared<Var>("Notification Messages", key, true);
	}
	void text(const QString &value, const Qt::TextFormat format) {
		m_text = value;
		m_textFormat = format;
	}

	void cancelDefault(const bool value) { m_cancelDefault = value; }
	void okText(const QString &value) { m_okText = value; }
	void icon(const QIcon &value) { m_icon = value; }
	void markdown(const QString &value) { text(value, Qt::MarkdownText); }
	void plainText(const QString &value) { text(value, Qt::PlainText); }
	void title(const QString &value) { m_title = value; }
private:
	bool m_cancelDefault = false;
	QIcon m_icon = QIcon();
	QString m_okText = QString();
	QString m_text = QString();
	QString m_title = QString();
	Qt::TextFormat m_textFormat = Qt::AutoText;
	std::shared_ptr<Var> m_showVar;
	Type m_type;
};
