# Copyright(c) 1986 Association of Universities for Research in Astronomy Inc.

include	<syserr.h>
include	<error.h>
include	<mach.h>
include	"mwcs.h"

.help WFPC
.nf -------------------------------------------------------------------------
WFPC -- WCS function driver for polynomial corrections.

These routines apply a polynomial correction to the intermediate
coordinates from up to three coupled axes plus a radial variable.  The
wtype for the axes is "pc" and the axtypes are the CTYPE strings for the
final world coordinate projection.  A secondary MWCS transformation is set
up using the axtype values to apply the world coordinate projection.  Any
of the MWCS function drivers may be used.

Driver routines:

	FN_INIT		wf_pc_init (fc, dir)
	FN_DESTROY	wf_pc_destroy (fc)
	FN_FWD		wf_pc_fwd (fc, v1, v2)
	FN_INV		wf_pc_inv (fc, v1, v2)

.endhelp --------------------------------------------------------------------

define	NAXMAX		3			 # Max axes
define	NDIMMAX		4			 # Max poly dimension

# Driver specific fields of function call (FC) descriptor.
define	FC_MW1		Memi[$1+FCU]		 # Secondary MWCS pointer
define	FC_CT1		Memi[$1+FCU+1]		 # Secondary CT pointer
define	FC_NAX		Memi[$1+FCU+2]		 # Number of axes
define	FC_NDIM		Memi[$1+FCU+3]		 # Polynomial dimensionality
define	FC_NPOLY	Memi[$1+FCU+4]		 # Number of defined polynomials
define	FC_LAST		Memd[P2D($1+FCU+6)+$2-1] # Last inverted coordinate
define	FC_P		Memi[$1+FCU+6+2*NAXMAX+$2-1]	 # Polynomial structures

# Polynomial structure.
define	P_LEN		(2*$1+4*NAXMAX+NDIMMAX)	 # Length of structure
define	P_N		Memd[P2D($1)+$2-1]	 # Normalizations
define	P_R		Memd[P2D($1+2*NAXMAX)+$2-1] # Radius coefficients
define	P_O		Memi[$1+4*NAXMAX+$2-1]	 # Orders
define	P_C		P2D($1+4*NAXMAX+NDIMMAX) # Coefficients



# WF_PC_INIT -- Initialize forward/inverse transform with polynomial correction.

procedure wf_pc_init (fc, dir)

pointer	fc			#I pointer to FC descriptor
int	dir			#I direction of transform

int	i, j, k, nax, ndim, ax, axes[NAXMAX], npc, strncmp()
pointer	mw, ct, wp, wv, pc, mw1, ct1, p, ptr, mw_open(), wf_pc_sctran()
pointer	sp, axtype, wtype, r, w, cd
bool	streq()

begin
	call smark (sp)
	call salloc (axtype, SZ_LINE, TY_CHAR)
	call salloc (wtype, SZ_LINE, TY_CHAR)
	call salloc (r, NAXMAX, TY_DOUBLE)
	call salloc (w, NAXMAX, TY_DOUBLE)
	call salloc (cd, 9, TY_DOUBLE)

	ct = FC_CT(fc)
	mw = CT_MW(ct)
	wp = FC_WCS(fc)

	nax = WCS_NDIM(wp)
	if (nax > NAXMAX)
	    call error (1, "WCS dimensionality is too large")

	# Open the secondary MWCS.
	mw1 = mw_open (NULL, nax)
	call mw_newsystem (mw1, "image", nax)

	# Set axes.  Identify celestial axes to be set later.
	j = 0
	Memc[wtype] = EOS
	do i = 1, nax {
	    ax = CT_AXIS(ct,FC_AXIS(fc,i))
	    ifnoerr (call mw_gwattrs (mw, ax, "axtype", Memc[axtype],SZ_LINE)) {
		if (strncmp (Memc[axtype], "ra--", 4) == 0 ||
		    strncmp (Memc[axtype], "dec-", 4) == 0 ||
		    strncmp (Memc[axtype+1], "lon", 3) == 0 ||
		    strncmp (Memc[axtype+1], "lat", 3) == 0) {
		    j = j + 1
		    axes[j] = ax
		    if (Memc[wtype] == EOS)
			call strcpy (Memc[axtype+5], Memc[wtype], 4)
		    else if (!streq (Memc[axtype+5], Memc[wtype]))
			call error (1, "Unmatched celestial axes")
		    if (Memc[axtype+2] == '-')
			Memc[axtype+2] = EOS
		    else if (Memc[axtype+3] == '-')
			Memc[axtype+3] = EOS
		    else
			Memc[axtype+4] = EOS
		    call mw_swattrs (mw1, ax, "axtype", Memc[axtype])
		} else
		    call mw_swtype (mw1, ax, 1, Memc[axtype], "")
	    }
	}

	# Set coupled celestial axes.
	if (j > 0) {
	    if (j != 2)
		call error (1, "Unmatched celestial axes")
	    call mw_swtype (mw1, axes, j, Memc[wtype], "")
	}

	# Set reference coordinate.
	wv = MI_DBUF(mw) + WCS_W(wp) - 1
	call mw_gwtermd (mw1, Memd[r], Memd[w], Memd[cd], nax)
	call mw_swtermd (mw1, Memd[r], Memd[wv], Memd[cd], nax)

	# Set transformation.
	if (dir == FORWARD)
	   ct1 = wf_pc_sctran (mw1, "physical", "world", 0)
	else
	   ct1 = wf_pc_sctran (mw1, "world", "physical", 0)

	FC_NAX(fc) = nax
	FC_MW1(fc) = mw1
	FC_CT1(fc) = ct1

	# Set polynomial structures.  The polynomial structure is NULL if
	# there are no coefficients.  The number of non-null polynomial
	# structures is recorded for later efficiency.

	ndim = nax + 1
	FC_NDIM(fc) = ndim
	FC_NPOLY(fc) = 0
	do i = 1, nax {
	    FC_P(fc,i) = NULL
	    FC_LAST(fc,i) = 0D0

	    npc = WCS_NPC(wp,i)
	    pc = WCS_PC(wp,i)
	    if (npc > ndim+2*nax && pc != NULL) {
		pc = MI_DBUF(mw) + pc - 1

		# Compute number of coefficients including default zeros.
		k = 1
		ptr = pc
		do j = 1, ndim {
		    k = k * (1 + nint (Memd[ptr]))
		    ptr = ptr + 1
		}

		# Allocate and set polynomial structure.
		if (k > 0) {
		    call calloc (FC_P(fc,i), P_LEN(k), TY_STRUCT)
		    FC_NPOLY(fc) = FC_NPOLY(fc) + 1

		    p = FC_P(fc,i)
		    ptr = pc
		    do j = 1, ndim {
			P_O(p,j) = nint (Memd[ptr])
			ptr = ptr + 1
		    }
		    do j = 1, nax {
			P_N(p,j) = 1D0 + Memd[ptr]
			ptr = ptr + 1
		    }
		    do j = 1, nax {
			P_R(p,j) = Memd[ptr]
			ptr = ptr + 1
		    }
		    call amovd (Memd[ptr], Memd[P_C(p)], npc-(ptr-pc))
		    FC_P(fc,i) = p
		}
	    }
	}
end


# WF_PC_DESTROY -- Destroy memory associated with function driver.

procedure wf_pc_destroy (fc)

pointer	fc			#I pointer to FC descriptor

int	i

begin
	do i = 1, FC_NAX(fc)
	    call mfree (FC_P(fc,i), TY_STRUCT)
	if (FC_MW1(fc) != NULL)
	    call wf_pc_close (FC_MW1(fc))
end


# WF_PC_FWD -- Forward transformation with polynomial correction.

procedure wf_pc_fwd (fc, p, w)

pointer	fc			#I pointer to FC descriptor
double	p[ARB]			#I physical coordinates
double	w[ARB]			#O world coordinates

double	p1[NAXMAX]

begin
	# Apply polynomial correction to each axis and then the projection.
	if (FC_NPOLY(fc) == 0)
	    call wf_pc_ctran (FC_CT1(fc), p, w, FC_NAX(fc))
	else {
	    call wf_pc_eval (fc, p, p1)
	    call wf_pc_ctran (FC_CT1(fc), p1, w, FC_NAX(fc))
	}
end


define	TOL	10000
define	NIT	1000
define	MAXINC	10

# WF_PC_INV -- Inverse transformation with polynomial correction.

procedure wf_pc_inv (fc, w, p)

pointer	fc			#I pointer to FC descriptor
double	w[ARB]			#I input world coordinates
double	p[ARB]			#I output physical coordinates

bool	done
int	i, j, nax, ninc
double	tol, p1[NAXMAX], p2[NAXMAX], dp[NAXMAX]
double	adp[NAXMAX], adplast[NAXMAX]
double	deriv[NAXMAX,NAXMAX], ideriv[NAXMAX,NAXMAX]

begin
	if (FC_NPOLY(fc) == 0)
	    call wf_pc_ctran (FC_CT1(fc), w, p, FC_NAX(fc))
	else {
	    nax = FC_NAX(fc)
	    call wf_pc_ctran (FC_CT1(fc), w, p1, nax)

	    ninc = -1
	    tol = TOL * EPSILOND
	    call amovkd (MAX_DOUBLE, adplast, nax)
	    do i = 1, NIT {
		do j = 1, nax
		    p[j] = FC_LAST(fc,j)
		call wf_pc_eval (fc, p, p2)
		done = true
		do j = 1, nax {
		    dp[j] = p1[j] - p2[j]
		    adp[j] = abs (dp[j])
		    if (adp[j] > tol)
			done = false
		    if (adp[j] <= adplast[j])
			ninc = -1
		}
		if (done)
		    return
		ninc = ninc + 1
		if (ninc > MAXINC)
		    break
		do j = 1, nax
		    adplast[j] = adp[j]
		call wf_pc_deriv (fc, p, deriv)
		call mw_invertd (deriv, ideriv, nax)
		call mw_vmuld (ideriv, dp, p2, nax)
		do j = 1, nax
		    FC_LAST(fc,j) = p[j] + p2[j]
	    }
	    do j = 1, nax
		p[j] = FC_LAST(fc,j)
	    call wf_pc_eval (fc, p, p2)
	    done = true
	    do j = 1, nax {
		dp[j] = p1[j] - p2[j]
		adp[j] = abs (dp[j])
		if (adp[j] > tol)
		    done = false
	    }
	    if (done)
		return
	    call error (1, "Did not converge")
	}
end


# WF_PC_EVAL -- Evaluate polynomials.

procedure wf_pc_eval (fc, x, y)

pointer	fc			#I pointer to FC descriptor
double	x[ARB]			#I input coordinates
double	y[ARB]			#O output coordinates

int	i, j, k, m, q, nax, ndim
double	r[NDIMMAX], rq, sum, xi, yj, zk
pointer	p, c

begin
	nax = FC_NAX(fc)
	ndim = FC_NDIM(fc)

	do m = 1, nax {
	    p = FC_P(fc,m)
	    if (P_O(p,ndim) > 0) {
		sum = 0
		do i = 1, nax
		    sum = sum + (P_R(p,i) * x[i] / P_N(p,i)) ** 2
		r[m] = sqrt (sum)
	    }
	}

	if (nax == 1) {
	    do m = 1, nax {
		p = FC_P(fc,m)
		if (p != NULL) {
		    c = P_C(p)
		    sum = x[m]
		    do q = 0, P_O(p,ndim) {
			rq = r[m] ** q
			do i = 0, P_O(p,1) {
			    xi = (x[1] / P_N(p,1)) ** i
			    sum = sum + Memd[c] * xi * rq
			    c = c + 1
			}
		    }
		    y[m] = sum
		} else
		    y[m] = x[m]
	    }
	} else if (nax == 2) {
	    do m = 1, nax {
		p = FC_P(fc,m)
		if (p != NULL) {
		    c = P_C(p)
		    sum = x[m]
		    do q = 0, P_O(p,ndim) {
			rq = r[m] ** q
			do j = 0, P_O(p,2) {
			    yj = rq * (x[2] / P_N(p,2)) ** j
			    do i = 0, P_O(p,1) {
				xi = (x[1] / P_N(p,1)) ** i
				sum = sum + Memd[c] * xi * yj
				c = c + 1
			    }
			}
		    }
		    y[m] = sum
		} else
		    y[m] = x[m]
	    }
	} else {
	    do m = 1, nax {
		p = FC_P(fc,m)
		if (p != NULL) {
		    c = P_C(p)
		    sum = x[m]
		    do q = 0, P_O(p,ndim) {
			rq = r[m] ** q
			do k = 0, P_O(p,3) {
			    zk = rq * (x[3] / P_N(p,2)) ** k
			    do j = 0, P_O(p,2) {
				yj = zk * (x[2] / P_N(p,2)) ** j
				do i = 0, P_O(p,1) {
				    xi = (x[1] / P_N(p,1)) ** i
				    sum = sum + Memd[c] * xi * yj
				    c = c + 1
				}
			    }
			}
		    }
		    y[m] = sum
		} else
		    y[m] = x[m]
	    }
	}
end


# WF_PC_DERIV -- Evaluate derivatives of polynomials.

procedure wf_pc_deriv (fc, x, y)

pointer	fc			#I pointer to FC descriptor
double	x[ARB]			#I input coordinates
double	y[ARB]			#O output derivatives

int	i, j, k, l, m, n, q, nax, ndim
double	r[NAXMAX], rq, drq, sum, xi, yj, zk
pointer	p, c

begin
	nax = FC_NAX(fc)
	ndim = FC_NDIM(fc)

	do m = 1, nax {
	    p = FC_P(fc,m)
	    if (P_O(p,ndim) > 0) {
		sum = 0
		do i = 1, nax
		    sum = sum + (P_R(p,i) * x[i] / P_N(p,i)) ** 2
		r[m] = sqrt (sum)
	    }
	}

        if (nax == 1) {
            y[1] = 1
            p = FC_P(fc,1)
            if (p != NULL) {
                sum = y[1]
                c = P_C(p) + 1
		do q = 0, P_O(p,ndim) {
		    if (q == 0) {
		       rq = 1
		       drq = 0
		    } else if (r[m] == 0.)
			break
		    else {
			rq = r[m] ** q
			xi = P_R(p,1) * x[1] / P_N(p,1)
			drq = q * r[m] ** (q-2) * P_R(p,1) * xi
		    }
		    do i = 0, P_O(p,1) {
			if (i == 0) {
			    if (q == 0) {
				c = c + 1
				next
			    }
			    xi = drq
			} else {
			    xi = x[1] / P_N(p,1)
			    xi = drq * xi ** i +
				 rq * i / P_N(p,1) * xi ** (i-1)
			}
			sum = sum + Memd[c] * xi
			c = c + 1
		    }
		}
                y[1] = sum
            }
	} else if (nax == 2) {
	    y[1] = 1; y[2] = 0
	    y[3] = 0; y[4] = 1
	    do m = 1, nax {
		p = FC_P(fc,m)
		if (p != NULL) {
		    l = nax * (m - 1)
		    do n = 1, nax {
			l = l + 1
			sum = y[l]
			c = P_C(p)
			do q = 0, P_O(p,ndim) {
			    if (q == 0) {
			       rq = 1
			       drq = 0
			    } else if (r[m] == 0.)
				break
			    else {
				rq = r[m] ** q
				xi = P_R(p,n) * x[n] / P_N(p,n)
				drq = q * r[m] ** (q-2) * P_R(p,n) * xi
			    }
			    do j = 0, P_O(p,2) {
				if (n == 2) {
				    if (j == 0) {
					if (q == 0) {
					    c = c + (P_O(p,1)+1)
					    next
					}
					yj = drq
				    } else {
					yj = x[2] / P_N(p,2)
					yj = drq * yj ** j +
					     rq * j / P_N(p,2) * yj ** (j-1)
				    }
				} else
				    yj = (x[2] / P_N(p,2)) ** j
				do i = 0, P_O(p,1) {
				    if (n == 1) {
					if (i == 0) {
					    if (q == 0) {
						c = c + 1
						next
					    }
					    xi = drq
					} else {
					    xi = x[1] / P_N(p,1)
					    xi = drq * xi ** i +
						 rq * i / P_N(p,1) * xi ** (i-1)
					}
				    } else
					xi = (x[1] / P_N(p,1)) ** i
				    sum = sum + Memd[c] * xi * yj
				    c = c + 1
				}
			    }
			}
			y[l] = sum
		    }
		}
	    }
	} else {
	    y[1] = 1; y[2] = 0; y[3] = 0
	    y[4] = 0; y[5] = 1; y[6] = 0
	    y[7] = 0; y[8] = 0; y[9] = 1
	    do m = 1, nax {
		p = FC_P(fc,m)
		if (p != NULL) {
		    l = nax * (m - 1)
		    do n = 1, nax {
			l = l + 1
			sum = y[l]
			c = P_C(p)
			do q = 0, P_O(p,ndim) {
			    if (q == 0) {
			       rq = 1
			       drq = 0
			    } else if (r[m] == 0.)
				break
			    else {
				rq = r[m] ** q
				xi = P_R(p,n) * x[n] / P_N(p,n)
				drq = q * r[m] ** (q-2) * P_R(p,n) * xi
			    }
			    do k = 0, P_O(p,3) {
				if (n == 3) {
				    if (k == 0) {
					if (q == 0) {
					    c = c + (P_O(p,1)+1) * (P_O(p,2)+1)
					    next
					}
					zk = drq
				    } else {
					zk = x[3] / P_N(p,3)
					zk = drq * zk ** k +
					     rq * k / P_N(p,3) * zk ** (k-1)
				    }
				} else
				    zk = (x[3] / P_N(p,3)) ** k
				do j = 0, P_O(p,2) {
				    if (n == 2) {
					if (j == 0) {
					    if (q == 0) {
						c = c + (P_O(p,1)+1)
						next
					    }
					    yj = drq
					} else {
					    yj = x[2] / P_N(p,2)
					    yj = drq * yj ** j +
						 rq * j / P_N(p,2) * yj ** (j-1)
					}
				    } else
					yj = (x[2] / P_N(p,2)) ** j
				    yj = yj * zk
				    do i = 0, P_O(p,1) {
					if (n == 1) {
					    if (i == 0) {
						if (q == 0) {
						    c = c + 1
						    next
						}
						xi = drq
					    } else {
						xi = x[1] / P_N(p,1)
						xi = drq * xi ** i +
						     rq * i/P_N(p,1) * xi**(i-1)
					    }
					} else
					    xi = (x[1] / P_N(p,1)) ** i
					sum = sum + Memd[c] * xi * yj
					c = c + 1
				    }
				}
			    }
			}
			y[l] = sum
		    }
		}
	    }
	}
end


################################################################################
# The following routines are copies of MW_SCTRAN and MW_CTRAND included
# here to avoid recursion.


# MW_SCTRAN -- Setup a coordinate transformation (CTRAN) descriptor.

pointer procedure wf_pc_sctran (mw, system1, system2, axbits)

pointer mw			#I pointer to MWCS descriptor
char	system1[ARB]		#I input coordinate system
char	system2[ARB]		#I output coordinate system
int	axbits			#I bitmap defining axes to be transformed

bool	newfunc
int	naxes, axis[MAX_DIM], wfno, fn, epa
int	i, j, k , matlen, ndata, ctlen, pdim
pointer	i_ltm, i_ltv, o_ltm, o_ltv, t_ltm, t_ltv, ltm, ltv
pointer	sp, w1, w2, ct, wf, fc, lp, ip, op, ct_r, sv_wcs

pointer	coerce()
errchk	syserr, syserrs, calloc, zcall2, mw_invertd, mw_ssystem
include	"mwcs.com"

begin
	call smark (sp)

	# Get pointers to the input and output systems.
	sv_wcs = MI_WCS(mw)
	iferr {
	    call mw_ssystem (mw, system1)
	    w1 = MI_WCS(mw)
	    call mw_ssystem (mw, system2)
	    w2 = MI_WCS(mw)
	} then {
	    MI_WCS(mw) = sv_wcs
	    call erract (EA_ERROR)
	} else
	    MI_WCS(mw) = sv_wcs

	# Get the physical axis list.  The bitflags in AXBITS define the axes
	# in the logical system; run these through the axis map (if enabled)
	# to get the list of physical axes for which the transformation is to
	# be prepared.

	call mw_gaxlist (mw, axbits, axis, naxes)

	# Allocate the CTRAN descriptor.  First we must figure out how
	# much space is required.  The space required is for the base
	# descriptor, plus additional space for the LTM and LTV, which vary
	# in size depending upon the dimensionality of the transformation.
	# The whole thing is then doubled to provide 2 versions of the
	# descriptor, providing both single and double precision versions
	# of the LTM and LTV.  Any additional storage utilized by the WCS
	# functions is separately allocated by the initialization routines
	# in the function drivers.

	matlen = naxes * naxes
	ndata = matlen + naxes
	ctlen = LEN_CTBASE + ndata * SZ_DOUBLE / SZ_STRUCT
	call calloc (ct, ctlen*2, TY_STRUCT)

	# Save a pointer to the CTRAN descriptor in the main MWCS descriptor,
	# to permit automatic deallocation at close time.

	do i = 1, MAX_CTRAN+1 {
	    if (i > MAX_CTRAN) {
		call mfree (ct, TY_STRUCT)
		call syserr (SYS_MWCTOVFL)
	    }

	    if (MI_CTRAN(mw,i) == NULL) {
		MI_CTRAN(mw,i) = ct
		break
	    }
	}

	CT_MW(ct) = mw
	CT_WCSI(ct) = w1
	CT_WCSO(ct) = w2
	CT_NDIM(ct) = naxes
	CT_R(ct) = ct + ctlen
	call amovi (axis, CT_AXIS(ct,1), naxes)
	CT_LTM(ct) = coerce (ct + LEN_CTBASE, TY_STRUCT, TY_DOUBLE)
	CT_LTV(ct) = CT_LTM(ct) + matlen

	ltm = CT_LTM(ct)
	ltv = CT_LTV(ct)

	# We also need some full-system matrix and vector buffers.
	pdim = min (WCS_NDIM(w1), WCS_NDIM(w2))
	pdim = min (MI_NDIM(mw), pdim)

	i = pdim * pdim
	call salloc (i_ltm, i, TY_DOUBLE)
	call salloc (i_ltv, pdim, TY_DOUBLE)
	call salloc (o_ltm, i, TY_DOUBLE)
	call salloc (o_ltv, pdim, TY_DOUBLE)
	call salloc (t_ltm, i, TY_DOUBLE)
	call salloc (t_ltv, pdim, TY_DOUBLE)

	# Compute the transformation.  A transformation between any two
	# world systems W1 and W2 consists of the transformation W1->P
	# from W1 to the physical system, followed by a transformation
	# P->W2 to the second world system.  The linear portions of these
	# two transformations can be combined to produce a single linear
	# transformation, and if no WCS function calls are involved at
	# either end, the entire transformation reduces to a single linear
	# transformation defined by LTM and LTV.  Note that as far as we
	# are concerned here, the special world systems "logical" and
	# "physical" are just like other world systems, except that both are
	# always linear systems.  The linear term for the logical system is
	# the MWCS Lterm; for the physical system it is the identity matrix.

	# Set up the transformation W1->P.  First we must determine if there
	# are any WCS function calls.  We do this by going ahead and compiling
	# the "in" function calls in the CTRAN descriptor.

	do i = 1, naxes {
	    wfno = WCS_AXCLASS(w1,axis[i])

	    # Skip to next axis if no WCS function is assigned to this axis.
	    if (wfno == 0)
		next

	    # Has function call for this axis already been compiled?
	    newfunc = true
	    do j = 1, CT_NCALLI(ct) {
		fc = CT_FCI(ct,j)
		do k = 1, FC_NAXES(fc)
		    if (FC_AXIS(fc,k) == i)
			newfunc = false
	    }

	    # Compile a function call for the inverse transformation.
	    if (newfunc) {
		CT_NCALLI(ct) = CT_NCALLI(ct) + 1
		if (CT_NCALLI(ct) > MAX_CALL)
		    call syserrs (SYS_MWFCOVFL, system1)

		fc = CT_FCI(ct,CT_NCALLI(ct))
		wf = WCS_FUNC(w1,wfno)
		fn = WF_FN(wf)

		FC_CT(fc)    = ct
		FC_WCS(fc)   = w1
		FC_WF(fc)    = wf
		FC_FCN(fc)   = FN_INV(fn)
		FC_NAXES(fc) = WF_NAXES(wf)

		# Store CTRAN-relative list of axes in function call
		# descriptor.  Verify that all the axes needed for the
		# function call are included in the transformation.
		# This requirement can theoretically be relaxed in
		# some cases but this is not supported in MWCS.

		do j = 1, WF_NAXES(wf) {
		    for (k=1;  k <= naxes;  k=k+1)
			if (axis[k] == WF_AXIS(wf,j)) {
			    FC_AXIS(fc,j) = k
			    break
			}
		    if (k > naxes)
			call syserrs (SYS_MWMISSAX, system1)
		}

		# Call the function driver to perform any driver dependent
		# initialization.

		epa = FN_INIT(fn)
		if (epa != NULL)
		    call zcall2 (epa, fc, INVERSE)
	    }
	}

	# Prepare the linear part of the input transformation W1->P.
	# This is LTM=inv(CD), and for axis I, LTV[i]=(R[i]-inv(CD)*W)
	# if no function call, or LTV[i]=R[i] if there is a function
	# assigned to axis I which already deals with the W[i].  All
	# this is done in the full dimension of the internal system for
	# now; extraction of the portion of the full system affecting
	# the CTRAN axes is done later to permit verification of the
	# legality of the reduction step required.

	# Invert CD matrix.
	if (WCS_CD(w1) == NULL)
	    call mw_mkidmd (Memd[i_ltm], pdim)
	else
	    call mw_invertd (D(mw,WCS_CD(w1)), Memd[i_ltm], pdim)

	# If no function calls for an axis and W is set, LTV=(R-inv(CD)*W).
	if (WCS_W(w1) != NULL) {
	    call amovd (D(mw,WCS_W(w1)), Memd[i_ltv], pdim)
	    do i = 1, CT_NCALLI(ct) {
		fc = CT_FCI(ct,i)
		do j = 1, FC_NAXES(fc) {
		    k = axis[FC_AXIS(fc,j)]
		    Memd[i_ltv+k-1] = 0.
		}
	    }
	    call mw_vmuld (Memd[i_ltm], Memd[i_ltv], Memd[t_ltv], pdim)
	    if (WCS_R(w1) == NULL)
		call anegd (Memd[t_ltv], Memd[i_ltv], pdim)
	    else
		call asubd (D(mw,WCS_R(w1)), Memd[t_ltv], Memd[i_ltv], pdim)
	} else {
	    if (WCS_R(w1) == NULL)
		call aclrd (Memd[i_ltv], pdim)
	    else
		call amovd (D(mw,WCS_R(w1)), Memd[i_ltv], pdim)
	}

	# Now prepare the output side of the transformation, from P->W2.
	# Like the input half, this consists of a linear term and a list
	# of zero or more function calls.

	# Compile the "out" function calls in the CTRAN descriptor.
	do i = 1, naxes {
	    wfno = WCS_AXCLASS(w2,axis[i])

	    # Skip to next axis if no WCS function is assigned to this axis.
	    if (wfno == 0)
		next

	    # Has function call for this axis already been compiled?
	    newfunc = true
	    do j = 1, CT_NCALLO(ct) {
		fc = CT_FCO(ct,j)
		do k = 1, FC_NAXES(fc)
		    if (FC_AXIS(fc,k) == i)
			newfunc = false
	    }

	    # Compile a function call for the forward transformation.
	    if (newfunc) {
		CT_NCALLO(ct) = CT_NCALLO(ct) + 1
		if (CT_NCALLO(ct) > MAX_CALL)
		    call syserrs (SYS_MWFCOVFL, system2)

		fc = CT_FCO(ct,CT_NCALLO(ct))
		wf = WCS_FUNC(w2,wfno)
		fn = WF_FN(wf)

		FC_CT(fc)    = ct
		FC_WCS(fc)   = w2
		FC_WF(fc)    = wf
		FC_FCN(fc)   = FN_FWD(fn)
		FC_NAXES(fc) = WF_NAXES(wf)

		# Store CTRAN-relative list of axes in function call
		# descriptor.  Verify that all the axes needed for the
		# function call are included in the transformation.

		do j = 1, WF_NAXES(wf) {
		    for (k=1;  k <= naxes;  k=k+1)
			if (axis[k] == WF_AXIS(wf,j)) {
			    FC_AXIS(fc,j) = k
			    break
			}
		    if (k > naxes)
			call syserrs (SYS_MWMISSAX, system2)
		}

		# Call the function driver to perform any driver dependent
		# initialization.

		epa = FN_INIT(fn)
		if (epa != NULL)
		    call zcall2 (epa, fc, FORWARD)
	    }
	}

	# Prepare the linear part of the input transformation P->W2.
	# This is LTM=CD, and for axis I, LTV[i]=(W-CD*R) if no function
	# call, or LTV[i]=(-CD*R) if there is a function assigned to axis
	# I which already deals with the W[i].

	# Copy CD matrix to LTM.
	if (WCS_CD(w2) == NULL)
	    call mw_mkidmd (Memd[o_ltm], pdim)
	else
	    call amovd (D(mw,WCS_CD(w2)), Memd[o_ltm], pdim*pdim)

	# Copy -R to t_ltv.
	if (WCS_R(w2) == NULL)
	    call aclrd (Memd[t_ltv], pdim)
	else
	    call amulkd (D(mw,WCS_R(w2)), -1.0D0, Memd[t_ltv], pdim)

	# Compute -CD*R in LTV.
	call mw_vmuld (Memd[o_ltm], Memd[t_ltv], Memd[o_ltv], pdim)

	# If no function calls for an axis and W is set, LTV=(W-CD*R).
	if (WCS_W(w2) != NULL) {
	    call amovd (D(mw,WCS_W(w2)), Memd[t_ltv], pdim)
	    call aaddd (Memd[t_ltv], Memd[o_ltv], Memd[o_ltv], pdim)
	    do i = 1, CT_NCALLO(ct) {
		fc = CT_FCO(ct,i)
		do j = 1, FC_NAXES(fc) {
		    k = axis[FC_AXIS(fc,j)]		# undo +W[k]
		    lp = o_ltv + k - 1
		    Memd[lp] = Memd[lp] - Memd[t_ltv+k-1]
		}
	    }
	}

	# Now combine the linear terms of the input and output transformations
	# to produce the linear portion of the full transformation.

	call mw_mmuld (Memd[o_ltm], Memd[i_ltm], Memd[t_ltm], pdim)
	call mw_vmuld (Memd[o_ltm], Memd[i_ltv], Memd[t_ltv], pdim)
	call    aaddd (Memd[o_ltv], Memd[t_ltv], Memd[t_ltv], pdim)

	# Extract the rows of the full linear transformation which are used
	# for the axes involved in the transformation we are compiling.
	# In the process we must examine the off-diagonal elements of the
	# matrix to verify that the system does not include any dependencies
	# upon axes other than those included in the transformation we are
	# compiling.  (This restriction prohibits dimensional reduction via
	# an image section which results in loss of a rotated axis).

	do i = 1, naxes {
	    # Get matrix line pointers for axis[i].
	    ip = t_ltm + (axis[i]-1) * pdim
	    op = ltm + (i-1) * naxes

	    do j = 1, pdim {
		# Is column J used by transform?
		for (k=1;  k <= naxes;  k=k+1)
		    if (axis[k] == j)
			break

		# If column J is not used in the transform but is not zero,
		# then transform I is dependent upon physical axis J and
		# we cannot do the transform.  If column J is used in the
		# transform, copy the value to the final output matrix LTM
		# discarding unused columns as we go.

		if (k > naxes) {
		    # Check for dependency on axis outside transform.
		    if (abs(Memd[ip+j-1]) > EPSILOND*100.0D0)
			call syserr (SYS_MWROTDEP)
		} else {
		    # Add matrix element to final LTM.
		    Memd[op+k-1] = Memd[ip+j-1]
		}
	    }

	    # Copy the LTV vector element.
	    Memd[ltv+i-1] = Memd[t_ltv+axis[i]-1]
	}

	# Determine the transformation type.  This is LNR for a purely
	# linear transformation with no rotational (off-diagonal) terms,
	# LRO for a purely linear transform with rotational terms, and
	# GEN for everything else.

	if (CT_NCALLI(ct) > 0 || CT_NCALLO(ct) > 0)
	    CT_TYPE(ct) = GEN
	else {
	    CT_TYPE(ct) = LNR
	    do j = 1, naxes
		do i = 1, naxes
		    if (i != j) {
			lp = ltm + (j-1)*naxes + i-1
			if (abs(Memd[lp]) > EPSILOND*100.0D0) {
			    CT_TYPE(ct) = LRO
			    break
			}
		    }
	}

	# Prepare the single precision part of the transform.
	call amovi (Memi[CT_D(ct)], Memi[CT_R(ct)], ctlen)

	ct_r = CT_R(ct)
	CT_LTM(ct_r) = coerce (ct_r + LEN_CTBASE, TY_STRUCT, TY_REAL)
	CT_LTV(ct_r) = CT_LTM(ct_r) + matlen
	call achtdr (Memd[CT_LTM(ct)], Memr[CT_LTM(ct_r)], matlen)
	call achtdr (Memd[CT_LTV(ct)], Memr[CT_LTV(ct_r)], naxes)

	call sfree (sp)
	return (ct)
end


# MW_CTRAN -- Transform a single N-dimensional point, using the optimized
# transformation set up by a prior call to MW_SCTRAN.

procedure wf_pc_ctran (a_ct, p1, p2, ndim)

pointer	a_ct			#I pointer to CTRAN descriptor
double	p1[ndim]		#I coordinates of point in input system
double	p2[ndim]		#O coordinates of point in output system
int	ndim			#I dimensionality of point

int	naxes, i, j
pointer	ct, fc, ltm, ltv, d_ct
double	v1[MAX_DIM], v2[MAX_DIM], iv[MAX_DIM], ov[MAX_DIM]
errchk	zcall3

begin
	# Get real or double version of descriptor.
	ct = CT_D(a_ct)

	ltm = CT_LTM(ct)
	ltv = CT_LTV(ct)

	# Specially optimized cases.
	if (CT_TYPE(ct) == LNR) {
	    # Simple linear, nonrotated transformation.
	    do i = 1, ndim
		p2[i] = Memd[ltm+(i-1)*(ndim+1)] * p1[i] + Memd[ltv+i-1]
	    return
	} else if (CT_TYPE(ct) == LRO) {
	    # Simple linear, rotated transformation.
	    call mw_ltrand (p1, p2, Memd[ltm], Memd[ltv], ndim)
	    return
	}

	# If we get here the transformation involves a call to one or more
	# WCS functions.  In this general case, the transformation consists
	# of zero or more calls to WCS functions to transform the input
	# world coordinates to the linear input system, followed by a general
	# linear transformation to the linear output system, followed by zero
	# or more calls to WCS functions to do the forward transformation
	# to generate the final output world coordinates.  The WCS function
	# calls are always evaluated in double precision.

	# Make zero or more WCS function calls for the different axes of the
	# input system (inverse transform).

	call achtdd (p1, iv, ndim)
	do j = 1, CT_NCALLI(ct) {
	    # Get pointer to function call descriptor.
	    fc = CT_FCI(ct,j)
	    naxes = FC_NAXES(fc)

	    # Extract the coordinate vector for the function call.
	    do i = 1, naxes
		v1[i] = p1[FC_AXIS(fc,i)]

	    # Call the WCS function.
	    call zcall3 (FC_FCN(fc), fc, v1, v2)

	    # Edit the vector IV, replacing the entries associated with
	    # the WCS function by the transformed values.

	    do i = 1, naxes
		iv[FC_AXIS(fc,i)] = v2[i]
	}

	# Apply the general linear transformation.  We may as well do this in
	# double since we already have to use double for the function calls.

	d_ct = CT_D(a_ct)
	call mw_ltrand (iv, ov, Memd[CT_LTM(d_ct)], Memd[CT_LTV(d_ct)], ndim)

	# Make zero or more WCS function calls for the different axes of the
	# output system (forward transform to final world system).

	call achtdd (ov, p2, ndim)
	do j = 1, CT_NCALLO(ct) {
	    # Get pointer to function call descriptor.
	    fc = CT_FCO(ct,j)
	    naxes = FC_NAXES(fc)

	    # Extract the coordinate vector for the function call.
	    do i = 1, naxes
		v1[i] = ov[FC_AXIS(fc,i)]

	    # Call the WCS function.
	    call zcall3 (FC_FCN(fc), fc, v1, v2)

	    # Edit the final output vector, replacing the entries for the
	    # function axes by their transformed values.

	    do i = 1, naxes
		p2[FC_AXIS(fc,i)] = v2[i]
	}
end


# MW_CLOSE -- Close a MWCS descriptor and deallocate all resources used
# by the descriptor.  Any CTRAN descriptors which have been opened on
# the MWCS are automatically closed if not already manually closed by
# the application.

procedure wf_pc_close (mw)

pointer	mw		#U pointer to MWCS descriptor

int	i
pointer	ct

begin
	# Free any still allocated CTRAN descriptors.
	do i = 1, MAX_CTRAN {
	    ct = MI_CTRAN(mw,i)
	    if (ct != NULL)
		iferr (call wf_pc_freect (ct))
		    call erract (EA_WARN)
	}

	# Free the string and data buffers.
	if (MI_SBUF(mw) != NULL)
	    call mfree (MI_SBUF(mw), TY_CHAR)
	if (MI_DBUF(mw) != NULL)
	    call mfree (MI_DBUF(mw), TY_DOUBLE)

	# Free the main descriptor.
	call mfree (mw, TY_STRUCT)
end


# MW_CTFREE -- Free a CTRAN (coordinate transformation) descriptor.  We keep
# track of all allocated CTRAN descriptors in the parent MWCS descriptor, and
# NULL the saved entry for a descriptor when it is freed, thus guaranteeing
# that a descriptor will be freed only once.

procedure wf_pc_freect (ct)

pointer	ct		#U pointer to CTRAN descriptor

int	fn, i, j
pointer mw, fc
include	"mwcs.com"

begin
	if (ct != NULL) {
	    mw = CT_MW(ct)
	    if (mw != NULL)
		do i = 1, MAX_CTRAN
		    if (MI_CTRAN(mw,i) == ct) {
			# Free private storage for any input WCS functions.
			do j = 1, CT_NCALLI(ct) {
			    fc = CT_FCI(ct,j)
			    fn = WF_FN(FC_WF(fc))
			    if (FN_DESTROY(fn) != NULL)
				call zcall1 (FN_DESTROY(fn), fc)
			}
			# Free private storage for any output WCS functions.
			do j = 1, CT_NCALLO(ct) {
			    fc = CT_FCO(ct,j)
			    fn = WF_FN(FC_WF(fc))
			    if (FN_DESTROY(fn) != NULL)
				call zcall1 (FN_DESTROY(fn), fc)
			}
			# Free the main CTRAN descriptor.
			call mfree (ct, TY_STRUCT)
			MI_CTRAN(mw,i) = NULL
			break
		    }
	}
end
