/******************************************************************************
 * This file is part of MPlayer Audio Visualization.                          *
 *                                                                            *
 *                                                                            *
 *  MPlayer Audio Visualization is free software; you can redistribute it     *
 *  and/or modify it under the terms of the GNU General Public License as     *
 *  published by the Free Software Foundation; either version 2 of the        *
 *  License, or (at your option) any later version.                           *
 *                                                                            *
 *  MPlayer Audio Visualization is distributed in the hope that it will be    *
 *  useful, but WITHOUT ANY WARRANTY; without even the implied warranty of    *
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU General *
 *  Public License for more details.                                          *
 *                                                                            * 
 *  You should have received a copy of the GNU General Public License         *
 *  along with MPlayer Audio Visualization; if not, write to the Free         *
 *  Software Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA         *
 *  02111-1307  USA                                                           *
 ******************************************************************************/
#ifndef _H_SCREEN_
#define _H_SCREEN_

/* Common area */

#include <config.h>
#include <msglist.h>


/* Global Functions */

/**
 * Init screen updater. It will launch a thread that updates the screen
 * when needed.
 *
 * @return 0 on succes. -1 Otherwise
 */
int screen_init();

/**
 * Stop the updater thread and clean up screen stuff
 */
void screen_quit();

/**
 * Toggle Fullscreen mode
 */
void screen_toggle_fullscreen();

/**
 * Set resolution
 *
 * @param width the horizontal resolution (width)
 * @param height the vertical resolution (height)
 * @return 0 on success; -2 SDL SetVideoMode error
 */
int screen_set_resolution( int width, int height );

/**
 * Increase resolution based on some pre-defined resolutions
 *
 * @return 0 on success; -1 if maximum resolution reached; -2 SDL SetVideoMode error
 * @see screen_set_resolution
 */
int screen_increase_resolution();

/**
 * Decrease resolution based on some pre-defined resolutions
 *
 * @return 0 on success; -1 if minimum resolution reached; -2 SDL SetVideoMode error
 * @see screen_set_resolution
 */
int screen_decrease_resolution();

/**
 * Toggle display messages on screen
 */
void screen_toggle_messages();


#ifndef _C_SCREEN_

/*
 * Global Area
 */

/* Global Vars */
extern SDL_Surface *screen;      /**< the SDL screen surface */
extern SDL_mutex *screen_mutex;   /**< mutex to screen access */
extern SDL_mutex *messages_mutex; /**< mutex to messages list access */
extern msglist_t *message_list;   /**< messages list */


#else // _C_SCREEN_

/*
 * Local Area: use only in screen.c 
 */

#include <SDL.h>
#include <SDL/SDL_ttf.h>
#include <SDL_thread.h>
#include <stdio.h>
#include <unistd.h>
#include <sys/time.h>
#include <mpav.h>
#include <draw_text.h>
#include <render.h>

#define TITLE "MPlayer Audio Visualization"
#define DEF_RES    3   /**< Default Resolution ( index of resolx/resoly in __screen_change_resolution ) */
#define SURFACE_FLAGS ( SDL_RESIZABLE | SDL_HWSURFACE | SDL_DOUBLEBUF | SDL_ASYNCBLIT ) /**< SDL SetVideoMode flags */

/* Global Vars */
SDL_Surface *screen = NULL;       /**< the SDL screen surface */
int is_fs = DEF_FS;               /**< is full screen? */
SDL_mutex *screen_mutex = NULL;   /**< mutex to screen access */
SDL_mutex *messages_mutex = NULL; /**< mutex to messages list access */
msglist_t *message_list = NULL;   /**< messages list */

/* Local Vars */
static SDL_Thread *screen_thread = NULL;  /**< screen thread pointer */
static int show_messages = 1;             /**< show messages on screen */
static float fps = 0.0f;

/* Local functions */

/**
 * Change screen resolution
 *
 * @parameter d 0 reset current resolution; -1 decrease resolution; 1 increase resolution
 * @return 0 on success; -1 if maximum or minimum resolution reached; -2 SDL SetVideoMode error 
 */
int __screen_change_resolution( int d );

/**
 * Function to be launched as thread. Update the screen when needed 
 */
void __screen_thread_func();

/**
 * Show all the messages in message_list 
 */
void __show_messages();

#endif // _C_SCREEN_

#endif // _H_SCREEN_
