# FIAIF is an Intelligent firewall
#
# Author: Anders Fugmann <afu at fugmann dot net>
#
# FIAIF is an Intelligent firewall
# Copyright (C) 2002-2011 Anders Peter Fugmann
#
# This program is free software; you can redistribute it and/or
# modify it under the terms of the GNU General Public License
# as published by the Free Software Foundation; either version 2
# of the License, or (at your option) any later version.
#
# This program is distributed in the hope that it will be useful,
# but WITHOUT ANY WARRANTY; without even the implied warranty of
# MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
# GNU General Public License for more details.
#
# You should have received a copy of the GNU General Public License
# along with this program; if not, write to the Free Software
# Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.

###############################################################################
# Clear all chains
# Arguments: 0 for close, 1 for open.
###############################################################################
function iptables_stop ()
{
    local DEBUG=$1
    local POLICY
    if (( DEBUG == 0 )); then
	POLICY=DROP
    else
	POLICY=ACCEPT
    fi

    echo -n "Clearing all rules: "
    IPTABLES -F
    IPTABLES -F -t nat
    IPTABLES -F -t mangle

    IPTABLES -X
    IPTABLES -X -t nat
    IPTABLES -X -t mangle

    IPTABLES -Z
    IPTABLES -Z -t nat
    IPTABLES -Z -t mangle

    #Always set default policy.
    IPTABLES -P INPUT ${POLICY}
    IPTABLES -P FORWARD ${POLICY}
    IPTABLES -P OUTPUT ${POLICY}

    IPTABLES -t nat -P PREROUTING ACCEPT
    IPTABLES -t nat -P POSTROUTING ACCEPT
    IPTABLES -t nat -P OUTPUT ACCEPT

    IPTABLES -t mangle -P PREROUTING ACCEPT
    IPTABLES -t mangle -P OUTPUT ACCEPT
    if (( EXTENDED_MANGLE == 1 )); then
	IPTABLES -t mangle -P INPUT ACCEPT
	IPTABLES -t mangle -P FORWARD ACCEPT
	IPTABLES -t mangle -P POSTROUTING ACCEPT
    fi

    echo "Done."
}


###############################################################################
# setup_default_chains
###############################################################################
function setup_default_chains ()
{
    debug_out "Setting up default chains."
    local POLICY
    local LOG_NAME
    local TARGET

    LOG_LIMIT=${LOG_LIMIT:=5/min}
    LOG_BURST=${LOG_BURST:=5}

    if (( VERBOSE == 1 )); then
	for TARGET in ${STANDARD_TARGETS}; do
	    IPTABLES -N LIMIT_LOGGING_${TARGET}
	    IPTABLES -A LIMIT_LOGGING_${TARGET} -m limit --limit ${LOG_LIMIT} \
		--limit-burst ${LOG_BURST} -j RETURN
	    if (( $? == 0 )); then
		IPTABLES -A LIMIT_LOGGING_${TARGET} -j ${TARGET}
	    fi
	done
    fi

    local LOG
    if (( ENABLE_ULOG == 1 )); then
	let ${LOG_LEVEL:=1}
	LOG="-j ULOG --ulog-nlgroup ${LOG_LEVEL} --ulog-prefix "
    else
	let ${LOG_LEVEL:=CRIT}
	LOG="-j LOG --log-level ${LOG_LEVEL} --log-prefix "
    fi

    # All DROP targets
    if (( DEBUG == 0 )); then
	POLICY=DROP
    else
	POLICY=ACCEPT
    fi
    for LOG_NAME in ${LOG_DROP_TARGETS}; do
	IPTABLES -N LOG_${LOG_NAME}
	IPTABLES -N LOG_${LOG_NAME}_NOLOG
	if (( VERBOSE == 1 )); then
	    IPTABLES -A LOG_${LOG_NAME} -j LIMIT_LOGGING_${POLICY}
	    IPTABLES -A LOG_${LOG_NAME} ${LOG} "[${LOG_PREFIX}${LOG_NAME}]:"
	fi
	IPTABLES -A LOG_${LOG_NAME} -j ${POLICY}
	IPTABLES -A LOG_${LOG_NAME}_NOLOG -j ${POLICY}
    done

    # All REJECT targets
    if (( DEBUG == 0 )); then
	POLICY=REJECT
    else
	POLICY=ACCEPT
    fi

    for LOG_NAME in ${LOG_REJECT_TARGETS}; do
	IPTABLES -N LOG_${LOG_NAME}
	IPTABLES -N LOG_${LOG_NAME}_NOLOG
	if (( VERBOSE == 1 )); then
	    IPTABLES -A LOG_${LOG_NAME} -j LIMIT_LOGGING_${POLICY}
	    IPTABLES -A LOG_${LOG_NAME} ${LOG} "[${LOG_PREFIX}${LOG_NAME}]:"
	fi
	IPTABLES -A LOG_${LOG_NAME} -j ${POLICY}
	IPTABLES -A LOG_${LOG_NAME}_NOLOG -j ${POLICY}
    done

    # All ACCEPT targets (Never log.)
    for LOG_NAME in ${LOG_ACCEPT_TARGETS}; do
	IPTABLES -N LOG_${LOG_NAME}
	IPTABLES -N LOG_${LOG_NAME}_LOG
	if (( VERBOSE == 1 )); then
	    IPTABLES -A LOG_${LOG_NAME}_LOG -j LIMIT_LOGGING_ACCEPT
	    IPTABLES -A LOG_${LOG_NAME}_LOG ${LOG} "[${LOG_PREFIX}${LOG_NAME}]:"
	fi
	IPTABLES -A LOG_${LOG_NAME} -j ACCEPT
	IPTABLES -A LOG_${LOG_NAME}_LOG -j ACCEPT
    done

    # Pure logging targets.
    for LOG_NAME in ${LOG_TARGETS}; do
	IPTABLES -N LOG_${LOG_NAME}
	IPTABLES -A LOG_${LOG_NAME} ${LIMIT} ${LOG} "[${LOG_PREFIX}${LOG_NAME}]:"
    done

    # Drop packets, but dont log them.
    if (( DEBUG == 0 )); then
	POLICY=DROP
    else
	POLICY=ACCEPT
    fi
    IPTABLES -N NOLOG_DROP
    IPTABLES -A NOLOG_DROP -j ${POLICY}

    # Make reserved chain
    local RESERVED
    # Either a list or a file.
    debug_out "RESERVED_NETWORKS=${RESERVED_NETWORKS}"
    IPTABLES -N RESERVED_SRC
    IPTABLES -N RESERVED_DST
    if [[ -f "${CONF_DIR}/${RESERVED_NETWORKS}" ]]; then
	cat "${CONF_DIR}/${RESERVED_NETWORKS}" | cut -d"#" -f1 | while read RESERVED; do
	    if [[ -z "${RESERVED}" ]]; then
		continue
	    fi
	    IPTABLES -A RESERVED_SRC -s ${RESERVED} -j LOG_MARTIAN
	    IPTABLES -A RESERVED_DST -d ${RESERVED} -j LOG_MARTIAN
	done
    else
	for RESERVED in ${RESERVED_NETWORKS}; do
	    IPTABLES -A RESERVED_SRC -s ${RESERVED} -j LOG_MARTIAN
	    IPTABLES -A RESERVED_DST -d ${RESERVED} -j LOG_MARTIAN
	done
    fi
    local PRIVATE
    debug_out "PRIVATE_NETWORKS=${PRIVATE_NETWORKS}"
    IPTABLES -N PRIVATE_SRC
    IPTABLES -N PRIVATE_DST
    if [[ -f "${CONF_DIR}/${PRIVATE_NETWORKS}" ]]; then
	cat "${CONF_DIR}/${PRIVATE_NETWORKS}" | cut -d"#" -f1 | while read PRIVATE; do
	    if [[ -z "${PRIVATE}" ]]; then
		continue
	    fi
	    IPTABLES -A PRIVATE_SRC -s ${PRIVATE} -j LOG_MARTIAN
	    IPTABLES -A PRIVATE_DST -d ${PRIVATE} -j LOG_MARTIAN
	done
    else
	for PRIVATE in ${PRIVATE_NETWORKS}; do
	    IPTABLES -A PRIVATE_SRC -s ${PRIVATE} -j LOG_MARTIAN
	    IPTABLES -A PRIVATE_DST -d ${PRIVATE} -j LOG_MARTIAN
	done
    fi

    debug_out "Setup packets sanity checks"
    IPTABLES -t filter -N SANITY
    add_sanity_check filter SANITY

    local DEVICE
    IPTABLES -N CHECK_IP
    for DEVICE in ${DEVICE_LIST}; do
	IPTABLES -N DEV_${DEVICE}_SRC
	IPTABLES -N DEV_${DEVICE}_DST
	IPTABLES -A CHECK_IP -i ${DEVICE} -j DEV_${DEVICE}_SRC
	IPTABLES -A CHECK_IP -o ${DEVICE} -j DEV_${DEVICE}_DST
    done

    # Add zone miss targets.
    if (( DEBUG == 0 )); then
	POLICY=DROP
    else
	POLICY=ACCEPT
    fi

    for ZONE in ${ZONES}; do
	IPTABLES -N LOG_MISS_${ZONE}
	if (( VERBOSE == 1 )); then
	    IPTABLES -A LOG_MISS_${ZONE} -j LIMIT_LOGGING_${POLICY}
	    IPTABLES -A LOG_MISS_${ZONE} ${LOG} "[${LOG_PREFIX}ZONE_MISS_${ZONE}]:"
	fi
	IPTABLES -A LOG_MISS_${ZONE} -j ${POLICY}
    done
}


###############################################################################
# Setup initial redirection to chains.
###############################################################################
function chain_redirect ()
{
    debug_out "Global chains"
    local CHAIN
    local PROTOCOL
    local DEVICE
    local PARAM

    IPTABLES -N GENERAL
    IPTABLES -A GENERAL -j SANITY
    IPTABLES -A GENERAL -m conntrack --ctstate RELATED,ESTABLISHED -j LOG_ACCEPT
    # We only examine new packets.
    IPTABLES -A GENERAL -j CHECK_IP

    for CHAIN in ${BUILT_IN_CHAINS_filter}; do
	IPTABLES -N ${CHAIN}_NEW
	IPTABLES -A ${CHAIN} -j GENERAL
	IPTABLES -A ${CHAIN} -m conntrack --ctstate NEW -j ${CHAIN}_NEW
	IPTABLES -A ${CHAIN} -j LOG_GLOBAL_MISS

	for DEVICE in ${DEVICE_LIST}; do
	    IPTABLES -N ${CHAIN}_NEW_${DEVICE}
	    case "${CHAIN}" in
                INPUT)
		    PARAM="-i ${DEVICE}"
		    ;;
		OUTPUT | FORWARD)
		    PARAM="-o ${DEVICE}"
		    ;;
	    esac

	    IPTABLES -A ${CHAIN}_NEW ${PARAM} -j ${CHAIN}_NEW_${DEVICE}
	done
    done

    if kernel_version_ge 2.4.21; then
        debug_out "Enable traceroute workarround:"
	IPTABLES -I OUTPUT -p ICMP -m conntrack --ctstate INVALID --icmp-type 11 \
	    -j OUTPUT_NEW
    fi

    for DEVICE in ${DEVICE_LIST}; do
	IPTABLES -N SEND_NEW_${DEVICE}
	# Insert before all other forward rules
	IPTABLES -I FORWARD_NEW -i ${DEVICE} -j SEND_NEW_${DEVICE}
    done

}

###############################################################################
# set_tos
# Params: TOS protocol ports
###############################################################################
function set_tos ()
{
    local CHAIN=$1
    local TOS=$2
    local PROTOCOL=$3
    local PORTS=$4

    if [[ -n "${PORTS}" ]]; then
	local PORTS_PARAM=""
	if [[ "${PORTS}" == "any" || "${PORTS}" == "ANY" ]]; then
	    IPTABLES -t mangle -A ${CHAIN} -p ${PROTOCOL} \
		-j TOS --set-tos ${TOS}
	else
	    if [[ "${PROTOCOL}" = "icmp" || "${PROTOCOL}" = "ICMP" ]]; then
		IPTABLES -t mangle -A ${CHAIN} -p ${PROTOCOL} \
		    --icmp-type ${PORTS} -j TOS --set-tos ${TOS}
	    else
		IPTABLES -t mangle -A ${CHAIN} -p ${PROTOCOL} \
		    -m multiport --dports ${PORTS} -j TOS --set-tos ${TOS}
		IPTABLES -t mangle -A ${CHAIN} -p ${PROTOCOL} \
		    -m multiport --sports ${PORTS} -j TOS --set-tos ${TOS}

	    fi
	fi
    fi
}

###############################################################################
# Read TOS_FILE, and call set_tos
###############################################################################
function read_tos()
{
    if [[ -n "${TOS_FILE}" ]]; then
	if [[ ! -f ${CONF_DIR}/${TOS_FILE} ]]; then
	    echo Could not locate TOS settings file: ${CONF_DIR}/${TOS_FILE}.
	else
	    source ${CONF_DIR}/${TOS_FILE}

	    # Create the chain.
	    local CHAIN="SET_TOS"
	    IPTABLES -t mangle -N ${CHAIN}

	    local TOS
            # Setup TOS values:
	    for TOS in ${!TOS_*}; do
		debug_out "${TOS}=${!TOS}"
		set_tos ${CHAIN} ${!TOS}
	    done
	    IPTABLES -t mangle -A ${CHAIN} -m helper --helper ftp -j TOS --set-tos Maximize-Throughput
	    # Route the the chain.
	    if (( EXTENDED_MANGLE == 1 )); then
		IPTABLES -t mangle -A INPUT -j ${CHAIN}
		IPTABLES -t mangle -A FORWARD -j ${CHAIN}
		IPTABLES -t mangle -A OUTPUT -j ${CHAIN}
	    else
		IPTABLES -t mangle -A PREROUTING -j ${CHAIN}
		IPTABLES -t mangle -A OUTPUT -j ${CHAIN}
	    fi
	fi
    fi
}


###############################################################################
# Setup zone LO. This zone is so specialized,
# so who would change rules on this.
# I still believe that the loopback interface is weired.
###############################################################################
function configure_zone_lo ()
{
    debug_out "Configuring zone: LO"

    #Do not allow forwarding.
    IPTABLES -A FORWARD -o lo -j LOG_DROP
    IPTABLES -A FORWARD -i lo -j LOG_DROP

    # Accept any packets.
    IPTABLES -A INPUT -i lo -j ACCEPT

    # Allow lo to send any packets.
    IPTABLES -A OUTPUT -o lo -j ACCEPT

}

###############################################################################
# State valid:
# returns 1, if the state file is valid.
###############################################################################
function state_valid()
{
    if [[ ! -f ${FIAIF_STATE_FILE} ]]; then
	return 2
    fi

    if [[ ${FIAIF_STATE_FILE} -ot ${CONF_FILE} ||
          ${FIAIF_STATE_FILE} -ot ${CONF_DIR}/${RESERVED_NETS} || \
          ${FIAIF_STATE_FILE} -ot ${CONF_DIR}/${PRIVATE_NETS}  || \
          ${FIAIF_STATE_FILE} -ot ${CONF_DIR}/${TOS_FILE} || \
          ${FIAIF_STATE_FILE} -ot ${VERSION_FILE} ]]; then
	return 1
    fi

    local ZONE_FILE
    local ZONE
    for ZONE in ${ZONES}; do
	ZONE_FILE=CONF_${ZONE}
	ZONE_FILE=${CONF_DIR}/${!ZONE_FILE}
	if [[ ${FIAIF_STATE_FILE} -ot ${ZONE_FILE} ]]; then
	    return 1
	fi
    done

    local FILE
    for FILE in ${CONF_DIR}/*; do
	if [[ ${FIAIF_STATE_FILE} -ot ${FILE} ]]; then
	    return 1
	fi
    done

    return 0
}

###############################################################################
# Test if a zone can be read and initialized.
# returns 1, if everything is ok.
###############################################################################
function test_zone ()
{
    local ZONE=$1
    local ZONE_LIST=${ZONES// /,}
    local TMP_FILE=$(mktemp /tmp/fiaif-tmp.XXXXXX)
    local ZONE_FILE=CONF_${ZONE}
    ZONE_FILE=${!ZONE_FILE}
    ZONE_FILE=${CONF_DIR}/${ZONE_FILE}
    local AVAILABLE=1
    local STRING

    # First test the syntax.
    # Remember to send all output to the debug file also.
    awk -v "ZONES=${ZONE_LIST}" \
	-f ${FIAIF_SHARED}/syntax.awk \
	-f ${FIAIF_SHARED}/zone_rules.awk < ${ZONE_FILE} > ${TMP_FILE}

    if (( $? != 0 )); then
	# We found an error.
	print_err "Syntax check for zone ${ZONE} failed. See errors below."
	cat ${TMP_FILE} | while read line; do
	    print_err "${line}"
	done
	AVAILABLE=0
    else
	# Try reading the zone.
	read_zone ${ZONE}
	if (( $? != 0 )); then
	    print_err "Warning: Device for ${ZONE} is not up."
	    print_err "Unable to configure zone."
	    AVAILABLE=0
	fi
    fi

    rm -f ${TMP_FILE}
    return ${AVAILABLE}
}
###############################################################################
# Main
###############################################################################
function iptables_setup ()
{
    # Damn bash. We really needed this to be in a seperate function,
    # but declare only declares locally to functions.
    local GLOBAL_INTERFACES=""
    local DYNAMIC_INTERFACES=""
    DEVICE_LIST=""

    local DEVICE
    for ZONE in ${ZONES}; do
	# Test the zone file.
	test_zone ${ZONE}
	if (( $? == 0 )); then
	    declare -ri ${ZONE}_AVAILABLE="0"
	    let DEV_ERRORS++
	else
	    declare -r ${ZONE}_DEV="${DEV}"
	    declare -r ${ZONE}_IP="${IP}"
	    declare -ri ${ZONE}_DYNAMIC="${DYNAMIC}"
	    declare -r ${ZONE}_BCAST="${BCAST}"
	    declare -ri ${ZONE}_GLOBAL="${GLOBAL}"
	    declare -r ${ZONE}_NETS="${NET} ${NET_EXTRA}"
	    declare -r ${ZONE}_IPS="${IP} ${IP_EXTRA}"
	    declare -ri ${ZONE}_AVAILABLE="1"

	    # Make sure that only one DYNAMIC/GLOBAL zone per interface.
	    local IFACE
	    for IFACE in ${GLOBAL_INTERFACES} ${DYNAMIC_INTERFACES}; do
		if [[ "${IFACE}" == "${DEV}" ]]; then
		    if (( GLOBAL == 1 )); then
			print_err "Error: Zone ${ZONE} declared GLOBAL, but a zone has already been defined for ${DEV}"
		    elif (( DYNAMIC == 1 )); then
			print_err "Error: Zone ${ZONE} declared DYNAMIC, but a zone has already been defined for ${DEV}"
		    else
			print_err "Error: Zone ${ZONE} covers interface ${DEV}, but a zone definition already exists"
			print_err "for ${DEV}, which is declared GLOBAL or DYNAMIC."
		    fi
    		    declare ${ZONE}_AVAILABLE="0"
		    let DEV_ERRORS++
		fi
	    done

	    local AVAILABLE=${ZONE}_AVAILABLE
	    AVAILABLE=${!AVAILABLE}
	    if (( AVAILABLE == 0 )); then
		continue;
	    fi

	    if (( GLOBAL == 1 )); then
		GLOBAL_INTERFACES="${GLOBAL_INTERFACES} ${DEV}"
	    fi
	    if (( DYNAMIC == 1 )); then
		DYNAMIC_INTERFACES="${DYNAMIC_INTERFACES} ${DEV}"
	    fi

	    # Create a device-based list
	    for DEVICE in ${DEVICE_LIST}; do
		if [[ "${DEVICE}" == "${DEV}" ]]; then
		    break
		fi
	    done

	    if [[ "${DEVICE}" != "${DEV}" ]]; then
		DEVICE_LIST="${DEVICE_LIST} ${DEV}"
	    fi

	    # Create zone specific chains.
	    zone_chains ${ZONE}

	fi
    done
    # Setup default global chains.
    setup_default_chains

    # Set TOS values
    read_tos

    # Configure the loopback device.
    configure_zone_lo

    # Setup chain redirection.
    chain_redirect

    # read aliases.
    if [[ -f "${CONF_DIR}/${ALIASES}" ]]; then
	local FILE=$(mktemp "/tmp/fiaif-tmp.XXXXXX")
	read_aliases ${CONF_DIR}/${ALIASES} ${FILE}
	source ${FILE}
	rm -f ${FILE}
    fi

    echo -n "Configuring zone:"
    for ZONE in ${ZONES}; do
	local ZONE_AVL=${ZONE}_AVAILABLE
	ZONE_AVL=${!ZONE_AVL}
	if (( ZONE_AVL == 1 )); then
	    echo -n " ${ZONE}"
	    configure_zone ${ZONE}
	fi
    done
    echo

    debug_out "Log martians"
    for DEVICE in ${DEVICE_LIST}; do
	IPTABLES -A DEV_${DEVICE}_SRC -j LOG_MARTIAN
	IPTABLES -A DEV_${DEVICE}_DST -j LOG_MARTIAN
    done

    return ${DEV_ERRORS}
}
