/* Copyright (C) 2006 to 2010 Chris Vine

The library comprised in this file or of which this file is part is
distributed by Chris Vine under the GNU Lesser General Public
License as follows:

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Lesser General Public License
   as published by the Free Software Foundation; either version 2.1 of
   the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Lesser General Public License, version 2.1, for more details.

   You should have received a copy of the GNU Lesser General Public
   License, version 2.1, along with this library (see the file LGPL.TXT
   which came with this source code package in the c++-gtk-utils
   sub-directory); if not, write to the Free Software Foundation, Inc.,
   59 Temple Place - Suite 330, Boston, MA, 02111-1307, USA.

*/

#include <c++-gtk-utils/lib_defs.h>

#include <c++-gtk-utils/mutex.h>

namespace Cgu {

namespace Thread {

/* since we have to compile some RecMutex methods into the library for
   the reasons mentioned below, we might as well compile in
   Cond::get_abs_time() and Cond::Cond as well, which also depend on
   configuration tests.
 */
void Cond::get_abs_time(timespec& ts, unsigned int millisec) {
#ifdef HAVE_MONOTONIC_CLOCK
  clock_gettime(CLOCK_MONOTONIC, &ts);
#else
  clock_gettime(CLOCK_REALTIME, &ts);
#endif
  unsigned long nanosec = ts.tv_nsec + ((millisec % 1000) * 1000000);
  ts.tv_sec += millisec/1000 + nanosec/1000000000;
  ts.tv_nsec = nanosec % 1000000000;
}

bool Cond::have_monotonic_clock() {
#ifdef HAVE_MONOTONIC_CLOCK
  return true;
#else
  return false;
#endif
}

Cond::Cond() {
#ifdef HAVE_MONOTONIC_CLOCK
  pthread_condattr_t attr;
  if (pthread_condattr_init(&attr))
    throw CondError();
  if (pthread_condattr_setclock(&attr, CLOCK_MONOTONIC)
      || pthread_cond_init(&cond, &attr)) {
    pthread_condattr_destroy(&attr);
    throw CondError();
  }
  pthread_condattr_destroy(&attr);
#else
  if (pthread_cond_init(&cond, 0))
    throw CondError();
#endif
}

/* we compile these RecMutex methods into the library because they
   depend on _XOPEN_SOURCE being defined as 600 and it is easier to
   ensure this is done on compiling the library rather than when the
   user compiles individual programs.  HAVE_RECURSIVE_MUTEX_HEADERS is
   defined in config.h and so included via lib_defs.h, and is tested
   for in the ./configure script.
 */
int RecMutex::test_support() {
#ifdef HAVE_RECURSIVE_MUTEX_HEADERS
  pthread_mutexattr_t attr;
  if (pthread_mutexattr_init(&attr))
    return 1;
  if (pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE)) {
    pthread_mutexattr_destroy(&attr);
    return -1;
  }
  pthread_mutexattr_destroy(&attr);
  return 0;
#else
  return -1;
#endif
}

RecMutex::RecMutex() {
#ifdef HAVE_RECURSIVE_MUTEX_HEADERS
  pthread_mutexattr_t attr;
  if (pthread_mutexattr_init(&attr))
    throw MutexError();
  if (pthread_mutexattr_settype(&attr, PTHREAD_MUTEX_RECURSIVE)) {
    pthread_mutexattr_destroy(&attr);
    throw RecMutexError();
  }
  if (pthread_mutex_init(&pthr_mutex, &attr)) {
    pthread_mutexattr_destroy(&attr);
    throw MutexError();
  }
  pthread_mutexattr_destroy(&attr);
#else
  throw RecMutexError();
#endif
}

} // namespace Thread

} // namespace Cgu
