@rem = '--*-Perl-*--
@echo off
if "%OS%" == "Windows_NT" goto WinNT
perl -x -S "%0" %1 %2 %3 %4 %5 %6 %7 %8 %9
goto endofperl
:WinNT
perl -x -S %0 %*
if NOT "%COMSPEC%" == "%SystemRoot%\system32\cmd.exe" goto endofperl
if %errorlevel% == 9009 echo You do not have Perl in your PATH.
if errorlevel 1 goto script_failed_so_exit_with_non_zero_val 2>nul
goto endofperl
@rem ';
#!/usr/bin/perl -w
#line 15
#
# ecaccess-file-mdelete: Delete Multiple ECaccess Files at once
#
# Laurent.Gougeon@ecmwf.int - 2010-10-15

use ECMWF::ECaccess;
use Getopt::Long;
use Pod::Usage;
use MIME::Base64;
use Term::Prompt;
use Number::Bytes::Human qw(format_bytes);

my %opt = ( force => 0, version => 0, help => 0, manual => 0, retry => 0, debug => 0 );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if !GetOptions(
	\%opt,
	qw(
	  force
	  version
	  help|?
	  manual
	  retry=i
	  debug
	  )
);

# Display version if requested
die ECMWF::ECaccess->VERSION . "\n" if ( $opt{version} );

pod2usage( -noperldoc => 1, -exit => 1, verbose => 1 ) if ( $opt{help} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 2 ) if ( $opt{manual} );
pod2usage( -noperldoc => 1, -exit => 1, verbose => 0, -msg => "No target-ecaccess-files specified!\n" ) if not(@ARGV);

# Create the ECaccess Controler
my $ecaccess = ECMWF::ECaccess->new( $opt{retry}, $opt{debug});

# Get the Token (using the Certificate in $HOME)
my $token = $ecaccess->getToken();

# Get the Control Channel
my $controlChannel = $ecaccess->getControlChannel();

# Process all source Files from the command-line
foreach (@ARGV) {
	my $sourceFiles = $_;

	# Get the list of Files to download
	my $files = $controlChannel->getDirList( $token, $sourceFiles );

	# Delete each File
	foreach my $file ( $files->valueof('//getDirListResponse/return') ) {

		# Set source filename
		my $source = $file->{domain} . "/" . $file->{name};

		# Do we delete this file? (don't process directories and symbolic links)
		if ( not( substr( $file->{permissions}, 0, 1 ) eq 'd' )
			&& not( substr( $file->{permissions}, 0, 1 ) eq 'l' )
			&& $file->{size} >= '0' && ( $opt{force} || &prompt( "y", "Delete " . $source, "y/n", "y" ) ) )
		{
			print "Deleting " . $source . " (" . ( format_bytes( $file->{size} ) ) . ") ...\n";

			# Delete the source file
			$controlChannel->deleteFile( $token, $source, SOAP::Data->type( boolean => 'true' ) );
		}
	}
}

# Logout
$ecaccess->releaseToken($token);

__END__

=head1 NAME

ecaccess-file-mdelete - Delete Multiple ECaccess Files at once

=head1 SYNOPSIS

B<ecaccess-file-mdelete -version|-help|-manual>

B<ecaccess-file-mdelete [-debug] [-force]> I<target-ecaccess-file> B<[>I<...>B<]>

=head1 DESCRIPTION

Allow deleting Multiple ECaccess Files at once.

Each I<target-ecaccess-file> is in the form [domain:][/user-id/]path. Please read the "Shell commands -> File Management"
section of the "ecaccess" guide for more information on the ECaccess File System.

=head1 ARGUMENTS

=over 8

=item I<target-ecaccess-file> B<[>I<...>B<]>

The name(s) of the ECaccess File(s) to delete.

=back

=head1 OPTIONS

=over 8

=item B<-force>

Overrides the interactive mode and delete each file without prompting.

=item B<-version>

Display version number and exits.

=item B<-help>

Print a brief help message and exits.

=item B<-manual>

Prints the manual page and exits.

=item B<-retry> I<count>

Number of SSL connection retries per 5s to ECMWF. This parameter only apply to the
initial SSL connection initiated by the command to the ECMWF server. It does not
apply to all the subsequent requests made afteward as it is mainly targeting errors
that can happen from time to time during the SSL handshake. Default is no retry.

=item B<-debug>

Display the SOAP and SSL messages exchanged.

=back

=head1 EXAMPLES

B<ecaccess-file-mdelete -force> I<'home:/xyz/bin/*.bin'>

Delete the I<*.bin> Files in the $HOME/bin directory of the user xyz without prompting.

=head1 SEE ALSO

B<ecaccess-file-delete>, B<ecaccess-file-get>, B<ecaccess-file-mget>, B<ecaccess-file-modtime>, B<ecaccess-file-mput>,
B<ecaccess-file-rmdir>, B<ecaccess-file-copy>, B<ecaccess-file-dir>, B<ecaccess-file-chmod>, B<ecaccess-file-mkdir>,
B<ecaccess-file-move>, B<ecaccess-file-put>, B<ecaccess-file-size> and B<ecaccess>.

=cut

__END__
:endofperl
