# Copyright © The Debusine Developers
# See the AUTHORS file at the top-level directory of this distribution
#
# This file is part of Debusine. It is subject to the license terms
# in the LICENSE file found in the top-level directory of this
# distribution. No part of Debusine, including this file, may be copied,
# modified, propagated, or distributed except according to the terms
# contained in the LICENSE file.

"""URLs for the server application - API."""

from django.urls import URLPattern, URLResolver, include, path
from rest_framework import routers

from debusine.server.views.artifacts import (
    ArtifactRelationsView,
    ArtifactView,
    UploadFileView,
)
from debusine.server.views.assets import AssetPermissionCheckView, AssetView
from debusine.server.views.base import SupersededAPIView, Whoami
from debusine.server.views.collections import (
    CollectionViewSet,
    TaskConfigurationCollectionView,
)
from debusine.server.views.enroll import EnrollView
from debusine.server.views.lookups import LookupMultipleView, LookupSingleView
from debusine.server.views.open_metrics import OpenMetricsView
from debusine.server.views.service_status import ServiceStatusView
from debusine.server.views.work_requests import (
    WorkRequestAbortView,
    WorkRequestConfirmView,
    WorkRequestExternalDebsignView,
    WorkRequestRetryView,
    WorkRequestUnblockView,
    WorkRequestView,
)
from debusine.server.views.workers import (
    GetNextWorkRequestView,
    RegisterView,
    UpdateWorkRequestAsCompletedView,
    UpdateWorkerDynamicMetadataView,
)
from debusine.server.views.workflows import (
    WorkflowTemplateViewSet,
    WorkflowView,
)
from debusine.server.views.workspaces import WorkspaceInheritanceView

app_name = 'server'

urlpatterns_1_0: list[URLPattern | URLResolver] = [
    path(
        'whoami/',
        Whoami.as_view(),
        name='whoami',
    ),
    path(
        'work-request/get-next-for-worker/',
        GetNextWorkRequestView.as_view(),
        name='work-request-get-next',
    ),
    path(
        'work-request/',
        WorkRequestView.as_view(),
        name='work-requests',
    ),
    path(
        'work-request/<int:work_request_id>/',
        WorkRequestView.as_view(),
        name='work-request-detail',
    ),
    path(
        'work-request/<int:work_request_id>/completed/',
        UpdateWorkRequestAsCompletedView.as_view(),
        name='work-request-completed',
    ),
    path(
        'work-request/<int:work_request_id>/retry/',
        WorkRequestRetryView.as_view(),
        name='work-requests-retry',
    ),
    path(
        'work-request/<int:work_request_id>/abort/',
        WorkRequestAbortView.as_view(),
        name='work-request-abort',
    ),
    path(
        'work-request/<int:work_request_id>/unblock/',
        WorkRequestUnblockView.as_view(),
        name='work-request-unblock',
    ),
    path(
        'work-request/<int:work_request_id>/external-debsign/',
        WorkRequestExternalDebsignView.as_view(),
        name='work-request-external-debsign',
    ),
    path(
        "work-request/<int:work_request_id>/confirm/",
        WorkRequestConfirmView.as_view(),
        name="work-request-confirm",
    ),
    path(
        "artifact/",
        ArtifactView.as_view(),
        name="artifact-create",
    ),
    path(
        "artifact/<int:artifact_id>/",
        ArtifactView.as_view(),
        name="artifact",
    ),
    path(
        "artifact/<int:artifact_id>/files/<path:file_path>/",
        UploadFileView.as_view(),
        name="upload-file",
    ),
    path(
        "asset/",
        AssetView.as_view(),
        name="asset",
    ),
    path(
        (
            "asset/<str:asset_category>/<str:asset_slug>/"
            "<str:permission_name>/"
        ),
        AssetPermissionCheckView.as_view(),
        name="asset-permission-check",
    ),
    path("lookup/single/", LookupSingleView.as_view(), name="lookup-single"),
    path(
        "lookup/multiple/",
        LookupMultipleView.as_view(),
        name="lookup-multiple",
    ),
    path('worker/register/', RegisterView.as_view(), name='register'),
    path(
        'worker/dynamic-metadata/',
        UpdateWorkerDynamicMetadataView.as_view(),
        name='worker-dynamic-metadata',
    ),
    path(
        "artifact-relation/",
        ArtifactRelationsView.as_view(),
        name="artifact-relation-list",
    ),
    path(
        "artifact-relation/<int:pk>/",
        ArtifactRelationsView.as_view(),
        name="artifact-relation-detail",
    ),
    path("workflow/", WorkflowView.as_view(), name="workflows"),
    path(
        "service-status/",
        ServiceStatusView.as_view(),
        name="service-status",
    ),
    path(
        "task-configuration/<str:workspace>/<str:name>/",
        TaskConfigurationCollectionView.as_view(),
        name="task-configuration-collection",
    ),
    path(
        "workspace/<str:workspace>/inheritance/",
        WorkspaceInheritanceView.as_view(),
        name="workspace-inheritance",
    ),
    path(
        "open-metrics/",
        OpenMetricsView.as_view(),
        name="open-metrics",
    ),
    # Deprecated: use the workflow-template viewset instead
    path(
        "workflow-template/",
        SupersededAPIView.as_view(),
        {
            "new_endpoint": "/1.0/workflow-template/<workspace>/",
            "client_version": "0.13.0",
        },
        name="workflow-templates",
    ),
]

router = routers.DefaultRouter()
router.include_root_view = False
router.register(
    "collection/(?P<workspace>[^/]+)",
    CollectionViewSet,
    basename="collection",
)
router.register(
    "workflow-template/(?P<workspace>[^/]+)",
    WorkflowTemplateViewSet,
    basename="workflow-template",
)
urlpatterns_1_0 += router.urls

urlpatterns = [
    path(
        'auth/',
        include('rest_framework.urls', namespace='rest_framework'),
    ),
    path("enroll/", EnrollView.as_view(), name="enroll"),
    path("1.0/", include(urlpatterns_1_0)),
]
