#include <stdio.h>
#include <stdlib.h>
#include <limits.h>
#include <string.h>

/* This file has all of the code for the first-order interface to
 * propositional model searching, including:
 *  1. Take (flat) first-order clauses and generate propositional clauses.
 *  2. Print models, e.g., multiplication table.
 *
 */

/* TO DO:
 *  1. When add units from assignments, also add neg units??
 *  2. Move unit preprocessing to DP code??
 *  3. Enumerated sorts.
 *  4. Allow assigned or enumerated constants in clauses.
 *  5. Otter input for sorted logic.
 */

/* #define PRINT_CLAUSES  /* print ground clauses that are sent to DP */

/****** External variables and routines *********/

 extern void insert_dp_clause(int c[], int n);
 extern void abend(char *s);
 extern int atom_value(int atom);
 extern long run_time(void);
 extern void exit_if_over_time_limit(void);
 extern void unsatisfiable_exit(void);

#define STR_IDENT(s,t) (strcmp(s,t) == 0)

/* Limits
 * WARNING: you can't simply change MAX_VARS or MAX_ARITY.
 */

#define MAX_VARS       12  /* If changed, change generate_prop_clauses(). */
#define MAX_ARITY       4  /* See generate_prop_clauses(), ATOM, etc. */
#define MAX_LITERALS   35  /* per clause */
#define MAX_CLAUSES   500  /* first-order clauses */
#define MAX_SYMBOLS    50  /* number of functors */
#define MAX_NAME       32  /* string length of functors */
#define MAX_SORTS      10  /* number of sorts */

/******** Type declarations *********/

struct sort {
    char name[MAX_NAME];    /* name of sort */
    int n;                  /* size of sort */
    };

struct symbol {
    int base;                    /* prop. atom numbers start here */
    int arity;                   /* of symbol */
    struct {
	int sort;                /* index of sort */
	int n;                   /* size of sort */
	int mult;                /* for constructing prop. variables */
	} args[MAX_ARITY];
    char type[MAX_NAME];         /* function or relation */
    char name[MAX_NAME];         /* for printing */
    char properties[MAX_NAME];   /* special properties of symbol */
    int assigned_value;          /* for distinct constants option */
    };

struct rel_lit {      /* first_order literal */
    int sign;         /* 1 or -1 */
    int functor;      /* index into Symbols */
    int a[MAX_ARITY]; /* args: vars are +, domain elements are -(e+100). */
    char ai[MAX_ARITY][MAX_NAME];  /* array of args in string form */
    };

struct rel_clause {     /* first_order clause */
    int id;
    int num_lits;
    struct rel_lit lits[MAX_LITERALS];
    };

/******* Global variables ***************/

static int Clauses_generated;
static int Clauses_inserted;

static struct sort Sorts[MAX_SORTS];
static int Num_sorts;

static struct symbol Symbols[MAX_SYMBOLS];
static int Num_symbols;
static int Next_base = 1;

static struct rel_clause Clauses[MAX_CLAUSES];
static int Num_rel_clauses;

static int Greatest_atom;
static int *Units;

static int Experiment;
static int Hole = -1;

/* The following macros take sequences of integers and give unique integers
 * that are the propositional variables.  The input, for example b,i,j,
 * represents application of symbol with "base" b to elements of the
 * domain.
 */

#define ATOM0(s) (s->base)
#define ATOM1(s,i) (i + s->base)
#define ATOM2(s,i,j) ((i)*s->args[0].mult + j + s->base)
#define ATOM3(s,i,j,k) ((i)*s->args[0].mult + (j)*s->args[1].mult + k + s->base)
#define ATOM4(s,i,j,k,l) ((i)*s->args[0].mult + (j)*s->args[1].mult + (k)*s->args[2].mult + l + s->base)

/*************
 *
 *   sort_index()
 *
 *************/

static int sort_index(char *s)
{
    int i;
    for (i = 0; i < Num_sorts; i++)
	if (STR_IDENT(s, Sorts[i].name))
	    return(i);
    return(-1);
}  /* sort_index */

/*************
 *
 *   declare_symbol_sorted()
 *
 *************/

static int declare_symbol_sorted(char *name, int arity, char at[][MAX_NAME])
{
    int i, j, m;
    struct symbol *s;

    s = Symbols + Num_symbols;
    Num_symbols++;
    strcpy(s->name, name);
    s->arity = arity;
    s->base = Next_base;
    s->assigned_value = -1;

    for (i = 0; i < arity; i++) {
	j = sort_index(at[i]);
	if (j == -1) {
	    printf("%s", at[i]);
	    abend("declare_symbol_sorted: sort not found");
	    }
	s->args[i].sort = j;
	s->args[i].n = Sorts[j].n;
	}

    for (i = arity-1, m = 1; i >= 0; i--) {
	s->args[i].mult = m;
	m = m * s->args[i].n;
	}

    Next_base += m;
    return(Num_symbols-1);
}  /* declare_symbol_sorted */

/*************
 *
 *   str_to_id()
 *
 *   Given the string form of a functor, return the ID (or -1 if not found).
 *
 *************/

int str_to_id(char *name)
{
    int i;
    for (i = 0; i < Num_symbols; i++)
	if (STR_IDENT(name, Symbols[i].name))
	    return(i);
    return(-1);
}  /* str_to_id */

/*************
 *
 *   decode_int()
 *
 *   Given a propositional variable (an integer), find the atom that
 *   it represents.  This is the inverse of the ATOM macros.
 *   This is used mostly for printing ground clauses in a readable way.
 *
 *************/

int decode_int(int atom, char **func, int *arity, int *args)
{
    int i, c;
    struct symbol *s;

    if (atom >= Next_base || atom < 1)
	return(-1);
    for (i = 0; i < Num_symbols; i++)
	 if (atom < Symbols[i].base)
	     break;
    i--;
    s = Symbols+i;
    c = atom - s->base;
    *func = s->name;
    *arity = s->arity;

    switch (*arity) {
      case 1:
	args[0] = c;
	break;
      case 2:
	args[1] = c % s->args[1].n; c = c / s->args[1].n;
	args[0] = c;
	break;
      case 3:
	args[2] = c % s->args[2].n; c = c / s->args[2].n;
	args[1] = c % s->args[1].n; c = c / s->args[1].n;
	args[0] = c;
	break;
      case 4:
	args[3] = c % s->args[3].n; c = c / s->args[3].n;
	args[2] = c % s->args[2].n; c = c / s->args[2].n;
	args[1] = c % s->args[1].n; c = c / s->args[1].n;
	args[0] = c;
	break;
	}
    return(1);
}  /* decode_int */

/*************
 *
 *   decode_and_p_clause()
 *
 *   Decode and print a propositional clause.
 *
 *************/

static void decode_and_p_clause(int c[], int n)
{
    char *name, *sign;
    int arity, args[MAX_ARITY], i, j;

    for (i = 0; i < n; i++) {
	sign = (c[i] < 0 ? "-" : "");
	decode_int(abs(c[i]), &name, &arity, args);
	printf("  %s%s", sign, name);
	for (j=0; j<arity; j++)
	    printf("%d", args[j]);
	}
    printf("\n");
}  /* decode_and_p_clause */

/*************
 *
 *   p_relational_clause()
 *
 *************/

void p_relational_clause(struct rel_clause *c)
{
    int v, sign, i, j, arity;
    
    for (i = 0; i < c->num_lits; i++) {
	int v, sign;

	sign = c->lits[i].sign;
	printf("%s%s(", (sign==1?"":"-"), Symbols[c->lits[i].functor].name);
	arity = Symbols[c->lits[i].functor].arity;
	for (j = 0; j < arity; j++) {
	    v = c->lits[i].a[j];
	    if (v >= 0)
		printf("v%d", v);  /* variable */
	    else
		printf("%d", -(v+100));  /* element */
	    printf("%s", j == arity-1 ? ") " : ",");
	    }
	}
    printf("\n");
    fflush(stdout);
}  /* p_relational_clause */

/*************
 *
 *   sort_clause() -- propositional clause, by atom.
 *
 *************/

static void sort_clause(int c[], int n)
{
    int i, j, temp;

    for (i = 0; i < n-1; i++)
	for (j = i+1; j < n; j++)
	    if (abs(c[j]) < abs(c[i])) {
		temp = c[i]; c[i] = c[j]; c[j] = temp;
		}

}  /* sort_clause */

/*************
 *
 *   unit_subsumed()
 *
 *   Is a given propositional clause subsumed by anything in the Units array?
 *
 *************/

static int unit_subsumed(int c[], int n)
{
    int i, a, sign;
    for (i = 0; i < n; i++) {
	a = abs(c[i]);
	sign = (c[i] < 0) ? -1 : 1;
	if (Units[a] == sign)
	    return(1);
	}
    return(0);
}  /* unit_subsumed */

/*************
 *
 *   tautology() -- propositional; assume sorted by atom.
 *
 *************/

static int tautology(int c[], int n)
{
    int i;
    for (i = 0; i < n-1; i++)
	if (c[i] == -c[i+1])
	    return(1);
    return(0);
}  /* tautology */

/*************
 *
 *   merge_lits() -- propositional; assume sorted by atom.
 *
 *************/

static int merge_lits(int c[], int n)
{
    int i, j, m;
    m = n;
    for (i = 0; i < m-1; i++)
	if (c[i] == c[i+1]) {
	    for (j = i+1; j < m-1; j++)
		c[j] = c[j+1];
	    m--;
	    i--;
	    }
    return(m);
}  /* merge_lits */

/*************
 *
 *   unit_delete()
 *
 *   Given a propositional clause, do unit resolution with Units array.
 *
 *************/

int unit_delete(int c[], int n)
{
    int i, j, l, a, v, m;
    m = n;
    for (i = 0; i < m; i++) {
	l = c[i]; a = abs(l); v = Units[a];
	if ((v == -1 && l > 0) || (v == 1 && l < 0)) {
	    for (j = i; j < m-1; j++)
		c[j] = c[j+1];
	    m--;
	    i--;
	    }
	}

    return(m);
}  /* unit_delete */

/*************
 *
 *   add_clause()
 *
 *   Given a propositional clause, do some preprocessing, and if it
 *   passes the tests, send it to insert_dp_clause(), which is part
 *   of the interface to the DP procedure.
 *
 *************/

static void add_clause(int c[], int n)
{
    Clauses_generated++;
#ifdef PRINT_CLAUSES
	decode_and_p_clause(c,n);
#endif
    n = unit_delete(c, n);
    if (n == 0) {
	printf("add_clause: propositionally unsatisfiable");
	unsatisfiable_exit();
        }
    sort_clause(c, n);
    if (!unit_subsumed(c, n) && !tautology(c, n)) {
	n = merge_lits(c, n);
	if (n == 1)
	    Units[abs(c[0])] = c[0] < 0 ? -1 : 1;
	insert_dp_clause(c, n);
	Clauses_inserted++;
	}
}  /* add_clause */

/*************
 *
 *   add_2clause()
 *
 *   Given 2 literals, make a 2-clause and send it to add_clause.
 *
 *************/

static void add_2clause(int a, int b)
{
    int c[2];
    c[0] = a; c[1] = b;
    add_clause(c, 2);
}  /* add_2clause */

/*************
 *
 *   add_unit()
 *
 *   Build a literal and send it to add_clause.
 *
 *************/

static void add_unit(int sign, struct symbol *s, int a0, int a1, int a2, int a3)
{
    int c[1];

    if (a0 >= s->args[0].n || a1 >= s->args[1].n || a2 >= s->args[2].n || a3 >= s->args[3].n)
	abend("add_unit: element out of domain");
    if (abs(sign) != 1)
	abend("add_unit: sign must be 1 or -1");

#if 0
    printf("add_unit: %s%s %d %d %d %d.\n", 
	   sign==1?"":"-", Symbols[functor].name, a0, a1, a2, a3);
#endif
    
    switch(s->arity) {
      case 0: c[0] = sign * ATOM0(s); break;
      case 1: c[0] = sign * ATOM1(s, a0); break;
      case 2: c[0] = sign * ATOM2(s, a0, a1); break;
      case 3: c[0] = sign * ATOM3(s, a0, a1, a2); break;
      case 4: c[0] = sign * ATOM4(s, a0, a1, a2, a3); break;
	}
    add_clause(c, 1);
}  /* add_unit */

/*
 * The following set of routines assert various properties of functions.
 * Recall that an n-ary function is handled as an (n+1)-ary relation,
 * so we have to say that the function is well-defined and closed.
 * Similar routines can be used to assert that functions are 1-1, onto,
 * cancellative, etc.
 */

/*************
 *
 *   well_defined_0() - constant cannot have 2 values.
 *
 *************/

static void well_defined_0(struct symbol *s)
{
    int y, z;
    int n0 = s->args[0].n;
    
    /* -By | -Bz, for y < z */
    for (y = 0; y < n0; y++)
	for (z = y+1; z < n0; z++)
	    add_2clause( - ATOM1(s,y), - ATOM1(s,z));
}  /* well_defined_0 */

/*************
 *
 *   well_defined_1()
 *
 *************/

static void well_defined_1(struct symbol *s)
{
    int x, y, z;
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    
    /* -Bxy | -Bxz, for y < z */
    for (x = 0; x < n0; x++)
	for (y = 0; y < n1; y++)
	    for (z = y+1; z < n1; z++)
		add_2clause( - ATOM2(s,x,y), - ATOM2(s,x,z));
}  /* well_defined_1 */

/*************
 *
 *   well_defined_2()
 *
 *************/

static void well_defined_2(struct symbol *s)
{
    int u, x, y, z;
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;

    /* -Buxy | -Buxz, for y < z */
    for (u = 0; u < n0; u++)
	for (x = 0; x < n1; x++)
	    for (y = 0; y < n2; y++)
		for (z = y+1; z < n2; z++)
		    add_2clause( - ATOM3(s,u,x,y), - ATOM3(s,u,x,z));
}  /* well_defined_2 */

/*************
 *
 *   well_defined_3()
 *
 *************/

static void well_defined_3(struct symbol *s)
{
    int v, u, x, y, z;
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;
    int n3 = s->args[3].n;

    /* -Bvuxy | -Bvuxz, for y < z */
    for (v = 0; v < n0; v++)
	for (u = 0; u < n1; u++)
	    for (x = 0; x < n2; x++)
		for (y = 0; y < n3; y++)
		    for (z = y+1; z < n3; z++)
			add_2clause( - ATOM4(s,v,u,x,y), - ATOM4(s,v,u,x,z));
}  /* well_defined_3 */

/*************
 *
 *   closed_0() - value of constant is one of the domain elements.
 *
 *************/

static void closed_0(struct symbol *s)
{
    int x, c[MAX_LITERALS];
    int n0 = s->args[0].n;

    /* B0 | B1 | .. | Bn-1 */
    for (x = 0; x < n0; x++)
	c[x] = ATOM1(s,x);
    add_clause(c, n0);
}  /* closed_0 */

/*************
 *
 *   closed_1()
 *
 *************/

static void closed_1(struct symbol *s)
{
    int x, y, c[MAX_LITERALS];
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;

    /* Bx0 | Bx1 | .. | Bxn-1, 0<=x<=n-1 */
    for (x = 0; x < n0; x++) {
	for (y = 0; y < n1; y++)
	    c[y] = ATOM2(s,x,y);
	add_clause(c, n1);
	}
}  /* closed_1 */

/*************
 *
 *   closed_2()
 *
 *************/

static void closed_2(struct symbol *s)
{
    int x, y, z, c[MAX_LITERALS];
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;

    /* Bxy0 | Bxy1 | .. | Bxyn-1, 0<=x<=n-1, 0<=y<=n-1 */
    for (x = 0; x < n0; x++)
	for (y = 0; y < n1; y++) {
	    for (z = 0; z < n2; z++)
		c[z] = ATOM3(s,x,y,z);
	    add_clause(c, n2);
	    }
}  /* closed_2 */

/*************
 *
 *   closed_2_holey()
 *
 *   A holey function (binary only) is a kind of partial function.
 *   same_hole(x,y) -> f(x,y) is not defined.
 *
 *************/

static void closed_2_holey(struct symbol *s, struct symbol *h)
{
    int x, y, z, c[MAX_LITERALS];
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;

    /* Bxy0 | Bxy1 | .. | Bxyn-1 | same_hole(x,y); 0<=x<=n-1, 0<=y<=n-1 */

    for (x = 0; x < n0; x++)
	for (y = 0; y < n1; y++) {
	    for (z = 0; z < n2; z++)
		c[z] = ATOM3(s,x,y,z);
	    c[n2] = ATOM2(h, x, y);
	    add_clause(c, n2+1);
	    }

    /* Holes are empty. */
    /* -same_hole(x,y) | -B(x,y,z). */
    for (x = 0; x < n0; x++)
	for (y = 0; y < n1; y++)
	    for (z = 0; z < n2; z++) {
		add_2clause( - ATOM2(h, x, y),
			     - ATOM3(s, x, y, z));
		}
}  /* closed_2_holey */

/*************
 *
 *   closed_3()
 *
 *************/

static void closed_3(struct symbol *s)
{
    int x, y, z, u, c[MAX_LITERALS];
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;
    int n3 = s->args[3].n;

    /* Bxyz0 | Bxyz1 | .. | Bxyzn-1, 0<=x<=n-1, 0<=y<=n-1, 0<=z<=n-1 */
    for (x = 0; x < n0; x++)
	for (y = 0; y < n1; y++)
	    for (z = 0; z < n2; z++) {
		for (u = 0; u < n3; u++)
		    c[u] = ATOM4(s,x,y,z,u);
		add_clause(c, n3);
		}
}  /* closed_3 */

/*************
 *
 *   well_defined()
 *
 *************/

void well_defined(struct symbol *s)
{
    switch(s->arity) {
      case 1: well_defined_0(s); break;
      case 2: well_defined_1(s); break;
      case 3: well_defined_2(s); break;
      case 4: well_defined_3(s); break;
      default: abend("well_defined, bad arity");
	}
}  /* well_defined */

/*************
 *
 *   closed()
 *
 *************/

void closed(struct symbol *s)
{
    switch(s->arity) {
      case 1: closed_0(s); break;
      case 2: closed_1(s); break;
      case 3: closed_2(s); break;
      case 4: closed_3(s); break;
      default: abend("closed, bad arity");
	}
}  /* closed */

/*************
 *
 *   one_one()
 *
 *************/

static void one_one(struct symbol *s)
{
    int x, y, z;
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;

    /* -Bxy | -Bzy, x < z */
    for (y = 0; y < n1; y++)
	for (x = 0; x < n0; x++)
	    for (z = x+1; z < n0; z++)
		add_2clause( - ATOM2(s,x,y), - ATOM2(s,z,y));

}  /* one_one */

/*************
 *
 *   onto()
 *
 *************/

static void onto(struct symbol *s)
{
    int x, y, c[MAX_LITERALS];
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;

    /* B0x | B1x | .. | Bn-1x, 0<=x<=n-1 */

    for (x = 0; x < n1; x++) {
	for (y = 0; y < n0; y++)
	    c[y] = ATOM2(s,y,x);
	add_clause(c, n1);
	}
}  /* onto */

/*************
 *
 *   left_cancel()
 *
 *************/

static void left_cancel(struct symbol *s)
{
    int u, x, y, z;
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;

    /* -Buyx | -Buzx, for y < z */

    for (u = 0; u < n0; u++)
	for (x = 0; x < n2; x++)
	    for (y = 0; y < n1; y++)
		for (z = y+1; z < n1; z++)
		    add_2clause( - ATOM3(s,u,y,x), - ATOM3(s,u,z,x));

}  /* left_cancel */

/*************
 *
 *   right_cancel()
 *
 *************/

static void right_cancel(struct symbol *s)
{
    int u, x, y, z;
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;

    /* -Byux | -Bzux, for y < z */

    for (u = 0; u < n1; u++)
	for (x = 0; x < n2; x++)
	    for (y = 0; y < n0; y++)
		for (z = y+1; z < n0; z++)
		    add_2clause( - ATOM3(s,y,u,x), - ATOM3(s,z,u,x));

}  /* right_cancel */

/*************
 *
 *   left_onto()
 *
 *************/

static void left_onto(struct symbol *s)
{
    int x, y, z, c[MAX_LITERALS];
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;

    /* Bx0y | Bx1y | .. | Bxn-1y, 0<=x<=n-1, 0<=y<=n-1 */

    for (x = 0; x < n0; x++)
	for (y = 0; y < n2; y++) {
	    for (z = 0; z < n1; z++)
		c[z] = ATOM3(s,x,z,y);
	    add_clause(c, n1);
	    }
}  /* left_onto */

/*************
 *
 *   left_onto_holey()
 *
 *************/

static void left_onto_holey(struct symbol *s, struct symbol *h)
{
    int x, y, z, c[MAX_LITERALS];
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;

    /* Bx0y | Bx1y | .. | Bxn-1y | same_hole(x,y); 0<=x<=n-1, 0<=y<=n-1 */

    for (x = 0; x < n0; x++)
	for (y = 0; y < n2; y++) {
	    for (z = 0; z < n1; z++)
		c[z] = ATOM3(s,x,z,y);
	    c[n1] = ATOM2(h, x, y);
	    add_clause(c, n1+1);
	    }

    /* -same_hole(x,y) | -B(x,z,y). */
    for (x = 0; x < n0; x++)
	for (y = 0; y < n2; y++)
	    for (z = 0; z < n1; z++) {
		add_2clause( - ATOM2(h, x, y),
			     - ATOM3(s, x, z, y));
		}

}  /* left_onto_holey */

/*************
 *
 *   right_onto()
 *
 *************/

static void right_onto(struct symbol *s)
{
    int x, y, z, c[MAX_LITERALS];
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;

    /* B0xy | B1xy | .. | Bn-1xy, 0<=x<=n-1, 0<=y<=n-1 */

    for (x = 0; x < n1; x++)
	for (y = 0; y < n2; y++) {
	    for (z = 0; z < n0; z++)
		c[z] = ATOM3(s,z,x,y);
	    add_clause(c, n0);
	    }
}  /* right_onto */

/*************
 *
 *   right_onto_holey()
 *
 *************/

static void right_onto_holey(struct symbol *s, struct symbol *h)
{
    int x, y, z, c[MAX_LITERALS];
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;

    /* B0xy | B1xy | .. | Bn-1xy | same_hole(x,y); 0<=x<=n-1, 0<=y<=n-1 */

    for (x = 0; x < n1; x++)
	for (y = 0; y < n2; y++) {
	    for (z = 0; z < n0; z++)
		c[z] = ATOM3(s,z,x,y);
	    c[n0] = ATOM2(h, x,y);
	    add_clause(c, n0+1);
	    }

    /* -same_hole(x,y) | -B(z,x,y). */
    for (x = 0; x < n1; x++)
	for (y = 0; y < n2; y++)
	    for (z = 0; z < n0; z++) {
		add_2clause( - ATOM2(h, x, y),
			     - ATOM3(s, z, x, y));
		}

}  /* right_onto_holey */

/*************
 *
 *   process_variables()
 *
 *   Check that all occurrences of each variable are in a position of the
 *   same sort, collect the domain sizes of the variables into an array,
 *   and return the biggest variable.
 *
 *************/

static int process_variables(struct rel_lit *lits, int num_lits, int domains[])
{
    int i, j, biggest_var, v;
    struct symbol *s;

    for (i = 0; i < MAX_VARS; i++)
	domains[i] = -1;
    biggest_var = -1;

    for (i = 0; i < num_lits; i++) {
	s = Symbols + lits[i].functor;
	for (j = 0; j < s->arity; j++) {
	    v = lits[i].a[j];
	    if (v >= 0) {
		if (v > biggest_var) {
		    if (v >= MAX_VARS)
			abend("process_variables, variable too big.");
		    biggest_var = v;
		    }
		if (domains[v] == -1)
		    domains[v] = s->args[j].sort;
		else if (domains[v] != s->args[j].sort)
		    abend("process_variables, badly sorted clause");
		}
	    }
	}

    /* Now replace sort indexes with sort sizes. */

    for (i = 0; i <= biggest_var; i++)
	domains[i] = Sorts[domains[i]].n;

    return(biggest_var);
}  /* process_variables */

/*************
 *
 *   generate_prop_clauses()
 *
 *   Given a flattened general clause, generate the corresponding
 *   ground clauses over the domain and send them to add_clause.
 *
 *   Note that limits on the arity and on the number of variables
 *   are wired into the code.
 *
 *************/

static void generate_prop_clauses(struct rel_clause *c)
{
    int v0, v0_lim, v1, v1_lim, v2, v2_lim, v3, v3_lim, v4, v4_lim, v5, v5_lim;
    int v6, v6_lim, v7, v7_lim, v8, v8_lim, v9, v9_lim, v10, v10_lim, v11, v11_lim;
    int biggest_var, val[MAX_VARS], i, j, sign, domains[MAX_VARS];
    int a0, a1, a2, a3;
    int prop_clause[MAX_LITERALS];
    struct symbol *s;
    struct rel_lit *lits = c->lits;
    int num_lits = c->num_lits;
	
    biggest_var = process_variables(lits, num_lits, domains);

#ifdef PRINT_CLAUSES
    printf("  biggest_var=%d: ", biggest_var);
    p_relational_clause(c);
#endif

    /* Now domains[] has the domain sizes for all the variables.
     * Fill in the remaining slots with 1 so that the nested loops
     * below work correctly.
     */

    for (i = biggest_var+1; i < MAX_VARS; i++)
	domains[i] = 1;

    v0_lim = domains[0]; v1_lim = domains[1]; v2_lim = domains[2];
    v3_lim = domains[3]; v4_lim = domains[4]; v5_lim = domains[5];
    v6_lim = domains[6]; v7_lim = domains[7]; v8_lim = domains[8];
    v9_lim = domains[9]; v10_lim = domains[10]; v11_lim = domains[11];

    for (v0 = 0; v0 < v0_lim; v0++) { val[0] = v0;
    for (v1 = 0; v1 < v1_lim; v1++) { val[1] = v1;
    for (v2 = 0; v2 < v2_lim; v2++) { val[2] = v2;
    for (v3 = 0; v3 < v3_lim; v3++) { val[3] = v3;
    for (v4 = 0; v4 < v4_lim; v4++) { val[4] = v4;
    for (v5 = 0; v5 < v5_lim; v5++) { val[5] = v5;
    for (v6 = 0; v6 < v6_lim; v6++) { val[6] = v6;
    for (v7 = 0; v7 < v7_lim; v7++) { val[7] = v7;
    for (v8 = 0; v8 < v8_lim; v8++) { val[8] = v8;
    for (v9 = 0; v9 < v9_lim; v9++) { val[9] = v9;
    for (v10 = 0; v10 < v10_lim; v10++) { val[10] = v10;
    for (v11 = 0; v11 < v11_lim; v11++) { val[11] = v11;

    for (i = 0; i < num_lits; i++) {
	s = Symbols + lits[i].functor;

	switch (s->arity) {  /* note no breaks */
	  case 4:
	    a3 = lits[i].a[3];
	    a3 = (a3 < 0 ? -(a3+100) : val[a3]);
	  case 3:
	    a2 = lits[i].a[2];
	    a2 = (a2 < 0 ? -(a2+100) : val[a2]);
	  case 2:
	    a1 = lits[i].a[1];
	    a1 = (a1 < 0 ? -(a1+100) : val[a1]);
	  case 1:
	    a0 = lits[i].a[0];
	    a0 = (a0 < 0 ? -(a0+100) : val[a0]);
	    }

	sign = lits[i].sign;
	switch (s->arity) {
	  case 0: prop_clause[i] = sign * ATOM0(s); break;
	  case 1: prop_clause[i] = sign * ATOM1(s,a0); break;
	  case 2: prop_clause[i] = sign * ATOM2(s,a0,a1); break;
	  case 3: prop_clause[i] = sign * ATOM3(s,a0,a1,a2); break;
	  case 4: prop_clause[i] = sign * ATOM4(s,a0,a1,a2,a3); break;
	    }
	}

    add_clause(prop_clause, i);

    }}}}}}}}}}}}

}  /* generate_prop_clauses */

/*************
 *
 *   add_clauses_for_experiment()
 *
 *   Add special-purpose constraints determined by the Experiment variable.
 *
 *************/

static void add_clauses_for_experiment()
{
    int i, j, k, ii, jj, kk, m, id, n;
    struct symbol *s;

    if (Num_sorts != 1)
	abend("experiment requires one-sorted logic");

    n = Sorts[0].n;

    if (Experiment == 1) {
	/* quasigroup constraint:  x*n != z, when z < x-1. */

	id = str_to_id("f");
	s = Symbols+id;

	printf("Adding basic QG constraints on last column of f.\n");

	for (i = 0; i < n; i++)
	    for (k = 0; k < i-1; k++) {
		add_unit(-1, s, i, n-1, k, -1);
		}

	}
    else if (Experiment == 2) {

	/* Diagonals count up (mod n). */
	
	id = str_to_id("f");
	s = Symbols+id;
	
	printf("Diagonals count up mod n.\n");
	
	for (i = 0; i < n; i++)
	    for (j = 0; j < n; j++)
		for (k = 0; k < n; k++) {
		    ii = (i+1)%n;  jj = (j+1)%n; kk = (k+1)%n;
		    add_2clause( - ATOM3(s, i, j, k),
				   ATOM3(s, ii, jj, kk));
		    }
	}
    else if (Experiment >= 11 && Experiment <= 19) {

	int x = Experiment-10;  /* x is size of special square in SE corner */

	printf("Special square (%d, %d).\n", n, x);

	m = n-x;  /* m is index of first element in special square. */

	id = str_to_id("f");
	s = Symbols+id;

	/* first m of last x rows and columns count up (mod m). */

	for (j = m; j < n; j++)
	    for (i = 0; i < m; i++)
		for (k = 0; k < m; k++) {
		    add_2clause( - ATOM3(s, i, j, k),
				   ATOM3(s, (i+1)%m, j, (k+1)%m));
		}

	for (i = m; i < n; i++)
	    for (j = 0; j < m; j++)
		for (k = 0; k < m; k++) {
		    add_2clause( - ATOM3(s, i, j, k),
				   ATOM3(s, i, (j+1)%m, (k+1)%m));
		    }

	/* Diagonals, except last x rows and columns. */
	
	for (i = 0; i < m; i++)
	    for (j = 0; j < m; j++) {
		if (i != j) {
		    ii = (i+1)%m;  jj = (j+1)%m;
		    for (k = 0; k < m; k++) {
			kk = (k+1)%m;
			/*
			printf("Adding -%s %d %d %d | %s %d %d %d.\n",
			       s->name, i, j, k, s->name, ii, jj, kk);
			*/
			add_2clause( - ATOM3(s, i, j, k),
				       ATOM3(s, ii, jj, kk));

			}
		    for (k = m; k < n; k++)
			add_2clause( - ATOM3(s, i, j, k),
				       ATOM3(s, ii, jj, k));
		    }
		}
	}
}  /* add_clauses_for_experiment */

/*************
 *
 *   make_holey_relation()
 *
 *   A holey relation is symmetric and transitive.  Assume the units
 *   to generate the relation have already been asserted.
 *   This routine asserts the rules to make it symmetric and transitive
 *   (which will in turn generate the rest of the appropriate positive units).
 *   Then make all other pairs false.
 *
 *************/

void make_holey_relation(int id)
{
    int i, j, k, c[3], n;
    struct symbol *s;

    if (Num_sorts != 1)
	abend("make_holey_relation: requires one-sorted logic");

    n = Sorts[0].n;

    /* Assume the units to generate the relation
     * have already been asserted.
     */

    s = Symbols+id;
    
    for (i = 0; i < n; i++)  /* symmetric */
	for (j = 0; j < n; j++)
	    add_2clause( - ATOM2(s,i,j), ATOM2(s,j,i));
    for (i = 0; i < n; i++)  /* transitive */
	for (j = 0; j < n; j++)
	    for (k = 0; k < n; k++) {
		c[0] = - ATOM2(s,i,j);
		c[1] = - ATOM2(s,j,k);
		c[2] =   ATOM2(s,i,k);
		add_clause(c,3);
		}
    /* If not true, make it false. */
    for (i = 0; i < n; i++)
	for (j = 0; j < n; j++) {
	    if (Units[ATOM2(s, i, j)] == 0)
		add_unit(-1, s, i, j, -1, -1);
	    }
}  /* make_holey_relation */

/*************
 *
 *   number_variables()
 *
 *   Given a first-order (flat) clause with string arguments
 *   (vars and domain elements), assign vars positive ints,
 *   and domain elements negative ints.
 *
 *************/

static int number_variables(struct rel_clause *c)
{
    char *vnames[MAX_VARS], *s;

    int i, j, k, arity, el;
    int max_assigned_element = -1;

    for (i = 0; i < MAX_VARS; i++)
	vnames[i] = NULL;

    for (i = 0; i < c->num_lits; i++) {
	arity = Symbols[c->lits[i].functor].arity;
	for (j = 0; j < arity; j++) {
	    s = c->lits[i].ai[j];
	    if (str_int(s, &el)) {  /* Domain element */
		if (el < 0 || el >= Symbols[c->lits[i].functor].args[j].n) {
		    printf("\nThe bad element is %d.\n", el);
		    abend("number_variables, element out of range");
		    }
		c->lits[i].a[j] = -(el+100);
		if (el > max_assigned_element)
		    max_assigned_element = el;
		}
	    else {  /* Variable */
		for (k = 0; k < MAX_VARS && vnames[k] && !STR_IDENT(s,vnames[k]); k++);
		if (k == MAX_VARS) {
		    printf("\nMaximum number of variables is %d.\n", MAX_VARS);
		    abend("number_variables, too many variables");
		    }
		else {
		    c->lits[i].a[j] = k;
		    if (vnames[k] == NULL)
			vnames[k] = s;
		    }
		}
	    }
	}
    return(max_assigned_element);
}  /* number_variables */

/*************
 *
 *   read_ordinary_clause()
 *
 *************/

void read_ordinary_clause(int id)
{
    char name[MAX_NAME];

    scanf("%s", name);
    while (!STR_IDENT(name, "."))
	scanf("%s", name);
    
}  /* read_ordinary_clause */

/*************
 *
 *   read_relational_clause()
 *
 *************/

static int read_relational_clause(int cid)
{
    struct rel_clause *c;
    struct rel_lit *l;
    int i, j, num_lits, arity, fid;
    char name[MAX_NAME];
    char *functor;
    struct symbol *s = NULL;
    int max_assigned_element;

    if (Num_rel_clauses == MAX_CLAUSES)
	abend("read_relational_clause: too many first_order clauses");
    c = Clauses + Num_rel_clauses;
    Num_rel_clauses++;
    c->id = cid;

    scanf("%d", &num_lits);

    for (j = 0; j < num_lits; j++) {
	l = c->lits + j;
	scanf("%s", name);
	if (name[0] == '-') {
	    functor = name+1; l->sign = -1;
	    }
	else {
	    functor = name; l->sign = 1;
	    }
	fid = str_to_id(functor);
	if (fid == -1) {
	    printf("Bad name=%s\n", name);
	    abend("read_relational_clause, bad functor");
	    }
	    
	s = Symbols + fid;
	arity = s->arity;
	l->functor = fid;

	for (i = 0; i < arity; i++) {
	    scanf("%s", l->ai[i]);
	    if (str_to_id(l->ai[i]) != -1) {
		printf("\nBad argument=%s\n", l->ai[i]);
		abend("read_relational_clause, argument expected");
		}
	    }
	}
    c->num_lits = num_lits;
    max_assigned_element = number_variables(c);
    
    scanf("%s", name);
    if (!STR_IDENT(name, "."))
	abend("read_relational_clause, no period");

#ifdef PRINT_CLAUSES
    p_relational_clause(c);
#endif
    return(max_assigned_element);
}  /* read_relational_clause */

/*************
 *
 *   read_gen_clauses_sorted()
 *
 *   Read a file of first-order (flat) clauses, and generate propositional
 *   clauses.
 *
 *   For option -n, the form of the input is:
 *
 *   <type symbol arity properties>
 *   end_of_symbols
 *   <clauses, terminated with .>
 *   end_of_clauses
 *   <assignments>
 *   end_of_assignments
 *
 *   For example, 
 *
 *   relation = 2 equality
 *   relation same_hole 2 hole
 *   function f 3 quasigroup_holey
 *   end_of_symbols
 *   
 *   f v0 v0 v0   same_hole v0 v0 .
 *   -f v0 v1 v2  f v1 v2 v0 .
 *   -f v0 v1 v2   -f v3 v4 v2   -f v1 v0 v5 -f v4 v3 v5  = v0 v3 = v1 v4 .
 *   end_of_clauses
 *   
 *   same_hole 7 8
 *   same_hole 8 9
 *   f 0 9 6
 *   end_of_assignments
 *   
 *   For option -z, the form of the input is:
 *
 *************/

void read_gen_clauses_sorted(FILE *fp,
			     int domain_size,
			     int distinct_constants,
			     int experiment)
{
    int i, j, k, id, num_lits, arity, sign, size, n;
    int a[MAX_ARITY];
    char arg_types[MAX_ARITY][MAX_NAME];
    char type[MAX_NAME], name[MAX_NAME], properties[MAX_NAME];
    char *functor;
    struct symbol *s = NULL;
    struct rel_clause *c;
    struct rel_lit *l;
    int max_assigned_element = -1;
    
    Experiment = experiment;

    /* Sort declarations */

    if (domain_size > 0) {
	strcpy(Sorts[0].name, "univ");
	Sorts[0].n = domain_size;
	Num_sorts = 1;
	}
    else {
	scanf("%s", type);
	while (!STR_IDENT(type, "end_of_sorts")) {
	    scanf("%d", &size);
	    strcpy(Sorts[Num_sorts].name, type);
	    Sorts[Num_sorts].n = size;
	    printf("Sort %s, size=%d.\n", type, size);
	    Num_sorts++;
	    scanf("%s", type);
	    }
	}

    /* Symbol declarations */

    scanf("%s", type);
    while (!STR_IDENT(type, "end_of_symbols")) {
	if (!(STR_IDENT(type, "function") || STR_IDENT(type, "relation")))
	    abend("read_gen_clauses_sort, type must be function or relation");

	scanf("%s %d", name, &arity);
	if (arity > MAX_ARITY) {
	    abend("arity too great");
	    }

	if (domain_size > 0) {
	    for (i = 0; i < arity; i++)
		strcpy(arg_types[i], "univ");
	    }
	else {
	    for (i = 0; i < arity; i++)
		scanf("%s", arg_types[i]);
	    }

	id = declare_symbol_sorted(name, arity, arg_types);

	s = Symbols + id;
	strcpy(s->type, type);
	scanf("%s", properties);
	strcpy(s->properties, properties);

	printf("Symbol id=%d, type=%s, name=%s, arity=%d, ",
	       id, s->type, s->name, s->arity);
	for (i = 0; i < arity; i++)
	    printf("%s/%d, ", Sorts[s->args[i].sort].name, s->args[i].n);
	printf("props=%s, base=%d.\n", s->properties, s->base);
	       
	scanf("%s", type);
	}

    Greatest_atom = Next_base-1;
    Units = malloc((Greatest_atom+1) * sizeof(int));
    for (i = 1; i <= Greatest_atom; i++)
        Units[i] = 0;

    /* Clauses */

    scanf("%s", name);

    while (!STR_IDENT(name, "end_of_clauses")) {

	str_int(name, &id);
	scanf("%s", type);

	if (STR_IDENT(type, "relational")) {
	    i = read_relational_clause(id);
	    if (i > max_assigned_element)
		max_assigned_element = i;
	    }
	else if (STR_IDENT(type, "ordinary"))
	    read_ordinary_clause(id);
	else
	    abend("read_gen_clauses_sort, bad clause type");

	scanf("%s", name);
	}

    /* User's unit assignments. */

    scanf("%s", name);
    while (!STR_IDENT(name, "end_of_assignments")) {
	if (name[0] == '-') {
	    functor = name+1; sign = -1;
	    }
	else {
	    functor = name; sign = 1;
	    }
	id = str_to_id(functor);
	if (id == -1) {
	    printf("Bad name=%s, \n", functor);
	    abend("read_gen_clauses_sort, bad functor");
	    }
	    
	s = Symbols + id;
	arity = s->arity;

	for (i = 0; i < MAX_ARITY; i++)
	    a[i] = -1;

	for (i = 0; i < s->arity; i++)
	    scanf("%d", &(a[i]));

	add_unit(sign, s, a[0], a[1], a[2], a[3]);
	printf("Adding unit assigning %s%s", sign == -1 ? "-" : "", functor);
	for (i = 0; i < s->arity; i++)
	    printf(" %d", a[i]);
	printf("\n");

	if (STR_IDENT(s->type, "function") && s->arity == 1) {
	    s->assigned_value = a[0];
	    if (a[0] > max_assigned_element)
		max_assigned_element = a[0];
	    }

	scanf("%s", name);
	}

    /* Assignments for "distinct_constants" option. */

    if (distinct_constants) {
	for (id = 0; id < Num_symbols; id++) {
	    s = Symbols + id;
	    if (STR_IDENT(s->type, "function") &&
		s->arity == 1 && s->assigned_value == -1) {
		if (max_assigned_element+1 == domain_size)
		    printf("Cannot make constant %s distinct, domain is too small.\n",
			   s->name);
		else {
		    max_assigned_element++;
		    printf("Adding unit (distinct_constants) assigning %s %d\n",
			   s->name, max_assigned_element);
		    add_unit(1, s, max_assigned_element, -1, -1, -1);
		    }
		}
	    }
	}

    /* Unit assignments for order, equality relations, if present. */
	
    for (id = 0; id < Num_symbols; id++) {
	s = Symbols + id;
	if (STR_IDENT(s->type, "relation") && s->arity == 2) {
	    if (strstr(s->properties, "order")) {
		if (s->args[0].sort != s->args[1].sort)
		    abend("order relation must have args of same type");
		n = s->args[0].n;
		printf("Adding units for order relation %s.\n", s->name);
		for (i = 0; i < n; i++)
		    for (j = 0; j < n; j++) {
			if (i < j)
			    add_unit(1, s, i, j, -1, -1);
			else
			    add_unit(-1, s, i, j, -1, -1);
			}
		}
	    else if (strstr(s->properties, "equality")) {
		if (s->args[0].sort != s->args[1].sort)
		    abend("equality relation must have args of same type");
		n = s->args[0].n;
		printf("Adding units for equality relation %s.\n", s->name);
		for (i = 0; i < n; i++)
		    for (j = 0; j < n; j++) {
			if (i == j)
			    add_unit(1, s, i, j, -1, -1);
			else
			    add_unit(-1, s, i, j, -1, -1);
			}
		}
	    else if (strstr(s->properties, "hole")) {
		printf("Completing %s as a hole relation.\n", s->name);
		Hole = id;
		make_holey_relation(id);
		}
	    }
	}

    if (Experiment > 0)
	add_clauses_for_experiment();

    /* When processing generated clauses, forward unit subsumption and
     * forward unit deletion are applied, but the backward operations are not.
     * So to save memory, it's best to carefully order the generation of
     * clauses.  Because of unit-deletion, it's not clear which way is best.
     */

    for (i = 0; i < Num_rel_clauses; i++) {
	c = Clauses + i;
	if (c->num_lits <= 2) {
	    printf("Processing clause: ");
	    p_relational_clause(c);
	    generate_prop_clauses(c);
	    exit_if_over_time_limit();
	    }
	}

    /* Well-defined, closed, quasigroup, bijection. */
	
    for (i = 0; i < Num_symbols; i++) {
	s = Symbols + i;
	if (STR_IDENT(s->type, "function")) {
	    printf("function %s, arity %d, well-defined and closed.\n",
		   s->name, s->arity);

	    well_defined(s);

	    if (s->arity == 3 && strstr(s->properties, "holey")) {
		printf("  (closed with holes).\n");
		closed_2_holey(s, Symbols+Hole);
		}
	    else
		closed(s);

	    if (s->arity == 2 && strstr(s->properties, "bijection")) {
		printf("function %s, arity %d, bijection.\n", s->name, s->arity);
		one_one(s);
		onto(s);
		}
	    else if (s->arity == 2 && strstr(s->properties, "one_one")) {
		printf("function %s, arity %d, one_one.\n", s->name, s->arity);
		one_one(s);
		}
	    else if (s->arity == 2 && strstr(s->properties, "onto")) {
		printf("function %s, arity %d, onto.\n", s->name, s->arity);
		onto(s);
		}
	    else if (s->arity == 3 && strstr(s->properties, "quasigroup")) {
		printf("function %s, arity %d, quasigroup.\n", s->name, s->arity);
		left_cancel(s);
		right_cancel(s);
		if (strstr(s->properties, "holey")) {
		    printf("  (left and right onto with holes).\n");
		    left_onto_holey(s, Symbols+Hole);
		    right_onto_holey(s, Symbols+Hole);
		    }
		else {
		    left_onto(s);
		    right_onto(s);
		    }
		}
	    }
	}

    for (i = 0; i < Num_rel_clauses; i++) {
	c = Clauses + i;
	if (c->num_lits > 2) {
	    printf("Processing clause: ");
	    p_relational_clause(c);
	    generate_prop_clauses(c);
	    exit_if_over_time_limit();
	    }
	}

    printf("\n%d clauses were generated; %d of those survived the first stage\n"
	   "of unit preprocessing; there are %d atoms.\n", 
	   Clauses_generated, Clauses_inserted, Greatest_atom);
	   
    fflush(stdout);
}  /* read_gen_clauses_sort */

/* function_value_n() -- these routines find the value of functions at
 * given points.  It is assumed that we have a model including the
 * well-defined and closed properties.  The DP code knows the
 * model, so we call atom_value.  This is used mostly for printing
 * models.
 */

/*************
 *
 *   function_value_0()
 *
 *************/

int function_value_0(struct symbol *s)
{
    int b, k;
    int n0 = s->args[0].n;

    b = ATOM1(s, 0);
    for (k = 0; k < n0 && atom_value(b+k) != 1; k++);
    return(k == n0 ? -1 : k);
}  /* function_value_0 */

/*************
 *
 *   function_value_1()
 *
 *************/

int function_value_1(struct symbol *s, int i)
{
    int b, k;
    int n1 = s->args[1].n;

    b = ATOM2(s, i, 0);
    for (k = 0; k < n1 && atom_value(b+k) != 1; k++);
    return(k == n1 ? -1 : k);
}  /* function_value_1 */

/*************
 *
 *   function_value_2()
 *
 *************/

int function_value_2(struct symbol *s, int i, int j)
{
    int b, k;
    int n2 = s->args[2].n;

    b = ATOM3(s, i, j, 0);
    for (k = 0; k < n2 && atom_value(b+k) != 1; k++);
    return(k == n2 ? -1 : k);
}  /* function_value_2 */

/*************
 *
 *   function_value_3()
 *
 *************/

int function_value_3(struct symbol *s, int i, int j, int a)
{
    int b, k;
    int n3 = s->args[3].n;

    b = ATOM4(s, i, j, a, 0);
    for (k = 0; k < n3 && atom_value(b+k) != 1; k++);
    return(k == n3 ? -1 : k);
}  /* function_value_3 */

/*************
 *
 *   print_table_f0()
 *
 *************/

void print_table_f0(FILE *fp, struct symbol *s)
{
    fprintf(fp, "\n %s: %d\n", s->name, function_value_0(s));
}  /* print_table_f0 */

/*************
 *
 *   print_table_f1()
 *
 *************/

void print_table_f1(FILE *fp, struct symbol *s)
{
    int i;
    char *s1, *s2;
    int n0 = s->args[0].n;

    if (n0 <= 10) {
	s1 = "%2d";
	s2 = "--";
	}
    else {
	s1 = "%3d";
	s2 = "---";
	}
    
    fprintf(fp, "\n %s :\n", s->name);
    fprintf(fp, "       ");
    for (i = 0; i < n0; i++)
	fprintf(fp, s1, i);
    fprintf(fp, "\n    ---");
    for (i = 0; i < n0; i++)
	fprintf(fp, s2);
    fprintf(fp, "\n       ");
    for (i = 0; i < n0; i++)
	fprintf(fp, s1, function_value_1(s, i));
    fprintf(fp, "\n");
}  /* print_table_f1 */

/*************
 *
 *   print_table_f2()
 *
 *************/

void print_table_f2(FILE *fp, struct symbol *s)
{
    int i, j, v;
    char *s1, *s2, *s3;
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;

    if (n1 <= 10) {
	s1 = "%2d";
	s2 = "--";
	s3 = " -";
	}
    else {
	s1 = "%3d";
	s2 = "---";
	s3 = "  -";
	}
    
    fprintf(fp, "\n %s :\n", s->name);
    fprintf(fp, "      |");
    for (i = 0; i < n1; i++)
	fprintf(fp, s1, i);
    fprintf(fp, "\n    --+");
    for (i = 0; i < n1; i++)
	fprintf(fp, s2);
    fprintf(fp, "\n");

    for (i = 0; i < n0; i++) {
	fprintf(fp, "%5d |", i);
	for (j = 0; j < n1; j++) {
	    v = function_value_2(s, i, j);
	    if (v < 0)
		fprintf(fp, s3);
	    else
		fprintf(fp, s1, v);
	    }
	fprintf(fp, "\n");
	}
}  /* print_table_f2 */

/*************
 *
 *   print_table_r0()
 *
 *************/

void print_table_r0(FILE *fp, struct symbol *s)
{
    int v;
    char *s1;
    v = atom_value(ATOM0(s));
    switch(v) {
      case 0: s1 = "F"; break;
      case 1: s1 = "T"; break;
      case 2: s1 = "F"; break;
      default: s1 = "?"; break;
	}
    fprintf(fp, "\n %s: %s\n", s->name, s1);
}  /* print_table_r0 */

/*************
 *
 *   print_table_r1()
 *
 *************/

void print_table_r1(FILE *fp, struct symbol *s)
{
    int i, v;
    char *s1;
    int n0=s->args[0].n;

    fprintf(fp, "\n %s :\n", s->name);
    fprintf(fp, "       ");
    for (i = 0; i < n0; i++)
	fprintf(fp, "%2d", i);
    fprintf(fp, "\n    ---");
    for (i = 0; i < n0; i++)
	fprintf(fp, "--");
    fprintf(fp, "\n       ");
    for (i = 0; i < n0; i++) {
	v = atom_value(ATOM1(s, i));
	switch(v) {
	  case 0: s1 = "F"; break;
	  case 1: s1 = "T"; break;
	  case 2: s1 = "F"; break;
	  default: s1 = "?"; break;
	    }
	fprintf(fp, "%2s", s1);
	}
    fprintf(fp, "\n");
}  /* print_table_r1 */

/*************
 *
 *   print_table_r2()
 *
 *************/

void print_table_r2(FILE *fp, struct symbol *s)
{
    int i, j, v;
    char *s1;
    int n0=s->args[0].n;
    int n1=s->args[1].n;

    fprintf(fp, "\n %s :\n", s->name);
    fprintf(fp, "      |");
    for (i = 0; i < n1; i++)
	fprintf(fp, "%2d", i);
    fprintf(fp, "\n    --+");
    for (i = 0; i < n1; i++)
	fprintf(fp, "--");
    fprintf(fp, "\n");

    for (i = 0; i < n0; i++) {
	fprintf(fp, "%5d |", i);
	for (j = 0; j < n1; j++) {
	    v = atom_value(ATOM2(s, i, j));
	    switch(v) {
	      case 0: s1 = "F"; break;
	      case 1: s1 = "T"; break;
	      case 2: s1 = "F"; break;
	      default: s1 = "?"; break;
		}
	    fprintf(fp, "%2s", s1);
	    }
	fprintf(fp, "\n");
	}
}  /* print_table_r2 */

/*************
 *
 *   print_model()
 *
 *   Print the current model as a first-order model.  The DP code
 *   knows about the propositional model, so we call atom_value()
 *   to get the information to print the first-order model.
 *
 *   Assume unassigned atoms are true, because dp determines that
 *   we have a model when all remaining clauses are positive.
 *
 *   Don't print equality, order, or hole relations.
 *
 *************/

void print_model(FILE *fp)
{
    int i;
    struct symbol *s;
    static int count;

#if 1
    fprintf(fp, "\n======================= Model #%d at %.2f seconds:\n",
	    ++count, run_time() / 1000.);
#endif

    for (i = 0; i < Num_symbols; i++) {
	s = Symbols + i;
	if (STR_IDENT(s->type, "function")) {
	    switch (s->arity) {
	      case 1: print_table_f0(fp, s); break;
	      case 2: print_table_f1(fp, s); break;
	      case 3: print_table_f2(fp, s); break;
	      default: fprintf(fp, "Dimension %d table for %s not printed\n",
			       s->arity-1, s->name);
		}
	    }
	else if (!strstr(s->properties, "order") &&
		 !strstr(s->properties, "equality") &&
		 !strstr(s->properties, "hole")) {
	    
	    switch (s->arity) {
	      case 0: print_table_r0(fp, s); break;
	      case 1: print_table_r1(fp, s); break;
	      case 2: print_table_r2(fp, s); break;
	      default: fprintf(fp, "Dimension %d table for %s not printed\n",
			       s->arity, s->name);
		}
	    }
	}
    fprintf(fp, "end_of_model\n");
}  /* print_model */

/*************
 *
 *   print_table_f0_portable()
 *
 *************/

void print_table_f0_portable(FILE *fp, struct symbol *s)
{
    fprintf(fp, "\n        function(%s, [%d])",s->name,function_value_0(s));
}  /* print_table_f0_portable */

/*************
 *
 *   print_table_f1_portable()
 *
 *************/

void print_table_f1_portable(FILE *fp, struct symbol *s)
{
    int i;
    int n0 = s->args[0].n;

    fprintf(fp, "\n        function(%s(_), [", s->name);
    for (i = 0; i < n0; i++) {
	fprintf(fp, "%d", function_value_1(s, i));
	if (i < n0-1)
	    fprintf(fp, ", ");
	else
	    fprintf(fp, "])");
	}
}  /* print_table_f1_portable */

/*************
 *
 *   print_table_f2_portable()
 *
 *************/

void print_table_f2_portable(FILE *fp, struct symbol *s)
{
    int i, j, v;
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;

    fprintf(fp, "\n        function(%s(_,_), [\n", s->name);

    for (i = 0; i < n0; i++) {
	fprintf(fp, "                ");
	/* print row */
        for (j = 0; j < n1; j++) {
	    v = function_value_2(s, i, j);
	    if (j < n1-1)
	      fprintf(fp, "%d, ", v);
	    else
	      fprintf(fp, "%d", v);
	    }
	if (i < n0-1)
	    fprintf(fp, ",\n");
	else
	    fprintf(fp, " ])");
	}
}  /* print_table_f2_portable */

/*************
 *
 *   print_table_r0_portable()
 *
 *************/

void print_table_r0_portable(FILE *fp, struct symbol *s)
{
    int v;
    char *s1;
    v = atom_value(ATOM0(s));
    switch(v) {
      case 0: s1 = "0"; break;
      case 1: s1 = "1"; break;
      case 2: s1 = "0"; break;
      default: s1 = "?"; break;
	}
    fprintf(fp, "\n        predicate(%s, [%s])",s->name, s1);
}  /* print_table_r0_portable */

/*************
 *
 *   print_table_r1_portable()
 *
 *************/

void print_table_r1_portable(FILE *fp, struct symbol *s)
{
    int i, v;
    char *s1;
    int n0=s->args[0].n;

    fprintf(fp, "\n        predicate(%s(_), [", s->name);
    for (i = 0; i < n0; i++) {
	v = atom_value(ATOM1(s, i));
	switch(v) {
	  case 0: s1 = "0"; break;
	  case 1: s1 = "1"; break;
	  case 2: s1 = "0"; break;
	  default: s1 = "?"; break;
	    }
	fprintf(fp, "%s", s1);
	if (i < n0-1)
	    fprintf(fp, ", ");
	else
	    fprintf(fp, "])");
	}

    for (i = 0; i < n0; i++) {
	}
}  /* print_table_r1_portable */

/*************
 *
 *   print_table_r2_portable()
 *
 *************/

void print_table_r2_portable(FILE *fp, struct symbol *s)
{
    int i, j, v;
    char *s1;
    int n0=s->args[0].n;
    int n1=s->args[1].n;

    fprintf(fp, "\n        predicate(%s(_,_), [\n", s->name);

    for (i = 0; i < n0; i++) {
	fprintf(fp, "                ");
	/* print row */
        for (j = 0; j < n1; j++) {
	    v = atom_value(ATOM2(s, i, j));
	    switch(v) {
	      case 0: s1 = "0"; break;
	      case 1: s1 = "1"; break;
	      case 2: s1 = "0"; break;
	      default: s1 = "?"; break;
		}
	    if (j < n1-1)
	      fprintf(fp, "%s, ", s1);
	    else
	      fprintf(fp, "%s", s1);
	    }
	if (i < n0-1)
	    fprintf(fp, ",\n");
	else
	    fprintf(fp, " ])");
	}

}  /* print_table_r2_portable */

/*************
 *
 *   print_model_portable()
 *
 *   Print the current model as a first-order model.  The DP code
 *   knows about the propositional model, so we call atom_value()
 *   to get the information to print the first-order model.
 *
 *   Assume unassigned atoms are true, because dp determines that
 *   we have a model when all remaining clauses are positive.
 *
 *   Don't print equality, order, or hole relations.
 *
 *************/

void print_model_portable(FILE *fp)
{
    int i, domain_size;
    struct symbol *s;
    static int count;

    if (Num_sorts > 1) {
	print_model(fp);
	return;
	}

    domain_size = Sorts[0].n;

    fprintf(fp, "\n======================= Model #%d at %.2f seconds:\n",
	    ++count, run_time() / 1000.);
    
    fprintf(fp, "interpretation( %d, [\n", domain_size);

    for (i = 0; i < Num_symbols; i++) {
	s = Symbols + i;
	if (!strstr(s->properties, "equality")) {
	    if (STR_IDENT(s->type, "function")) {
		switch (s->arity) {
		  case 1: print_table_f0_portable(fp, s); break;
		  case 2: print_table_f1_portable(fp, s); break;
		  case 3: print_table_f2_portable(fp, s); break;
		  default: fprintf(fp, "Dimension %d table for %s not printed",
				   s->arity-1, s->name);
		    }
		}
	    else {
		
		switch (s->arity) {
		  case 0: print_table_r0_portable(fp, s); break;
		  case 1: print_table_r1_portable(fp, s); break;
		  case 2: print_table_r2_portable(fp, s); break;
		  default: fprintf(fp, "Dimension %d table for %s not printed\n",
				   s->arity, s->name);
		    }
		}
	    /* Following assumes that an equality relation is not last. */
	    if (i < Num_symbols-1)
		fprintf(fp, ",\n");
	    else
		fprintf(fp, "\n]).\n");
	    }
	}

    fprintf(fp, "end_of_model\n");
}  /* print_model_portable */

/*************
 *
 *   print_table_f0_ivy()
 *
 *************/

void print_table_f0_ivy(FILE *fp, struct symbol *s)
{
  fprintf(fp, "   ((%s . 0) . ((nil . %d)))\n",s->name,function_value_0(s));
}  /* print_table_f0_ivy */

/*************
 *
 *   print_table_f1_ivy()
 *
 *************/

void print_table_f1_ivy(FILE *fp, struct symbol *s)
{
  int i;
  int n0 = s->args[0].n;

  fprintf(fp, "   ((%s . 1) . (\n", s->name);
  for (i = 0; i < n0; i++) {
    fprintf(fp, "               ((%d) . %d)\n", i, function_value_1(s, i));
  }
  fprintf(fp, "              ))\n");
}  /* print_table_f1_ivy */

/*************
 *
 *   print_table_f2_ivy()
 *
 *************/

void print_table_f2_ivy(FILE *fp, struct symbol *s)
{
    int i, j, v;
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;

  fprintf(fp, "   ((%s . 2) . (\n", s->name);
  for (i = 0; i < n0; i++) {
    for (j = 0; j < n1; j++) {
      fprintf(fp, "               ((%d %d) . %d)\n", i, j, function_value_2(s, i, j));
    }
  }
  fprintf(fp, "              ))\n");
}  /* print_table_f2_ivy */

/*************
 *
 *   print_table_f3_ivy()
 *
 *************/

void print_table_f3_ivy(FILE *fp, struct symbol *s)
{
    int i, j, k, v;
    int n0 = s->args[0].n;
    int n1 = s->args[1].n;
    int n2 = s->args[2].n;

  fprintf(fp, "   ((%s . 3) . (\n", s->name);
  for (i = 0; i < n0; i++) {
    for (j = 0; j < n1; j++) {
      for (k = 0; k < n2; k++) {
	fprintf(fp, "               ((%d %d %d) . %d)\n",i,j,k,function_value_3(s, i, j, k));
      }
    }
  }
  fprintf(fp, "              ))\n");
}  /* print_table_f3_ivy */

/*************
 *
 *   print_table_r0_ivy()
 *
 *************/

void print_table_r0_ivy(FILE *fp, struct symbol *s)
{
  int v;
  char *s1;
  v = atom_value(ATOM0(s));
  switch(v) {
  case 0: s1 = "nil"; break;
  case 1: s1 = "t"; break;
  case 2: s1 = "nil"; break;
  default: s1 = "?"; break;
  }
  fprintf(fp, "   ((%s . 0) . ((nil . %s)))\n", s->name, s1);
}  /* print_table_r0_ivy */

/*************
 *
 *   print_table_r1_ivy()
 *
 *************/

void print_table_r1_ivy(FILE *fp, struct symbol *s)
{
  int i, v;
  char *s1;
  int n0=s->args[0].n;

  fprintf(fp, "   ((%s . 1) . (\n", s->name);
  for (i = 0; i < n0; i++) {
    v = atom_value(ATOM1(s, i));
    switch(v) {
    case 0: s1 = "nil"; break;
    case 1: s1 = "t"; break;
    case 2: s1 = "nil"; break;
    default: s1 = "nil"; break;
    }
    fprintf(fp, "               ((%d) . %s)\n", i, s1);
  }
  fprintf(fp, "              ))\n");
}  /* print_table_r1_ivy */

/*************
 *
 *   print_table_r2_ivy()
 *
 *************/

void print_table_r2_ivy(FILE *fp, struct symbol *s)
{
    int i, j, v;
    char *s1;
    int n0=s->args[0].n;
    int n1=s->args[1].n;

  fprintf(fp, "   ((%s . 2) . (\n", s->name);
  for (i = 0; i < n0; i++) {
    for (j = 0; j < n1; j++) {
      v = atom_value(ATOM2(s, i, j));
      switch(v) {
      case 0: s1 = "nil"; break;
      case 1: s1 = "t"; break;
      case 2: s1 = "nil"; break;
      default: s1 = "nil"; break;
      }
      fprintf(fp, "               ((%d %d) . %s)\n", i, j, s1);
    }
  }
  fprintf(fp, "              ))\n");
}  /* print_table_r2_ivy */

/*************
 *
 *   print_table_r3_ivy()
 *
 *************/

void print_table_r3_ivy(FILE *fp, struct symbol *s)
{
    int i, j, k, v;
    char *s1;
    int n0=s->args[0].n;
    int n1=s->args[1].n;
    int n2=s->args[2].n;

  fprintf(fp, "   ((%s . 3) . (\n", s->name);
  for (i = 0; i < n0; i++) {
    for (j = 0; j < n1; j++) {
      for (k = 0; k < n2; k++) {
	v = atom_value(ATOM3(s, i, j, k));
	switch(v) {
	case 0: s1 = "nil"; break;
	case 1: s1 = "t"; break;
	case 2: s1 = "nil"; break;
	default: s1 = "nil"; break;
	}
	fprintf(fp, "               ((%d %d %d) . %s)\n", i, j, k, s1);
      }
    }
  }
  fprintf(fp, "              ))\n");
}  /* print_table_r3_ivy */

/*************
 *
 *   print_table_r4_ivy()
 *
 *************/

void print_table_r4_ivy(FILE *fp, struct symbol *s)
{
    int i, j, k, l, v;
    char *s1;
    int n0=s->args[0].n;
    int n1=s->args[1].n;
    int n2=s->args[2].n;
    int n3=s->args[3].n;

  fprintf(fp, "   ((%s . 4) . (\n", s->name);
  for (i = 0; i < n0; i++) {
    for (j = 0; j < n1; j++) {
      for (k = 0; k < n2; k++) {
	for (l = 0; l < n3; l++) {
	  v = atom_value(ATOM4(s, i, j, k, l));
	  switch(v) {
	  case 0: s1 = "nil"; break;
	  case 1: s1 = "t"; break;
	  case 2: s1 = "nil"; break;
	  default: s1 = "nil"; break;
	  }
	  fprintf(fp, "               ((%d %d %d %d) . %s)\n", i, j, k, l, s1);
	}
      }
    }
  }
  fprintf(fp, "              ))\n");
}  /* print_table_r4_ivy */

/*************
 *
 *   print_model_ivy()
 *
 *   Print the current model as a first-order model.  The DP code
 *   knows about the propositional model, so we call atom_value()
 *   to get the information to print the first-order model.
 *
 *   Assume unassigned atoms are true, because dp determines that
 *   we have a model when all remaining clauses are positive.
 *
 *   Don't print equality, order, or hole relations.
 *
 *************/

void print_model_ivy(FILE *fp)
{
  int i, domain_size;
  struct symbol *s;
  static int count;

  if (Num_sorts > 1) {
    print_model(fp);
    return;
  }

  domain_size = Sorts[0].n;

  fprintf(fp, "\n======================= Model #%d at %.2f seconds:\n",
	  ++count, run_time() / 1000.);
  fprintf(fp, "\n;; BEGINNING OF IVY MODEL\n");

  if (domain_size == 1)
    fprintf(fp, "(0 .\n");
  else {
    int i;
    fprintf(fp, "((");
    for (i = 0; i < domain_size-1; i++)
      fprintf(fp, "%d ", i);
    fprintf(fp, ". %d) .\n (\n  (\n", domain_size-1);
  }

  for (i = 0; i < Num_symbols; i++) {
    s = Symbols + i;
    if (STR_IDENT(s->type, "relation") &&
	!strstr(s->properties, "equality")) {
      switch (s->arity) {
      case 0: print_table_r0_ivy(fp, s); break;
      case 1: print_table_r1_ivy(fp, s); break;
      case 2: print_table_r2_ivy(fp, s); break;
      case 3: print_table_r3_ivy(fp, s); break;
      case 4: print_table_r4_ivy(fp, s); break;
      default: abend("print_model_ivy, max relation arity is 4");
      }
    }
  }

  fprintf(fp, "  )\n  .\n  (\n");

  for (i = 0; i < Num_symbols; i++) {
    s = Symbols + i;
    if (STR_IDENT(s->type, "function")) {
      switch (s->arity) {
      case 1: print_table_f0_ivy(fp, s); break;
      case 2: print_table_f1_ivy(fp, s); break;
      case 3: print_table_f2_ivy(fp, s); break;
      case 4: print_table_f3_ivy(fp, s); break;
      default: abend("print_model_ivy, max function arity is 3");
      }
    }
  }

  fprintf(fp, "  )\n )\n)\n");
  fprintf(fp, ";; END OF IVY MODEL\n");
}  /* print_model_ivy */

