#!/usr/bin/env python

import nsysstats

class KHRPushPopSummary(nsysstats.Report):

    EVENT_TYPE_KHR_DEBUG_PUSHPOP_RANGE = 62

    usage = f"""{{SCRIPT}} -- OpenGL KHR_debug Range Summary

    No arguments.

    Output: All time values default to nanoseconds
        Time(%) : Percentage of "Total Time"
        Total Time : The total time used by all instances of this range
        Instances: The number of instances of this range
        Average : The average execution time of this range
        Minimum : The smallest execution time of this range
        Maximum : The largest execution time of this range
        StdDev : The standard deviation of execution time of this range
        Range : The name of the range

    This report provides a summary of OpenGL KHR_debug CPU PUSH/POP debug Ranges,
    and their execution times. Note that the "Time(%)" column
    is calculated using a summation of the "Total Time" column, and represents
    that range's percent of the execution time of the ranges listed, and not a
    percentage of the application wall or CPU execution time.
"""

    query = f"""
WITH
    maxts AS(
        SELECT max(max(start), max(end)) AS m
        FROM   KHR_DEBUG_EVENTS
    ),
    khrDebug AS (
        SELECT
            coalesce(khrDebugEvents.end, (SELECT m FROM maxts)) - khrDebugEvents.start AS duration,
            CASE
                WHEN sid.value IS NOT NULL
                    THEN sid.value
                ELSE khrDebugEvents.id
            END AS tag
        FROM
            KHR_DEBUG_EVENTS AS khrDebugEvents
        LEFT OUTER JOIN
            StringIds AS sid
            ON khrDebugEvents.textId == sid.id
        WHERE
            khrDebugEvents.eventClass == {EVENT_TYPE_KHR_DEBUG_PUSHPOP_RANGE}
    ),
    summary AS (
        SELECT
            tag AS name,
            sum(duration) AS total,
            count(*) AS num,
            avg(duration) AS avg,
            min(duration) AS min,
            max(duration) AS max,
            stdev(duration) AS stddev
        FROM
            khrDebug
        GROUP BY 1
    ),
    totals AS (
        SELECT sum(total) AS total
        FROM summary
    )

SELECT
    round(total * 100.0 / (SELECT total FROM totals), 1) AS "Time(%)",
    total AS "Total Time:dur_ns",
    num AS "Instances",
    round(avg, 1) AS "Average:dur_ns",
    min AS "Minimum:dur_ns",
    max AS "Maximum:dur_ns",
    round(stddev, 1) AS "StdDev:dur_ns",
    name AS "Range"
FROM
    summary
WHERE summary.total > 0
ORDER BY 2 DESC
;
"""
    table_checks = {
        'KHR_DEBUG_EVENTS':
            "{DBFILE} does not contain KHR Extension (KHR_DEBUG) data"
    }

if __name__ == "__main__":
    KHRPushPopSummary.Main()
