/* -*- Mode: C; tab-width: 4; indent-tabs-mode: nil; c-basic-offset: 4 -*- */
/*
 * libqmi-glib -- GLib/GIO based library to control QMI devices
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 *
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301 USA.
 *
 * Copyright (C) 2012 Lanedo GmbH
 * Copyright (C) 2015 Velocloud Inc.
 * Copyright (C) 2012-2017 Aleksander Morgado <aleksander@aleksander.es>
 */

#ifndef _LIBQMI_GLIB_QMI_ENUMS_WDS_H_
#define _LIBQMI_GLIB_QMI_ENUMS_WDS_H_

#if !defined (__LIBQMI_GLIB_H_INSIDE__) && !defined (LIBQMI_GLIB_COMPILATION)
#error "Only <libqmi-glib.h> can be included directly."
#endif

/**
 * SECTION: qmi-enums-wds
 * @title: WDS enumerations and flags
 *
 * This section defines enumerations and flags used in the WDS service
 * interface.
 */

#include <glib.h>

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Start Network' message */

/**
 * QmiWdsIpFamily:
 * @QMI_WDS_IP_FAMILY_UNKNOWN: unknown. Since: 1.18.
 * @QMI_WDS_IP_FAMILY_IPV4: IPv4.
 * @QMI_WDS_IP_FAMILY_IPV6: IPv6.
 * @QMI_WDS_IP_FAMILY_UNSPECIFIED: None specified.
 *
 * Type of IP family.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_IP_FAMILY_UNKNOWN     = 0,
    QMI_WDS_IP_FAMILY_IPV4        = 4,
    QMI_WDS_IP_FAMILY_IPV6        = 6,
    QMI_WDS_IP_FAMILY_UNSPECIFIED = 8
} QmiWdsIpFamily;

/**
 * qmi_wds_ip_family_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsTechnologyPreference:
 * @QMI_WDS_TECHNOLOGY_PREFERENCE_ALLOW_3GPP: 3GPP allowed.
 * @QMI_WDS_TECHNOLOGY_PREFERENCE_ALLOW_3GPP2: 3GPP2 allowed.
 *
 * Type of network allowed when trying to connect.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_TECHNOLOGY_PREFERENCE_ALLOW_3GPP  = 1 << 0,
    QMI_WDS_TECHNOLOGY_PREFERENCE_ALLOW_3GPP2 = 1 << 1
} QmiWdsTechnologyPreference;

/**
 * qmi_wds_technology_preference_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsExtendedTechnologyPreference:
 * @QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_CDMA: Use CDMA.
 * @QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_UMTS: Use UMTS.
 * @QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_EPC: Use EPC (LTE).
 * @QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_EMBMS: Use eMBMS.
 * @QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_MODEM_LINK_LOCAL: Use modem link-local.
 *
 * Type of network allowed when trying to connect.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_CDMA             = 32769,
    QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_UMTS             = 32772,
    QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_EPC              = 34944,
    QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_EMBMS            = 34946,
    QMI_WDS_EXTENDED_TECHNOLOGY_PREFERENCE_MODEM_LINK_LOCAL = 34952,
} QmiWdsExtendedTechnologyPreference;

/**
 * qmi_wds_extended_technology_preference_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsCallType:
 * @QMI_WDS_CALL_TYPE_LAPTOP: Laptop call.
 * @QMI_WDS_CALL_TYPE_EMBEDDED: Embedded call.
 *
 * Type of call to originate.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_CALL_TYPE_LAPTOP   = 0,
    QMI_WDS_CALL_TYPE_EMBEDDED = 1
} QmiWdsCallType;

/**
 * qmi_wds_call_type_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsCallEndReason:
 * @QMI_WDS_CALL_END_REASON_GENERIC_UNSPECIFIED: Unspecified reason.
 * @QMI_WDS_CALL_END_REASON_GENERIC_CLIENT_END: Client end.
 * @QMI_WDS_CALL_END_REASON_GENERIC_NO_SERVICE: No service.
 * @QMI_WDS_CALL_END_REASON_GENERIC_FADE: Fade.
 * @QMI_WDS_CALL_END_REASON_GENERIC_RELEASE_NORMAL: Release normal.
 * @QMI_WDS_CALL_END_REASON_GENERIC_ACCESS_ATTEMPT_IN_PROGRESS: Access attempt in progress.
 * @QMI_WDS_CALL_END_REASON_GENERIC_ACCESS_FAILURE: Access Failure.
 * @QMI_WDS_CALL_END_REASON_GENERIC_REDIRECTION_OR_HANDOFF: Redirection or handoff.
 * @QMI_WDS_CALL_END_REASON_GENERIC_CLOSE_IN_PROGRESS: Close in progress.
 * @QMI_WDS_CALL_END_REASON_GENERIC_AUTHENTICATION_FAILED: Authentication failed.
 * @QMI_WDS_CALL_END_REASON_GENERIC_INTERNAL_ERROR: Internal error.
 * @QMI_WDS_CALL_END_REASON_CDMA_LOCK: (CDMA) Phone is CDMA-locked.
 * @QMI_WDS_CALL_END_REASON_CDMA_INTERCEPT: (CDMA) Received intercept from the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_REORDER: (CDMA) Received reorder from the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_RELEASE_SO_REJECT: (CDMA) Received release from the BS, SO reject.
 * @QMI_WDS_CALL_END_REASON_CDMA_INCOMING_CALL: (CDMA) Received incoming call from the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_ALERT_STOP: (CDMA) Received alert stop from the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_ACTIVATION: (CDMA) Received end activation.
 * @QMI_WDS_CALL_END_REASON_CDMA_MAX_ACCESS_PROBES: (CDMA) Maximum access probes transmitted.
 * @QMI_WDS_CALL_END_REASON_CDMA_CCS_NOT_SUPPORTED_BY_BS: (CDMA) Concurrent service not supported by the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_NO_RESPONSE_FROM_BS: (CDMA) No response received from the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_REJECTED_BY_BS: (CDMA) Rejected by the BS.
 * @QMI_WDS_CALL_END_REASON_CDMA_INCOMPATIBLE: (CDMA) Concurrent services requested are incompatible.
 * @QMI_WDS_CALL_END_REASON_CDMA_ALREADY_IN_TC: (CDMA) Already in TC.
 * @QMI_WDS_CALL_END_REASON_CDMA_USER_CALL_ORIGINATED_DURING_GPS: (CDMA) Call originated during GPS.
 * @QMI_WDS_CALL_END_REASON_CDMA_USER_CALL_ORIGINATED_DURING_SMS: (CDMA) Call originated during SMS.
 * @QMI_WDS_CALL_END_REASON_CDMA_NO_SERVICE: (CDMA) No service.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_CONFERENCE_FAILED: (GSM/WCDMA) Call origination request failed.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_INCOMING_REJECTED: (GSM/WCDMA) Client rejected incoming call.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_NO_SERVICE: (GSM/WCDMA) No service.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_NETWORK_END: (GSM/WCDMA) Network ended the call.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_LLC_SNDCP_FAILURE: (GSM/WCDMA) LLC or SNDCP failure.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_INSUFFICIENT_RESOURCES: (GSM/WCDMA) Insufficient resources.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_TEMPORARILY_OUT_OF_ORDER: (GSM/WCDMA) Service option temporarily out of order.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_NSAPI_ALREADY_USED: (GSM/WCDMA) NSAPI already used.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_REGULAR_DEACTIVATION: (GSM/WCDMA) Regular PDP context deactivation.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_NETWORK_FAILURE: (GSM/WCDMA) Network failure.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_REATTACH_REQUIRED: (GSM/WCDMA) Reattach required.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_PROTOCOL_ERROR: (GSM/WCDMA) Protocol error.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPERATOR_DETERMINED_BARRING: (GSM/WCDMA) Operator-determined barring.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_APN: (GSM/WCDMA) Unknown or missing APN.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_PDP: (GSM/WCDMA) Unknown PDP address or type.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_GGSN_REJECT: (GSM/WCDMA) Activation rejected by GGSN.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_ACTIVATION_REJECT: (GSM/WCDMA) Activation rejected.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_NOT_SUPPORTED: (GSM/WCDMA) Service option not supported.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_UNSUBSCRIBED: (GSM/WCDMA) Service option not subscribed.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_QOS_NOT_ACCEPTED: (GSM/WCDMA) QoS not accepted.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_TFT_SEMANTIC_ERROR: (GSM/WCDMA) Semantic error in TFT operation.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_TFT_SYNTAX_ERROR: (GSM/WCDMA) Syntax error in TFT operation.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_PDP_CONTEXT: (GSM/WCDMA) Unknown PDP context.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_FILTER_SEMANTIC_ERROR: (GSM/WCDMA) Semantic error in packet filters.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_FILTER_SYNTAX_ERROR: (GSM/WCDMA) Syntax error in packet filters.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_PDP_WITHOUT_ACTIVE_TFT: (GSM/WCDMA) PDP context without TFT activated.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_INVALID_TRANSACTION_ID: (GSM/WCDMA) Invalid transaction ID.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_INCORRECT_SEMANTIC: (GSM/WCDMA) Message incorrect semantically.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_INVALID_MANDATORY_INFO: (GSM/WCDMA) Invalid mandatory information.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_TYPE_UNSUPPORTED: (GSM/WCDMA) Message type not implemented.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_TYPE_NONCOMPATIBLE_STATE: (GSM/WCDMA) Message not compatible with state.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_INFO_ELEMENT: (GSM/WCDMA) Information element unknown.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_CONDITIONAL_IE_ERROR: (GSM/WCDMA) Conditional IE error.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_AND_PROTOCOL_STATE_UNCOMPATIBLE: (GSM/WCDMA) Message and protocol state uncompatible.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_APN_TYPE_CONFLICT: (GSM/WCDMA) APN type conflict.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_NO_GPRS_CONTEXT: (GSM/WCDMA) No GPRS context.
 * @QMI_WDS_CALL_END_REASON_GSM_WCDMA_FEATURE_NOT_SUPPORTED: (GSM/WCDMA) Feature not supported.
 * @QMI_WDS_CALL_END_REASON_EVDO_CONNECTION_DENY_GENERAL_OR_BUSY: (EV-DO) Received Connection Deny (General or Network busy).
 * @QMI_WDS_CALL_END_REASON_EVDO_CONNECTION_DENY_BILLING_OR_AUTHENTICATION_FAILURE: (EV-DO) Received Connection Deny (Billing or Authentication failure).
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_CHANGE: (EV-DO) Change HDR.
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_EXIT: (EV-DO) Exit HDR.
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_NO_SESSION: (EV-DO) No HDR session.
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_ORIGINATION_DURING_GPS_FIX: (EV-DO) HDR call ended in favor of a GPS fix.
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_CONNECTION_SETUP_TIMEOUT: (EV-DO) Connection setup timeout.
 * @QMI_WDS_CALL_END_REASON_EVDO_HDR_RELEASED_BY_CM: (EV-DO) Released HDR call by call manager.
 *
 * Reason for ending the call.
 *
 * Since: 1.0
 */
typedef enum {
    /* Generic reasons */
    QMI_WDS_CALL_END_REASON_GENERIC_UNSPECIFIED                = 1,
    QMI_WDS_CALL_END_REASON_GENERIC_CLIENT_END                 = 2,
    QMI_WDS_CALL_END_REASON_GENERIC_NO_SERVICE                 = 3,
    QMI_WDS_CALL_END_REASON_GENERIC_FADE                       = 4,
    QMI_WDS_CALL_END_REASON_GENERIC_RELEASE_NORMAL             = 5,
    QMI_WDS_CALL_END_REASON_GENERIC_ACCESS_ATTEMPT_IN_PROGRESS = 6,
    QMI_WDS_CALL_END_REASON_GENERIC_ACCESS_FAILURE             = 7,
    QMI_WDS_CALL_END_REASON_GENERIC_REDIRECTION_OR_HANDOFF     = 8,
    QMI_WDS_CALL_END_REASON_GENERIC_CLOSE_IN_PROGRESS          = 9,
    QMI_WDS_CALL_END_REASON_GENERIC_AUTHENTICATION_FAILED      = 10,
    QMI_WDS_CALL_END_REASON_GENERIC_INTERNAL_ERROR             = 11,

    /* CDMA specific reasons */
    QMI_WDS_CALL_END_REASON_CDMA_LOCK                            = 500,
    QMI_WDS_CALL_END_REASON_CDMA_INTERCEPT                       = 501,
    QMI_WDS_CALL_END_REASON_CDMA_REORDER                         = 502,
    QMI_WDS_CALL_END_REASON_CDMA_RELEASE_SO_REJECT               = 503,
    QMI_WDS_CALL_END_REASON_CDMA_INCOMING_CALL                   = 504,
    QMI_WDS_CALL_END_REASON_CDMA_ALERT_STOP                      = 505,
    QMI_WDS_CALL_END_REASON_CDMA_ACTIVATION                      = 506,
    QMI_WDS_CALL_END_REASON_CDMA_MAX_ACCESS_PROBES               = 507,
    QMI_WDS_CALL_END_REASON_CDMA_CCS_NOT_SUPPORTED_BY_BS         = 508,
    QMI_WDS_CALL_END_REASON_CDMA_NO_RESPONSE_FROM_BS             = 509,
    QMI_WDS_CALL_END_REASON_CDMA_REJECTED_BY_BS                  = 510,
    QMI_WDS_CALL_END_REASON_CDMA_INCOMPATIBLE                    = 511,
    QMI_WDS_CALL_END_REASON_CDMA_ALREADY_IN_TC                   = 512,
    QMI_WDS_CALL_END_REASON_CDMA_USER_CALL_ORIGINATED_DURING_GPS = 513,
    QMI_WDS_CALL_END_REASON_CDMA_USER_CALL_ORIGINATED_DURING_SMS = 514,
    QMI_WDS_CALL_END_REASON_CDMA_NO_SERVICE                      = 515,

    /* GSM/WCDMA specific reasons */
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_CONFERENCE_FAILED                       = 1000,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_INCOMING_REJECTED                       = 1001,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_NO_SERVICE                              = 1002,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_NETWORK_END                             = 1003,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_LLC_SNDCP_FAILURE                       = 1004,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_INSUFFICIENT_RESOURCES                  = 1005,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_TEMPORARILY_OUT_OF_ORDER         = 1006,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_NSAPI_ALREADY_USED                      = 1007,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_REGULAR_DEACTIVATION                    = 1008,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_NETWORK_FAILURE                         = 1009,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_REATTACH_REQUIRED                       = 1010,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_PROTOCOL_ERROR                          = 1011,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPERATOR_DETERMINED_BARRING             = 1012,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_APN                             = 1013,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_PDP                             = 1014,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_GGSN_REJECT                             = 1015,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_ACTIVATION_REJECT                       = 1016,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_NOT_SUPPORTED                    = 1017,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_OPTION_UNSUBSCRIBED                     = 1018,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_QOS_NOT_ACCEPTED                        = 1019,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_TFT_SEMANTIC_ERROR                      = 1020,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_TFT_SYNTAX_ERROR                        = 1021,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_PDP_CONTEXT                     = 1022,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_FILTER_SEMANTIC_ERROR                   = 1023,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_FILTER_SYNTAX_ERROR                     = 1024,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_PDP_WITHOUT_ACTIVE_TFT                  = 1025,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_INVALID_TRANSACTION_ID                  = 1026,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_INCORRECT_SEMANTIC              = 1027,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_INVALID_MANDATORY_INFO                  = 1028,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_TYPE_UNSUPPORTED                = 1029,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_TYPE_NONCOMPATIBLE_STATE        = 1030,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_UNKNOWN_INFO_ELEMENT                    = 1031,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_CONDITIONAL_IE_ERROR                    = 1032,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_MESSAGE_AND_PROTOCOL_STATE_UNCOMPATIBLE = 1033,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_APN_TYPE_CONFLICT                       = 1034,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_NO_GPRS_CONTEXT                         = 1035,
    QMI_WDS_CALL_END_REASON_GSM_WCDMA_FEATURE_NOT_SUPPORTED                   = 1036,

    /* EV-DO specific reasons */
    QMI_WDS_CALL_END_REASON_EVDO_CONNECTION_DENY_GENERAL_OR_BUSY                   = 1500,
    QMI_WDS_CALL_END_REASON_EVDO_CONNECTION_DENY_BILLING_OR_AUTHENTICATION_FAILURE = 1501,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_CHANGE                                        = 1502,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_EXIT                                          = 1503,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_NO_SESSION                                    = 1504,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_ORIGINATION_DURING_GPS_FIX                    = 1505,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_CONNECTION_SETUP_TIMEOUT                      = 1506,
    QMI_WDS_CALL_END_REASON_EVDO_HDR_RELEASED_BY_CM                                = 1507
} QmiWdsCallEndReason;

/**
 * qmi_wds_call_end_reason_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsVerboseCallEndReasonType:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_MIP: Mobile IP.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_INTERNAL: Internal.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_CM: Call manager.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_3GPP: 3GPP.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_PPP: PPP.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_EHRPD: eHRPD.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_IPV6: IPv6.
 *
 * Type of verbose call end reason.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_MIP      = 1,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_INTERNAL = 2,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_CM       = 3,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_3GPP     = 6,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_PPP      = 7,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_EHRPD    = 8,
    QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_IPV6     = 9
} QmiWdsVerboseCallEndReasonType;

/**
 * qmi_wds_verbose_call_end_reason_type_get_string:
 *
 * Since: 1.0
 */

/**
 * qmi_wds_verbose_call_end_reason_get_string:
 * @type: a #QmiWdsVerboseCallEndReasonType.
 * @reason: the reason id.
 *
 * Get verbose call end reason based on the reason type.
 *
 * Returns: (transfer none): a string with the reason, or %NULL if not found. Do not free the returned value.
 *
 * Since: 1.0
 */
const gchar *qmi_wds_verbose_call_end_reason_get_string (QmiWdsVerboseCallEndReasonType type,
                                                         gint16                         reason);

/**
 * QmiWdsVerboseCallEndReasonMip:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_ERROR_REASON_UNKNOWN: Unknown reason.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REASON_UNSPECIFIED: (FA error) Reason unspecified.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_ADMINISTRATIVELY_PROHIBITED: (FA error) Administratively prohibited.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_INSUFFICIENT_RESOURCES: (FA error) Insufficient resources.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MOBILE_NODE_AUTHENTICATION_FAILURE: (FA error) Mobile node authenticatin failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_HA_AUTHENTICATION_FAILURE: (FA error) HA authentication failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REQUESTED_LIFETIME_TOO_LONG: (FA error) Requested lifetime too long.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MALFORMED_REQUEST: (FA error) Malformed request.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MALFORMED_REPLY: (FA error) Malformed reply.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_ENCAPSULATION_UNAVAILABLE: (FA error) Encapsulation unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_VJHC_UNAVAILABLE: (FA error) VJHC unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REVERSE_TUNNEL_UNAVAILABLE: (FA error) Reverse tunnel unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REVERSE_TUNNEL_MANDATORY_AND_T_BIT_NOT_SET: (FA error) Reverse tunnel mandatory and T bit not set.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_DELIVERY_STYLE_NOT_SUPPORTED: (FA error) Delivery style not supported.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_NAI: (FA error) Missing NAI.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_HA: (FA error) Missing HA.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_HOME_ADDRESS: (FA error) Missing home address.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_UNKNOWN_CHALLENGE: (FA error) Unknown challenge.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_CHALLENGE: (FA error) Missing challenge.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_STALE_CHALLENGE: (FA error) Stale challenge.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REASON_UNSPECIFIED: (HA error) Reason unspecified.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_ADMINISTRATIVELY_PROHIBITED: (HA error) Administratively prohibited.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_INSUFFICIENT_RESOURCES: (HA error) Insufficient resources.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_MOBILE_NODE_AUTHENTICATION_FAILURE: (HA error) Mobile node authentication failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_FA_AUTHENTICATION_FAILURE: (HA error) FA authentication failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REGISTRATION_ID_MISMATCH: (HA error) Registration ID mismatch.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_MALFORMED_REQUEST: (HA error) Malformed request.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_UNKNOWN_HA_ADDRESS: (HA error) Unknown HA address.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REVERSE_TUNNEL_UNAVAILABLE: (HA error) Reverse tunnel unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REVERSE_TUNNEL_MANDATORY_AND_T_BIT_NOT_SET: (HA error) Reverse tunnel mandatory and T bit not set.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_ENCAPSULATION_UNAVAILABLE: (HA error) Encapsulation unavailable.
 *
 * Mobile IP specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_MIP.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_ERROR_REASON_UNKNOWN                                = -1,

    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REASON_UNSPECIFIED                         = 64,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_ADMINISTRATIVELY_PROHIBITED                = 65,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_INSUFFICIENT_RESOURCES                     = 66,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MOBILE_NODE_AUTHENTICATION_FAILURE         = 67,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_HA_AUTHENTICATION_FAILURE                  = 68,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REQUESTED_LIFETIME_TOO_LONG                = 69,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MALFORMED_REQUEST                          = 70,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MALFORMED_REPLY                            = 71,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_ENCAPSULATION_UNAVAILABLE                  = 72,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_VJHC_UNAVAILABLE                           = 73,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REVERSE_TUNNEL_UNAVAILABLE                 = 74,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_REVERSE_TUNNEL_MANDATORY_AND_T_BIT_NOT_SET = 75,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_DELIVERY_STYLE_NOT_SUPPORTED               = 79,

    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_NAI                                = 97,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_HA                                 = 98,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_HOME_ADDRESS                       = 99,

    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_UNKNOWN_CHALLENGE                          = 104,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_MISSING_CHALLENGE                          = 105,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_FA_ERROR_STALE_CHALLENGE                            = 106,

    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REASON_UNSPECIFIED                         = 128,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_ADMINISTRATIVELY_PROHIBITED                = 129,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_INSUFFICIENT_RESOURCES                     = 130,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_MOBILE_NODE_AUTHENTICATION_FAILURE         = 131,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_FA_AUTHENTICATION_FAILURE                  = 132,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REGISTRATION_ID_MISMATCH                   = 133,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_MALFORMED_REQUEST                          = 134,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_UNKNOWN_HA_ADDRESS                         = 136,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REVERSE_TUNNEL_UNAVAILABLE                 = 137,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_REVERSE_TUNNEL_MANDATORY_AND_T_BIT_NOT_SET = 138,
    QMI_WDS_VERBOSE_CALL_END_REASON_MIP_HA_ERROR_ENCAPSULATION_UNAVAILABLE                  = 139
} QmiWdsVerboseCallEndReasonMip;

/**
 * qmi_wds_verbose_call_end_reason_mip_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsVerboseCallEndReasonInternal:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR: Internal error.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CALL_ENDED: Call ended.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNKNOWN_INTERNAL_CAUSE: Unknown internal cause.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNKNOWN_CAUSE: Unknown cause.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CLOSE_IN_PROGRESS: Close in progress.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_NETWORK_INITIATED_TERMINATION: Network initiated termination.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APP_PREEMPTED: App preempted.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV4_CALL_DISALLOWED: PDN IPv4 call disallowed. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV4_CALL_THROTTLED: PDN IPv4 call throttled. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV6_CALL_DISALLOWED: PDN IPv6 call disallowed. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV6_CALL_THROTTLED: PDN IPv6 call throttled. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MODEM_RESTART: Modem restart. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDP_PPP_NOT_SUPPORTED: PDP PPP not supported. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNPREFERRED_RAT: Unpreferred RAT. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PHYSICAL_LINK_CLOSE_IN_PROGRESS: Physical link close in progress. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_PENDING_HANDOVER: APN pending handover. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PROFILE_BEARER_INCOMPATIBLE: Profile bearer incompatible. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MMGDSI_CARD_EVENT: MMGDSI Card event. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_LPM_OR_POWER_DOWN: LPM or power down. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_DISABLED: APN disabled. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MPIT_EXPIRED: MPIT expired. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_IPV6_ADDRESS_TRANSFER_FAILED: IPv6 address transfer failed. Since: 1.14.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_TRAT_SWAP_FAILED: TRAT swap failed. Since: 1.14.
 *
 * Internal call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_INTERNAL.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_ERROR                           = 201,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CALL_ENDED                      = 202,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNKNOWN_INTERNAL_CAUSE          = 203,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNKNOWN_CAUSE                   = 204,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_CLOSE_IN_PROGRESS               = 205,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_NETWORK_INITIATED_TERMINATION   = 206,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APP_PREEMPTED                   = 207,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV4_CALL_DISALLOWED        = 208,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV4_CALL_THROTTLED         = 209,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV6_CALL_DISALLOWED        = 210,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDN_IPV6_CALL_THROTTLED         = 211,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MODEM_RESTART                   = 212,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PDP_PPP_NOT_SUPPORTED           = 213,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_UNPREFERRED_RAT                 = 214,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PHYSICAL_LINK_CLOSE_IN_PROGRESS = 215,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_PENDING_HANDOVER            = 216,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_PROFILE_BEARER_INCOMPATIBLE     = 217,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MMGDSI_CARD_EVENT               = 218,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_LPM_OR_POWER_DOWN               = 219,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_APN_DISABLED                    = 220,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_MPIT_EXPIRED                    = 221,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_IPV6_ADDRESS_TRANSFER_FAILED    = 222,
    QMI_WDS_VERBOSE_CALL_END_REASON_INTERNAL_TRAT_SWAP_FAILED                = 223,
} QmiWdsVerboseCallEndReasonInternal;

/**
 * qmi_wds_verbose_call_end_reason_internal_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsVerboseCallEndReasonCm:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CDMA_LOCK: (CDMA) Phone is CDMA-locked.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INTERCEPT: (CDMA) Received intercept from the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_REORDER: (CDMA) Received reorder from the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RELEASE_SO_REJECT: (CDMA) Received release from the BS, SO reject.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMING_CALL: (CDMA) Received incoming call from the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ALERT_STOP: (CDMA) Received alert stop from the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACTIVATION: (CDMA) Received end activation.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_MAX_ACCESS_PROBES: (CDMA) Maximum access probes transmitted.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CCS_NOT_SUPPORTED_BY_BS: (CDMA) Concurrent service not supported by the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_RESPONSE_FROM_BS: (CDMA) No response received from the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_REJECTED_BY_BS: (CDMA) Rejected by the BS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMPATIBLE: (CDMA) Concurrent services requested are incompatible.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ALREADY_IN_TC: (CDMA) Already in TC.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_USER_CALL_ORIGINATED_DURING_GPS: (CDMA) Call originated during GPS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_USER_CALL_ORIGINATED_DURING_SMS: (CDMA) Call originated during SMS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_CDMA_SERVICE: (CDMA) No service.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RETRY_ORDER: Retry order.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONFIGURATION_FAILED: Configuration failed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMING_REJECTED: Incoming rejected.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_GATEWAY_SERVICE: No gateway service.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_GPRS_CONTEXT: No GPRS context.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ILLEGAL_MS: Illegal MS.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ILLEGAL_ME: Illegal ME.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_AND_NON_GPRS_SERVICES_NOT_ALLOWED: GPRS and non GPRS services not allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_SERVICES_NOT_ALLOWED: GPRS services not allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_MS_IDENTITY_NOT_DERIVED_BY_THE_NETWORK: MS identity not derived by the network.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_IMPLICITLY_DETACHED: Implicitly detached.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PLMN_NOT_ALLOWED: PLMN not allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_LA_NOT_ALLOWED: LA not allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_SERVICES_NOT_ALLOWED_IN_PLMN: GPRS services not allowed in PLMN.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_DUPLICATE: PDP duplicate.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_UE_RAT_CHANGE: UE radio access technology change.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONGESTION: Congestion.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_PDP_CONTEXT_ACTIVATED: No PDP context activated.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_CLASS_DSAC_REJECTION: Access class DSAC rejection.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONNECTION_DENY_GENERAL_OR_BUSY: (EV-DO) Received Connection Deny (General or Network busy).
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONNECTION_DENY_BILLING_OR_AUTHENTICATION_FAILURE: (EV-DO) Received Connection Deny (Billing or Authentication failure).
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_CHANGE: (EV-DO) Change HDR.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_EXIT: (EV-DO) Exit HDR.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_NO_SESSION: (EV-DO) No HDR session.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_ORIGINATION_DURING_GPS_FIX: (EV-DO) HDR call ended in favor of a GPS fix.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_CONNECTION_SETUP_TIMEOUT: (EV-DO) Connection setup timeout.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_RELEASED_BY_CM: (EV-DO) Released HDR call by call manager.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_HYBRID_HDR_SERVICE: (EV-DO) No hybrid HDR service.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_CLIENT_END: Client end.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_SERVICE: No service.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_FADE: Fade.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_RELEASE_NORMAL: Release normal.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_ATTEMPT_IN_PROGRESS: Access attempt in progress.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_FAILURE: Access Failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_CM_REDIRECTION_OR_HANDOFF: Redirection or handoff.
 *
 * Call manager specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_CM.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CDMA_LOCK                       = 500,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INTERCEPT                       = 501,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_REORDER                         = 502,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RELEASE_SO_REJECT               = 503,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMING_CALL                   = 504,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ALERT_STOP                      = 505,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACTIVATION                      = 506,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_MAX_ACCESS_PROBES               = 507,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CCS_NOT_SUPPORTED_BY_BS         = 508,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_RESPONSE_FROM_BS             = 509,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_REJECTED_BY_BS                  = 510,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMPATIBLE                    = 511,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ALREADY_IN_TC                   = 512,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_USER_CALL_ORIGINATED_DURING_GPS = 513,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_USER_CALL_ORIGINATED_DURING_SMS = 514,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_CDMA_SERVICE                 = 515,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RETRY_ORDER                     = 519,

    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONFIGURATION_FAILED                   = 1000,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_INCOMING_REJECTED                      = 1001,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_GATEWAY_SERVICE                     = 1002,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_GPRS_CONTEXT                        = 1003,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ILLEGAL_MS                             = 1004,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ILLEGAL_ME                             = 1005,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_AND_NON_GPRS_SERVICES_NOT_ALLOWED = 1006,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_SERVICES_NOT_ALLOWED              = 1007,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_MS_IDENTITY_NOT_DERIVED_BY_THE_NETWORK = 1008,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_IMPLICITLY_DETACHED                    = 1009,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PLMN_NOT_ALLOWED                       = 1010,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_LA_NOT_ALLOWED                         = 1011,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_GPRS_SERVICES_NOT_ALLOWED_IN_PLMN      = 1012,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_PDP_DUPLICATE                          = 1013,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_UE_RAT_CHANGE                          = 1014,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONGESTION                             = 1015,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_PDP_CONTEXT_ACTIVATED               = 1016,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_CLASS_DSAC_REJECTION            = 1017,

    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONNECTION_DENY_GENERAL_OR_BUSY                   = 1500,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CONNECTION_DENY_BILLING_OR_AUTHENTICATION_FAILURE = 1501,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_CHANGE                                        = 1502,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_EXIT                                          = 1503,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_NO_SESSION                                    = 1504,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_ORIGINATION_DURING_GPS_FIX                    = 1505,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_CONNECTION_SETUP_TIMEOUT                      = 1506,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_HDR_RELEASED_BY_CM                                = 1507,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_HYBRID_HDR_SERVICE                             = 1510,

    QMI_WDS_VERBOSE_CALL_END_REASON_CM_CLIENT_END                 = 2000,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_NO_SERVICE                 = 2001,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_FADE                       = 2002,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_RELEASE_NORMAL             = 2003,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_ATTEMPT_IN_PROGRESS = 2004,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_ACCESS_FAILURE             = 2005,
    QMI_WDS_VERBOSE_CALL_END_REASON_CM_REDIRECTION_OR_HANDOFF     = 2006
} QmiWdsVerboseCallEndReasonCm;

/**
 * qmi_wds_verbose_call_end_reason_cm_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsVerboseCallEndReason3gpp:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPERATOR_DETERMINED_BARRING: Operator-determined barring.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_LLC_SNDCP_FAILURE: LLC or SNDCP failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INSUFFICIENT_RESOURCES: Insufficient resources.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_APN:  Unknown or missing APN.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_PDP: Unknown PDP address or type.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_AUTHENTICATION_FAILED: Authentication failed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_GGSN_REJECT: Activation rejected by GGSN.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_ACTIVATION_REJECT: Activation rejected.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_NOT_SUPPORTED:  Service option not supported.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_UNSUBSCRIBED: Service option not subscribed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_TEMPORARILY_OUT_OF_ORDER: Service option temporarily out of order.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_NSAPI_ALREADY_USED: NSAPI already used.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_REGULAR_DEACTIVATION: Regular PDP context deactivation.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_QOS_NOT_ACCEPTED: QoS not accepted.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_NETWORK_FAILURE: Network failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_REATTACH_REQUIRED: Reattach required.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FEATURE_NOT_SUPPORTED: Feature not supported.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_TFT_SEMANTIC_ERROR: Semantic error in TFT operation.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_TFT_SYNTAX_ERROR: Syntax error in TFT operation.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_PDP_CONTEXT: Unknown PDP context.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FILTER_SEMANTIC_ERROR: Semantic error in packet filters.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FILTER_SYNTAX_ERROR: Syntax error in packet filters.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PDP_WITHOUT_ACTIVE_TFT: PDP context without TFT activated.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_IPV4_ONLY_ALLOWED: IPv4 only allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_IPV6_ONLY_ALLOWED: IPv6 only allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_SINGLE_ADDRESS_BEARER_ONLY: Single address bearer only.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_ESM_INFO_NOT_RECEIVED: ESM information not received.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PDN_CONNECTION_DOES_NOT_EXIST: PDN connection does not exist.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MULTIPLE_CONNECTION_TO_SAME_PDN_NOT_ALLOWED: Multiple connection to same PDN not allowed.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_TRANSACTION_ID: Invalid transaction ID.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_INCORRECT_SEMANTIC: Message incorrect semantically.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_MANDATORY_INFO:  Invalid mandatory information.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_TYPE_UNSUPPORTED: Message type not implemented.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_TYPE_NONCOMPATIBLE_STATE: Message not compatible with state.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_INFO_ELEMENT: Information element unknown.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_CONDITIONAL_IE_ERROR: Conditional IE error.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_AND_PROTOCOL_STATE_UNCOMPATIBLE: Message and protocol state uncompatible.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PROTOCOL_ERROR: Protocol error.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_APN_TYPE_CONFLICT: APN type conflict.
 *
 * 3GPP-specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_3GPP.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_verbose_call_end_reason_3gpp >*/
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPERATOR_DETERMINED_BARRING = 8,

    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_LLC_SNDCP_FAILURE               = 25,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INSUFFICIENT_RESOURCES          = 26,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_APN                     = 27,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_PDP                     = 28,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_AUTHENTICATION_FAILED           = 29,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_GGSN_REJECT                     = 30,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_ACTIVATION_REJECT               = 31,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_NOT_SUPPORTED            = 32,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_UNSUBSCRIBED             = 33,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_OPTION_TEMPORARILY_OUT_OF_ORDER = 34,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_NSAPI_ALREADY_USED              = 35,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_REGULAR_DEACTIVATION            = 36,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_QOS_NOT_ACCEPTED                = 37,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_NETWORK_FAILURE                 = 38,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_REATTACH_REQUIRED               = 39,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FEATURE_NOT_SUPPORTED           = 40,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_TFT_SEMANTIC_ERROR              = 41,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_TFT_SYNTAX_ERROR                = 42,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_PDP_CONTEXT             = 43,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FILTER_SEMANTIC_ERROR           = 44,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_FILTER_SYNTAX_ERROR             = 45,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PDP_WITHOUT_ACTIVE_TFT          = 46,

    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_IPV4_ONLY_ALLOWED                           = 50,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_IPV6_ONLY_ALLOWED                           = 51,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_SINGLE_ADDRESS_BEARER_ONLY                  = 52,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_ESM_INFO_NOT_RECEIVED                       = 53,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PDN_CONNECTION_DOES_NOT_EXIST               = 54,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MULTIPLE_CONNECTION_TO_SAME_PDN_NOT_ALLOWED = 55,

    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_TRANSACTION_ID = 81,

    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_INCORRECT_SEMANTIC              = 95,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_INVALID_MANDATORY_INFO                  = 96,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_TYPE_UNSUPPORTED                = 97,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_TYPE_NONCOMPATIBLE_STATE        = 98,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_UNKNOWN_INFO_ELEMENT                    = 99,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_CONDITIONAL_IE_ERROR                    = 100,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_MESSAGE_AND_PROTOCOL_STATE_UNCOMPATIBLE = 101,

    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_PROTOCOL_ERROR    = 111,
    QMI_WDS_VERBOSE_CALL_END_REASON_3GPP_APN_TYPE_CONFLICT = 112
} QmiWdsVerboseCallEndReason3gpp;

/**
 * qmi_wds_verbose_call_end_reason_3gpp_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsVerboseCallEndReasonPpp:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_UNKNOWN: Unknown error.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_TIMEOUT: Timeout.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_AUTHENTICATION_FAILURE: Authentication failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_OPTION_MISMATCH: Option mismatch.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_PAP_FAILURE: PAP failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_PPP_CHAP_FAILURE: CHAP failure.
 *
 * PPP-specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_PPP.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_UNKNOWN = -1,

    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_TIMEOUT                = 1,
    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_AUTHENTICATION_FAILURE = 2,
    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_OPTION_MISMATCH        = 3,

    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_PAP_FAILURE  = 31,
    QMI_WDS_VERBOSE_CALL_END_REASON_PPP_CHAP_FAILURE = 32
} QmiWdsVerboseCallEndReasonPpp;

/**
 * qmi_wds_verbose_call_end_reason_ppp_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsVerboseCallEndReasonEhrpd:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_SUBSCRIPTION_LIMITED_TO_IPV4: Subscription limited to IPv4.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_SUBSCRIPTION_LIMITED_TO_IPV6: Subscription limited to IPv6.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_TIMEOUT: (VSNCP) timeout.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_FAILURE: (VSNCP) failure.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_GENERAL_ERROR: (VSCNP) 3GPP2 general error.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_UNAUTHENTICATED_APN: (VSCNP) 3GPP2 unauthenticated APN.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_LIMIT_EXCEEDED: (VSCNP) 3GPP2 PDN limit exceeded.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_NO_PDN_GATEWAY: (VSCNP) 3GPP2 no PDN gateway.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_GATEWAY_UNREACHABLE: (VSCNP) 3GPP2 PDN gateway unreachable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_GATEWAY_REJECTED: (VSCNP) 3GPP2 PDN gateway rejected.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_INSUFFICIENT_PARAMETERS: (VSCNP) 3GPP2 insufficient parameters.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_RESOURCE_UNAVAILABLE: (VSCNP) 3GPP2 resource unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_ADMINISTRATIVELY_PROHIBITED: (VSCNP) 3GPP2 administratively prohibited.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_ID_IN_USE: (VSCNP) 3GPP2 PDN ID in use.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_SUBSCRIPTION_LIMITATION: (VSCNP) 3GPP2 subscription limitation.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_EXISTS_FOR_THIS_APN: (VSCNP) 3GPP2 PDN exists for this APN.
 *
 * eHRPD-specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_EHRPD.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_SUBSCRIPTION_LIMITED_TO_IPV4 = 1,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_SUBSCRIPTION_LIMITED_TO_IPV6 = 2,

    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_TIMEOUT                           = 4,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_FAILURE                           = 5,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_GENERAL_ERROR               = 6,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_UNAUTHENTICATED_APN         = 7,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_LIMIT_EXCEEDED          = 8,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_NO_PDN_GATEWAY              = 9,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_GATEWAY_UNREACHABLE     = 10,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_GATEWAY_REJECTED        = 11,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_INSUFFICIENT_PARAMETERS     = 12,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_RESOURCE_UNAVAILABLE        = 13,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_ADMINISTRATIVELY_PROHIBITED = 14,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_ID_IN_USE               = 15,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_SUBSCRIPTION_LIMITATION     = 16,
    QMI_WDS_VERBOSE_CALL_END_REASON_EHRPD_VSNCP_3GPP2_PDN_EXISTS_FOR_THIS_APN     = 17
} QmiWdsVerboseCallEndReasonEhrpd;

/**
 * qmi_wds_verbose_call_end_reason_ehrpd_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsVerboseCallEndReasonIpv6:
 * @QMI_WDS_VERBOSE_CALL_END_REASON_IPV6_PREFIX_UNAVAILABLE: Prefix unavailable.
 * @QMI_WDS_VERBOSE_CALL_END_REASON_IPV6_HRPD_IPV6_DISABLED: HRDP IPv6 disabled.
 *
 * IPv6-specific call end reasons, given when the @QmiWdsVerboseCallEndReasonType is #QMI_WDS_VERBOSE_CALL_END_REASON_TYPE_IPV6.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_VERBOSE_CALL_END_REASON_IPV6_PREFIX_UNAVAILABLE = 1,
    QMI_WDS_VERBOSE_CALL_END_REASON_IPV6_HRPD_IPV6_DISABLED = 2
} QmiWdsVerboseCallEndReasonIpv6;

/**
 * qmi_wds_verbose_call_end_reason_ipv6_get_string:
 *
 * Since: 1.0
 */

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Packet Service Status' message */

/**
 * QmiWdsConnectionStatus:
 * @QMI_WDS_CONNECTION_STATUS_UNKNOWN: Unknown status.
 * @QMI_WDS_CONNECTION_STATUS_DISCONNECTED: Network is disconnected
 * @QMI_WDS_CONNECTION_STATUS_CONNECTED: Network is connected.
 * @QMI_WDS_CONNECTION_STATUS_SUSPENDED: Network connection is suspended.
 * @QMI_WDS_CONNECTION_STATUS_AUTHENTICATING: Network authentication is ongoing.
 *
 * Status of the network connection.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_CONNECTION_STATUS_UNKNOWN        = 0,
    QMI_WDS_CONNECTION_STATUS_DISCONNECTED   = 1,
    QMI_WDS_CONNECTION_STATUS_CONNECTED      = 2,
    QMI_WDS_CONNECTION_STATUS_SUSPENDED      = 3,
    QMI_WDS_CONNECTION_STATUS_AUTHENTICATING = 4
} QmiWdsConnectionStatus;

/**
 * qmi_wds_connection_status_get_string:
 *
 * Since: 1.0
 */

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Data Bearer Technology' message */

/**
 * QmiWdsDataBearerTechnology:
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_UNKNOWN: Unknown.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_CDMA20001X: CDMA2000 1x.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_1xEVDO: CDMA2000 HRPD 1xEV-DO.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_GSM: GSM.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_UMTS: UMTS.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_1xEVDO_REVA: CDMA2000 HRPD 1xEV-DO RevA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_EDGE: EDGE.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPA: HSDPA and WCDMA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSUPA: WCDMA and HSUPA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPA_HSUPDA: HSDPA and HSUPA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_LTE: LTE.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_EHRPD: CDMA2000 eHRPD.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS: HSDPA+ and WCDMA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_HSUPA: HSDPA+ and HSUPA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_DCHSDPAPLUS: DC-HSDPA+ and WCDMA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_DCHSDPAPLUS_HSUPA: DC-HSDPA+ and HSUPA.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_PLUS_64QAM: HSDPA+ with 64QAM. Since: 1.18.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_PLUS_64QAM_HSUPA: HSDPA+ with 64QAM and HSUPA. Since: 1.18.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA: TD-SCDMA. Since: 1.18.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA_HSDPA: TD-SCDMA and HSDPA. Since: 1.18.
 * @QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA_HSUPA: TD-SCDMA and HSUPA. Since: 1.18.
 *
 * Data bearer technology.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_DATA_BEARER_TECHNOLOGY_UNKNOWN                    = -1,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_CDMA20001X                 = 0x01,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_1xEVDO                     = 0x02,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_GSM                        = 0x03,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_UMTS                       = 0x04,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_1xEVDO_REVA                = 0x05,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_EDGE                       = 0x06,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPA                      = 0x07,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSUPA                      = 0x08,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPA_HSUPDA               = 0x09,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_LTE                        = 0x0A,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_EHRPD                      = 0x0B,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS                  = 0x0C,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_HSUPA            = 0x0D,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_DCHSDPAPLUS                = 0x0E,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_DCHSDPAPLUS_HSUPA          = 0x0F,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_PLUS_64QAM       = 0x10,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_HSDPAPLUS_PLUS_64QAM_HSUPA = 0x11,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA                    = 0x12,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA_HSDPA              = 0x13,
    QMI_WDS_DATA_BEARER_TECHNOLOGY_TDSCDMA_HSUPA              = 0x14,
} QmiWdsDataBearerTechnology;

/**
 * qmi_wds_data_bearer_technology_get_string:
 *
 * Since: 1.0
 */

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Current Data Bearer Technology' message */

/**
 * QmiWdsNetworkType:
 * @QMI_WDS_NETWORK_TYPE_UNKNOWN: Unknown.
 * @QMI_WDS_NETWORK_TYPE_3GPP2: 3GPP2 network type.
 * @QMI_WDS_NETWORK_TYPE_3GPP: 3GPP network type.
 *
 * Network type of the data bearer.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_NETWORK_TYPE_UNKNOWN = 0,
    QMI_WDS_NETWORK_TYPE_3GPP2   = 1,
    QMI_WDS_NETWORK_TYPE_3GPP    = 2
} QmiWdsNetworkType;

/**
 * qmi_wds_network_type_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsDataSystemNetworkType:
 * @QMI_WDS_DATA_SYSTEM_NETWORK_TYPE_3GPP2: 3GPP2 network type.
 * @QMI_WDS_DATA_SYSTEM_NETWORK_TYPE_3GPP: 3GPP network type.
 *
 * Network type of the data system.
 *
 * Since: 1.14
 */
typedef enum {
    QMI_WDS_DATA_SYSTEM_NETWORK_TYPE_3GPP  = 0,
    QMI_WDS_DATA_SYSTEM_NETWORK_TYPE_3GPP2 = 1,
} QmiWdsDataSystemNetworkType;

/**
 * qmi_wds_data_system_network_type_get_string:
 *
 * Since: 1.14
 */

/**
 * QmiWdsDataSystem:
 * @QMI_WDS_DATA_SYSTEM_UNKNOWN: unknown
 * @QMI_WDS_DATA_SYSTEM_CDMA1X: CDMA 1x
 * @QMI_WDS_DATA_SYSTEM_CDMA1X_EVDO: CDMA 1x EVDO
 * @QMI_WDS_DATA_SYSTEM_GPRS: GPRS
 * @QMI_WDS_DATA_SYSTEM_WCDMA: WCDMA
 * @QMI_WDS_DATA_SYSTEM_LTE: LTE
 * @QMI_WDS_DATA_SYSTEM_TDSCDMA: TD-SCDMA.
 *
 * Data system type.
 *
 * Since: 1.18
 */
typedef enum {
    QMI_WDS_DATA_SYSTEM_UNKNOWN     = 0,
    QMI_WDS_DATA_SYSTEM_CDMA1X      = 1,
    QMI_WDS_DATA_SYSTEM_CDMA1X_EVDO = 2,
    QMI_WDS_DATA_SYSTEM_GPRS        = 3,
    QMI_WDS_DATA_SYSTEM_WCDMA       = 4,
    QMI_WDS_DATA_SYSTEM_LTE         = 5,
    QMI_WDS_DATA_SYSTEM_TDSCDMA     = 6,
} QmiWdsDataSystem;

/**
 * qmi_wds_data_system_get_string:
 *
 * Since: 1.18
 */

/**
 * QmiWdsRat3gpp2:
 * @QMI_WDS_RAT_3GPP2_NONE: Unknown, to be ignored.
 * @QMI_WDS_RAT_3GPP2_CDMA1X: CDMA 1x.
 * @QMI_WDS_RAT_3GPP2_EVDO_REV0: EVDO Rev0.
 * @QMI_WDS_RAT_3GPP2_EVDO_REVA: EVDO RevA.
 * @QMI_WDS_RAT_3GPP2_EVDO_REVB: EVDO RevB.
 * @QMI_WDS_RAT_3GPP2_EHRPD: eHRPD. Since: 1.18.
 * @QMI_WDS_RAT_3GPP2_FMC: FMC. Since: 1.18.
 * @QMI_WDS_RAT_3GPP2_NULL_BEARER: No bearer.
 *
 * Flags specifying the 3GPP2-specific Radio Access Technology, when the data
 * bearer network type is @QMI_WDS_NETWORK_TYPE_3GPP2.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_rat_3gpp2 >*/
    QMI_WDS_RAT_3GPP2_NONE        = 0,
    QMI_WDS_RAT_3GPP2_CDMA1X      = 1 << 0,
    QMI_WDS_RAT_3GPP2_EVDO_REV0   = 1 << 1,
    QMI_WDS_RAT_3GPP2_EVDO_REVA   = 1 << 2,
    QMI_WDS_RAT_3GPP2_EVDO_REVB   = 1 << 3,
    QMI_WDS_RAT_3GPP2_EHRPD       = 1 << 4,
    QMI_WDS_RAT_3GPP2_FMC         = 1 << 5,
    QMI_WDS_RAT_3GPP2_NULL_BEARER = 1 << 15
} QmiWdsRat3gpp2;

/**
 * qmi_wds_rat_3gpp2_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsRat3gpp:
 * @QMI_WDS_RAT_3GPP_NONE: Unknown, to be ignored.
 * @QMI_WDS_RAT_3GPP_WCDMA: WCDMA.
 * @QMI_WDS_RAT_3GPP_GPRS: GPRS.
 * @QMI_WDS_RAT_3GPP_HSDPA: HSDPA.
 * @QMI_WDS_RAT_3GPP_HSUPA: HSUPA.
 * @QMI_WDS_RAT_3GPP_EDGE: EDGE.
 * @QMI_WDS_RAT_3GPP_LTE: LTE.
 * @QMI_WDS_RAT_3GPP_HSDPAPLUS: HSDPA+.
 * @QMI_WDS_RAT_3GPP_DCHSDPAPLUS: DC-HSDPA+.
 * @QMI_WDS_RAT_3GPP_64QAM: 64QAM. Since: 1.18.
 * @QMI_WDS_RAT_3GPP_TDSCDMA: TD-SCDMA. Since: 1.18.
 * @QMI_WDS_RAT_3GPP_NULL_BEARER: No bearer.
 *
 * Flags specifying the 3GPP-specific Radio Access Technology, when the data
 * bearer network type is @QMI_WDS_NETWORK_TYPE_3GPP.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_rat_3gpp >*/
    QMI_WDS_RAT_3GPP_NONE        = 0,
    QMI_WDS_RAT_3GPP_WCDMA       = 1 << 0,
    QMI_WDS_RAT_3GPP_GPRS        = 1 << 1,
    QMI_WDS_RAT_3GPP_HSDPA       = 1 << 2,
    QMI_WDS_RAT_3GPP_HSUPA       = 1 << 3,
    QMI_WDS_RAT_3GPP_EDGE        = 1 << 4,
    QMI_WDS_RAT_3GPP_LTE         = 1 << 5,
    QMI_WDS_RAT_3GPP_HSDPAPLUS   = 1 << 6,
    QMI_WDS_RAT_3GPP_DCHSDPAPLUS = 1 << 7,
    QMI_WDS_RAT_3GPP_64QAM       = 1 << 8,
    QMI_WDS_RAT_3GPP_TDSCDMA     = 1 << 9,
    QMI_WDS_RAT_3GPP_NULL_BEARER = 1 << 15
} QmiWdsRat3gpp;

/**
 * qmi_wds_rat_3gpp_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsSoCdma1x:
 * @QMI_WDS_SO_CDMA1X_NONE: Unknown, to be ignored.
 * @QMI_WDS_SO_CDMA1X_IS95: IS95.
 * @QMI_WDS_SO_CDMA1X_IS2000: IS2000.
 * @QMI_WDS_SO_CDMA1X_IS2000_REL_A: IS2000 RelA.
 *
 * Flags specifying the Service Option when the bearer network type is
 * @QMI_WDS_NETWORK_TYPE_3GPP2 and when the Radio Access Technology mask
 * contains @QMI_WDS_RAT_3GPP2_CDMA1X.
 *
 * Since: 1.0
 */
typedef enum {
    QMI_WDS_SO_CDMA1X_NONE         = 0,
    QMI_WDS_SO_CDMA1X_IS95         = 1 << 0,
    QMI_WDS_SO_CDMA1X_IS2000       = 1 << 1,
    QMI_WDS_SO_CDMA1X_IS2000_REL_A = 1 << 2
} QmiWdsSoCdma1x;

/**
 * qmi_wds_so_cdma1x_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsSoEvdoRev0:
 * @QMI_WDS_SO_EVDO_REV0_NONE: Unknown, to be ignored.
 * @QMI_WDS_SO_EVDO_REV0_DPA: DPA.
 *
 * Flags specifying the Service Option when the bearer network type is
 * @QMI_WDS_NETWORK_TYPE_3GPP2 and when the Radio Access Technology mask
 * contains @QMI_WDS_RAT_3GPP2_EVDO_REV0.
 *
 * Since: 1.18
 */
typedef enum { /*< underscore_name=qmi_wds_so_evdo_rev0 >*/
    QMI_WDS_SO_EVDO_REV0_NONE = 0,
    QMI_WDS_SO_EVDO_REV0_DPA  = 1 << 0,
} QmiWdsSoEvdoRev0;

/**
 * qmi_wds_so_evdo_rev0_get_string:
 *
 * Since: 1.18
 */

/**
 * QmiWdsSoEvdoRevA:
 * @QMI_WDS_SO_EVDO_REVA_NONE: Unknown, to be ignored.
 * @QMI_WDS_SO_EVDO_REVA_DPA: DPA.
 * @QMI_WDS_SO_EVDO_REVA_MFPA: MFPA.
 * @QMI_WDS_SO_EVDO_REVA_EMPA: EMPA.
 * @QMI_WDS_SO_EVDO_REVA_EMPA_EHRPD: EMPA eHRPD.
 *
 * Flags specifying the Service Option when the bearer network type is
 * @QMI_WDS_NETWORK_TYPE_3GPP2 and when the Radio Access Technology mask
 * contains @QMI_WDS_RAT_3GPP2_EVDO_REVA.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_so_evdo_reva >*/
    QMI_WDS_SO_EVDO_REVA_NONE       = 0,
    QMI_WDS_SO_EVDO_REVA_DPA        = 1 << 0,
    QMI_WDS_SO_EVDO_REVA_MFPA       = 1 << 1,
    QMI_WDS_SO_EVDO_REVA_EMPA       = 1 << 2,
    QMI_WDS_SO_EVDO_REVA_EMPA_EHRPD = 1 << 3,
} QmiWdsSoEvdoRevA;

/**
 * qmi_wds_so_evdo_reva_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsSoEvdoRevB:
 * @QMI_WDS_SO_EVDO_REVB_NONE: Unknown, to be ignored.
 * @QMI_WDS_SO_EVDO_REVB_DPA: DPA.
 * @QMI_WDS_SO_EVDO_REVB_MFPA: MFPA.
 * @QMI_WDS_SO_EVDO_REVB_EMPA: EMPA.
 * @QMI_WDS_SO_EVDO_REVB_EMPA_EHRPD: EMPA eHRPD.
 * @QMI_WDS_SO_EVDO_REVB_MMPA: MMPA.
 * @QMI_WDS_SO_EVDO_REVB_MMPA_EHRPD: MMPA eHRPD.
 *
 * Flags specifying the Service Option when the bearer network type is
 * @QMI_WDS_NETWORK_TYPE_3GPP2 and when the Radio Access Technology mask
 * contains @QMI_WDS_RAT_3GPP2_EVDO_REVB.
 *
 * Since: 1.18
 */
typedef enum { /*< underscore_name=qmi_wds_so_evdo_revb >*/
    QMI_WDS_SO_EVDO_REVB_NONE       = 0,
    QMI_WDS_SO_EVDO_REVB_DPA        = 1 << 0,
    QMI_WDS_SO_EVDO_REVB_MFPA       = 1 << 1,
    QMI_WDS_SO_EVDO_REVB_EMPA       = 1 << 2,
    QMI_WDS_SO_EVDO_REVB_EMPA_EHRPD = 1 << 3,
    QMI_WDS_SO_EVDO_REVB_MMPA       = 1 << 4,
    QMI_WDS_SO_EVDO_REVB_MMPA_EHRPD = 1 << 5,
} QmiWdsSoEvdoRevB;

/**
 * qmi_wds_so_evdo_revb_get_string:
 *
 * Since: 1.18
 */

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Current Settings' message */

/**
 * QmiWdsGetCurrentSettingsRequestedSettings:
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_NONE: no settings requested
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PROFILE_ID: request profile ID
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PROFILE_NAME: request profile name
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PDP_TYPE: request PDP context type
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_APN_NAME: request APN name
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_DNS_ADDRESS: request DNS server addresses
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_GRANTED_QOS: request granted QoS
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_USERNAME: request username
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_AUTH_PROTOCOL: request authentication protocol (ie PAP/CHAP/none)
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_IP_ADDRESS: request IP address
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_GATEWAY_INFO: request gateway address
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PCSCF_ADDRESS: request PCSCF address
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PCSCF_SERVER_ADDRESS_LIST: request PCSCF server address list
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PCSCF_DOMAIN_NAME_LIST: request PCSCF domain name list
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_MTU: request MTU
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_DOMAIN_NAME_LIST: request domain name list
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_IP_FAMILY: request IP family (ie IPv4 or IPv6)
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_IMCN_FLAG: request IMCN flag
 * @QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_EXTENDED_TECHNOLOGY: request extended technology info
 *
 * Flags specifying which specific settings to return when requesting the
 * current WDS bearer settings.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_get_current_settings_requested_settings >*/
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_NONE                      = 0,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PROFILE_ID                = 1 << 0,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PROFILE_NAME              = 1 << 1,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PDP_TYPE                  = 1 << 2,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_APN_NAME                  = 1 << 3,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_DNS_ADDRESS               = 1 << 4,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_GRANTED_QOS               = 1 << 5,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_USERNAME                  = 1 << 6,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_AUTH_PROTOCOL             = 1 << 7,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_IP_ADDRESS                = 1 << 8,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_GATEWAY_INFO              = 1 << 9,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PCSCF_ADDRESS             = 1 << 10,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PCSCF_SERVER_ADDRESS_LIST = 1 << 11,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_PCSCF_DOMAIN_NAME_LIST    = 1 << 12,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_MTU                       = 1 << 13,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_DOMAIN_NAME_LIST          = 1 << 14,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_IP_FAMILY                 = 1 << 15,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_IMCN_FLAG                 = 1 << 16,
    QMI_WDS_GET_CURRENT_SETTINGS_REQUESTED_SETTINGS_EXTENDED_TECHNOLOGY       = 1 << 17,
} QmiWdsGetCurrentSettingsRequestedSettings;

/**
 * qmi_wds_get_current_settings_requested_settings_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsPdpType:
 * @QMI_WDS_PDP_TYPE_IPV4: IPv4
 * @QMI_WDS_PDP_TYPE_PPP: PPP
 * @QMI_WDS_PDP_TYPE_IPV6: IPv6
 * @QMI_WDS_PDP_TYPE_IPV4_OR_IPV6: IPv4 and IPv6 combined context
 *
 * PDP context type.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_pdp_type >*/
    QMI_WDS_PDP_TYPE_IPV4         = 0,
    QMI_WDS_PDP_TYPE_PPP          = 1,
    QMI_WDS_PDP_TYPE_IPV6         = 2,
    QMI_WDS_PDP_TYPE_IPV4_OR_IPV6 = 3
} QmiWdsPdpType;

/**
 * qmi_wds_pdp_type_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsTrafficClass:
 * @QMI_WDS_TRAFFIC_CLASS_SUBSCRIBED: default (?) class, best-effort
 * @QMI_WDS_TRAFFIC_CLASS_CONVERSATIONAL: two-way video/voice, most delay sensitive
 * @QMI_WDS_TRAFFIC_CLASS_STREAMING: one-way video/audio, delay sensitive
 * @QMI_WDS_TRAFFIC_CLASS_INTERACTIVE: delay-insensitive (browsing, SSH)
 * @QMI_WDS_TRAFFIC_CLASS_BACKGROUND: delay-insensitive (downloads, email)
 *
 * QoS Traffic Classes.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_traffic_class >*/
    QMI_WDS_TRAFFIC_CLASS_SUBSCRIBED     = 0,
    QMI_WDS_TRAFFIC_CLASS_CONVERSATIONAL = 1,
    QMI_WDS_TRAFFIC_CLASS_STREAMING      = 2,
    QMI_WDS_TRAFFIC_CLASS_INTERACTIVE    = 3,
    QMI_WDS_TRAFFIC_CLASS_BACKGROUND     = 4
} QmiWdsTrafficClass;

/**
 * qmi_wds_traffic_class_get_string:
 *
 * Since: 1.0
 */

/**
 * QmiWdsAuthentication:
 * @QMI_WDS_AUTHENTICATION_NONE: no authentication
 * @QMI_WDS_AUTHENTICATION_PAP: PAP authentication
 * @QMI_WDS_AUTHENTICATION_CHAP: CHAP authentication
 *
 * PDP context authentication protocols.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_authentication >*/
    QMI_WDS_AUTHENTICATION_NONE = 0,
    QMI_WDS_AUTHENTICATION_PAP  = 1 << 0,
    QMI_WDS_AUTHENTICATION_CHAP = 1 << 1
} QmiWdsAuthentication;

/**
 * qmi_wds_authentication_build_string_from_mask:
 *
 * Since: 1.0
 */

/**
 * QmiWdsProfileType:
 * @QMI_WDS_PROFILE_TYPE_3GPP: 3GPP profile type.
 * @QMI_WDS_PROFILE_TYPE_3GPP2: 3GPP2 profile type.
 *
 * Profile network type family.
 *
 * Since: 1.0
 */
typedef enum { /*< underscore_name=qmi_wds_profile_type >*/
    QMI_WDS_PROFILE_TYPE_3GPP    = 0,
    QMI_WDS_PROFILE_TYPE_3GPP2   = 1
} QmiWdsProfileType;

/**
 * qmi_wds_profile_type_get_string:
 *
 * Since: 1.0
 */

 /**
  * QmiWdsProfileFamily:
  * @QMI_WDS_PROFILE_FAMILY_EMBEDDED: Embedded profile family.
  * @QMI_WDS_PROFILE_FAMILY_TETHERED: Tethered profile family.
  *
  * Profile family.
  *
  * Since: 1.22
  */
 typedef enum { /*< underscore_name=qmi_wds_profile_family >*/
     QMI_WDS_PROFILE_FAMILY_EMBEDDED = 0,
     QMI_WDS_PROFILE_FAMILY_TETHERED = 1
 } QmiWdsProfileFamily;

 /**
  * qmi_wds_profile_family_get_string:
  *
  * Since: 1.22
  */

/**
 * QmiWdsDeliveryOrder:
 * @QMI_WDS_QOS_DELIVERY_ORDER_SUBSCRIBE: Subscribe.
 * @QMI_WDS_QOS_DELIVERY_ORDER_ON: On.
 * @QMI_WDS_QOS_DELIVERY_ORDER_OFF: Off.
 *
 * QoS delivery order.
 *
 * Since: 1.14
 */
typedef enum { /*< underscore_name=qmi_wds_delivery_order >*/
    QMI_WDS_QOS_DELIVERY_ORDER_SUBSCRIBE = 0,
    QMI_WDS_QOS_DELIVERY_ORDER_ON        = 1,
    QMI_WDS_QOS_DELIVERY_ORDER_OFF       = 2
} QmiWdsDeliveryOrder;

/**
 * qmi_wds_delivery_order_get_string:
 *
 * Since: 1.14
 */

/**
 * QmiWdsSduErrorRatio:
 * @QMI_WDS_SDU_ERROR_RATIO_SUBSCRIBE: Subscribe.
 * @QMI_WDS_SDU_ERROR_RATIO_1E2: 1x10^2.
 * @QMI_WDS_SDU_ERROR_RATIO_7E3: 7x10^3.
 * @QMI_WDS_SDU_ERROR_RATIO_1E3: 1x10^3.
 * @QMI_WDS_SDU_ERROR_RATIO_1E4: 1x10^4.
 * @QMI_WDS_SDU_ERROR_RATIO_1E5: 1x10^5.
 * @QMI_WDS_SDU_ERROR_RATIO_1E6: 1x10^6.
 * @QMI_WDS_SDU_ERROR_RATIO_1E1: 1x10^7.
 *
 * Target value for the fraction of SDUs lost or detected as erroneous.
 *
 * Since: 1.14
 */
typedef enum { /*< underscore_name=qmi_wds_sdu_error_ratio >*/
    QMI_WDS_SDU_ERROR_RATIO_SUBSCRIBE = 0,
    QMI_WDS_SDU_ERROR_RATIO_1E2       = 1,
    QMI_WDS_SDU_ERROR_RATIO_7E3       = 2,
    QMI_WDS_SDU_ERROR_RATIO_1E3       = 3,
    QMI_WDS_SDU_ERROR_RATIO_1E4       = 4,
    QMI_WDS_SDU_ERROR_RATIO_1E5       = 5,
    QMI_WDS_SDU_ERROR_RATIO_1E6       = 6,
    QMI_WDS_SDU_ERROR_RATIO_1E1       = 7
} QmiWdsSduErrorRatio;

/**
 * qmi_wds_sdu_error_ratio_get_string:
 *
 * Since: 1.14
 */

/**
 * QmiWdsSduResidualBitErrorRatio:
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_SUBSCRIBE: Subscribe.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_5E2: 5x10^2.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E2: 1x10^2.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_5E3: 5x10^3.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_4E3: 4x10^3.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E3: 1x10^3.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E4: 1x10^4.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E5: 1x10^5.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E6: 1x10^6.
 * @QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_6E8: 6x10^8.
 *
 * Target value for the undetected bit error ratio in the delivered SDUs.
 *
 * Since: 1.14
 */
typedef enum { /*< underscore_name=qmi_wds_sdu_residual_bit_error_ratio >*/
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_SUBSCRIBE = 0,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_5E2       = 1,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E2       = 2,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_5E3       = 3,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_4E3       = 4,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E3       = 5,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E4       = 6,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E5       = 7,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_1E6       = 8,
    QMI_WDS_SDU_RESIDUAL_BIT_ERROR_RATIO_6E8       = 9
} QmiWdsSduResidualBitErrorRatio;

/**
 * qmi_wds_sdu_residual_bit_error_ratio_get_string:
 *
 * Since: 1.14
 */

/**
 * QmiWdsSduErroneousDelivery:
 * @QMI_WDS_SDU_ERRONEOUS_DELIVERY_SUBSCRIBE: Subscribe.
 * @QMI_WDS_SDU_ERRONEOUS_DELIVERY_NO_DETECTION: No detection.
 * @QMI_WDS_SDU_ERRONEOUS_DELIVERY_ENABLED: Erroneous SDUs are delivered.
 * @QMI_WDS_SDU_ERRONEOUS_DELIVERY_DISABLED: Erroneous SDUs are delivered.
 *
 * Whether SDUs detected as erroneous are delivered or not.
 *
 * Since: 1.14
 */
typedef enum { /*< underscore_name=qmi_wds_sdu_erroneous_delivery >*/
    QMI_WDS_SDU_ERRONEOUS_DELIVERY_SUBSCRIBE    = 0,
    QMI_WDS_SDU_ERRONEOUS_DELIVERY_NO_DETECTION = 1,
    QMI_WDS_SDU_ERRONEOUS_DELIVERY_ENABLED      = 2,
    QMI_WDS_SDU_ERRONEOUS_DELIVERY_DISABLED     = 3
} QmiWdsSduErroneousDelivery;

/**
 * qmi_wds_sdu_erroneous_delivery_get_string:
 *
 * Since: 1.14
 */

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Packet Statistics' message */

/**
 * QmiWdsPacketStatisticsMaskFlag:
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_OK: Request count of correctly sent packets.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_OK: Request count of correctly received packets.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_ERROR: Request count of sent packets with error.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_ERROR: Request count of received packets with error.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_OVERFLOWS: Request count of transmitter overflows.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_OVERFLOWS: Request count of receiver overflows.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_BYTES_OK: Request count of correctly sent bytes.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_BYTES_OK: Request count of correctly received bytes.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_DROPPED: Request count of dropped packets in transmission.
 * @QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_DROPPED: Request count of dropped packets in reception.
 *
 * Mask to use when requesting packet statistics.
 *
 * Since: 1.6
 */
typedef enum {
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_OK      = 1 << 0,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_OK      = 1 << 1,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_ERROR   = 1 << 2,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_ERROR   = 1 << 3,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_OVERFLOWS       = 1 << 4,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_OVERFLOWS       = 1 << 5,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_BYTES_OK        = 1 << 6,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_BYTES_OK        = 1 << 7,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_TX_PACKETS_DROPPED = 1 << 8,
    QMI_WDS_PACKET_STATISTICS_MASK_FLAG_RX_PACKETS_DROPPED = 1 << 9
} QmiWdsPacketStatisticsMaskFlag;

/**
 * qmi_wds_packet_statistics_mask_flag_build_string_from_mask:
 *
 * Since: 1.6
 */

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Get Profile List' message */

/**
 * QmiWdsDsProfileError:
 * @QMI_WDS_DS_PROFILE_ERROR_UNKNOWN: Unknown DS profile error.
 * @QMI_WDS_DS_PROFILE_ERROR_FAIL: General failure.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_HANDLE: Invalid handle.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_OPERATION: Invalid operation.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_TYPE: Invalid profile type.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_NUMBER: Invalid profile number.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_ID: Invalid profile identifier.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_ARGUMENT: Invalid argument.
 * @QMI_WDS_DS_PROFILE_ERROR_REGISTRY_NOT_INITIALIZED: Registry not initialized.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_PARAMETER_LENGTH: Invalid parameter length.
 * @QMI_WDS_DS_PROFILE_ERROR_LIST_END: List end, profile not found.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_SUBSCRIPTION_ID: Invalid subscription identifier.
 * @QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_FAMILY: Invalid profile family.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_INVALID_PROFILE_FAMILY: Invalid 3GPP profile family.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_ACCESS_ERROR: Error accessing 3GPP profiles.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_CONTEXT_NOT_DEFINED: 3GPP profile doesn't have a valid context.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_VALID_FLAG_NOT_SET: 3GPP profile doesn't have valid flag set.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_READ_ONLY_FLAG_SET: 3GPP profile has read-only flag set.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP_OUT_OF_PROFILES: Out of 3GPP profiles.
 * @QMI_WDS_DS_PROFILE_ERROR_3GPP2_INVALID_PROFILE_ID: Invalid 3GPP2 profile identifier.
 *
 * Extended DS profile errors.
 *
 * Since: 1.8
 */
typedef enum {
    QMI_WDS_DS_PROFILE_ERROR_UNKNOWN                     = 0,
    QMI_WDS_DS_PROFILE_ERROR_FAIL                        = 1,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_HANDLE              = 2,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_OPERATION           = 3,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_TYPE        = 4,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_NUMBER      = 5,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_ID          = 6,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_ARGUMENT            = 7,
    QMI_WDS_DS_PROFILE_ERROR_REGISTRY_NOT_INITIALIZED    = 8,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_PARAMETER_LENGTH    = 9,
    QMI_WDS_DS_PROFILE_ERROR_LIST_END                    = 10,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_SUBSCRIPTION_ID     = 11,
    QMI_WDS_DS_PROFILE_ERROR_INVALID_PROFILE_FAMILY      = 12,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_INVALID_PROFILE_FAMILY = 1001,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_ACCESS_ERROR           = 1002,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_CONTEXT_NOT_DEFINED    = 1003,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_VALID_FLAG_NOT_SET     = 1004,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_READ_ONLY_FLAG_SET     = 1005,
    QMI_WDS_DS_PROFILE_ERROR_3GPP_OUT_OF_PROFILES        = 1006,
    QMI_WDS_DS_PROFILE_ERROR_3GPP2_INVALID_PROFILE_ID    = 1101,
} QmiWdsDsProfileError;

/**
 * qmi_wds_ds_profile_error_get_string:
 *
 * Since: 1.8
 */

/**
 * QmiWdsAutoconnectSetting:
 * @QMI_WDS_AUTOCONNECT_SETTING_DISABLED: Disabled.
 * @QMI_WDS_AUTOCONNECT_SETTING_ENABLED: Enabled.
 * @QMI_WDS_AUTOCONNECT_SETTING_PAUSED: Paused (resume on power cycle).
 *
 * Status of the autoconnect setting.
 *
 * Since: 1.14
 */
typedef enum {
    QMI_WDS_AUTOCONNECT_SETTING_DISABLED = 0,
    QMI_WDS_AUTOCONNECT_SETTING_ENABLED  = 1,
    QMI_WDS_AUTOCONNECT_SETTING_PAUSED   = 2,
} QmiWdsAutoconnectSetting;

/**
 * qmi_wds_autoconnect_setting_get_string:
 *
 * Since: 1.14
 */

/**
 * QmiWdsAutoconnectSettingRoaming:
 * @QMI_WDS_AUTOCONNECT_SETTING_ROAMING_ALLOWED: Autoconnect always allowed, even when roaming.
 * @QMI_WDS_AUTOCONNECT_SETTING_ROAMING_HOME_ONLY: Autoconnect allowed only in home network.
 *
 * Status of the autoconnect setting while roaming.
 *
 * Since: 1.14
 */
typedef enum {
    QMI_WDS_AUTOCONNECT_SETTING_ROAMING_ALLOWED   = 0,
    QMI_WDS_AUTOCONNECT_SETTING_ROAMING_HOME_ONLY = 1,
} QmiWdsAutoconnectSettingRoaming;

/**
 * qmi_wds_autoconnect_setting_roaming_get_string:
 *
 * Since: 1.14
 */

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Set Event Report' message */

/**
 * QmiWdsSetEventReportTransferStatistics:
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_OK: report TX packet successes.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_OK: report RX packet successes.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_ERROR: report TX packet errors.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_ERROR: report RX packet errors.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_OVERFLOWS: report TX overflows.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_OVERFLOWS: report RX overflows.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_BYTES_OK: report TX byte total.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_BYTES_OK: report RX byte total.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_DROPPED: report TX packets dropped.
 * @QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_DROPPED: report RX packets dropped.
 *
 * Flags indicating which WDS Set Event Report Transfer Statistics indicators
 * should be emitted.
 *
 * Since: 1.18
 */
typedef enum {
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_OK      = 1 << 0,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_OK      = 1 << 1,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_ERROR   = 1 << 2,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_ERROR   = 1 << 3,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_OVERFLOWS       = 1 << 4,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_OVERFLOWS       = 1 << 5,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_BYTES_OK        = 1 << 6,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_BYTES_OK        = 1 << 7,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_TX_PACKETS_DROPPED = 1 << 8,
    QMI_WDS_SET_EVENT_REPORT_TRANSFER_STATISTICS_REPORT_RX_PACKETS_DROPPED = 1 << 9,
} QmiWdsSetEventReportTransferStatistics;

/**
 * qmi_wds_set_event_report_transfer_statistics_get_string:
 *
 * Since: 1.18
 */

/**
 * QmiWdsDormancyStatus:
 * @QMI_WDS_DORMANCY_STATUS_UNKNOWN: dormancy status unknown.
 * @QMI_WDS_DORMANCY_STATUS_TRAFFIC_CHANNEL_DORMANT: traffic channel is dormant.
 * @QMI_WDS_DORMANCY_STATUS_TRAFFIC_CHANNEL_ACTIVE: traffic channel is active.
 *
 * Dormancy status.
 *
 * Since: 1.18
 */
typedef enum {
    QMI_WDS_DORMANCY_STATUS_UNKNOWN                 = 0,
    QMI_WDS_DORMANCY_STATUS_TRAFFIC_CHANNEL_DORMANT = 1,
    QMI_WDS_DORMANCY_STATUS_TRAFFIC_CHANNEL_ACTIVE  = 2,
} QmiWdsDormancyStatus;

/**
 * qmi_wds_dormancy_status_get_string:
 *
 * Since: 1.18
 */

/**
 * QmiWdsDataCallStatus:
 * @QMI_WDS_DATA_CALL_STATUS_UNKNOWN: data call status unknown.
 * @QMI_WDS_DATA_CALL_STATUS_ACTIVATED: data call status activated.
 * @QMI_WDS_DATA_CALL_STATUS_TERMINATED: data call status terminated.
 *
 * Data call status.
 *
 * Since: 1.18
 */
typedef enum {
    QMI_WDS_DATA_CALL_STATUS_UNKNOWN    = 0,
    QMI_WDS_DATA_CALL_STATUS_ACTIVATED  = 1,
    QMI_WDS_DATA_CALL_STATUS_TERMINATED = 2,
} QmiWdsDataCallStatus;

/**
 * qmi_wds_data_call_status_get_string:
 *
 * Since: 1.18
 */

/**
 * QmiWdsDataCallType:
 * @QMI_WDS_DATA_CALL_TYPE_UNKNOWN: data call type unknown.
 * @QMI_WDS_DATA_CALL_TYPE_EMBEDDED: embedded data call.
 * @QMI_WDS_DATA_CALL_TYPE_TETHERED: tethered data call.
 * @QMI_WDS_DATA_CALL_TYPE_MODEM_EMBEDDED: modem embedded data call.
 *
 * Data call type.
 *
 * Since: 1.18
 */
typedef enum {
    QMI_WDS_DATA_CALL_TYPE_UNKNOWN        = 0,
    QMI_WDS_DATA_CALL_TYPE_EMBEDDED       = 1,
    QMI_WDS_DATA_CALL_TYPE_TETHERED       = 2,
    QMI_WDS_DATA_CALL_TYPE_MODEM_EMBEDDED = 3,
} QmiWdsDataCallType;

/**
 * qmi_wds_data_call_type_get_string:
 *
 * Since: 1.18
 */

/**
 * QmiWdsTetheredCallType:
 * @QMI_WDS_TETHERED_CALL_TYPE_NON_TETHERED: non-tethered.
 * @QMI_WDS_TETHERED_CALL_TYPE_RMNET: RmNet.
 * @QMI_WDS_TETHERED_CALL_TYPE_DUN: DUN.
 *
 * Tethered call type.
 *
 * Since: 1.18
 */
typedef enum {
    QMI_WDS_TETHERED_CALL_TYPE_NON_TETHERED = 0,
    QMI_WDS_TETHERED_CALL_TYPE_RMNET        = 1,
    QMI_WDS_TETHERED_CALL_TYPE_DUN          = 2,
} QmiWdsTetheredCallType;

/**
 * qmi_wds_tethered_call_type_get_string:
 *
 * Since: 1.18
 */

/**
 * QmiWdsRadioAccessTechnology:
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_NULL_BEARER: null bearer
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_WCDMA: WCDMA.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_GERAN: GERAN.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_LTE: LTE.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_TDSCDMA: TD-SDCMA.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_WLAN: 3GPP WLAN.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_MAXIMUM: 3GPP maximum.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_1X: CDMA 1x.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_HRPD: CDMA EVDO (HRPD).
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_EHRPD: CDMA EVDO with eHRPD.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_WLAN: 3GPP2 WLAN.
 * @QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_MAXIMUM: 3GPP2 maximum.
 *
 * Radio access technology.
 *
 * Since: 1.18
 */
typedef enum {
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_NULL_BEARER   = 0,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_WCDMA    = 1,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_GERAN    = 2,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_LTE      = 3,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_TDSCDMA  = 4,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_WLAN     = 5,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP_MAXIMUM  = 100,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_1X      = 101,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_HRPD    = 102,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_EHRPD   = 103,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_WLAN    = 104,
    QMI_WDS_RADIO_ACCESS_TECHNOLOGY_3GPP2_MAXIMUM = 200,
} QmiWdsRadioAccessTechnology;

/**
 * qmi_wds_radio_access_technology_get_string:
 *
 * Since: 1.18
 */

/**
 * QmiWdsExtendedDataBearerTechnology3gpp:
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_UNKNOWN: unknown.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_WCDMA: WCDMA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSDPA: HSDPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSUPA: HSUPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSDPA_PLUS: HSDPA+.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_DC_HSDPA_PLUS: Dual-cell HSDPA+.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_64QAM: 64-QAM.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSPA: HSPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_GPRS: GPRS.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_EDGE: EDGE.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_GSM: GSM.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_S2B: S2B.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_LIMITED_SERVICE: LTE limited service.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_FDD: LTE FDD.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_TDD: LTE TDD.
 *
 * 3GPP extended data bearer technologies.
 *
 * Since: 1.18
 */
typedef enum { /*< underscore_name=qmi_wds_extended_data_bearer_technology_3gpp >*/
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_UNKNOWN             = 0,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_WCDMA               = 1 << 0,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSDPA               = 1 << 1,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSUPA               = 1 << 2,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSDPA_PLUS          = 1 << 3,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_DC_HSDPA_PLUS       = 1 << 4,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_64QAM               = 1 << 5,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_HSPA                = 1 << 6,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_GPRS                = 1 << 7,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_EDGE                = 1 << 8,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_GSM                 = 1 << 9,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_S2B                 = 1 << 10,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_LIMITED_SERVICE = 1 << 11,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_FDD             = 1 << 12,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP_LTE_TDD             = 1 << 13,
} QmiWdsExtendedDataBearerTechnology3gpp;

/**
 * qmi_wds_extended_data_bearer_technology_3gpp_build_string_from_mask:
 *
 * Since: 1.18
 */

/**
 * QmiWdsExtendedDataBearerTechnology3gpp2:
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_UNKNOWN: unknown.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_RESERVED: reserved.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS95: CDMA1x IS-95.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS2000: CDMA1x IS-2000.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS2000_REL_A: CDMA1x IS-2000 Release A.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_0_DPA: HDR (EVDO) Rev 0 DPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_DPA: HDR (EVDO) Rev A DPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_DPA: HDR (EVDO) Rev B DPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_MPA: HDR (EVDO) Rev A MPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_MPA: HDR (EVDO) Rev B MPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_EMPA: HDR (EVDO) Rev A EMPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_EMPA: HDR (EVDO) Rev B EMPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_MMPA: HDR (EVDO) Rev B MMPA.
 * @QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_EVDO_FMC: HDR EVDO FMC.
 *
 * 3GPP2 extended data bearer technologies.
 *
 * Since: 1.18
 */
typedef enum { /*< underscore_name=qmi_wds_extended_data_bearer_technology_3gpp2 >*/
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_UNKNOWN             = 0,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_RESERVED            = 1 << 0,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS95         = 1 << 1,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS2000       = 1 << 2,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_CDMA1X_IS2000_REL_A = 1 << 3,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_0_DPA       = 1 << 4,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_DPA       = 1 << 5,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_DPA       = 1 << 6,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_MPA       = 1 << 7,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_MPA       = 1 << 8,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_A_EMPA      = 1 << 9,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_EMPA      = 1 << 10,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_REV_B_MMPA      = 1 << 11,
    QMI_WDS_EXTENDED_DATA_BEARER_TECHNOLOGY_3GPP2_HDR_EVDO_FMC        = 1 << 12,
} QmiWdsExtendedDataBearerTechnology3gpp2;

/**
 * qmi_wds_extended_data_bearer_technology_3gpp2_build_string_from_mask:
 *
 * Since: 1.18
 */

/*****************************************************************************/
/* Helper enums for the 'QMI WDS Profile'-related messages */

/**
 * QmiWdsPdpHeaderCompressionType:
 * @QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_OFF: no header compression
 * @QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_MANUFACTURER_PREFERRED: manufacturer preferred
 * @QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC1144: RFC1144
 * @QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC2507: RFC2507
 * @QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC3095: RFC3095
 *
 * PDP header compression types.
 *
 * Since: 1.18
 */
typedef enum { /*< underscore_name=qmi_wds_pdp_header_compression_type >*/
    QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_OFF                    = 0,
    QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_MANUFACTURER_PREFERRED = 1,
    QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC1144                = 2,
    QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC2507                = 3,
    QMI_WDS_PDP_HEADER_COMPRESSION_TYPE_RFC3095                = 4,
} QmiWdsPdpHeaderCompressionType;

/**
 * qmi_wds_pdp_header_compression_type_get_string:
 *
 * Since: 1.18
 */

/**
 * QmiWdsPdpDataCompressionType:
 * @QMI_WDS_PDP_DATA_COMPRESSION_TYPE_OFF: no data compression
 * @QMI_WDS_PDP_DATA_COMPRESSION_TYPE_MANUFACTURER_PREFERRED: manufacturer preferred
 * @QMI_WDS_PDP_DATA_COMPRESSION_TYPE_V42BIS: V42bis
 * @QMI_WDS_PDP_DATA_COMPRESSION_TYPE_V44: V44
 *
 * PDP data compression types.
 *
 * Since: 1.18
 */
typedef enum { /*< underscore_name=qmi_wds_pdp_data_compression_type >*/
    QMI_WDS_PDP_DATA_COMPRESSION_TYPE_OFF                    = 0,
    QMI_WDS_PDP_DATA_COMPRESSION_TYPE_MANUFACTURER_PREFERRED = 1,
    QMI_WDS_PDP_DATA_COMPRESSION_TYPE_V42BIS                 = 2,
    QMI_WDS_PDP_DATA_COMPRESSION_TYPE_V44                    = 3,
} QmiWdsPdpDataCompressionType;

/**
 * qmi_wds_pdp_data_compression_type_get_string:
 *
 * Since: 1.18
 */

/**
 * QmiWdsQosClassIdentifier:
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_NETWORK_ASSIGN: network assigned
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE1: guaranteed bitrate 1
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE2: guaranteed bitrate 2
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE3: guaranteed bitrate 3
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE4: guaranteed bitrate 4
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE5: non-guaranteed bitrate 5
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE6: non-guaranteed bitrate 6
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE7: non-guaranteed bitrate 7
 * @QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE8: non-guaranteed bitrate 8
 *
 * QoS Class Identifier.
 *
 * Since: 1.18
 */
typedef enum { /*< underscore_name=qmi_wds_qos_class_identifier >*/
    QMI_WDS_QOS_CLASS_IDENTIFIER_NETWORK_ASSIGN          = 0,
    QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE1     = 1,
    QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE2     = 2,
    QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE3     = 3,
    QMI_WDS_QOS_CLASS_IDENTIFIER_GUARANTEED_BITRATE4     = 4,
    QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE5 = 5,
    QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE6 = 6,
    QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE7 = 7,
    QMI_WDS_QOS_CLASS_IDENTIFIER_NON_GUARANTEED_BITRATE8 = 8,
} QmiWdsQosClassIdentifier;

/**
 * qmi_wds_qos_class_identifier_get_string:
 *
 * Since: 1.18
 */

/**
 * QmiWdsClientType:
 * @QMI_WDS_CLIENT_TYPE_TETHERED: client type tethered
 * @QMI_WDS_CLIENT_TYPE_UNDEFINED: no client type defined
 *
 * Client Type
 *
 * Since: 1.18
 */
typedef enum { /*< underscore_name=qmi_wds_client_type > */
    QMI_WDS_CLIENT_TYPE_TETHERED = 0x01,
    QMI_WDS_CLIENT_TYPE_UNDEFINED = 0xFF,
} QmiWdsClientType;

/**
 * qmi_wds_client_type_get_string:
 *
 * Since: 1.18
 */
#endif /* _LIBQMI_GLIB_QMI_ENUMS_WDS_H_ */
