/*
	Copyright (C) 2021 - 2022
	by Iris Morelle <shadowm@wesnoth.org>
	Part of the Battle for Wesnoth Project https://www.wesnoth.org/

	This program is free software; you can redistribute it and/or modify
	it under the terms of the GNU General Public License as published by
	the Free Software Foundation; either version 2 of the License, or
	(at your option) any later version.
	This program is distributed in the hope that it will be useful,
	but WITHOUT ANY WARRANTY.

	See the COPYING file for more details.
*/

#pragma once

/**
 * @file
 * Transitional API for porting SDL_ttf-based code to Pango. Do NOT use in new code!
 *
 * @note GUI1 markup is not supported by this transitional API for cost-benefit reasons.
 * Not only does implementing it require a lot more work to go over text line by line,
 * it also had major design flaws -- namely, only applying to whole lines with variable
 * spans that would be decided by the layout algorithm depending on available space,
 * rather than on a physical line basis (markup start till EOL) or fixed span basis (e.g.
 * the special markup used by the Help browser, or Pango markup).
 */

#include "font/text.hpp"

class CVideo;

namespace font {

/**
 * Returns a SDL surface containing the rendered text.
 */
surface pango_render_text(const std::string& text, int size, const color_t& color, font::pango_text::FONT_STYLE style = font::pango_text::STYLE_NORMAL, bool use_markup = false, int max_width = -1);

/**
 * Determine the width and height of a line of text given a certain font size.
 */
std::pair<int, int> pango_line_size(const std::string& line, int font_size, font::pango_text::FONT_STYLE font_style = font::pango_text::STYLE_NORMAL);

/**
 * Determine the width of a line of text given a certain font size.
 */
inline int pango_line_width(const std::string& line, int font_size, font::pango_text::FONT_STYLE font_style = font::pango_text::STYLE_NORMAL)
{
	return pango_line_size(line, font_size, font_style).first;
}

/**
 * If the text exceeds the specified max width, end it with an ellipsis (...)
 */
std::string pango_line_ellipsize(const std::string& text, int font_size, int max_width, font::pango_text::FONT_STYLE font_style = font::pango_text::STYLE_NORMAL);

/**
 * Uses Pango to word wrap text.
 */
std::string pango_word_wrap(const std::string& unwrapped_text, int font_size, int max_width, int max_height = -1, int max_lines = -1, bool partial_line = false);

/**
 * Draws text on a surface.
 *
 * The text will be clipped to area.  If the text runs outside of area
 * horizontally, an ellipsis will be displayed at the end of it.
 *
 * If use_tooltips is true, then text with an ellipsis will have a tooltip
 * set for it equivalent to the entire contents of the text.
 *
 * A bounding rectangle of the text is returned. If dst is nullptr, then the
 * text will not be drawn, and a bounding rectangle only will be returned.
 */
SDL_Rect pango_draw_text(surface& dst, const SDL_Rect& area, int size, const color_t& color, const std::string& text, int x, int y, bool use_tooltips = false, pango_text::FONT_STYLE style = pango_text::STYLE_NORMAL);

/**
 * Draws text on the screen.
 *
 * gui can be nullptr, in which case the bounding rectangle will still be
 * returned.
 */
SDL_Rect pango_draw_text(CVideo* gui, const SDL_Rect& area, int size, const color_t& color, const std::string& text, int x, int y, bool use_tooltips = false, pango_text::FONT_STYLE style = pango_text::STYLE_NORMAL);

} // end namespace font
