/*
  function.hh, copyright (c) 2006 by Vincent Fourmond: 
  The (public) definition of functions.
  
  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 2 of the License, or
  (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details (in the COPYING file).
  
*/

namespace SCalc {


  /** \brief A function definition with any number of parameters

      The FuncDef class represents a function definition. A function
      can have any number of parameters. A typical way to create a
      function definition is the following:

      \code
      SCalc::Session sess;
      SCalc::FuncDef * funcdef = sess.eval("f : x-> x** 2 + 1");
      \endcode
      
      There are different kind of functions:
      - SCalc::ExprFunc is an expression-based function definition, such as
      the one in the above example;
      - SCalc::CFunc points to a function written in C. They
      have to be created manually. Unless you provide a derivative using
      SCalc::CFunc::set_derivative(), it will not be possible to derive them
      (returns systematically 0).
  */
  class FuncDef : public ParserResult {
  protected:
    
    int _nb_params;

    /// The name is something particularly important. FuncDefs split
    /// themselves into two categories:
    ///  * anonymous functions, that probably should be disposed
    ///    of after use; (when to determine this is something
    ///    tricky...).
    ///  * named functions, which *should* be registered to the session.
    std::string _name;
  public:
    FuncDef(Session * s, int nb) : ParserResult(s)
    { _nb_params = nb; };

    /// Yes, this is a function definition
    virtual int is_func_def() { return 1;};

    /// Pretty printing of the result ?
    virtual std::string pretty_print();

    /// Register the function to the session if it has a name.
    int register_self();

    /// The number of params the function takes
    int nb_params() { return _nb_params;};

    /// Set the name of the function. (does not register it).
    /// Be careful, as setting the name to a function and not
    /// registering it soon will cause memory leaks.
    void set_name(const char * name) { _name = std::string(name);};

    std::string name() { return _name;};


    /// Evaluate the function; first argument is the undefined variables
    /// second is the actual function arguments.
    virtual double evaluate(const double * vars, const double * args) = 0;

    /// This function registers common functions to the given session.
    static void register_common_functions(Session * sess);
    virtual ~FuncDef() {;};

    /// The derivative with regards to the argument nb.
    virtual FuncDef * derivative(int nb) = 0;

    /// Delete the derivative if anonymous.
    virtual void destroy_anonymous_derivatives() {;};

    /// We can delete this function freely unless it has a name -- that
    /// is, unless it is registered.
    virtual int can_delete() { return _name.empty();};
  };

  /** A basic C function with one parameter */
  class CFunc : public FuncDef {
  public:
    /// The type of arguments it accepts
    typedef double (*c_function_t)(double);

  protected:
    /// The C function to be called.
    c_function_t func;

    /// The derivative. A pointer to a function. If NULL, then the
    /// function has to take care to provide it when necessary (or fail)
    /// and to free it when it dies.
    FuncDef * deriv;
  public:
    CFunc(Session * s, const char * n, 
	  c_function_t func, 
	  FuncDef * derivat = NULL);

    virtual ~CFunc() {;};

    /// The function doing the actual job...
    virtual double evaluate(const double * vars, const double * args);

    /// We can set the derivative later, as this can come in really
    /// useful...
    void set_derivative(FuncDef * d) { deriv = d;};

    /// Delete the derivative if anonymous.
    virtual void destroy_anonymous_derivatives();

    /// Gets the derivative
    virtual FuncDef * derivative(int nb)
    { if(nb) return NULL; return deriv; };
  };

  /** A basic C function with one argument and one external parameter */
  class CFuncParam : public CFunc {
  public:
    /// The type of arguments it accepts
    typedef double (*c_function_t)(void *, double);

  protected:
    /// The C function to be called.
    c_function_t func;

    /// The derivative. A pointer to a function. If NULL, then the
    /// function has to take care to provide it when necessary (or fail)
    /// and to free it when it dies.
    FuncDef * deriv;

    /// The parameter !
    void * _param;
  public:
    CFuncParam(Session * s, const char * n, 
	       c_function_t func, void * param,
	       FuncDef * derivat = NULL);

    virtual ~CFuncParam() {;};

    /// The function doing the actual job...
    virtual double evaluate(const double * vars, const double * args);

    void * param() { return _param;};
    void set_param(void * p) { _param = p;};
  };

  /** An expression-based definition of a function */
  class ExprFunc : public FuncDef {
    /// produced on the fly.
    std::map<int, FuncDef *> cached_derivatives; 
    
    Expression * exp;
  public:
    /// construction of the function.
    ExprFunc(Session * s, Expression * expr, int nb_args) :
      FuncDef(s, nb_args) { exp = expr;};
    virtual ~ExprFunc() { delete exp;};

    /// Delete the derivative if anonymous.
    virtual void destroy_anonymous_derivatives();

    virtual FuncDef * derivative(int nb);

    /// The function doing the actual job... pretty easy, isn't it ?
    virtual double evaluate(const double * vars, 
			    const double * args)
    { return exp->evaluate(vars, args);};

    
    virtual std::string pretty_print();

  };

};
