/**
 * Copyright (c) Members of the EGEE Collaboration. 2004-2010.
 * See http://www.eu-egee.org/partners/ for details on the copyright
 * holders.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 *
 *  Authors:
 *  2004-
 *     Oscar Koeroo <okoeroo@nikhef.nl>
 *     NIKHEF Amsterdam, the Netherlands
 *     <grid-mw-security@nikhef.nl>
 *
 */

/* Needed for (un)setenv */
#define _XOPEN_SOURCE	600

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <strings.h>
#include <time.h>
#include <errno.h>

#include <openssl/evp.h>
#include <openssl/x509.h>
#include <openssl/x509v3.h>

#include "verify_x509.h"
#include "_verify_log.h"
#include "verify_x509_datatypes.h"
#include "_verify_x509.h"
#include "_verify_proxy_certinfo.h"

/************************************************************************/
/* DEFINES								*/	
/************************************************************************/

#define EXT_TEXT_LEN	80

/************************************************************************/
/* INTERNAL PROTOTYPES							*/	
/************************************************************************/

/* Note that timegm() is non-standard. Linux manpage advices the following
 * substition instead. */
static time_t my_timegm(struct tm *tm);

/* Convert OID to NID and log error when it's undefined
 * return nid or NID_undef */
static int my_oid2nid(const char *oid);

/* Obtains and adds proxy language type to the proxy type p_type
 * return 0 on success, -1 on error */
static int get_proxy_lang(PROXYPOLICY *policy,
			  proxy_cert_info_type_t basic_type,
			  proxy_type_t *p_type);

/************************************************************************/
/* FUNCTION DEFINITIONS							*/	
/************************************************************************/

/**
 * Check if certificate can be used as a CA to sign standard X509 certs,
 * i.e. it must have the keyCertSign key usage specified, even if it does
 * not have the CA=True
 * Return 1 if true; 0 if not.
 */
int verify_x509IsCA(X509 *cert)
{
    int idret;

    /* final argument to X509_check_purpose() is whether to check for CAness */
    idret = X509_check_purpose(cert, X509_PURPOSE_SSL_CLIENT, 1);
    if (idret == 1)
        return 1;
    else if (idret == 0)
        return 0;
    else
    {
        verify_log( L_WARN, "Purpose warning code = %d", idret );
        return 1;
    }

#if 0
    BASIC_CONSTRAINTS *                 x509v3_bc = NULL;
    int                                 index = -1;
    int                                 critical;
    if((x509v3_bc = X509_get_ext_d2i(cert,
                    NID_basic_constraints,
                    &critical,
                    &index)) && x509v3_bc->ca)
    {
        *type = GLOBUS_GSI_CERT_UTILS_TYPE_CA;
        goto exit;
    }
#endif
}

/**
 * Converts ASN1 time string (in a ASN1_TIME) to time_t
 * using ASN1_STRING_data() ASN1_STRING_get0_data() to convert
 * ASN1_GENERALIZEDTIME to const char *, then calling
 * verify_str_asn1TimeToTimeT()*/
time_t verify_asn1TimeToTimeT(ASN1_TIME *asn1time)  {
    /* protect against NULL pointers */
    if (asn1time==NULL)
	return 0;

#if OPENSSL_VERSION_NUMBER < 0x010100000L
    return verify_str_asn1TimeToTimeT((const char*)ASN1_STRING_data(asn1time));
#else
    return verify_str_asn1TimeToTimeT((const char*)ASN1_STRING_get0_data(asn1time));
#endif
}

/**
 * Converts ASN1 time string (in a const char *) to time_t. See also
 * verify_asn1TimeToTimeT()
 */
time_t verify_str_asn1TimeToTimeT(const char *asn1time)
{
   char   zone;
   struct tm time_tm;
   size_t len;

   /* Make sure to start with an empty struct */
   memset(&time_tm,0,sizeof(struct tm));

   len = strlen(asn1time);

   if ((len != 13) && (len != 15)) return 0; /* dont understand */

   if ((len == 13) &&
       ((sscanf(asn1time, "%02d%02d%02d%02d%02d%02d%c",
         &(time_tm.tm_year),
         &(time_tm.tm_mon),
         &(time_tm.tm_mday),
         &(time_tm.tm_hour),
         &(time_tm.tm_min),
         &(time_tm.tm_sec),
         &zone) != 7) || (zone != 'Z'))) return 0; /* dont understand */

   if ((len == 15) &&
       ((sscanf(asn1time, "20%02d%02d%02d%02d%02d%02d%c",
         &(time_tm.tm_year),
         &(time_tm.tm_mon),
         &(time_tm.tm_mday),
         &(time_tm.tm_hour),
         &(time_tm.tm_min),
         &(time_tm.tm_sec),
         &zone) != 7) || (zone != 'Z'))) return 0; /* dont understand */

   /* time format fixups */

   if (time_tm.tm_year < 90) time_tm.tm_year += 100;
   --(time_tm.tm_mon);

   return my_timegm(&time_tm);
}


/**
 * Returns type of a certificate.
 * Valid values are defined in verify_x509_datatypes.h
 */
proxy_type_t verify_type_of_proxy(X509 * cert) {
    const char *logstr="verify_type_of_proxy";
    proxy_type_t pt = NONE;
    char * cert_subjectdn = NULL;
    char * cert_issuerdn = NULL;
    char * tail_str = NULL;
    size_t len_subject_dn, len_issuer_dn, len_proxy, len_limited_proxy;

    PROXYCERTINFO     *  pci_rfc = NULL;
    PROXYCERTINFO_GT3 *  pci_gt3 = NULL;
    int              rfc_proxy_nid, globus_proxy_v3_nid;
    int              crit;

    int mixed_gt3 = 0;

    if ( (rfc_proxy_nid=my_oid2nid(RFC_PROXY_OID))==NID_undef ||
	 (globus_proxy_v3_nid=my_oid2nid(GLOBUS_PROXY_V3_OID))== NID_undef)
	goto failure;

    /* Is it a CA certificate */
    if (verify_x509IsCA(cert)) {
        /* verify_log (L_DEBUG, "%s: Detected CA certificate", logstr); */
	/* Don't try other types when we found CA type */
        pt = CA;
        goto finalize;
    }

    /* Check by OID */

    /* Get RFC proxy Proxy certificate info extension */
    if ( (pci_rfc=X509_get_ext_d2i(cert, rfc_proxy_nid, &crit, NULL) ) != NULL )
    {
	/* Found exactly one (valid) RFC proxy extension */

	/* Check it is critical */
	if (crit==0)	{
	    verify_error(logstr,
		    "Found RFC PROXYCERTINFO extension which is not critical");
	    goto failure;
	}

	/* Get proxy language type (e.g. limited) */
	if (get_proxy_lang(pci_rfc->policy, RFC_TYPE, &pt)<0)
	    goto failure;

	/* all done: don't finalize yet, we need to check we don't also have
	 * different (conflicting) extensions. */
    } else {
	/* Did not find exactly one (valid) RFC proxy extension, check the
	 * reason, before trying other proxy types */
	if (crit==-2)   {	/* more than one extension */
	    verify_error(logstr, "Found more than one RFC PCI extension");
	    goto failure;
	}
	if (crit>=0)    {   /* could not convert: X509V3_EXT_d2i() failed */
	    verify_error(logstr,
		    "Can't convert DER encoded RFC PROXYCERTINFO extension to "
		    "internal form");
	    goto failure;
	}
    }

    /* Also check GT3 proxy */

    pci_gt3=X509_get_ext_d2i(cert, globus_proxy_v3_nid, &crit, NULL);
    /* If we did not find a GT3, it could be that the proxyCertInfo is RFC */
    if (pci_gt3 == NULL && crit!=-2 && crit>=0)	{
	/* temporarily change the extension matching code */
	mixed_gt3 = 1;  /* indicate we're looking for a mixed GT3 proxy */

	/* Set proxyCertificateInfo type to RFC ordering */
	set_gt3_pci_type_rfc();

	pci_gt3=X509_get_ext_d2i(cert, globus_proxy_v3_nid, &crit, NULL);

	/* Set proxyCertificateInfo type back to GT3 ordering */
	set_gt3_pci_type_gt3();
    }

    /* either found exactly one valid RFC proxy or crit == -1 (i.e. no RFC proxy
     * critical extension), now try (also) GT3 */
    if ( pci_gt3 != NULL )
    {
	/* If we found a GT3 proxy with a RFC-type proxyCertInfo, warn!
	 * caNL-java and hence voms-proxy-init3 can produce these */
	if (mixed_gt3)  {
	    verify_log(L_WARN,
		    "Found mixed GT3 PCI extension: "
		    "GT3 OID with RFC proxyCertInfo");
	}

	/* Found valid GT3 proxy extension: check we didn't ALSO have an RFC
	 * proxy extension */
	if (CERTISTYPE(pt, RFC_TYPE))   {
	    verify_error(logstr, "Found both RFC and GT3 PCI extensions");
	    goto failure;
	}

	/* Don't check whether critical is set to 1, since VOMS doesn't set it
	 * for GT3 */

	/* Get proxy language type (e.g. limited) */
	if (get_proxy_lang(pci_gt3->policy, GT3_TYPE, &pt)<0)
	    goto failure;

	/* all done */
	goto finalize;
    }

    /* Did not find a valid GT3 proxy extension, check the reason */
    if (crit==-2)   {	/* more than one extension */
	verify_error(logstr, "Found more than one PCI extension");
	goto failure;
    }
    if (crit>=0)    {   /* could not convert: X509V3_EXT_d2i() failed */
	verify_error(logstr,
		"Can't convert DER encoded GT3 PROXYCERTINFO extension to "
		"internal form");
	goto failure;
    }
    
    /* crit == -1 -> no GT3 proxy critical extension, but maybe we already had a
     * RFC proxy */
    if (pt!=NONE)
	goto finalize;


    /* Options left: GT2_PROXY, GT2_LIMITED_PROXY, EEC */
    /* Extract Subject DN - Needs free */
    if (!(cert_subjectdn = X509_NAME_oneline (X509_get_subject_name (cert), NULL, 0))) {
        verify_error (logstr,
		"Error: Couldn't get the subject DN from the certificate.");
        goto failure;
    }
    if (!(cert_issuerdn = X509_NAME_oneline (X509_get_issuer_name (cert), NULL, 0))) {
        verify_error (logstr,
		"Error: Couldn't get the issuer DN from the certificate.");
        goto failure;
    }

    /* Check length of the DNs */
    len_subject_dn = strlen(cert_subjectdn);
    len_issuer_dn  = strlen(cert_issuerdn);

    /* Proxies always has a longer subject_dn then a issuer_dn and
     * the issuer_dn is a substring of the subject_dn
     */
    if ( (len_issuer_dn < len_subject_dn) &&
	 (strncmp(cert_subjectdn, cert_issuerdn, len_issuer_dn) == 0) )
    {
        /* Check for GT2_PROXY tail */
	len_proxy=strlen("/cn=proxy");
        if ( len_subject_dn >= len_proxy &&
	     (tail_str = &cert_subjectdn[len_subject_dn - len_proxy]) &&
	     (strcasecmp(tail_str, "/cn=proxy") == 0)
           ) {
            /* verify_log (L_DEBUG, "%s: Detected GT2 proxy certificate", logstr); */
            pt = GT2_PROXY;
            goto finalize;
        }

        /* Check for GT2_LIMITED_PROXY tail */
	len_limited_proxy=strlen("/cn=limited proxy");
        if ( len_subject_dn >= len_limited_proxy &&
	     (tail_str = &cert_subjectdn[len_subject_dn - len_limited_proxy]) &&
	     (strcasecmp(tail_str, "/cn=limited proxy") == 0)
           ) {
            /* verify_log (L_DEBUG, "%s: Detected GT2 limited proxy certificate", logstr); */
            pt = GT2_LIMITED_PROXY;
            goto finalize;
        }

	/* Don't know the type of proxy, we have checked RFC and GT3 proxies
	 * already before, since we have added the OIDs by hand if OpenSSL
	 * didn't provide them. */
        goto failure;
    }


    /* I have no idea what else it is, so I conclude that it's an EEC */
    pt = EEC;
    goto finalize;

failure:
    /* On failure, or non-distinct selections of the certificate, indicate NONE */
    pt = NONE;
finalize:
    if (pci_rfc)
	/* Free proxy cert info */
	PROXYCERTINFO_free(pci_rfc);
    if (pci_gt3)
	/* Free proxy cert info */
	PROXYCERTINFO_GT3_free(pci_gt3);
    if (cert_subjectdn)
        free(cert_subjectdn);
    if (cert_issuerdn)
        free(cert_issuerdn);

    return pt;
}

/******************************************************************************
 * INTERNAL FUNCTIONS
 ******************************************************************************/

/**
 * Note that timegm() is non-standard. Linux manpage advices the following
 * substition instead.
 */
static time_t my_timegm(struct tm *tm)
{
   time_t ret;
   char *tz;

   tz = getenv("TZ");
   setenv("TZ", "", 1);
   tzset();
   ret = mktime(tm);
   if (tz)
       setenv("TZ", tz, 1);
   else
       unsetenv("TZ");
   tzset();

   return ret;
}

/**
 * Convert OID to NID and log error when it's undefined
 * return nid or NID_undef
 */
static int my_oid2nid(const char *oid)  {
    ASN1_OBJECT *obj=OBJ_txt2obj(oid,0);
    int nid=OBJ_obj2nid(obj);

    ASN1_OBJECT_free(obj);

    if (nid == NID_undef)
	verify_error("my_oid2nid", "OID %s not defined", oid);


    return nid;
}

/**
 * Obtains and adds proxy language type to the proxy type p_type
 * return 0 on success, -1 on error
 */
static int get_proxy_lang(PROXYPOLICY *policy,
			  proxy_cert_info_type_t basic_type,
			  proxy_type_t *p_type) {
    const char      *logstr="get_proxy_lang";
    int		     nid, impersonation_proxy_nid, independent_proxy_nid,
		     limited_proxy_nid, any_language_nid;
    ASN1_OBJECT *    policy_lang = NULL;
    proxy_type_t     pt = *p_type;
    char             s[EXT_TEXT_LEN];

    /* Pull a certificate policy from the extension, note: pci!=NULL since
     * we've checked that */
    if( policy == NULL) {
	verify_error(logstr,
		"Can't get policy from PROXYCERTINFO extension");
	return -1;
    }

    /* Get policy language */
    if( (policy_lang = policy->policy_language) == NULL) {
	verify_error(logstr,
		"Can't get policy language from PROXYCERTINFO extension");
	return -1;
    }

    /* First get all the nids: do indirectly via OBJ_txt2obj() to be able to
     * specify no_name==0 and to ensure we can check for NID_undef (conform
     * manpage and code instead of only code) */
    /* RFC proxy certificate info */
    if ( (impersonation_proxy_nid=my_oid2nid(IMPERSONATION_PROXY_OID))==NID_undef ||
	 (independent_proxy_nid=my_oid2nid(INDEPENDENT_PROXY_OID))==NID_undef ||
	 (any_language_nid=my_oid2nid(ANY_LANGUAGE_OID))==NID_undef ||
	 (limited_proxy_nid=my_oid2nid(LIMITED_PROXY_OID))==NID_undef)
	return -1;

    /* Lang to NID, lang's NID holds RFC Proxy type, like
     * limited. Impersonation is the default */
    nid=OBJ_obj2nid(policy_lang);
    if (nid==impersonation_proxy_nid)
	pt = basic_type | IMPERSONATION;
    else if (nid==independent_proxy_nid)
	pt = basic_type | INDEPENDENT;
    else if (nid==limited_proxy_nid)
	pt = basic_type | LIMITED;
    else if (nid==any_language_nid)
	pt = basic_type | ANYLANG;
    else    { /* Note: this included NID_undef */
	pt = basic_type | RESTRICTED;
	OBJ_obj2txt(s, EXT_TEXT_LEN, policy_lang, 1);
	verify_log(L_WARN,
	    "%s: Found %s (policy language: %s)",
	    logstr, verify_certificate_type_str(pt) ,s);
    }

    /* all done */
    *p_type=pt;
    return 0;
}
