/*
 * Copyright (c) 2021, 2023-2024 Arm Limited.
 *
 * SPDX-License-Identifier: MIT
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to
 * deal in the Software without restriction, including without limitation the
 * rights to use, copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */

#include <cstddef>
#include <cstdint>

#if defined(ARM_COMPUTE_ENABLE_SVE)

namespace arm_conv {
namespace depthwise {

void sve_fp32_nhwc_3x3_s1_output4x4_mla_depthfirst_direct_impl(
  const unsigned int n_tile_rows,
  const unsigned int n_tile_cols,
  const float *inptr,
  int64_t ld_input_row,
  int64_t ld_input_col,
  float *outptr,
  int64_t ld_output_row,
  int64_t ld_output_col,
  const void *params,
  unsigned int n_channels,
  const float activation_min,
  const float activation_max
)
{
  struct Args
  {
    const uint64_t n_tile_rows, n_tile_cols;
    const float *inptr;
    const uint64_t ld_input_row;
    const uint64_t ld_input_col;
    float *outptr;
    const uint64_t ld_output_row;
    const uint64_t ld_output_col;
    const void *params;
    const float min, max;

    uint64_t tile_i = 0, tile_j = 0;

    Args(
      const unsigned int n_tile_rows,
      const unsigned int n_tile_cols,
      const float *inptr,
      int64_t ld_input_row,
      int64_t ld_input_col,
      float *outptr,
      int64_t ld_output_row,
      int64_t ld_output_col,
      const void *params,
      const float activation_min,
      const float activation_max
    ) : n_tile_rows(n_tile_rows), n_tile_cols(n_tile_cols), inptr(inptr),
        ld_input_row(ld_input_row), ld_input_col(ld_input_col), outptr(outptr),
        ld_output_row(ld_output_row), ld_output_col(ld_output_col),
        params(params), min(activation_min), max(activation_max)
    {
    }
  };

  Args params_struct(
    n_tile_rows, n_tile_cols,
    inptr, ld_input_row, ld_input_col,
    outptr, ld_output_row, ld_output_col,
    params, activation_min, activation_max
  );

  __asm__ __volatile__(
    "ptrue p3.b\n"
    "mov x1, #0x0\n"
    "mov x2, #0x0\n"
    "1:"  // Tile loop
    "str x1, [%x[params_struct], %[offsetof_args_tile_i]]\n"
    "mov x20, #0x4\n"
    "mov x25, #0x4\n"
    "str x2, [%x[params_struct], %[offsetof_args_tile_j]]\n"
    "ldr x24, [%x[params_struct], %[offsetof_args_ld_input_row]]\n"
    "ldr x23, [%x[params_struct], %[offsetof_args_ld_output_row]]\n"
    "cntw x3\n"
    "ldr x4, [%x[params_struct], %[offsetof_args_ld_input_col]]\n"
    "ldr x5, [%x[params_struct], %[offsetof_args_ld_output_col]]\n"
    "whilelt p2.s, XZR, %x[n_channels]\n"
    "mov x6, #0x0\n"
    "ldr x7, [%x[params_struct], %[offsetof_args_inptr]]\n"
    "ldr x8, [%x[params_struct], %[offsetof_args_params]]\n"
    "mul x22, x1, x24\n"  // offset = tile_i * ld_input_row
    "mul x21, x1, x23\n"  // offset = tile_i * ld_output_row
    "ldr x17, [%x[params_struct], %[offsetof_args_outptr]]\n"
    "cmp x3, %x[n_channels]\n"
    "ld1rw { z27.s }, p3/Z, [%x[params_struct], %[offsetof_args_min]]\n"
    "add x16, x4, x4\n"
    "add x15, x5, x5\n"
    "ld1rw { z29.s }, p3/Z, [%x[params_struct], %[offsetof_args_max]]\n"
    "madd x22, x2, x4, x22\n"  // offset += tile_j * ld_input_col
    "add x14, x16, x4\n"
    "ld1w { z13.s }, p3/Z, [x8]\n"
    "ld1w { z0.s }, p3/Z, [x8, #1, MUL VL]\n"
    "add x13, x15, x5\n"
    "madd x21, x2, x5, x21\n"  // offset += tile_j * ld_output_col
    "ld1w { z1.s }, p3/Z, [x8, #2, MUL VL]\n"
    "ld1w { z2.s }, p3/Z, [x8, #3, MUL VL]\n"
    "add x12, x14, x4\n"
    "mul x22, x22, x20\n"  // offset *= kernel_stride * output_size
    "ld1w { z3.s }, p3/Z, [x8, #4, MUL VL]\n"
    "ld1w { z4.s }, p3/Z, [x8, #5, MUL VL]\n"
    "add x11, x12, x4\n"
    "ld1w { z5.s }, p3/Z, [x8, #6, MUL VL]\n"
    "ld1w { z6.s }, p3/Z, [x8, #7, MUL VL]\n"
    "addvl x8, x8, #16\n"
    "sub x20, XZR, x3\n"
    "mul x21, x21, x25\n"  // offset *= output_tile_size
    "add x7, x7, x22, LSL #2\n"  // inptr[0] += offset * sizeof(float)
    "add x10, x7, x24, LSL #2\n"
    "add x9, x10, x24, LSL #2\n"
    "ld1w { z10.s }, p2/Z, [x7]\n"
    "ld1w { z11.s }, p2/Z, [x7, x11, LSL #2]\n"
    "add x28, x9, x24, LSL #2\n"
    "add x27, x28, x24, LSL #2\n"
    "ld1w { z7.s }, p3/Z, [x8, #-8, MUL VL]\n"
    "ld1w { z8.s }, p3/Z, [x8, #-7, MUL VL]\n"
    "addvl x8, x8, #-6\n"
    "add x17, x17, x21, LSL #2\n"  // outptrs[0] += offset * sizeof(float)
    "add x26, x27, x24, LSL #2\n"
    "ld1w { z9.s }, p2/Z, [x9, x16, LSL #2]\n"
    "ld1w { z12.s }, p2/Z, [x9, x14, LSL #2]\n"
    "add x25, x17, x23, LSL #2\n"
    "add x24, x25, x23, LSL #2\n"
    "add x23, x24, x23, LSL #2\n"
    "bge 3f\n"
    "2:"  // Tile loop: Channel loop
    "movprfx z14, z13\n fmla z14.s, p3/M, z4.s, z9.s\n"
    "movprfx z19, z13\n fmla z19.s, p3/M, z8.s, z9.s\n"
    "whilelt p1.s, x3, %x[n_channels]\n"
    "incw x6\n"
    "movprfx z18, z13\n fmla z18.s, p3/M, z3.s, z9.s\n"
    "movprfx z26, z13\n fmla z26.s, p3/M, z1.s, z9.s\n"
    "incw x3\n"
    "mov p0.b, p2.b\n"
    "movprfx z15, z13\n fmla z15.s, p3/M, z0.s, z9.s\n"
    "movprfx z30, z13\n fmla z30.s, p3/M, z7.s, z9.s\n"
    "incw x20\n"
    "movprfx z28, z13\n fmla z28.s, p3/M, z6.s, z9.s\n"
    "movprfx z21, z13\n fmla z21.s, p3/M, z5.s, z9.s\n"
    "fmla z14.s, p3/M, z5.s, z12.s\n"
    "movprfx z24, z13\n fmla z24.s, p3/M, z2.s, z9.s\n"
    "ld1w { z9.s }, p2/Z, [x28, x16, LSL #2]\n"
    "fmla z19.s, p3/M, z0.s, z10.s\n"
    "movprfx z22, z13\n fmla z22.s, p3/M, z2.s, z11.s\n"
    "ld1w { z11.s }, p2/Z, [x26]\n"
    "ld1w { z10.s }, p2/Z, [x26, x11, LSL #2]\n"
    "fmla z18.s, p3/M, z4.s, z12.s\n"
    "fmla z26.s, p3/M, z2.s, z12.s\n"
    "fmla z15.s, p3/M, z1.s, z12.s\n"
    "fmla z30.s, p3/M, z8.s, z12.s\n"
    "movprfx z25, z13\n fmla z25.s, p3/M, z6.s, z11.s\n"
    "fmla z14.s, p3/M, z7.s, z9.s\n"
    "ld1w { z11.s }, p2/Z, [x28, x14, LSL #2]\n"
    "fmla z28.s, p3/M, z7.s, z12.s\n"
    "fmla z22.s, p3/M, z6.s, z12.s\n"
    "movprfx z31, z13\n fmla z31.s, p3/M, z3.s, z12.s\n"
    "movprfx z17, z13\n fmla z17.s, p3/M, z0.s, z12.s\n"
    "ld1w { z12.s }, p2/Z, [x7, x4, LSL #2]\n"
    "movprfx z20, z13\n fmla z20.s, p3/M, z8.s, z10.s\n"
    "fmla z18.s, p3/M, z6.s, z9.s\n"
    "ld1w { z10.s }, p2/Z, [x7, x12, LSL #2]\n"
    "fmla z26.s, p3/M, z4.s, z9.s\n"
    "fmla z15.s, p3/M, z3.s, z9.s\n"
    "movprfx z16, z13\n fmla z16.s, p3/M, z1.s, z9.s\n"
    "movprfx z23, z13\n fmla z23.s, p3/M, z0.s, z9.s\n"
    "ld1w { z13.s }, p3/Z, [x8]\n"
    "fmla z21.s, p3/M, z8.s, z9.s\n"
    "fmla z24.s, p3/M, z5.s, z9.s\n"
    "fmla z25.s, p3/M, z2.s, z9.s\n"
    "fmla z14.s, p3/M, z8.s, z11.s\n"
    "ld1w { z9.s }, p2/Z, [x10]\n"
    "fmla z19.s, p3/M, z1.s, z12.s\n"
    "fmla z30.s, p3/M, z0.s, z12.s\n"
    "ld1w { z12.s }, p2/Z, [x10, x11, LSL #2]\n"
    "fmla z28.s, p3/M, z2.s, z10.s\n"
    "fmla z22.s, p3/M, z1.s, z10.s\n"
    "ld1w { z10.s }, p2/Z, [x27]\n"
    "fmla z18.s, p3/M, z7.s, z11.s\n"
    "fmla z31.s, p3/M, z6.s, z11.s\n"
    "fmla z26.s, p3/M, z5.s, z11.s\n"
    "fmla z15.s, p3/M, z4.s, z11.s\n"
    "fmla z17.s, p3/M, z3.s, z11.s\n"
    "fmla z16.s, p3/M, z2.s, z11.s\n"
    "fmla z23.s, p3/M, z1.s, z11.s\n"
    "fmla z20.s, p3/M, z0.s, z11.s\n"
    "ld1w { z11.s }, p2/Z, [x10, x16, LSL #2]\n"
    "fmla z21.s, p3/M, z0.s, z9.s\n"
    "fmla z24.s, p3/M, z6.s, z10.s\n"
    "fmla z25.s, p3/M, z3.s, z10.s\n"
    "ld1w { z10.s }, p2/Z, [x27, x11, LSL #2]\n"
    "fmla z19.s, p3/M, z3.s, z9.s\n"
    "fmla z14.s, p3/M, z1.s, z11.s\n"
    "fmla z22.s, p3/M, z5.s, z12.s\n"
    "fmla z31.s, p3/M, z2.s, z12.s\n"
    "fmla z30.s, p3/M, z4.s, z11.s\n"
    "ld1w { z12.s }, p2/Z, [x10, x14, LSL #2]\n"
    "fmla z28.s, p3/M, z3.s, z11.s\n"
    "fmla z18.s, p3/M, z0.s, z11.s\n"
    "fmla z17.s, p3/M, z8.s, z10.s\n"
    "fmla z20.s, p3/M, z5.s, z10.s\n"
    "ld1w { z10.s }, p2/Z, [x26, x4, LSL #2]\n"
    "fmla z21.s, p3/M, z2.s, z11.s\n"
    "fmla z14.s, p3/M, z2.s, z12.s\n"
    "fmla z19.s, p3/M, z5.s, z11.s\n"
    "fmla z30.s, p3/M, z5.s, z12.s\n"
    "ld1w { z11.s }, p2/Z, [x9, x4, LSL #2]\n"
    "fmla z28.s, p3/M, z4.s, z12.s\n"
    "fmla z22.s, p3/M, z3.s, z12.s\n"
    "fmla z18.s, p3/M, z1.s, z12.s\n"
    "fmla z31.s, p3/M, z0.s, z12.s\n"
    "ld1w { z9.s }, p2/Z, [x9, x12, LSL #2]\n"
    "fmla z25.s, p3/M, z7.s, z10.s\n"
    "fmla z16.s, p3/M, z6.s, z10.s\n"
    "ld1w { z12.s }, p2/Z, [x26, x12, LSL #2]\n"
    "fmla z21.s, p3/M, z4.s, z11.s\n"
    "fmla z14.s, p3/M, z3.s, z11.s\n"
    "fmla z24.s, p3/M, z1.s, z11.s\n"
    "fmla z26.s, p3/M, z0.s, z11.s\n"
    "fmla z19.s, p3/M, z7.s, z11.s\n"
    "fmla z30.s, p3/M, z6.s, z11.s\n"
    "ld1w { z11.s }, p2/Z, [x7, x16, LSL #2]\n"
    "fmla z23.s, p3/M, z8.s, z12.s\n"
    "fmla z20.s, p3/M, z7.s, z12.s\n"
    "ld1w { z10.s }, p2/Z, [x28, x4, LSL #2]\n"
    "fmla z28.s, p3/M, z8.s, z9.s\n"
    "fmla z22.s, p3/M, z7.s, z9.s\n"
    "fmla z18.s, p3/M, z5.s, z9.s\n"
    "fmla z31.s, p3/M, z4.s, z9.s\n"
    "fmla z15.s, p3/M, z2.s, z9.s\n"
    "fmla z17.s, p3/M, z1.s, z9.s\n"
    "ld1w { z9.s }, p2/Z, [x7, x14, LSL #2]\n"
    "addvl x7, x7, #1\n"
    "fmla z21.s, p3/M, z7.s, z10.s\n"
    "fmla z14.s, p3/M, z6.s, z10.s\n"
    "fmla z24.s, p3/M, z4.s, z10.s\n"
    "fmla z26.s, p3/M, z3.s, z10.s\n"
    "fmla z25.s, p3/M, z1.s, z10.s\n"
    "fmla z16.s, p3/M, z0.s, z10.s\n"
    "ld1w { z12.s }, p2/Z, [x28, x12, LSL #2]\n"
    "fmla z19.s, p3/M, z2.s, z11.s\n"
    "fmla z30.s, p3/M, z1.s, z11.s\n"
    "fmla z28.s, p3/M, z0.s, z11.s\n"
    "ld1w { z10.s }, p2/Z, [x9]\n"
    "fmla z22.s, p3/M, z0.s, z9.s\n"
    "fmla z23.s, p3/M, z2.s, z12.s\n"
    "fmla z18.s, p3/M, z8.s, z12.s\n"
    "fmla z31.s, p3/M, z7.s, z12.s\n"
    "fmla z15.s, p3/M, z5.s, z12.s\n"
    "fmla z21.s, p3/M, z3.s, z10.s\n"
    "fmla z24.s, p3/M, z0.s, z10.s\n"
    "fmla z17.s, p3/M, z4.s, z12.s\n"
    "fmla z20.s, p3/M, z1.s, z12.s\n"
    "ld1w { z12.s }, p2/Z, [x27, x16, LSL #2]\n"
    "fmla z30.s, p3/M, z2.s, z9.s\n"
    "fmla z28.s, p3/M, z1.s, z9.s\n"
    "ld1w { z11.s }, p2/Z, [x9, x11, LSL #2]\n"
    "addvl x9, x9, #1\n"
    "fmla z19.s, p3/M, z6.s, z10.s\n"
    "ld1w { z10.s }, p2/Z, [x28]\n"
    "fmla z16.s, p3/M, z4.s, z12.s\n"
    "fmla z23.s, p3/M, z3.s, z12.s\n"
    "fmla z26.s, p3/M, z7.s, z12.s\n"
    "fmla z22.s, p3/M, z8.s, z11.s\n"
    "fmla z31.s, p3/M, z5.s, z11.s\n"
    "ld1w { z9.s }, p1/Z, [x9, x16, LSL #2]\n"
    "fmla z17.s, p3/M, z2.s, z11.s\n"
    "fmla z21.s, p3/M, z6.s, z10.s\n"
    "ld1w { z11.s }, p2/Z, [x28, x11, LSL #2]\n"
    "addvl x28, x28, #1\n"
    "fmla z24.s, p3/M, z3.s, z10.s\n"
    "fmla z25.s, p3/M, z0.s, z10.s\n"
    "ld1w { z10.s }, p2/Z, [x26, x16, LSL #2]\n"
    "fmla z15.s, p3/M, z6.s, z12.s\n"
    "fmla z20.s, p3/M, z2.s, z11.s\n"
    "fmla z31.s, p3/M, z8.s, z11.s\n"
    "fmla z16.s, p3/M, z7.s, z10.s\n"
    "fmla z23.s, p3/M, z6.s, z10.s\n"
    "fmla z17.s, p3/M, z5.s, z11.s\n"
    "ld1w { z11.s }, p2/Z, [x26, x14, LSL #2]\n"
    "addvl x26, x26, #1\n"
    "fmla z24.s, p3/M, z8.s, z12.s\n"
    "fmla z25.s, p3/M, z5.s, z12.s\n"
    "ld1w { z12.s }, p2/Z, [x27, x14, LSL #2]\n"
    "fmla z16.s, p3/M, z5.s, z12.s\n"
    "fmla z23.s, p3/M, z4.s, z12.s\n"
    "fmla z20.s, p3/M, z3.s, z12.s\n"
    "fmla z26.s, p3/M, z8.s, z12.s\n"
    "fmla z15.s, p3/M, z7.s, z12.s\n"
    "fmla z17.s, p3/M, z6.s, z12.s\n"
    "ld1w { z12.s }, p2/Z, [x10, x12, LSL #2]\n"
    "fmla z25.s, p3/M, z8.s, z10.s\n"
    "ld1w { z10.s }, p2/Z, [x10, x4, LSL #2]\n"
    "addvl x10, x10, #1\n"
    "fmla z16.s, p3/M, z8.s, z11.s\n"
    "fmla z23.s, p3/M, z7.s, z11.s\n"
    "fmla z20.s, p3/M, z6.s, z11.s\n"
    "ld1w { z11.s }, p2/Z, [x27, x4, LSL #2]\n"
    "fmla z28.s, p3/M, z5.s, z12.s\n"
    "fmla z22.s, p3/M, z4.s, z12.s\n"
    "fmla z19.s, p3/M, z4.s, z10.s\n"
    "fmla z30.s, p3/M, z3.s, z10.s\n"
    "fmla z21.s, p3/M, z1.s, z10.s\n"
    "fmla z14.s, p3/M, z0.s, z10.s\n"
    "ld1w { z10.s }, p2/Z, [x27, x12, LSL #2]\n"
    "ld1w { z0.s }, p3/Z, [x8, #1, MUL VL]\n"
    "fmla z18.s, p3/M, z2.s, z12.s\n"
    "fmla z31.s, p3/M, z1.s, z12.s\n"
    "ld1w { z1.s }, p3/Z, [x8, #2, MUL VL]\n"
    "ld1w { z2.s }, p3/Z, [x8, #3, MUL VL]\n"
    "fmla z24.s, p3/M, z7.s, z11.s\n"
    "fmla z26.s, p3/M, z6.s, z11.s\n"
    "fmax z28.s, p3/M, z28.s, z27.s\n"
    "fmax z22.s, p3/M, z22.s, z27.s\n"
    "fmla z25.s, p3/M, z4.s, z11.s\n"
    "fmla z16.s, p3/M, z3.s, z11.s\n"
    "fmax z19.s, p3/M, z19.s, z27.s\n"
    "fmax z30.s, p3/M, z30.s, z27.s\n"
    "fmla z15.s, p3/M, z8.s, z10.s\n"
    "fmla z17.s, p3/M, z7.s, z10.s\n"
    "fmax z21.s, p3/M, z21.s, z27.s\n"
    "fmax z14.s, p3/M, z14.s, z27.s\n"
    "fmla z23.s, p3/M, z5.s, z10.s\n"
    "fmla z20.s, p3/M, z4.s, z10.s\n"
    "fmax z18.s, p3/M, z18.s, z27.s\n"
    "fmax z31.s, p3/M, z31.s, z27.s\n"
    "fmax z24.s, p3/M, z24.s, z27.s\n"
    "fmax z26.s, p3/M, z26.s, z27.s\n"
    "ld1w { z3.s }, p3/Z, [x8, #4, MUL VL]\n"
    "ld1w { z4.s }, p3/Z, [x8, #5, MUL VL]\n"
    "fmax z25.s, p3/M, z25.s, z27.s\n"
    "fmax z16.s, p3/M, z16.s, z27.s\n"
    "ld1w { z5.s }, p3/Z, [x8, #6, MUL VL]\n"
    "ld1w { z6.s }, p3/Z, [x8, #7, MUL VL]\n"
    "fmax z15.s, p3/M, z15.s, z27.s\n"
    "fmax z17.s, p3/M, z17.s, z27.s\n"
    "ld1w { z10.s }, p1/Z, [x7]\n"
    "ld1w { z11.s }, p1/Z, [x7, x11, LSL #2]\n"
    "fmax z23.s, p3/M, z23.s, z27.s\n"
    "fmax z20.s, p3/M, z20.s, z27.s\n"
    "ld1w { z12.s }, p1/Z, [x9, x14, LSL #2]\n"
    "addvl x8, x8, #16\n"
    "whilelt p2.s, x6, %x[n_channels]\n"
    "cmp x3, %x[n_channels]\n"
    "fmin z19.s, p3/M, z19.s, z29.s\n"
    "fmin z30.s, p3/M, z30.s, z29.s\n"
    "fmin z28.s, p3/M, z28.s, z29.s\n"
    "fmin z22.s, p3/M, z22.s, z29.s\n"
    "fmin z21.s, p3/M, z21.s, z29.s\n"
    "ld1w { z7.s }, p3/Z, [x8, #-8, MUL VL]\n"
    "ld1w { z8.s }, p3/Z, [x8, #-7, MUL VL]\n"
    "fmin z14.s, p3/M, z14.s, z29.s\n"
    "fmin z18.s, p3/M, z18.s, z29.s\n"
    "st1w { z19.s }, p0, [x17]\n"
    "fmin z31.s, p3/M, z31.s, z29.s\n"
    "fmin z24.s, p3/M, z24.s, z29.s\n"
    "st1w { z30.s }, p0, [x17, x5, LSL #2]\n"
    "fmin z26.s, p3/M, z26.s, z29.s\n"
    "fmin z15.s, p3/M, z15.s, z29.s\n"
    "st1w { z28.s }, p0, [x17, x15, LSL #2]\n"
    "fmin z17.s, p3/M, z17.s, z29.s\n"
    "fmin z25.s, p3/M, z25.s, z29.s\n"
    "st1w { z22.s }, p0, [x17, x13, LSL #2]\n"
    "fmin z16.s, p3/M, z16.s, z29.s\n"
    "fmin z23.s, p3/M, z23.s, z29.s\n"
    "st1w { z21.s }, p0, [x25]\n"
    "fmin z20.s, p3/M, z20.s, z29.s\n"
    "addvl x27, x27, #1\n"
    "st1w { z14.s }, p0, [x25, x5, LSL #2]\n"
    "st1w { z18.s }, p0, [x25, x15, LSL #2]\n"
    "addvl x17, x17, #1\n"
    "addvl x8, x8, #-6\n"
    "st1w { z31.s }, p0, [x25, x13, LSL #2]\n"
    "addvl x25, x25, #1\n"
    "st1w { z24.s }, p0, [x24]\n"
    "st1w { z26.s }, p0, [x24, x5, LSL #2]\n"
    "st1w { z15.s }, p0, [x24, x15, LSL #2]\n"
    "st1w { z17.s }, p0, [x24, x13, LSL #2]\n"
    "addvl x24, x24, #1\n"
    "st1w { z25.s }, p0, [x23]\n"
    "st1w { z16.s }, p0, [x23, x5, LSL #2]\n"
    "st1w { z23.s }, p0, [x23, x15, LSL #2]\n"
    "st1w { z20.s }, p0, [x23, x13, LSL #2]\n"
    "addvl x23, x23, #1\n"
    "blt 2b\n"
    "3:"  // Tile loop: Channel tail
    "movprfx z14, z13\n fmla z14.s, p3/M, z4.s, z9.s\n"
    "movprfx z18, z13\n fmla z18.s, p3/M, z8.s, z9.s\n"
    "ldr x2, [%x[params_struct], %[offsetof_args_tile_j]]\n"
    "ldr x1, [%x[params_struct], %[offsetof_args_tile_i]]\n"
    "movprfx z23, z13\n fmla z23.s, p3/M, z3.s, z9.s\n"
    "movprfx z30, z13\n fmla z30.s, p3/M, z1.s, z9.s\n"
    "ldr x22, [%x[params_struct], %[offsetof_args_n_tile_cols]]\n"
    "ldr x21, [%x[params_struct], %[offsetof_args_n_tile_rows]]\n"
    "movprfx z20, z13\n fmla z20.s, p3/M, z0.s, z9.s\n"
    "movprfx z25, z13\n fmla z25.s, p3/M, z7.s, z9.s\n"
    "mov p0.b, p2.b\n"
    "movprfx z19, z13\n fmla z19.s, p3/M, z6.s, z9.s\n"
    "movprfx z26, z13\n fmla z26.s, p3/M, z5.s, z9.s\n"
    "add x2, x2, #0x1\n"
    "add x20, x1, #0x1\n"
    "fmla z14.s, p3/M, z5.s, z12.s\n"
    "movprfx z28, z13\n fmla z28.s, p3/M, z2.s, z9.s\n"
    "ld1w { z15.s }, p2/Z, [x28, x16, LSL #2]\n"
    "cmp x2, x22\n"
    "fmla z18.s, p3/M, z0.s, z10.s\n"
    "movprfx z9, z13\n fmla z9.s, p3/M, z2.s, z11.s\n"
    "ld1w { z17.s }, p2/Z, [x26]\n"
    "ld1w { z24.s }, p2/Z, [x26, x11, LSL #2]\n"
    "fmla z23.s, p3/M, z4.s, z12.s\n"
    "fmla z30.s, p3/M, z2.s, z12.s\n"
    "csel x1, x1, x20, LT\n"
    "csel x2, x2, XZR, LT\n"
    "fmla z20.s, p3/M, z1.s, z12.s\n"
    "fmla z25.s, p3/M, z8.s, z12.s\n"
    "movprfx z22, z13\n fmla z22.s, p3/M, z6.s, z17.s\n"
    "fmla z14.s, p3/M, z7.s, z15.s\n"
    "ld1w { z10.s }, p2/Z, [x28, x14, LSL #2]\n"
    "fmla z19.s, p3/M, z7.s, z12.s\n"
    "fmla z9.s, p3/M, z6.s, z12.s\n"
    "cmp x1, x21\n"
    "movprfx z31, z13\n fmla z31.s, p3/M, z3.s, z12.s\n"
    "movprfx z11, z13\n fmla z11.s, p3/M, z0.s, z12.s\n"
    "ld1w { z16.s }, p2/Z, [x7, x4, LSL #2]\n"
    "movprfx z12, z13\n fmla z12.s, p3/M, z8.s, z24.s\n"
    "fmla z23.s, p3/M, z6.s, z15.s\n"
    "ld1w { z17.s }, p2/Z, [x7, x12, LSL #2]\n"
    "fmla z30.s, p3/M, z4.s, z15.s\n"
    "fmla z20.s, p3/M, z3.s, z15.s\n"
    "movprfx z24, z13\n fmla z24.s, p3/M, z1.s, z15.s\n"
    "fmla z13.s, p3/M, z0.s, z15.s\n"
    "fmla z26.s, p3/M, z8.s, z15.s\n"
    "fmla z28.s, p3/M, z5.s, z15.s\n"
    "fmla z22.s, p3/M, z2.s, z15.s\n"
    "fmla z14.s, p3/M, z8.s, z10.s\n"
    "ld1w { z15.s }, p2/Z, [x10]\n"
    "fmla z18.s, p3/M, z1.s, z16.s\n"
    "fmla z25.s, p3/M, z0.s, z16.s\n"
    "ld1w { z21.s }, p2/Z, [x10, x11, LSL #2]\n"
    "fmla z19.s, p3/M, z2.s, z17.s\n"
    "fmla z9.s, p3/M, z1.s, z17.s\n"
    "ld1w { z16.s }, p2/Z, [x27]\n"
    "fmla z23.s, p3/M, z7.s, z10.s\n"
    "fmla z31.s, p3/M, z6.s, z10.s\n"
    "fmla z30.s, p3/M, z5.s, z10.s\n"
    "fmla z20.s, p3/M, z4.s, z10.s\n"
    "fmla z11.s, p3/M, z3.s, z10.s\n"
    "fmla z24.s, p3/M, z2.s, z10.s\n"
    "fmla z13.s, p3/M, z1.s, z10.s\n"
    "fmla z12.s, p3/M, z0.s, z10.s\n"
    "ld1w { z17.s }, p2/Z, [x10, x16, LSL #2]\n"
    "fmla z26.s, p3/M, z0.s, z15.s\n"
    "fmla z28.s, p3/M, z6.s, z16.s\n"
    "fmla z22.s, p3/M, z3.s, z16.s\n"
    "ld1w { z16.s }, p2/Z, [x27, x11, LSL #2]\n"
    "fmla z18.s, p3/M, z3.s, z15.s\n"
    "fmla z14.s, p3/M, z1.s, z17.s\n"
    "fmla z9.s, p3/M, z5.s, z21.s\n"
    "fmla z31.s, p3/M, z2.s, z21.s\n"
    "fmla z25.s, p3/M, z4.s, z17.s\n"
    "ld1w { z21.s }, p2/Z, [x10, x14, LSL #2]\n"
    "fmla z19.s, p3/M, z3.s, z17.s\n"
    "fmla z23.s, p3/M, z0.s, z17.s\n"
    "fmla z11.s, p3/M, z8.s, z16.s\n"
    "fmla z12.s, p3/M, z5.s, z16.s\n"
    "ld1w { z16.s }, p2/Z, [x26, x4, LSL #2]\n"
    "fmla z26.s, p3/M, z2.s, z17.s\n"
    "fmla z14.s, p3/M, z2.s, z21.s\n"
    "fmla z18.s, p3/M, z5.s, z17.s\n"
    "fmla z25.s, p3/M, z5.s, z21.s\n"
    "ld1w { z17.s }, p2/Z, [x9, x4, LSL #2]\n"
    "fmla z19.s, p3/M, z4.s, z21.s\n"
    "fmla z9.s, p3/M, z3.s, z21.s\n"
    "fmla z23.s, p3/M, z1.s, z21.s\n"
    "fmla z31.s, p3/M, z0.s, z21.s\n"
    "ld1w { z21.s }, p2/Z, [x9, x12, LSL #2]\n"
    "fmla z22.s, p3/M, z7.s, z16.s\n"
    "fmla z24.s, p3/M, z6.s, z16.s\n"
    "ld1w { z16.s }, p2/Z, [x26, x12, LSL #2]\n"
    "fmla z26.s, p3/M, z4.s, z17.s\n"
    "fmla z14.s, p3/M, z3.s, z17.s\n"
    "fmla z28.s, p3/M, z1.s, z17.s\n"
    "fmla z30.s, p3/M, z0.s, z17.s\n"
    "fmla z18.s, p3/M, z7.s, z17.s\n"
    "fmla z25.s, p3/M, z6.s, z17.s\n"
    "ld1w { z17.s }, p2/Z, [x7, x16, LSL #2]\n"
    "fmla z13.s, p3/M, z8.s, z16.s\n"
    "fmla z12.s, p3/M, z7.s, z16.s\n"
    "ld1w { z16.s }, p2/Z, [x28, x4, LSL #2]\n"
    "fmla z19.s, p3/M, z8.s, z21.s\n"
    "fmla z9.s, p3/M, z7.s, z21.s\n"
    "fmla z23.s, p3/M, z5.s, z21.s\n"
    "fmla z31.s, p3/M, z4.s, z21.s\n"
    "fmla z20.s, p3/M, z2.s, z21.s\n"
    "fmla z11.s, p3/M, z1.s, z21.s\n"
    "ld1w { z21.s }, p2/Z, [x7, x14, LSL #2]\n"
    "fmla z26.s, p3/M, z7.s, z16.s\n"
    "fmla z14.s, p3/M, z6.s, z16.s\n"
    "fmla z28.s, p3/M, z4.s, z16.s\n"
    "fmla z30.s, p3/M, z3.s, z16.s\n"
    "fmla z22.s, p3/M, z1.s, z16.s\n"
    "fmla z24.s, p3/M, z0.s, z16.s\n"
    "ld1w { z16.s }, p2/Z, [x28, x12, LSL #2]\n"
    "fmla z18.s, p3/M, z2.s, z17.s\n"
    "fmla z25.s, p3/M, z1.s, z17.s\n"
    "fmla z19.s, p3/M, z0.s, z17.s\n"
    "ld1w { z17.s }, p2/Z, [x9]\n"
    "fmla z9.s, p3/M, z0.s, z21.s\n"
    "fmla z13.s, p3/M, z2.s, z16.s\n"
    "fmla z23.s, p3/M, z8.s, z16.s\n"
    "fmla z31.s, p3/M, z7.s, z16.s\n"
    "fmla z20.s, p3/M, z5.s, z16.s\n"
    "fmla z26.s, p3/M, z3.s, z17.s\n"
    "fmla z28.s, p3/M, z0.s, z17.s\n"
    "fmla z11.s, p3/M, z4.s, z16.s\n"
    "fmla z12.s, p3/M, z1.s, z16.s\n"
    "ld1w { z15.s }, p2/Z, [x27, x16, LSL #2]\n"
    "fmla z25.s, p3/M, z2.s, z21.s\n"
    "fmla z19.s, p3/M, z1.s, z21.s\n"
    "ld1w { z16.s }, p2/Z, [x9, x11, LSL #2]\n"
    "fmla z18.s, p3/M, z6.s, z17.s\n"
    "ld1w { z17.s }, p2/Z, [x28]\n"
    "fmla z24.s, p3/M, z4.s, z15.s\n"
    "fmla z13.s, p3/M, z3.s, z15.s\n"
    "fmla z30.s, p3/M, z7.s, z15.s\n"
    "fmla z9.s, p3/M, z8.s, z16.s\n"
    "fmla z31.s, p3/M, z5.s, z16.s\n"
    "fmla z11.s, p3/M, z2.s, z16.s\n"
    "fmla z26.s, p3/M, z6.s, z17.s\n"
    "ld1w { z16.s }, p2/Z, [x28, x11, LSL #2]\n"
    "fmla z28.s, p3/M, z3.s, z17.s\n"
    "fmla z22.s, p3/M, z0.s, z17.s\n"
    "ld1w { z21.s }, p2/Z, [x26, x16, LSL #2]\n"
    "fmla z20.s, p3/M, z6.s, z15.s\n"
    "fmla z12.s, p3/M, z2.s, z16.s\n"
    "fmla z31.s, p3/M, z8.s, z16.s\n"
    "fmla z24.s, p3/M, z7.s, z21.s\n"
    "fmla z13.s, p3/M, z6.s, z21.s\n"
    "fmla z11.s, p3/M, z5.s, z16.s\n"
    "ld1w { z17.s }, p2/Z, [x26, x14, LSL #2]\n"
    "fmla z28.s, p3/M, z8.s, z15.s\n"
    "fmla z22.s, p3/M, z5.s, z15.s\n"
    "ld1w { z16.s }, p2/Z, [x27, x14, LSL #2]\n"
    "fmla z24.s, p3/M, z5.s, z16.s\n"
    "fmla z13.s, p3/M, z4.s, z16.s\n"
    "fmla z12.s, p3/M, z3.s, z16.s\n"
    "fmla z30.s, p3/M, z8.s, z16.s\n"
    "fmla z20.s, p3/M, z7.s, z16.s\n"
    "fmla z11.s, p3/M, z6.s, z16.s\n"
    "ld1w { z15.s }, p2/Z, [x10, x12, LSL #2]\n"
    "fmla z22.s, p3/M, z8.s, z21.s\n"
    "ld1w { z16.s }, p2/Z, [x10, x4, LSL #2]\n"
    "fmla z24.s, p3/M, z8.s, z17.s\n"
    "fmla z13.s, p3/M, z7.s, z17.s\n"
    "fmla z12.s, p3/M, z6.s, z17.s\n"
    "ld1w { z17.s }, p2/Z, [x27, x4, LSL #2]\n"
    "fmla z19.s, p3/M, z5.s, z15.s\n"
    "fmla z9.s, p3/M, z4.s, z15.s\n"
    "fmla z18.s, p3/M, z4.s, z16.s\n"
    "fmla z25.s, p3/M, z3.s, z16.s\n"
    "fmla z26.s, p3/M, z1.s, z16.s\n"
    "fmla z14.s, p3/M, z0.s, z16.s\n"
    "ld1w { z16.s }, p2/Z, [x27, x12, LSL #2]\n"
    "fmla z23.s, p3/M, z2.s, z15.s\n"
    "fmla z31.s, p3/M, z1.s, z15.s\n"
    "fmla z28.s, p3/M, z7.s, z17.s\n"
    "fmla z30.s, p3/M, z6.s, z17.s\n"
    "fmax z19.s, p3/M, z19.s, z27.s\n"
    "fmax z9.s, p3/M, z9.s, z27.s\n"
    "fmla z22.s, p3/M, z4.s, z17.s\n"
    "fmla z24.s, p3/M, z3.s, z17.s\n"
    "fmax z18.s, p3/M, z18.s, z27.s\n"
    "fmax z25.s, p3/M, z25.s, z27.s\n"
    "fmla z20.s, p3/M, z8.s, z16.s\n"
    "fmla z11.s, p3/M, z7.s, z16.s\n"
    "fmax z26.s, p3/M, z26.s, z27.s\n"
    "fmax z14.s, p3/M, z14.s, z27.s\n"
    "fmla z13.s, p3/M, z5.s, z16.s\n"
    "fmla z12.s, p3/M, z4.s, z16.s\n"
    "fmax z23.s, p3/M, z23.s, z27.s\n"
    "fmax z31.s, p3/M, z31.s, z27.s\n"
    "fmax z28.s, p3/M, z28.s, z27.s\n"
    "fmax z30.s, p3/M, z30.s, z27.s\n"
    "fmax z22.s, p3/M, z22.s, z27.s\n"
    "fmax z24.s, p3/M, z24.s, z27.s\n"
    "fmax z20.s, p3/M, z20.s, z27.s\n"
    "fmax z11.s, p3/M, z11.s, z27.s\n"
    "fmax z13.s, p3/M, z13.s, z27.s\n"
    "fmax z12.s, p3/M, z12.s, z27.s\n"
    "fmin z18.s, p3/M, z18.s, z29.s\n"
    "fmin z25.s, p3/M, z25.s, z29.s\n"
    "fmin z19.s, p3/M, z19.s, z29.s\n"
    "fmin z9.s, p3/M, z9.s, z29.s\n"
    "fmin z26.s, p3/M, z26.s, z29.s\n"
    "fmin z14.s, p3/M, z14.s, z29.s\n"
    "fmin z23.s, p3/M, z23.s, z29.s\n"
    "fmin z31.s, p3/M, z31.s, z29.s\n"
    "st1w { z18.s }, p0, [x17]\n"
    "fmin z28.s, p3/M, z28.s, z29.s\n"
    "fmin z30.s, p3/M, z30.s, z29.s\n"
    "st1w { z25.s }, p0, [x17, x5, LSL #2]\n"
    "fmin z20.s, p3/M, z20.s, z29.s\n"
    "fmin z11.s, p3/M, z11.s, z29.s\n"
    "st1w { z19.s }, p0, [x17, x15, LSL #2]\n"
    "fmin z22.s, p3/M, z22.s, z29.s\n"
    "fmin z24.s, p3/M, z24.s, z29.s\n"
    "st1w { z9.s }, p0, [x17, x13, LSL #2]\n"
    "fmin z13.s, p3/M, z13.s, z29.s\n"
    "fmin z12.s, p3/M, z12.s, z29.s\n"
    "st1w { z26.s }, p0, [x25]\n"
    "st1w { z14.s }, p0, [x25, x5, LSL #2]\n"
    "st1w { z23.s }, p0, [x25, x15, LSL #2]\n"
    "st1w { z31.s }, p0, [x25, x13, LSL #2]\n"
    "st1w { z28.s }, p0, [x24]\n"
    "st1w { z30.s }, p0, [x24, x5, LSL #2]\n"
    "st1w { z20.s }, p0, [x24, x15, LSL #2]\n"
    "st1w { z11.s }, p0, [x24, x13, LSL #2]\n"
    "st1w { z22.s }, p0, [x23]\n"
    "st1w { z24.s }, p0, [x23, x5, LSL #2]\n"
    "st1w { z13.s }, p0, [x23, x15, LSL #2]\n"
    "st1w { z12.s }, p0, [x23, x13, LSL #2]\n"
    "blt 1b\n"
    :
    : [n_channels] "r" ((unsigned long) n_channels), [offsetof_args_inptr] "I" (offsetof(Args, inptr)), [offsetof_args_ld_input_col] "I" (offsetof(Args, ld_input_col)), [offsetof_args_ld_input_row] "I" (offsetof(Args, ld_input_row)), [offsetof_args_ld_output_col] "I" (offsetof(Args, ld_output_col)), [offsetof_args_ld_output_row] "I" (offsetof(Args, ld_output_row)), [offsetof_args_max] "I" (offsetof(Args, max)), [offsetof_args_min] "I" (offsetof(Args, min)), [offsetof_args_n_tile_cols] "I" (offsetof(Args, n_tile_cols)), [offsetof_args_n_tile_rows] "I" (offsetof(Args, n_tile_rows)), [offsetof_args_outptr] "I" (offsetof(Args, outptr)), [offsetof_args_params] "I" (offsetof(Args, params)), [offsetof_args_tile_i] "I" (offsetof(Args, tile_i)), [offsetof_args_tile_j] "I" (offsetof(Args, tile_j)), [params_struct] "r" (&params_struct)
    : "cc", "memory", "p0", "p1", "p2", "p3", "x1", "x2", "x3", "x4", "x5", "x6", "x7", "x8", "x9", "x10", "x11", "x12", "x13", "x14", "x15", "x16", "x17", "x20", "x21", "x22", "x23", "x24", "x25", "x26", "x27", "x28", "z0", "z1", "z2", "z3", "z4", "z5", "z6", "z7", "z8", "z9", "z10", "z11", "z12", "z13", "z14", "z15", "z16", "z17", "z18", "z19", "z20", "z21", "z22", "z23", "z24", "z25", "z26", "z27", "z28", "z29", "z30", "z31"
  );
}

}  // namespace depthwise
}  // namespace arm_conv

#endif  // defined(ARM_COMPUTE_ENABLE_SVE)
